-- Copyright (c) 2004-2013, EnterpriseDB Corporation. All Rights Reserved.

GRANT USAGE ON SCHEMA sys TO PUBLIC;

CREATE DOMAIN sys.clob AS text;
CREATE DOMAIN sys.xmltype AS xml;


-- Supporting function for sys.all_* views
-- Lowers the input if not quoted. If quoted, removes quoting and added '"'
-- This is basically to reverse quote_ident_upper() translation
CREATE OR REPLACE FUNCTION sys.lower_if_not_quoted(str text) RETURN text IS
BEGIN
  IF str LIKE '"%"' THEN
    RETURN replace(substr(str, 2, length(str) - 2), '""', '"');
  ELSE
    RETURN lower(str);
  END IF;
END;


--------------------------------------------------------------------------------
--                       Redwood compatible views                             --
--------------------------------------------------------------------------------

-- In the views below, schema_name shows the schema containing the object, and
-- owner displays the user who owns the object.

--------------------------------------------------------------------------------
--                       sys.*_users views                                    --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_users AS
SELECT quote_ident_upper(rolname) AS username,
       pga.oid                    AS user_id,
       rolpassword::varchar2(30)  AS password,
       CASE WHEN now() > rolvaliduntil THEN 'EXPIRED & LOCKED'::varchar2(32)
            WHEN rolcanlogin THEN 'OPEN'::varchar2(32)
            ELSE 'LOCKED'::varchar2(32)
       END                        AS account_status,               -- NOT NULL
       null::date                 AS lock_date,
       rolvaliduntil::date        AS expiry_date,
       pgt.spcname::varchar2(30)  AS default_tablespace,           -- NOT NULL
       ''::varchar2(30)           AS temporary_tablespace,         -- NOT NULL
       null::date                 AS created,
       ''::varchar2(30)           AS profile,                      -- NOT NULL
       null::varchar2(30)         AS initial_rsrc_consumer_group,
       null::varchar2(4000)       AS external_name
FROM   pg_authid pga, pg_database pgd, pg_tablespace pgt
WHERE  pgd.datname = current_database() AND
       pgd.dattablespace = pgt.oid;

CREATE OR REPLACE VIEW sys.all_users AS
SELECT username, user_id, created
FROM   sys.dba_users;

GRANT SELECT ON sys.all_users TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_users AS
SELECT username, user_id, account_status, lock_date, expiry_date,
       default_tablespace, temporary_tablespace, created,
       initial_rsrc_consumer_group, external_name
FROM   sys.dba_users
WHERE  username = quote_ident_upper(user);

GRANT SELECT ON sys.user_users TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_tables views                                   --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_tables AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::varchar) AS owner,
       quote_ident_upper(n.nspname)   AS schema_name,
       quote_ident_upper(c.relname)   AS table_name,
       quote_ident_upper(t.spcname)   AS tablespace_name,
       'VALID'::character varying(5)  AS status,
       (CASE WHEN n.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END)::char AS temporary
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = c.reltablespace
WHERE  c.relkind = 'r'::"char";

CREATE OR REPLACE VIEW sys.all_tables AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::varchar) AS owner,
       quote_ident_upper(n.nspname)   AS schema_name,
       quote_ident_upper(c.relname)   AS table_name,
       quote_ident_upper(t.spcname)   AS tablespace_name,
       'VALID'::character varying(5)  AS status,
       (CASE WHEN n.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END)::char AS temporary
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = c.reltablespace
WHERE  c.relkind = 'r'::"char" AND
       has_table_privilege(c.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_tables TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_tables AS
SELECT schema_name, table_name, tablespace_name, status, temporary
FROM   sys.all_tables
WHERE  owner = quote_ident_upper(user);

GRANT SELECT ON sys.user_tables TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_constraints views                              --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_constraints AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(pn.nspname)       AS schema_name,
       quote_ident_upper(pc.conname)       AS constraint_name,
       quote_ident_upper(pc.contype)       AS constraint_type,
       quote_ident_upper(c.relname)        AS table_name,
       quote_ident_upper(pc.consrc)        AS search_condition,
       (CASE WHEN pc.contype='f' THEN quote_ident_upper(pn2.nspname)  ELSE NULL END) AS r_owner,
       (CASE WHEN pc.contype ='f' THEN quote_ident_upper(pc2.conname) ELSE NULL END) AS r_constraint_name,
       quote_ident_upper(pc.confdeltype)   AS delete_rule,
       pc.condeferrable                    AS deferrable,
       pc.condeferred                      AS deferred,
       quote_ident_upper(i.schemaname)     AS index_owner,
       quote_ident_upper(i.indexname)      AS index_name,
       pg_get_constraintdef(pc.oid)        AS constraint_def
FROM   pg_constraint pc
       LEFT outer JOIN pg_namespace pn ON pn.oid =pc.connamespace
       LEFT outer JOIN pg_class c ON c.oid =  pc.conrelid
       LEFT outer JOIN pg_indexes i ON i.indexname=pc.conname
       LEFT outer JOIN pg_class a ON a.oid =pc.confrelid
       LEFT outer JOIN pg_constraint pc2 ON (pc2.conrelid=pc.confrelid AND (pc2.contype='p' AND pc.contype='f'))
       LEFT outer JOIN pg_namespace pn2 ON pn2.oid =pc2.connamespace;

CREATE OR REPLACE VIEW sys.all_constraints AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(pn.nspname)       AS schema_name,
       quote_ident_upper(pc.conname)       AS constraint_name,
       quote_ident_upper(pc.contype)       AS constraint_type,
       quote_ident_upper(c.relname)        AS table_name,
       quote_ident_upper(pc.consrc)        AS search_condition,
       (CASE WHEN pc.contype='f' THEN quote_ident_upper(pn2.nspname)  ELSE NULL END) AS r_owner,
       (CASE WHEN pc.contype ='f' THEN quote_ident_upper(pc2.conname) ELSE NULL END) AS r_constraint_name,
       quote_ident_upper(pc.confdeltype)   AS delete_rule,
       pc.condeferrable                    AS deferrable,
       pc.condeferred                      AS deferred,
       quote_ident_upper(i.schemaname)     AS index_owner,
       quote_ident_upper(i.indexname)      AS index_name,
       pg_get_constraintdef(pc.oid)        AS constraint_def
FROM   pg_constraint pc
       LEFT outer JOIN pg_namespace pn ON pn.oid =pc.connamespace
       LEFT outer JOIN pg_class c ON c.oid =  pc.conrelid
       LEFT outer JOIN pg_indexes i ON i.indexname=pc.conname
       LEFT outer JOIN pg_class a ON a.oid =pc.confrelid
       LEFT outer JOIN pg_constraint pc2 ON (pc2.conrelid=pc.confrelid AND (pc2.contype='p' AND pc.contype='f'))
       LEFT outer JOIN pg_namespace pn2 ON pn2.oid =pc2.connamespace
WHERE  has_table_privilege(c.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       pn.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_constraints TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_constraints AS
SELECT owner, schema_name, constraint_name, constraint_type, table_name,
       search_condition, r_owner, r_constraint_name, delete_rule, "deferrable",
       deferred, index_owner, index_name, constraint_def
FROM   sys.all_constraints
WHERE  owner = quote_ident_upper(user);

GRANT SELECT ON sys.user_constraints TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_all_tables views                               --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_all_tables AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(n.nspname) AS schema_name,
       quote_ident_upper(c.relname) AS table_name,
       quote_ident_upper(t.spcname) AS tablespace_name,
       'VALID'::character varying(5) AS status,
       (CASE WHEN n.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = c.reltablespace
WHERE  c.relkind = 'r'::"char";

CREATE OR REPLACE VIEW sys.all_all_tables AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(n.nspname) AS schema_name,
       quote_ident_upper(c.relname) AS table_name,
       quote_ident_upper(t.spcname) AS tablespace_name,
       'VALID'::character varying(5) AS status,
       (CASE WHEN n.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = c.reltablespace
WHERE  c.relkind = 'r'::"char" AND
       has_table_privilege(c.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_all_tables TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_all_tables AS
SELECT schema_name, table_name, tablespace_name, status, temporary
FROM   sys.all_all_tables
WHERE  owner = quote_ident_upper(user);

GRANT SELECT ON sys.user_all_tables TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_triggers views                                 --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_triggers AS
SELECT quote_ident_upper(pg_get_userbyid(c.relowner)) AS owner,
       quote_ident_upper(n.nspname)                   AS schema_name,
       quote_ident_upper(t.tgname)                    AS trigger_name,
       (CASE WHEN ((t.tgtype)::integer & 2) = 2 THEN 'BEFORE'::text  ELSE 'AFTER'::text END)
       ||' '|| (CASE WHEN ((t.tgtype)::integer & 1) = 1 THEN 'ROW'::text ELSE 'STATEMENT'::text END) AS trigger_type,
       ((em.text)::text)                              AS triggering_event,
       quote_ident_upper(pg_get_userbyid(c.relowner)) AS table_owner,
       'TABLE'::text                                  AS base_object_type,
       quote_ident_upper(c.relname)                   AS table_name,
       'REFERENCING NEW AS NEW OLD AS OLD'::text      AS referencing_names,
       (CASE WHEN tgenabled !='D' THEN 'ENABLED' ELSE 'DISABLED' END) AS status,
       'see TRIGGER_BODY for text'::text              AS description,
       pg_get_triggerdef(t.oid)                       AS trigger_body,
       "substring"(pg_get_triggerdef(t.oid), "position"("substring"(pg_get_triggerdef(t.oid), 48), 'EXECUTE PROCEDURE'::text) + 47) AS action_statement
FROM   pg_namespace n,
       pg_class c,
       pg_trigger t,
       ((SELECT 4, 'INSERT' UNION ALL SELECT 8, 'DELETE') UNION ALL SELECT 16, 'UPDATE') em(num, text)
WHERE  n.oid = c.relnamespace AND
       c.oid = t.tgrelid AND
       ((t.tgtype)::integer & em.num) <> 0 AND
       t.tgconstraint = 0 AND
       NOT pg_is_other_temp_schema(n.oid);

CREATE OR REPLACE VIEW sys.all_triggers AS
SELECT quote_ident_upper(pg_get_userbyid(c.relowner)) AS owner,
       quote_ident_upper(n.nspname)                   AS schema_name,
       quote_ident_upper(t.tgname)                    AS trigger_name,
       (CASE WHEN ((t.tgtype)::integer & 2) = 2 THEN 'BEFORE'::text  ELSE 'AFTER'::text END)
       ||' '|| (CASE WHEN ((t.tgtype)::integer & 1) = 1 THEN 'ROW'::text ELSE 'STATEMENT'::text END) AS trigger_type,
       ((em.text)::text)                              AS triggering_event,
       quote_ident_upper(pg_get_userbyid(c.relowner)) AS table_owner,
       'TABLE'::text                                  AS base_object_type,
       quote_ident_upper(c.relname)                   AS table_name,
       'REFERENCING NEW AS NEW OLD AS OLD'::text      AS referencing_names,
       (CASE WHEN tgenabled !='D' THEN 'ENABLED' ELSE 'DISABLED' END) AS status,
       'see TRIGGER_BODY for text'::text              AS description,
       pg_get_triggerdef(t.oid)                       AS trigger_body,
       "substring"(pg_get_triggerdef(t.oid), "position"("substring"(pg_get_triggerdef(t.oid), 48), 'EXECUTE PROCEDURE'::text) + 47) AS action_statement
FROM   pg_namespace n,
       pg_class c,
       pg_trigger t,
       ((SELECT 4, 'INSERT' UNION ALL SELECT 8, 'DELETE') UNION ALL SELECT 16, 'UPDATE') em(num, text)
WHERE  n.oid = c.relnamespace AND
       c.oid = t.tgrelid AND
       ((t.tgtype)::integer & em.num) <> 0 AND
       t.tgconstraint = 0 AND
       NOT pg_is_other_temp_schema(n.oid) AND
       has_table_privilege(c.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_triggers TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_triggers AS
SELECT schema_name, trigger_name, trigger_type, triggering_event, table_owner,
       base_object_type, table_name, referencing_names, status, description,
       trigger_body, action_statement
FROM   sys.all_triggers
WHERE  owner = quote_ident_upper(user);

GRANT SELECT ON sys.user_triggers TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_tab_columns views                              --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_tab_columns AS
SELECT quote_ident_upper(isc.owner)::character varying       AS owner,
       quote_ident_upper(isc.schema_name)::character varying AS schema_name,
       quote_ident_upper(isc.table_name)::character varying  AS table_name,
       quote_ident_upper(isc.column_name)::character varying AS column_name,
       DECODE(isc.data_type,
           'interval'                   , 'INTERVAL',
           'timestamp with time zone'   , 'TIMESTAMP WITH TIME ZONE',
           'time with time zone'        , 'TIME WITH TIME ZONE',
           'bigint'                     , 'BIGINT',
           'character varying'          , 'VARCHAR2',
           'boolean'                    , 'BOOLEAN',
           'smallint'                   , 'SMALLINT',
           'double precision'           , 'DOUBLE PRECISION',
           'integer'                    , 'INTEGER',
           'character'                  , 'CHAR',
           'timestamp without time zone', 'DATE',
           'time without time zone'     , 'TIME WITHOUT TIME ZONE',
           'real'                       , 'REAL',
           'numeric'                    , 'NUMBER',
           'bit'                        , 'BIT',
           'time'                       , 'TIME',
           'bit varying'                , 'BIT VARYING',
           'BYTEA'                      , 'BLOB',
           'TEXT'                       , 'CLOB',
           isc.data_type
       )::character varying                                  AS data_type,
       isc.data_length::numeric                              AS data_length,
       isc.data_precision::numeric                           AS data_precision,
       isc.data_scale::numeric                               AS data_scale,
       isc.nullable::character(1)                            AS nullable,
       isc.column_id::numeric                                AS column_id,
       isc.data_default::character varying                   AS data_default
FROM   (SELECT pg_get_userbyid(c.relowner)     AS owner,
             nc.nspname                        AS schema_name,
             c.relname                         AS table_name,
             a.attname                         AS column_name,
             CASE WHEN t.typtype = 'd'::"char" THEN
                  CASE WHEN bt.typelem <> 0::oid AND bt.typlen = (-1) THEN 'ARRAY'
                       ELSE CASE WHEN quote_ident(format_type(t.typbasetype, NULL::integer)) = format_type(t.typbasetype, NULL::integer) THEN quote_ident_upper(format_type(t.typbasetype, NULL::integer))
                                 ELSE format_type(t.typbasetype, NULL::integer)
                            END
                  END
                  ELSE CASE WHEN t.typelem <> 0::oid AND t.typlen = (-1) THEN 'ARRAY'
                            ELSE CASE WHEN quote_ident(format_type(a.atttypid, NULL::integer)) = format_type(a.atttypid, NULL::integer) THEN quote_ident_upper(format_type(a.atttypid, NULL::integer))
                                      ELSE format_type(a.atttypid, NULL::integer)
                                 END
                       END
             END                               AS data_type,
             information_schema._pg_char_max_length(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))   AS data_length,
             information_schema._pg_numeric_precision(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*)) AS data_precision,
             information_schema._pg_numeric_scale(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))     AS data_scale,
             a.attnum                          AS column_id,
             CASE WHEN a.attnotnull OR t.typtype = 'd'::"char" AND t.typnotnull THEN 'N'::character(1)
                  ELSE 'Y'::character(1)
             END                               AS nullable,
             pg_get_expr(ad.adbin, ad.adrelid) AS data_default
       FROM  pg_attribute a
             LEFT JOIN pg_attrdef ad ON a.attrelid = ad.adrelid AND a.attnum = ad.adnum
             JOIN (pg_class c JOIN pg_namespace nc ON c.relnamespace = nc.oid) ON a.attrelid = c.oid
             JOIN (pg_type t JOIN pg_namespace nt ON t.typnamespace = nt.oid) ON a.atttypid = t.oid
             LEFT JOIN (pg_type bt JOIN pg_namespace nbt ON bt.typnamespace = nbt.oid) ON t.typtype = 'd'::"char" AND t.typbasetype = bt.oid
             LEFT JOIN (pg_collation co JOIN pg_namespace nco ON co.collnamespace = nco.oid) ON a.attcollation = co.oid AND (nco.nspname <> 'pg_catalog'::name OR co.collname <> 'default'::name)
       WHERE NOT pg_is_other_temp_schema(nc.oid) AND
             a.attnum > 0 AND
             NOT a.attisdropped AND
             c.relkind = 'r'::"char"
       ) isc
ORDER BY 1, 2;

CREATE OR REPLACE VIEW sys.all_tab_columns AS
SELECT quote_ident_upper(isc.owner)::character varying       AS owner,
       quote_ident_upper(isc.schema_name)::character varying AS schema_name,
       quote_ident_upper(isc.table_name)::character varying  AS table_name,
       quote_ident_upper(isc.column_name)::character varying AS column_name,
       DECODE(isc.data_type,
           'interval'                   , 'INTERVAL',
           'timestamp with time zone'   , 'TIMESTAMP WITH TIME ZONE',
           'time with time zone'        , 'TIME WITH TIME ZONE',
           'bigint'                     , 'BIGINT',
           'character varying'          , 'VARCHAR2',
           'boolean'                    , 'BOOLEAN',
           'smallint'                   , 'SMALLINT',
           'double precision'           , 'DOUBLE PRECISION',
           'integer'                    , 'INTEGER',
           'character'                  , 'CHAR',
           'timestamp without time zone', 'DATE',
           'time without time zone'     , 'TIME WITHOUT TIME ZONE',
           'real'                       , 'REAL',
           'numeric'                    , 'NUMBER',
           'bit'                        , 'BIT',
           'time'                       , 'TIME',
           'bit varying'                , 'BIT VARYING',
           'BYTEA'                      , 'BLOB',
           'TEXT'                       , 'CLOB',
           isc.data_type
       )::character varying                                  AS data_type,
       isc.data_length::numeric                              AS data_length,
       isc.data_precision::numeric                           AS data_precision,
       isc.data_scale::numeric                               AS data_scale,
       isc.nullable::character(1)                            AS nullable,
       isc.column_id::numeric                                AS column_id,
       isc.data_default::character varying                   AS data_default
FROM   (SELECT pg_get_userbyid(c.relowner)     AS owner,
             nc.nspname                        AS schema_name,
             c.relname                         AS table_name,
             a.attname                         AS column_name,
             CASE WHEN t.typtype = 'd'::"char" THEN
                  CASE WHEN bt.typelem <> 0::oid AND bt.typlen = (-1) THEN 'ARRAY'
                       ELSE CASE WHEN quote_ident(format_type(t.typbasetype, NULL::integer)) = format_type(t.typbasetype, NULL::integer) THEN quote_ident_upper(format_type(t.typbasetype, NULL::integer))
                                 ELSE format_type(t.typbasetype, NULL::integer)
                            END
                  END
                  ELSE CASE WHEN t.typelem <> 0::oid AND t.typlen = (-1) THEN 'ARRAY'
                            ELSE CASE WHEN quote_ident(format_type(a.atttypid, NULL::integer)) = format_type(a.atttypid, NULL::integer) THEN quote_ident_upper(format_type(a.atttypid, NULL::integer))
                                      ELSE format_type(a.atttypid, NULL::integer)
                                 END
                       END
             END                               AS data_type,
             information_schema._pg_char_max_length(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))   AS data_length,
             information_schema._pg_numeric_precision(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*)) AS data_precision,
             information_schema._pg_numeric_scale(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))     AS data_scale,
             a.attnum                          AS column_id,
             CASE WHEN a.attnotnull OR t.typtype = 'd'::"char" AND t.typnotnull THEN 'N'::character(1)
                  ELSE 'Y'::character(1)
             END                               AS nullable,
             pg_get_expr(ad.adbin, ad.adrelid) AS data_default
       FROM  pg_attribute a
             LEFT JOIN pg_attrdef ad ON a.attrelid = ad.adrelid AND a.attnum = ad.adnum
             JOIN (pg_class c JOIN pg_namespace nc ON c.relnamespace = nc.oid) ON a.attrelid = c.oid
             JOIN (pg_type t JOIN pg_namespace nt ON t.typnamespace = nt.oid) ON a.atttypid = t.oid
             LEFT JOIN (pg_type bt JOIN pg_namespace nbt ON bt.typnamespace = nbt.oid) ON t.typtype = 'd'::"char" AND t.typbasetype = bt.oid
             LEFT JOIN (pg_collation co JOIN pg_namespace nco ON co.collnamespace = nco.oid) ON a.attcollation = co.oid AND (nco.nspname <> 'pg_catalog'::name OR co.collname <> 'default'::name)
       WHERE NOT pg_is_other_temp_schema(nc.oid) AND
             a.attnum > 0 AND
             NOT a.attisdropped AND
             c.relkind = 'r'::"char" AND
             has_column_privilege(c.oid, a.attnum, 'SELECT, INSERT, UPDATE, REFERENCES') AND
             nc.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
       ) isc
ORDER BY 1, 2;

GRANT SELECT ON sys.all_tab_columns TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_tab_columns AS
SELECT schema_name, table_name, column_name, data_type, data_length,
       data_precision, data_scale, nullable, column_id, data_default
FROM   sys.all_tab_columns
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_tab_columns TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_synonyms views                                 --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_synonyms AS
SELECT quote_ident_upper(pg_get_userbyid(synowner))   AS owner,
       quote_ident_upper(pnname.nspname)              As schema_name,
       quote_ident_upper(sy.synname)                  AS synonym_name,
       quote_ident_upper(pg_get_userbyid(c.relowner)) AS table_owner,
       quote_ident_upper(sy.synobjschema)             AS table_schema_name,
       quote_ident_upper(sy.synobjname)               AS table_name,
       quote_ident_upper(sy.synlink)                  AS db_link
FROM   pg_synonym sy
       LEFT OUTER JOIN pg_namespace pnoid ON pnoid.nspname=sy.synobjschema and pnoid.nspparent=0
       LEFT OUTER JOIN pg_class c ON c.relname=sy.synobjname AND c.relnamespace = pnoid.oid
       LEFT OUTER JOIN pg_namespace pnname ON pnname.oid=sy.synnamespace;

CREATE OR REPLACE VIEW sys.all_synonyms AS
SELECT quote_ident_upper(pg_get_userbyid(synowner))   AS owner,
       quote_ident_upper(pnname.nspname)              As schema_name,
       quote_ident_upper(sy.synname)                  AS synonym_name,
       quote_ident_upper(pg_get_userbyid(c.relowner)) AS table_owner,
       quote_ident_upper(sy.synobjschema)             AS table_schema_name,
       quote_ident_upper(sy.synobjname)               AS table_name,
       quote_ident_upper(sy.synlink)                  AS db_link
FROM   pg_synonym sy
       LEFT OUTER JOIN pg_namespace pnoid ON pnoid.nspname=sy.synobjschema and pnoid.nspparent=0
       LEFT OUTER JOIN pg_class c ON c.relname=sy.synobjname AND c.relnamespace = pnoid.oid
       LEFT OUTER JOIN pg_namespace pnname ON pnname.oid=sy.synnamespace
       -- We have no idea for which object synonym is created. So just have check over role
       -- However, public synonym resides in public schema and as per Redwood, all public synonyms
       -- are part of all_synonyms view. These public synonyms should be displayed irrespective of
       -- the owner and it's pointing object
WHERE  (pnname.nspname = 'public' OR pg_has_role(synowner, 'USAGE'::text)) AND
       pnname.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_synonyms TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_synonyms AS
SELECT schema_name, synonym_name, table_owner, table_schema_name, table_name, db_link
FROM   sys.all_synonyms
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_synonyms TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_views views                                    --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_views AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(n.nspname)                           AS schema_name,
       quote_ident_upper(c.relname)                           AS view_name,
       pg_get_viewdef(c.oid)                                  AS text
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
WHERE  c.relkind = 'v'::"char";

CREATE OR REPLACE VIEW sys.all_views AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(n.nspname)                           AS schema_name,
       quote_ident_upper(c.relname)                           AS view_name,
       pg_get_viewdef(c.oid)                                  AS text
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
WHERE  c.relkind = 'v'::"char" AND
       has_table_privilege(c.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_views TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_views AS
SELECT schema_name, view_name, text
FROM   sys.all_views
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_views TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_cons_columns views                             --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_cons_columns AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text)  AS owner,
       quote_ident_upper(pn.nspname)                           AS schema_name,
       quote_ident_upper(pc.conname)                           AS constraint_name,
       quote_ident_upper(c.relname)                            AS table_name,
       quote_ident_upper(pa.attname)                           AS column_name,
       pa.attnum                                               AS position,
       quote_ident_upper((pg_get_constraintdef(pc.oid))::text) AS constraint_def
FROM   pg_attribute pa JOIN pg_constraint pc ON pc.conrelid = pa.attrelid AND pa.attnum = any (pc.conkey)
       JOIN pg_namespace pn ON pn.oid =pc.connamespace
       LEFT OUTER JOIN pg_class c ON c.oid =  pc.conrelid;

CREATE OR REPLACE VIEW sys.all_cons_columns AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text)  AS owner,
       quote_ident_upper(pn.nspname)                           AS schema_name,
       quote_ident_upper(pc.conname)                           AS constraint_name,
       quote_ident_upper(c.relname)                            AS table_name,
       quote_ident_upper(pa.attname)                           AS column_name,
       pa.attnum                                               AS position,
       quote_ident_upper((pg_get_constraintdef(pc.oid))::text) AS constraint_def
FROM   pg_attribute pa JOIN pg_constraint pc ON pc.conrelid = pa.attrelid AND pa.attnum = any (pc.conkey)
       JOIN pg_namespace pn ON pn.oid = pc.connamespace
       LEFT OUTER JOIN pg_class c ON c.oid =  pc.conrelid
WHERE  has_column_privilege(c.oid, pa.attnum, 'SELECT, INSERT, UPDATE, REFERENCES') AND
       pn.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_cons_columns TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_cons_columns AS
SELECT owner, schema_name, constraint_name, table_name, column_name, position,
       constraint_def
FROM   sys.all_cons_columns
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_cons_columns TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_indexes views                                 --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_indexes AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(n.nspname) AS schema_name,
       quote_ident_upper(i.relname) AS index_name,
       'BTREE'::text                AS index_type,
       quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS table_owner,
       quote_ident_upper(c.relname) AS table_name,
       'TABLE'::text                AS table_type,
       (CASE WHEN x.indisunique ='t' THEN 'UNIQUE' ELSE 'NONUNIQUE' END) AS uniqueness,
       'N'::char(1)                 AS compression,
       quote_ident_upper(t.spcname) AS tablespace_name,
       'LOGGING'::text              AS logging,
       (CASE WHEN x.indisvalid ='t' THEN 'VALID' ELSE 'INVALID' END) AS status,
       'NO'::char(3)                AS partitioned,
       (CASE WHEN n.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END)::char(1) AS temporary,
       'N'::char(1)                 AS secondary,
       'NO'::char(3)                AS join_index,
       'NO'::char(3)                AS dropped
FROM   pg_index x
       JOIN pg_class c ON c.oid = x.indrelid
       JOIN pg_class i ON i.oid = x.indexrelid
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = i.reltablespace
WHERE  c.relkind = 'r'::"char" AND i.relkind = 'i'::"char";

CREATE OR REPLACE VIEW sys.all_indexes AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS owner,
       quote_ident_upper(n.nspname) AS schema_name,
       quote_ident_upper(i.relname) AS index_name,
       'BTREE'::text                AS index_type,
       quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS table_owner,
       quote_ident_upper(c.relname) AS table_name,
       'TABLE'::text                AS table_type,
       (CASE WHEN x.indisunique ='t' THEN 'UNIQUE' ELSE 'NONUNIQUE' END) AS uniqueness,
       'N'::char(1)                 AS compression,
       quote_ident_upper(t.spcname) AS tablespace_name,
       'LOGGING'::text              AS logging,
       (CASE WHEN x.indisvalid ='t' THEN 'VALID' ELSE 'INVALID' END) AS status,
       'NO'::char(3)                AS partitioned,
       (CASE WHEN n.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END)::char(1) AS temporary,
       'N'::char(1)                 AS secondary,
       'NO'::char(3)                AS join_index,
       'NO'::char(3)                AS dropped
FROM   pg_index x
       JOIN pg_class c ON c.oid = x.indrelid
       JOIN pg_class i ON i.oid = x.indexrelid
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = i.reltablespace
WHERE  c.relkind = 'r'::"char" AND i.relkind = 'i'::"char" AND
       has_table_privilege(c.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_indexes TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_indexes AS
SELECT schema_name, index_name, index_type, table_owner, table_name, table_type,
       uniqueness, compression, tablespace_name, logging, status, partitioned,
       temporary, secondary, join_index, dropped
FROM   sys.all_indexes
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_indexes TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_ind_columns views                              --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_ind_columns AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text)  AS index_owner,
       quote_ident_upper(n.nspname) AS schema_name,
       quote_ident_upper(i.relname) AS index_name,
       quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS table_owner,
       quote_ident_upper(c.relname) AS table_name,
       quote_ident_upper(j.attname) AS column_name,
       j.attnum                     AS column_position,
       j.attlen                     AS column_length,
       (CASE WHEN j.atttypid =18 THEN j.attlen ELSE 0 END)::number AS char_length,
       'Y'::char                    AS descend
FROM   pg_index x
       JOIN pg_class c ON c.oid = x.indrelid
       JOIN pg_class i ON i.oid = x.indexrelid
       JOIN pg_attribute j ON j.attrelid=i.oid
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = i.reltablespace
WHERE  c.relkind = 'r'::"char" AND i.relkind = 'i'::"char"
ORDER BY i.relname, j.attnum;

CREATE OR REPLACE VIEW sys.all_ind_columns AS
SELECT quote_ident_upper((pg_get_userbyid(c.relowner))::text)  AS index_owner,
       quote_ident_upper(n.nspname) AS schema_name,
       quote_ident_upper(i.relname) AS index_name,
       quote_ident_upper((pg_get_userbyid(c.relowner))::text) AS table_owner,
       quote_ident_upper(c.relname) AS table_name,
       quote_ident_upper(j.attname) AS column_name,
       j.attnum                     AS column_position,
       j.attlen                     AS column_length,
       (CASE WHEN j.atttypid =18 THEN j.attlen ELSE 0 END)::number AS char_length,
       'Y'::char                    AS descend
FROM   pg_index x
       JOIN pg_class c ON c.oid = x.indrelid
       JOIN pg_class i ON i.oid = x.indexrelid
       JOIN pg_attribute j ON j.attrelid = i.oid
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
       LEFT JOIN pg_tablespace t ON t.oid = i.reltablespace
WHERE  c.relkind = 'r'::"char" AND i.relkind = 'i'::"char" AND
       has_column_privilege(c.oid, j.attnum, 'SELECT, INSERT, UPDATE, REFERENCES') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
ORDER BY i.relname, j.attnum;

GRANT SELECT ON sys.all_ind_columns TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_ind_columns AS
SELECT schema_name, index_name, table_name, column_name,
       column_position, column_length, char_length, descend
FROM   sys.all_ind_columns
WHERE  index_owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_ind_columns TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_view_columns views                             --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_view_columns AS
SELECT quote_ident_upper(isc.owner)::character varying       AS owner,
       quote_ident_upper(isc.schema_name)::character varying AS schema_name,
       quote_ident_upper(isc.table_name)::character varying  AS view_name,
       quote_ident_upper(isc.column_name)::character varying AS column_name,
       DECODE(isc.data_type,
           'interval'                   , 'INTERVAL',
           'timestamp with time zone'   , 'TIMESTAMP WITH TIME ZONE',
           'time with time zone'        , 'TIME WITH TIME ZONE',
           'bigint'                     , 'BIGINT',
           'character varying'          , 'VARCHAR2',
           'boolean'                    , 'BOOLEAN',
           'smallint'                   , 'SMALLINT',
           'double precision'           , 'DOUBLE PRECISION',
           'integer'                    , 'INTEGER',
           'character'                  , 'CHAR',
           'timestamp without time zone', 'DATE',
           'time without time zone'     , 'TIME WITHOUT TIME ZONE',
           'real'                       , 'REAL',
           'numeric'                    , 'NUMBER',
           'bit'                        , 'BIT',
           'time'                       , 'TIME',
           'bit varying'                , 'BIT VARYING',
           'BYTEA'                      , 'BLOB',
           'TEXT'                       , 'CLOB',
           isc.data_type
       )::character varying                                  AS data_type,
       isc.data_length::numeric                              AS data_length,
       isc.data_precision::numeric                           AS data_precision,
       isc.data_scale::numeric                               AS data_scale,
       isc.nullable::character(1)                            AS nullable,
       isc.column_id::numeric                                AS column_id,
       isc.data_default::character varying                   AS data_default
FROM   (SELECT pg_get_userbyid(c.relowner)     AS owner,
             nc.nspname                        AS schema_name,
             c.relname                         AS table_name,
             a.attname                         AS column_name,
             CASE WHEN t.typtype = 'd'::"char" THEN
                  CASE WHEN bt.typelem <> 0::oid AND bt.typlen = (-1) THEN 'ARRAY'
                       ELSE CASE WHEN quote_ident(format_type(t.typbasetype, NULL::integer)) = format_type(t.typbasetype, NULL::integer) THEN quote_ident_upper(format_type(t.typbasetype, NULL::integer))
                                 ELSE format_type(t.typbasetype, NULL::integer)
                            END
                  END
                  ELSE CASE WHEN t.typelem <> 0::oid AND t.typlen = (-1) THEN 'ARRAY'
                            ELSE CASE WHEN quote_ident(format_type(a.atttypid, NULL::integer)) = format_type(a.atttypid, NULL::integer) THEN quote_ident_upper(format_type(a.atttypid, NULL::integer))
                                      ELSE format_type(a.atttypid, NULL::integer)
                                 END
                       END
             END                               AS data_type,
             information_schema._pg_char_max_length(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))   AS data_length,
             information_schema._pg_numeric_precision(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*)) AS data_precision,
             information_schema._pg_numeric_scale(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))     AS data_scale,
             a.attnum                          AS column_id,
             CASE WHEN a.attnotnull OR t.typtype = 'd'::"char" AND t.typnotnull THEN 'N'::character(1)
                  ELSE 'Y'::character(1)
             END                               AS nullable,
             pg_get_expr(ad.adbin, ad.adrelid) AS data_default
       FROM  pg_attribute a
             LEFT JOIN pg_attrdef ad ON a.attrelid = ad.adrelid AND a.attnum = ad.adnum
             JOIN (pg_class c JOIN pg_namespace nc ON c.relnamespace = nc.oid) ON a.attrelid = c.oid
             JOIN (pg_type t JOIN pg_namespace nt ON t.typnamespace = nt.oid) ON a.atttypid = t.oid
             LEFT JOIN (pg_type bt JOIN pg_namespace nbt ON bt.typnamespace = nbt.oid) ON t.typtype = 'd'::"char" AND t.typbasetype = bt.oid
             LEFT JOIN (pg_collation co JOIN pg_namespace nco ON co.collnamespace = nco.oid) ON a.attcollation = co.oid AND (nco.nspname <> 'pg_catalog'::name OR co.collname <> 'default'::name)
       WHERE NOT pg_is_other_temp_schema(nc.oid) AND
             a.attnum > 0 AND
             NOT a.attisdropped AND
             c.relkind = 'v'::"char"
       ) isc
ORDER BY 1, 2;

CREATE OR REPLACE VIEW sys.all_view_columns AS
SELECT quote_ident_upper(isc.owner)::character varying       AS owner,
       quote_ident_upper(isc.schema_name)::character varying AS schema_name,
       quote_ident_upper(isc.table_name)::character varying  AS view_name,
       quote_ident_upper(isc.column_name)::character varying AS column_name,
       DECODE(isc.data_type,
           'interval'                   , 'INTERVAL',
           'timestamp with time zone'   , 'TIMESTAMP WITH TIME ZONE',
           'time with time zone'        , 'TIME WITH TIME ZONE',
           'bigint'                     , 'BIGINT',
           'character varying'          , 'VARCHAR2',
           'boolean'                    , 'BOOLEAN',
           'smallint'                   , 'SMALLINT',
           'double precision'           , 'DOUBLE PRECISION',
           'integer'                    , 'INTEGER',
           'character'                  , 'CHAR',
           'timestamp without time zone', 'DATE',
           'time without time zone'     , 'TIME WITHOUT TIME ZONE',
           'real'                       , 'REAL',
           'numeric'                    , 'NUMBER',
           'bit'                        , 'BIT',
           'time'                       , 'TIME',
           'bit varying'                , 'BIT VARYING',
           'BYTEA'                      , 'BLOB',
           'TEXT'                       , 'CLOB',
           isc.data_type
       )::character varying                                  AS data_type,
       isc.data_length::numeric                              AS data_length,
       isc.data_precision::numeric                           AS data_precision,
       isc.data_scale::numeric                               AS data_scale,
       isc.nullable::character(1)                            AS nullable,
       isc.column_id::numeric                                AS column_id,
       isc.data_default::character varying                   AS data_default
FROM   (SELECT pg_get_userbyid(c.relowner)     AS owner,
             nc.nspname                        AS schema_name,
             c.relname                         AS table_name,
             a.attname                         AS column_name,
             CASE WHEN t.typtype = 'd'::"char" THEN
                  CASE WHEN bt.typelem <> 0::oid AND bt.typlen = (-1) THEN 'ARRAY'
                       ELSE CASE WHEN quote_ident(format_type(t.typbasetype, NULL::integer)) = format_type(t.typbasetype, NULL::integer) THEN quote_ident_upper(format_type(t.typbasetype, NULL::integer))
                                 ELSE format_type(t.typbasetype, NULL::integer)
                            END
                  END
                  ELSE CASE WHEN t.typelem <> 0::oid AND t.typlen = (-1) THEN 'ARRAY'
                            ELSE CASE WHEN quote_ident(format_type(a.atttypid, NULL::integer)) = format_type(a.atttypid, NULL::integer) THEN quote_ident_upper(format_type(a.atttypid, NULL::integer))
                                      ELSE format_type(a.atttypid, NULL::integer)
                                 END
                       END
             END                               AS data_type,
             information_schema._pg_char_max_length(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))   AS data_length,
             information_schema._pg_numeric_precision(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*)) AS data_precision,
             information_schema._pg_numeric_scale(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))     AS data_scale,
             a.attnum                          AS column_id,
             CASE WHEN a.attnotnull OR t.typtype = 'd'::"char" AND t.typnotnull THEN 'N'::character(1)
                  ELSE 'Y'::character(1)
             END                               AS nullable,
             pg_get_expr(ad.adbin, ad.adrelid) AS data_default
       FROM  pg_attribute a
             LEFT JOIN pg_attrdef ad ON a.attrelid = ad.adrelid AND a.attnum = ad.adnum
             JOIN (pg_class c JOIN pg_namespace nc ON c.relnamespace = nc.oid) ON a.attrelid = c.oid
             JOIN (pg_type t JOIN pg_namespace nt ON t.typnamespace = nt.oid) ON a.atttypid = t.oid
             LEFT JOIN (pg_type bt JOIN pg_namespace nbt ON bt.typnamespace = nbt.oid) ON t.typtype = 'd'::"char" AND t.typbasetype = bt.oid
             LEFT JOIN (pg_collation co JOIN pg_namespace nco ON co.collnamespace = nco.oid) ON a.attcollation = co.oid AND (nco.nspname <> 'pg_catalog'::name OR co.collname <> 'default'::name)
       WHERE NOT pg_is_other_temp_schema(nc.oid) AND
             a.attnum > 0 AND
             NOT a.attisdropped AND
             c.relkind = 'v'::"char" AND
             has_column_privilege(c.oid, a.attnum, 'SELECT, INSERT, UPDATE, REFERENCES') AND
             nc.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
       ) isc
ORDER BY 1, 2;

GRANT SELECT ON sys.all_view_columns TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_view_columns AS
SELECT schema_name, view_name, column_name, data_type, data_length,
       data_precision, data_scale, nullable, column_id, data_default
FROM   sys.all_view_columns
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_view_columns TO PUBLIC;


--------------------------------------------------------------------------------
--                       edb package related views                            --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW edb_pkgelements AS
SELECT varpackage AS packageoid, varname AS eltname, varaccess AS visibilty,
       'V' AS eltclass, vartype AS eltdatatype,
       NULL AS nargs, NULL AS argtypes, NULL AS argmodes,
       NULL AS argnames, NULL AS argdefvals
FROM   edb_variable
UNION
SELECT pronamespace AS packageoid, proname AS eltname, proaccess AS visibility,
       DECODE(protype,'0','F','1','P' ) AS eltclass, prorettype AS eltdatatype,
       pronargs AS nargs, proargtypes AS argtypes, proargmodes AS argmodes,
       proargnames AS argnames, proargdefaults AS argdefaults
FROM   pg_proc
WHERE  pg_proc.pronamespace IN (SELECT oid FROM pg_namespace WHERE nspparent != 0);

GRANT SELECT ON edb_pkgelements TO PUBLIC;

CREATE OR REPLACE VIEW edb_package AS
SELECT oid, nspname AS pkgname, nspparent AS pkgnamespace, nspowner AS pkgowner,
       pg_catalog.edb_get_packageheaddef(oid) AS pkgheadsrc,
       pg_catalog.edb_get_packagebodydef(oid) AS pkgbodysrc,
       'P'::char AS pkgproperties, nspacl AS pkgacl, cmin, xmin, cmax, xmax, ctid
FROM   pg_namespace
WHERE  (nspparent != 0) AND (nspobjecttype = 0);

GRANT SELECT ON edb_package TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_types views                                    --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_types AS
SELECT quote_ident_upper(pg_get_userbyid(t.typowner)) AS owner,
       quote_ident_upper(n.nspname)                   AS schema_name,
       quote_ident_upper(t.typname)                   AS type_name,
       t.oid                                          AS type_oid,
       (CASE WHEN t.typtype = 'c'              THEN 'OBJECT'
             WHEN t.typtype IN ('A', 'N', 'V') THEN 'COLLECTION'
             ELSE 'OTHER' END)                        AS typecode,
       (CASE WHEN c.relkind = 'c'
             THEN c.relnatts ELSE 0 END)              AS attributes
FROM   pg_type t
       LEFT JOIN pg_namespace n ON n.oid = t.typnamespace
       LEFT JOIN pg_class c     ON t.typrelid = c.oid
WHERE  NOT (t.typelem != 0 AND t.typlen = -1 AND typarray = 0) AND  -- Exclude implicit array types
       (c.relkind IS NULL OR c.relkind = 'c' ) AND                  -- Exclude implicit table types
       n.nspname NOT IN (SELECT pkgname FROM edb_package);          -- On redwood package types are not displayed

CREATE OR REPLACE VIEW sys.all_types AS
SELECT quote_ident_upper(pg_get_userbyid(t.typowner)) AS owner,
       quote_ident_upper(n.nspname)                   AS schema_name,
       quote_ident_upper(t.typname)                   AS type_name,
       t.oid                                          AS type_oid,
       (CASE WHEN t.typtype = 'c'              THEN 'OBJECT'
             WHEN t.typtype IN ('A', 'N', 'V') THEN 'COLLECTION'
             ELSE 'OTHER' END)                        AS typecode,
       (CASE WHEN c.relkind = 'c'
             THEN c.relnatts ELSE 0 END)              AS attributes
FROM   pg_type t
       LEFT JOIN pg_namespace n ON n.oid = t.typnamespace
       LEFT JOIN pg_class c     ON t.typrelid = c.oid
WHERE  NOT (t.typelem != 0 AND t.typlen = -1 AND typarray = 0) AND  -- Exclude implicit array types
       (c.relkind IS NULL OR c.relkind = 'c' ) AND                  -- Exclude implicit table types
       n.nspname NOT IN (SELECT pkgname FROM edb_package) AND       -- On redwood package types are not displayed
       has_type_privilege(t.oid, 'USAGE') AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_types TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_types AS
SELECT schema_name, type_name, type_oid, typecode, attributes
FROM   sys.all_types
WHERE  owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_types TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_roles views                                    --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_roles AS
SELECT quote_ident_upper(groname) AS role,
       'N'::text                  AS password_required
FROM   pg_group;

-- sys.all_roles and sys.user_roles are not present on Oracle 10g


--------------------------------------------------------------------------------
--                       sys.*_role_privs views                               --
--------------------------------------------------------------------------------


CREATE OR REPLACE VIEW sys.dba_role_privs AS
SELECT quote_ident_upper(a.rolname)                       AS grantee,
       quote_ident_upper(b.rolname)                       AS granted_role,
       CASE WHEN am.admin_option THEN 'YES' ELSE 'NO' END AS admin_option,
       CASE WHEN a.rolinherit THEN 'YES' ELSE 'NO' END    AS default_role
FROM   pg_roles a
       INNER JOIN pg_auth_members am ON am.member = a.oid
       INNER JOIN pg_roles b ON b.oid = am.roleid
WHERE  a.rolcanlogin;

-- sys.all_role_privs is not present on Oracle 10g

CREATE OR REPLACE VIEW sys.user_role_privs AS
SELECT grantee AS username, granted_role, admin_option, default_role,
      'NO'::VARCHAR2(3) AS os_granted
FROM   sys.dba_role_privs
WHERE  grantee = quote_ident_upper(USER);

GRANT SELECT ON sys.user_role_privs TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_sequences views                                --
--------------------------------------------------------------------------------

-- Supporting function for sys.*_sequences views
CREATE OR REPLACE FUNCTION sys.showseq(seqoid OID, MIN_VALUE OUT number,
    MAX_VALUE OUT number, INCREMENT_BY OUT number, CYCLE_FLAG OUT varchar2(1),
    ORDER_FLAG OUT varchar2(1), CACHE_SIZE OUT number, LAST_NUMBER OUT number)
AS $$
BEGIN
    EXECUTE 'SELECT last_value , increment_by , max_value , min_value , cache_value , decode(is_cycled, FALSE, ''N'', TRUE, ''Y'') , ''Y'' FROM ' || seqoid::regclass::text INTO LAST_NUMBER, INCREMENT_BY, MAX_VALUE, MIN_VALUE, CACHE_SIZE, CYCLE_FLAG, ORDER_FLAG;
END;
$$ LANGUAGE PLPGSQL;

CREATE OR REPLACE VIEW sys.dba_sequences AS
SELECT quote_ident_upper(pg_get_userbyid(c.relowner)) AS sequence_owner,
       quote_ident_upper(n.nspname)                   AS schema_name,
       quote_ident_upper(c.relname)                   AS sequence_name,
       (showseq(c.oid::regclass)).*
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
WHERE  c.relkind = 'S';

CREATE OR REPLACE VIEW sys.all_sequences AS
SELECT quote_ident_upper(pg_get_userbyid(c.relowner)) AS sequence_owner,
       quote_ident_upper(n.nspname)                   AS schema_name,
       quote_ident_upper(c.relname)                   AS sequence_name,
       (showseq(c.oid::regclass)).*
FROM   pg_class c
       LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
WHERE  c.relkind = 'S' AND
       -- We cannot call has_sequence_privilege() as it will error out in case
       -- of non sequence oid. So call has_table_privilege() instead. But it
       -- don't have USAGE privilege, Thus check USAGE on role instead
       -- Previous implementation was like this
       (pg_has_role(c.relowner, 'USAGE') OR
       has_table_privilege(c.oid, 'SELECT, UPDATE')) AND
       n.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_sequences TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_sequences AS
SELECT schema_name, sequence_name, min_value, max_value, increment_by,
       cycle_flag, order_flag, cache_size, last_number
FROM   sys.all_sequences
WHERE  sequence_owner = quote_ident_upper(USER);

GRANT SELECT ON sys.user_sequences TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_db_links views                                 --
--------------------------------------------------------------------------------

-- edb_dblink is defined in system_views.sql, because it's needed in
-- PostgreSQL mode too, because pg_dump and other tools depend on it.

CREATE OR REPLACE VIEW sys.dba_db_links AS
SELECT CASE WHEN e.lnkispublic = true THEN 'PUBLIC'::text WHEN lnkispublic = false THEN quote_ident_upper(pg_get_userbyid(e.lnkowner)) END AS owner,
       upper(e.lnkname)                                AS db_link,
       decode(e.lnktype::text,'0','REDWOOD','1','EDB') AS type,
       quote_ident_upper(e.lnkuser)                    AS username,
       e.lnkconnstr                                    AS host
FROM   edb_dblink e;

CREATE OR REPLACE VIEW sys.all_db_links AS
SELECT CASE WHEN e.lnkispublic = true THEN 'PUBLIC'::text WHEN lnkispublic = false THEN quote_ident_upper(pg_get_userbyid(e.lnkowner)) END AS owner,
       upper(e.lnkname)                                AS db_link,
       decode(e.lnktype::text,'0','REDWOOD','1','EDB') AS type,
       quote_ident_upper(e.lnkuser)                    AS username,
       e.lnkconnstr                                    AS host
FROM   edb_dblink e
WHERE  e.lnkispublic = true OR
       has_server_privilege(e.oid, 'USAGE');

GRANT SELECT ON all_db_links TO PUBLIC;

-- We need to display password in this view, thus cannot user dba_db_links
CREATE OR REPLACE VIEW sys.user_db_links AS
SELECT upper(e.lnkname)                                AS db_link,
       decode(e.lnktype::text,'0','REDWOOD','1','EDB') AS type,
       quote_ident_upper(e.lnkuser)                    AS username,
       e.lnkpassword                                   AS password,
       e.lnkconnstr                                    AS host
FROM   edb_dblink e
WHERE  pg_get_userbyid(e.lnkowner) = user AND
       e.lnkispublic = false;

GRANT SELECT ON user_db_links to public;


--
--  pg_proc.protype enumerations
--
--  T_FUNC = '0',
--  T_PROC = '1' ,
--  T_TRIG = '2'
--

--------------------------------------------------------------------------------
--                       sys.*_objects views                                  --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_objects AS
SELECT owner,
       schema_name,
       index_name       AS object_name,
       'INDEX'          AS object_type,
       status::varchar2 AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_indexes
UNION ALL
SELECT sequence_owner   AS owner,
       schema_name,
       sequence_name    AS object_name,
       'SEQUENCE'       AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_sequences
UNION ALL
SELECT owner,
       schema_name,
       table_name       AS object_name,
       'TABLE'          AS object_type,
       status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_tables
UNION ALL
SELECT owner,
       schema_name,
       synonym_name     AS object_name,
       'SYNONYM'        AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_synonyms
UNION ALL
SELECT owner,
       schema_name,
       view_name        AS object_name,
       'VIEW'           AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_views
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.proowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.proname)                   AS object_name,
       'PROCEDURE'                                     AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_proc pr, pg_namespace ns ,pg_language ln , pg_type tp
WHERE  ns.oid = pr.pronamespace AND ln.oid = pr.prolang AND
       pr.pronamespace = ns.oid AND prorettype = tp.oid AND
       pr.protype = '1' AND ns.nspparent = 0 AND tp.typname <> 'trigger'
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.proowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.proname)                   AS object_name,
       'FUNCTION'                                      AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_proc pr, pg_namespace ns ,pg_language ln , pg_type tp
WHERE  ns.oid = pr.pronamespace AND ln.oid = pr.prolang AND
       pr.pronamespace = ns.oid AND prorettype = tp.oid AND
       pr.protype = '0' AND ns.nspparent = 0 AND tp.typname <> 'trigger'
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.pkgowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.pkgname)                   AS object_name,
       'PACKAGE'                                       AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   edb_package pr, pg_namespace ns
WHERE  ns.oid = pr.pkgnamespace AND pr.pkgnamespace = ns.oid
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.pkgowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.pkgname)                   AS object_name,
       'PACKAGE BODY'                                  AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   edb_package pr, pg_namespace ns
WHERE  ns.oid = pr.pkgnamespace AND pr.pkgnamespace = ns.oid AND
       pr.pkgbodysrc IS NOT NULL
UNION ALL
SELECT  quote_ident_upper(pg_get_userbyid(cl.relowner)) AS owner,
        quote_ident_upper(ns.nspname)                   AS schema_name,
        quote_ident_upper(tr.tgname)                    AS object_name,
       'TRIGGER'                                        AS object_type,
       'VALID'                                          AS status,
        (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_trigger tr ,pg_class cl, pg_namespace ns
WHERE  cl.oid = tr.tgrelid AND ns.oid  = cl.relnamespace AND
       quote_ident_upper(tr.tgname) NOT LIKE 'RI_CONSTRAINTTRIGGER_%'
UNION ALL
SELECT owner,
       schema_name,
       type_name        AS object_name,
       'TYPE'           AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_types
UNION ALL
SELECT owner,
       schema_name,
       type_name        AS object_name,
       'TYPE BODY'      AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.dba_types, pg_namespace
WHERE  type_oid = nspobjecttype AND nspbodysrc IS NOT NULL;

CREATE OR REPLACE VIEW sys.all_objects AS
SELECT owner,
       schema_name,
       index_name       AS object_name,
       'INDEX'          AS object_type,
       status::varchar2 AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_indexes
UNION ALL
SELECT sequence_owner   AS owner,
       schema_name,
       sequence_name    AS object_name,
       'SEQUENCE'       AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_sequences
UNION ALL
SELECT owner,
       schema_name,
       table_name       AS object_name,
       'TABLE'          AS object_type,
       status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_tables
UNION ALL
SELECT owner,
       schema_name,
       synonym_name     AS object_name,
       'SYNONYM'        AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_synonyms
UNION ALL
SELECT owner,
       schema_name,
       view_name        AS object_name,
       'VIEW'           AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_views
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.proowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.proname)                   AS object_name,
       'PROCEDURE'                                     AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_proc pr, pg_namespace ns ,pg_language ln , pg_type tp
WHERE  ns.oid = pr.pronamespace AND ln.oid = pr.prolang AND
       pr.pronamespace = ns.oid AND prorettype = tp.oid AND
       pr.protype = '1' AND ns.nspparent = 0 AND tp.typname <> 'trigger' AND
       has_function_privilege(pr.oid, 'EXECUTE') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.proowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.proname)                   AS object_name,
       'FUNCTION'                                      AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_proc pr, pg_namespace ns ,pg_language ln , pg_type tp
WHERE  ns.oid = pr.pronamespace AND ln.oid = pr.prolang AND
       pr.pronamespace = ns.oid AND prorettype = tp.oid AND
       pr.protype = '0' AND ns.nspparent = 0 AND tp.typname <> 'trigger' AND
       has_function_privilege(pr.oid, 'EXECUTE') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.pkgowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.pkgname)                   AS object_name,
       'PACKAGE'                                       AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   edb_package pr, pg_namespace ns
WHERE  ns.oid = pr.pkgnamespace AND
       has_schema_privilege(pr.oid, 'USAGE') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
UNION ALL
SELECT quote_ident_upper(pg_get_userbyid(pr.pkgowner)) AS owner,
       quote_ident_upper(ns.nspname)                   AS schema_name,
       quote_ident_upper(pr.pkgname)                   AS object_name,
       'PACKAGE BODY'                                  AS object_type,
       'VALID'                                         AS status,
       (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   edb_package pr, pg_namespace ns
WHERE  ns.oid = pr.pkgnamespace AND
       pr.pkgbodysrc IS NOT NULL AND
       has_schema_privilege(pr.oid, 'USAGE') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
UNION ALL
SELECT  quote_ident_upper(pg_get_userbyid(cl.relowner)) AS owner,
        quote_ident_upper(ns.nspname)                   AS schema_name,
        quote_ident_upper(tr.tgname)                    AS object_name,
       'TRIGGER'                                        AS object_type,
       'VALID'                                          AS status,
        (CASE WHEN ns.nspname like 'pg_temp_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   pg_trigger tr ,pg_class cl, pg_namespace ns
WHERE  cl.oid = tr.tgrelid AND ns.oid  = cl.relnamespace AND
       quote_ident_upper(tr.tgname) NOT LIKE 'RI_CONSTRAINTTRIGGER_%' AND
       has_table_privilege(cl.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
UNION ALL
SELECT owner,
       schema_name,
       type_name        AS object_name,
       'TYPE'           AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_types
UNION ALL
SELECT owner,
       schema_name,
       type_name        AS object_name,
       'TYPE BODY'      AS object_type,
       'VALID'          AS status,
       (CASE WHEN schema_name like 'PG_TEMP_%' THEN 'Y' ELSE 'N' END) AS temporary
FROM   sys.all_types, pg_namespace
WHERE  type_oid = nspobjecttype AND nspbodysrc IS NOT NULL;

GRANT SELECT ON sys.all_objects TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_objects AS
SELECT schema_name, object_name, object_type, status, temporary
FROM   sys.all_objects
WHERE  owner = quote_ident_upper(user);

GRANT SELECT ON sys.user_objects TO PUBLIC;


------------- Oracle compatible product information table ----------------------

CREATE TABLE SYS.PRODUCT_COMPONENT_VERSION
(
	PRODUCT    VARCHAR2(74),
	VERSION     VARCHAR2(74),
	STATUS       VARCHAR2(74)
);
		
INSERT INTO SYS.PRODUCT_COMPONENT_VERSION (PRODUCT, VERSION, STATUS) 
VALUES ('Oracle Database 10g Edition is similar', '10.2', 'Available');

INSERT INTO SYS.PRODUCT_COMPONENT_VERSION (PRODUCT, VERSION, STATUS) 
VALUES ('PL/SQL is similar', '10.2', 'Available');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'product_component_version'), 0, 'p');


/*
 * What strings should we print in "sys.v$version"? This is not just
 * for use by DBA's to check the version; there's the version() function
 * for that. This is for compatibility with applications that query
 * sys.v$version to determine whether feature X can be used. How exactly
 * a given application does that varies, but this will for example match
 * a query that does a LIKE '%Release 8%' query. Others might search for
 * '8i' for example, or something else, so we can't easily make everyone
 * happy. If nothing else helps, the superuser can drop and replace this
 * with whatever his/her application needs.
 */
CREATE VIEW sys.v$version as
SELECT	'Oracle Database 10g Edition Release 10.2.0.1.0 - Product is similar' AS banner
FROM pg_catalog.pg_settings WHERE name = 'server_version'
UNION ALL
SELECT	'PL/SQL Release 10.2.0.1.0 - Production is similar' AS banner
FROM pg_catalog.pg_settings WHERE name = 'server_version'
UNION ALL
SELECT	'CORE 10.2.0.1.0 Production is similar' AS banner
FROM pg_catalog.pg_settings WHERE name = 'server_version';

GRANT SELECT ON sys.v$version TO PUBLIC;

------------------------------------------------
--new_time: this function returns a date in US time zone1 to a date in
-- US time zone2.
------------------------------------------------

CREATE FUNCTION sys.new_time (datein timestamp, sourcezone varchar, destzone varchar) RETURNS timestamp  AS $$
DECLARE
	sourceoffset varchar;
	destoffset   varchar;
BEGIN
	sourceoffset := CASE sourcezone
		WHEN 'AST' THEN 'UTC+4'
		WHEN 'ADT' THEN 'UTC+3'
		WHEN 'BST' THEN 'UTC+11'
		WHEN 'BDT' THEN 'UTC+10'
		WHEN 'CST' THEN 'UTC+6'
		WHEN 'CDT' THEN 'UTC+5'
		WHEN 'EST' THEN 'UTC+5'
		WHEN 'EDT' THEN 'UTC+4'
		WHEN 'GMT' THEN 'UTC'
		WHEN 'HST' THEN 'UTC+10'
		WHEN 'HDT' THEN 'UTC+9'
		WHEN 'MST' THEN 'UTC+7'
		WHEN 'MDT' THEN 'UTC+6'
		WHEN 'NST' THEN 'UTC+3:30'
		WHEN 'PST' THEN 'UTC+8'
		WHEN 'PDT' THEN 'UTC+7'
		WHEN 'YST' THEN 'UTC+9'
		WHEN 'YDT' THEN 'UTC+8' END;

	IF sourceoffset IS NULL THEN
		RAISE EXCEPTION 'timezone % not recognized for new_time(), Use SQL standard AT TIME ZONE construct which supports all timezones', sourcezone;
	END IF;

	destoffset := CASE destzone
		WHEN 'AST' THEN 'UTC+4'
		WHEN 'ADT' THEN 'UTC+3'
		WHEN 'BST' THEN 'UTC+11'
		WHEN 'BDT' THEN 'UTC+10'
		WHEN 'CST' THEN 'UTC+6'
		WHEN 'CDT' THEN 'UTC+5'
		WHEN 'EST' THEN 'UTC+5'
		WHEN 'EDT' THEN 'UTC+4'
		WHEN 'GMT' THEN 'UTC'
		WHEN 'HST' THEN 'UTC+10'
		WHEN 'HDT' THEN 'UTC+9'
		WHEN 'MST' THEN 'UTC+7'
		WHEN 'MDT' THEN 'UTC+6'
		WHEN 'NST' THEN 'UTC+3:30'
		WHEN 'PST' THEN 'UTC+8'
		WHEN 'PDT' THEN 'UTC+7'
		WHEN 'YST' THEN 'UTC+9'
		WHEN 'YDT' THEN 'UTC+8' END;

	IF destoffset IS NULL THEN
		RAISE EXCEPTION 'timezone % not recognized for new_time(). Use SQL standard AT TIME ZONE construct which supports all timezones', destzone;
	END IF;

	RETURN ((datein AT TIME ZONE sourceoffset)  AT TIME ZONE destoffset);
END;
$$ LANGUAGE PLPGSQL STRICT IMMUTABLE;

------------------Create Redmond Views--------------------------------------------------------------

GRANT USAGE ON SCHEMA dbo TO PUBLIC;

CREATE VIEW dbo.sysusers AS
SELECT user_id AS uid, LOWER(username) AS name
  FROM sys.all_users;

GRANT SELECT ON dbo.sysusers TO PUBLIC;

CREATE VIEW dbo.sysobjects AS
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'U' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type = 'TABLE'
UNION
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'I' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type = 'INDEX'
UNION
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'S' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type = 'SEQUENCE'
UNION
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'V' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type = 'VIEW'
UNION
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'P' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type IN ('PROCEDURE', 'FUNCTION', 'PACKAGE')
UNION
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'TR' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type = 'TRIGGER'
UNION
SELECT su.uid, LOWER(ao.schema_name) || '.' || LOWER(ao.object_name) AS name
     , 'M' AS xtype 
  FROM sys.all_objects ao, dbo.sysusers su 
 WHERE LOWER(ao.owner) = su.name 
   AND ao.object_type = 'SYNONYM';

GRANT SELECT ON dbo.sysobjects TO PUBLIC;

CREATE VIEW dbo.systypes AS
SELECT typname as name, typlen AS length 
  FROM pg_type
 WHERE typrelid = 0 and typelem = 0 and typnamespace = 11;

GRANT SELECT ON dbo.systypes TO PUBLIC;

CREATE VIEW dbo.sysindexes AS
SELECT LOWER(schema_name) || '.' || LOWER(object_name) AS name 
  FROM sys.all_objects
 WHERE object_type = 'INDEX';

GRANT SELECT ON dbo.sysindexes TO PUBLIC;

CREATE VIEW dbo.systables AS
SELECT LOWER(schema_name) || '.' || LOWER(object_name) AS name 
  FROM sys.all_objects
 WHERE object_type = 'TABLE';

GRANT SELECT ON dbo.systables TO PUBLIC;


CREATE OR REPLACE FUNCTION empty_blob() RETURNS blob as 'SELECT ''''::blob;' LANGUAGE sql;
CREATE OR REPLACE FUNCTION empty_clob() RETURNS clob as 'SELECT ''''::clob;' LANGUAGE sql;

-- /////////////////////////////////////////////////////////

--Do not remove entry for dual table
CREATE table sys.dual (DUMMY VARCHAR2(1));
GRANT SELECT ON sys.dual TO PUBLIC;

insert into sys.dual select 'X';

insert into pg_depend values (0,0,0,
							(select oid from pg_class where oid = 'pg_catalog.pg_namespace'::regclass),
							(select oid from pg_namespace where nspname='sys'),
							0,
							'p');

-- disallow removal of system views

insert into pg_depend values (0,0,0,
	(select oid from pg_class where relname = 'pg_class'),
	(select oid from pg_class where relname = 'dual'), 0, 'p');


--- Redwood ---

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_all_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_cons_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_constraints'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_db_links'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_ind_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_indexes'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_sequences'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_synonyms'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_tab_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_triggers'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_types'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_users'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_view_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_views'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_all_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_cons_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_constraints'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_db_links'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_ind_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_indexes'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_role_privs'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_roles'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_sequences'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_synonyms'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_tab_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_triggers'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_types'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_users'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_view_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_views'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_all_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_cons_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_constraints'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_db_links'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_ind_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_indexes'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_role_privs'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_sequences'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_synonyms'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_tab_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_triggers'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_types'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_users'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_view_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_views'), 0, 'p');


--- Redmond ---

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'sysusers'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'sysobjects'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'systypes'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'sysindexes'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'systables'), 0, 'p');

----------------------- Function to convert args direction tooid vector ( for backward compatibility )-------------------
CREATE or replace function ConvertArgDir ( x _char, length smallint ) returns oidvector as $$

declare 
	y oidvector;
	i integer;
	str1 varchar = '';
	str2 varchar = '';
begin
	if x is not null then 
		i := 0;
		y := '';
	loop 
		if i <= length then 
			select case x[i] when 'i' then '1 ' 
					 when 'o' then '2 '  
					 when 'b' then '3 ' 
					 else '' 
				end
			into str2;
		
			str1 = str1 || str2;
			i := i + 1;
		else
			exit;
		end if;
	end loop;
end if;
	y := str1;
	return y;
end; $$ 
language 'plpgsql';

--------------------------------------------------------------------------------------------------------------------------------
-----------------------------raise_application_error function-------------------------

CREATE OR REPLACE PROCEDURE raise_application_error(code numeric, msg text) AS
BEGIN
  IF code IS NULL THEN
    RAISE 'null value for sql code not allowed' USING ERRCODE='invalid_parameter_value';
  END IF;

/*
 * Oracle limits the range of error codes like to -20999 to -20000, but for
 * historical reasons we accept a wider range. We accept all 5 digits positive
 * and negative integers.
 */
  IF code NOT BETWEEN -20999 AND -20000 THEN
    RAISE 'error code is out of range' USING DETAIL = 'Valid range is -20999 to -20000', ERRCODE='invalid_parameter_value';
  END IF;
  RAISE 'EDB%: %', code::integer, COALESCE(msg, '<NULL>') USING ERRCODE = 'raise_exception', SQLCODE = code::integer;
END;

------------------- bool -> smallint and smallint to bool casts ------------------------

CREATE OR REPLACE FUNCTION bool_to_smallint ( BOOLEAN ) RETURNS SMALLINT AS
' SELECT CASE WHEN $1 IS TRUE THEN 1::SMALLINT ELSE 0::SMALLINT END; '
LANGUAGE sql;
CREATE CAST (BOOLEAN AS SMALLINT) WITH FUNCTION bool_to_smallint(BOOLEAN) AS implicit;

CREATE OR REPLACE FUNCTION smallint_to_bool ( SMALLINT ) RETURNS BOOLEAN AS
' SELECT CASE WHEN $1 <> 0 THEN TRUE ELSE FALSE END; '
LANGUAGE sql;
CREATE CAST (SMALLINT AS BOOLEAN) WITH FUNCTION smallint_to_bool(SMALLINT) AS implicit;
-----------------------------pg_procedure view-------------------------

CREATE VIEW pg_procedure AS
SELECT
	pg_proc.tableoid,
	pg_proc.oid,
	pg_proc.proname,
	pg_proc.pronamespace,
	pg_proc.proowner,
	pg_proc.prolang,
	pg_proc.proisagg,
	pg_proc.prosecdef,
	pg_proc.proisstrict,
	pg_proc.proretset,
	pg_proc.provolatile,
	pg_proc.pronargs,
	pg_proc.prorettype,
	pg_proc.proargtypes,
	pg_proc.proallargtypes,
	pg_proc.proargmodes as proargdirs,
	pg_proc.proargnames,
	pg_proc.prosrc,
	pg_proc.probin,
	pg_proc.proacl
FROM
	pg_proc
WHERE
	protype='1';
 
GRANT SELECT ON pg_procedure TO PUBLIC;

-- -----------------------------pg_function view-------------------------
-- 
CREATE VIEW pg_function AS
SELECT
	pg_proc.tableoid,
	pg_proc.oid,
	pg_proc.proname AS funname,
	pg_proc.pronamespace AS funnamespace,
	pg_proc.proowner AS funowner,
	pg_proc.prolang AS funlang,
	pg_proc.proisagg AS funisagg,
	pg_proc.prosecdef AS funsecdef,
	pg_proc.proisstrict AS funisstrict,
	pg_proc.proretset AS funretset,
	pg_proc.provolatile AS funvolatile,
	pg_proc.pronargs AS funnargs,
	pg_proc.prorettype AS funrettype,
	pg_proc.proargtypes AS funargtypes,
	pg_proc.proallargtypes AS funallargtypes ,
	ConvertArgDir(pg_proc.proargmodes,pg_proc.pronargs) AS funargdirs,
	pg_proc.proargnames AS funargnames ,
	pg_proc.prosrc AS funsrc,
	pg_proc.probin AS funbin,
	pg_proc.proacl AS funacl
FROM
	pg_proc
WHERE
	protype = '0';

GRANT SELECT ON pg_function TO PUBLIC;

SET SEARCH_PATH = PG_CATALOG;

/* CONNECT BY support function */
CREATE OR REPLACE FUNCTION pg_catalog.connectby_cyclecheck(path anyarray, node anyelement)
 RETURNS boolean AS $$
BEGIN
  IF node = ANY (path) THEN
    RAISE 'CONNECT BY loop in user data.';
  END IF;
  RETURN TRUE;
END;
$$ LANGUAGE 'plpgsql' IMMUTABLE;

CREATE OPERATOR ** (
	    PROCEDURE=numeric_power,
    	LEFTARG=numeric,
	    RIGHTARG=numeric);

CREATE OPERATOR ** (
    	PROCEDURE=dpow,
	    LEFTARG=double precision,
	    RIGHTARG=double precision);

/* Oracle's SYS_EXTRACT_UTC() function */
create or replace function sys.SYS_EXTRACT_UTC(timestamptz)
returns timestamp as $$
    select $1 at time zone 'UTC'
$$ language SQL;

/* On Redwood, int / int is not integer division like on PostgreSQL */
CREATE OR REPLACE FUNCTION sys.int4div(integer, integer)
RETURNS numeric
AS $$select $1/$2::numeric;$$
LANGUAGE SQL IMMUTABLE;

CREATE OPERATOR sys./ (PROCEDURE=sys.int4div, leftarg=INT4, rightarg=INT4);

COMMENT ON FUNCTION sys.int4div(integer, integer) IS 'Redwood-style int/int divison, returning a decimal number';

--------------------------------------------------------------------------------
--                           DBMS_RLS interface
--------------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE SYS.DBMS_RLS AUTHID CURRENT_USER AS

  STATIC				   CONSTANT INTEGER := 1;
  SHARED_STATIC			   CONSTANT INTEGER := 2;
  CONTEXT_SENSITIVE		   CONSTANT INTEGER := 3;
  SHARED_CONTEXT_SENSITIVE CONSTANT INTEGER := 4;
  DYNAMIC                  CONSTANT INTEGER := 5;

  ALL_ROWS				   CONSTANT INTEGER := 1;

  PROCEDURE add_policy(object_schema   IN       TEXT      := NULL,
                       object_name     IN       TEXT,
                       policy_name     IN       TEXT,
                       function_schema IN       TEXT      := NULL,
                       policy_function IN       TEXT,
                       statement_types IN       TEXT      := 'insert,update,delete,select',
                       update_check    IN       BOOLEAN   := FALSE,
                       enable          IN       BOOLEAN   := TRUE,
                       static_policy   IN       BOOLEAN   := FALSE,
                       policy_type     IN       INTEGER   := NULL,
                       long_predicate           BOOLEAN   := FALSE,
                       sec_relevant_cols IN     TEXT      := NULL,
                       sec_relevant_cols_opt IN INTEGER   := NULL);
  
  PROCEDURE drop_policy(object_schema  IN       TEXT      := NULL,
                        object_name    IN       TEXT,
                        policy_name    IN       TEXT    ); 

   PROCEDURE enable_policy(object_schema  IN       TEXT    := NULL,
                           object_name    IN       TEXT,
                           policy_name    IN       TEXT,
						   enable         IN       BOOLEAN);
END DBMS_RLS;

--------------------------------------------------------------------------------
--                           DBMS_RLS implementation                          --
--------------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE BODY SYS.DBMS_RLS AS
 
  ----------------------------------------------------------------------
  PROCEDURE add_policy(object_schema   IN       TEXT      := NULL,
                       object_name     IN       TEXT,
                       policy_name     IN       TEXT,
                       function_schema IN       TEXT      := NULL,
                       policy_function IN       TEXT,
                       statement_types IN       TEXT      := 'insert,update,delete,select',
                       update_check    IN       BOOLEAN   := FALSE,
                       enable          IN       BOOLEAN   := TRUE,
                       static_policy   IN       BOOLEAN   := FALSE,
                       policy_type     IN       INTEGER   := NULL,
                       long_predicate           BOOLEAN   := FALSE,
                       sec_relevant_cols IN     TEXT      := NULL,
                       sec_relevant_cols_opt IN INTEGER   := NULL) IS
   BEGIN
     PERFORM pg_catalog._add_policy(object_schema, object_name, 
                           policy_name, 
                           function_schema, policy_function,
                           statement_types,
                           update_check,
                           enable,
						   sec_relevant_cols_opt, sec_relevant_cols);
                           
   END add_policy;

  ----------------------------------------------------------------------
  PROCEDURE drop_policy(object_schema  IN       TEXT      := NULL,
                        object_name    IN       TEXT,
                        policy_name    IN       TEXT    ) IS
   BEGIN
     PERFORM pg_catalog._drop_policy(object_schema, object_name, policy_name);
   END drop_policy;

  ----------------------------------------------------------------------
   PROCEDURE enable_policy(object_schema  IN       TEXT      := NULL,
                           object_name    IN       TEXT,
                           policy_name    IN       TEXT,
						   enable         IN       BOOLEAN) IS
   BEGIN
     PERFORM pg_catalog._toggle_policy(object_schema, object_name, policy_name, enable);
   END enable_policy;

END DBMS_RLS;

--------------------------------------------------------------------------------
--                           SYS_CONTEXT implementation                       --
--                                                                            --
-- NOTE: this implementation is incomplete, but still useful in many          --
--       applications.  We do not yet support user-defined application        --
--       contexts, only the USERENV context (and then we only support a       --
--       few USERENV attributes)                                              --
--------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION pg_catalog.sys_context(namespace varchar2, attribute varchar2)
  RETURN varchar2 
  AUTHID CURRENT_USER AS
BEGIN

  IF UPPER(namespace) = 'USERENV' THEN
    CASE UPPER(attribute) 
	  WHEN 'SESSION_USER' 
	    THEN RETURN pg_catalog.session_user;
      WHEN 'CURRENT_USER' 
	    THEN RETURN pg_catalog.current_user;
      WHEN 'CURRENT_SCHEMA'
	    THEN RETURN pg_catalog.current_schema;
      WHEN 'HOST'
	    THEN RETURN pg_catalog.inet_client_addr;
      WHEN 'IP_ADDRESS'
	    THEN RETURN pg_catalog.inet_client_addr;
	  WHEN 'SERVER_HOST'
	    THEN RETURN pg_catalog.inet_server_addr;
      ELSE 
	    RAISE 'invalid USERENV parameter';
    END CASE;
  ELSE
    RETURN NULL;
  END IF;

END;


--------------------------------------------------------------------------------
--                           sys.*_policies views                             --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_policies AS
SELECT CASE p.policykind
            WHEN 'r' THEN pg_get_userbyid(objclass.relowner)
            WHEN 'v' THEN pg_get_userbyid(objclass.relowner)
            WHEN '=' THEN pg_get_userbyid(objsyn.synowner)
       END                                                         AS object_owner,
       CASE p.policykind
            WHEN 'r' THEN objclassschema.nspname
            WHEN 'v' THEN objclassschema.nspname
            WHEN '=' THEN objsynschema.nspname
       END                                                         AS schema_name,
       CASE p.policykind
            WHEN 'r' THEN objclass.relname
            WHEN 'v' THEN objclass.relname
            WHEN '=' THEN objsyn.synname
       END                                                         AS object_name,
       ''::text                                                    AS policy_group,
       policyname                                                  AS policy_name,
       CASE WHEN proschema.nspparent = 0 THEN proschema.nspname
            ELSE pkgschema.nspname END                             AS pf_owner,
       CASE WHEN proschema.nspparent = 0 THEN NULL
            ELSE proschema.nspname END                             AS package,
       proc.proname                                                AS function,
       CASE p.policyselect      WHEN true THEN 'YES' ELSE 'NO' END AS sel,
       CASE p.policyinsert      WHEN true THEN 'YES' ELSE 'NO' END AS ins,
       CASE p.policyupdate      WHEN true THEN 'YES' ELSE 'NO' END AS upd,
       CASE p.policydelete      WHEN true THEN 'YES' ELSE 'NO' END AS del,
       CASE p.policyindex       WHEN true THEN 'YES' ELSE 'NO' END AS idx,
       CASE p.policyupdatecheck WHEN true THEN 'YES' ELSE 'NO' END AS chk_option,
       CASE p.policyenabled     WHEN true THEN 'YES' ELSE 'NO' END AS enable,
       CASE p.policystatic      WHEN true THEN 'YES' ELSE 'NO' END AS static_policy,
       CASE p.policytype        -- Note must match DBMS_RLS constants
            WHEN 1 THEN 'STATIC'
            WHEN 2 THEN 'SHARED_STATIC'
            WHEN 3 THEN 'CONTEXT_SENSITIVE'
            WHEN 4 THEN 'SHARED_CONTEXT_SENSITIVE'
            WHEN 5 THEN 'DYNAMIC'
            ELSE 'UNKNOWN'
       END                                                         AS policy_type,
       'YES'::TEXT                                                 AS long_predicate
FROM   pg_catalog.edb_policy p                                                                                  -- policies
       JOIN pg_catalog.pg_proc proc ON (proc.oid = p.policyproc)                                                    -- policy function
       JOIN pg_catalog.pg_namespace proschema ON (proschema.oid = proc.pronamespace)                                -- schema for policy function
       LEFT OUTER JOIN pg_catalog.pg_namespace pkgschema ON (pkgschema.oid = proschema.nspparent)                   -- package for policy function
       LEFT OUTER JOIN pg_catalog.pg_class objclass ON (objclass.oid = p.policyobject)                              -- pg_class for object (views, tables)
       LEFT OUTER JOIN pg_catalog.pg_synonym objsyn ON (objsyn.oid = p.policyobject)                                -- pg_synonym for object (synonyms)
       LEFT OUTER JOIN pg_catalog.pg_namespace objclassschema ON (objclassschema.oid = objclass.relnamespace)       -- schema for object class (views, tables)
       LEFT OUTER JOIN pg_catalog.pg_namespace objsynschema ON (objsynschema.oid = objsyn.synnamespace);            -- schema for object synonym (synonyms)

CREATE OR REPLACE VIEW sys.all_policies AS
SELECT object_owner, schema_name, object_name, policy_group, policy_name, pf_owner, package, function,
       sel, ins, upd, del, idx, chk_option, enable, static_policy, policy_type,
       long_predicate
FROM   (SELECT CASE p.policykind
                    WHEN 'r' THEN pg_get_userbyid(objclass.relowner)
                    WHEN 'v' THEN pg_get_userbyid(objclass.relowner)
                    WHEN '=' THEN pg_get_userbyid(objsyn.synowner)
               END                                                         AS object_owner,
               CASE p.policykind
                    WHEN 'r' THEN objclassschema.nspname
                    WHEN 'v' THEN objclassschema.nspname
                    WHEN '=' THEN objsynschema.nspname
               END                                                         AS schema_name,
               CASE p.policykind
                    WHEN 'r' THEN objclass.relname
                    WHEN 'v' THEN objclass.relname
                    WHEN '=' THEN objsyn.synname
               END                                                         AS object_name,
               CASE p.policykind
                    WHEN 'r' THEN has_table_privilege(objclass.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER')
                    WHEN 'v' THEN has_table_privilege(objclass.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER')
                    WHEN '=' THEN (objsynschema.nspname = 'public' OR pg_has_role(objsyn.synowner, 'USAGE'::text))
               END                                                         AS can_access,
               ''::text                                                    AS policy_group,
               policyname                                                  AS policy_name,
               CASE WHEN proschema.nspparent = 0 THEN proschema.nspname
                    ELSE pkgschema.nspname END                             AS pf_owner,
               CASE WHEN proschema.nspparent = 0 THEN NULL
                    ELSE proschema.nspname END                             AS package,
               proc.proname                                                AS function,
               CASE p.policyselect      WHEN true THEN 'YES' ELSE 'NO' END AS sel,
               CASE p.policyinsert      WHEN true THEN 'YES' ELSE 'NO' END AS ins,
               CASE p.policyupdate      WHEN true THEN 'YES' ELSE 'NO' END AS upd,
               CASE p.policydelete      WHEN true THEN 'YES' ELSE 'NO' END AS del,
               CASE p.policyindex       WHEN true THEN 'YES' ELSE 'NO' END AS idx,
               CASE p.policyupdatecheck WHEN true THEN 'YES' ELSE 'NO' END AS chk_option,
               CASE p.policyenabled     WHEN true THEN 'YES' ELSE 'NO' END AS enable,
               CASE p.policystatic      WHEN true THEN 'YES' ELSE 'NO' END AS static_policy,
               CASE p.policytype        -- Note must match DBMS_RLS constants
                    WHEN 1 THEN 'STATIC'
                    WHEN 2 THEN 'SHARED_STATIC'
                    WHEN 3 THEN 'CONTEXT_SENSITIVE'
                    WHEN 4 THEN 'SHARED_CONTEXT_SENSITIVE'
                    WHEN 5 THEN 'DYNAMIC'
                    ELSE 'UNKNOWN'
               END                                                         AS policy_type,
               'YES'::TEXT                                                 AS long_predicate
        FROM   pg_catalog.edb_policy p                                                                                  -- policies
               JOIN pg_catalog.pg_proc proc ON (proc.oid = p.policyproc)                                                    -- policy function
               JOIN pg_catalog.pg_namespace proschema ON (proschema.oid = proc.pronamespace)                                -- schema for policy function
               LEFT OUTER JOIN pg_catalog.pg_namespace pkgschema ON (pkgschema.oid = proschema.nspparent)                   -- package for policy function
               LEFT OUTER JOIN pg_catalog.pg_class objclass ON (objclass.oid = p.policyobject)                              -- pg_class for object (views, tables)
               LEFT OUTER JOIN pg_catalog.pg_synonym objsyn ON (objsyn.oid = p.policyobject)                                -- pg_synonym for object (synonyms)
               LEFT OUTER JOIN pg_catalog.pg_namespace objclassschema ON (objclassschema.oid = objclass.relnamespace)       -- schema for object class (views, tables)
               LEFT OUTER JOIN pg_catalog.pg_namespace objsynschema ON (objsynschema.oid = objsyn.synnamespace)             -- schema for object synonym (synonyms)
       ) ep
WHERE  ep.can_access = true AND
       ep.schema_name NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_policies TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_policies AS
SELECT schema_name, object_name, policy_group, policy_name, pf_owner, package, function,
       sel, ins, upd, del, idx, chk_option, enable, static_policy, policy_type,
       long_predicate
FROM   sys.all_policies
WHERE  object_owner = user;

GRANT SELECT ON sys.user_policies TO PUBLIC;


-- Redwood-compatibility functions for converting between hex and raw
CREATE OR REPLACE FUNCTION sys.hextoraw(text) RETURNS bytea AS
$$
  -- pg_catalog.decode() expects the string to have an even number of
  -- chars, but Redwood hextoraw() does not. Pad the input with a
  -- leading zero if necessary.
  SELECT pg_catalog.decode(
    CASE WHEN pg_catalog.length($1) % 2 = 1 THEN ('0' || $1) ELSE $1 END,
    'hex')
$$ LANGUAGE SQL STRICT IMMUTABLE;

COMMENT ON FUNCTION sys.hextoraw(text)
	IS 'Convert hex-encoded text string to a bytea value';

CREATE OR REPLACE FUNCTION sys.rawtohex(bytea) RETURNS text AS
$$
  SELECT pg_catalog.encode($1, 'hex')
$$ LANGUAGE SQL STRICT IMMUTABLE;

COMMENT ON FUNCTION sys.rawtohex(bytea)
	IS 'Convert a bytea value into hex-encoded text string';

--------------------------------------------------------------------------------
--                          PARTITIONING views                                --
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
--                          sys.*_part_tables views                           --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_part_tables AS
SELECT pg_get_userbyid(class.relowner)            AS owner,
       ns.nspname                                 AS schema_name,
       class.relname                              AS table_name,
       CASE pd.pdeftype
            WHEN 'r' THEN 'RANGE'
            WHEN 'l' THEN 'LIST'
            ELSE 'UNKNOWN'
       END                                        AS partitioning_type,
       CASE pd.pdefsubtype
            WHEN 'r' THEN 'RANGE'
            WHEN 'l' THEN 'LIST'
            WHEN '?' THEN 'NONE'
            ELSE 'UNKNOWN'
       END                                        AS subpartitioning_type,
       (SELECT count(*)
        FROM   pg_catalog.edb_partition
        WHERE  partpdefid = pd.oid)               AS partition_count,
       0                                          AS def_subpartition_count,
       pg_catalog.array_length(pd.pdefcols, 1)    AS partitioning_key_count,
       pg_catalog.array_length(pd.pdefsubcols, 1) AS subpartitioning_key_count,
       'VALID'::VARCHAR2(8)                       AS status,
       NULL::VARCHAR2(30)                         AS def_tablespace_name,
       NULL::NUMBER                               AS def_pct_free,
       NULL::NUMBER                               AS def_pct_used,
       NULL::NUMBER                               AS def_ini_trans,
       NULL::NUMBER                               AS def_max_trans,
       NULL::VARCHAR2(40)                         AS def_initial_extent,
       NULL::VARCHAR2(40)                         AS def_next_extent,
       NULL::VARCHAR2(40)                         AS def_min_extents,
       NULL::VARCHAR2(40)                         AS def_max_extents,
       NULL::VARCHAR2(40)                         AS def_pct_increase,
       NULL::NUMBER                               AS def_freelists,
       NULL::NUMBER                               AS def_freelist_groups,
       'YES'::VARCHAR2(7)                         AS def_logging,
       'NONE'::VARCHAR2(8)                        AS def_compression,
       'DEFAULT'::VARCHAR2(7)                     AS def_buffer_pool,
       NULL::VARCHAR2(30)                         AS ref_ptn_constraint_name,
       NULL::VARCHAR2(1000)                       AS interval
FROM   pg_catalog.edb_partdef pd
       JOIN pg_catalog.pg_class class ON(class.oid = pd.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON(ns.oid = class.relnamespace);

CREATE OR REPLACE VIEW sys.all_part_tables AS
SELECT pg_get_userbyid(class.relowner)            AS owner,
       ns.nspname                                 AS schema_name,
       class.relname                              AS table_name,
       CASE pd.pdeftype
            WHEN 'r' THEN 'RANGE'
            WHEN 'l' THEN 'LIST'
            ELSE 'UNKNOWN'
       END                                        AS partitioning_type,
       CASE pd.pdefsubtype
            WHEN 'r' THEN 'RANGE'
            WHEN 'l' THEN 'LIST'
            WHEN '?' THEN 'NONE'
            ELSE 'UNKNOWN'
       END                                        AS subpartitioning_type,
       (SELECT count(*)
        FROM   pg_catalog.edb_partition
        WHERE  partpdefid = pd.oid)               AS partition_count,
       0                                          AS def_subpartition_count,
       pg_catalog.array_length(pd.pdefcols, 1)    AS partitioning_key_count,
       pg_catalog.array_length(pd.pdefsubcols, 1) AS subpartitioning_key_count,
       'VALID'::VARCHAR2(8)                       AS status,
       NULL::VARCHAR2(30)                         AS def_tablespace_name,
       NULL::NUMBER                               AS def_pct_free,
       NULL::NUMBER                               AS def_pct_used,
       NULL::NUMBER                               AS def_ini_trans,
       NULL::NUMBER                               AS def_max_trans,
       NULL::VARCHAR2(40)                         AS def_initial_extent,
       NULL::VARCHAR2(40)                         AS def_next_extent,
       NULL::VARCHAR2(40)                         AS def_min_extents,
       NULL::VARCHAR2(40)                         AS def_max_extents,
       NULL::VARCHAR2(40)                         AS def_pct_increase,
       NULL::NUMBER                               AS def_freelists,
       NULL::NUMBER                               AS def_freelist_groups,
       'YES'::VARCHAR2(7)                         AS def_logging,
       'NONE'::VARCHAR2(8)                        AS def_compression,
       'DEFAULT'::VARCHAR2(7)                     AS def_buffer_pool,
       NULL::VARCHAR2(30)                         AS ref_ptn_constraint_name,
       NULL::VARCHAR2(1000)                       AS interval
FROM   pg_catalog.edb_partdef pd
       JOIN pg_catalog.pg_class class ON (class.oid = pd.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
WHERE  has_table_privilege(class.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_part_tables TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_part_tables AS
SELECT schema_name, table_name, partitioning_type, subpartitioning_type,
       partition_count, def_subpartition_count, partitioning_key_count,
       subpartitioning_key_count, status, def_tablespace_name, def_pct_free,
       def_pct_used, def_ini_trans, def_max_trans, def_initial_extent,
       def_min_extents, def_max_extents, def_pct_increase, def_freelists,
       def_freelist_groups, def_logging, def_compression, def_buffer_pool,
       ref_ptn_constraint_name, interval
FROM   sys.all_part_tables
WHERE  owner = user;

GRANT SELECT ON sys.user_part_tables TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_tab_partitions views                           --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_tab_partitions AS
SELECT pg_get_userbyid(class.relowner)                       AS table_owner,
       ns.nspname                                            AS schema_name,
       class.relname                                         AS table_name,
       CASE pd.pdefsubtype WHEN '?' THEN 'NO' ELSE 'YES' END AS composite,
       part.partname                                         AS partition_name,
       (SELECT count(*)
        FROM   pg_catalog.edb_partition sub
        WHERE sub.partparent = part.oid)                     AS subpartition_count,
	   pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid) AS high_value,
	   pg_catalog.length(pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid)) AS high_value_length,
       part.partpos                                          AS partition_position,
       ts.spcname                                            AS tablespace_name,
       0::NUMBER                                             AS pct_free,
       0::NUMBER                                             AS pct_used,
       0::NUMBER                                             AS ini_trans,
       0::number                                             AS max_trans,
       NULL::number                                          AS initial_extent,
       NULL::number                                          AS next_extent,
       0::number                                             AS min_extent,
       0::number                                             AS max_extent,
       0::number                                             AS pct_increase,
       NULL::NUMBER                                          AS freelists,
       NULL::NUMBER                                          AS freelist_groups,
       'YES'::VARCHAR2(7)                                    AS logging,
       'NONE'::VARCHAR2(8)                                   AS compression,
       class.reltuples::NUMBER                               AS num_rows,
       class.relpages                                        AS blocks,
       NULL::NUMBER                                          AS empty_blocks,
       NULL::NUMBER                                          AS avg_space,
       NULL::NUMBER                                          AS chain_cnt,
       NULL::NUMBER                                          AS avg_row_len,
       NULL::NUMBER                                          AS sample_size,
       NULL::DATE                                            AS last_analyzed,
       NULL::VARCHAR2(7)                                     AS buffer_pool,
       'YES'::VARCHAR2(3)                                    AS global_stats,
       'NO'::VARCHAR2(3)                                     AS user_stats,
       part.partrelid::regclass                              AS backing_table
FROM   pg_catalog.edb_partition part
       JOIN pg_catalog.edb_partdef pd ON (pd.oid = part.partpdefid)
       JOIN pg_catalog.pg_class class ON (class.oid = pd.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
       LEFT JOIN pg_catalog.pg_tablespace ts ON(ts.oid = part.parttablespace)
WHERE  part.partparent = 0;

CREATE OR REPLACE VIEW sys.all_tab_partitions AS
SELECT pg_get_userbyid(class.relowner)                       AS table_owner,
       ns.nspname                                            AS schema_name,
       class.relname                                         AS table_name,
       CASE pd.pdefsubtype WHEN '?' THEN 'NO' ELSE 'YES' END AS composite,
       part.partname                                         AS partition_name,
       (SELECT count(*)
        FROM   pg_catalog.edb_partition sub
        WHERE sub.partparent = part.oid)                     AS subpartition_count,
	   pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid) AS high_value,
	   pg_catalog.length(pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid)) AS high_value_length,
       part.partpos                                          AS partition_position,
       ts.spcname                                            AS tablespace_name,
       0::NUMBER                                             AS pct_free,
       0::NUMBER                                             AS pct_used,
       0::NUMBER                                             AS ini_trans,
       0::number                                             AS max_trans,
       NULL::number                                          AS initial_extent,
       NULL::number                                          AS next_extent,
       0::number                                             AS min_extent,
       0::number                                             AS max_extent,
       0::number                                             AS pct_increase,
       NULL::NUMBER                                          AS freelists,
       NULL::NUMBER                                          AS freelist_groups,
       'YES'::VARCHAR2(7)                                    AS logging,
       'NONE'::VARCHAR2(8)                                   AS compression,
       class.reltuples::NUMBER                               AS num_rows,
       class.relpages                                        AS blocks,
       NULL::NUMBER                                          AS empty_blocks,
       NULL::NUMBER                                          AS avg_space,
       NULL::NUMBER                                          AS chain_cnt,
       NULL::NUMBER                                          AS avg_row_len,
       NULL::NUMBER                                          AS sample_size,
       NULL::DATE                                            AS last_analyzed,
       NULL::VARCHAR2(7)                                     AS buffer_pool,
       'YES'::VARCHAR2(3)                                    AS global_stats,
       'NO'::VARCHAR2(3)                                     AS user_stats,
       part.partrelid::regclass                              AS backing_table
FROM   pg_catalog.edb_partition part
       JOIN pg_catalog.edb_partdef pd ON (pd.oid = part.partpdefid)
       JOIN pg_catalog.pg_class class ON (class.oid = pd.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
       LEFT JOIN pg_catalog.pg_tablespace ts ON(ts.oid = part.parttablespace)
WHERE  part.partparent = 0 AND
       has_table_privilege(class.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_tab_partitions TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_tab_partitions AS
SELECT schema_name, table_name, composite, partition_name, subpartition_count,
       high_value, high_value_length, partition_position, tablespace_name,
       pct_free, pct_used, ini_trans, max_trans, initial_extent, next_extent,
       min_extent, max_extent, pct_increase, freelists, freelist_groups,
       logging, compression, num_rows, blocks, empty_blocks, avg_space, chain_cnt,
       avg_row_len, sample_size, last_analyzed, buffer_pool, global_stats,
       user_stats, backing_table
FROM   sys.all_tab_partitions
WHERE  table_owner = user;

GRANT SELECT ON sys.user_tab_partitions TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_tab_subpartitions views                        --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_tab_subpartitions AS
SELECT pg_get_userbyid(class.relowner)                       AS table_owner,
       ns.nspname                                            AS schema_name,
       class.relname                                         AS table_name,
       parent.partname                                       AS partition_name,
       part.partname                                         AS subpartition_name,
	   pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid) AS high_value,
	   pg_catalog.length(pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid)) AS high_value_length,
       part.partpos                                          AS subpartition_position,
       ts.spcname                                            AS tablespace_name,
       0::NUMBER                                             AS pct_free,
       0::NUMBER                                             AS pct_used,
       0::NUMBER                                             AS ini_trans,
       0::number                                             AS max_trans,
       NULL::number                                          AS initial_extent,
       NULL::number                                          AS next_extent,
       0::number                                             AS min_extent,
       0::number                                             AS max_extent,
       0::number                                             AS pct_increase,
       NULL::NUMBER                                          AS freelists,
       NULL::NUMBER                                          AS freelist_groups,
       'YES'::VARCHAR2(7)                                    AS logging,
       'NONE'::VARCHAR2(8)                                   AS compression,
       class.reltuples::NUMBER                               AS num_rows,
       class.relpages                                        AS blocks,
       NULL::NUMBER                                          AS empty_blocks,
       NULL::NUMBER                                          AS avg_space,
       NULL::NUMBER                                          AS chain_cnt,
       NULL::NUMBER                                          AS avg_row_len,
       NULL::NUMBER                                          AS sample_size,
       NULL::DATE                                            AS last_analyzed,
       NULL::VARCHAR2(7)                                     AS buffer_pool,
       'YES'::VARCHAR2(3)                                    AS global_stats,
       'NO'::VARCHAR2(3)                                     AS user_stats,
       part.partrelid::regclass                              AS backing_table
FROM   pg_catalog.edb_partition part
       JOIN pg_catalog.edb_partition parent ON (parent.oid = part.partparent)
       JOIN pg_catalog.edb_partdef pd ON (pd.oid = part.partpdefid)
       JOIN pg_catalog.pg_class class ON (class.oid = pd.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
       LEFT JOIN pg_catalog.pg_tablespace ts ON(ts.oid = part.parttablespace);

CREATE OR REPLACE VIEW sys.all_tab_subpartitions AS
SELECT pg_get_userbyid(class.relowner)                       AS table_owner,
       ns.nspname                                            AS schema_name,
       class.relname                                         AS table_name,
       parent.partname                                       AS partition_name,
       part.partname                                         AS subpartition_name,
	   pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid) AS high_value,
	   pg_catalog.length(pg_catalog.pg_get_expr_inexact(part.partvals, part.partrelid)) AS high_value_length,
       part.partpos                                          AS subpartition_position,
       ts.spcname                                            AS tablespace_name,
       0::NUMBER                                             AS pct_free,
       0::NUMBER                                             AS pct_used,
       0::NUMBER                                             AS ini_trans,
       0::number                                             AS max_trans,
       NULL::number                                          AS initial_extent,
       NULL::number                                          AS next_extent,
       0::number                                             AS min_extent,
       0::number                                             AS max_extent,
       0::number                                             AS pct_increase,
       NULL::NUMBER                                          AS freelists,
       NULL::NUMBER                                          AS freelist_groups,
       'YES'::VARCHAR2(7)                                    AS logging,
       'NONE'::VARCHAR2(8)                                   AS compression,
       class.reltuples::NUMBER                               AS num_rows,
       class.relpages                                        AS blocks,
       NULL::NUMBER                                          AS empty_blocks,
       NULL::NUMBER                                          AS avg_space,
       NULL::NUMBER                                          AS chain_cnt,
       NULL::NUMBER                                          AS avg_row_len,
       NULL::NUMBER                                          AS sample_size,
       NULL::DATE                                            AS last_analyzed,
       NULL::VARCHAR2(7)                                     AS buffer_pool,
       'YES'::VARCHAR2(3)                                    AS global_stats,
       'NO'::VARCHAR2(3)                                     AS user_stats,
       part.partrelid::regclass                              AS backing_table
FROM   pg_catalog.edb_partition part
       JOIN pg_catalog.edb_partition parent ON (parent.oid = part.partparent)
       JOIN pg_catalog.edb_partdef pd ON (pd.oid = part.partpdefid)
       JOIN pg_catalog.pg_class class ON (class.oid = pd.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
       LEFT JOIN pg_catalog.pg_tablespace ts ON(ts.oid = part.parttablespace)
WHERE  has_table_privilege(class.oid, 'SELECT, INSERT, UPDATE, DELETE, TRUNCATE, REFERENCES, TRIGGER') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast');

GRANT SELECT ON sys.all_tab_subpartitions TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_tab_subpartitions AS
SELECT schema_name, table_name, partition_name, subpartition_name, high_value,
       high_value_length, subpartition_position, tablespace_name, pct_free,
       pct_used, ini_trans, max_trans, initial_extent, next_extent, min_extent,
       max_extent, pct_increase, freelists, freelist_groups, logging,
       compression, num_rows, blocks, empty_blocks, avg_space, chain_cnt,
       avg_row_len, sample_size, last_analyzed, buffer_pool, global_stats,
       user_stats, backing_table
FROM   sys.all_tab_subpartitions
WHERE  table_owner = user;

GRANT SELECT ON sys.user_tab_subpartitions TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_part_key_columns views                         --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_part_key_columns AS
SELECT pg_get_userbyid(class.relowner) AS owner,
       ns.nspname                      AS schema_name,
       class.relname                   AS name,
       'TABLE'::CHAR(5)                AS object_type,
       a.attname                       AS column_name,
       s.ordinal + 1                   AS column_position
FROM   (SELECT pdefrel, pdefcols, pg_catalog.generate_subscripts(pdefcols, 1) ordinal FROM edb_partdef) s
       JOIN pg_catalog.pg_attribute a ON (a.attrelid = s.pdefrel AND a.attnum = s.pdefcols[s.ordinal])
       JOIN pg_catalog.pg_class class ON(class.oid = s.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
ORDER BY class.relname, s.ordinal;

CREATE OR REPLACE VIEW sys.all_part_key_columns AS
SELECT pg_get_userbyid(class.relowner) AS owner,
       ns.nspname                      AS schema_name,
       class.relname                   AS name,
       'TABLE'::CHAR(5)                AS object_type,
       a.attname                       AS column_name,
       s.ordinal + 1                   AS column_position
FROM   (SELECT pdefrel, pdefcols, pg_catalog.generate_subscripts(pdefcols, 1) ordinal FROM edb_partdef) s
       JOIN pg_catalog.pg_attribute a ON (a.attrelid = s.pdefrel AND a.attnum = s.pdefcols[s.ordinal])
       JOIN pg_catalog.pg_class class ON(class.oid = s.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
WHERE  has_column_privilege(class.oid, a.attnum, 'SELECT, INSERT, UPDATE, REFERENCES') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
ORDER BY class.relname, s.ordinal;

GRANT SELECT ON sys.all_part_key_columns TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_part_key_columns AS
SELECT schema_name, name, object_type, column_name, column_position
FROM   sys.all_part_key_columns
WHERE  owner = user;

GRANT SELECT ON sys.user_part_key_columns TO PUBLIC;


--------------------------------------------------------------------------------
--                       sys.*_subpart_key_columns views                      --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_subpart_key_columns AS
SELECT pg_get_userbyid(class.relowner) AS owner,
       ns.nspname                      AS schema_name,
       class.relname                   AS name,
       'TABLE'::CHAR(5)                AS object_type,
       a.attname                       AS column_name,
       s.ordinal + 1                   AS column_position
FROM   (SELECT pdefrel, pdefsubcols, pg_catalog.generate_subscripts(pdefsubcols, 1) ordinal FROM edb_partdef) s
       JOIN pg_catalog.pg_attribute a ON (a.attrelid = s.pdefrel AND a.attnum = s.pdefsubcols[s.ordinal])
       JOIN pg_catalog.pg_class class ON(class.oid = s.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
ORDER BY class.relname, s.ordinal;

CREATE OR REPLACE VIEW sys.all_subpart_key_columns AS
SELECT pg_get_userbyid(class.relowner) AS owner,
       ns.nspname                      AS schema_name,
       class.relname                   AS name,
       'TABLE'::CHAR(5)                AS object_type,
       a.attname                       AS column_name,
       s.ordinal + 1                   AS column_position
FROM   (SELECT pdefrel, pdefsubcols, pg_catalog.generate_subscripts(pdefsubcols, 1) ordinal FROM edb_partdef) s
       JOIN pg_catalog.pg_attribute a ON (a.attrelid = s.pdefrel AND a.attnum = s.pdefsubcols[s.ordinal])
       JOIN pg_catalog.pg_class class ON(class.oid = s.pdefrel)
       JOIN pg_catalog.pg_namespace ns ON (ns.oid = class.relnamespace)
WHERE  has_column_privilege(class.oid, a.attnum, 'SELECT, INSERT, UPDATE, REFERENCES') AND
       ns.nspname NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
ORDER BY class.relname, s.ordinal;

GRANT SELECT ON sys.all_subpart_key_columns TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_subpart_key_columns AS
SELECT schema_name, name, object_type, column_name, column_position
FROM   sys.all_subpart_key_columns
WHERE  owner = user;

GRANT SELECT ON sys.user_subpart_key_columns TO PUBLIC;


-- disallow removal of system views

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_part_key_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_part_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_policies'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_subpart_key_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_tab_partitions'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'all_tab_subpartitions'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_part_key_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_part_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_policies'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_subpart_key_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_tab_partitions'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'dba_tab_subpartitions'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_part_key_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_part_tables'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_policies'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_subpart_key_columns'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_tab_partitions'), 0, 'p');

insert into pg_depend values (0,0,0,
    (select oid from pg_class where relname = 'pg_class'),
    (select oid from pg_class where relname = 'user_tab_subpartitions'), 0, 'p');


-- REGEXP_INSTR: Returns the position at which requested pattern found.
CREATE OR REPLACE FUNCTION sys.regexp_instr(srcstr TEXT, pattern TEXT,
	"position" NUMBER DEFAULT 1, occurrence NUMBER DEFAULT 1,
	returnparam NUMBER DEFAULT 0, modifier TEXT DEFAULT NULL,
	subexpression NUMBER DEFAULT 0)
RETURNS INT
AS $$SELECT pg_catalog.regexp_instr7(srcstr, pattern, position::INTEGER,
	occurrence::INTEGER, returnparam::INTEGER,
	modifier, subexpression::INTEGER);$$
LANGUAGE SQL IMMUTABLE;

COMMENT ON FUNCTION
	sys.regexp_instr(text, text, number, number, number, text, number)
	IS 'find match location for regexp';

-- REGEXP_SUBSTR: Returns the text matching a given regular expression.
CREATE OR REPLACE FUNCTION sys.regexp_substr(srcstr TEXT, pattern TEXT,
	"position" NUMBER DEFAULT 1, occurrence NUMBER DEFAULT 1,
	modifier TEXT DEFAULT NULL, subexpression NUMBER DEFAULT 0)
RETURNS TEXT
AS $$SELECT pg_catalog.regexp_substr6(srcstr, pattern, position::INTEGER,
	occurrence::INTEGER, modifier, subexpression::INTEGER);$$
LANGUAGE SQL IMMUTABLE;

COMMENT ON FUNCTION sys.regexp_substr(text, text, number, number, text, number)
	IS 'extract one match of regexp';

-- REGEXP_COUNT: Returns the number of patterns found matching a given regular expression.
CREATE OR REPLACE FUNCTION sys.regexp_count(srcstr TEXT, pattern TEXT,
	"position" NUMBER DEFAULT 1, modifier TEXT DEFAULT NULL)
RETURNS INT
AS $$SELECT pg_catalog.regexp_count4(srcstr, pattern, position::INTEGER,
	modifier);$$
LANGUAGE SQL IMMUTABLE;

COMMENT ON FUNCTION sys.regexp_count(text, text, number, text)
	IS 'count all regexp matches';
