-- DBMS_SCHEDULER extension
--   Creates required catalog tables, function, triggers etc to be used in
-- dbms_scheduler package

-- EDB:
-- Tables created here are all from their corresponding types defined in .plb
-- If you want to change something related to table, do change corresponding
-- type in .plb

-- complain if script is sourced in psql, rather than via CREATE EXTENSION
\echo Use "CREATE EXTENSION dbms_scheduler" to load this file. \quit

-- scheduler_0100_component_name dummy table to check the uniqueness of
-- (program/schedule/job)_name
CREATE TABLE scheduler_0100_component_name OF scheduler_0100_component_name_type(
  dsc_name WITH OPTIONS UNIQUE
);

-- Trigger function, which will take care of insert metadata entry into
-- scheduler_0100_component_name during create_* and also take care of deleting entry
-- during drop_* procedure. By making entry into scheduler_0100_component_name we will
-- ensure the uniqueness of (program/schedule/job)_name across other
-- dbms_scheduler component_name
CREATE OR REPLACE FUNCTION scheduler_check_component_name() RETURN trigger AS
DECLARE
  name VARCHAR2;
BEGIN

  IF TG_OP = 'INSERT'  THEN
    CASE (upper(TG_ARGV[0]))
      WHEN 'JOB' THEN
        name := NEW.dsj_job_name;
      WHEN 'PROGRAM' THEN
        name := NEW.dsp_program_name;
      WHEN 'SCHEDULE' THEN
        name := NEW.dss_schedule_name;
    END CASE;

    -- Component name NULL means its internal component and we don't want to do
    -- anything for that
    IF name IS NOT NULL THEN
      -- check component_name is unique across other dbms_scheduler component
      INSERT INTO sys.scheduler_0100_component_name VALUES (name);
    END IF;
    -- UPDATE is not possible here, so only need to handle DELETE
  ELSE
    CASE (upper(TG_ARGV[0]))
      WHEN 'JOB' THEN
        name := OLD.dsj_job_name;
      WHEN 'PROGRAM' THEN
        name := OLD.dsp_program_name;
      WHEN 'SCHEDULE' THEN
        name := OLD.dss_schedule_name;
    END CASE;

    -- Component name NULL means its internal component and we don't want to do
    -- anything for that
    IF name IS NOT NULL THEN
      -- Delete the component name entry from scheduler_0100_component_name
      DELETE FROM sys.scheduler_0100_component_name WHERE dsc_name = name;
    END IF;
  END IF;

  RETURN NEW;

  EXCEPTION
    WHEN unique_violation THEN
      raise 'dbms_scheduler component ''%'' already exists', name;
END;

-- Create sequence for dsp_program_id
CREATE SEQUENCE scheduler_0200_program_dsp_program_id_seq;

-- Catalog table to hold program details
CREATE TABLE scheduler_0200_program OF scheduler_0200_program_type(
  dsp_program_id          WITH OPTIONS NOT NULL DEFAULT nextval('scheduler_0200_program_dsp_program_id_seq'::regclass),
  dsp_owner               WITH OPTIONS DEFAULT session_user,
  dsp_program_type        WITH OPTIONS NOT NULL,
  dsp_program_action      WITH OPTIONS NOT NULL,
  dsp_number_of_arguments WITH OPTIONS DEFAULT 0,
  dsp_enabled             WITH OPTIONS DEFAULT FALSE,
  dsp_comments            WITH OPTIONS DEFAULT NULL,
  PRIMARY KEY (dsp_program_id)
);

-- Redwood do not allow duplicate program names so create unique index to
-- search on program names.
CREATE UNIQUE INDEX dsp_program_name_idx ON sys.scheduler_0200_program(dsp_program_name);

-- Trigger to check the program_name uniqueness across other dbms_scheduler
-- components
CREATE TRIGGER scheduler_0200_program_name_isunique BEFORE INSERT
  ON sys.scheduler_0200_program FOR EACH ROW
  EXECUTE PROCEDURE sys.scheduler_check_component_name('PROGRAM');

-- Trigger to delete the scheduler_0100_component_name metadata entry
CREATE TRIGGER scheduler_0200_program_name_delete AFTER DELETE
  ON sys.scheduler_0200_program FOR EACH ROW
  EXECUTE PROCEDURE sys.scheduler_check_component_name('PROGRAM');

-- Catalog table to hold program argument details
CREATE TABLE scheduler_0250_program_argument OF scheduler_0250_program_argument_type(
  dspa_owner             WITH OPTIONS DEFAULT session_user,
  dspa_argument_name     WITH OPTIONS DEFAULT NULL,
  dspa_default_value_set WITH OPTIONS DEFAULT FALSE,
  dspa_out_argument      WITH OPTIONS DEFAULT FALSE,
  PRIMARY KEY (dspa_program_id, dspa_argument_position),
  FOREIGN KEY (dspa_program_id) REFERENCES sys.scheduler_0200_program(dsp_program_id)
);

-- Create sequence for dsp_schedule_id
CREATE SEQUENCE scheduler_0300_schedule_dss_schedule_id_seq;

-- Catalog table to hold schedule details
CREATE TABLE scheduler_0300_schedule OF scheduler_0300_schedule_type(
  dss_owner           WITH OPTIONS DEFAULT session_user,
  dss_schedule_id     WITH OPTIONS NOT NULL DEFAULT nextval('scheduler_0300_schedule_dss_schedule_id_seq'::regclass),
  dss_start_date      WITH OPTIONS DEFAULT NULL,
  dss_repeat_interval WITH OPTIONS DEFAULT NULL,
  dss_end_date        WITH OPTIONS DEFAULT NULL,
  dss_comments        WITH OPTIONS DEFAULT NULL,
  dss_bymonth         WITH OPTIONS NOT NULL DEFAULT '{f,f,f,f,f,f,f,f,f,f, f,f}',
  dss_bymonthday      WITH OPTIONS NOT NULL DEFAULT '{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f}',
  dss_byday           WITH OPTIONS NOT NULL DEFAULT '{f,f,f,f,f,f,f}',
  dss_byhour          WITH OPTIONS NOT NULL DEFAULT '{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f}',
  dss_byminute        WITH OPTIONS NOT NULL DEFAULT '{f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f, f,f,f,f,f,f,f,f,f,f}',

  PRIMARY KEY (dss_schedule_id),

  CONSTRAINT dss_bymonth_ub    CHECK (array_upper(dss_bymonth, 1) = 12),
  CONSTRAINT dss_bymonthday_ub CHECK (array_upper(dss_bymonthday, 1) = 31),
  CONSTRAINT dss_byday_ub      CHECK (array_upper(dss_byday, 1) = 7),
  CONSTRAINT dss_byhour_ub     CHECK (array_upper(dss_byhour, 1) = 24),
  CONSTRAINT dss_byminute_ub   CHECK (array_upper(dss_byminute, 1) = 60),

  CONSTRAINT dss_bymonth_lb    CHECK (array_lower(dss_bymonth, 1) = 1),
  CONSTRAINT dss_bymonthday_lb CHECK (array_lower(dss_bymonthday, 1) = 1),
  CONSTRAINT dss_byday_lb      CHECK (array_lower(dss_byday, 1) = 1),
  CONSTRAINT dss_byhour_lb     CHECK (array_lower(dss_byhour, 1) = 1),
  CONSTRAINT dss_byminute_lb   CHECK (array_lower(dss_byminute, 1) = 1)
);

-- Redwood do not allow duplicate schedule names so create unique index to
-- search on schedule names.
CREATE UNIQUE INDEX dss_schedule_name_idx ON sys.scheduler_0300_schedule(dss_schedule_name);

-- Trigger to check the schedule_name uniqueness across other dbms_scheduler
-- components
CREATE TRIGGER scheduler_0300_schedule_name_isunique BEFORE INSERT
  ON sys.scheduler_0300_schedule FOR EACH ROW
  EXECUTE PROCEDURE sys.scheduler_check_component_name('SCHEDULE');

-- Trigger to delete the scheduler_0100_component_name metadata entry
CREATE TRIGGER scheduler_0300_schedule_name_delete AFTER DELETE
  ON sys.scheduler_0300_schedule FOR EACH ROW
  EXECUTE PROCEDURE sys.scheduler_check_component_name('SCHEDULE');

-- Catalog table to hold job details
CREATE TABLE scheduler_0400_job OF scheduler_0400_job_type(
  dsj_owner     WITH OPTIONS DEFAULT session_user,
  dsj_job_class WITH OPTIONS DEFAULT 'DEFAULT_JOB_CLASS',
  dsj_enabled   WITH OPTIONS DEFAULT FALSE,
  dsj_auto_drop WITH OPTIONS DEFAULT TRUE,
  dsj_comments  WITH OPTIONS DEFAULT NULL,
  PRIMARY KEY (dsj_job_id),
  FOREIGN KEY (dsj_program_id) REFERENCES sys.scheduler_0200_program(dsp_program_id),
  FOREIGN KEY (dsj_schedule_id) REFERENCES sys.scheduler_0300_schedule(dss_schedule_id)
);

-- Redwood do not allow duplicate job names so create unique index to
-- search on job names.
CREATE UNIQUE INDEX dsj_job_name_idx ON sys.scheduler_0400_job(dsj_job_name);

-- Trigger to check the job_name uniqueness across other dbms_scheduler
-- components
CREATE TRIGGER scheduler_0400_job_name_isunique BEFORE INSERT
  ON sys.scheduler_0400_job FOR EACH ROW
  EXECUTE PROCEDURE sys.scheduler_check_component_name('JOB');

-- Trigger to delete the scheduler_0100_component_name metadata entry
CREATE TRIGGER scheduler_0400_job_name_delete AFTER DELETE
  ON sys.scheduler_0400_job FOR EACH ROW
  EXECUTE PROCEDURE sys.scheduler_check_component_name('JOB');

CREATE TABLE scheduler_0450_job_argument OF scheduler_0450_job_argument_type(
  dsja_owner         WITH OPTIONS DEFAULT session_user,
  dsja_argument_name WITH OPTIONS DEFAULT NULL,
  PRIMARY KEY (dsja_job_id, dsja_argument_position),
  FOREIGN KEY (dsja_job_id) REFERENCES sys.scheduler_0400_job(dsj_job_id)
);

-- Need to dump data from these tables
SELECT pg_catalog.pg_extension_config_dump('scheduler_0200_program', '');
SELECT pg_catalog.pg_extension_config_dump('scheduler_0250_program_argument', '');
SELECT pg_catalog.pg_extension_config_dump('scheduler_0300_schedule', '');
SELECT pg_catalog.pg_extension_config_dump('scheduler_0400_job', '');
SELECT pg_catalog.pg_extension_config_dump('scheduler_0450_job_argument', '');


--------------------------------------------------------------------------------
--                        DBMS_SCHEDULER views                                --
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
--                   sys.*_scheduler_programs views                           --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_scheduler_programs AS
SELECT dsp_owner               AS OWNER,
       dsp_program_name        AS PROGRAM_NAME,
       dsp_program_type        AS PROGRAM_TYPE,
       dsp_program_action      AS PROGRAM_ACTION,
       dsp_number_of_arguments AS NUMBER_OF_ARGUMENTS,
       dsp_enabled             AS ENABLED,
       NULL::VARCHAR2(5)       AS DETACHED,
       NULL::INTERVAL          AS SCHEDULE_LIMIT,
       NULL::NUMBER            AS PRIORITY,
       NULL::NUMBER            AS WEIGHT,
       NULL::NUMBER            AS MAX_RUNS,
       NULL::NUMBER            AS MAX_FAILURES,
       NULL::INTERVAL          AS MAX_RUN_DURATION,
       NULL::VARCHAR(4000)     AS NLS_ENV,
       dsp_comments            AS COMMENTS
FROM   sys.scheduler_0200_program
WHERE  dsp_program_name IS NOT NULL;

CREATE OR REPLACE VIEW sys.all_scheduler_programs AS
SELECT *
FROM   sys.dba_scheduler_programs
WHERE  pg_has_role(owner, 'USAGE');

GRANT SELECT ON sys.all_scheduler_programs TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_scheduler_programs AS
SELECT PROGRAM_NAME, PROGRAM_TYPE, PROGRAM_ACTION, NUMBER_OF_ARGUMENTS, ENABLED,
       DETACHED, SCHEDULE_LIMIT, PRIORITY, WEIGHT, MAX_RUNS, MAX_FAILURES,
       MAX_RUN_DURATION, NLS_ENV, COMMENTS
FROM   sys.all_scheduler_programs
WHERE  OWNER = pg_catalog.current_user;

GRANT SELECT ON sys.user_scheduler_programs TO PUBLIC;

--------------------------------------------------------------------------------
--               sys.*_scheduler_program_args views                           --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_scheduler_program_args AS
SELECT dsp.dsp_owner               AS OWNER,
       dsp.dsp_program_name        AS PROGRAM_NAME,
       dspa.dspa_argument_name     AS ARGUMENT_NAME,
       dspa.dspa_argument_position AS ARGUMENT_POSITION,
       dspa.dspa_argument_type     AS ARGUMENT_TYPE,
       NULL::VARCHAR2(19)          AS METADATA_ATTRIBUTE,
       dspa.dspa_default_value     AS DEFAULT_VALUE,
       NULL::TEXT                  AS DEFAULT_ANYDATA_VALUE,
       dspa.dspa_out_argument      AS OUT_ARGUMENT
FROM   sys.scheduler_0200_program AS dsp, sys.scheduler_0250_program_argument AS dspa
WHERE  dsp.dsp_program_id = dspa.dspa_program_id;

CREATE OR REPLACE VIEW sys.all_scheduler_program_args AS
SELECT *
FROM   sys.dba_scheduler_program_args
WHERE  pg_has_role(owner, 'USAGE');

GRANT SELECT ON sys.all_scheduler_program_args TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_scheduler_program_args AS
SELECT PROGRAM_NAME, ARGUMENT_NAME, ARGUMENT_POSITION, ARGUMENT_TYPE,
       METADATA_ATTRIBUTE, DEFAULT_VALUE, DEFAULT_ANYDATA_VALUE, OUT_ARGUMENT
FROM   sys.all_scheduler_program_args
WHERE  OWNER = pg_catalog.current_user;

GRANT SELECT ON sys.user_scheduler_program_args TO PUBLIC;

--------------------------------------------------------------------------------
--                  sys.*_scheduler_schedules views                           --
--------------------------------------------------------------------------------
CREATE OR REPLACE VIEW sys.dba_scheduler_schedules AS
SELECT dss_owner           AS OWNER,
       dss_schedule_name   AS SCHEDULE_NAME,
       NULL::VARCHAR(12)   AS SCHEDULE_TYPE,
       dss_start_date      AS START_DATE,
       dss_repeat_interval AS REPEAT_INTERVAL,
       NULL::VARCHAR(30)   AS EVENT_QUEUE_OWNER,
       NULL::VARCHAR(30)   AS EVENT_QUEUE_NAME,
       NULL::VARCHAR(30)   AS EVENT_QUEUE_AGENT,
       NULL::VARCHAR(4000) AS EVENT_CONDITION,
       NULL::VARCHAR(260)  AS FILE_WATCHER_OWNER,
       NULL::VARCHAR(260)  AS FILE_WATCHER_NAME,
       dss_end_date        AS END_DATE,
       dss_comments        AS COMMENTS
FROM   sys.scheduler_0300_schedule
WHERE  dss_schedule_name IS NOT NULL;

CREATE OR REPLACE VIEW sys.all_scheduler_schedules AS
SELECT *
FROM   sys.dba_scheduler_schedules
WHERE  pg_has_role(owner, 'USAGE');

GRANT SELECT ON sys.all_scheduler_schedules TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_scheduler_schedules AS
SELECT SCHEDULE_NAME, SCHEDULE_TYPE, START_DATE, REPEAT_INTERVAL, EVENT_QUEUE_OWNER,
       EVENT_QUEUE_NAME, EVENT_QUEUE_AGENT, EVENT_CONDITION, FILE_WATCHER_OWNER,
       FILE_WATCHER_NAME, END_DATE, COMMENTS
FROM   sys.all_scheduler_schedules
WHERE  OWNER = pg_catalog.current_user;

GRANT SELECT ON sys.user_scheduler_schedules TO PUBLIC;

--------------------------------------------------------------------------------
-- Supporting functions which fetches details from pgagent                    --
--------------------------------------------------------------------------------

-- Get run count
CREATE OR REPLACE FUNCTION sys.ds_pga_joblog_run_count(id integer) RETURN integer AS
DECLARE
  cnt integer;
BEGIN
  SELECT COUNT(*) INTO cnt FROM pgagent.pga_joblog WHERE jlgjobid = id;
  RETURN cnt;
END;

-- Get failure count
CREATE OR REPLACE FUNCTION sys.ds_pga_joblog_failure_count(id integer) RETURN integer AS
DECLARE
  cnt integer;
BEGIN
  SELECT COUNT(*) INTO cnt FROM pgagent.pga_joblog WHERE jlgjobid = id AND jlgstatus = 'f';
  RETURN cnt;
END;

-- Get start date for last job
CREATE OR REPLACE FUNCTION sys.ds_pga_joblog_last_start_date(id integer) RETURN timestamptz AS
DECLARE
  lsd timestamptz;
BEGIN
  SELECT jlgstart INTO lsd FROM pgagent.pga_joblog WHERE jlgid =
    (SELECT max(jlgid) FROM pgagent.pga_joblog WHERE jlgjobid = id AND jlgstatus = 's');
  RETURN lsd;
EXCEPTION
  WHEN others THEN
    RETURN null;
END;

-- Get run duration for last run job
CREATE OR REPLACE FUNCTION sys.ds_pga_joblog_last_run_duration(id integer) RETURN interval AS
DECLARE
  lrd interval;
BEGIN
  SELECT jlgduration INTO lrd FROM pgagent.pga_joblog WHERE jlgid =
    (SELECT max(jlgid) FROM pgagent.pga_joblog WHERE jlgjobid = id AND jlgstatus = 's');
  RETURN lrd;
EXCEPTION
  WHEN others THEN
    RETURN null;
END;

-- Get next run date from pga_job for given job
CREATE OR REPLACE FUNCTION sys.ds_pga_job_next_run_date(id integer) RETURN timestamptz AS
DECLARE
  nrd timestamptz;
BEGIN
  SELECT jobnextrun INTO nrd FROM pgagent.pga_job WHERE jobid = id;
  RETURN nrd;
EXCEPTION
  WHEN others THEN
    RETURN null;
END;


--------------------------------------------------------------------------------
--                  sys.*_scheduler_jobs views                                --
--------------------------------------------------------------------------------
CREATE OR REPLACE VIEW sys.dba_scheduler_jobs AS
SELECT dsj.dsj_owner               AS OWNER,
       dsj.dsj_job_name            AS JOB_NAME,
       NULL::VARCHAR2(30)          AS JOB_SUBNAME,
       NULL::VARCHAR2(11)          AS JOB_STYLE,
       NULL::VARCHAR2(30)          AS JOB_CREATOR,
       NULL::VARCHAR2(64)          AS CLIENT_ID,
       NULL::VARCHAR2(32)          AS GLOBAL_UID,
       dsp.dsp_owner               AS PROGRAM_OWNER,
       dsp.dsp_program_name        AS PROGRAM_NAME,
       dsp.dsp_program_type        AS JOB_TYPE,
       dsp.dsp_program_action      AS JOB_ACTION,
       dsp.dsp_number_of_arguments AS NUMBER_OF_ARGUMENTS,
       dss.dss_owner               AS SCHEDULE_OWNER,
       dss.dss_schedule_name       AS SCHEDULE_NAME,
       NULL::VARCHAR2(12)          AS SCHEDULE_TYPE,
       dss.dss_start_date          AS START_DATE,
       dss.dss_repeat_interval     AS REPEAT_INTERVAL,
       NULL::VARCHAR2(30)          AS EVENT_QUEUE_OWNER,
       NULL::VARCHAR2(30)          AS EVENT_QUEUE_NAME,
       NULL::VARCHAR2(256)         AS EVENT_QUEUE_AGENT,
       NULL::VARCHAR2(4000)        AS EVENT_CONDITION,
       NULL::VARCHAR2(65)          AS EVENT_RULE,
       NULL::VARCHAR2(260)         AS FILE_WATCHER_OWNER,
       NULL::VARCHAR2(260)         AS FILE_WATCHER_NAME,
       dss.dss_end_date            AS END_DATE,
       dsj.dsj_job_class           AS JOB_CLASS,
       dsj.dsj_enabled             AS ENABLED,
       dsj.dsj_auto_drop           AS AUTO_DROP,
       NULL::VARCHAR2(5)           AS RESTARTABLE,
       NULL::VARCHAR2(15)          AS STATE,
       NULL::NUMBER                AS JOB_PRIORITY,
       (SELECT sys.ds_pga_joblog_run_count(dsj.dsj_job_id))
                                   AS RUN_COUNT,
       NULL::NUMBER                AS MAX_RUNS,
       (SELECT sys.ds_pga_joblog_failure_count(dsj.dsj_job_id))
                                   AS FAILURE_COUNT,
       NULL::NUMBER                AS MAX_FAILURES,
       NULL::NUMBER                AS RETRY_COUNT,
       (SELECT sys.ds_pga_joblog_last_start_date(dsj.dsj_job_id))
                                   AS LAST_START_DATE,
       (SELECT sys.ds_pga_joblog_last_run_duration(dsj.dsj_job_id))
                                   AS LAST_RUN_DURATION,
       (SELECT sys.ds_pga_job_next_run_date(dsj.dsj_job_id))
                                   AS NEXT_RUN_DATE,
       NULL::INTERVAL              AS SCHEDULE_LIMIT,
       NULL::INTERVAL              AS MAX_RUN_DURATION,
       NULL::VARCHAR2(11)          AS LOGGING_LEVEL,
       NULL::VARCHAR2(5)           AS STOP_ON_WINDOW_CLOSE,
       NULL::VARCHAR2(5)           AS INSTANCE_STICKINESS,
       NULL::VARCHAR2(4000)        AS RAISE_EVENTS,
       NULL::VARCHAR2(5)           AS SYSTEM,
       NULL::NUMBER                AS JOB_WEIGHT,
       NULL::VARCHAR2(4000)        AS NLS_ENV,
       NULL::VARCHAR2(128)         AS SOURCE,
       NULL::NUMBER                AS NUMBER_OF_DESTINATIONS,
       NULL::VARCHAR2(512)         AS DESTINATION_OWNER,
       NULL::VARCHAR2(512)         AS DESTINATION,
       NULL::VARCHAR2(30)          AS CREDENTIAL_OWNER,
       NULL::VARCHAR2(30)          AS CREDENTIAL_NAME,
       NULL::NUMBER                AS INSTANCE_ID,
       NULL::VARCHAR2(5)           AS DEFERRED_DROP,
       NULL::VARCHAR2(5)           AS ALLOW_RUNS_IN_RESTRICTED_MODE,
       dsj.dsj_comments            AS COMMENTS,
       NULL::NUMBER                AS FLAGS
FROM   sys.scheduler_0400_job dsj, sys.scheduler_0200_program dsp, sys.scheduler_0300_schedule dss
WHERE  dsj.dsj_schedule_id = dss.dss_schedule_id AND
       dsj.dsj_program_id = dsp.dsp_program_id;

CREATE OR REPLACE VIEW sys.all_scheduler_jobs AS
SELECT *
FROM   sys.dba_scheduler_jobs
WHERE  pg_has_role(owner, 'USAGE');

GRANT SELECT ON sys.all_scheduler_jobs TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_scheduler_jobs AS
SELECT JOB_NAME, JOB_SUBNAME, JOB_STYLE, JOB_CREATOR, CLIENT_ID, GLOBAL_UID,
       PROGRAM_OWNER, PROGRAM_NAME, JOB_TYPE, JOB_ACTION, NUMBER_OF_ARGUMENTS,
       SCHEDULE_OWNER, SCHEDULE_NAME, SCHEDULE_TYPE, START_DATE, REPEAT_INTERVAL,
       EVENT_QUEUE_OWNER, EVENT_QUEUE_NAME, EVENT_QUEUE_AGENT,	EVENT_CONDITION,
       EVENT_RULE, FILE_WATCHER_OWNER, FILE_WATCHER_NAME, END_DATE, JOB_CLASS,
       ENABLED, AUTO_DROP, RESTARTABLE, STATE, JOB_PRIORITY, RUN_COUNT, MAX_RUNS,
       FAILURE_COUNT, MAX_FAILURES, RETRY_COUNT, LAST_START_DATE, LAST_RUN_DURATION,
       NEXT_RUN_DATE, SCHEDULE_LIMIT, MAX_RUN_DURATION, LOGGING_LEVEL,
       STOP_ON_WINDOW_CLOSE, INSTANCE_STICKINESS, RAISE_EVENTS, SYSTEM, JOB_WEIGHT,
       NLS_ENV, SOURCE, NUMBER_OF_DESTINATIONS, DESTINATION_OWNER, DESTINATION,
       CREDENTIAL_OWNER, CREDENTIAL_NAME, INSTANCE_ID, DEFERRED_DROP,
       ALLOW_RUNS_IN_RESTRICTED_MODE, COMMENTS, FLAGS
FROM   sys.all_scheduler_jobs
WHERE  OWNER = pg_catalog.current_user;

GRANT SELECT ON sys.user_scheduler_jobs TO PUBLIC;

--------------------------------------------------------------------------------
--               sys.*_scheduler_job_args views                               --
--------------------------------------------------------------------------------

CREATE OR REPLACE VIEW sys.dba_scheduler_job_args AS
SELECT dsj.dsj_owner               AS OWNER,
       dsj.dsj_job_name            AS JOB_NAME,
       dsja.dsja_argument_name     AS ARGUMENT_NAME,
       dsja.dsja_argument_position AS ARGUMENT_POSITION,
       dspa.dspa_argument_type     AS ARGUMENT_TYPE,
       dsja.dsja_argument_value    AS VALUE,
       NULL::TEXT                  AS ANYDATA_VALUE,
       dspa.dspa_out_argument      AS OUT_ARGUMENT
FROM   sys.scheduler_0450_job_argument AS dsja, sys.scheduler_0400_job AS dsj, sys.scheduler_0250_program_argument AS dspa
WHERE  dsj.dsj_job_id = dsja.dsja_job_id AND
       dsj.dsj_program_id = dspa.dspa_program_id AND
       (dsja.dsja_argument_position = dspa.dspa_argument_position OR dsja.dsja_argument_name = dspa.dspa_argument_name);

CREATE OR REPLACE VIEW sys.all_scheduler_job_args AS
SELECT *
FROM   sys.dba_scheduler_job_args
WHERE  pg_has_role(owner, 'USAGE');

GRANT SELECT ON sys.all_scheduler_job_args TO PUBLIC;

CREATE OR REPLACE VIEW sys.user_scheduler_job_args AS
SELECT JOB_NAME, ARGUMENT_NAME, ARGUMENT_POSITION, ARGUMENT_TYPE, VALUE,
       ANYDATA_VALUE, OUT_ARGUMENT
FROM   sys.all_scheduler_job_args
WHERE  OWNER = pg_catalog.current_user;

GRANT SELECT ON sys.user_scheduler_job_args TO PUBLIC;
