/*
 * Decompiled with CFR 0.152.
 */
package com.tandbergtv.neptune.i18n.service.io;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.tandbergtv.neptune.i18n.service.bundle.ResourceBundleFileNameHelper;
import com.tandbergtv.neptune.i18n.service.io.FileEventHandler;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.log4j.Logger;

public class FileResourceWatcher {
    private Logger logger = Logger.getLogger(FileResourceWatcher.class);
    private ExecutorService threadPoolExecutor;
    private FileEventHandler resourceEventHandler;
    private List<String> watchPaths;
    private volatile boolean started;
    private Future<?> future;
    private Map<Path, FileTime> cachedFilesModifiedTimeMap;
    private static final long DEFAULT_FOLDER_SCAN_INTERVAL_MILLSECS = 30000L;
    private static final int MAX_DELAY_SCAN_COUNTER = 10;
    private long noFileChangeCounter;
    private long folderScanIntervalMillSecs;
    private int maxDelayScanCounter;

    public FileResourceWatcher(List<String> paths) {
        this(paths, Executors.newSingleThreadExecutor(), new FileEventHandler(), 30000L, 10);
    }

    FileResourceWatcher(List<String> paths, ExecutorService threapPoolExecutor, FileEventHandler eventHandler, long scanInterval, int maxDelayScanCounter) {
        this.threadPoolExecutor = threapPoolExecutor;
        this.resourceEventHandler = eventHandler;
        if (paths == null) {
            throw new IllegalArgumentException("Watch paths can be null");
        }
        this.watchPaths = paths;
        this.folderScanIntervalMillSecs = scanInterval;
        this.maxDelayScanCounter = maxDelayScanCounter;
    }

    private void initFilesModifiedTimeMapCache() {
        this.logger.debug((Object)"Scan file from watch paths ");
        this.cachedFilesModifiedTimeMap = this.getFileModifiedTimeMap();
    }

    private Map<Path, FileTime> getFileModifiedTimeMap() {
        HashMap fileModifiedTimeMap = Maps.newHashMap();
        for (String pathString : this.watchPaths) {
            Path scanPath = Paths.get(pathString, new String[0]);
            fileModifiedTimeMap.putAll(this.getFilesModifiedTimeUnderPath(scanPath));
        }
        return fileModifiedTimeMap;
    }

    private Map<Path, FileTime> getFilesModifiedTimeUnderPath(Path scanPath) {
        this.logger.debug((Object)("Scan files from path:" + scanPath));
        HashMap modifiedMap = Maps.newHashMap();
        if (!scanPath.toFile().isDirectory()) {
            return modifiedMap;
        }
        for (Path path : this.listFiles(scanPath)) {
            try {
                modifiedMap.put(path, Files.getLastModifiedTime(path, new LinkOption[0]));
            }
            catch (IOException e) {
                this.logger.warn((Object)("Failed to get file's modified time:" + path), (Throwable)e);
            }
        }
        return modifiedMap;
    }

    public List<Path> listFiles(Path scanPath) {
        ArrayList filesUnderPaths = Lists.newArrayList();
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(scanPath, this.getFilter());){
            for (Path path : directoryStream) {
                this.logger.debug((Object)("Adding file " + path.toString() + " to list"));
                filesUnderPaths.add(path);
            }
        }
        catch (IOException e) {
            this.logger.debug((Object)("Can't access file under folder :" + scanPath + " "), (Throwable)e);
        }
        catch (Exception ex) {
            this.logger.debug((Object)"Scan folder failed ", (Throwable)ex);
        }
        return filesUnderPaths;
    }

    private DirectoryStream.Filter<Path> getFilter() {
        return file -> {
            if (Files.isDirectory(file, new LinkOption[0])) {
                this.logger.debug((Object)("file " + file.toString() + " is a folder"));
                return false;
            }
            if (file.getFileName().toString().endsWith(".swp")) {
                this.logger.debug((Object)("file " + file.toString() + " is a swp file"));
                return false;
            }
            return ResourceBundleFileNameHelper.validateFileName(file.getFileName().toString());
        };
    }

    private boolean scanPathsForChange() throws IOException, InterruptedException {
        this.logger.debug((Object)"Scan file from watch paths ");
        Map<Path, FileTime> latestFileModifiedTimeMap = this.getFileModifiedTimeMap();
        ArrayList events = Lists.newArrayList();
        this.logger.debug((Object)("latestChangeMap size : " + latestFileModifiedTimeMap.size()));
        this.logger.debug((Object)("cachedFilesModifiedTimeMap size : " + this.cachedFilesModifiedTimeMap.size()));
        Sets.SetView filesMightBeModified = Sets.intersection(latestFileModifiedTimeMap.keySet(), this.cachedFilesModifiedTimeMap.keySet());
        Sets.SetView fileCreated = Sets.difference(latestFileModifiedTimeMap.keySet(), this.cachedFilesModifiedTimeMap.keySet());
        Sets.SetView fileDeleted = Sets.difference(this.cachedFilesModifiedTimeMap.keySet(), latestFileModifiedTimeMap.keySet());
        for (Path p2 : filesMightBeModified) {
            if (!this.isFileModified(latestFileModifiedTimeMap, p2)) continue;
            this.logger.debug((Object)("detected file modified : " + p2));
            events.add(new Event<Path>(StandardWatchEventKinds.ENTRY_MODIFY, p2));
            this.cachedFilesModifiedTimeMap.put(p2, latestFileModifiedTimeMap.get(p2));
        }
        for (Path p2 : fileCreated) {
            this.logger.debug((Object)("detected file created : " + p2));
            events.add(new Event<Path>(StandardWatchEventKinds.ENTRY_CREATE, p2));
            this.cachedFilesModifiedTimeMap.put(p2, latestFileModifiedTimeMap.get(p2));
        }
        ArrayList deletePaths = Lists.newArrayList();
        for (Path p3 : fileDeleted) {
            this.logger.debug((Object)("detected file delete : " + p3));
            events.add(new Event<Path>(StandardWatchEventKinds.ENTRY_DELETE, p3));
            deletePaths.add(p3);
        }
        deletePaths.forEach(p -> this.cachedFilesModifiedTimeMap.remove(p));
        this.logger.debug((Object)(events.size() + " events generated "));
        for (WatchEvent event : events) {
            try {
                this.logger.debug((Object)("handling event type:" + event.kind() + " for path " + ((Path)event.context()).toString()));
                this.resourceEventHandler.handleEvent(event);
            }
            catch (Exception e) {
                this.logger.error((Object)("handle event " + event + " failed, continue to the next event"), (Throwable)e);
            }
        }
        return !events.isEmpty();
    }

    private boolean isFileModified(Map<Path, FileTime> latestFileModifiedTimeMap, Path p) {
        return latestFileModifiedTimeMap.get(p).compareTo(this.cachedFilesModifiedTimeMap.get(p)) > 0;
    }

    public synchronized void start() {
        if (this.started) {
            return;
        }
        this.initFilesModifiedTimeMapCache();
        this.future = this.threadPoolExecutor.submit(() -> {
            while (!Thread.interrupted()) {
                try {
                    if (this.scanPathsForChange()) {
                        this.resetFileChangeCounter();
                    }
                    this.increaseFileChangeCounter();
                    Thread.sleep(this.getSleepTime());
                }
                catch (InterruptedException ex) {
                    this.logger.error((Object)"folder watcher was interrupted.", (Throwable)ex);
                    Thread.currentThread().interrupt();
                }
                catch (Exception exx) {
                    this.logger.error((Object)"failed to scan folder:", (Throwable)exx);
                }
            }
        });
        this.logger.debug((Object)"folder watcher started");
        this.started = true;
    }

    private void increaseFileChangeCounter() {
        if (this.noFileChangeCounter > (long)this.maxDelayScanCounter) {
            this.resetFileChangeCounter();
        }
        ++this.noFileChangeCounter;
    }

    private void resetFileChangeCounter() {
        this.noFileChangeCounter = 0L;
    }

    private long getSleepTime() {
        return this.noFileChangeCounter * this.folderScanIntervalMillSecs;
    }

    public synchronized void stop() {
        if (!this.started) {
            return;
        }
        this.future.cancel(true);
        this.threadPoolExecutor.shutdown();
        this.logger.debug((Object)"folder watcher stop");
        this.started = false;
    }

    public boolean isStarted() {
        return this.started;
    }

    private static class Event<T>
    implements WatchEvent<T> {
        private final WatchEvent.Kind<T> kind;
        private final T context;
        private int count;

        Event(WatchEvent.Kind<T> type, T context) {
            this.kind = type;
            this.context = context;
            this.count = 1;
        }

        @Override
        public WatchEvent.Kind<T> kind() {
            return this.kind;
        }

        @Override
        public T context() {
            return this.context;
        }

        @Override
        public int count() {
            return this.count;
        }

        void increment() {
            ++this.count;
        }
    }
}

