# this file manages the VMs created as part of the deployment
# VM Specific networking is also in this file
# General netwoking is managed in network.tf


# Create virtual machine as NFS server
resource "azurerm_virtual_machine" "nfs" {
    name                  = "nfs-${var.install_name}"
    location              = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name   = "${azurerm_resource_group.cms-rg.name}"
    network_interface_ids = ["${azurerm_network_interface.nfs_nic.id}"]
    vm_size               = "${var.dev_mode ? var.aws_instance_type_nfs_dev : var.aws_instance_type_nfs}"

    storage_os_disk {
        name              = "nfs-${var.install_name}-os-disk"
        caching           = "ReadWrite"
        create_option     = "FromImage"
        managed_disk_type = "${var.root_volume_type}"
#        disk_size_gb      = "${var.instance_block_size_cs}"
    }

    delete_os_disk_on_termination = "${var.delete_disks_on_termination}"

    storage_data_disk {
        name              = "nfs-${var.install_name}-data-disk"
        caching           = "ReadWrite"
        create_option     = "Empty"
        lun               = "3"
        managed_disk_type = "${var.block_volume_type}"
        disk_size_gb      = "${var.instance_block_size_nfs}"
    }

    delete_data_disks_on_termination = "${var.delete_nfs_disks_on_termination}"

    storage_image_reference {
        publisher = "${var.os_publisher}"
        offer     = "${var.os_offer}"
        sku       = "${var.os_sku}"
        version   = "${var.os_version}"
    }

    os_profile {
        computer_name  = "nfs-${var.install_name}"
        admin_username = "${var.ssh_user}"
        admin_password = "${var.ssh_user_password}"
    }

    os_profile_linux_config {
        disable_password_authentication = false
        ssh_keys {
            path     = "/home/${var.ssh_user}/.ssh/authorized_keys"
            key_data = "${file(var.public_key_path)}"
        }
    }

	tags = {
        environment = "${var.tag_environment}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo -S <<< ${var.ssh_user_password} sh -c \"echo -e '${var.ssh_user} ALL=(ALL) NOPASSWD: ALL\n' >> /etc/sudoers\"",
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            host = "${azurerm_public_ip.nfs_public_ip.fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
        connection {
            host = "${azurerm_public_ip.nfs_public_ip.fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }


    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            #"sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup space and NFS share for use by other vms
            "sudo mkdir -p /vol1",
            "sudo bash /opt/vmTools/mountVMdisk_azure.sh /vol1 \"0% 100%\"",
            "sudo mkdir -p /vol1/cms /vol1/content",
            
            "sudo bash /opt/vmTools/setupNFSServer.sh /vol1/cms",
            "sudo bash /opt/vmTools/setupNFSServer.sh /vol1/content",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh nfs-${var.install_name}",

            # have to tell the network NOT to create an /etc/resolv.conf based on what comes back from AWS DHCP - we need cms/ours not theirs
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #turn off SELinux
            "sudo setenforce 0",
            "sudo sed -i --follow-symlinks 's/^SELINUX=.*/SELINUX=disabled/g' /etc/sysconfig/selinux",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",
        ]
        connection {
            host = "${azurerm_public_ip.nfs_public_ip.fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }

}


# App Virtual NIC
resource "azurerm_network_interface" "nfs_nic" {
    name                = "nic-nfs-${var.install_name}"
    location            = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_id = "${azurerm_network_security_group.admin.id}"
    ip_configuration {
        name                          = "nic-nfs-${var.install_name}"
        subnet_id                     = "${azurerm_subnet.cms-subnet.id}"
        private_ip_address_allocation = "Dynamic"
        public_ip_address_id          = "${azurerm_public_ip.nfs_public_ip.id}"
    }

    tags = {
        environment = "${var.tag_environment}"
    }
}


# App public IPs
resource "azurerm_public_ip" "nfs_public_ip" {
    name                         = "public-ip-nfs-${var.install_name}"
    location                     = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
    allocation_method            = "Dynamic"
    domain_name_label             = "nfs-pub-${var.install_name}"


    tags = {
        environment = "${var.tag_environment}"
    }
}

output "nfs" {
    value = "${azurerm_public_ip.nfs_public_ip.*.ip_address}"
}

