# this file manages the VMs created as part of the deployment
# VM Specific networking is also in this file
# General netwoking is managed in network.tf


# Create virtual machine pt1
resource "azurerm_virtual_machine" "pt" {
    name                  = "pt${count.index + 1}-${var.install_name}"
    count = "${var.pt_count}"
    location              = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name   = "${azurerm_resource_group.cms-rg.name}"
    network_interface_ids = ["${azurerm_network_interface.pt_nic[count.index].id}"]
    vm_size               = "${var.dev_mode ? var.aws_instance_type_pt_dev : var.aws_instance_type_pt}"
    depends_on 		  = ["azurerm_virtual_machine.cs[0]"]
    availability_set_id   = "${azurerm_availability_set.cms-avset-pt[0].id}"

    storage_os_disk {
        name              = "pt${count.index + 1}-${var.install_name}-os-disk"
        caching           = "ReadWrite"
        create_option     = "FromImage"
        managed_disk_type = "${var.root_volume_type}"
#        disk_size_gb      = "${var.instance_block_size_app}"
    }

    delete_os_disk_on_termination = "${var.delete_disks_on_termination}"

    #storage_data_disk {
    #    name              = "pt${count.index + 1}-${var.install_name}-data-disk"
    #    caching           = "ReadWrite"
    #    create_option     = "Empty"
    #    lun               = "3"
    #    managed_disk_type = "${var.root_volume_type}"
    #    disk_size_gb      = "${var.instance_block_size_pt}"
    #}
#
    #delete_data_disks_on_termination = "${var.delete_disks_on_termination}"

    storage_image_reference {
        publisher = "${var.os_publisher}"
        offer     = "${var.os_offer}"
        sku       = "${var.os_sku}"
        version   = "${var.os_version}"
    }

    os_profile {
        computer_name  = "pt${count.index + 1}-${var.install_name}"
        admin_username = "${var.ssh_user}"
        admin_password = "${var.ssh_user_password}"
    }

    os_profile_linux_config {
        disable_password_authentication = false
        ssh_keys {
            path     = "/home/${var.ssh_user}/.ssh/authorized_keys"
            key_data = "${file(var.public_key_path)}"
        }
    }

	tags = {
        environment = "${var.tag_environment}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo -S <<< ${var.ssh_user_password} sh -c \"echo -e '${var.ssh_user} ALL=(ALL) NOPASSWD: ALL\n' >> /etc/sudoers\"",
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            host = "${azurerm_public_ip.pt_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
        connection {
            host = "${azurerm_public_ip.pt_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }
}

# Operations to be done on all pt nodes
resource "null_resource" "pt_installcms" {
    depends_on = ["azurerm_network_security_rule.pt_sr","null_resource.prepare_rpms"]
    count = "${var.pt_count}"

  # run this in all pt nodes
    connection {
        host = "${data.azurerm_public_ip.pt_pub_ip[count.index].ip_address}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

#            # create new disk space
#           "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",


            # setup for using nfs client
            "sudo systemctl enable rpcbind.service",
            "sudo systemctl start rpcbind",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_pt}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",

            #make create a settings file to override dbsettings
            "sudo mkdir -p /opt/db/scripts/install",
            "echo '#!/bin/bash' | sudo tee /opt/db/scripts/install/settings",
            "echo 'DB_ADMIN_PW=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'PGPD=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_PORT=${var.rds_db_port}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_TYPE=azure' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_USER=${var.rds_db_user}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_VERSION=${var.rds_db_version}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_NAME=${var.rds_db_name}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'HOST_NAME=dbserver' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'AZURE_DB=${var.install_name}edb11' | sudo tee -a /opt/db/scripts/install/settings",
            "sudo chown nobody:nobody /opt/db/scripts/install/settings",
            "sudo chmod 400 /opt/db/scripts/install/settings",

            #sleep to stagger pt instances
            "sleep ${270*count.index}",

            "sudo bash installcmsnode.sh -t 900 -n ${var.pt_count} pt${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
    }
}

# pt Virtual NIC
resource "azurerm_network_interface" "pt_nic" {
    name                = "nic-pt${count.index + 1}-${var.install_name}"
    location            = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_id = "${azurerm_network_security_group.admin.id}"
    count = "${var.pt_count}"
    ip_configuration {
        name                          = "nic-pt${count.index + 1}-${var.install_name}"
        subnet_id                     = "${azurerm_subnet.cms-subnet.id}"
        private_ip_address_allocation = "Dynamic"
        public_ip_address_id          = "${azurerm_public_ip.pt_public_ip[count.index].id}"
    }

    tags = {
        environment = "${var.tag_environment}"
    }
}

# Availability Set
resource "azurerm_availability_set" "cms-avset-pt" {
  name                         = "${var.install_name}-cms-availabilityset-pt"
  count 		               = "${ (var.pt_count == 0 ? 0 : 1) }"
  location                     = "${azurerm_resource_group.cms-rg.location}"
  resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
  platform_fault_domain_count  = "${var.platform_fault_domain_count}"
  platform_update_domain_count = "${var.platform_update_domain_count}"
  managed                      = true
}

# PT public IPs
resource "azurerm_public_ip" "pt_public_ip" {
    name                         = "public-ip-pt${count.index + 1}-${var.install_name}"
    count = "${var.pt_count}"
    location                     = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
    allocation_method            = "Dynamic"
    domain_name_label             = "pt${count.index + 1}-pub-${var.install_name}"


    tags = {
        environment = "${var.tag_environment}"
    }
}

data "azurerm_public_ip" "pt_pub_ip" {
  count               = "${var.pt_count}"
  name                = "${azurerm_public_ip.pt_public_ip[count.index].name}"
  resource_group_name = "${azurerm_virtual_machine.pt[count.index].resource_group_name}"
}

resource "azurerm_network_security_rule" "pt_sr" {
  count                       = "${var.pt_count}"
  depends_on = ["azurerm_virtual_machine.pt"]
  name                        = "sr-pt${count.index + 1}-${var.install_name}"
  priority                    = "${401 + count.index}"
  direction                   = "Inbound"
  access                      = "Allow"
  protocol                    = "*"
  source_port_range           = "*"
  destination_port_range      = "*"
  #source_address_prefix       = "${azurerm_public_ip.pt_public_ip[count.index].ip_address}"
  source_address_prefix       = "${data.azurerm_public_ip.pt_pub_ip[count.index].ip_address}"
  destination_address_prefix  = "*"
  resource_group_name         = "${azurerm_resource_group.cms-rg.name}"
  network_security_group_name = "admin_sg"
}

output "pt_public_ip_address" {
  value = "${data.azurerm_public_ip.pt_pub_ip.*.ip_address}"
}

#output "pt_nodes" {
#    value = "${azurerm_public_ip.pt_public_ip.*.ip_address}"
#}
