/*
-- On 10/12/2016 Converted the EPG database scripts from EDB to Postgres format by Suresh Neravati.
-- Verified by Kriss from CGC.
*/

CREATE OR REPLACE FUNCTION createobject(vsql varchar, errmsg varchar, mask varchar default 'N') RETURNS smallint AS $func_createobject$
DECLARE
  verrmsg varchar:='';
BEGIN
  --raise notice '%', vsql;
  EXECUTE vsql;
  return 0;
EXCEPTION 
  WHEN others THEN 
    select substring(errmsg,1,200) into verrmsg;
    verrmsg := sqlstate||' - Error : "'||verrmsg ||'", '||sqlerrm;
    if upper(mask) != 'Y' then
      INSERT INTO install_table values(now(),verrmsg);
      raise exception 'Error: % %',errmsg ||' '||sqlerrm, SQLSTATE;
      raise notice 'Error: % %',errmsg ||' '||sqlerrm, SQLSTATE;
    end if;
    return 1;
END;
$func_createobject$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION RenameIndex(vschema varchar, orgIndexName varchar, newIndexName varchar) RETURNS smallint AS $func_RenameIndex$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter index '||vschema||'.' || orgIndexName || ' rename to '|| newIndexName;
    msg := 'rename index ' || orgIndexName || ' to '|| newIndexName;
    
    return createObject(sql, msg); 
end;
$func_RenameIndex$ language plpgsql;
  
  
CREATE OR REPLACE FUNCTION DropTable(vschema varchar, tableName text) RETURNS smallint AS $func_DropTable$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'drop table '|| vschema||'.'||tableName ||' cascade';
    msg := 'drop table '||tableName ;
    
    return createObject(sql, msg); 
end;
$func_DropTable$ language plpgsql;
  
  
CREATE OR REPLACE FUNCTION AddColumn(vschema varchar, tableName varchar, columnName varchar, dataType varchar) RETURNS smallint AS $func_AddColumn$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter table '||vschema||'.'||tableName||' add '||columnName||' '||dataType;
    msg := 'add column '||columnName||' in '||tableName ;
    
    return createObject(sql, msg); 
end;
$func_AddColumn$ language plpgsql;
 
    
CREATE OR REPLACE FUNCTION DropColumn(vschema varchar, tableName varchar, columnName varchar) RETURNS smallint AS $func_DropColumn$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter table '||vschema||'.'||tableName||' drop '||columnName;
    msg := 'drop column '||columnName||' in '||tableName ;
    
    return createObject(sql, msg); 
end;
$func_DropColumn$ language plpgsql;
   
    
CREATE OR REPLACE FUNCTION DropNotNullConstraint(vschema varchar, tableName varchar, columnWithNotNull varchar) RETURNS smallint AS $func_DropNotNullConstraint$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter table '||vschema||'.'||tableName||' alter column '||columnWithNotNull||'  drop NOT NULL';
    msg := 'drop NOT NULL constraint on '||columnWithNotNull||' in '||tableName ;
    
    return createObject(sql, msg); 
end;
$func_DropNotNullConstraint$ language plpgsql;
  

CREATE OR REPLACE FUNCTION DropConstraint(vschema varchar, tableName varchar, constraintName varchar, isCascade varchar default 'Y') RETURNS smallint AS $func_DropConstraint$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter table '||vschema||'.'||tableName||' drop constraint if exists '||constraintName ||' ';
    if upper(isCascade) = 'Y' then
       sql := sql || ' cascade';
    end if;
    
    msg := 'drop constraint ' || constraintName || ' of '|| tableName ;
    
    return createObject(sql, msg); 
end;
$func_DropConstraint$ language plpgsql;
  

CREATE OR REPLACE FUNCTION AddPrimaryKey(vschema varchar, tableName varchar, keyName varchar, keys varchar) RETURNS smallint AS $func_AddPrimaryKey$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter table '||vschema||'.'||tableName||' add constraint '||keyName||' primary key('||keys||')';
    msg := 'add primary key '||keyName||' to '||tableName ;
    
    return createObject(sql, msg); 
end;
$func_AddPrimaryKey$ language plpgsql;
    
  
  
CREATE OR REPLACE FUNCTION AddForeignKey(vschema varchar, tableName varchar, refTable varchar, keyName varchar, keys varchar) RETURNS smallint AS $func_AddForeignKey$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql := 'alter table '||vschema||'.'||tableName||' add constraint '||keyName||' foreign key('||keys||') references '||vschema||'.'||refTable ||' ('||keys||')';
    msg := 'add foreign key '||keyName||' to '||tableName ;
    
	return createObject(sql, msg); 
end;
$func_AddForeignKey$ language plpgsql;
      
  
CREATE OR REPLACE FUNCTION CreateIndex(vschema varchar, tableName varchar, columnName varchar, indexName varchar) RETURNS smallint AS $func_CreateIndex$
declare
    sql varchar(32000);
begin
    sql:='create index ' || indexName || ' ON '||VSCHEMA||'.' || tableName || '(' || columnName || ')';  
    return createObject(sql, 'create index ' || indexName); 
end;
$func_CreateIndex$ language plpgsql;
 
  
CREATE OR REPLACE FUNCTION CreateUniqueIndex(vschema varchar, tableName varchar, columnName varchar, indexName varchar) RETURNS smallint AS $func_CreateUniqueIndex$
declare
    sql varchar(32000);
    msg varchar(200);
begin
    sql:='create unique index ' || indexName || ' ON '||VSCHEMA||'.' || tableName || '(' || columnName || ')';  
    return createObject(sql, 'create unique index ' || indexName); 
end;
$func_CreateUniqueIndex$ language plpgsql;


CREATE OR REPLACE FUNCTION DropIndex(vschema varchar, indexName varchar) RETURNS smallint AS $func_DropIndex$
declare
    sql varchar(32000);
begin
    sql:='drop index if exists ' || indexName;  
	return createObject(sql, 'drop index ' || indexName); 
end;
$func_DropIndex$ language plpgsql;
  

/* Updates the UpdateTime column with the current date and time when any
   update statement is issued against the table.
*/  
CREATE OR REPLACE FUNCTION createupdatetrg(vschema VARCHAR, vtabname VARCHAR) RETURNS smallint AS $func_createupdatetrg$
DECLARE
    verrmsg varchar:='';
    vcount  smallint;
    vsql    varchar;
BEGIN
    EXECUTE 'SELECT count(*) FROM information_schema.columns WHERE table_schema = lower('''||vschema||''') AND upper(table_name) = upper('''||vtabname||''') AND upper(column_name) = ''UPDATE_DATE''' INTO vcount;
    if ( vcount > 0 ) then
      vsql := 'CREATE OR REPLACE FUNCTION FUNC_'||substr(vtabname,1,26)||'UPD() RETURNS TRIGGER AS $$ BEGIN NEW.UPDATE_DATE = now(); RETURN NEW; END; $$ language plpgsql';
      EXECUTE vsql;
      vsql := 'CREATE TRIGGER '||substr(vtabname,1,26)||'UPD BEFORE UPDATE ON '||vtabname||' FOR EACH ROW EXECUTE PROCEDURE  FUNC_'||substr(vtabname,1,26)||'UPD()';
      EXECUTE vsql;
   end if;
   return 0;
EXCEPTION
   WHEN others THEN
      verrmsg := sqlstate||' - Error : While creating trigger, '||sqlerrm;
      raise notice 'Error: % ',verrmsg;
      raise exception 'Error: % ',verrmsg;
      return 1;
  END;
$func_createupdatetrg$ language plpgsql;
  
  
CREATE OR REPLACE FUNCTION InsertEntityMapping(vschema varchar, id integer, mapname varchar, maptable varchar, mapcolumn varchar, maptype varchar,mapcriteria integer, mappredicate integer, displayorder integer, max_length integer, nullable integer) RETURNS smallint AS $func_InsertEntityMapping$
declare
    vsql varchar(32000);
    verrmsg varchar;
begin
      vsql :='INSERT INTO '||vschema||'.ENTITY_MAPPING(ID, NAME, TABLE_NAME, COLUMN_NAME, DATA_TYPE, ALLOW_AS_CRITERIA, ALLOW_AS_PREDICATE, DISP_ORDER, MAX_LENGTH, NULLABLE) values ('||id||','''||mapname||''', '''||maptable||''','''||mapcolumn||''','''||maptype||''','||mapcriteria||','||mappredicate||','||displayorder||','||max_length||','||nullable||')';
      execute vsql;
   return 0;
EXCEPTION
   WHEN others THEN
      verrmsg := sqlstate||' - Error : While creating trigger, '||sqlerrm;
      raise notice 'Error: % ',verrmsg;
      raise exception 'Error: % ',verrmsg;
      return 1;
end;
$func_InsertEntityMapping$ language plpgsql;



CREATE OR REPLACE FUNCTION UpdateEntityMapping(vschema varchar, id smallint, max_length smallint, nullable smallint) RETURNS smallint AS $func_UpdateEntityMapping$
DECLARE
    vsql varchar;
    tmp_result varchar;
BEGIN
    vsql :='UPDATE '||vschema||'.ENTITY_MAPPING SET MAX_LENGTH = '||max_length||' WHERE ID = '|| id ;
    SELECT createObject(vsql,'UPDATE EntityMapping max_length') INTO tmp_result;
     
    vsql :='UPDATE '||vschema||'.ENTITY_MAPPING SET NULLABLE = '||nullable||' WHERE ID = '|| id ;
	return createObject(vsql,'UPDATE EntityMapping nullable'); 
END;
$func_UpdateEntityMapping$ LANGUAGE plpgsql;  
  
 
CREATE OR REPLACE FUNCTION create_epgmanager(vschema VARCHAR, epg_db_version VARCHAR) RETURNS smallint AS $func_create_epgmanager$
DECLARE
    vsql          varchar;
    wk_message    varchar(500);
    tmp_result    smallint;
    errcnt        smallint;
BEGIN
        wk_message := '';
        -- Create install_table
        vsql:='CREATE TABLE '||VSCHEMA||'.install_table (msgdate date default current_timestamp, install_message varchar(4000))';
        SELECT createObject(vsql,'CREATE install_table') INTO tmp_result;
        vsql:='CREATE TABLE '||VSCHEMA||'.SEQ_GENERATOR
               ( SEQ_NAME varchar(50) NOT NULL,
               SEQ_NUMBER numeric(19,0) NOT NULL,
               CONSTRAINT PK_SEQ_GENERATOR PRIMARY KEY (SEQ_NAME)
        )';
        SELECT createObject(vsql,'Create Table SEQ_GENERATOR','Y') INTO tmp_result;

        -- Epg_Format 
        vsql:='CREATE TABLE '||VSCHEMA||'.EPG_FORMAT(
              ID varchar(100) NOT NULL, 
              NAME varchar(50) NOT NULL, 
              JAR_NAME varchar(255),
              TYPE varchar(50) NOT NULL,
              INFO varchar(255),
              VERSION varchar(255),
              CREATED_BY varchar (50)  NOT NULL ,
              UPDATED_BY varchar (50)  NOT NULL ,
              CREATE_DATE TIMESTAMP  NOT NULL ,
              UPDATE_DATE TIMESTAMP  NOT NULL, 
              STATE numeric(1,0) DEFAULT 2 NOT NULL,
              CONSTRAINT PK_EPG_FORMAT_ID PRIMARY KEY(ID))'; 
        SELECT createObject(vsql,'Create Table EPG_FORMAT')  INTO tmp_result;
           
        -- Endpoint
         vsql:='CREATE TABLE '||VSCHEMA||'.ENDPOINT(
            ID numeric(19, 0) NOT NULL,
            TASK_ID numeric(19, 0) NOT NULL,
            FEATURE_TYPE varchar(20) NOT NULL,
            ENDPOINT_TYPE varchar(20) NOT NULL,
            SETTING varchar(4000),
            SEQ_ORDER numeric(7,0),
            FEATURE_CODE varchar(500),
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL,
            CONSTRAINT PK_ENDPOINT_ID PRIMARY KEY(ID) )';
        SELECT createObject(vsql,'Create Table ENDPOINT') INTO tmp_result;

 
        vsql:='CREATE TABLE '||VSCHEMA||'.NORM_RULE_GROUP
        (
            ID numeric(19,0) NOT NULL,
            NAME varchar(200) NOT NULL,
            ENABLED numeric(1,0) NOT NULL,
            IS_FOR_INGEST numeric(1,0) NOT NULL DEFAULT 1,
            GLOBAL_FOR_PROVIDERS numeric(1,0) NOT NULL,
            GLOBAL_FOR_PUBLISHERS numeric(1,0) NOT NULL,
            ORDERING    numeric(19,0) NOT NULL,
            CREATED_BY      varchar(50) NOT NULL,
            UPDATED_BY      varchar(50) NOT NULL,
            CREATE_DATE     TIMESTAMP(0) WITH TIME ZONE DEFAULT (CURRENT_TIMESTAMP) NOT NULL,
            UPDATE_DATE     TIMESTAMP(0) WITH TIME ZONE DEFAULT (CURRENT_TIMESTAMP) NOT NULL,
            PROVIDER_ID     numeric(19,0),
            UUID varchar(128),
            CONSTRAINT PK_NORM_RULE_GROUP_ID PRIMARY KEY(ID),
            CONSTRAINT U_NORM_RULE_GROUP_NAME UNIQUE (NAME),
			CONSTRAINT U_NORM_RULE_GROUP_UUID UNIQUE (UUID)
        )';
        SELECT createObject(vsql,'create Table NORM_RULE_GROUP') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.VALID_RULE_GROUP
        (
            ID numeric(19,0) NOT NULL,
            NAME varchar(200) NOT NULL,
            ENABLED numeric(1,0) NOT NULL,
            GLOBAL numeric(1,0) NOT NULL,
            CREATED_BY      varchar(50) NOT NULL,
            UPDATED_BY      varchar(50) NOT NULL,
            CREATE_DATE     TIMESTAMP(0) WITH TIME ZONE DEFAULT (CURRENT_TIMESTAMP) NOT NULL,
            UPDATE_DATE     TIMESTAMP(0) WITH TIME ZONE DEFAULT (CURRENT_TIMESTAMP) NOT NULL,
            UUID varchar(128),
            CONSTRAINT PK_VALID_RULE_GROUP_ID PRIMARY KEY(ID),
            CONSTRAINT U_VALID_RULE_GROUP_NAME UNIQUE (NAME)
        )';
        SELECT createObject(vsql,'create table VALID_RULE_GROUP') INTO tmp_result;
                 
        vsql:='CREATE TABLE '||VSCHEMA||'.PARTNER
        (
            ID numeric(19, 0) NOT NULL,
            NAME varchar(200) NOT NULL,
            CONTACT varchar(200),
            EMAIL varchar(100),
            PARTNER_TYPE CHAR(8), 
            IS_ACTIVE numeric(1,0) NOT NULL,
            RECEIVE_WARNING numeric(1,0),
            SETTING varchar(2000),
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL,  
            UUID varchar(128),
            CONSTRAINT U_PARTNER_NAME UNIQUE (NAME), 
            CONSTRAINT U_PARTNER_UUID UNIQUE (UUID),
            CONSTRAINT PK_PARTNER_ID PRIMARY KEY (ID)
        )';
        SELECT createObject(vsql,'Create table PARTNER') INTO tmp_result;
        
                   
        --Provider reference table end
            
        -- stage tables 
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_HEADEND
        (   
            FILENAME varchar(4000),
            HEADEND_ID varchar(100),
            COMMUNITY_NAME varchar(200),
            COUNTY_NAME varchar(200),
            COUNTY_SIZE varchar(1),
            ST_COUNTY_CODE varchar(100),
            STATE varchar(64),
            ZIPCODE varchar(100),
            DMA_CODE varchar(100),
            DMA_NAME varchar(300),
            MSO_CODE numeric,
            DMA_RANK varchar(100),
            HEADEND_NAME varchar(1000),
            HEADEND_LOCATION varchar(100),
            MSO_NAME varchar(1000),
            TIME_ZONE varchar(100),
            HEADEND_TYPE varchar(100),
            HEADEND_COUNTRY varchar(100),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_STG_HEADEND PRIMARY KEY (PROVIDER_ID, HEADEND_ID)
        )';
        SELECT createObject(vsql,'create table stg_headend') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_HEADEND', 'PROVIDER_ID', 'IDX_STG_HEADEND') INTO tmp_result;
            
                
        -- stg_headend generic 
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_HEADEND_GENERIC
        (    
             FILENAME varchar(4000),
             HEADEND_ID varchar(100) NOT NULL, 
             ASSET varchar(50) NULL, 
             NAME varchar(200) NULL, 
             VALUE varchar(2000) NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT STG_HEADEND_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, HEADEND_ID, ASSET, NAME) 
        )';
        SELECT createObject(vsql,'create table STG_HEADEND_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_HEADEND_GENERIC', 'PROVIDER_ID', 'IDX_STG_HEADEND_GENERIC') INTO tmp_result;
               
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_STATION
        (   FILENAME varchar(4000),
            STATION_ID varchar(100),
            TIME_ZONE varchar(100),
            STATION_NAME varchar(200),
            CALL_SIGN varchar(100),
            AFFILIATION varchar(100),
            CITY varchar(100),
            STATE varchar(100),
            ZIPCODE varchar(100),
            COUNTRY varchar(100),
            DMA_NAME varchar(200),
            DMA_NUMBER numeric(12),
            FCC_CHANNEL_NUM numeric(12),
            PROGRAM_GENRE varchar(200),
            USER_DATA1 varchar(1000),
            USER_DATA2 varchar(1000),
            USER_DATA3 varchar(1000),
            USER_DATA4 varchar(1000),
            USER_DATA5 varchar(1000),
            USER_DATA6 varchar(1000),
            USER_DATA7 varchar(1000),
            USER_DATA8 varchar(1000),
            USER_DATA9 varchar(1000),
            USER_DATA10 varchar(1000),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_STG_STATION PRIMARY KEY (PROVIDER_ID, STATION_ID)
        )';
        SELECT createObject(vsql,'create table stg_station') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_STATION', 'PROVIDER_ID', 'IDX_STG_STATION') INTO tmp_result;
        
        -- stg_station generic 
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_STATION_GENERIC
        (    
            FILENAME varchar(4000),
            STATION_ID varchar(100) NOT NULL, 
            ASSET varchar(50) NULL, 
            NAME varchar(200) NULL, 
            VALUE varchar(2000) NULL, 
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT STG_STATION_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, STATION_ID, ASSET, NAME) 
        )';
        SELECT createObject(vsql,'create table STG_STATION_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_STATION_GENERIC', 'PROVIDER_ID', 'IDX_STG_STATION_GENERIC') INTO tmp_result;

                
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_LINEUP
        (   
            FILENAME varchar(4000),
            LINEUP_ID varchar(100),
            STATION_ID varchar(100),
            HEADEND_ID varchar(100),
            DEVICE varchar(2),
            TMS_CHANNEL varchar(20),
            SERVICE_TIER varchar(100),
            EFFECTIVE_DATE TIMESTAMP (0)  WITH TIME ZONE,
            EXPIRATION_DATE TIMESTAMP (0) WITH TIME ZONE,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_STG_LINEUP PRIMARY KEY (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID),
            CONSTRAINT FK_STG_LINEUP_STATION FOREIGN KEY (PROVIDER_ID, STATION_ID) REFERENCES '||VSCHEMA||'.STG_STATION(PROVIDER_ID, STATION_ID),
            CONSTRAINT FK_STG_HEADEND_LINEUP FOREIGN KEY (PROVIDER_ID, HEADEND_ID) REFERENCES '||VSCHEMA||'.STG_HEADEND(PROVIDER_ID, HEADEND_ID)
        )';
        SELECT createObject(vsql,'create table stg_lineup') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_LINEUP', 'PROVIDER_ID', 'IDX_STG_LINEUP') INTO tmp_result;
            
             -- stg_lineup generic
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_LINEUP_GENERIC
               (    FILENAME varchar(4000),
                 LINEUP_ID varchar(100) NOT NULL, 
                 HEADEND_ID varchar(100) NOT NULL,
                 STATION_ID varchar(100) NOT NULL,
                 ASSET varchar(50) NULL, 
                 NAME varchar(200) NULL, 
                 VALUE varchar(2000) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_LINEUP_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID, ASSET, NAME) 
               )';
        SELECT createObject(vsql,'create table STG_LINEUP_GENERIC') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_LINEUP_GENERIC', 'PROVIDER_ID', 'IDX_STG_LINEUP_GENERIC') INTO tmp_result;
            
               
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM
            (   FILENAME varchar(4000),
              PROGRAM_ID varchar(100),
              REF_STATION_ID varchar (100),
              REF_START_TIME TIMESTAMP (0) WITH TIME ZONE,
              LANGUAGE varchar(100),
              TITLE varchar(1000),
              REDUCED_TITLE1 varchar(1000),
              REDUCED_TITLE2 varchar(1000),
              REDUCED_TITLE3 varchar(1000),
              REDUCED_TITLE4 varchar(1000),
              ALT_TITLE varchar(1000),
              REDUCED_DESCRIPTION1 varchar(1000),
              REDUCED_DESCRIPTION2 varchar(1000),
              REDUCED_DESCRIPTION3 varchar(1000),
              REDUCED_DESCRIPTION4 varchar(1000),
              ADVISORY_DESC1 varchar(100),
              ADVISORY_DESC2 varchar(100),
              ADVISORY_DESC3 varchar(100),
              ADVISORY_DESC4 varchar(100),
              ADVISORY_DESC5 varchar(100),
              ADVISORY_DESC6 varchar(100),
              GENRE_DESC1 varchar(100),
              GENRE_DESC2 varchar(100),
              GENRE_DESC3 varchar(100),
              GENRE_DESC4 varchar(100),
              GENRE_DESC5 varchar(100),
              GENRE_DESC6 varchar(100),
              DESCRIPTION1 varchar(2000),
              DESCRIPTION2 varchar(2000),
              YEAR_PROGRAM varchar(100),
              MPAA_RATING varchar(1000),
              STAR_RATING varchar(1000),
              RUN_TIME varchar(100),
              COLOR_CODE varchar(100),
              PROGRAM_LANGUAGE varchar(100),
              COUNTRY_OF_ORIGIN varchar(100),
              MADE_FOR_TV varchar(1),
              SOURCE_TYPE varchar(100),
              SHOW_TYPE varchar(200),
              HOLIDAY varchar(100),
              SYNDICATE_EPISODE_NUM varchar(100),
              ALT_SYNDICATE_EPI_NUM varchar(100),
              EPISODE_TITLE varchar(1000),
              NET_SYN_SOURCE varchar(100),
              NET_SYN_TYPE varchar(100),
              ORG_STUDIO varchar(100),
              GAME_DATETIME TIMESTAMP (0) WITH TIME ZONE,
              GAME_TIMEZONE varchar(100),
              ORG_AIR_DATE TIMESTAMP (0) WITH TIME ZONE,
              UNIQUE_ID varchar(100),
              SEASON_ID varchar(100),
              SERIES_ID varchar(100),
              USER_DATA1 varchar(1000),
              USER_DATA2 varchar(1000),
              USER_DATA3 varchar(1000),
              USER_DATA4 varchar(1000),
              USER_DATA5 varchar(1000),
              USER_DATA6 varchar(1000),
              USER_DATA7 varchar(1000),
              USER_DATA8 varchar(1000),
              USER_DATA9 varchar(1000),
              USER_DATA10 varchar(1000),
              USER_DATA11 varchar(1000),
              USER_DATA12 varchar(1000),
              USER_DATA13 varchar(1000),
              PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT PK_STG_PROGRAM PRIMARY KEY (PROVIDER_ID, PROGRAM_ID)
            )';
        SELECT createObject(vsql,'create table STG_PROGRAM') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_PROGRAM', 'PROVIDER_ID', 'IDX_STG_PROGRAM') INTO tmp_result;
    
          -- stg_Program Rating
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_RATING
               (    FILENAME varchar(4000),
                 PROGRAM_ID varchar(100) NOT NULL, 
                 AUTHORITY varchar(200) NULL, 
                 VALUE varchar(2000) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT PK_STG_PROGRAM_RATING   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, AUTHORITY) 
                )';
        SELECT createObject(vsql,'create table STG_PROGRAM_RATING') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_PROGRAM_RATING', 'PROVIDER_ID', 'IDX_STG_PROGRAM_RATING') INTO tmp_result;
            
            -- stg_program generic 
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_GENERIC
               (    FILENAME varchar(4000),
                 PROGRAM_ID varchar(100) NOT NULL, 
                 ASSET varchar(50) NULL, 
                 NAME varchar(200) NULL, 
                 VALUE varchar(2000) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_PROGRAM_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, ASSET, NAME) 
               )';
        SELECT createObject(vsql,'create table STG_PROGRAM_GENERIC') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_PROGRAM_GENERIC', 'PROVIDER_ID', 'IDX_STG_PROGRAM_GENERIC') INTO tmp_result;
            
            
           -- stg_program audio component
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_AUDIO_COMP
               (    FILENAME varchar(4000),
                 PROGRAM_ID varchar(100) NOT NULL, 
                 TLA varchar(600) NOT NULL,
                 TYPE varchar(200) NULL, 
                 LANGUAGE varchar(200) NULL, 
                 ATTRIBUTE_TYPE varchar(200) NULL,
                 VALUE varchar(200) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_PROGRAM_AUDIO_COMP_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, TLA) 
               )';
        SELECT createObject(vsql,'create table STG_PROGRAM_AUDIO_COMP') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_PROGRAM_AUDIO_COMP', 'PROVIDER_ID', 'IDX_STG_PROGRAM_AUDIO_COMP') INTO tmp_result;
            
            
           -- stg_program subtitle component
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_SUBTITLE_COMP
               (    FILENAME varchar(4000),
                 PROGRAM_ID varchar(100) NOT NULL, 
                 TLA varchar(600) NOT NULL,
                 TYPE varchar(200) NULL, 
                 LANGUAGE varchar(200) NULL, 
                 ATTRIBUTE_TYPE varchar(200) NULL,
                 VALUE varchar(200) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_PROGRAM_SUBTITLE_COMP_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID,TLA) 
               )';
        SELECT createObject(vsql,'create table STG_PROGRAM_SUBTITLE_COMP') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_PROGRAM_SUBTITLE_COMP', 'PROVIDER_ID', 'IDX_STG_PROGRAM_SUBTITLE_COMP') INTO tmp_result;
            

           vsql:='CREATE TABLE '||VSCHEMA||'.STG_SCHEDULE
               (    FILENAME varchar(4000),
              SCHEDULE_ID varchar (100), 
              STATION_ID varchar (100),
              PROGRAM_ID varchar(100),
              SCHEDULE_LANGUAGE varchar(100),
              START_TIME TIMESTAMP(0) with time zone,
              DURATION numeric,
              REF_STATION_ID varchar (100),
              REF_START_TIME TIMESTAMP (0) WITH TIME ZONE,
              PART_NUMBER numeric,
              SERIES_ID varchar(200),
              NO_OF_PARTS numeric,
              CC varchar(1),
              STEREO varchar(1),
              REPEAT varchar(1),
              LIVE_TAPE_DELAY varchar(100),
              SUBTITLED varchar(1),
              PREMIERE_FINALE varchar(100),
              JOINED_IN_PROGRESS varchar(1),
              CABLE_IN_CLASSROOM varchar(1),
              TV_RATING varchar(1000),
              SAP varchar(1),
              BLACKOUT varchar(1),
              SEX_RATING varchar(1),
              VIOLENCE_RATING varchar(1),
              LANGUAGE_RATING varchar(1),
              DIALOG_RATING varchar(1),
              FV_RATING varchar(1),
              ENHANCED varchar(1),
              THREE_D varchar(1),
              LETTERBOX varchar(1),
              HD_TV varchar(1),
              DOLBY varchar(100),
              DVS varchar(1),
              USER_DATA1 varchar(1000),
              USER_DATA2 varchar(1000),
              USER_DATA3 varchar(1000),
              USER_DATA4 varchar(1000),
              USER_DATA5 varchar(1000),
              USER_DATA6 varchar(1000),
              USER_DATA7 varchar(1000),
              USER_DATA8 varchar(1000),
              USER_DATA9 varchar(1000),
              USER_DATA10 varchar(1000),
              USER_DATA11 varchar(1000),
              REQUEST_RECORD varchar(1),
              PROVIDER_ID numeric(19,0) NOT NULL
               )';
        SELECT createObject(vsql,'create table STG_SCHEDULE') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_SCHEDULE', 'PROVIDER_ID', 'IDX_STG_SCHEDULE') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'STG_SCHEDULE', 'SCHEDULE_ID, PROVIDER_ID', 'IDX_STG_SCHED_PROVID_SCHEDID') INTO tmp_result;
            
            --stg_schedule_event_id 
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_SCHEDULE_EVENT_ID( FILENAME varchar(4000),
                 PROGRAM_ID varchar(100), 
                 IDENTIFIER_TYPE varchar(200) NULL, 
                 VALUE varchar(200) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_SCHEDULE_EVENT_IDS_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, IDENTIFIER_TYPE) 
                )';
        SELECT createObject(vsql,'create table STG_SCHEDULE_EVENT_ID') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_SCHEDULE_EVENT_ID', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_EVENT_ID') INTO tmp_result;
            
            
            -- stg_schedule_series_attribute 
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_SCHEDULE_SERIES_ATTRIB(FILENAME varchar(4000),
                 PROGRAM_ID varchar(100), 
                 SERIES_ID varchar(100), 
                 ATTRIBUTE_TYPE varchar(200) NULL, 
                 VALUE varchar(2000) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_SCHEDULE_SERIES_ATTRIBS_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, ATTRIBUTE_TYPE) 
                )';
        SELECT createObject(vsql,'create table STG_SCHEDULE_SERIES_ATTRIB') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_SCHEDULE_SERIES_ATTRIB', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_SERIES_ATTRIB') INTO tmp_result;
            
         
           -- stg_schedule generic 
            vsql:='CREATE TABLE '||VSCHEMA||'.STG_SCHEDULE_GENERIC
               (    FILENAME varchar(4000),
                 SCHEDULE_ID varchar(100), 
                 ASSET varchar(50) NULL, 
                 NAME varchar(200) NULL, 
                 VALUE varchar(2000) NULL, 
                 PROVIDER_ID numeric(19,0) NOT NULL,
                 CONSTRAINT STG_SCHEDULE_GENERIC_PK   PRIMARY KEY (SCHEDULE_ID, PROVIDER_ID, ASSET, NAME) 
               )';
        SELECT createObject(vsql,'create table STG_SCHEDULE_GENERIC') INTO tmp_result;
            
        SELECT CreateIndex(vschema,'STG_SCHEDULE_GENERIC', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_GENERIC') INTO tmp_result;
            
            
            -- EPGMANAGER Production tables
            vsql:='CREATE TABLE '||VSCHEMA||'.CONFIG_SETTING
            (
             KEY varchar (1024)  NOT NULL ,
             VALUE varchar (2000) ,
             CREATED_BY varchar (50)  NOT NULL ,
             UPDATED_BY varchar (50)  NOT NULL ,
             CREATE_DATE TIMESTAMP DEFAULT (CURRENT_TIMESTAMP) NOT NULL ,
             UPDATE_DATE TIMESTAMP DEFAULT (CURRENT_TIMESTAMP) NOT NULL,
             CONSTRAINT PK_CONFIG_SETTING_EPG PRIMARY KEY ( KEY )
            )';
        SELECT createObject(vsql,'create TABLE CONFIG_SETTING') INTO tmp_result;
            
            vsql:='CREATE TABLE '||VSCHEMA||'.ALARM_ACTIVITYLOG
            (
              ID            numeric(19,0) NOT NULL,
              SEVERITY      varchar(64) NOT NULL,
              COMPONENT     varchar(64) NOT NULL,
              DESCRIPTION   varchar(2000) NOT NULL,    
              CREATED_BY    varchar(50) NOT NULL,
              UPDATED_BY    varchar(50) NOT NULL,
              CREATE_DATE   TIMESTAMP(0) WITH TIME ZONE NOT NULL,
              UPDATE_DATE   TIMESTAMP(0) WITH TIME ZONE NOT NULL,
              SOURCE_ID     numeric(19,0),
              SOURCE_TYPE   varchar(20), 
              CONSTRAINT PK_ALARM_ACTIVITYLOG_EPG PRIMARY KEY(ID)
              )';
        SELECT createObject(vsql,'create table ALARM_ACTIVITYLOG') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'ALARM_ACTIVITYLOG', 'COMPONENT', 'IDX_AAL_COMPONENT') INTO tmp_result;
        SELECT CreateIndex(vschema,'ALARM_ACTIVITYLOG', 'SEVERITY', 'IDX_AAL_SEVERITY') INTO tmp_result;
        SELECT CreateIndex(vschema,'ALARM_ACTIVITYLOG', 'CREATE_DATE', 'IDX_AAL_CREATE_DATE') INTO tmp_result;
        
           -- Rules related table creation start
/*
           vsql:='CREATE TABLE '||VSCHEMA||'.ENTITY_MAPPING 
           (  
             ID NUMERIC(19,0) NOT NULL, 
             NAME varchar(200) NOT NULL, 
             TABLE_NAME varchar(200) NOT NULL, 
             COLUMN_NAME varchar(200) NOT NULL, 
             ALT_COLUMN_NAME1 varchar(200) NULL, 
             ALT_COLUMN_VALUE1 varchar(200) NULL, 
             ALT_COLUMN_NAME2 varchar(200) NULL, 
             ALT_COLUMN_VALUE2 varchar(200) NULL, 
             DATA_TYPE varchar(10) NULL, 
             ALLOW_AS_CRITERIA numeric(1,0) DEFAULT 1 NOT NULL,
             ALLOW_AS_PREDICATE numeric(1,0)DEFAULT 1 NOT NULL,
             MAX_LENGTH numeric(4,0) DEFAULT 1 NOT NULL,
             DISP_ORDER numeric(3,0) DEFAULT 0 NOT NULL,
             NULLABLE numeric(1,0) DEFAULT 1 NOT NULL,
             PARTNER_ID numeric(19,0) DEFAULT -1 NOT NULL,
             DISPLAY_NAME varchar(200), 
             CONSTRAINT Entity_Mapping_PK   PRIMARY KEY (ID) 
           )'; 
*/
           -- Rules related table creation start
           vsql:='CREATE TABLE '||VSCHEMA||'.ENTITY_MAPPING 
           (  
             ID NUMERIC(19) NOT NULL, 
             NAME varchar(200) NOT NULL, 
             TABLE_NAME varchar(200) NOT NULL, 
             COLUMN_NAME varchar(200) NOT NULL, 
             ALT_COLUMN_NAME1 varchar(200) NULL, 
             ALT_COLUMN_VALUE1 varchar(200) NULL, 
             ALT_COLUMN_NAME2 varchar(200) NULL, 
             ALT_COLUMN_VALUE2 varchar(200) NULL, 
             DATA_TYPE varchar(10) NULL, 
             ALLOW_AS_CRITERIA numeric(1) DEFAULT 1 NOT NULL,
             ALLOW_AS_PREDICATE numeric(1)DEFAULT 1 NOT NULL,
             MAX_LENGTH numeric(4) DEFAULT 1 NOT NULL,
             DISP_ORDER numeric(3) DEFAULT 0 NOT NULL,
             NULLABLE numeric(1) DEFAULT 1 NOT NULL,
             PARTNER_ID numeric(19) DEFAULT -1 NOT NULL,
             DISPLAY_NAME varchar(200), 
             CONSTRAINT Entity_Mapping_PK   PRIMARY KEY (ID) 
           )'; 
        SELECT createObject(vsql,'create table Entity_Mapping') INTO tmp_result;
            
        vsql:='CREATE TABLE '||VSCHEMA||'.PREDICATE
        (
            ID numeric(19,0) NOT NULL,
            RULE_ID numeric(19,0) NOT NULL,
            PREDICATE_TYPE varchar(64) NOT NULL,
            ERROR_LEVEL varchar(64),
            LOGMESSAGE varchar(2000),
            ENTITY_MAPPING numeric(19,0),
            VALUE_TYPE varchar(64),
            LITERAL_VALUE varchar(2000),
            MAPPED_VALUE numeric(19,0),
            ORDERING    numeric(19,0) NOT NULL DEFAULT 0,
            CREATED_BY      varchar(50) NOT NULL,
            UPDATED_BY      varchar(50) NOT NULL,
            CREATE_DATE     TIMESTAMP(0) WITH TIME ZONE NOT NULL,
            UPDATE_DATE     TIMESTAMP(0) WITH TIME ZONE NOT NULL,
            CONSTRAINT PK_PREDICATE_ID PRIMARY KEY(ID),
            CONSTRAINT FK_PREDICATE_ENTITY_MAPPING FOREIGN KEY(ENTITY_MAPPING) REFERENCES '||VSCHEMA||'.ENTITY_MAPPING(ID) ON DELETE CASCADE,
            CONSTRAINT FK_PREDICATE_MAPPED_VALUE FOREIGN KEY(MAPPED_VALUE) REFERENCES '||VSCHEMA||'.ENTITY_MAPPING(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table PREDICATE') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.RULE
        (
            ID numeric(19,0) NOT NULL,
            NAME varchar(200) NOT NULL,
            RULE_TYPE varchar(64) NOT NULL,
            ORDERING    numeric(19,0) NOT NULL,
            CREATED_BY      varchar(50) NOT NULL,
            UPDATED_BY      varchar(50) NOT NULL,
            CREATE_DATE     TIMESTAMP(0) WITH TIME ZONE NOT NULL,
            UPDATE_DATE     TIMESTAMP(0) WITH TIME ZONE NOT NULL,
            CONSTRAINT PK_RULE_ID PRIMARY KEY(ID)
        )';
        
        SELECT createObject(vsql,'create table RULE') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.CLAUSE
        (
            ID numeric(19,0) NOT NULL,
            ENTITY_MAPPING numeric(19,0),
            COMPARE_OPERATOR varchar(64) NOT NULL,
            JOIN_PREFIX varchar(64) NOT NULL,
            CONDITION_VALUE_TYPE varchar(64) NOT NULL,
            LITERAL_VALUE varchar(2000),
            MAPPED_VALUE numeric(19,0),
            LAYER_NUMBER numeric(4,0) NOT NULL,
            ORDERING numeric(4,0) NOT NULL,
            GROUP_NUMBER numeric(4,0) NOT NULL,
            NEGATE numeric(1,0) NOT NULL,
            IGNORECASE numeric(1,0) NULL,
            CREATED_BY      varchar(50) NOT NULL,
            UPDATED_BY      varchar(50) NOT NULL,
            CREATE_DATE     TIMESTAMP(0) WITH TIME ZONE NOT NULL,
            UPDATE_DATE     TIMESTAMP(0) WITH TIME ZONE NOT NULL,
            CONSTRAINT PK_CLAUSE_ID PRIMARY KEY(ID),
            CONSTRAINT FK_CLAUSE_ENTITY_MAPPING FOREIGN KEY(ENTITY_MAPPING) REFERENCES '||VSCHEMA||'.ENTITY_MAPPING(ID) ON DELETE CASCADE,
            CONSTRAINT FK_CLAUSE_MAPPED_VALUE FOREIGN KEY(MAPPED_VALUE) REFERENCES '||VSCHEMA||'.ENTITY_MAPPING(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table CLAUSE') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.RULE_CLAUSE_MAPPING
        (
            RULE_ID numeric(19,0),
            CLAUSE_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_CLAUSE_ID UNIQUE (CLAUSE_ID),
            CONSTRAINT FK_RULE_CLAUSE_RULE FOREIGN KEY(RULE_ID) REFERENCES '||VSCHEMA||'.RULE(ID) ON DELETE CASCADE,
            CONSTRAINT FK_RULE_CLAUSE_CLAUSE FOREIGN KEY(CLAUSE_ID) REFERENCES '||VSCHEMA||'.CLAUSE(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table RULE_CLAUSE_MAPPING') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.NORM_GROUP_RULE_MAPPING
        (
            NORM_RULE_GROUP_ID numeric(19,0),
            RULE_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_NGRM_RULE_ID UNIQUE (RULE_ID),
            CONSTRAINT FK_NGRM_GROUP FOREIGN KEY(NORM_RULE_GROUP_ID) REFERENCES '||VSCHEMA||'.NORM_RULE_GROUP(ID) ON DELETE CASCADE,
            CONSTRAINT FK_NGRM_RULE FOREIGN KEY(RULE_ID) REFERENCES '||VSCHEMA||'.RULE(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table NORM_GROUP_RULE_MAPPING') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.VALID_GROUP_RULE_MAPPING
        (
            VALID_RULE_GROUP_ID numeric(19,0),
            RULE_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_VGRM_RULE_ID UNIQUE (RULE_ID),
            CONSTRAINT FK_VGRM_GROUP FOREIGN KEY(VALID_RULE_GROUP_ID) REFERENCES '||VSCHEMA||'.VALID_RULE_GROUP(ID) ON DELETE CASCADE,
            CONSTRAINT FK_VGRM_RULE FOREIGN KEY(RULE_ID) REFERENCES '||VSCHEMA||'.RULE(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table VALID_GROUP_RULE_MAPPING') INTO tmp_result;
        
        
        --- Rules related table creation end
        
        -- Plugin_Properties
        vsql:='CREATE TABLE '||VSCHEMA||'.PLUGIN_PROPERTY(ID varchar(2101) NOT NULL, 
            FORMAT varchar(100) NOT NULL,
            PROPERTY_NAME varchar(2000) NOT NULL,
            PROPERTY_DEFAULT_VALUE varchar(2000),
            CREATED_BY varchar (50) NOT NULL ,
            UPDATED_BY varchar (50) NOT NULL ,
            CREATE_DATE TIMESTAMP NOT NULL ,
            UPDATE_DATE TIMESTAMP NOT NULL, 
            CONSTRAINT PK_PLUGIN_PROPERTY_ID PRIMARY KEY(ID))'; 
        SELECT createObject(vsql,'Create Table PLUGIN_PROPERTY') INTO tmp_result;
        
        -- Compression
        vsql:='CREATE TABLE '||VSCHEMA||'.COMPRESSION(ID numeric(19, 0) NOT NULL, 
            TYPE varchar(50) NOT NULL,
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL,
            CONSTRAINT PK_COMPRESS_ID PRIMARY KEY(ID))'; 
        SELECT createObject(vsql,'Create Table COMPRESSION') INTO tmp_result;
                
        -- Headend, lineup, station 
        vsql:='CREATE TABLE '||VSCHEMA||'.HEADEND
        (
            HEADEND_ID varchar(100) NOT NULL ,
            COMMUNITY_NAME varchar (200) ,
            COUNTY_NAME varchar (200) ,
            COUNTY_SIZE varchar (1) ,
            ST_COUNTY_CODE varchar (100) ,
            STATE varchar (64) ,
            ZIPCODE varchar (100),
            DMA_CODE varchar (100) ,
            DMA_NAME varchar (300) ,
            MSO_CODE numeric(22,0) ,
            DMA_RANK varchar (100) ,
            HEADEND_NAME varchar (1000) ,
            HEADEND_LOCATION varchar (100) NOT NULL ,
            MSO_NAME varchar (1000) NOT NULL ,
            TIME_ZONE varchar (100) NOT NULL ,
            HEADEND_TYPE varchar(100),
            HEADEND_COUNTRY varchar(100),
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_HEADEND_EPG PRIMARY KEY (PROVIDER_ID, HEADEND_ID)
        )';
        SELECT createObject(vsql,'create TABLE HEADEND') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'HEADEND', 'PROVIDER_ID', 'IDX_HEADEND') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.HEADEND_GENERIC 
        (  
             HEADEND_ID varchar (100) NULL, 
             ASSET varchar(50) NULL, 
             NAME varchar(200) NULL, 
             VALUE varchar(2000) NULL, 
             CREATED_BY varchar (50) NOT NULL, 
             UPDATED_BY varchar (50) NOT NULL,  
             CREATE_DATE TIMESTAMP NOT NULL, 
             UPDATE_DATE TIMESTAMP NOT NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT HEADEND_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, HEADEND_ID, ASSET, NAME),
             CONSTRAINT FK_HEADEND_GENERIC_HEADEND_ID FOREIGN KEY (PROVIDER_ID, HEADEND_ID) REFERENCES '||VSCHEMA||'.HEADEND (PROVIDER_ID, HEADEND_ID) ON DELETE CASCADE 
        )'; 
        SELECT createObject(vsql,'create table HEADEND_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'HEADEND_GENERIC', 'PROVIDER_ID', 'IDX_HEADEND_GENERIC') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STATION
        (
            STATION_ID varchar (100)  NOT NULL ,
            STATION_NAME varchar (200) ,
            TIME_ZONE varchar (100) ,
            CALL_SIGN varchar (100),
            CITY varchar (100) ,
            STATE varchar (100) ,
            ZIPCODE varchar (100) ,
            COUNTRY varchar (100) ,
            DMA_NAME varchar (200) ,
            DMA_NUMBER numeric(12) ,
            FCC_CHANNEL_NUM numeric(12) ,
            PROGRAM_GENRE varchar(200) ,
            AFFILIATION varchar (100),
            STATION_LOCKED varchar(1) DEFAULT ''N'',
            USER_DATA1 varchar(1000),
            USER_DATA2 varchar(1000),
            USER_DATA3 varchar(1000),
            USER_DATA4 varchar(1000),
            USER_DATA5 varchar(1000),
            USER_DATA6 varchar(1000),
            USER_DATA7 varchar(1000),
            USER_DATA8 varchar(1000),
            USER_DATA9 varchar(1000),
            USER_DATA10 varchar(1000),
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_STATION_EPG PRIMARY KEY (PROVIDER_ID, STATION_ID)
        )';
        SELECT createObject(vsql,'create table STATION') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STATION', 'PROVIDER_ID', 'IDX_STATION') INTO tmp_result;
        SELECT CreateIndex(vschema,'STATION', 'CALL_SIGN', 'IDX_STATION_CALLSIGN') INTO tmp_result;
        SELECT CreateIndex(vschema,'STATION', 'STATION_NAME', 'IDX_STATION_STATIONNAME') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STATION_GENERIC 
        (  
             STATION_ID varchar (100) NULL, 
             ASSET varchar(50) NULL, 
             NAME varchar(200) NULL, 
             VALUE varchar(2000) NULL, 
             CREATED_BY varchar (50) NOT NULL, 
             UPDATED_BY varchar (50) NOT NULL,  
             CREATE_DATE TIMESTAMP NOT NULL, 
             UPDATE_DATE TIMESTAMP NOT NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT STATION_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, STATION_ID, ASSET, NAME),
             CONSTRAINT FK_STATION_GENERIC_STATION_ID FOREIGN KEY (PROVIDER_ID, STATION_ID) REFERENCES '||VSCHEMA||'.STATION (PROVIDER_ID, STATION_ID) ON DELETE CASCADE 
        )'; 
        SELECT createObject(vsql,'create table STATION_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STATION_GENERIC', 'PROVIDER_ID', 'IDX_STATION_GENERIC') INTO tmp_result;
         
        
        vsql:='CREATE TABLE '||VSCHEMA||'.LINEUP
        (
            LINEUP_ID varchar(100) NOT NULL ,
            HEADEND_ID varchar(100) NOT NULL ,
            STATION_ID varchar (100)  NOT NULL,
            DEVICE varchar (2) ,
            TMS_CHANNEL varchar (20) NOT NULL ,
            SERVICE_TIER varchar (100) NOT NULL ,
            EFFECTIVE_DATE TIMESTAMP(0) WITH TIME ZONE ,
            EXPIRATION_DATE TIMESTAMP(0) WITH TIME ZONE ,
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL ,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_LINEUP_EPG PRIMARY KEY (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID),
            CONSTRAINT FK_LINEUP_STATION FOREIGN KEY (PROVIDER_ID, STATION_ID) REFERENCES '||VSCHEMA||'.STATION(PROVIDER_ID, STATION_ID),
            CONSTRAINT FK_HEADEND_LINEUP FOREIGN KEY (PROVIDER_ID, HEADEND_ID) REFERENCES '||VSCHEMA||'.HEADEND (PROVIDER_ID, HEADEND_ID)
        )';
        SELECT createObject(vsql,'create TABLE LINEUP') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'LINEUP', 'PROVIDER_ID', 'IDX_LINEUP') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'LINEUP', 'STATION_ID,HEADEND_ID', 'IDX_L_STATIONID_HEADENDID') INTO tmp_result;
        
         vsql:='CREATE TABLE '||VSCHEMA||'.LINEUP_GENERIC 
        (  
             LINEUP_ID varchar (100) NOT NULL,
             HEADEND_ID varchar (100) NOT NULL,
             STATION_ID varchar (100) NOT NULL, 
             ASSET varchar(50) NULL, 
             NAME varchar(200) NULL, 
             VALUE varchar(2000) NULL, 
             CREATED_BY varchar (50) NOT NULL, 
             UPDATED_BY varchar (50) NOT NULL,  
             CREATE_DATE TIMESTAMP NOT NULL, 
             UPDATE_DATE TIMESTAMP NOT NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT LINEUP_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID, ASSET, NAME),
             CONSTRAINT FK_LINEUP_GENERIC_LINEUP_ID FOREIGN KEY (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID) REFERENCES '||VSCHEMA||'.LINEUP (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID) ON DELETE CASCADE 
        )'; 
        
        SELECT createObject(vsql,'create table LINEUP_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'LINEUP_GENERIC', 'PROVIDER_ID', 'IDX_LINEUP_GENERIC') INTO tmp_result;
        
        -- Program 
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM
        (
            PROGRAM_ID varchar (100)  NOT NULL ,
            REF_STATION_ID varchar (100),
            REF_START_TIME TIMESTAMP (0) WITH TIME ZONE,
            LANGUAGE varchar (100),
            TITLE varchar (1000),
            REDUCED_TITLE1 varchar (1000),
            REDUCED_TITLE2 varchar (1000) ,
            REDUCED_TITLE3 varchar (1000) ,
            REDUCED_TITLE4 varchar (1000) ,
            ALT_TITLE varchar (1000) ,
            REDUCED_DESCRIPTION1 varchar (1000) ,
            REDUCED_DESCRIPTION2 varchar (1000) ,
            REDUCED_DESCRIPTION3 varchar (1000) ,
            REDUCED_DESCRIPTION4 varchar (1000) ,
            ADVISORY_DESC1 varchar (100) ,
            ADVISORY_DESC2 varchar (100) ,
            ADVISORY_DESC3 varchar (100) ,
            ADVISORY_DESC4 varchar (100) ,
            ADVISORY_DESC5 varchar (100) ,
            ADVISORY_DESC6 varchar (100) ,
            GENRE_DESC1 varchar (100) ,
            GENRE_DESC2 varchar (100) ,
            GENRE_DESC3 varchar (100) ,
            GENRE_DESC4 varchar (100) ,
            GENRE_DESC5 varchar (100) ,
            GENRE_DESC6 varchar (100) ,
            DESCRIPTION1 varchar (2000),
            DESCRIPTION2 varchar (2000) ,
            YEAR_PROGRAM varchar (100) ,
            MPAA_RATING varchar (1000) ,
            STAR_RATING varchar (1000) ,
            RUN_TIME varchar (100) ,
            COLOR_CODE varchar (100) ,
            PROGRAM_LANGUAGE varchar (100) ,
            COUNTRY_OF_ORIGIN varchar (100) ,
            MADE_FOR_TV varchar (1) ,
            SOURCE_TYPE varchar (100) ,
            SHOW_TYPE varchar (200) ,
            HOLIDAY varchar (100) ,
            SYNDICATE_EPISODE_NUM varchar (100) ,
            ALT_SYNDICATE_EPI_NUM varchar (100) ,
            EPISODE_TITLE varchar (1000) ,
            NET_SYN_SOURCE varchar (100) ,
            NET_SYN_TYPE varchar (100) ,
            ORG_STUDIO varchar (100),
            GAME_DATETIME TIMESTAMP(0) WITH TIME ZONE ,
            GAME_TIMEZONE varchar (100) ,
            ORG_AIR_DATE TIMESTAMP(0)  WITH TIME ZONE,
            UNIQUE_ID varchar (100) ,
            SEASON_ID  varchar (100) ,
            SERIES_ID  varchar (100) ,
            USER_DATA1 varchar(1000),
            USER_DATA2 varchar(1000),
            USER_DATA3 varchar(1000),
            USER_DATA4 varchar(1000),
            USER_DATA5 varchar(1000),
            USER_DATA6 varchar(1000),
            USER_DATA7 varchar(1000),
            USER_DATA8 varchar(1000),
            USER_DATA9 varchar(1000),
            USER_DATA10 varchar(1000),
            USER_DATA11 varchar(1000),
            USER_DATA12 varchar(1000),
            USER_DATA13 varchar(1000),
            CREATED_BY varchar (50)  NOT NULL ,
            UPDATED_BY varchar (50)  NOT NULL ,
            CREATE_DATE TIMESTAMP  NOT NULL ,
            UPDATE_DATE TIMESTAMP  NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT PK_PROGRAM_EPG PRIMARY KEY (PROVIDER_ID, PROGRAM_ID)
        )';
        SELECT createObject(vsql,'create TABLE PROGRAM') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM', 'PROVIDER_ID', 'IDX_PROGRAM') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM', 'PROGRAM_ID', 'IDX_PROGRAM_PROGRAMID') INTO tmp_result;
        
        -- support tables for program 
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_SUBTITLE_COMP 
        (  
             PROGRAM_ID varchar (100) NOT NULL, 
             TLA varchar(600) NOT NULL,
             TYPE varchar(200) NULL, 
             LANGUAGE varchar(200) NULL, 
             ATTRIBUTE_TYPE varchar(200) NULL,
             VALUE varchar(200) NULL, 
             CREATED_BY varchar (50) NOT NULL, 
             UPDATED_BY varchar (50) NOT NULL,  
             CREATE_DATE TIMESTAMP NOT NULL, 
             UPDATE_DATE TIMESTAMP NOT NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT PROGRAM_SUBTITLE_COMP_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, TLA),
             CONSTRAINT FK_PROGRAM_SUBTITLE_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        
        )'; 
        SELECT createObject(vsql,'create table PROGRAM_SUBTITLE_COMP') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_SUBTITLE_COMP', 'PROVIDER_ID', 'IDX_PROGRAM_SUBTITLE_COMP') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_AUDIO_COMP 
        (  
         PROGRAM_ID varchar (100) NOT NULL, 
         TLA varchar(600) NOT NULL,
         TYPE varchar(200) NULL, 
         LANGUAGE varchar(200) NULL, 
         ATTRIBUTE_TYPE varchar(200) NULL,
         VALUE varchar(200) NULL, 
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT PROGRAM_AUDIO_COMP_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, TLA),
         CONSTRAINT FK_PROGRAM_AUDIO_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        )'; 
        SELECT createObject(vsql,'create table PROGRAM_AUDIO_COMP') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_AUDIO_COMP', 'PROVIDER_ID', 'IDX_PROGRAM_AUDIO_COMP') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_GENERIC 
        (  
         PROGRAM_ID varchar (100) NULL, 
         ASSET varchar(50) NULL, 
         NAME varchar(200) NULL, 
         VALUE varchar(2000) NULL, 
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT PROGRAM_GENERIC_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, ASSET, NAME),
         CONSTRAINT FK_PROGRAM_GENERIC_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE 
        )'; 
        
        SELECT createObject(vsql,'create table PROGRAM_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_GENERIC', 'PROVIDER_ID', 'IDX_PROGRAM_GENERIC') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_RATING 
        (  
         PROGRAM_ID varchar(100) NULL, 
         AUTHORITY varchar(200) NULL, 
         VALUE varchar(2000) NULL, 
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT PROGRAM_RATING_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, AUTHORITY),
         CONSTRAINT FK_PROGRAM_RATING_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        )'; 
        SELECT createObject(vsql,'create table PROGRAM_RATING') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_RATING', 'PROVIDER_ID', 'IDX_PROGRAM_RATING') INTO tmp_result;
        
        
        -- Schedule 
        vsql:='CREATE TABLE '||VSCHEMA||'.SCHEDULE
        (
        SCHEDULE_ID varchar (100) NOT NULL ,
        SCHEDULE_LANGUAGE varchar (100) ,
        STATION_ID varchar (100) NOT NULL ,
        PROGRAM_ID varchar (100),
        START_TIME TIMESTAMP (0) WITH TIME ZONE  NOT NULL ,
        DURATION numeric(10,0) ,
        EXPIRED_TIME TIMESTAMP,
        REF_STATION_ID varchar (100),
        REF_START_TIME TIMESTAMP (0) WITH TIME ZONE,
        TV_RATING varchar (1000) ,
        HD_TV varchar (1) ,
        PART_NUMBER numeric(20,0) ,
        NO_OF_PARTS numeric(20,0) ,
        SERIES_ID varchar(200) , 
        CC varchar (1) ,
        STEREO varchar (1) ,
        LIVE_TAPE_DELAY varchar (100) ,
        SUBTITLED varchar (1) ,
        PREMIERE_FINALE varchar (100) ,
        JOINED_IN_PROGRESS varchar (1) ,
        CABLE_IN_CLASSROOM varchar (1) ,
        SAP varchar (1) ,
        SEX_RATING varchar (1) ,
        VIOLENCE_RATING varchar (1) ,
        LANGUAGE_RATING varchar (1) ,
        DIALOG_RATING varchar (1) ,
        FV_RATING varchar (1) ,
        ENHANCED varchar (1) ,
        THREE_D varchar (1) ,
        LETTERBOX varchar (1) ,
        DVS varchar (1) ,
        REPEAT varchar (1) ,
        BLACKOUT varchar (1) ,
        DOLBY varchar (100) ,
        REQUEST_RECORD varchar(1),
        USER_DATA1 varchar(1000),
        USER_DATA2 varchar(1000),
        USER_DATA3 varchar(1000),
        USER_DATA4 varchar(1000),
        USER_DATA5 varchar(1000),
        USER_DATA6 varchar(1000),
        USER_DATA7 varchar(1000),
        USER_DATA8 varchar(1000),
        USER_DATA9 varchar(1000),
        USER_DATA10 varchar(1000),
        USER_DATA11 varchar(1000),
        CREATED_BY varchar (50)  NOT NULL ,
        UPDATED_BY varchar (50)  NOT NULL ,
        CREATE_DATE TIMESTAMP  NOT NULL ,
        UPDATE_DATE TIMESTAMP  NOT NULL,
        PROVIDER_ID numeric(19,0) NOT NULL
        )';
        SELECT createObject(vsql,'create table SCHEDULE') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'SCHEDULE', 'PROVIDER_ID', 'IDX_SCHEDULE') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'SCHEDULE', 'SCHEDULE_ID, PROVIDER_ID', 'IDX_SCHED_PROVID_SCHEDID') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'SCHEDULE', 'PROGRAM_ID,STATION_ID,START_TIME, PROVIDER_ID', 'IDX_SCHED_UNIQ_IND') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE', 'START_TIME', 'IDX_SCHED_START_TIME') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE', 'STATION_ID', 'IDX_SCH_STATION_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE', 'UPDATE_DATE', 'IDX_SCHED_UPDATE_DATE') INTO tmp_result;
        
        -- schedule support tables
        vsql:='CREATE TABLE '||VSCHEMA||'.SCHEDULE_EVENT_ID 
        (  
         PROGRAM_ID varchar(100) NULL, 
        IDENTIFIER_TYPE varchar(200) NULL, 
        VALUE varchar(200) NULL, 
        CREATED_BY varchar (50) NOT NULL, 
        UPDATED_BY varchar (50) NOT NULL,  
        CREATE_DATE TIMESTAMP NOT NULL, 
        UPDATE_DATE TIMESTAMP NOT NULL, 
        PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT SCHEDULE_EVENT_IDS_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, IDENTIFIER_TYPE) 
        )'; 
        SELECT createObject(vsql,'create table SCHEDULE_EVENT_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'SCHEDULE_EVENT_ID', 'PROVIDER_ID', 'IDX_SCHEDULE_EVENT_ID') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.SCHEDULE_SERIES_ATTRIB 
        (  
         PROGRAM_ID varchar(100) NULL, 
         SERIES_ID varchar(100), 
        ATTRIBUTE_TYPE varchar(200) NULL, 
        VALUE varchar(2000) NULL, 
        CREATED_BY varchar (50) NOT NULL, 
        UPDATED_BY varchar (50) NOT NULL,  
        CREATE_DATE TIMESTAMP NOT NULL, 
        UPDATE_DATE TIMESTAMP NOT NULL, 
        PROVIDER_ID numeric(19,0) NOT NULL,
        CONSTRAINT SCHEDULE_SERIES_ATTRIB_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, ATTRIBUTE_TYPE) 
        )'; 
        SELECT createObject(vsql,'create table SCHEDULE_SERIES_ATTRIB') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'SCHEDULE_SERIES_ATTRIB', 'PROVIDER_ID', 'IDX_SCHEDULE_SERIES_ATTRIB') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.SCHEDULE_GENERIC 
        (  
         SCHEDULE_ID varchar(100) NULL, 
        ASSET varchar(50) NULL, 
        NAME varchar(200) NULL, 
        VALUE varchar(2000) NULL, 
        CREATED_BY varchar (50) NOT NULL, 
        UPDATED_BY varchar (50) NOT NULL,  
        CREATE_DATE TIMESTAMP NOT NULL, 
        UPDATE_DATE TIMESTAMP NOT NULL, 
        PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT SCHEDULE_GENERIC_PK   PRIMARY KEY (SCHEDULE_ID, PROVIDER_ID, ASSET, NAME) 
        )'; 
        SELECT createObject(vsql,'create table SCHEDULE_GENERIC') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'SCHEDULE_GENERIC', 'PROVIDER_ID', 'IDX_SCHEDULE_GENERIC') INTO tmp_result;
        
        -- stg_program_credits
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_CREDITS
        (
            FILENAME varchar(4000), 
            PROGRAM_ID varchar(100) NOT NULL, 
            C_ID varchar(50) NULL,
            SEQ numeric(10) DEFAULT 1 NOT NULL,
            ROLE varchar(200) NOT NULL, 
            FIRST_NAME varchar(200) NULL, 
            MIDDLE_NAME varchar(200) NULL, 
            LAST_NAME varchar(500) NULL, 
			LANGUAGE varchar(100),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_STG_PROGRAM_CREDITS   UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE) 
        )'; 
        SELECT createObject(vsql,'create table STG_PROGRAM_CREDITS') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_PROGRAM_CREDITS', 'PROVIDER_ID', 'IDX_STG_PROGRAM_CREDITS') INTO tmp_result;
        
        -- stg_program cast
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_CAST
        (   
            FILENAME varchar(4000), 
            PROGRAM_ID varchar(100) NOT NULL, 
            C_ID varchar(50) NULL,
            SEQ numeric(10) DEFAULT 1 NOT NULL,
            ROLE varchar(200) NOT NULL, 
            FIRST_NAME varchar(200) NULL, 
            MIDDLE_NAME varchar(200) NULL, 
            LAST_NAME varchar(500) NULL, 
			LANGUAGE varchar(100),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_STG_PROGRAM_CAST   UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE) 
        )';
        SELECT createObject(vsql,'create table STG_PROGRAM_CAST') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_PROGRAM_CAST', 'PROVIDER_ID', 'IDX_STG_PROGRAM_CAST') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_CREDITS 
        (  
            PROGRAM_ID varchar (100) NOT NULL, 
            C_ID varchar(50) NULL,
            SEQ numeric(10) DEFAULT 1 NOT NULL ,
            ROLE varchar(200) NOT NULL, 
            FIRST_NAME varchar(200) NULL, 
            MIDDLE_NAME varchar(200) NULL, 
            LAST_NAME varchar(500) NULL, 
            CREATED_BY varchar (50) NOT NULL, 
            UPDATED_BY varchar (50) NOT NULL,  
            CREATE_DATE TIMESTAMP NOT NULL, 
            UPDATE_DATE TIMESTAMP NOT NULL, 
			LANGUAGE varchar(100),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_PROGRAM_CREDITS   UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE),
            CONSTRAINT FK_PROGRAM_CREDIT_PROG_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        )'; 
        SELECT createObject(vsql,'create table PROGRAM_CREDITS') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_CREDITS', 'PROVIDER_ID', 'IDX_PROGRAM_CREDITS') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_CAST
        (  
            PROGRAM_ID varchar (100) NOT NULL, 
            C_ID varchar(50) NULL,
            SEQ numeric(10) DEFAULT 1 NOT NULL,
            ROLE varchar(200) NOT NULL, 
            FIRST_NAME varchar(200) NULL, 
            MIDDLE_NAME varchar(200) NULL,
            LAST_NAME varchar(500) NULL, 
            CREATED_BY varchar (50) NOT NULL, 
            UPDATED_BY varchar (50) NOT NULL,  
            CREATE_DATE TIMESTAMP NOT NULL, 
            UPDATE_DATE TIMESTAMP NOT NULL, 
			LANGUAGE varchar(100),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_PROGRAM_CAST   UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE),
            CONSTRAINT FK_PROGRAM_CAST_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        )'; 
        SELECT createObject(vsql,'create table PROGRAM_CAST') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_CAST', 'PROVIDER_ID', 'IDX_PROGRAM_CAST') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_STATION_IMAGE 
        (  
             FILENAME varchar(4000),
             STATION_ID varchar (100) NOT NULL, 
             IMAGE_ID varchar(100) NOT NULL, 
             IMAGE_TYPE_ID varchar(100), 
             ACTIVE_DATE TIMESTAMP,
             EXPIRATION_DATE TIMESTAMP,
             LAST_UPDATE_DATE TIMESTAMP,
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT STG_STATION_IMAGE_PK  PRIMARY KEY (PROVIDER_ID, STATION_ID, IMAGE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_STATION_IMAGE') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_STATION_IMAGE', 'PROVIDER_ID', 'IDX_STG_STATION_IMAGE') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_STATION_IMAGE', 'STATION_ID', 'IDX_STG_STATION_IMAGE_STA') INTO tmp_result;

        vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_IMAGE 
        (  
         FILENAME varchar(4000),
         PROGRAM_ID varchar(100) NOT NULL,   
         SERIES_ID varchar(100),
         SEASON_PROGRAM_ID varchar(100),
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_ID varchar(100),
         ACTIVE_DATE TIMESTAMP,
         EXPIRATION_DATE TIMESTAMP,
         NUMBER_OF_PEOPLE numeric(12,0),
         ZOOM_LEVEL numeric(12,0),
         CAST_WEIGHT numeric(12,2),
         CAPTION varchar(2000) NULL, 
         CREDIT varchar(256) NULL, 
         LAST_UPDATE_DATE TIMESTAMP,
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT STG_PROGRAM_IMAGE_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, IMAGE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_PROGRAM_IMAGE') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_PROGRAM_IMAGE', 'PROVIDER_ID', 'IDX_STG_PROGRAM_IMAGE') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_IMAGE', 'PROGRAM_ID', 'IDX_STG_PROGRAM_IMAGE_PROGRA') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_IMAGE_FILES 
        (  
         FILENAME varchar(4000),         
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_FILE_ID varchar(100),
         FILE_URL varchar(2000),
         FILE_FORMAT_ID numeric(12,0),
         FILE_FORMAT varchar(50),
         FILE_WIDTH numeric(12,0),
         FILE_HEIGHT numeric(12,0),
         FILE_QUALITY_FACTOR numeric(12,0),
         FILE_BIT_DEPTH varchar(100),
         FILE_MARGIN numeric(12,0),
         FILE_BG_COLOR numeric(12,0),
         FILE_SIZE numeric(12,0),
         LAST_UPDATE_DATE TIMESTAMP,
         SCALING_FACTOR numeric(8,2),
         IS_FILE_READY numeric(1,0),
         TOUCHED numeric(1,0) default 0,
         GROUP_ID numeric(12,0),
         ENTITY_TYPE varchar(100),
         ENTITY_IDS varchar(2000),
         IMAGE_TYPE_ID varchar(100),
         LOCKED_BY varchar(100),
         LOCKED_AT TIMESTAMP,
         INTERNAL_FILE_ID varchar(100),
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT STG_IMAGE_FILES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_FILE_ID, ENTITY_TYPE)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_IMAGE_FILES') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_IMAGE_FILES', 'GROUP_ID', 'IDX_STG_IMAGE_FILES_GROUP_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_IMAGE_FILES', 'PROVIDER_ID', 'IDX_STG_IMAGE_FILES') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_IMAGE_TYPES 
        (  
         FILENAME varchar(4000),
         IMAGE_TYPE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_NAME varchar(100),
         IMAGE_CATEGORY varchar(100),
         LAST_UPDATE_DATE TIMESTAMP,
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT STG_IMAGE_TYPES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_TYPE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_IMAGE_TYPES') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_IMAGE_TYPES', 'PROVIDER_ID', 'IDX_STG_IMAGE_TYPES') INTO tmp_result;
        

        vsql:='CREATE TABLE '||VSCHEMA||'.STATION_IMAGE 
        (  
             STATION_ID varchar (100) NOT NULL, 
             IMAGE_ID varchar(100) NOT NULL, 
             IMAGE_TYPE_ID varchar(100), 
             ACTIVE_DATE TIMESTAMP,
             EXPIRATION_DATE TIMESTAMP,
             LAST_UPDATE_DATE TIMESTAMP,
             CREATED_BY varchar (50) NOT NULL, 
             UPDATED_BY varchar (50) NOT NULL,  
             CREATE_DATE TIMESTAMP NOT NULL, 
             UPDATE_DATE TIMESTAMP NOT NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT STATION_IMAGE_PK   PRIMARY KEY (PROVIDER_ID, STATION_ID, IMAGE_ID),
             CONSTRAINT FK_STATION_IMAGE_STATION_ID FOREIGN KEY (PROVIDER_ID, STATION_ID) REFERENCES '||VSCHEMA||'.STATION (PROVIDER_ID, STATION_ID) ON DELETE CASCADE 
        )'; 
        SELECT createObject(vsql,'create TABLE STATION_IMAGE') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STATION_IMAGE', 'PROVIDER_ID', 'IDX_STATION_IMAGE') INTO tmp_result;
        SELECT CreateIndex(vschema,'STATION_IMAGE', 'STATION_ID', 'IDX_STATION_IMAGE_STATION_ID')      INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_IMAGE 
        (  
         PROGRAM_ID varchar(100) NOT NULL,   
         SERIES_ID varchar(100),
         SEASON_PROGRAM_ID varchar(100),
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_ID varchar(100),
         ACTIVE_DATE TIMESTAMP,
         EXPIRATION_DATE TIMESTAMP,
         NUMBER_OF_PEOPLE numeric(12,0),
         ZOOM_LEVEL numeric(12,0),
         CAST_WEIGHT numeric(12,2),
         CAPTION varchar(2000) NULL, 
         CREDIT varchar(256) NULL, 
         LAST_UPDATE_DATE TIMESTAMP,
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT PROGRAM_IMAGE_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, IMAGE_ID),
         CONSTRAINT FK_PROGRAM_IMAGE_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES '||VSCHEMA||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        )'; 
        SELECT createObject(vsql,'create TABLE PROGRAM_IMAGE') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM_IMAGE', 'PROVIDER_ID', 'IDX_PROGRAM_IMAGE')  INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_IMAGE', 'PROGRAM_ID', 'IDX_PROGRAM_IMAGE_PROGRAM_ID')  INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.IMAGE_FILES 
        (  
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_FILE_ID varchar(100),
         FILE_URL varchar(2000),
         FILE_FORMAT_ID numeric(12,0),
         FILE_FORMAT varchar(50),
         FILE_WIDTH numeric(12,0),
         FILE_HEIGHT numeric(12,0),
         FILE_QUALITY_FACTOR numeric(12,0),
         FILE_BIT_DEPTH varchar(100),
         FILE_MARGIN numeric(12,0),
         FILE_BG_COLOR numeric(12,0),
         FILE_SIZE numeric(12,0),
         LAST_UPDATE_DATE TIMESTAMP,
         SCALING_FACTOR numeric(8,2),
         IS_FILE_READY numeric(1,0),
         TOUCHED numeric(1,0) default 0,
         GROUP_ID numeric(12,0),
         ENTITY_TYPE varchar(100),
         ENTITY_IDS varchar(2000),
         IMAGE_TYPE_ID varchar(100),
         LOCKED_BY varchar(100),
         LOCKED_AT TIMESTAMP,
         INTERNAL_FILE_ID varchar(100),
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT IMAGE_FILES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_FILE_ID, ENTITY_TYPE)
        )'; 
        SELECT createObject(vsql,'create TABLE IMAGE_FILES') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'IMAGE_FILES', 'GROUP_ID', 'IDX_IMAGE_FILES_GROUP_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'IMAGE_FILES', 'PROVIDER_ID', 'IDX_IMAGE_FILES') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.IMAGE_TYPES 
        (  
         IMAGE_TYPE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_NAME varchar(100),
         IMAGE_CATEGORY varchar(100),
         LAST_UPDATE_DATE TIMESTAMP,
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT IMAGE_TYPES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_TYPE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE IMAGE_TYPES') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'IMAGE_TYPES', 'PROVIDER_ID', 'IDX_IMAGE_TYPES') INTO tmp_result;
        
        vsql:='create table '||VSCHEMA||'.temp_station_sch_range
                (
                STATION_ID varchar(100) primary key,
                first_schedule_start_time timestamp(0) with time zone,
                last_schedule_end_time timestamp(0) with time zone,
                PROVIDER_ID numeric(19,0) NOT NULL
                )';
        SELECT createObject(vsql, 'create TABLE temp_station_sch_range') INTO tmp_result;
        
        
        vsql:='CREATE TABLE '||VSCHEMA||'.EPG_JOB 
            (JOB_ID varchar(50), 
            PARTNER_ID numeric(19,0),
            TASK_ID numeric(19,0), 
            TASK_NAME varchar(200),
            TYPE varchar(6), 
            START_DATE TIMESTAMP, 
            END_DATE TIMESTAMP, 
            STATUS varchar(10), 
            PARAMETERS varchar(200), 
            TRIGGER_BY varchar(20), 
            TRIGGER_DATE DATE, 
			ERROR_KEY varchar(100),
            ERROR_MESSAGE varchar(200), 
            PROGRESS numeric(3,0), 
            STEP_INFO varchar(4000), 
            TRIGGER_TYPE varchar(20) DEFAULT ''FREQUENCY'',
            RUN_ON varchar(41),
            COMMENTS varchar(2000),
            CONSTRAINT PK_EPG_JOB PRIMARY KEY (JOB_ID))';
        SELECT createObject(vsql, 'create table EPG_JOB' ) INTO tmp_result;
        
        SELECT CreateIndex(vschema,'EPG_JOB', 'PARTNER_ID', 'IDX_JOB_PARTNERID') INTO tmp_result;
        SELECT CreateIndex(vschema,'EPG_JOB', 'START_DATE', 'IDX_JOB_STARTDATE') INTO tmp_result;
        
                -- task related tables
        vsql:='CREATE TABLE '||VSCHEMA||'.TASK
           (
            ID numeric(19,0) NOT NULL,
            NAME varchar(200),
            PARTNER_ID numeric(19,0) NOT NULL,
            TYPE varchar(6),
            TRIGGER_TYPE varchar(20),
            TRIGGER_VALUE varchar(200),
            FORMAT_NAME varchar(50),
            IS_SEND_EMAIL numeric(1,0),
            IS_ENABLED numeric(1,0),
            CREATED_BY varchar (50) NOT NULL,
            UPDATED_BY varchar (50) NOT NULL,
            CREATE_DATE TIMESTAMP NOT NULL,
            UPDATE_DATE TIMESTAMP NOT NULL,
            UUID varchar(128),
            CONSTRAINT PK_TASK_ID PRIMARY KEY(ID),
            CONSTRAINT U_TASK UNIQUE (NAME, PARTNER_ID),
            CONSTRAINT U_TASK_UUID UNIQUE (UUID)
            )';
        SELECT createObject(vsql, 'create table TASK') INTO tmp_result;
        
       vsql:='CREATE TABLE '||VSCHEMA||'.TASK_EXE_INFO 
            (TASK_ID numeric(19,0) NOT NULL, 
            IDENTIFY varchar(100) NOT NULL,
            TASK_NAME varchar(200) NOT NULL,
            PARTNER_ID numeric(19,0),
            START_DATE DATE, 
            END_DATE DATE, 
            FOLDERS varchar(2000), 
            FILE_NAMES varchar(2000) NOT NULL,
            COMMENTS varchar(2000),
            CONSTRAINT PK_TASK_EXE_INFO PRIMARY KEY (TASK_ID, IDENTIFY))';
        SELECT createObject(vsql, 'create TABLE TASK_EXE_INFO') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.TASK_SETTING(
            ID numeric(19,0) NOT NULL,
            TASK_ID numeric(19,0) NOT NULL,
            KEY varchar(100) NOT NULL,
            SETTING varchar(20000),
            CREATED_BY varchar (50) NOT NULL,
            UPDATED_BY varchar (50) NOT NULL,
            CREATE_DATE TIMESTAMP NOT NULL,
            UPDATE_DATE TIMESTAMP NOT NULL, 
            CONSTRAINT PK_TASK_SETTING_ID PRIMARY KEY(ID),
            CONSTRAINT FK_TASK_SETTING_TASK_ID FOREIGN KEY(TASK_ID) REFERENCES '||VSCHEMA||'.TASK(ID) ON DELETE CASCADE,
            CONSTRAINT U_TASK_SETTING UNIQUE (TASK_ID, KEY)
            )';
        SELECT createObject(vsql, 'create table TASK_SETTING') INTO tmp_result;
        
        --- tasks mapping to rule groups         
        vsql:='CREATE TABLE '||VSCHEMA||'.TASK_VALID_GROUP_MAPPING
        (
            TASK_ID numeric(19,0) NOT NULL,
            VALID_GROUP_ID numeric(19,0) NOT NULL,
            CONSTRAINT FK_TVGM_TASK FOREIGN KEY(TASK_ID) REFERENCES '||VSCHEMA||'.TASK(ID) ON DELETE CASCADE,
            CONSTRAINT FK_TVGM_VALID_GROUP FOREIGN KEY(VALID_GROUP_ID) REFERENCES '||VSCHEMA||'.VALID_RULE_GROUP(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create TABLE TASK_VALID_GROUP_MAPPING') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.TASK_NORM_GROUP_MAPPING 
        (    
            TASK_ID numeric(19,0) NOT NULL, 
            NORM_GROUP_ID numeric(19,0) NOT NULL,
            CONSTRAINT FK_TNGM_TASK FOREIGN KEY(TASK_ID) REFERENCES '||VSCHEMA||'.TASK(ID) ON DELETE CASCADE,
            CONSTRAINT FK_TNGM_VALID_GROUP FOREIGN KEY(NORM_GROUP_ID) REFERENCES '||VSCHEMA||'.NORM_RULE_GROUP(ID) ON DELETE CASCADE
        )';        
        SELECT createObject(vsql,'create TABLE TASK_NORM_GROUP_MAPPING')        INTO tmp_result;

        
        vsql:='CREATE TABLE '||VSCHEMA||'.PLUGIN_PROPERTY_VALUE(
            ID numeric(19,0) NOT NULL,
            TASK_ID numeric(19,0) NOT NULL,
            PLUGIN_PROPERTY_ID varchar(2101) NOT NULL,
            VALUE varchar(2000),
            CREATED_BY varchar (50) NOT NULL,
            UPDATED_BY varchar (50) NOT NULL,
            CREATE_DATE TIMESTAMP NOT NULL,
            UPDATE_DATE TIMESTAMP NOT NULL, 
            CONSTRAINT PK_PLUGIN_PROPERTY_VALUE_ID PRIMARY KEY(ID),
            CONSTRAINT FK_PPV_TASK_ID FOREIGN KEY(TASK_ID) REFERENCES '||VSCHEMA||'.TASK(ID)            
            )';
        SELECT createObject(vsql, 'create table PLUGIN_PROPERTY_VALUE') INTO tmp_result;
                
        
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''PLUGIN_PROPERTY_VALUE'',1)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for PLUGIN_PROPERTY_VALUE id generating') INTO tmp_result;
        
        vsql := 'CREATE TABLE ' || vschema || '.JOB_TRACE(
            JOB_ID varchar(50) NOT NULL,
            IS_FINISHED numeric(1,0) DEFAULT 0,
            START_TIME TIMESTAMP,
            NODE_ADDRESS varchar(60),
            NODE_PORT numeric(5,0),
            CREATED_BY varchar (50) NOT NULL,
            UPDATED_BY varchar (50) NOT NULL,
            CREATE_DATE TIMESTAMP NOT NULL,
            UPDATE_DATE TIMESTAMP NOT NULL,
            CONSTRAINT PK_JOB_TRACE_ID PRIMARY KEY(JOB_ID),
            CONSTRAINT FK_JOB_TRACE_JOB_ID FOREIGN KEY(JOB_ID) REFERENCES '||VSCHEMA||'.EPG_JOB(JOB_ID)
        )';
        SELECT createObject(vsql, 'create table JOB_TRACE') INTO tmp_result;
        
        vsql := 'CREATE TABLE ' || vschema || '.CUSTOM_RULE_DATA(
 			ID numeric(19,0) NOT NULL,
			RULE_ID numeric(19,0) NOT NULL,
		  	RULE_CLASS varchar(200) NOT NULL,
		  	RULE_DESC varchar(200),
		  	PARAMETERS varchar(2000),
			GLOBAL_NAME varchar(200) NOT NULL,
		  	CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
		  	CONSTRAINT PK_CUSTOM_RULE_DATA PRIMARY KEY (ID)
        )';
        SELECT createObject(vsql, 'create table CUSTOM_RULE_DATA') INTO tmp_result;
        
	    vsql:='CREATE TABLE ' || vschema || '.EDITABLE_PG_FIELDS_SETTING
        (
          PROVIDER_ID numeric(19, 0) NOT NULL,
		  SETTING varchar(3000),
		  CONSTRAINT PK_EDITABLE_PG_FIELDS_SETTING PRIMARY KEY (PROVIDER_ID),
		  CONSTRAINT FK_EDITABLE_PG_FIELDS_ID FOREIGN KEY (PROVIDER_ID) REFERENCES '||VSCHEMA||'.PARTNER (ID) ON DELETE CASCADE 
		)';
        SELECT createObject(vsql,'create TABLE EDITABLE_PG_FIELDS_SETTING') INTO tmp_result;
        
         -- Bo 10/11/2018 Add two new table for store cast and CREDITS images.
        vsql := 'CREATE TABLE '|| vschema||'.CREDITS_CAST_IMAGE
        (  
            PROGRAM_ID varchar (100) NOT NULL, 
            SEQ numeric(10) DEFAULT 1 NOT NULL ,
            ROLE varchar(200) NOT NULL, 
            PROVIDER_ID numeric(19,0) NOT NULL,
            IMAGE_ID varchar(100) NOT NULL,
            ACTOR_TYPE numeric(1) NOT NULL,
            CONSTRAINT U_CREDITS_CAST_IMAGE UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, ACTOR_TYPE, IMAGE_ID)
        )'; 
        SELECT createObject(vsql,'create table CREDITS_CAST_IMAGE') INTO tmp_result;
        SELECT CreateIndex(vschema,'CREDITS_CAST_IMAGE', 'PROVIDER_ID', 'IDX_CREDITS_CAST_IMAGE') INTO tmp_result;
      
      
        vsql := 'CREATE TABLE '|| vschema||'.STG_CREDITS_CAST_IMAGE 
        (  
		    FILENAME varchar (4000),
            PROGRAM_ID varchar (100) NOT NULL, 
            SEQ numeric(10) DEFAULT 1 NOT NULL,
            ROLE varchar(200) NOT NULL, 
            PROVIDER_ID numeric(19,0) NOT NULL,
            IMAGE_ID varchar(100) NOT NULL,
            ACTOR_TYPE numeric(1) NOT NULL,
            CONSTRAINT U_STG_CREDITS_CAST_IMAGE UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, ACTOR_TYPE, IMAGE_ID)
        )'; 
        SELECT createObject(vsql,'create table STG_CREDITS_CAST_IMAGE') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_CREDITS_CAST_IMAGE', 'PROVIDER_ID', 'IDX_STG_CREDITS_CAST_IMAGE') INTO tmp_result;
        
        --Bo 29/12/2018 add tables for support season and series
        vsql := 'CREATE TABLE '|| vschema||'.GENERIC_ENTRY_DEFINITION 
        (  
		    DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_GENERIC_ENTRY_DEFINITION UNIQUE (PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE)
        )'; 
        SELECT createObject(vsql,'create table GENERIC_ENTRY_DEFINITION') INTO tmp_result;
        SELECT CreateIndex(vschema,'GENERIC_ENTRY_DEFINITION', 'PROVIDER_ID', 'IDX_GENERIC_ENTRY_DEFINITION_PROVIDER_ID') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_GENERIC_ENTRY_DEFINITION 
        (  
            FILENAME        varchar(4000),
		    DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_STG_GENERIC_ENTRY_DEFINITION UNIQUE (PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE)
        )';
        SELECT createObject(vsql,'create table STG_GENERIC_ENTRY_DEFINITION') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_GENERIC_ENTRY_DEFINITION', 'PROVIDER_ID', 'IDX_STG_GENERIC_ENTRY_DEFINITION_PROVIDER_ID') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.ENTRY_DEFINITION_TRANSLATIONS 
        (  
            DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50) NOT NULL,
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL
        )';
        SELECT createObject(vsql,'create table ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'ENTRY_DEFINITION_TRANSLATIONS', 'PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE, LANGUAGE', 'IDX_ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_ENTRY_DEFINITION_TRANSLATIONS 
        (  
		    FILENAME         varchar (4000),
            DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50) NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL
        )';
        SELECT createObject(vsql,'create table STG_ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'STG_ENTRY_DEFINITION_TRANSLATIONS', 'PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE, LANGUAGE', 'IDX_STG_ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON 
        (  
            ID      varchar (100), 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_SERIES_SEASON PRIMARY KEY (PROVIDER_ID, ID, TYPE)
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON', 'PROVIDER_ID', 'IDX_SERIES_SEASON') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON 
        (   FILENAME varchar(4000),
            ID      varchar (100) , 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON PRIMARY KEY (PROVIDER_ID, ID, TYPE)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_TRANSLATIONS 
        (  
            ID      varchar (100), 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_SERIES_SEASON_TRANSLATIONS PRIMARY KEY (PROVIDER_ID, ID, TYPE, LANGUAGE),
            CONSTRAINT FK_SERIES_SEASON_TRANSLATIONS FOREIGN KEY (PROVIDER_ID, ID, TYPE) REFERENCES '||VSCHEMA||'.SERIES_SEASON (PROVIDER_ID, ID, TYPE) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_TRANSLATIONS', 'PROVIDER_ID', 'IDX_SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_TRANSLATIONS 
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON_TRANSLATIONS PRIMARY KEY (PROVIDER_ID, ID, TYPE, LANGUAGE)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_TRANSLATIONS', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_GENERIC 
        (   
            ID      varchar (100), 
            ASSET   varchar (50),
            NAME    varchar (200),
            VALUE   varchar (2000), 
            TYPE    varchar (10),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_SERIES_SEASON_GENERIC PRIMARY KEY (PROVIDER_ID, ID, TYPE, ASSET, NAME),
            CONSTRAINT FK_SERIES_SEASON_GENERIC FOREIGN KEY (PROVIDER_ID, ID, TYPE) REFERENCES '||VSCHEMA||'.SERIES_SEASON (PROVIDER_ID, ID, TYPE) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_GENERIC') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_GENERIC', 'PROVIDER_ID', 'IDX_SERIES_SEASON_GENERIC') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_GENERIC 
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            ASSET   varchar (50),
            NAME    varchar (200),
            VALUE   varchar (2000), 
            TYPE    varchar (10),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON_GENERIC PRIMARY KEY (PROVIDER_ID, ID, TYPE, ASSET, NAME)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_GENERIC') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_GENERIC', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_GENERIC') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_IMAGES
        (   
            ID      varchar (100), 
            IMAGE_ID   varchar (100),
            IMAGE_TYPE    varchar (10),
            PROVIDER_ID numeric(19,0),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            CONSTRAINT PK_SERIES_SEASON_IMAGES PRIMARY KEY (PROVIDER_ID, ID, IMAGE_ID, IMAGE_TYPE)
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_IMAGES') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_IMAGES', 'PROVIDER_ID', 'IDX_SERIES_SEASON_IMAGES') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_IMAGES
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            IMAGE_ID   varchar (100),
            IMAGE_TYPE    varchar (10),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON_IMAGES PRIMARY KEY (PROVIDER_ID, ID, IMAGE_ID, IMAGE_TYPE)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_IMAGES') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_IMAGES', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_IMAGES') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_CAST
        (   
            ID      varchar (100), 
            C_ID   varchar (50),
            SEQ    numeric(10) DEFAULT 1 NOT NULL,
            ROLE   varchar (200),
            TYPE   varchar (10),
            FIRST_NAME   varchar (200),
            MIDDLE_NAME  varchar (200),
            LAST_NAME    varchar (500),
            LANGUAGE        varchar (50),
            PROVIDER_ID  numeric(19,0),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            CONSTRAINT U_SERIES_SEASON_CAST UNIQUE (PROVIDER_ID, ID, TYPE, SEQ, ROLE),
            CONSTRAINT FK_SERIES_SEASON_CAST FOREIGN KEY (PROVIDER_ID, ID, TYPE) REFERENCES '||VSCHEMA||'.SERIES_SEASON (PROVIDER_ID, ID, TYPE) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_CAST') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_CAST', 'PROVIDER_ID', 'IDX_SERIES_SEASON_CAST') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_CAST
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            C_ID   varchar (50),
            SEQ    numeric(10) DEFAULT 1 NOT NULL,
            ROLE   varchar (200),
            TYPE   varchar (10),
            FIRST_NAME   varchar (200),
            MIDDLE_NAME  varchar (200),
            LAST_NAME    varchar (500),
            LANGUAGE        varchar (50),
            PROVIDER_ID  numeric(19,0),
            CONSTRAINT U_STG_SERIES_SEASON_CAST UNIQUE (PROVIDER_ID, ID, TYPE, SEQ, ROLE)
        )';
        
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_CAST') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_CAST', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_CAST') INTO tmp_result;
        
        -- entity mappings 
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',1,''dynamicEntityMappingField.scheduleId'',''SCHEDULE'',''SCHEDULE_ID'',''Int'',0,0,100,19,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',2,''dynamicEntityMappingField.scheduleStationId'',''SCHEDULE'',''STATION_ID'',''String'',1,0,100,15,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',3,''dynamicEntityMappingField.scheduleProgramId'',''SCHEDULE'',''PROGRAM_ID'',''String'',1,0,100,32,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',4,''dynamicEntityMappingField.scheduleLanguage'',''SCHEDULE'',''SCHEDULE_LANGUAGE'',''String'',1,1,100,12,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',5,''dynamicEntityMappingField.startTime'',''SCHEDULE'',''START_TIME'',''DateTime'',1,1,100,19,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',6,''dynamicEntityMappingField.endTime'',''SCHEDULE'',''END_TIME'',''DateTime'',1,0,100,19,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',8,''dynamicEntityMappingField.tvRating'',''SCHEDULE'',''TV_RATING'',''String'',1,1,100,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',9,''dynamicEntityMappingField.hd'',''SCHEDULE'',''HD_TV'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',10,''dynamicEntityMappingField.partNumber'',''SCHEDULE'',''PART_NUMBER'',''Int'',1,1,100,12,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',11,''dynamicEntityMappingField.noOfParts'',''SCHEDULE'',''NO_OF_PARTS'',''Int'',1,1,100,12,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',12,''dynamicEntityMappingField.cc'',''SCHEDULE'',''CC'',''Boolean'',1,1,100,1,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',13,''dynamicEntityMappingField.stereo'',''SCHEDULE'',''STEREO'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',14,''dynamicEntityMappingField.liveTapeDelay'',''SCHEDULE'',''LIVE_TAPE_DELAY'',''String'',1,1,100,5,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',15,''dynamicEntityMappingField.subtitled'',''SCHEDULE'',''SUBTITLED'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',16,''dynamicEntityMappingField.premiereFinale'',''SCHEDULE'',''PREMIERE_FINALE'',''String'',1,1,100,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',17,''dynamicEntityMappingField.joinedInProgress'',''SCHEDULE'',''JOINED_IN_PROGRESS'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',18,''dynamicEntityMappingField.cableInClassroom'',''SCHEDULE'',''CABLE_IN_CLASSROOM'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',19,''dynamicEntityMappingField.sap'',''SCHEDULE'',''SAP'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',20,''dynamicEntityMappingField.sexRating'',''SCHEDULE'',''SEX_RATING'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',21,''dynamicEntityMappingField.violenceRating'',''SCHEDULE'',''VIOLENCE_RATING'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',22,''dynamicEntityMappingField.languageRating'',''SCHEDULE'',''LANGUAGE_RATING'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',23,''dynamicEntityMappingField.dialogRating'',''SCHEDULE'',''DIALOG_RATING'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',24,''dynamicEntityMappingField.fvRating'',''SCHEDULE'',''FV_RATING'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',25,''dynamicEntityMappingField.enhanced'',''SCHEDULE'',''ENHANCED'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',26,''dynamicEntityMappingField.threeD'',''SCHEDULE'',''THREE_D'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',27,''dynamicEntityMappingField.letterBox'',''SCHEDULE'',''LETTERBOX'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',28,''dynamicEntityMappingField.dvs'',''SCHEDULE'',''DVS'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',29,''dynamicEntityMappingField.repeat'',''SCHEDULE'',''REPEAT'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',30,''dynamicEntityMappingField.blackout'',''SCHEDULE'',''BLACKOUT'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',31,''dynamicEntityMappingField.dolby'',''SCHEDULE'',''DOLBY'',''String'',1,1,100,5,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',32,''dynamicEntityMappingField.requestToRecord'',''SCHEDULE'',''REQUEST_RECORD'',''Boolean'',1,1,100,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',33,''dynamicEntityMappingField.programId'',''PROGRAM'',''PROGRAM_ID'',''String'',1,0,200,100,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',34,''dynamicEntityMappingField.language'',''PROGRAM'',''LANGUAGE'',''String'',1,1,200,12,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',35,''dynamicEntityMappingField.title'',''PROGRAM'',''TITLE'',''String'',1,1,200,512,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',36,''dynamicEntityMappingField.reducedTitle1'',''PROGRAM'',''REDUCED_TITLE1'',''String'',1,1,201,256,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',37,''dynamicEntityMappingField.reducedTitle2'',''PROGRAM'',''REDUCED_TITLE2'',''String'',1,1,202,256,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',38,''dynamicEntityMappingField.reducedTitle3'',''PROGRAM'',''REDUCED_TITLE3'',''String'',1,1,203,256,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',39,''dynamicEntityMappingField.reducedTitle4'',''PROGRAM'',''REDUCED_TITLE4'',''String'',1,1,204,256,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',40,''dynamicEntityMappingField.altTitle'',''PROGRAM'',''ALT_TITLE'',''String'',1,1,200,512,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',41,''dynamicEntityMappingField.reducedDescription1'',''PROGRAM'',''REDUCED_DESCRIPTION1'',''String'',1,1,220,512,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',42,''dynamicEntityMappingField.reducedDescription2'',''PROGRAM'',''REDUCED_DESCRIPTION2'',''String'',1,1,221,512,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',43,''dynamicEntityMappingField.reducedDescription3'',''PROGRAM'',''REDUCED_DESCRIPTION3'',''String'',1,1,222,512,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',44,''dynamicEntityMappingField.reducedDescription4'',''PROGRAM'',''REDUCED_DESCRIPTION4'',''String'',1,1,223,512,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',45,''dynamicEntityMappingField.advisoryDescription1'',''PROGRAM'',''ADVISORY_DESC1'',''String'',1,1,230,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',46,''dynamicEntityMappingField.advisoryDescription2'',''PROGRAM'',''ADVISORY_DESC2'',''String'',1,1,231,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',47,''dynamicEntityMappingField.advisoryDescription3'',''PROGRAM'',''ADVISORY_DESC3'',''String'',1,1,232,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',48,''dynamicEntityMappingField.advisoryDescription4'',''PROGRAM'',''ADVISORY_DESC4'',''String'',1,1,233,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',49,''dynamicEntityMappingField.advisoryDescription5'',''PROGRAM'',''ADVISORY_DESC5'',''String'',1,1,234,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',50,''dynamicEntityMappingField.advisoryDescription6'',''PROGRAM'',''ADVISORY_DESC6'',''String'',1,1,235,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',51,''dynamicEntityMappingField.genereDescription1'',''PROGRAM'',''GENRE_DESC1'',''String'',1,1,240,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',52,''dynamicEntityMappingField.genereDescription2'',''PROGRAM'',''GENRE_DESC2'',''String'',1,1,241,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',53,''dynamicEntityMappingField.genereDescription3'',''PROGRAM'',''GENRE_DESC3'',''String'',1,1,242,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',54,''dynamicEntityMappingField.genereDescription4'',''PROGRAM'',''GENRE_DESC4'',''String'',1,1,243,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',55,''dynamicEntityMappingField.genereDescription5'',''PROGRAM'',''GENRE_DESC5'',''String'',1,1,244,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',56,''dynamicEntityMappingField.genereDescription6'',''PROGRAM'',''GENRE_DESC6'',''String'',1,1,245,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',57,''dynamicEntityMappingField.description1'',''PROGRAM'',''DESCRIPTION1'',''String'',1,1,210,1024,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',58,''dynamicEntityMappingField.description2'',''PROGRAM'',''DESCRIPTION2'',''String'',1,1,211,1024,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',59,''dynamicEntityMappingField.year'',''PROGRAM'',''YEAR_PROGRAM'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',60,''dynamicEntityMappingField.mpaaRating'',''PROGRAM'',''MPAA_RATING'',''String'',1,1,200,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',61,''dynamicEntityMappingField.starRating'',''PROGRAM'',''STAR_RATING'',''String'',1,1,200,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',62,''dynamicEntityMappingField.runTime'',''PROGRAM'',''RUN_TIME'',''String'',1,1,200,4,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',63,''dynamicEntityMappingField.colorCode'',''PROGRAM'',''COLOR_CODE'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',64,''dynamicEntityMappingField.programLanguage'',''PROGRAM'',''PROGRAM_LANGUAGE'',''String'',1,1,200,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',65,''dynamicEntityMappingField.countryOfOrigin'',''PROGRAM'',''COUNTRY_OF_ORIGIN'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',66,''dynamicEntityMappingField.madeForTV'',''PROGRAM'',''MADE_FOR_TV'',''Boolean'',1,1,200,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',67,''dynamicEntityMappingField.sourceType'',''PROGRAM'',''SOURCE_TYPE'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',68,''dynamicEntityMappingField.showType'',''PROGRAM'',''SHOW_TYPE'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',69,''dynamicEntityMappingField.holiday'',''PROGRAM'',''HOLIDAY'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',70,''dynamicEntityMappingField.syndicateEpisodeNum'',''PROGRAM'',''SYNDICATE_EPISODE_NUM'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',71,''dynamicEntityMappingField.altSyndicateEpisodeNum'',''PROGRAM'',''ALT_SYNDICATE_EPI_NUM'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',72,''dynamicEntityMappingField.episodeTitle'',''PROGRAM'',''EPISODE_TITLE'',''String'',1,1,200,256,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',73,''dynamicEntityMappingField.netSynSource'',''PROGRAM'',''NET_SYN_SOURCE'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',74,''dynamicEntityMappingField.netSynType'',''PROGRAM'',''NET_SYN_TYPE'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',75,''dynamicEntityMappingField.orgStudio'',''PROGRAM'',''ORG_STUDIO'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',76,''dynamicEntityMappingField.gameDateTime'',''PROGRAM'',''GAME_DATETIME'',''DateTime'',1,1,200,19,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',77,''dynamicEntityMappingField.gameTimezone'',''PROGRAM'',''GAME_TIMEZONE'',''String'',1,1,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',78,''dynamicEntityMappingField.orgAirDate'',''PROGRAM'',''ORG_AIR_DATE'',''DateTime'',1,1,200,19,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',79,''dynamicEntityMappingField.uniqueId'',''PROGRAM'',''UNIQUE_ID'',''String'',1,0,200,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',80,''dynamicEntityMappingField.seriesId'',''SCHEDULE'',''SERIES_ID'',''String'',1,0,100,200,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',81,''dynamicEntityMappingField.stationId'',''STATION'',''STATION_ID'',''String'',1,0,300,100,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',82,''dynamicEntityMappingField.stationName'',''STATION'',''STATION_NAME'',''String'',1,1,300,200,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',83,''dynamicEntityMappingField.stationTimezone'',''STATION'',''TIME_ZONE'',''String'',1,1,300,50,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',84,''dynamicEntityMappingField.callSign'',''STATION'',''CALL_SIGN'',''String'',1,1,300,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',85,''dynamicEntityMappingField.stationCity'',''STATION'',''CITY'',''String'',1,1,300,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',86,''dynamicEntityMappingField.stationState'',''STATION'',''STATE'',''String'',1,1,300,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',87,''dynamicEntityMappingField.stationZipcode'',''STATION'',''ZIPCODE'',''String'',1,1,300,50,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',88,''dynamicEntityMappingField.stationCountry'',''STATION'',''COUNTRY'',''String'',1,1,300,15,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',89,''dynamicEntityMappingField.dmaName'',''STATION'',''DMA_NAME'',''String'',1,1,300,120,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',90,''dynamicEntityMappingField.dmaNumber'',''STATION'',''DMA_NUMBER'',''Int'',1,1,300,12,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',91,''dynamicEntityMappingField.fccChannelNumber'',''STATION'',''FCC_CHANNEL_NUM'',''Int'',1,1,300,12,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',92,''dynamicEntityMappingField.affiliation'',''STATION'',''AFFILIATION'',''String'',1,1,300,60,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',152,''dynamicEntityMappingField.programGenre'',''STATION'',''PROGRAM_GENRE'',''String'',1,1,300,200,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',93,''dynamicEntityMappingField.programUserData1'',''PROGRAM'',''USER_DATA1'',''String'',1,1,250,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',94,''dynamicEntityMappingField.programUserData2'',''PROGRAM'',''USER_DATA2'',''String'',1,1,251,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',95,''dynamicEntityMappingField.programUserData3'',''PROGRAM'',''USER_DATA3'',''String'',1,1,252,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',96,''dynamicEntityMappingField.programUserData4'',''PROGRAM'',''USER_DATA4'',''String'',1,1,253,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',97,''dynamicEntityMappingField.programUserData5'',''PROGRAM'',''USER_DATA5'',''String'',1,1,254,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',98,''dynamicEntityMappingField.programUserData6'',''PROGRAM'',''USER_DATA6'',''String'',1,1,255,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',99,''dynamicEntityMappingField.programUserData7'',''PROGRAM'',''USER_DATA7'',''String'',1,1,256,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',100,''dynamicEntityMappingField.programUserData8'',''PROGRAM'',''USER_DATA8'',''String'',1,1,257,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',101,''dynamicEntityMappingField.programUserData9'',''PROGRAM'',''USER_DATA9'',''String'',1,1,258,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',102,''dynamicEntityMappingField.programUserData10'',''PROGRAM'',''USER_DATA10'',''String'',1,1,259,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',103,''dynamicEntityMappingField.programUserData11'',''PROGRAM'',''USER_DATA11'',''String'',1,1,260,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',104,''dynamicEntityMappingField.programUserData12'',''PROGRAM'',''USER_DATA12'',''String'',1,1,261,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',105,''dynamicEntityMappingField.programUserData13'',''PROGRAM'',''USER_DATA13'',''String'',1,1,262,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',106,''dynamicEntityMappingField.scheduleUserData1'',''SCHEDULE'',''USER_DATA1'',''String'',1,1,101,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',107,''dynamicEntityMappingField.scheduleUserData2'',''SCHEDULE'',''USER_DATA2'',''String'',1,1,102,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',108,''dynamicEntityMappingField.scheduleUserData3'',''SCHEDULE'',''USER_DATA3'',''String'',1,1,103,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',109,''dynamicEntityMappingField.scheduleUserData4'',''SCHEDULE'',''USER_DATA4'',''String'',1,1,104,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',110,''dynamicEntityMappingField.scheduleUserData5'',''SCHEDULE'',''USER_DATA5'',''String'',1,1,105,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',111,''dynamicEntityMappingField.scheduleUserData6'',''SCHEDULE'',''USER_DATA6'',''String'',1,1,106,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',112,''dynamicEntityMappingField.scheduleUserData7'',''SCHEDULE'',''USER_DATA7'',''String'',1,1,107,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',113,''dynamicEntityMappingField.scheduleUserData8'',''SCHEDULE'',''USER_DATA8'',''String'',1,1,108,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',114,''dynamicEntityMappingField.scheduleUserData9'',''SCHEDULE'',''USER_DATA9'',''String'',1,1,109,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',115,''dynamicEntityMappingField.scheduleUserData10'',''SCHEDULE'',''USER_DATA10'',''String'',1,1,110,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',116,''dynamicEntityMappingField.scheduleUserData11'',''SCHEDULE'',''USER_DATA11'',''String'',1,1,111,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',117,''dynamicEntityMappingField.stationUserData1'',''STATION'',''USER_DATA1'',''String'',1,1,301,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',118,''dynamicEntityMappingField.stationUserData2'',''STATION'',''USER_DATA2'',''String'',1,1,302,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',119,''dynamicEntityMappingField.stationUserData3'',''STATION'',''USER_DATA3'',''String'',1,1,303,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',120,''dynamicEntityMappingField.stationUserData4'',''STATION'',''USER_DATA4'',''String'',1,1,304,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',121,''dynamicEntityMappingField.stationUserData5'',''STATION'',''USER_DATA5'',''String'',1,1,305,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',122,''dynamicEntityMappingField.stationUserData6'',''STATION'',''USER_DATA6'',''String'',1,1,306,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',123,''dynamicEntityMappingField.stationUserData7'',''STATION'',''USER_DATA7'',''String'',1,1,307,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',124,''dynamicEntityMappingField.stationUserData8'',''STATION'',''USER_DATA8'',''String'',1,1,308,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',125,''dynamicEntityMappingField.stationUserData9'',''STATION'',''USER_DATA9'',''String'',1,1,309,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',126,''dynamicEntityMappingField.stationUserData10'',''STATION'',''USER_DATA10'',''String'',1,1,310,1000,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',127,''dynamicEntityMappingField.headendId'',''HEADEND'',''HEADEND_ID'',''String'',1,0,400,15,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',128,''dynamicEntityMappingField.headendCommunityName'',''HEADEND'',''COMMUNITY_NAME'',''String'',1,1,400,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',129,''dynamicEntityMappingField.headendCountyName'',''HEADEND'',''COUNTY_NAME'',''String'',1,1,400,100,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',130,''dynamicEntityMappingField.headendStCountyCode'',''HEADEND'',''ST_COUNTY_CODE'',''String'',1,1,400,5,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',131,''dynamicEntityMappingField.headendCountySize'',''HEADEND'',''COUNTY_SIZE'',''String'',1,1,400,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',132,''dynamicEntityMappingField.headendState'',''HEADEND'',''STATE'',''String'',1,1,400,2,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',133,''dynamicEntityMappingField.headendZipCode'',''HEADEND'',''ZIPCODE'',''String'',1,1,400,50,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',134,''dynamicEntityMappingField.headendDMACode'',''HEADEND'',''DMA_CODE'',''String'',1,1,400,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',135,''dynamicEntityMappingField.headendDMAName'',''HEADEND'',''DMA_NAME'',''String'',1,1,400,120,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',136,''dynamicEntityMappingField.headendMSOCode'',''HEADEND'',''MSO_CODE'',''Int'',1,1,400,12,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',137,''dynamicEntityMappingField.headendDMARank'',''HEADEND'',''DMA_RANK'',''String'',1,1,400,15,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',138,''dynamicEntityMappingField.headendName'',''HEADEND'',''HEADEND_NAME'',''String'',1,1,400,400,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',139,''dynamicEntityMappingField.headendLocation'',''HEADEND'',''HEADEND_LOCATION'',''String'',1,1,400,60,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',140,''dynamicEntityMappingField.headendMSOName'',''HEADEND'',''MSO_NAME'',''String'',1,1,400,1000,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',141,''dynamicEntityMappingField.headendTimezone'',''HEADEND'',''TIME_ZONE'',''String'',1,1,400,50,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',150,''dynamicEntityMappingField.headendType'',''HEADEND'',''HEADEND_TYPE'',''String'',1,1,400,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',151,''dynamicEntityMappingField.headendCountry'',''HEADEND'',''HEADEND_COUNTRY'',''String'',1,1,400,50,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',142,''dynamicEntityMappingField.lineupId'',''LINEUP'',''LINEUP_ID'',''String'',1,0,500,12,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',143,''dynamicEntityMappingField.lineupStationId'',''LINEUP'',''STATION_ID'',''String'',1,0,500,15,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',144,''dynamicEntityMappingField.lineupHeadendId'',''LINEUP'',''HEADEND_ID'',''String'',1,0,500,15,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',145,''dynamicEntityMappingField.lineupDevice'',''LINEUP'',''DEVICE'',''String'',1,1,500,1,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',146,''dynamicEntityMappingField.lineupTMSChannel'',''LINEUP'',''TMS_CHANNEL'',''String'',1,1,500,15,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',147,''dynamicEntityMappingField.lineupServiceTier'',''LINEUP'',''SERVICE_TIER'',''String'',1,1,500,60,0)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',148,''dynamicEntityMappingField.lineupEffectiveDate'',''LINEUP'',''EFFECTIVE_DATE'',''DateTime'',1,1,500,19,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',149,''dynamicEntityMappingField.lineupExpirationDate'',''LINEUP'',''EXPIRATION_DATE'',''DateTime'',1,1,500,19,1)' INTO tmp_result;
        SELECT CreateUpdateTrg(vschema,'CONFIG_SETTING') INTO tmp_result;
        SELECT CreateUpdateTrg(vschema,'NORM_RULE_GROUP') INTO tmp_result;
        SELECT CreateUpdateTrg(vschema,'VALID_RULE_GROUP') INTO tmp_result;
    
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''ENDPOINT'',0)';
        SELECT createObject(vsql,'insert into Table SEQ_GENERATOR for ENDPOINT id generating')         INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''NORM_RULE_GROUP'',0)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for NORM_RULE_GROUP id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''VALID_RULE_GROUP'',0)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for VALID_RULE_GROUP id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''PARTNER'',0)';
        SELECT createObject(vsql,'insert into Table SEQ_GENERATOR for PARTNER id generating')  INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME, SEQ_NUMBER) VALUES(''ALARM_ACTIVITYLOG'',0)';
        SELECT createObject(vsql, 'insert into table SEQ_GENERATOR for ALARM_ACTIVITYLOG') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME, SEQ_NUMBER) VALUES(''ENTITY_MAPPING'',5)';
        SELECT createObject(vsql, 'insert into table SEQ_GENERATOR for ENTITY_MAPPING') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''PREDICATE'',0)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for PREDICATE id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''RULE'',0)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for RULE id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''CLAUSE'',0)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for CLAUSE id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''TASK'',1)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for TASK id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''TASK_SETTING'',1)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for TASK_SETTING id generating') INTO tmp_result;
        vsql:='INSERT INTO '||VSCHEMA||'.SEQ_GENERATOR(SEQ_NAME,SEQ_NUMBER) VALUES(''CUSTOM_RULE_DATA'',1)';
        SELECT createObject(vsql,'insert into table SEQ_GENERATOR for CUSTOM_RULE_DATA id generating') INTO tmp_result;
                        
        vsql:='INSERT INTO '||VSCHEMA||'.COMPRESSION (id, type, CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE)
        VALUES(1, ''DUMMY'', ''System'', ''System'', current_timestamp, current_timestamp)';
        SELECT createObject(vsql,'INSERT INTO COMPRESSION') INTO tmp_result;
                
        --Version information will be updated by installer.
		
		--TODO: consider how to pass epg_db_version to this script. Now just use a hard code value.
        --  vsql:='INSERT INTO '||VSCHEMA||'.config_setting (key,value,CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE)
        --   VALUES(''epgmanagerversion'', ''&epg_db_version'', ''System'', ''System'', current_timestamp, current_timestamp)';
		
        vsql:='INSERT INTO '||VSCHEMA||'.config_setting (key,value,CREATED_BY, UPDATED_BY, CREATE_DATE, UPDATE_DATE)
        VALUES(''epgmanagerversion'', ''' ||epg_db_version ||''', ''System'', ''System'', current_timestamp, current_timestamp)';
		
        SELECT createObject(vsql,'INSERT INTO CONFIG_SETTING') INTO tmp_result;
        
		vsql:='INSERT INTO '||VSCHEMA||'.install_table (install_message) values(''EPGManager Version ' || epg_db_version || ' install has completed SUCCESSFULLY.'')';
		
        return createObject(vsql,'insert into install_table','Y');
end;
$func_create_epgmanager$ language plpgsql;  


create or replace function set_version(vschema varchar, newVersion varchar) returns smallint as $func_set_version$
declare
    vsql text;
    msg varchar(200);
begin
        --Version information will be updated by installer.
        vsql := ' update '||VSCHEMA||'.config_setting set VALUE = '''||newVersion||''', UPDATE_DATE = current_timestamp where KEY = ''epgmanagerversion''';
        return createObject(vsql,' Updating the EPGMGR db version ');
end;
$func_set_version$ language plpgsql;  
  
  
create or replace function convertepgmanager(vschema varchar, epg_db_version varchar) returns smallint as $func_convertepgmanager$
declare
    vsql varchar;
    msg varchar(200);
    tmp_result smallint;
    vepgdbver varchar(100);
begin
      execute 'SELECT value from config_setting where key=''epgmanagerversion'''  into vepgdbver;
    
    -- the algorithm here is to compare the version of the DB against the known versions ever created.
    -- then implement the changes needed in each block to move from what is known to what is current.
    
   if vepgdbver < '4.0.1' then
        -- drop headend primary key
        vsql := 'alter table '|| vschema||'.headend drop constraint if exists PK_HEADEND cascade';
        SELECT createObject(vsql,'drop primary key of HEADEND','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.HEADEND_GENERIC drop constraint if exists HEADEND_GENERIC_PK cascade';
        SELECT createObject(vsql,'drop primary key of HEADEND_GENERIC','Y') INTO tmp_result;

        -- drop station primary key
        vsql := 'alter table '|| vschema||'.STATION drop constraint if exists PK_STATION cascade';
        SELECT createObject(vsql,'drop primary key of STATION','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.STATION_GENERIC drop constraint if exists STATION_GENERIC_PK cascade';
        SELECT createObject(vsql,'drop primary key of STATION_GENERIC','Y') INTO tmp_result;

        -- drop lineup primary key
        vsql := 'alter table '|| vschema||'.LINEUP drop constraint if exists PK_LINEUP cascade';
        SELECT createObject(vsql,'drop primary key of LINEUP','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.LINEUP_GENERIC drop constraint if exists LINEUP_GENERIC_PK cascade';
        SELECT createObject(vsql,'drop primary key of LINEUP_GENERIC','Y') INTO tmp_result;

        -- drop program primary key
        vsql := 'alter table '|| vschema||'.PROGRAM drop constraint if exists PK_PROGRAM cascade';
        SELECT createObject(vsql,'drop primary key of PROGRAM','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_SUBTITLE_COMP drop constraint if exists PROGRAM_SUBTITLE_COMP_PK cascade';
        SELECT createObject(vsql,'drop primary key of PROGRAM_SUBTITLE_COMP','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_AUDIO_COMP drop constraint if exists PROGRAM_AUDIO_COMP_PK cascade';
        SELECT createObject(vsql,'drop primary key of PROGRAM_AUDIO_COMP','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_GENERIC drop constraint if exists PROGRAM_GENERIC_PK cascade';
        SELECT createObject(vsql,'drop primary key of PROGRAM_GENERIC','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_RATING drop constraint if exists PROGRAM_RATING_PK cascade';
        SELECT createObject(vsql,'drop primary key of PROGRAM_RATING','Y') INTO tmp_result;

        -- drop schedule primary key
        vsql := 'alter table '|| vschema||'.SCHEDULE drop constraint if exists PK_SCHEDULE cascade';
        SELECT createObject(vsql,'drop primary key of SCHEDULE','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.SCHEDULE_EVENT_ID drop constraint if exists SCHEDULE_EVENT_IDS_PK cascade';
        SELECT createObject(vsql,'drop primary key of SCHEDULE_EVENT_ID','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.SCHEDULE_SERIES_ATTRIB drop constraint if exists SCHEDULE_SERIES_ATTRIB_PK cascade';
        SELECT createObject(vsql,'drop primary key of SCHEDULE_SERIES_ATTRIB','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.SCHEDULE_GENERIC drop constraint if exists SCHEDULE_GENERIC_PK cascade';
        SELECT createObject(vsql,'drop primary key of SCHEDULE_GENERIC','Y') INTO tmp_result;


        -- add primary key and other constraints
        vsql := 'alter table '|| vschema||'.HEADEND add constraint PK_HEADEND primary key(PROVIDER_ID, HEADEND_ID)';
        SELECT createObject(vsql,'add primary key of HEADEND','Y') INTO tmp_result;


        vsql := 'alter table '|| vschema||'.HEADEND_GENERIC add constraint HEADEND_GENERIC_PK primary key(PROVIDER_ID, HEADEND_ID, ASSET, NAME)';
        SELECT createObject(vsql,'add primay key of HEADEND_GENERIC','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.headend_generic 
                 add constraint FK_HEADEND_GENERIC_HEADEND_ID foreign key(PROVIDER_ID, HEADEND_ID) 
                 REFERENCES '|| vschema||'.HEADEND (PROVIDER_ID, HEADEND_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of HEADEND_GENERIC','Y') INTO tmp_result;

        -- station
        vsql := 'alter table '|| vschema||'.STATION add constraint PK_STATION primary key(PROVIDER_ID, STATION_ID)';
        SELECT createObject(vsql,'add primary key of STATION','Y') INTO tmp_result;

        -- station_generic
        vsql := 'alter table '|| vschema||'.STATION_GENERIC add constraint STATION_GENERIC_PK primary key(PROVIDER_ID, STATION_ID, ASSET, NAME)';
        SELECT createObject(vsql,'add primary key of STATION_GENERIC','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.STATION_GENERIC
                 add constraint FK_STATION_GENERIC_STATION_ID foreign key(PROVIDER_ID, STATION_ID) 
                 REFERENCES '|| vschema||'.STATION (PROVIDER_ID, STATION_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of STATION_GENERIC','Y') INTO tmp_result;

        -- lineup 
        vsql := 'alter table '|| vschema||'.LINEUP add constraint PK_LINEUP primary key(PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID)';
        SELECT createObject(vsql,'add primay key of LINEUP','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.LINEUP
                 add constraint FK_LINEUP_STATION foreign key(PROVIDER_ID, STATION_ID) 
                 REFERENCES '|| vschema||'.STATION (PROVIDER_ID, STATION_ID)';
        SELECT createObject(vsql,'add foreign key of LINEUP','Y') INTO tmp_result;


        vsql := 'alter table '|| vschema||'.LINEUP
                 add constraint FK_HEADEND_LINEUP foreign key(PROVIDER_ID, HEADEND_ID) 
                 REFERENCES '|| vschema||'.HEADEND (PROVIDER_ID, HEADEND_ID)';
        SELECT createObject(vsql,'add foreign key of LINEUP','Y') INTO tmp_result;


        -- lineup_generic
        vsql := 'alter table '|| vschema||'.LINEUP_GENERIC add constraint LINEUP_GENERIC_PK primary key(PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID, ASSET, NAME)';
        SELECT createObject(vsql,'add primay key of LINEUP_GENERIC','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.LINEUP_GENERIC
                 add constraint FK_LINEUP_GENERIC_LINEUP_ID foreign key(PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID) 
                 REFERENCES '|| vschema||'.LINEUP (PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of LINEUP_GENERIC','Y') INTO tmp_result;


        -- program
        vsql := 'alter table '|| vschema||'.PROGRAM add constraint PK_PROGRAM  primary key(PROVIDER_ID, PROGRAM_ID)';
        SELECT createObject(vsql,'add primary key of PROGRAM','Y') INTO tmp_result;
        
       
        -- program_credits        
        vsql := 'alter table '|| vschema||'.PROGRAM_CREDITS
                  add constraint FK_PROGRAM_CREDIT_PROG_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                  REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_CREDITS','Y') INTO tmp_result;

        -- program_cast
        vsql := 'alter table '|| vschema||'.PROGRAM_CAST
                  add constraint FK_PROGRAM_CAST_PROGRAM_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                  REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_CAST','Y') INTO tmp_result;

        -- program_subtitle_comp
        vsql := 'alter table '|| vschema||'.PROGRAM_SUBTITLE_COMP add constraint PROGRAM_SUBTITLE_COMP_PK primary key(PROVIDER_ID, PROGRAM_ID, TLA)';
        SELECT createObject(vsql,'add primary key of PROGRAM_SUBTITLE_COMP','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_SUBTITLE_COMP
                  add constraint FK_PROGRAM_SUBTITLE_PROGRAM_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                  REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_SUBTITLE_COMP','Y') INTO tmp_result;

        -- program_audio_comp
        vsql := 'alter table '|| vschema||'.PROGRAM_AUDIO_COMP add constraint PROGRAM_AUDIO_COMP_PK primary key(PROVIDER_ID, PROGRAM_ID, TLA)';
        SELECT createObject(vsql,'add primary key of PROGRAM_AUDIO_COMP','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_AUDIO_COMP
                 add constraint FK_PROGRAM_AUDIO_PROGRAM_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                 REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_AUDIO_COMP','Y') INTO tmp_result;

        -- program_generic
        vsql := 'alter table '|| vschema||'.PROGRAM_GENERIC add constraint PROGRAM_GENERIC_PK primary key(PROVIDER_ID, PROGRAM_ID, ASSET, NAME)';
        SELECT createObject(vsql,'add primary key of PROGRAM_GENERIC','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PROGRAM_GENERIC
                 add constraint FK_PROGRAM_GENERIC_PROGRAM_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                 REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_GENERIC','Y') INTO tmp_result;

        -- program_rating
        vsql := 'alter table '|| vschema||'.PROGRAM_RATING add constraint PROGRAM_RATING_PK primary key(PROVIDER_ID, PROGRAM_ID, AUTHORITY)';
        SELECT createObject(vsql,'add primary key of PROGRAM_RATING','Y') INTO tmp_result;

        --  this following statement has errrors, fix it in next version
        vsql := 'alter table '|| vschema||'.epgmanager.PROGRAM_RATING
                 add constraint FK_PROGRAM_RATING_PROGRAM_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                 REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_RATING','Y') INTO tmp_result;

        -- schedule
        vsql := 'alter table '|| vschema||'.SCHEDULE add constraint PK_SCHEDULE primary key(PROVIDER_ID, SCHEDULE_ID)';
        SELECT createObject(vsql,'add primary key of SCHEDULE','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.SCHEDULE
                 add constraint FK_STATION_SCHEDULE foreign key(PROVIDER_ID, STATION_ID) 
                 REFERENCES '|| vschema||'.STATION (PROVIDER_ID, STATION_ID)';
        SELECT createObject(vsql,'add foreign key of SCHEDULE','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.SCHEDULE
                 add constraint FK_SCHEDULE_PROGRAM foreign key(PROVIDER_ID, PROGRAM_ID) 
                 REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID)';
        SELECT createObject(vsql,'add foreign key of SCHEDULE','Y') INTO tmp_result;


        -- schedule_event_id
        vsql := 'alter table '|| vschema||'.SCHEDULE_EVENT_ID add constraint SCHEDULE_EVENT_IDS_PK primary key(PROVIDER_ID, PROGRAM_ID, IDENTIFIER_TYPE)';
        SELECT createObject(vsql,'add primay key of SCHEDULE_EVENT_ID','Y') INTO tmp_result;

        -- schedule_series_attrib
        vsql := 'alter table '|| vschema||'.SCHEDULE_SERIES_ATTRIB add constraint SCHEDULE_SERIES_ATTRIB_PK primary key(PROVIDER_ID, PROGRAM_ID, ATTRIBUTE_TYPE)';
        SELECT createObject(vsql,'add primay key of SCHEDULE_SERIES_ATTRIB','Y') INTO tmp_result;

        -- schedule_generic
        vsql := 'alter table '|| vschema||'.SCHEDULE_GENERIC add constraint SSCHEDULE_GENERIC_PK primary key(PROVIDER_ID, PROGRAM_ID, ASSET, NAME)';
        SELECT createObject(vsql,'add primay key of SCHEDULE_GENERIC','Y') INTO tmp_result;
        
        SELECT set_version(vschema,'4.0.1') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.0.2' then
        -- this foreign key is missing in the 4.0.1, add it back.
        vsql := 'alter table '|| vschema||'.PROGRAM_RATING
                 add constraint FK_PROGRAM_RATING_PROGRAM_ID foreign key(PROVIDER_ID, PROGRAM_ID) 
                 REFERENCES '|| vschema||'.PROGRAM (PROVIDER_ID, PROGRAM_ID) on delete cascade';
        SELECT createObject(vsql,'add foreign key of PROGRAM_RATING','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.0.2') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.0.3' then
        vsql := 'alter table '|| vschema||'.ENDPOINT ADD RETRY_COUNT NUMERIC(2,0) default 0, ADD FILE_NAME_MATCHER VARCHAR(100), ADD RETRY_DELAY NUMERIC(8,0) default 0 ';
        SELECT createObject(vsql,'add column RETRY_COUNT, FILE_NAME_MATCHER, RETRY_DELAY to ENDPOINT','Y')  INTO tmp_result;
        
        SELECT set_version(vschema,'4.0.3') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.0.4' then
        -- disable all the tasks and publishers before cleanup the EPG_FORMAT
        vsql := 'update '|| vschema||'.TASK set IS_ENABLED = 0';
        SELECT createObject(vsql,'disable all the tasks ','Y') INTO tmp_result;
        
        vsql := 'update '|| vschema||'.PUBLISHER set IS_ACTIVE = 0';
        SELECT createObject(vsql,'disable all the publishers ','Y') INTO tmp_result;
   
        -- remove all the old EPG format record
        vsql := 'delete '|| vschema||'.EPG_FORMAT';
        SELECT createObject(vsql,'delete dummy etv pbulisher plugin record','Y') INTO tmp_result;
        
        -- drop the foreign key to the EPG_FORMAT
        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY drop constraint if exists FK_PLUGIN_PROPERTY_EPG_FORMAT cascade';
        SELECT createObject(vsql,'drop foreign key FK_PLUGIN_PROPERTY_EPG_FORMAT of PLUGIN_PROPERTY','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.PUBLISHER drop constraint if exists FK_PUBLISHER_FORMAT_TYPE cascade';
        SELECT createObject(vsql,'drop foreign key FK_PUBLISHER_FORMAT_TYPE of PUBLISHER','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.TASK drop constraint if exists FK_TASK_FORMAT_ID cascade';
        SELECT createObject(vsql,'drop foreign key FK_TASK_FORMAT_ID of TASK','Y') INTO tmp_result;
        
        -- change the epg format id to string type
        vsql := 'alter table '|| vschema||'.EPG_FORMAT      alter column ID type varchar(100)';
        SELECT createObject(vsql,'alter id of EPG_FORMAT to varchar(100)','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY alter column FORMAT_ID type varchar(100)';
        SELECT createObject(vsql,'alter format_id of PLUGIN_PROPERTY to varchar(100)','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.PUBLISHER       alter column FORMAT_TYPE type varchar(100)';
        SELECT createObject(vsql,'alter FORMAT_TYPE of PUBLISHER to varchar(100)','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.TASK            alter column FORMAT_ID type varchar(100)';
        SELECT createObject(vsql,'alter FORMAT_ID of TASK to varchar(100)','Y') INTO tmp_result;
        
        
        -- add back the foreign key to the EPG_FORMAT
        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY
                 add constraint FK_PLUGIN_PROPERTY_EPG_FORMAT foreign key(FORMAT_ID) 
                 REFERENCES '|| vschema||'.EPG_FORMAT(ID)';
        SELECT createObject(vsql,'add epg format contraint for PLUGIN_PROPERTY','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.ENDPOINT alter column FILE_NAME_MATCHER type VARCHAR(3000)';
        SELECT createObject(vsql,'alter column FILE_NAME_MATCHER of ENDPOINT to varchar(3000)','Y') INTO tmp_result;
                 
        SELECT set_version(vschema,'4.0.4') INTO tmp_result;
   end if;
   
     
   if vepgdbver < '4.0.5' then
        vsql := 'alter table '|| vschema||'.TASK alter column FORMAT_ID drop not null';
        SELECT createObject(vsql,'drop the not null constraint of FORMAT_ID column in TASK','Y')  INTO tmp_result;
        
        SELECT set_version(vschema,'4.0.5') INTO tmp_result;
   end if;

   if vepgdbver < '4.0.6' then
        vsql := 'alter table '|| vschema||'.EPG_JOB ADD TASK_NAME varchar(60)';
        SELECT createObject(vsql,'add column TASK_NAME to EPG_JOB','Y') INTO tmp_result;

        SELECT set_version(vschema,'4.0.6') INTO tmp_result;
   end if;
   

   if vepgdbver < '4.0.7' then
        --truncate table EPG_FORMAT will cascade to PLUGIN_PROPERTY
        vsql := 'truncate table '|| vschema||'.EPG_FORMAT cascade';
        SELECT createObject(vsql,'truncate table TASK','Y') INTO tmp_result;
   
        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY alter column ID type varchar(2101)';
        SELECT createObject(vsql,'alter id of PLUGIN_PROPERTY to varchar(2101)','Y') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY_VALUE alter column PLUGIN_PROPERTY_ID type varchar(2101)';
        SELECT createObject(vsql,'alter id of PLUGIN_PROPERTY_VALUE to varchar(2101)','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY_VALUE drop constraint if exists FK_PPV_PLUGIN_PROPERTY_ID cascade';
        SELECT createObject(vsql,'drop foreign key FK_PPV_PLUGIN_PROPERTY_ID of PLUGIN_PROPERTY_VALUE','Y') INTO tmp_result;

        --truncate table TASK will cascade to TASK_SETTING and PLUGIN_PROPERTY_VALUE
        vsql := 'truncate table '|| vschema||'.TASK cascade';
        SELECT createObject(vsql,'truncate table TASK','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.TASK rename column FORMAT_ID to FORMAT_NAME';
        SELECT createObject(vsql,'rename FORMAT_ID to FORMAT_NAME in TASK','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.TASK alter column FORMAT_NAME type varchar(50)';
        SELECT createObject(vsql,'change FORMAT_NAME of TASK to varchar(50)','Y') INTO tmp_result;
        
        vsql := 'drop table '|| vschema||'.PUBLISHER cascade';
        SELECT createObject(vsql,'drop table PUBLISHER','Y') INTO tmp_result;

        SELECT set_version(vschema,'4.0.7') INTO tmp_result;
   end if;

   if vepgdbver < '4.0.8' then
        vsql := 'alter table '|| vschema||'.STG_SCHEDULE add SCHEDULE_ID numeric(19,0)';
        SELECT createObject(vsql,'add column SCHEDULE_ID to STG_SCHEDULE','Y') INTO tmp_result;
        
        -- drop all the triggers on the production tables
        vsql := 'drop trigger if exists headendcrt on '|| vschema||'.HEADEND';
        SELECT createObject(vsql,'drop trigger headendcrt of HEADEND','Y') INTO tmp_result;
        vsql := 'drop trigger if exists headendupd on '|| vschema||'.HEADEND';
        SELECT createObject(vsql,'drop trigger headendupd of HEADEND','Y') INTO tmp_result;

        vsql := 'drop trigger if exists headend_genericcrt on '|| vschema||'.HEADEND_GENERIC';
        SELECT createObject(vsql,'drop trigger headend_genericcrt of HEADEND_GENERIC','Y') INTO tmp_result;
        vsql := 'drop trigger if exists headend_genericupd on '|| vschema||'.HEADEND_GENERIC';
        SELECT createObject(vsql,'drop trigger headend_genericupd of HEADEND_GENERIC','Y') INTO tmp_result;

        vsql := 'drop trigger if exists lineupcrt on '|| vschema||'.LINEUP';
        SELECT createObject(vsql,'drop trigger lineupcrt of LINEUP','Y') INTO tmp_result;
        vsql := 'drop trigger if exists lineupupd on '|| vschema||'.LINEUP';
        SELECT createObject(vsql,'drop trigger lineupupd of LINEUP','Y') INTO tmp_result;

        vsql := 'drop trigger if exists lineup_genericcrt on '|| vschema||'.LINEUP_GENERIC';
        SELECT createObject(vsql,'drop trigger lineup_genericcrt of LINEUP_GENERIC','Y') INTO tmp_result;
        vsql := 'drop trigger if exists lineup_genericupd on '|| vschema||'.LINEUP_GENERIC';
        SELECT createObject(vsql,'drop trigger lineup_genericupd of LINEUP_GENERIC','Y') INTO tmp_result;

        vsql := 'drop trigger if exists stationcrt on '|| vschema||'.STATION';
        SELECT createObject(vsql,'drop trigger stationcrt of STATION','Y') INTO tmp_result;
        vsql := 'drop trigger if exists stationupd on '|| vschema||'.STATION';
        SELECT createObject(vsql,'drop trigger stationupd of STATION','Y') INTO tmp_result;

        vsql := 'drop trigger if exists station_genericcrt on '|| vschema||'.STATION_GENERIC';
        SELECT createObject(vsql,'drop trigger station_genericcrt of STATION_GENERIC','Y') INTO tmp_result;
        vsql := 'drop trigger if exists station_genericupd on '|| vschema||'.STATION_GENERIC';
        SELECT createObject(vsql,'drop trigger station_genericupd of STATION_GENERIC','Y') INTO tmp_result;

        vsql := 'drop trigger if exists programcrt on '|| vschema||'.PROGRAM';
        SELECT createObject(vsql,'drop trigger programcrt of PROGRAM','Y') INTO tmp_result;
        vsql := 'drop trigger if exists programupd on '|| vschema||'.PROGRAM';
        SELECT createObject(vsql,'drop trigger programupd of PROGRAM','Y') INTO tmp_result;

        vsql := 'drop trigger if exists program_audio_compcrt on '|| vschema||'.PROGRAM_AUDIO_COMP';
        SELECT createObject(vsql,'drop trigger program_audio_compcrt of PROGRAM_AUDIO_COMP','Y') INTO tmp_result;
        vsql := 'drop trigger if exists program_audio_compupd on '|| vschema||'.PROGRAM_AUDIO_COMP';
        SELECT createObject(vsql,'drop trigger program_audio_compupd of PROGRAM_AUDIO_COMP','Y') INTO tmp_result;

        vsql := 'drop trigger if exists program_castcrt on '|| vschema||'.PROGRAM_CAST';
        SELECT createObject(vsql,'drop trigger program_castcrt of PROGRAM_CAST','Y') INTO tmp_result;
        vsql := 'drop trigger if exists program_castupd on '|| vschema||'.PROGRAM_CAST';
        SELECT createObject(vsql,'drop trigger program_castupd of PROGRAM_CAST','Y') INTO tmp_result;

        vsql := 'drop trigger if exists program_creditscrt on '|| vschema||'.PROGRAM_CREDITS';
        SELECT createObject(vsql,'drop trigger program_creditscrt of PROGRAM_CREDITS','Y') INTO tmp_result;
        vsql := 'drop trigger if exists program_creditsupd on '|| vschema||'.PROGRAM_CREDITS';
        SELECT createObject(vsql,'drop trigger program_creditsupd of PROGRAM_CREDITS','Y') INTO tmp_result;

        vsql := 'drop trigger if exists program_genericcrt on '|| vschema||'.PROGRAM_GENERIC';
        SELECT createObject(vsql,'drop trigger program_genericcrt of PROGRAM_GENERIC','Y') INTO tmp_result;
        vsql := 'drop trigger if exists program_genericupd on '|| vschema||'.PROGRAM_GENERIC';
        SELECT createObject(vsql,'drop trigger program_genericupd of PROGRAM_GENERIC','Y') INTO tmp_result;

        vsql := 'drop trigger if exists program_ratingcrt on '|| vschema||'.PROGRAM_RATING';
        SELECT createObject(vsql,'drop trigger program_ratingcrt of PROGRAM_RATING','Y') INTO tmp_result;
        vsql := 'drop trigger if exists program_ratingupd on '|| vschema||'.PROGRAM_RATING';
        SELECT createObject(vsql,'drop trigger program_ratingupd of PROGRAM_RATING','Y') INTO tmp_result;

        vsql := 'drop trigger if exists program_subtitle_compcrt on '|| vschema||'.PROGRAM_SUBTITLE_COMP';
        SELECT createObject(vsql,'drop trigger program_subtitle_compcrt of PROGRAM_SUBTITLE_COMP','Y') INTO tmp_result;
        vsql := 'drop trigger if exists program_subtitle_compupd on '|| vschema||'.PROGRAM_SUBTITLE_COMP';
        SELECT createObject(vsql,'drop trigger program_subtitle_compupd of PROGRAM_SUBTITLE_COMP','Y') INTO tmp_result;

        vsql := 'drop trigger if exists schedulecrt on '|| vschema||'.SCHEDULE';
        SELECT createObject(vsql,'drop trigger schedulecrt of SCHEDULE','Y') INTO tmp_result;
        vsql := 'drop trigger if exists scheduleupd on '|| vschema||'.SCHEDULE';
        SELECT createObject(vsql,'drop trigger scheduleupd of SCHEDULE','Y') INTO tmp_result;

        vsql := 'drop trigger if exists schedule_event_idcrt on '|| vschema||'.SCHEDULE_EVENT_ID';
        SELECT createObject(vsql,'drop trigger schedule_event_idcrt of SCHEDULE_EVENT_ID','Y') INTO tmp_result;
        vsql := 'drop trigger if exists schedule_event_idupd on '|| vschema||'.SCHEDULE_EVENT_ID';
        SELECT createObject(vsql,'drop trigger schedule_event_idupd of SCHEDULE_EVENT_ID','Y') INTO tmp_result;

        vsql := 'drop trigger if exists schedule_genericcrt on '|| vschema||'.SCHEDULE_GENERIC';
        SELECT createObject(vsql,'drop trigger schedule_genericcrt of SCHEDULE_GENERIC','Y') INTO tmp_result;
        vsql := 'drop trigger if exists schedule_genericupd on '|| vschema||'.SCHEDULE_GENERIC';
        SELECT createObject(vsql,'drop trigger schedule_genericupd of SCHEDULE_GENERIC','Y') INTO tmp_result;

        vsql := 'drop trigger if exists schedule_series_attribcrt on '|| vschema||'.SCHEDULE_SERIES_ATTRIB';
        SELECT createObject(vsql,'drop trigger schedule_series_attribcrt of SCHEDULE_SERIES_ATTRIB','Y') INTO tmp_result;
        vsql := 'drop trigger if exists schedule_series_attribupd on '|| vschema||'.SCHEDULE_SERIES_ATTRIB';
        SELECT createObject(vsql,'drop trigger schedule_series_attribupd of SCHEDULE_SERIES_ATTRIB','Y') INTO tmp_result;

        SELECT set_version(vschema,'4.0.8') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.0.9' then
        vsql := 'alter table '|| vschema||'.JOB_TRACE alter column NODE_ADDRESS type varchar(60)';
        SELECT createObject(vsql,'extend column NODE_ADDRESS length','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.0.9') INTO tmp_result;
   end if;

   if vepgdbver < '4.1.0' then
        SELECT DropConstraint(vschema,'STG_HEADEND', 'PK_STG_HEADEND') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_HEADEND_GENERIC', 'STG_HEADEND_GENERIC_PK') INTO tmp_result;

        SELECT DropConstraint(vschema,'STG_STATION', 'PK_STG_STATION') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_STATION_GENERIC', 'STG_STATION_GENERIC_PK') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'STG_LINEUP', 'PK_STG_LINEUP') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_LINEUP_GENERIC', 'STG_LINEUP_GENERIC_PK') INTO tmp_result;

        SELECT DropConstraint(vschema,'STG_PROGRAM', 'PK_STG_PROGRAM') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_SUBTITLE_COMP', 'STG_PROGRAM_SUBTITLE_COMP_PK') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_AUDIO_COMP', 'STG_PROGRAM_AUDIO_COMP_PK') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_GENERIC', 'STG_PROGRAM_GENERIC_PK') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_RATING', 'PK_STG_PROGRAM_RATING') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'STG_SCHEDULE_EVENT_ID', 'STG_SCHEDULE_EVENT_IDS_PK') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE_SERIES_ATTRIB', 'STG_SCHEDULE_SERIES_ATTRIBS_PK') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE_GENERIC', 'STG_SCHEDULE_GENERIC_PK') INTO tmp_result;
        
        
        SELECT AddPrimaryKey(vschema,'STG_HEADEND',         'PK_STG_HEADEND', 'PROVIDER_ID, HEADEND_ID') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_HEADEND_GENERIC', 'STG_HEADEND_GENERIC_PK', 'PROVIDER_ID, HEADEND_ID, ASSET, NAME') INTO tmp_result;
        
        SELECT AddPrimaryKey(vschema,'STG_STATION', 'PK_STG_STATION', 'PROVIDER_ID, STATION_ID') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_STATION_GENERIC', 'STG_STATION_GENERIC_PK', 'PROVIDER_ID, STATION_ID, ASSET, NAME') INTO tmp_result;
        
        SELECT AddPrimaryKey(vschema,'STG_LINEUP', 'PK_STG_LINEUP', 'PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID') INTO tmp_result;
        SELECT AddForeignKey(vschema,'STG_LINEUP', 'STG_STATION', 'FK_STG_LINEUP_STATION', 'PROVIDER_ID, STATION_ID') INTO tmp_result;
        SELECT AddForeignKey(vschema,'STG_LINEUP', 'STG_HEADEND', 'FK_STG_HEADEND_LINEUP', 'PROVIDER_ID, HEADEND_ID') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_LINEUP_GENERIC', 'STG_LINEUP_GENERIC_PK', 'PROVIDER_ID, LINEUP_ID, HEADEND_ID, STATION_ID, ASSET, NAME') INTO tmp_result;

        SELECT AddPrimaryKey(vschema,'STG_PROGRAM', 'PK_STG_PROGRAM', 'PROVIDER_ID, PROGRAM_ID') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_PROGRAM_SUBTITLE_COMP',  'STG_PROGRAM_SUBTITLE_COMP_PK', 'PROVIDER_ID, PROGRAM_ID, TLA') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_PROGRAM_AUDIO_COMP', 'STG_PROGRAM_AUDIO_COMP_PK', 'PROVIDER_ID, PROGRAM_ID, TLA') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_PROGRAM_GENERIC', 'STG_PROGRAM_GENERIC_PK', 'PROVIDER_ID, PROGRAM_ID, ASSET, NAME') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_PROGRAM_RATING', 'PK_STG_PROGRAM_RATING', 'PROVIDER_ID, PROGRAM_ID, AUTHORITY') INTO tmp_result;
        
        
        SELECT AddForeignKey(vschema,'STG_SCHEDULE', 'STG_STATION', 'FK_STG_STATION_SCHEDULE', 'PROVIDER_ID, STATION_ID') INTO tmp_result;
        SELECT AddForeignKey(vschema,'STG_SCHEDULE', 'STG_PROGRAM', 'FK_STG_SCHEDULE_PROGRAM', 'PROVIDER_ID, PROGRAM_ID') INTO tmp_result;
        
        SELECT AddPrimaryKey(vschema,'STG_SCHEDULE_EVENT_ID', 'STG_SCHEDULE_EVENT_IDS_PK', 'PROVIDER_ID, PROGRAM_ID, IDENTIFIER_TYPE') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_SCHEDULE_SERIES_ATTRIB', 'STG_SCHEDULE_SERIES_ATTRIBS_PK', 'PROVIDER_ID, PROGRAM_ID, ATTRIBUTE_TYPE') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_SCHEDULE_GENERIC', 'STG_SCHEDULE_GENERIC_PK', 'PROVIDER_ID, PROGRAM_ID, ASSET, NAME') INTO tmp_result;
       
        SELECT set_version(vschema,'4.1.0') INTO tmp_result;
   end if;   
   
   if vepgdbver < '4.1.1' then
        vsql := 'alter table '|| vschema||'.TASK alter column NAME type varchar(200)';
        SELECT createObject(vsql,'extend column TASK NAME length','Y') INTO tmp_result;
        
        vsql := 'alter table '|| vschema||'.EPG_JOB
                    alter column TASK_NAME type varchar(200),
                    alter column COMMENTS type varchar(2000)';
        SELECT createObject(vsql,'extend column TASK NAME and COMMENTS length','Y') INTO tmp_result;
        
        vsql := 'alter table '|| vschema||'.PARTNER
                    alter column NAME type varchar(200),
                    alter column CONTACT type varchar(200)';
        SELECT createObject(vsql,'extend column partner NAME and CONTACT length','Y') INTO tmp_result;
        
        vsql := 'alter table '|| vschema||'.ENDPOINT alter column USER_NAME type varchar(200)';
        SELECT createObject(vsql,'extend column endpoint USER_NAME length','Y') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_JOB_PARTNERID ON EPG_JOB(PARTNER_ID)';
        SELECT createObject(vsql,'create IDX_JOB_PARTNERID','Y') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_JOB_STARTDATE ON EPG_JOB(START_DATE)';
        SELECT createObject(vsql,'create IDX_JOB_STARTDATE','Y') INTO tmp_result;
        
        SELECT set_version(vschema,'4.1.1') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.1.2' then
        SELECT DropConstraint(vschema,'PLUGIN_PROPERTY', 'FK_PLUGIN_PROPERTY_EPG_FORMAT') INTO tmp_result;

        vsql := 'alter table '|| vschema||'.PLUGIN_PROPERTY rename column FORMAT_ID to FORMAT';
        SELECT createObject(vsql,'rename FORMAT_ID to FORMAT in PLUGIN_PROPERTY') INTO tmp_result;
        
        SELECT set_version(vschema,'4.1.2') INTO tmp_result;
   end if;
   
   
   if vepgdbver < '4.1.3' then
        -- drop constraint on staging tables
        SELECT DropConstraint(vschema,'STG_HEADEND', 'FK_STG_HEADEND_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_HEADEND_GENERIC', 'FK_STG_HEADEND_GEN_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'STG_STATION', 'FK_STG_STATION_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_STATION_GENERIC', 'FK_STG_STATION_GEN_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'STG_LINEUP', 'FK_STG_LINEUP_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_LINEUP_GENERIC', 'FK_STG_LINE_GENE_PROVIDER') INTO tmp_result;

        SELECT DropConstraint(vschema,'STG_PROGRAM', 'FK_STG_PROGRAM_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_RATING', 'FK_STG_PRO_RAT_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_GENERIC', 'FK_STG_PROG_GEN_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_AUDIO_COMP', 'FK_STG_PRO_AUD_C_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_SUBTITLE_COMP', 'FK_STG_PRO_SUB_C_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_CREDITS', 'FK_STG_PRO_CRED_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_PROGRAM_CAST', 'FK_STG_PRO_CAST_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'STG_SCHEDULE', 'FK_STG_STG_SCHED_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE_EVENT_ID', 'FK_STG_SCHED_E_ID_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE_SERIES_ATTRIB', 'FK_STG_SCHE_SER_ATT_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE_GENERIC', 'FK_STG_SCHE_GEN_PROVIDER') INTO tmp_result;
        
        -- drop constraints on prod tables
        SELECT DropConstraint(vschema,'HEADEND', 'FK_HEADEND_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'HEADEND_GENERIC', 'FK_HEADEND_GENERIC_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'STATION', 'FK_STATION_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'STATION_GENERIC', 'FK_STATION_GENERIC_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'LINEUP', 'FK_LINEUP_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'LINEUP_GENERIC', 'FK_LINEUP_GENERIC_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'PROGRAM', 'FK_PROGRAM_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'PROGRAM_SUBTITLE_COMP', 'FK_PRO_SUB_COMP_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'PROGRAM_AUDIO_COMP', 'FK_PRO_AU_COMP_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'PROGRAM_GENERIC', 'FK_PRO_GEN_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'PROGRAM_RATING', 'FK_PRO_RAT_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'PROGRAM_CREDITS', 'FK_PRO_CRED_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'PROGRAM_CAST', 'FK_PROGRAM_CAST_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'SCHEDULE', 'FK_SCHEDULE_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'SCHEDULE_EVENT_ID', 'FK_SCHE_EVENT_ID_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'SCHEDULE_SERIES_ATTRIB', 'FK_SCHE_SER_ATT_PROVIDER') INTO tmp_result;
        SELECT DropConstraint(vschema,'SCHEDULE_GENERIC', 'FK_SCHE_GEN_PROVIDER') INTO tmp_result;
        
        SELECT DropConstraint(vschema,'temp_station_sch_range', 'FK_TEM_ST_S_R_PROVIDER') INTO tmp_result;
        
        --create index for stating tabels
        SELECT CreateIndex(vschema,'STG_HEADEND', 'PROVIDER_ID', 'IDX_STG_HEADEND_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_HEADEND_GENERIC', 'PROVIDER_ID','IDX_STG_HEADEND_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_STATION', 'PROVIDER_ID', 'IDX_STG_STATION_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_STATION_GENERIC', 'PROVIDER_ID', 'IDX_STG_STATION_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_LINEUP', 'PROVIDER_ID', 'IDX_STG_LINEUP_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_LINEUP_GENERIC', 'PROVIDER_ID', 'IDX_STG_LINEUP_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_PROGRAM', 'PROVIDER_ID', 'IDX_STG_PROGRAM_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_RATING', 'PROVIDER_ID', 'IDX_STG_PROGRAM_RATING_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_GENERIC', 'PROVIDER_ID', 'IDX_STG_PROGRAM_GENERIC_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_AUDIO_COMP', 'PROVIDER_ID', 'IDX_STG_PROGRAM_AUDIO_COMP_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_SUBTITLE_COMP', 'PROVIDER_ID', 'IDX_STG_PROGRAM_SUBTITLE_COMP_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_CREDITS', 'PROVIDER_ID', 'IDX_STG_PROGRAM_CREDITS_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_PROGRAM_CAST', 'PROVIDER_ID', 'IDX_STG_PROGRAM_CAST_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STG_SCHEDULE', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SCHEDULE_EVENT_ID', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_EVENT_ID_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SCHEDULE_SERIES_ATTRIB', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_SERIES_ATTRIB_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SCHEDULE_GENERIC', 'PROVIDER_ID', 'IDX_STG_SCHEDULE_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        --create index for prodution tables
        SELECT CreateIndex(vschema,'HEADEND', 'PROVIDER_ID', 'IDX_HEADEND_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'HEADEND_GENERIC', 'PROVIDER_ID', 'IDX_HEADEND_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'STATION', 'PROVIDER_ID', 'IDX_STATION_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'STATION_GENERIC', 'PROVIDER_ID', 'IDX_STATION_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'LINEUP', 'PROVIDER_ID', 'IDX_LINEUP_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'LINEUP_GENERIC', 'PROVIDER_ID', 'IDX_LINEUP_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'PROGRAM', 'PROVIDER_ID', 'IDX_PROGRAM_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_SUBTITLE_COMP', 'PROVIDER_ID', 'IDX_PROGRAM_SUBTITLE_COMP_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_AUDIO_COMP', 'PROVIDER_ID', 'IDX_PROGRAM_AUDIO_COMP_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_GENERIC', 'PROVIDER_ID', 'IDX_PROGRAM_GENERIC_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_RATING', 'PROVIDER_ID', 'IDX_PROGRAM_RATING_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_CREDITS', 'PROVIDER_ID', 'IDX_PROGRAM_CREDITS_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'PROGRAM_CAST', 'PROVIDER_ID', 'IDX_PROGRAM_CAST_PROVIDER_ID') INTO tmp_result;
        
        SELECT CreateIndex(vschema,'SCHEDULE', 'PROVIDER_ID', 'IDX_SCHEDULE_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE_EVENT_ID', 'PROVIDER_ID', 'IDX_SCHEDULE_EVENT_ID_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE_SERIES_ATTRIB', 'PROVIDER_ID', 'IDX_SCHEDULE_SERIES_ATTRIB_PROVIDER_ID') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE_GENERIC', 'PROVIDER_ID', 'IDX_SCHEDULE_GENERIC_PROVIDER_ID') INTO tmp_result;
        
        vsql := 'update '|| vschema||'.predicate set predicate_type=''DateAddPredicate'' , literal_value = substring(literal_value, 2) where predicate_type =''UpdateByFormulaPredicate'' and literal_value like ''+%''';
        SELECT createObject(vsql,'update predicate type DateAddPredicate') INTO tmp_result;
        vsql := 'update '|| vschema||'.predicate set predicate_type=''DateSubPredicate'' , literal_value = substring(literal_value, 2) where predicate_type =''UpdateByFormulaPredicate'' and literal_value like ''-%''';
        SELECT createObject(vsql,'update predicate type DateSubPredicate')        INTO tmp_result;
        
        vsql := 'alter table '|| vschema||'.ENTITY_MAPPING ADD MAX_LENGTH numeric(4,0) default 1';
        SELECT createObject(vsql,'ADD COLUMN MAX_LENGTH TO ENTITY_MAPPING') INTO tmp_result;
 
        vsql := 'alter table '|| vschema||'.ENTITY_MAPPING ADD NULLABLE numeric(1,0) default 1 NOT NULL';
        SELECT createObject(vsql,'ADD COLUMN NULLABLE TO ENTITY_MAPPING')       INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',150,''dynamicEntityMappingField.headendType'',''HEADEND'',''HEADEND_TYPE'',''String'',1,1,400,10,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',151,''dynamicEntityMappingField.headendCountry'',''HEADEND'',''HEADEND_COUNTRY'',''String'',1,1,400,50,1)' INTO tmp_result;
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',152,''dynamicEntityMappingField.programGenre'',''STATION'',''PROGRAM_GENRE'',''String'',1,1,300,200,1)' INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,1,19,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,2,15,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,3,32,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,4,12,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,5,19,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,7,6,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,8,10,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,9,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,10,12,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,11,12,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,12,1,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,13,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,14,5,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,15,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,16,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,17,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,18,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,19,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,20,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,21,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,22,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,23,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,24,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,25,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,26,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,27,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,28,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,29,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,30,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,31,5,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,32,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,33,100,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,34,12,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,35,512,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,36,256,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,37,256,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,38,256,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,39,256,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,40,512,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,41,512,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,42,512,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,43,512,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,44,512,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,45,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,46,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,47,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,48,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,49,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,50,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,51,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,52,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,53,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,54,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,55,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,56,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,57,1024,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,58,1024,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,59,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,60,10,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,61,10,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,62,4,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,63,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,64,10,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,65,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,66,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,67,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,68,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,69,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,70,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,71,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,72,256,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,73,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,74,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,75,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,76,19,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,77,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,78,19,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,79,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,80,200,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,81,100,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,82,200,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,83,50,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,84,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,85,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,86,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,87,50,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,88,15,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,89,120,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,90,12,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,91,12,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,92,60,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,93,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,94,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,95,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,96,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,97,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,98,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,99,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,100,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,101,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,102,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,103,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,104,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,105,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,106,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,107,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,108,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,109,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,110,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,111,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,112,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,113,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,114,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,115,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,116,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,117,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,118,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,119,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,120,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,121,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,122,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,123,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,124,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,125,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,126,1000,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,127,15,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,128,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,129,100,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,130,5,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,131,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,132,2,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,133,50,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,134,10,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,135,120,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,136,12,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,137,15,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,138,400,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,139,60,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,140,1000,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,141,50,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,150,10,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,151,50,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,142,12,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,143,15,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,144,15,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,145,1,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,146,15,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,147,60,0) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,148,19,1) INTO tmp_result;
        SELECT UpdateEntityMapping(vschema,149,19,1)   INTO tmp_result;

        
        vsql := 'alter table '|| vschema||'.ENDPOINT ADD AUTH_KEY varchar(2048)';
        SELECT createObject(vsql,'ADD AUTH_KEY to ENDPOINT') INTO tmp_result;
                
        vsql := 'alter table '|| vschema||'.ENDPOINT ADD AUTH_TYPE varchar(6)';
        SELECT createObject(vsql,'ADD AUTH_TYPE to ENDPOINT') INTO tmp_result;
        
        vsql := 'update '|| vschema||'.ENDPOINT set AUTH_TYPE=''ByPass'' where endpoint_type=''sftp''';
        SELECT createObject(vsql,'ADD AUTH_TYPE to ENDPOINT') INTO tmp_result;

        
        --for ETV-Lite support
        --for staging tables
        SELECT AddColumn(vschema,'STG_PROGRAM', 'REF_STATION_ID', 'varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'STG_PROGRAM', 'REF_START_TIME', 'TIMESTAMP (0) WITH TIME ZONE') INTO tmp_result;
        
        SELECT AddColumn(vschema,'STG_SCHEDULE', 'REF_STATION_ID', 'varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'STG_SCHEDULE', 'REF_START_TIME', 'TIMESTAMP (0) WITH TIME ZONE') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE', 'FK_STG_STATION_SCHEDULE') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_SCHEDULE', 'FK_STG_SCHEDULE_PROGRAM') INTO tmp_result;

        --update production tables
        SELECT DropNotNullConstraint(vschema,'STATION', 'CALL_SIGN') INTO tmp_result;
        SELECT DropNotNullConstraint(vschema,'STATION', 'AFFILIATION') INTO tmp_result;
       
        SELECT DropNotNullConstraint(vschema,'PROGRAM', 'LANGUAGE') INTO tmp_result;
        SELECT DropNotNullConstraint(vschema,'PROGRAM', 'TITLE') INTO tmp_result;
        SELECT DropNotNullConstraint(vschema,'PROGRAM', 'REDUCED_TITLE1') INTO tmp_result;
        SELECT DropNotNullConstraint(vschema,'PROGRAM', 'DESCRIPTION1') INTO tmp_result;
        SELECT DropNotNullConstraint(vschema,'PROGRAM', 'ORG_STUDIO') INTO tmp_result;
        SELECT AddColumn(vschema,'PROGRAM', 'REF_STATION_ID', 'varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'PROGRAM', 'REF_START_TIME', 'TIMESTAMP (0) WITH TIME ZONE') INTO tmp_result;
       
        SELECT DropNotNullConstraint(vschema,'SCHEDULE','PROGRAM_ID') INTO tmp_result;
        SELECT DropConstraint(vschema,'SCHEDULE', 'FK_STATION_SCHEDULE') INTO tmp_result;
        SELECT DropConstraint(vschema,'SCHEDULE', 'FK_SCHEDULE_PROGRAM') INTO tmp_result;
        SELECT AddColumn(vschema,'SCHEDULE', 'REF_STATION_ID', 'varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'SCHEDULE', 'REF_START_TIME', 'TIMESTAMP (0) WITH TIME ZONE') INTO tmp_result;
   
   
        SELECT AddColumn(vschema,'ENDPOINT', 'IS_BINARY', 'NUMERIC(1) DEFAULT 0') INTO tmp_result;
        SELECT AddColumn(vschema,'ENDPOINT', 'IS_PASSIVE', 'NUMERIC(1) DEFAULT 0') INTO tmp_result;
        
   
        SELECT DropColumn(vschema,'STG_SCHEDULE', 'END_TIME') INTO tmp_result;
        SELECT DropColumn(vschema,'SCHEDULE', 'END_TIME') INTO tmp_result;
        SELECT CreateIndex(vschema,'SCHEDULE', 'START_TIME', 'IDX_SCHED_START_TIME') INTO tmp_result;
        
        vsql := 'delete '|| vschema||'.ENTITY_MAPPING where table_name = ''SCHEDULE'' and column_name=''END_TIME''';
        SELECT createObject(vsql,'delete SCHEDULE:END_TIME from ENTITY_MAPPING') INTO tmp_result;
        
   
       --for adding provider to the entity mapping table
        SELECT AddColumn(vschema,'ENTITY_MAPPING', 'PARTNER_ID', 'numeric(19,0) DEFAULT -1 NOT NULL') INTO tmp_result;
        SELECT AddColumn(vschema,'ENTITY_MAPPING', 'DISPLAY_NAME', 'varchar(200)') INTO tmp_result;
        
        SELECT AddColumn(vschema,'NORM_RULE_GROUP', 'PROVIDER_ID', 'numeric(19,0)') INTO tmp_result;

        --for adding one column to PARTNER table
        SELECT AddColumn(vschema,'PARTNER', 'SETTING', 'varchar(4000)') INTO tmp_result;
        
        vsql := 'alter table '|| vschema||'.ENDPOINT alter column FOLDER type varchar(4000)';
        SELECT createObject(vsql,'extend column ENDPOINT FOLDER length','Y') INTO tmp_result;
        
        vsql := 'alter table '|| vschema||'.ENDPOINT alter column FILE_NAME type varchar(4000)';
        SELECT createObject(vsql,'extend column ENDPOINT FILE_NAME length','Y') INTO tmp_result;
        
        SELECT set_version(vschema,'4.1.3') INTO tmp_result;

   end if;
   
   if vepgdbver < '4.1.4' then
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_STATION_IMAGE 
        (  
             FILENAME varchar(4000),
             STATION_ID varchar (100) NOT NULL, 
             IMAGE_ID varchar(100) NOT NULL, 
             IMAGE_TYPE_ID varchar(100), 
             ACTIVE_DATE TIMESTAMP,
             EXPIRATION_DATE TIMESTAMP,
             LAST_UPDATE_DATE TIMESTAMP,
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT STG_STATION_IMAGE_PK  PRIMARY KEY (PROVIDER_ID, STATION_ID, IMAGE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_STATION_IMAGE') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_STATION_IMAGE_PROVIDER_ID ON STG_STATION_IMAGE(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_STG_STATION_IMAGE_PROVIDER_ID') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_STATION_IMAGE_STATION_ID ON STG_STATION_IMAGE(STATION_ID)';
        SELECT createObject(vsql,'create IDX_STG_STATION_IMAGE_STATION_ID') INTO tmp_result;

        vsql:='CREATE TABLE '||VSCHEMA||'.STG_PROGRAM_IMAGE 
        (  
         FILENAME varchar(4000),
         PROGRAM_ID varchar(100) NOT NULL,   
         SERIES_ID varchar(100),
         SEASON_PROGRAM_ID varchar(100),
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_ID varchar(100),
         ACTIVE_DATE TIMESTAMP,
         EXPIRATION_DATE TIMESTAMP,
         NUMBER_OF_PEOPLE numeric(12,0),
         ZOOM_LEVEL numeric(12,0),
         CAST_WEIGHT numeric(12,2),
         CAPTION varchar(4000) NULL, 
         CREDIT varchar(256) NULL, 
         LAST_UPDATE_DATE TIMESTAMP,
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT STG_PROGRAM_IMAGE_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, IMAGE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_PROGRAM_IMAGE') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_PROGRAM_IMAGE_PROVIDER_ID ON STG_PROGRAM_IMAGE(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_STG_PROGRAM_IMAGE_PROVIDER_ID') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_PROGRAM_IMAGE_PROGRAM_ID ON STG_PROGRAM_IMAGE(PROGRAM_ID)';
        SELECT createObject(vsql,'create IDX_STG_PROGRAM_IMAGE_PROGRAM_ID') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_IMAGE_FILES 
        (  
         FILENAME varchar(4000),         
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_FILE_ID varchar(100),
         FILE_URL varchar(4000),
         FILE_FORMAT_ID numeric(12,0),
         FILE_FORMAT varchar(50),
         FILE_WIDTH numeric(12,0),
         FILE_HEIGHT numeric(12,0),
         FILE_QUALITY_FACTOR numeric(12,0),
         FILE_BIT_DEPTH varchar(100),
         FILE_MARGIN numeric(12,0),
         FILE_BG_COLOR numeric(12,0),
         FILE_SIZE numeric(12,0),
         LAST_UPDATE_DATE TIMESTAMP,
         SCALING_FACTOR numeric(8,2),
         IS_FILE_READY numeric(1,0),
         GROUP_ID numeric(12,0),
         ENTITY_TYPE varchar(100),
         ENTITY_ID varchar(100),
         IMAGE_TYPE_ID varchar(100),
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT STG_IMAGE_FILES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_FILE_ID, ENTITY_TYPE, ENTITY_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_IMAGE_FILES') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_IMAGE_FILES_GROUP_ID ON STG_IMAGE_FILES(GROUP_ID)';
        SELECT createObject(vsql,'create IDX_STG_IMAGE_FILES_GROUP_ID') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_IMAGE_FILES_PROVIDER_ID ON STG_IMAGE_FILES(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_STG_IMAGE_FILES_PROVIDER_ID') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.STG_IMAGE_TYPES 
        (  
         FILENAME varchar(4000),
         IMAGE_TYPE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_NAME varchar(100),
         IMAGE_CATEGORY varchar(100),
         LAST_UPDATE_DATE TIMESTAMP,
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT STG_IMAGE_TYPES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_TYPE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE STG_IMAGE_TYPES') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STG_IMAGE_TYPES_PROVIDER_ID ON STG_IMAGE_TYPES(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_STG_IMAGE_TYPES_PROVIDER_ID') INTO tmp_result;
   
        vsql:='CREATE TABLE '||VSCHEMA||'.STATION_IMAGE 
        (  
             STATION_ID varchar (100) NOT NULL, 
             IMAGE_ID varchar(100) NOT NULL, 
             IMAGE_TYPE_ID varchar(100), 
             ACTIVE_DATE TIMESTAMP,
             EXPIRATION_DATE TIMESTAMP,
             LAST_UPDATE_DATE TIMESTAMP,
             CREATED_BY varchar (50) NOT NULL, 
             UPDATED_BY varchar (50) NOT NULL,  
             CREATE_DATE TIMESTAMP NOT NULL, 
             UPDATE_DATE TIMESTAMP NOT NULL, 
             PROVIDER_ID numeric(19,0) NOT NULL,
             CONSTRAINT STATION_IMAGE_PK   PRIMARY KEY (PROVIDER_ID, STATION_ID, IMAGE_ID),
             CONSTRAINT FK_STATION_IMAGE_STATION_ID FOREIGN KEY (PROVIDER_ID, STATION_ID) REFERENCES STATION (PROVIDER_ID, STATION_ID) ON DELETE CASCADE 
        )'; 
        SELECT createObject(vsql,'create TABLE STATION_IMAGE') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STATION_IMAGE_PROVIDER_ID ON STATION_IMAGE(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_STATION_IMAGE_PROVIDER_ID') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_STATION_IMAGE_STATION_ID ON STATION_IMAGE(STATION_ID)';
        SELECT createObject(vsql,'create IDX_STATION_IMAGE_STATION_ID')         INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.PROGRAM_IMAGE 
        (  
         PROGRAM_ID varchar(100) NOT NULL,   
         SERIES_ID varchar(100),
         SEASON_PROGRAM_ID varchar(100),
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_ID varchar(100),
         ACTIVE_DATE TIMESTAMP,
         EXPIRATION_DATE TIMESTAMP,
         NUMBER_OF_PEOPLE numeric(12,0),
         ZOOM_LEVEL numeric(12,0),
         CAST_WEIGHT numeric(12,2),
         CAPTION varchar(4000) NULL, 
         CREDIT varchar(256) NULL, 
         LAST_UPDATE_DATE TIMESTAMP,
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT PROGRAM_IMAGE_PK   PRIMARY KEY (PROVIDER_ID, PROGRAM_ID, IMAGE_ID),
         CONSTRAINT FK_PROGRAM_IMAGE_PROGRAM_ID FOREIGN KEY (PROVIDER_ID, PROGRAM_ID) REFERENCES PROGRAM (PROVIDER_ID, PROGRAM_ID) ON DELETE CASCADE
        )'; 
        SELECT createObject(vsql,'create TABLE PROGRAM_IMAGE') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_PROGRAM_IMAGE_PROVIDER_ID ON PROGRAM_IMAGE(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_PROGRAM_IMAGE_PROVIDER_ID') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_PROGRAM_IMAGE_PROGRAM_ID ON PROGRAM_IMAGE(PROGRAM_ID)';
        SELECT createObject(vsql,'create IDX_PROGRAM_IMAGE_PROGRAM_ID') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.IMAGE_FILES 
        (  
         IMAGE_ID varchar(100) NOT NULL,
         IMAGE_FILE_ID varchar(100),
         FILE_URL varchar(4000),
         FILE_FORMAT_ID numeric(12,0),
         FILE_FORMAT varchar(50),
         FILE_WIDTH numeric(12,0),
         FILE_HEIGHT numeric(12,0),
         FILE_QUALITY_FACTOR numeric(12,0),
         FILE_BIT_DEPTH varchar(100),
         FILE_MARGIN numeric(12,0),
         FILE_BG_COLOR numeric(12,0),
         FILE_SIZE numeric(12,0),
         LAST_UPDATE_DATE TIMESTAMP,
         SCALING_FACTOR numeric(8,2),
         IS_FILE_READY numeric(1,0),
         GROUP_ID numeric(12,0),
         LOCKED_BY varchar(100),
         LOCKED_AT TIMESTAMP,
         INTERNAL_FILE_ID varchar(100),
         ENTITY_TYPE varchar(100),
         ENTITY_ID varchar(100),
         IMAGE_TYPE_ID varchar(100),
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT IMAGE_FILES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_FILE_ID, ENTITY_TYPE, ENTITY_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE IMAGE_FILES') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_IMAGE_FILES_GROUP_ID ON IMAGE_FILES(GROUP_ID)';
        SELECT createObject(vsql,'create IDX_IMAGE_FILES_GROUP_ID') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_IMAGE_FILES_PROVIDER_ID ON IMAGE_FILES(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_IMAGE_FILES_PROVIDER_ID') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.IMAGE_TYPES 
        (  
         IMAGE_TYPE_ID varchar(100) NOT NULL,
         IMAGE_TYPE_NAME varchar(100),
         IMAGE_CATEGORY varchar(100),
         LAST_UPDATE_DATE TIMESTAMP,
         CREATED_BY varchar (50) NOT NULL, 
         UPDATED_BY varchar (50) NOT NULL,  
         CREATE_DATE TIMESTAMP NOT NULL, 
         UPDATE_DATE TIMESTAMP NOT NULL, 
         PROVIDER_ID numeric(19,0) NOT NULL,
         CONSTRAINT IMAGE_TYPES_PK   PRIMARY KEY (PROVIDER_ID, IMAGE_TYPE_ID)
        )'; 
        SELECT createObject(vsql,'create TABLE IMAGE_TYPES') INTO tmp_result;
        
        vsql:='CREATE INDEX IDX_IMAGE_TYPES_PROVIDER_ID ON IMAGE_TYPES(PROVIDER_ID)';
        SELECT createObject(vsql,'create IDX_IMAGE_TYPES_PROVIDER_ID') INTO tmp_result;
        
        -- Kevin, refactor Endpoint table, don't keep History data.
        vsql := 'delete from '||vschema||'.task_setting';
        SELECT createObject(vsql, 'delete task_setting record','Y') INTO tmp_result;
        
        vsql := 'delete from '||vschema||'.task';
        SELECT createObject(vsql, 'delete task record','Y') INTO tmp_result;
        
        vsql := 'delete from '||vschema||'.endpoint';
        SELECT createObject(vsql, 'delete endpoint record','Y') INTO tmp_result;
        
        SELECT DropColumn(vschema,'TASK', 'ENDPOINT_ID') INTO tmp_result;
        SELECT DropConstraint(vschema,'TASK', 'FK_TASK_ENDPOINT_ID') INTO tmp_result;
        
        SELECT DropColumn(vschema,'ENDPOINT', 'SERVER_IP') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'PORT') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'FOLDER') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'FILE_NAME') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'USER_NAME') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'PASSWORD') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'DONE_FILE') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'IS_SECURE') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'RETRY_COUNT') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'FILE_NAME_MATCHER') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'RETRY_DELAY') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'AUTH_TYPE') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'AUTH_KEY') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'IS_BINARY') INTO tmp_result;
        SELECT DropColumn(vschema,'ENDPOINT', 'IS_PASSIVE') INTO tmp_result;
        
        SELECT AddColumn(vschema,'ENDPOINT', 'TASK_ID', 'numeric(19,0) NOT NULL') INTO tmp_result;
        SELECT AddColumn(vschema,'ENDPOINT', 'FEATURE_TYPE', 'varchar(20)  NOT NULL') INTO tmp_result;
        SELECT AddColumn(vschema,'ENDPOINT', 'SETTING', 'varchar(4000)') INTO tmp_result;
        SELECT AddColumn(vschema,'ENDPOINT', 'SEQ_ORDER', 'numeric(7,0)') INTO tmp_result;
        SELECT AddColumn(vschema,'ENDPOINT', 'FEATURE_CODE', 'varchar(500)') INTO tmp_result;
        
        EXECUTE 'SELECT InsertEntityMapping('''||vschema||''',6,''dynamicEntityMappingField.endTime'',''SCHEDULE'',''END_TIME'',''DateTime'',1,1,100,19,0)' INTO tmp_result;
        vsql := 'delete '|| vschema||'.ENTITY_MAPPING where table_name = ''SCHEDULE'' and column_name=''DURATION''';
        SELECT createObject(vsql,'delete SCHEDULE:DURATION from ENTITY_MAPPING') INTO tmp_result;
        
        SELECT set_version(vschema,'4.1.4') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.1.5' then
        SELECT RenameIndex(vschema,'IDX_STG_HEADEND_PROVIDER_ID', 'IDX_STG_HEADEND') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_HEADEND_GENERIC_PROVIDER_ID', 'IDX_STG_HEADEND_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_STATION_PROVIDER_ID', 'IDX_STG_STATION') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_STATION_GENERIC_PROVIDER_ID', 'IDX_STG_STATION_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_LINEUP_PROVIDER_ID', 'IDX_STG_LINEUP') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_LINEUP_GENERIC_PROVIDER_ID', 'IDX_STG_LINEUP_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_PROVIDER_ID', 'IDX_STG_PROGRAM') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_RATING_PROVIDER_ID', 'IDX_STG_PROGRAM_RATING') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_GENERIC_PROVIDER_ID', 'IDX_STG_PROGRAM_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_AUDIO_COMP_PROVIDER_ID', 'IDX_STG_PROGRAM_AUDIO_COMP') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_SUBTITLE_COMP_PROVIDER_ID', 'IDX_STG_PROGRAM_SUBTITLE_COMP') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_SCHEDULE_PROVIDER_ID', 'IDX_STG_SCHEDULE') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_SCHEDULE_EVENT_ID_PROVIDER_ID', 'IDX_STG_SCHEDULE_EVENT_ID') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_SCHEDULE_SERIES_ATTRIB_PROVIDER_ID', 'IDX_STG_SCHEDULE_SERIES_ATTRIB') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_SCHEDULE_GENERIC_PROVIDER_ID', 'IDX_STG_SCHEDULE_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_HEADEND_PROVIDER_ID', 'IDX_HEADEND') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_HEADEND_GENERIC_PROVIDER_ID', 'IDX_HEADEND_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STATION_PROVIDER_ID', 'IDX_STATION') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STATION_GENERIC_PROVIDER_ID', 'IDX_STATION_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_LINEUP_PROVIDER_ID', 'IDX_LINEUP') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_LINEUP_GENERIC_PROVIDER_ID', 'IDX_LINEUP_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_PROVIDER_ID', 'IDX_PROGRAM') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_SUBTITLE_COMP_PROVIDER_ID', 'IDX_PROGRAM_SUBTITLE_COMP') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_AUDIO_COMP_PROVIDER_ID', 'IDX_PROGRAM_AUDIO_COMP') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_GENERIC_PROVIDER_ID', 'IDX_PROGRAM_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_RATING_PROVIDER_ID', 'IDX_PROGRAM_RATING') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_SCHEDULE_PROVIDER_ID', 'IDX_SCHEDULE') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_SCHEDULE_EVENT_ID_PROVIDER_ID', 'IDX_SCHEDULE_EVENT_ID') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_SCHEDULE_SERIES_ATTRIB_PROVIDER_ID', 'IDX_SCHEDULE_SERIES_ATTRIB') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_SCHEDULE_GENERIC_PROVIDER_ID', 'IDX_SCHEDULE_GENERIC') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_CREDITS_PROVIDER_ID', 'IDX_STG_PROGRAM_CREDITS') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_CAST_PROVIDER_ID', 'IDX_STG_PROGRAM_CAST') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_CREDITS_PROVIDER_ID', 'IDX_PROGRAM_CREDITS') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_CAST_PROVIDER_ID', 'IDX_PROGRAM_CAST') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_STATION_IMAGE_PROVIDER_ID', 'IDX_STG_STATION_IMAGE') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_IMAGE_PROVIDER_ID', 'IDX_STG_PROGRAM_IMAGE') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_IMAGE_FILES_PROVIDER_ID', 'IDX_STG_IMAGE_FILES') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STG_IMAGE_TYPES_PROVIDER_ID', 'IDX_STG_IMAGE_TYPES') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_STATION_IMAGE_PROVIDER_ID', 'IDX_STATION_IMAGE') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_PROGRAM_IMAGE_PROVIDER_ID', 'IDX_PROGRAM_IMAGE') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_IMAGE_FILES_PROVIDER_ID', 'IDX_IMAGE_FILES') INTO tmp_result;
        SELECT RenameIndex(vschema,'IDX_IMAGE_TYPES_PROVIDER_ID', 'IDX_IMAGE_TYPES') INTO tmp_result;
        
        SELECT RenameIndex(vschema,'IDX_STG_PROGRAM_IMAGE_PROGRAM_ID', 'IDX_STG_PROGRAM_IMAGE_PROGRA') INTO tmp_result;
        
        vsql := 'drop table '|| vschema||'.PUBLISHER_PLUGIN_VALUE cascade';
        SELECT createObject(vsql,'drop table PUBLISHER_PLUGIN_VALUE','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.1.5') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.1.6' then
        SELECT DropColumn(vschema,'STG_IMAGE_FILES', 'ENTITY_TYPE') INTO tmp_result;
        SELECT DropColumn(vschema,'STG_IMAGE_FILES', 'ENTITY_ID') INTO tmp_result;
        SELECT DropColumn(vschema,'STG_IMAGE_FILES', 'IMAGE_TYPE_ID') INTO tmp_result;
        --when the column in primary key is dropped, the key will be dropped automatically. Recreate it.
        SELECT AddPrimaryKey(vschema,'STG_IMAGE_FILES', 'STG_IMAGE_FILES_PK', 'PROVIDER_ID, IMAGE_FILE_ID') INTO tmp_result;
        
        SELECT DropColumn(vschema,'IMAGE_FILES', 'ENTITY_TYPE') INTO tmp_result;
        SELECT DropColumn(vschema,'IMAGE_FILES', 'ENTITY_ID') INTO tmp_result;
        SELECT DropColumn(vschema,'IMAGE_FILES', 'IMAGE_TYPE_ID') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'IMAGE_FILES', 'IMAGE_FILES_PK', 'PROVIDER_ID, IMAGE_FILE_ID') INTO tmp_result;
        
        SELECT set_version(vschema,'4.1.6') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.1.7' then
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'ENTITY_TYPE', 'varchar(100)') INTO tmp_result;
        SELECT DropConstraint(vschema,'STG_IMAGE_FILES', 'STG_IMAGE_FILES_PK', 'N') INTO tmp_result;
        vsql := 'update '|| vschema||'.STG_IMAGE_FILES s set s.entity_type = ''S'' where exists (select 1 from '|| vschema||'.stg_station_image i where s.image_id = i.image_id and s.provider_id = i.provider_id)';
        SELECT createObject(vsql,'set ENTITY_TYPE value to S for stg_image_files table','Y') INTO tmp_result;
        vsql := 'update '|| vschema||'.STG_IMAGE_FILES s set s.entity_type = ''P'' where entity_type is null';
        SELECT createObject(vsql,'set ENTITY_TYPE value to P for stg_image_files table','Y') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'STG_IMAGE_FILES', 'STG_IMAGE_FILES_PK', 'PROVIDER_ID, IMAGE_FILE_ID, ENTITY_TYPE') INTO tmp_result;
        
        SELECT AddColumn(vschema,'IMAGE_FILES', 'ENTITY_TYPE', 'varchar(100)') INTO tmp_result;
        SELECT DropConstraint(vschema,'IMAGE_FILES', 'IMAGE_FILES_PK', 'N') INTO tmp_result;
        vsql := 'update '|| vschema||'.IMAGE_FILES s set s.entity_type = ''S'' where exists (select 1 from '|| vschema||'.station_image i where s.image_id = i.image_id and s.provider_id = i.provider_id)';
        SELECT createObject(vsql,'set ENTITY_TYPE value to S for image_files table','Y') INTO tmp_result;
        vsql := 'update '|| vschema||'.IMAGE_FILES s set s.entity_type = ''P'' where entity_type is null';
        SELECT createObject(vsql,'set ENTITY_TYPE value to P for image_files table','Y') INTO tmp_result;
        SELECT AddPrimaryKey(vschema,'IMAGE_FILES', 'IMAGE_FILES_PK', 'PROVIDER_ID, IMAGE_FILE_ID, ENTITY_TYPE') INTO tmp_result;
        
        SELECT set_version(vschema,'4.1.7') INTO tmp_result;
   end if;

   if vepgdbver < '4.1.8' then
        vsql := 'alter table '|| vschema||'.EPG_JOB ADD ERROR_KEY varchar(100)';
        SELECT createObject(vsql,'add column ERROR_KEY to EPG_JOB','Y')  INTO tmp_result;

        SELECT set_version(vschema,'4.1.8') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.1.9' then
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'TOUCHED', 'numeric(1,0) default 0') INTO tmp_result;
        SELECT AddColumn(vschema,'IMAGE_FILES', 'TOUCHED', 'numeric(1,0) default 0') INTO tmp_result;

        SELECT set_version(vschema,'4.1.9') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.0' then
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'LOCKED_BY', 'varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'LOCKED_AT', 'TIMESTAMP') INTO tmp_result;
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'INTERNAL_FILE_ID', 'varchar(100)') INTO tmp_result;

        SELECT set_version(vschema,'4.2.0') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.1' then
        SELECT AddColumn(vschema,'PREDICATE', 'ORDERING', ' numeric(19,0) NOT NULL default 0') INTO tmp_result;
        
        SELECT set_version(vschema,'4.2.1') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.2' then
        SELECT AddColumn(vschema,'PREDICATE', 'RULE_ID', 'numeric(19,0) NOT NULL default 0')  INTO tmp_result;
        vsql := 'update '|| vschema||'.PREDICATE set RULE_ID=RULE.ID from RULE where RULE.PREDICATE_ID = PREDICATE.ID';
        SELECT createObject(vsql,'migrate predicat id into predict table')    INTO tmp_result;
        SELECT DropColumn(vschema,'RULE', 'PREDICATE_ID') INTO tmp_result;
        SELECT set_version(vschema,'4.2.2') INTO tmp_result;
   end if;
     
   if vepgdbver < '4.2.3' then
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'ENTITY_IDS', ' varchar(2000)') INTO tmp_result;
        SELECT AddColumn(vschema,'STG_IMAGE_FILES', 'IMAGE_TYPE_ID', ' varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'IMAGE_FILES', 'ENTITY_IDS', ' varchar(2000)') INTO tmp_result;
        SELECT AddColumn(vschema,'IMAGE_FILES', 'IMAGE_TYPE_ID', ' varchar(100)') INTO tmp_result;
        
        SELECT set_version(vschema,'4.2.3') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.4' then
	   vsql:='CREATE TABLE '||VSCHEMA||'.TASK_EXE_INFO 
            (TASK_ID numeric(19,0) NOT NULL, 
            IDENTIFY varchar(100) NOT NULL,
            TASK_NAME varchar(200) NOT NULL,
            PARTNER_ID numeric(19,0),
            START_DATE DATE, 
            END_DATE DATE, 
            FOLDERS varchar(2000), 
            FILE_NAMES varchar(2000) NOT NULL,
            COMMENTS varchar(2000),
            CONSTRAINT PK_TASK_EXE_INFO PRIMARY KEY (TASK_ID, IDENTIFY))';
        SELECT createObject(vsql,'create TABLE TASK_EXE_INFO') INTO tmp_result;
        
        SELECT set_version(vschema,'4.2.4') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.5' then
        vsql:='ALTER TABLE '||VSCHEMA||'.EPG_JOB ALTER COLUMN JOB_ID TYPE varchar(50)';
		SELECT createObject(vsql,'alter table EPG_JOB convert JOB_ID to varchar') INTO tmp_result;
		
        vsql:='ALTER TABLE '||VSCHEMA||'.JOB_TRACE ALTER COLUMN JOB_ID TYPE varchar(50)';
		SELECT createObject(vsql,'alter table JOB_TRACE convert JOB_ID to varchar') INTO tmp_result;

        SELECT set_version(vschema,'4.2.5') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.6' then
        SELECT AddColumn(vschema,'STG_SCHEDULE_SERIES_ATTRIB', 'SERIES_ID', 'varchar(100)') INTO tmp_result;
        SELECT AddColumn(vschema,'SCHEDULE_SERIES_ATTRIB', 'SERIES_ID', 'varchar(100)') INTO tmp_result;
        
        vsql := 'update '|| vschema||'.STG_SCHEDULE_SERIES_ATTRIB a set a.SERIES_ID = (select value from '|| vschema||'.STG_SCHEDULE_SERIES_ATTRIB b where a.program_id = b.program_id and b.attribute_type = ''seriesId'' and a.provider_id = b.provider_id)';
        SELECT createObject(vsql,'migrate series_id info for stg_schedule_series_attrib table ')   INTO tmp_result;
        
        vsql := 'update '|| vschema||'.SCHEDULE_SERIES_ATTRIB a set a.SERIES_ID = (select value from '|| vschema||'.SCHEDULE_SERIES_ATTRIB b where a.program_id = b.program_id and b.attribute_type = ''seriesId'' and a.provider_id = b.provider_id)';
        SELECT createObject(vsql,'migrate series_id info for schedule_series_attrib table ')   INTO tmp_result;
        
        SELECT set_version(vschema,'4.2.6') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.2.7' then
        SELECT AddColumn(vschema,'PARTNER', 'UUID', 'varchar(128)') INTO tmp_result;
        SELECT AddColumn(vschema,'TASK', 'UUID', 'varchar(128)') INTO tmp_result;
        SELECT AddColumn(vschema,'NORM_RULE_GROUP', 'UUID', 'varchar(128)') INTO tmp_result;
        SELECT AddColumn(vschema,'VALID_RULE_GROUP', 'UUID', 'varchar(128)') INTO tmp_result;
        vsql := 'update PARTNER set UUID = id; update TASK set UUID = id; update NORM_RULE_GROUP set UUID = id; update VALID_RULE_GROUP set UUID = id; ';
        SELECT createObject(vsql, 'update history data of UUID')   INTO tmp_result;
        SELECT DropConstraint(vschema,'RULE', 'u_rule_name', 'N') INTO tmp_result;
        SELECT set_version(vschema,'4.2.7') INTO tmp_result;
   end if;
   
   
   if vepgdbver < '4.2.8' then
        drop table if exists EPGMANAGER.PUBLISHER cascade;
        
        SELECT DropTable(vschema,'PUBLISHER_STATIONS') INTO tmp_result;
        SELECT DropTable(vschema,'PUBLISHER_NORM_GROUP_MAPPING') INTO tmp_result;
        
        SELECT set_version(vschema,'4.2.8') INTO tmp_result;
   end if; 

   if vepgdbver < '4.2.9' then
        vsql := 'ALTER TABLE '|| vschema||'.ALARM_ACTIVITYLOG RENAME CONSTRAINT PK_ALARM_ACTIVITYLOG TO PK_ALARM_ACTIVITYLOG_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.ALARM_ACTIVITYLOG RENAME CONSTRAINT PK_ALARM_ACTIVITYLOG TO PK_ALARM_ACTIVITYLOG_EPG','Y') INTO tmp_result;

        vsql := 'ALTER TABLE '|| vschema||'.HEADEND RENAME CONSTRAINT PK_HEADEND TO PK_HEADEND_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.HEADEND RENAME CONSTRAINT PK_HEADEND TO PK_HEADEND_EPG','Y') INTO tmp_result;

        vsql := 'ALTER TABLE '|| vschema||'.STATION RENAME CONSTRAINT PK_STATION TO PK_STATION_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.STATION RENAME CONSTRAINT PK_STATION TO PK_STATION_EPG','Y') INTO tmp_result;

        vsql := 'ALTER TABLE '|| vschema||'.LINEUP RENAME CONSTRAINT PK_LINEUP TO PK_LINEUP_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.LINEUP RENAME CONSTRAINT PK_LINEUP TO PK_LINEUP_EPG','Y') INTO tmp_result;

        vsql := 'ALTER TABLE '|| vschema||'.PROGRAM RENAME CONSTRAINT PK_PROGRAM TO PK_PROGRAM_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.PROGRAM RENAME CONSTRAINT PK_PROGRAM TO PK_PROGRAM_EPG','Y') INTO tmp_result;

        vsql := 'ALTER TABLE '|| vschema||'.SCHEDULE RENAME CONSTRAINT PK_SCHEDULE TO PK_SCHEDULE_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.SCHEDULE RENAME CONSTRAINT PK_SCHEDULE TO PK_SCHEDULE_EPG','Y') INTO tmp_result;

        vsql := 'ALTER TABLE '|| vschema||'.CONFIG_SETTING RENAME CONSTRAINT PK_CONFIG_SETTING TO PK_CONFIG_SETTING_EPG';
        SELECT createObject(vsql, 'Unable to ALTER TABLE '|| vschema||'.CONFIG_SETTING RENAME CONSTRAINT PK_CONFIG_SETTING TO PK_CONFIG_SETTING_EPG','Y') INTO tmp_result;

        SELECT set_version(vschema,'4.2.9') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.0' then
        vsql := 'alter table '|| vschema||'.ENDPOINT alter column SETTING type varchar(4000)';
        SELECT createObject(vsql,'extend column ENDPOINT SETTING length','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.3.0') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.1' then
        vsql := 'alter table '|| vschema||'.SCHEDULE alter column schedule_language drop not null';
        SELECT createObject(vsql,'drop schedule column schedule_language  not null','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.SCHEDULE alter column duration drop not null';
        SELECT createObject(vsql,'drop schedule column duration  not null','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.SCHEDULE alter column cc drop not null';
        SELECT createObject(vsql,'drop schedule column cc  not null','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.EPG_JOB alter column step_info TYPE character varying(4000)';
		SELECT createObject(vsql,'alter epg_job column step_info type character varying(4000)','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.3.1') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.2' then
        vsql := 'alter table '|| vschema||'.TASK add constraint U_TASK_UUID unique (UUID)';
        SELECT createObject(vsql,'add unique constraint of TASK(UUID)','Y') INTO tmp_result;
        vsql := 'alter table '|| vschema||'.PARTNER add constraint U_PARTNER_UUID unique (UUID)';
        SELECT createObject(vsql,'add unique constraint of PARTNER(UUID)','Y') INTO tmp_result;
        SELECT CreateIndex(vschema,'ALARM_ACTIVITYLOG', 'CREATE_DATE', 'IDX_AAL_CREATE_DATE') INTO tmp_result;
        SELECT set_version(vschema,'4.3.2') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.3' then
        vsql := 'alter table '|| vschema||'.CLAUSE alter column ENTITY_MAPPING drop not null';
        SELECT createObject(vsql,'drop not null constraint of CLAUSE(ENTITY_MAPPING)','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.3.3') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.4' then
        vsql := 'alter table '|| vschema||'.NORM_RULE_GROUP add constraint U_NORM_RULE_GROUP_UUID unique (UUID)';
        SELECT createObject(vsql,'add unique constraint of NORM_RULE_GROUP(UUID)','Y') INTO tmp_result;
        SELECT set_version(vschema,'4.3.4') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.5' then
        vsql:='CREATE TABLE '||VSCHEMA||'.TASK_VALID_GROUP_MAPPING
        (
            TASK_ID numeric(19,0) NOT NULL,
            VALID_GROUP_ID numeric(19,0) NOT NULL,
            CONSTRAINT FK_TVGM_TASK FOREIGN KEY(TASK_ID) REFERENCES '||VSCHEMA||'.TASK(ID) ON DELETE CASCADE,
            CONSTRAINT FK_TVGM_VALID_GROUP FOREIGN KEY(VALID_GROUP_ID) REFERENCES '||VSCHEMA||'.VALID_RULE_GROUP(ID) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create TABLE TASK_VALID_GROUP_MAPPING') INTO tmp_result;
        
        vsql:='CREATE TABLE '||VSCHEMA||'.TASK_NORM_GROUP_MAPPING 
        (    
            TASK_ID numeric(19,0) NOT NULL, 
            NORM_GROUP_ID numeric(19,0) NOT NULL,
            CONSTRAINT FK_TNGM_TASK FOREIGN KEY(TASK_ID) REFERENCES '||VSCHEMA||'.TASK(ID) ON DELETE CASCADE,
            CONSTRAINT FK_TNGM_VALID_GROUP FOREIGN KEY(NORM_GROUP_ID) REFERENCES '||VSCHEMA||'.NORM_RULE_GROUP(ID) ON DELETE CASCADE
        )';        
        SELECT createObject(vsql,'create TABLE TASK_NORM_GROUP_MAPPING')        INTO tmp_result;
		
		vsql :='INSERT INTO '||vschema||'.TASK_NORM_GROUP_MAPPING(TASK_ID, NORM_GROUP_ID) (SELECT t.ID, p_ng.NORM_GROUP_ID FROM '||vschema||'.TASK t INNER JOIN PARTNER_NORM_GROUP_MAPPING p_ng ON t.PARTNER_ID = p_ng.PARTNER_ID )';
     	SELECT createObject(vsql,'Migrate data from PARTNER_NORM_GROUP_MAPPING table to TASK_NORM_GROUP_MAPPING table') INTO tmp_result;
     	
     	vsql :='INSERT INTO '||vschema||'.TASK_VALID_GROUP_MAPPING(TASK_ID, VALID_GROUP_ID) (SELECT t.ID, p_vg.VALID_GROUP_ID FROM '||vschema||'.TASK t INNER JOIN PARTNER_VALID_GROUP_MAPPING p_vg ON t.PARTNER_ID =p_vg.PARTNER_ID )';
     	SELECT createObject(vsql,'Migrate data from PARTNER_VALID_GROUP_MAPPING table to TASK_VALID_GROUP_MAPPING table') INTO tmp_result;
        
        SELECT DropTable(vschema,'PARTNER_VALID_GROUP_MAPPING') INTO tmp_result;
   		SELECT DropTable(vschema,'PARTNER_NORM_GROUP_MAPPING') INTO tmp_result;
        
        SELECT set_version(vschema,'4.3.5') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.6' then
   
        vsql:='CREATE TABLE '||VSCHEMA||'.CUSTOM_RULE_DATA
        (
		  ID numeric(19,0) NOT NULL,
		  RULE_ID numeric(19,0) NOT NULL,
		  RULE_CLASS varchar(200) NOT NULL,
		  RULE_DESC varchar(200),
		  PARAMETERS varchar(2000),
		  GLOBAL_NAME varchar(200) NOT NULL,
		  CREATED_BY varchar(50) NOT NULL,
		  UPDATED_BY varchar(50) NOT NULL,
		  CREATE_DATE TIMESTAMP NOT NULL,
		  UPDATE_DATE TIMESTAMP NOT NULL,
		  CONSTRAINT PK_CUSTOM_RULE_DATA PRIMARY KEY (ID)
		)';
        SELECT createObject(vsql,'create TABLE CUSTOM_RULE_DATA') INTO tmp_result;
        
        SELECT set_version(vschema,'4.3.6') INTO tmp_result;
   end if;
   
   if vepgdbver < '4.3.7' then
   	  --STG_SCHEDULE
   	  SELECT DropColumn(vschema,'STG_SCHEDULE', 'SCHEDULE_ID') INTO tmp_result;
      
      SELECT AddColumn(vschema,'STG_SCHEDULE', 'SCHEDULE_ID', 'varchar(100)') INTO tmp_result;
      
      SELECT CreateUniqueIndex(vschema,'STG_SCHEDULE', 'SCHEDULE_ID', 'IDX_STG_SCHED_SCHEDULEID') INTO tmp_result;
      
      --STG_SCHEDULE_GENERIC
	  SELECT DropConstraint(vschema,'STG_SCHEDULE_GENERIC', 'STG_SCHEDULE_GENERIC_PK') INTO tmp_result;
      
      vsql := 'ALTER TABLE '||vschema||'.STG_SCHEDULE_GENERIC RENAME COLUMN PROGRAM_ID TO SCHEDULE_ID';
	  SELECT createObject(vsql,'Rename PROGRAM_ID to SCHEDULE_ID table STG_SCHEDULE_GENERIC','Y') INTO tmp_result;
      
      SELECT AddPrimaryKey(vschema,'STG_SCHEDULE_GENERIC',         'STG_SCHEDULE_GENERIC_PK', 'SCHEDULE_ID, PROVIDER_ID, ASSET, NAME') INTO tmp_result;
      
      --SCHEDULE
	  SELECT DropConstraint(vschema,'SCHEDULE', 'PK_SCHEDULE_EPG') INTO tmp_result;
	  
      SELECT DropColumn(vschema,'SCHEDULE', 'SCHEDULE_ID') INTO tmp_result;
      
      SELECT AddColumn(vschema,'SCHEDULE', 'SCHEDULE_ID', 'varchar(100)') INTO tmp_result;
      
      SELECT CreateUniqueIndex(vschema,'SCHEDULE', 'SCHEDULE_ID', 'IDX_SCHED_SCHEDULEID') INTO tmp_result;
	  
      --SCHEDULE_GENERIC
	  SELECT DropConstraint(vschema,'SCHEDULE_GENERIC', 'SCHEDULE_GENERIC_PK') INTO tmp_result;
	  
	  vsql := 'ALTER TABLE '||vschema||'.SCHEDULE_GENERIC RENAME COLUMN PROGRAM_ID TO SCHEDULE_ID';
	  SELECT createObject(vsql,'Rename PROGRAM_ID to SCHEDULE_ID table SCHEDULE_GENERIC','Y') INTO tmp_result;
	  
      SELECT AddPrimaryKey(vschema,'SCHEDULE_GENERIC',         'SCHEDULE_GENERIC_PK', 'SCHEDULE_ID, PROVIDER_ID, ASSET, NAME') INTO tmp_result;
        
      SELECT set_version(vschema,'4.3.7') INTO tmp_result;
   end if;

   if vepgdbver < '4.3.8' then
      vsql:='CREATE TABLE '||VSCHEMA||'.EDITABLE_PG_FIELDS_SETTING
        (
          PROVIDER_ID numeric(19, 0) NOT NULL,
		  SETTING varchar(3000),
		  CONSTRAINT PK_EDITABLE_PG_FIELDS_SETTING PRIMARY KEY (PROVIDER_ID),
		  CONSTRAINT FK_EDITABLE_PG_FIELDS_ID FOREIGN KEY (PROVIDER_ID) REFERENCES '||VSCHEMA||'.PARTNER (ID) ON DELETE CASCADE 
		)';
      SELECT createObject(vsql,'create TABLE EDITABLE_PG_FIELDS_SETTING') INTO tmp_result;
        
      SELECT AddColumn(vschema,'SCHEDULE', 'EXPIRED_TIME', 'TIMESTAMP') INTO tmp_result;
        
      SELECT AddColumn(vschema,'NORM_RULE_GROUP', 'IS_FOR_INGEST', 'numeric(1,0) NOT NULL DEFAULT 1') INTO tmp_result;
      
      vsql := 'Update '||vschema||'.NORM_RULE_GROUP set IS_FOR_INGEST = 0 where GLOBAL_FOR_PUBLISHERS = 1';
      SELECT createObject(vsql,'update NORM_RULE_GROUP IS_FOR_INGEST value') INTO tmp_result;
      
      vsql := 'Update '||vschema||'.NORM_RULE_GROUP ng set ng.IS_FOR_INGEST = 0 
               where ng.id in (select tngm.norm_group_id from '||vschema||'.TASK_NORM_GROUP_MAPPING tngm join TASK t on tngm.task_id = t.id and T.type = ''EXPORT'')';
      SELECT createObject(vsql,'update NORM_RULE_GROUP IS_FOR_INGEST value') INTO tmp_result;
        

      vsql := 'alter table '|| vschema||'.TASK_SETTING alter column SETTING type varchar(20000)';
      SELECT createObject(vsql,'alter SETTING of TASK_SETTING to varchar(20000)','Y') INTO tmp_result;

      SELECT CreateIndex(vschema,'PROGRAM', 'PROGRAM_ID', 'IDX_PROGRAM_PROGRAMID') INTO tmp_result;
      SELECT CreateIndex(vschema,'SCHEDULE', 'UPDATE_DATE', 'IDX_SCHED_UPDATE_DATE') INTO tmp_result;
      SELECT set_version(vschema,'4.3.8') INTO tmp_result;
   end if;      
   
   if vepgdbver < '4.3.9' then
      SELECT DropIndex(vschema,'IDX_SCHED_SCHEDULEID') INTO tmp_result;
      SELECT DropIndex(vschema,'IDX_STG_SCHED_SCHEDULEID') INTO tmp_result;
      
      SELECT CreateUniqueIndex(vschema,'STG_SCHEDULE', 'SCHEDULE_ID, PROVIDER_ID', 'IDX_STG_SCHED_PROVID_SCHEDID') INTO tmp_result;
      SELECT CreateUniqueIndex(vschema,'SCHEDULE', 'SCHEDULE_ID, PROVIDER_ID', 'IDX_SCHED_PROVID_SCHEDID') INTO tmp_result;
      
      vsql :='UPDATE '||vschema||'.ENTITY_MAPPING SET ALLOW_AS_PREDICATE = 0 WHERE ID = 6';
      SELECT createObject(vsql,'UPDATE schedule end time predicate to 0') INTO tmp_result;
      SELECT set_version(vschema,'4.3.9') INTO tmp_result;
   end if;  
   
   if vepgdbver < '6.0.0' then
      EXECUTE 'ALTER TABLE '|| vschema||'.CONFIG_SETTING ALTER COLUMN CREATE_DATE SET DEFAULT (CURRENT_TIMESTAMP), ALTER COLUMN UPDATE_DATE SET DEFAULT (CURRENT_TIMESTAMP)';
      EXECUTE 'ALTER TABLE '|| vschema||'.NORM_RULE_GROUP ALTER COLUMN CREATE_DATE SET DEFAULT (CURRENT_TIMESTAMP), ALTER COLUMN UPDATE_DATE SET DEFAULT (CURRENT_TIMESTAMP)';
      EXECUTE 'ALTER TABLE '|| vschema||'.VALID_RULE_GROUP ALTER COLUMN CREATE_DATE SET DEFAULT (CURRENT_TIMESTAMP), ALTER COLUMN UPDATE_DATE SET DEFAULT (CURRENT_TIMESTAMP)';
      SELECT set_version(vschema,'6.0.0') INTO tmp_result;
   end if;  

   if vepgdbver < '6.0.1' then
       -- Suresh found the 8 table column names have numeric in column name instead of NUMBER on 10/17/2016.
      vsql :='ALTER TABLE '||vschema||'.SEQ_GENERATOR RENAME COLUMN SEQ_numeric TO SEQ_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.SEQ_GENERATOR RENAME COLUMN SEQ_numeric TO SEQ_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.STG_STATION RENAME COLUMN DMA_numeric TO DMA_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.STG_STATION RENAME COLUMN DMA_numeric TO DMA_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.STG_SCHEDULE RENAME COLUMN PART_numeric TO PART_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.STG_SCHEDULE RENAME COLUMN PART_numeric TO PART_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.CLAUSE RENAME COLUMN LAYER_numeric TO LAYER_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.CLAUSE RENAME COLUMN LAYER_numeric TO LAYER_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.CLAUSE RENAME COLUMN GROUP_numeric TO GROUP_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.CLAUSE RENAME COLUMN GROUP_numeric TO GROUP_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.STATION RENAME COLUMN DMA_numeric TO DMA_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.STATION RENAME COLUMN DMA_numeric TO DMA_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.SCHEDULE RENAME COLUMN PART_numeric TO PART_NUMBER';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.SCHEDULE RENAME COLUMN PART_numeric TO PART_NUMBER','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.STG_PROGRAM_IMAGE RENAME COLUMN numeric_OF_PEOPLE TO NUMBER_OF_PEOPLE';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.STG_PROGRAM_IMAGE RENAME COLUMN numeric_OF_PEOPLE TO NUMBER_OF_PEOPLE','Y') INTO tmp_result;

      vsql :='ALTER TABLE '||vschema||'.PROGRAM_IMAGE RENAME COLUMN numeric_OF_PEOPLE TO NUMBER_OF_PEOPLE';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.PROGRAM_IMAGE RENAME COLUMN numeric_OF_PEOPLE TO NUMBER_OF_PEOPLE','Y') INTO tmp_result;

      vsql :='UPDATE '||vschema||'.entity_mapping SET NAME=''dynamicEntityMappingField.partNumber'',  column_name=''PART_NUMBER'' where name=''dynamicEntityMappingField.partnumeric'' and column_name=''PART_numeric''';
      SELECT createObject(vsql,'UPDATE '||vschema||'.entity_mapping SET NAME=''dynamicEntityMappingField.partnumber'' where name=''dynamicEntityMappingField.partnumeric''','Y') INTO tmp_result;

      vsql :='UPDATE '||vschema||'.entity_mapping SET NAME=''dynamicEntityMappingField.dmaNumber'', column_name=''DMA_NUMBER'' where name=''dynamicEntityMappingField.dmanumeric'' and column_name=''DMA_numeric''';
      SELECT createObject(vsql,'UPDATE '||vschema||'.entity_mapping SET NAME=''dynamicEntityMappingField.dmanumber'' where name=''dynamicEntityMappingField.dmanumeric''','Y') INTO tmp_result;

      SELECT set_version(vschema,'6.0.1') INTO tmp_result;
   end if;

   if vepgdbver < '6.0.2' then

      vsql :='UPDATE '||vschema||'.ENTITY_MAPPING SET NAME=''dynamicEntityMappingField.fccChannelNumber'' WHERE NAME = ''dynamicEntityMappingField.fccChannelnumeric''';
      SELECT createObject(vsql,'UPDATE '||vschema||'.ENTITY_MAPPING SET NAME=''dynamicEntityMappingField.fccChannelNumber''','Y') INTO tmp_result;      

      SELECT set_version(vschema,'6.0.2') INTO tmp_result;
   end if;

   if vepgdbver < '6.1.0' then

      vsql :='ALTER TABLE '||vschema||'.install_table ALTER COLUMN msgdate SET default current_timestamp';
      SELECT createObject(vsql,'ALTER TABLE '||vschema||'.install_table ALTER COLUMN msgdate SET default current_timestamp','Y') INTO tmp_result;      

      SELECT set_version(vschema,'6.1.0') INTO tmp_result;
   end if;

   if vepgdbver < '7.0.0' then
      -- )n 03/16/2018 Changed the database from edb to PostgreSQL10 and incremented the db version number.
      SELECT set_version(vschema,'7.0.0') INTO tmp_result;
   end if;
   
   if vepgdbver < '7.1.0' then
      -- Bo 10/11/2018 alter START_DATE of EPG_JOB to TIMESTAMP.
      vsql := 'alter table '|| vschema||'.EPG_JOB alter column START_DATE type TIMESTAMP';
      SELECT createObject(vsql,'alter START_DATE of EPG_JOB to TIMESTAMP ','Y') INTO tmp_result;
      
      vsql := 'alter table '|| vschema||'.EPG_JOB alter column END_DATE type TIMESTAMP';
      SELECT createObject(vsql,'alter END_DATE of EPG_JOB to TIMESTAMP ','Y') INTO tmp_result;
      SELECT set_version(vschema,'7.1.0') INTO tmp_result;
   end if;
   
   if vepgdbver < '7.1.1' then
      -- Bo 10/11/2018 Add two new table for store cast and CREDITS images.
      vsql := 'CREATE TABLE '|| vschema||'.CREDITS_CAST_IMAGE
        (  
            PROGRAM_ID varchar (100) NOT NULL, 
            SEQ numeric(10) DEFAULT 1 NOT NULL ,
            ROLE varchar(200) NOT NULL, 
            PROVIDER_ID numeric(19,0) NOT NULL,
            IMAGE_ID varchar(100) NOT NULL,
            ACTOR_TYPE numeric(1) NOT NULL,
            CONSTRAINT U_CREDITS_CAST_IMAGE UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, ACTOR_TYPE, IMAGE_ID)
        )'; 
      SELECT createObject(vsql,'create table CREDITS_CAST_IMAGE') INTO tmp_result;
      SELECT CreateIndex(vschema,'CREDITS_CAST_IMAGE', 'PROVIDER_ID', 'IDX_CREDITS_CAST_IMAGE') INTO tmp_result;
      
      
      vsql := 'CREATE TABLE '|| vschema||'.STG_CREDITS_CAST_IMAGE 
        (  
		    FILENAME varchar (4000),
            PROGRAM_ID varchar (100) NOT NULL, 
            SEQ numeric(10) DEFAULT 1 NOT NULL ,
            ROLE varchar(200) NOT NULL, 
            PROVIDER_ID numeric(19,0) NOT NULL,
            IMAGE_ID varchar(100) NOT NULL,
            ACTOR_TYPE numeric(1) NOT NULL,
            CONSTRAINT U_STG_CREDITS_CAST_IMAGE UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, ACTOR_TYPE, IMAGE_ID)
        )'; 
       SELECT createObject(vsql,'create table STG_CREDITS_CAST_IMAGE') INTO tmp_result;
       SELECT CreateIndex(vschema,'STG_CREDITS_CAST_IMAGE', 'PROVIDER_ID', 'IDX_STG_CREDITS_CAST_IMAGE') INTO tmp_result;

      SELECT set_version(vschema,'7.1.1') INTO tmp_result;
   end if;
   
   if vepgdbver < '7.1.2' then
       -- Bo 26/12/2018 Add Language Field in Cast/credits Table.
       vsql := 'alter table '|| vschema||'.PROGRAM_CAST add column language varchar(100) default ''en-US'' not null ';
       SELECT createObject(vsql,'add column language to PROGRAM_CAST','Y') INTO tmp_result;
       vsql := 'alter table '|| vschema||'.PROGRAM_CREDITS add column language varchar(100) default ''en-US'' not null ';
       SELECT createObject(vsql,'add column language to PROGRAM_CREDITS','Y') INTO tmp_result;
       vsql := 'alter table '|| vschema||'.STG_PROGRAM_CAST add column language varchar(100) default ''en-US'' not null ';
       SELECT createObject(vsql,'add column language to STG_PROGRAM_CAST','Y') INTO tmp_result;
       vsql := 'alter table '|| vschema||'.STG_PROGRAM_CREDITS add column language varchar(100) default ''en-US'' not null ';
       SELECT createObject(vsql,'add column language to STG_PROGRAM_CREDITS','Y') INTO tmp_result;
       
       SELECT DropConstraint(vschema,'PROGRAM_CAST', 'u_program_cast') INTO tmp_result;
       SELECT DropConstraint(vschema,'PROGRAM_CREDITS', 'u_program_credits') INTO tmp_result;
       SELECT DropConstraint(vschema,'STG_PROGRAM_CAST', 'u_stg_program_cast') INTO tmp_result;
       SELECT DropConstraint(vschema,'STG_PROGRAM_CREDITS', 'u_stg_program_credits') INTO tmp_result;
       
       vsql := 'alter table '|| vschema||'.PROGRAM_CAST add constraint u_program_cast UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE)';
       SELECT createObject(vsql,'add unique constraint of PROGRAM_CAST to PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE ','Y') INTO tmp_result;
      
       vsql := 'alter table '|| vschema||'.PROGRAM_CREDITS add constraint u_program_credits UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE)';
       SELECT createObject(vsql,'add unique constraint of PROGRAM_CREDITS to PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE ','Y') INTO tmp_result;
       
       vsql := 'alter table '|| vschema||'.STG_PROGRAM_CAST add constraint u_stg_program_cast UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE)';
       SELECT createObject(vsql,'add unique constraint of STG_PROGRAM_CAST to PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE ','Y') INTO tmp_result;
      
       vsql := 'alter table '|| vschema||'.STG_PROGRAM_CREDITS add constraint u_stg_program_credits UNIQUE (PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE)';
       SELECT createObject(vsql,'add unique constraint of STG_PROGRAM_CREDITS to PROVIDER_ID, PROGRAM_ID, SEQ, ROLE, LANGUAGE ','Y') INTO tmp_result;
       
       
       SELECT set_version(vschema,'7.1.2') INTO tmp_result;
   end if;
   
   if vepgdbver < '7.1.3' then
        --Bo 29/12/2018 add tables for support season and series
        vsql := 'CREATE TABLE '|| vschema||'.GENERIC_ENTRY_DEFINITION 
        (  
		    DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_GENERIC_ENTRY_DEFINITION UNIQUE (PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE)
        )'; 
        SELECT createObject(vsql,'create table GENERIC_ENTRY_DEFINITION') INTO tmp_result;
        SELECT CreateIndex(vschema,'GENERIC_ENTRY_DEFINITION', 'PROVIDER_ID', 'IDX_GENERIC_ENTRY_DEFINITION_PROVIDER_ID') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_GENERIC_ENTRY_DEFINITION 
        (  
            FILENAME        varchar(4000),
		    DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50),
            PROVIDER_ID numeric(19,0) NOT NULL,
            CONSTRAINT U_STG_GENERIC_ENTRY_DEFINITION UNIQUE (PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE)
        )';
        SELECT createObject(vsql,'create table STG_GENERIC_ENTRY_DEFINITION') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_GENERIC_ENTRY_DEFINITION', 'PROVIDER_ID', 'IDX_STG_GENERIC_ENTRY_DEFINITION_PROVIDER_ID') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.ENTRY_DEFINITION_TRANSLATIONS 
        (  
            DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL
        )';
        SELECT createObject(vsql,'create table ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'ENTRY_DEFINITION_TRANSLATIONS', 'PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE, LANGUAGE', 'IDX_ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_ENTRY_DEFINITION_TRANSLATIONS 
        (   FILENAME varchar(4000),
            DEFINITION_ID    varchar (10) NOT NULL, 
            DEFINITION_TYPE  varchar (10) NOT NULL,
            DEFINITION_VALUE varchar(4000) NOT NULL, 
            LANGUAGE        varchar (50) NOT NULL,
            PROVIDER_ID numeric(19,0) NOT NULL
        )';
        SELECT createObject(vsql,'create table STG_ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        SELECT CreateUniqueIndex(vschema,'STG_ENTRY_DEFINITION_TRANSLATIONS', 'PROVIDER_ID, DEFINITION_ID, DEFINITION_TYPE, LANGUAGE', 'IDX_STG_ENTRY_DEFINITION_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON 
        (  
            ID      varchar (100), 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_SERIES_SEASON PRIMARY KEY (PROVIDER_ID, ID, TYPE)
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON', 'PROVIDER_ID', 'IDX_SERIES_SEASON') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON 
        (   FILENAME varchar(4000),
            ID      varchar (100) , 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON PRIMARY KEY (PROVIDER_ID, ID, TYPE)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_TRANSLATIONS 
        (  
            ID      varchar (100), 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_SERIES_SEASON_TRANSLATIONS PRIMARY KEY (PROVIDER_ID, ID, TYPE, LANGUAGE),
            CONSTRAINT FK_SERIES_SEASON_TRANSLATIONS FOREIGN KEY (PROVIDER_ID, ID, TYPE) REFERENCES '||VSCHEMA||'.SERIES_SEASON (PROVIDER_ID, ID, TYPE) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_TRANSLATIONS', 'PROVIDER_ID', 'IDX_SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_TRANSLATIONS 
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            TITLE   varchar (300),
            DESCRIPTION    varchar (1000), 
            TYPE    varchar (50),
            LANGUAGE        varchar (50),
            REDUCED_TITLE    varchar (300),
            REDUCED_DESC     varchar (1000),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON_TRANSLATIONS PRIMARY KEY (PROVIDER_ID, ID, TYPE, LANGUAGE)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_TRANSLATIONS', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_TRANSLATIONS') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_GENERIC 
        (   
            ID      varchar (100), 
            ASSET   varchar (50),
            NAME    varchar (200),
            VALUE   varchar (2000), 
            TYPE    varchar (10),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_SERIES_SEASON_GENERIC PRIMARY KEY (PROVIDER_ID, ID, TYPE, ASSET, NAME),
            CONSTRAINT FK_SERIES_SEASON_GENERIC FOREIGN KEY (PROVIDER_ID, ID, TYPE) REFERENCES '||VSCHEMA||'.SERIES_SEASON (PROVIDER_ID, ID, TYPE) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_GENERIC') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_GENERIC', 'PROVIDER_ID', 'IDX_SERIES_SEASON_GENERIC') INTO tmp_result;
        
        vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_GENERIC 
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            ASSET   varchar (50),
            NAME    varchar (200),
            VALUE   varchar (2000), 
            TYPE    varchar (10),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON_GENERIC PRIMARY KEY (PROVIDER_ID, ID, TYPE, ASSET, NAME)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_GENERIC') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_GENERIC', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_GENERIC') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_IMAGES
        (   
            ID      varchar (100), 
            IMAGE_ID   varchar (100),
            IMAGE_TYPE    varchar (10),
            PROVIDER_ID numeric(19,0),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            CONSTRAINT PK_SERIES_SEASON_IMAGES PRIMARY KEY (PROVIDER_ID, ID, IMAGE_ID, IMAGE_TYPE)
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_IMAGES') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_IMAGES', 'PROVIDER_ID', 'IDX_SERIES_SEASON_IMAGES') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_IMAGES
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            IMAGE_ID   varchar (100),
            IMAGE_TYPE    varchar (10),
            PROVIDER_ID numeric(19,0),
            CONSTRAINT PK_STG_SERIES_SEASON_IMAGES PRIMARY KEY (PROVIDER_ID, ID, IMAGE_ID, IMAGE_TYPE)
        )';
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_IMAGES') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_IMAGES', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_IMAGES') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.SERIES_SEASON_CAST
        (   
            ID      varchar (100), 
            C_ID   varchar (50),
            SEQ    numeric(10) DEFAULT 1 NOT NULL,
            ROLE   varchar (200),
            TYPE   varchar (10),
            FIRST_NAME   varchar (200),
            MIDDLE_NAME  varchar (200),
            LAST_NAME    varchar (500),
            PROVIDER_ID  numeric(19,0),
            CREATED_BY varchar(50) NOT NULL,
		  	UPDATED_BY varchar(50) NOT NULL,
		  	CREATE_DATE TIMESTAMP NOT NULL,
		  	UPDATE_DATE TIMESTAMP NOT NULL,
            CONSTRAINT U_SERIES_SEASON_CAST UNIQUE (PROVIDER_ID, ID, TYPE, SEQ, ROLE),
            CONSTRAINT FK_SERIES_SEASON_CAST FOREIGN KEY (PROVIDER_ID, ID, TYPE) REFERENCES '||VSCHEMA||'.SERIES_SEASON (PROVIDER_ID, ID, TYPE) ON DELETE CASCADE
        )';
        SELECT createObject(vsql,'create table SERIES_SEASON_CAST') INTO tmp_result;
        SELECT CreateIndex(vschema,'SERIES_SEASON_CAST', 'PROVIDER_ID', 'IDX_SERIES_SEASON_CAST') INTO tmp_result;
        
         vsql := 'CREATE TABLE '|| vschema||'.STG_SERIES_SEASON_CAST
        (   FILENAME varchar(4000),
            ID      varchar (100), 
            C_ID   varchar (50),
            SEQ    numeric(10) DEFAULT 1 NOT NULL,
            ROLE   varchar (200),
            TYPE   varchar (10),
            FIRST_NAME   varchar (200),
            MIDDLE_NAME  varchar (200),
            LAST_NAME    varchar (500),
            PROVIDER_ID  numeric(19,0),
            CONSTRAINT U_STG_SERIES_SEASON_CAST UNIQUE (PROVIDER_ID, ID, TYPE, SEQ, ROLE)
        )';
        
        SELECT createObject(vsql,'create table STG_SERIES_SEASON_CAST') INTO tmp_result;
        SELECT CreateIndex(vschema,'STG_SERIES_SEASON_CAST', 'PROVIDER_ID', 'IDX_STG_SERIES_SEASON_CAST') INTO tmp_result;
        
        SELECT set_version(vschema,'7.1.3') INTO tmp_result;
   end if;
   
   if vepgdbver < '7.1.4' then
       vsql := 'alter table '|| vschema||'.STG_SERIES_SEASON_CAST add column language varchar(50)';
       SELECT createObject(vsql,'add column language to STG_SERIES_SEASON_CAST','Y') INTO tmp_result;
       
       vsql := 'alter table '|| vschema||'.SERIES_SEASON_CAST add column language varchar(50)';
       SELECT createObject(vsql,'add column language to SERIES_SEASON_CAST','Y') INTO tmp_result;
       
       SELECT set_version(vschema,'7.1.4') INTO tmp_result;
   end if;
   if vepgdbver < '7.1.5' then
       vsql := 'alter table '|| vschema||'.PROGRAM add column season_id varchar(100)';
       SELECT createObject(vsql,'add column season_id to PROGRAM','Y') INTO tmp_result;
       
       vsql := 'alter table '|| vschema||'.PROGRAM add column series_id varchar(100)';
       SELECT createObject(vsql,'add column series_id to PROGRAM','Y') INTO tmp_result;
       
       vsql := 'alter table '|| vschema||'.STG_PROGRAM add column season_id varchar(100)';
       SELECT createObject(vsql,'add column season_id to STG_PROGRAM','Y') INTO tmp_result;
       vsql := 'alter table '|| vschema||'.STG_PROGRAM add column series_id varchar(100)';
       SELECT createObject(vsql,'add column series_id to STG_PROGRAM','Y') INTO tmp_result;
       
       SELECT set_version(vschema,'7.1.5') INTO tmp_result;
   end if;
   
   if vepgdbver < '7.2.0' then
       --Bo 5/5/2019 change the db server to 7.2 for epg 7.2 release.
       SELECT set_version(vschema,'7.2.0') INTO tmp_result;
   end if;
   
   EXECUTE 'INSERT INTO '|| vschema||'.install_table (install_message) values (''EPGManager upgrade to version '||epg_db_version||' completed SUCCESSFULLY.'')';
   return 0;
end;
$func_convertepgmanager$ LANGUAGE plpgsql;  
 

CREATE OR REPLACE FUNCTION install_epgmanager() RETURNS smallint AS $func_install_epgmanager$
DECLARE
    wk_count       numeric;
    vschema        varchar(100) := 'epgmanager';
    epg_db_version varchar := '7.2.0';
    func_results   smallint;
    prev_release   varchar:='';
BEGIN
    select count(*) into wk_count from pg_tables where lower(schemaname) = lower(vschema); 
-- 71 tables for the clean install for the first time
    IF wk_count = 0 THEN
       SELECT create_epgmanager(vschema, epg_db_version) INTO func_results;
    ELSE
       EXECUTE 'SELECT value from '|| vschema||'.config_setting where key=''epgmanagerversion'''  INTO prev_release; 
       IF prev_release < epg_db_version  THEN
          SELECT convertepgmanager(vschema, epg_db_version) INTO func_results;
       ELSIF prev_release = epg_db_version  THEN
          EXECUTE 'Insert into '||VSCHEMA||'.install_table(install_message) values (''There are no database changes in EPG Manager to upgrade'')';
       END IF;
    END IF;
    return func_results;
end;
$func_install_epgmanager$ LANGUAGE plpgsql;  
 
DO $$DECLARE
   results smallint;
BEGIN
   SELECT install_epgmanager() INTO results;
END$$;

