/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_WriteEventLog
        [ETL METADATA]
        *****************************************************************************
        Inserts a record in wfsdw.MD_EventLog. This table is the "last resort" for metadata
        errors -- used only when metadata information cannot be saved in the usual
        MD tables. In most cases, this will be related to coding errors.

        [RETURNS]       Nothing.

        *****************************************************************************
        [USED BY]       Only Metadata procedures.
        *****************************************************************************
        [By Laura Francheri 09/19/2006]
        *****************************************************************************
        Change Log:
            06/22/2014 - Suresh Neravati - Migrated the script to EDB.
            06/17/2017 - Suresh Neravati - Converted to PL/PgSQL.
            06/21/2017 - Suresh Neravati - Added atx PL/PgSQL function.
        *****************************************************************************
*/
--1
CREATE OR REPLACE FUNCTION spMD_WriteEventLog ( 
        process_name varchar, 
        error_code varchar, 
        error_message varchar DEFAULT '', 
        section varchar DEFAULT '' )
RETURNS smallint AS $func_spMD_WriteEventLog$
DECLARE
        v_process_name  varchar (100);
        v_error_message varchar (2000);
        v_section       varchar (500);
BEGIN
    -- Get the error message, if not provided
    IF error_message IS NULL OR error_message = '' THEN
        --select SUBSTR (SQLERRM, 1, 2000) into v_error_message;
        v_error_message := '';
    ELSE
        v_error_message := error_message;
    END IF;
    -- Process name:
    select COALESCE (process_name, '[UNKNOWN PROCESS]') into v_process_name;
    -- Insert!
    INSERT  INTO WFSDW.MD_EVENTLOG 
           (ProcessName, ErrorCode, ErrorMessage, Section, CreateDate)
    VALUES 
           (v_process_name,error_code,v_error_message,section,now()::timestamp);
    RETURN 0;
EXCEPTION
    WHEN OTHERS THEN
        raise notice '.....at exception of spMD_WriteEventLog_atx Error : % %', sqlstate, SQLERRM ;
        RETURN 1;
END;
$func_spMD_WriteEventLog$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_LogInfoMessage
        [ETL METADATA]
        *****************************************************************************
        Inserts an info message in the Metadata table wfsdw.MD_ETLProcessMessage for a
        certain RunInstance.

        [RETURNS]       Nothing. Will save an EventLog if an error is found.

        [NOTE]          At this point I'm not raising application errors as this code
                        will only be called from within Oracle. If needed, just uncomment
                        the raise_application_error calls.

        *****************************************************************************
        [USED BY]       Any ETL processes, for logging purposes.
        *****************************************************************************
        [SQL Server version by Laura Francheri 01/24/2002]
        [Oracle version by Venkat - revised by Laura on 09/20/2006]
        *****************************************************************************
        Change Log:     06/20/2017 - Suresh Neravati - Converted to PL/PgSQL.

        *****************************************************************************
*/
--2
CREATE OR REPLACE FUNCTION spMD_LogInfoMessage (
             etl_process        NUMERIC,
             run_instance       NUMERIC,
             message            VARCHAR)
RETURNS SMALLINT AS $func_spMD_LogInfoMessage$
DECLARE
        --PRAGMA AUTONOMOUS_TRANSACTION;
        new_message_numeric     NUMERIC;
        countRecords            NUMERIC;
        v_sqlcode               TEXT;
        tmp_results             NUMERIC;
BEGIN
        -- Check parameters
        SELECT count(*) INTO countRecords 
		  FROM wfsdw.MD_ETLProcessRunInstance 
		 WHERE ETLProcess = etl_process 
		   AND RunInstance = run_instance;

        IF ( etl_process IS NULL OR run_instance IS NULL OR message IS NULL OR countRecords = 0 )  THEN
                -- Process not found or NULL
                -- raise_application_error(-20001, 'spMD_LogInfoMessage ERROR: Null arguments or process/runinstance does not exist')
                SELECT spMD_WriteEventLog ('spMD_LogInfoMessage', '-1','','Null parameters or process/runinstance doesn''t exist!') into tmp_results;
                RETURN 1;
        END IF;

        -- Acquire a lock of this table so I can make sure nobody else does an insert until I do!
        BEGIN
                LOCK TABLE wfsdw.MD_ETLProcessMessage IN EXCLUSIVE MODE;
        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        --ROLLBACK;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogInfoMessage ERROR: could not acquire lock of wfsdw.MD_ETLPROCESSRUNINSTANCE table.');
                        SELECT spMD_WriteEventLog ('spMD_LogInfoMessage', v_sqlcode, '', 'Locking wfsdw.MD_ETLProcessMessage') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;
        -- Get the next message numeric
        SELECT  MAX(MessageNumber) + 1 INTO new_message_numeric
        FROM    wfsdw.MD_ETLProcessMessage
        WHERE   ETLProcess = etl_process AND RunInstance = run_instance;
        IF      new_message_numeric IS NULL THEN
                new_message_numeric := 1;
        END IF;
        -- Insert the message!
        INSERT INTO wfsdw.MD_ETLProcessMessage VALUES (etl_process, run_instance, new_message_numeric, 'INFO', message, now()::timestamp);
        RETURN 0;
EXCEPTION
        WHEN OTHERS THEN
                v_sqlcode := SQLSTATE;
                -- Not raising application errors at this point!
                -- raise_application_error(-20001, 'spMD_LogInfoMessage ERROR: error inserting wfsdw.MD_ETLProcessMessage');
                SELECT spMD_WriteEventLog ('spMD_LogInfoMessage', v_sqlcode, '', 'Inserting into wfsdw.MD_ETLProcessMessage') into tmp_results;
                --COMMIT;
                RETURN 1;
END;
$func_spMD_LogInfoMessage$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_LogErrorMessage
        [ETL METADATA]
        *****************************************************************************
        Inserts an Error Message in the Metadata table wfsdw.MD_ETLProcessMessage for a
        certain RunInstance.

        [RETURNS]       Nothing. Will save an EventLog if an error is found.

        [NOTE]          At this point I'm not raising application errors as this code
                        will only be called from within Oracle. If needed, just uncomment
                        the raise_application_error calls.

        *****************************************************************************
        [USED BY]       Any ETL processes, for logging purposes.
        *****************************************************************************
        [SQL Server version by Laura Francheri 01/24/2002]
        [Oracle version by Venkat - revised by Laura on 09/20/2006]
        *****************************************************************************
        Change Log:     06/21/2017 - Suresh Neravati - Converted to PL/PgSQL.

        *****************************************************************************
*/
--3
CREATE OR REPLACE FUNCTION spMD_LogErrorMessage (
	etl_process                     NUMERIC,
	run_instance                    NUMERIC,
	message                         VARCHAR)
RETURNS smallint AS $func_spMD_LogErrorMessage$
DECLARE
        --PRAGMA AUTONOMOUS_TRANSACTION;
        new_message_numeric             NUMERIC;
        countRecords                    NUMERIC;
        v_sqlcode                       TEXT;
        tmp_results                     NUMERIC;
BEGIN

-- Check parameters
        SELECT count(*)
          INTO countRecords
          FROM wfsdw.MD_ETLProcessRunInstance
         WHERE ETLProcess = etl_process
           AND RunInstance = run_instance;

        IF ( etl_process IS NULL OR run_instance IS NULL OR message IS NULL OR countRecords = 0 ) THEN
                -- Process not found or NULL
                -- raise_application_error(-20001, 'spMD_LogErrorMessage ERROR: Null arguments or process/runinstance does not exist')
                SELECT spMD_WriteEventLog ('spMD_LogErrorMessage', '-1','', 'Null parameters or process/runinstance doesn''t exist!') into tmp_results;
                --COMMIT;
            RETURN 1;
        END IF;

        -- Acquire a lock of this table so I can make sure nobody else does an insert until I do!
        BEGIN
                LOCK TABLE wfsdw.MD_ETLProcessMessage IN EXCLUSIVE MODE;
                NULL;
        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        --ROLLBACK;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogErrorMessage ERROR: could not acquire lock of wfsdw.MD_ETLPROCESSRUNINSTANCE table.');
                        SELECT spMD_WriteEventLog ('spMD_LogErrorMessage', v_sqlcode, '', 'Locking wfsdw.MD_ETLProcessMessage') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;

        -- Get the next message numeric
        SELECT  MAX(MessageNumber) + 1
        INTO    new_message_numeric
        FROM    wfsdw.MD_ETLProcessMessage
        WHERE   ETLProcess = etl_process
        AND     RunInstance = run_instance;

        IF new_message_numeric IS NULL THEN
                new_message_numeric := 1;
        END IF;

        -- Insert the message!
        INSERT INTO wfsdw.MD_ETLProcessMessage VALUES (etl_process, run_instance, new_message_numeric, 'ERROR', message, now()::timestamp);
        --COMMIT ;
        RETURN 0;
EXCEPTION
        WHEN OTHERS THEN
                v_sqlcode := SQLSTATE;
                --ROLLBACK;
                -- Not raising application errors at this point!
                -- raise_application_error(-20001, 'spMD_LogErrorMessage ERROR: error inserting wfsdw.MD_ETLProcessMessage');
                SELECT spMD_WriteEventLog ('spMD_LogErrorMessage', v_sqlcode, '', 'Inserting into wfsdw.MD_ETLProcessMessage') into tmp_results;
                --COMMIT;
                RETURN 1;
END;
$func_spMD_LogErrorMessage$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_LogProcessFinishOK
        [ETL METADATA]
        *****************************************************************************
        Updates an entry in the Metadata table wfsdw.MD_ETLProcessRunInstance, indicating
        that the current RunInstance for this process has finished OK.

        [RETURNS]       ERROR: Returns -1.

        [NOTE]          At this point I'm not raising application errors as this code
                                will only be called from within Oracle. If needed, just uncomment
                                the raise_application_error calls.
                                The email code is commented out as we currently don't have email
                                support. If uncommented, it would need to be tested as well.

        *****************************************************************************
        [USED BY]       Should be used by every ETL process upon succesful completion.
        *****************************************************************************
        [SQL Server version by Laura Francheri 01/24/2002]
        [Oracle version by Venkat - revised by Laura on 09/15/2006]
        *****************************************************************************
        Change Log:     06/21/2017 - Suresh Neravati - Converted to PL/PgSQL.

        *****************************************************************************
*/
--4
CREATE OR REPLACE FUNCTION spMD_LogProcessFinishOK (
        etl_process                          NUMERIC,
        run_instance                         NUMERIC,
        user_comments                        VARCHAR,
        affected_rows                        NUMERIC     DEFAULT NULL)
RETURNS smallint AS $func_spMD_LogProcessFinishOK$
DECLARE
        --PRAGMA AUTONOMOUS_TRANSACTION;
        process_name                         VARCHAR (50);
        object_loaded                        NUMERIC;
        object_name                          VARCHAR (50);
        comments_var                         VARCHAR (500);
        v_sqlcode                            TEXT;
        sendemail_onsuccess                  NUMERIC;
        email_recipients                     VARCHAR(255);
        email_subject                        VARCHAR(255);
        msg                                  VARCHAR(5000);
        control_etlprocess                   NUMERIC;
        start_date                           TIMESTAMP;
        email_ret_val                        NUMERIC;
        tmp_results                          NUMERIC;
        error_status                         NUMERIC;
BEGIN
        -- Initialize...
        error_status := -1;

        BEGIN
                -- Process name and object being loaded
                SELECT  Name, LockableObjectLoaded, SendEmailOnSuccess, EmailRecipients
                INTO    process_name, object_loaded, sendemail_onsuccess, email_recipients
                FROM    wfsdw.MD_ETLProcess
                WHERE   ETLProcessKey = etl_process;

                -- StartDate and ETLProcess for control
                SELECT  etlprocess, startdate
                INTO    control_etlprocess, start_date
                FROM    wfsdw.MD_ETLProcessRunInstance
                WHERE   ETLProcess = etl_process
                AND     RunInstance = run_instance;

        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        raise notice '...3e..% %', SQLSTATE, SQLERRM;
                        SELECT spMD_WriteEventLog ('spMD_LogProcessFinishOK', v_sqlcode,'', '(OTHERS) Null parameters or process/runinstance doesn''t exist!') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;

        -- Check for errors
        IF ( etl_process IS NULL OR run_instance IS NULL OR process_name IS NULL OR control_etlprocess IS NULL ) THEN
                -- Process not found or NULL
                -- raise_application_error(-20001, 'spMD_LogProcessFinishError ERROR: Null arguments or process/runinstance does not exist')
                SELECT spMD_WriteEventLog ('spMD_LogProcessFinishOK', '-1','', 'Null parameters or process/runinstance doesn''t exist!') into tmp_results;
                --COMMIT;
            RETURN 1;
        END IF;

        IF user_comments IS NULL OR user_comments = '' THEN
                IF object_loaded IS NULL THEN
                        comments_var := process_name || ' finished OK';
                ELSE
                        -- Retrieve object name
                        SELECT  NAME
                        INTO    object_name
                        FROM    wfsdw.MD_LockableObject
                        WHERE   LockableObjectKey = object_loaded;

                        comments_var := process_name || ' (loads: ' || object_name || ') finished OK';
                END IF;
        ELSE
                comments_var := user_comments;
        END IF;
        -- Update run instance!
        BEGIN
                UPDATE  wfsdw.MD_ETLProcessRunInstance
                SET     status = 0, -- Finished OK
                        comments = comments_var,
                        affectedrows = affected_rows,
                        enddate = now()::timestamp
                WHERE   etlprocess = etl_process
                AND     runinstance = run_instance;
        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        --ROLLBACK;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogProcessFinishError ERROR: error updating wfsdw.MD_ETLProcessRunInstance');
                        raise notice '.....at exception of spMD_LogErrorMessage.....Error : % %', sqlstate, sqlerrm ;
                        SELECT spMD_WriteEventLog ('spMD_LogProcessFinishOK', v_sqlcode, '', 'Updating process status in wfsdw.MD_ETLProcessRunInstance') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;
        --COMMIT;
        error_status := 0;
        RETURN error_status;
END;
$func_spMD_LogProcessFinishOK$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_LogProcessFinishError
        [ETL METADATA]
        *****************************************************************************
        Updates an entry in the Metadata table wfsdw.MD_ETLProcessRunInstance, indicating
        that the current RunInstance for this process has finished with an Error.

        [RETURNS]       ERROR: Returns -1.

        [NOTE]          At this point I'm not raising application errors as this code
                                will only be called from within Oracle. If needed, just uncomment
                                the raise_application_error calls.
                                The email code is commented out as we currently don't have email
                                support. If uncommented, it would need to be tested as well.

        *****************************************************************************
        [USED BY]       Should be used by every ETL process if there's an error.
        *****************************************************************************
        [SQL Server version by Laura Francheri 01/24/2002]
        [Oracle version by Venkat - revised by Laura on 09/15/2006]
        *****************************************************************************
        Change Log:     06/22/2017 - Suresh Neravati - Converted to PL/PgSQL.

        *****************************************************************************
*/
--5
CREATE OR REPLACE FUNCTION spMD_LogProcessFinishError (
        etl_process                        NUMERIC,
        run_instance                       NUMERIC,
        user_comments                      VARCHAR,
        affected_rows                      NUMERIC     DEFAULT NULL,
        error                              TEXT        DEFAULT NULL)
RETURNS smallint AS $func_spMD_LogProcessFinishError$
DECLARE
        --PRAGMA AUTONOMOUS_TRANSACTION;
        process_name                       VARCHAR (50);
        object_loaded                      NUMERIC;
        object_name                        VARCHAR (50);
        comments_var                       VARCHAR (500);
        v_sqlcode                          TEXT;
        sendemail_onerror                   NUMERIC;
        email_recipients                   VARCHAR (255);
        email_subject                      VARCHAR (255);
        msg                                VARCHAR (5000);
        control_etlprocess                 NUMERIC;
        start_date                         TIMESTAMP;
        email_ret_val                      NUMERIC;
        tmp_results                        NUMERIC;
        error_status                       NUMERIC;
BEGIN
        -- Initialize...
        error_status := -1;
        BEGIN
                -- Process name and object being loaded
                SELECT  Name, LockableObjectLoaded, SendEmailOnError, EmailRecipients
                INTO    process_name, object_loaded, sendemail_onerror, email_recipients
                FROM    wfsdw.MD_ETLProcess
                WHERE   ETLProcessKey = etl_process;

                -- StartDate and ETLProcess for control
                SELECT  etlprocess, startdate
                INTO    control_etlprocess, start_date
                FROM    wfsdw.MD_ETLProcessRunInstance
                WHERE   ETLProcess = etl_process
                AND     RunInstance = run_instance;

        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        raise notice '.....at first exception of spMD_LogProcessFinishError.....Error : % %', sqlstate, sqlerrm ;
                        SELECT spMD_WriteEventLog ('spMD_LogProcessFinishError',v_sqlcode,'','(OTHERS) Null parameters or process/runinstance doesn''t exist!') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;

        -- Check for errors
        IF ( etl_process IS NULL OR run_instance IS NULL OR process_name IS NULL OR control_etlprocess IS NULL ) THEN
                -- Process not found or NULL
                -- raise_application_error(-20001, 'spMD_LogProcessFinishError ERROR: Null arguments or process/runinstance does not exist')
                SELECT spMD_WriteEventLog ('spMD_LogProcessFinishError', '-1','', 'Null parameters or process/runinstance doesn''t exist!') into tmp_results;
                --COMMIT;
                RETURN 1;
        END IF;

        IF user_comments IS NULL OR user_comments = '' THEN
                IF object_loaded IS NULL THEN
                        comments_var := process_name || ' FAILED!';
                ELSE
                        -- Retrieve object name
                        SELECT       NAME
                        INTO         object_name
                        FROM         wfsdw.MD_LockableObject
                        WHERE        LockableObjectKey = object_loaded;

                       comments_var := process_name || ' (loads: ' || object_name || ') FAILED!';
                END IF;
        ELSE
                comments_var := user_comments;
        END IF;

        -- Update run instance!
        BEGIN
                UPDATE  wfsdw.MD_ETLProcessRunInstance
                SET     status = 2, -- Error
                        comments = comments_var,
                        affectedrows = affected_rows,
                        enddate = now()::timestamp,
                        syserrorcode = error
                WHERE   etlprocess = etl_process
                AND     runinstance = run_instance;

        EXCEPTION
        WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        --ROLLBACK;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogProcessFinishError ERROR: error updating wfsdw.MD_ETLProcessRunInstance');
                        raise notice '.....at exception of spMD_LogProcessFinishError.....Error : % %', sqlstate, sqlerrm ;
                        SELECT spMD_WriteEventLog ('spMD_LogProcessFinishError', v_sqlcode, '', 'Updating process status in wfsdw.MD_ETLProcessRunInstance') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;

        --COMMIT;
        error_status := 0;
        RETURN error_status;
END;
$func_spMD_LogProcessFinishError$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_ReleaseObject
        [ETL METADATA]
        *****************************************************************************
        Releases a LockableObject.
        This procedure should be called after processing (succesfully or not) any
        LockableObject that was previously locked using spMD_LockObject.
        Only the process that acquired the lock will be able to unlock it.

        [RETURNS]       ERROR: Returns -1.

        [NOTE]          At this point I'm not raising application errors as this code
                                will only be called from within Oracle. If needed, just uncomment
                                the raise_application_error calls.

        *****************************************************************************
        [USED BY]       Should be used by every ETL process that performs an
                                incremental load of a standard fact table or a dimension
                                table.
        *****************************************************************************
        [SQL Server version by Laura Francheri 08/08/2002]
        [Oracle version 09/20/2006]
        *****************************************************************************
        Change Log:     06/22/2017 - Suresh Neravati - Converted to PL/PgSQL.

        *****************************************************************************
*/
--6
CREATE OR REPLACE FUNCTION spMD_ReleaseObject (
        userobject           NUMERIC,
        etl_process          NUMERIC,
        run_instance         NUMERIC,  -- Added underscore
        last_syncdate        TIMESTAMP,
        has_error            NUMERIC, -- 0=OK, 1=ERROR
        nonverbose           NUMERIC DEFAULT 0)
RETURNS smallint AS $func_spMD_ReleaseObject$
DECLARE
        --PRAGMA AUTONOMOUS_TRANSACTION;
        error_var               NUMERIC;
        rowcount_var            NUMERIC;
        error_message           VARCHAR (500);
        objectkey_control       NUMERIC;
        object_name             VARCHAR (50);
        new_load_status         NUMERIC;
        new_sync_date           TIMESTAMP;
        old_sync_date           TIMESTAMP;
        etlprocess_object       NUMERIC;
        message                 VARCHAR (500);
        last_etlprocess         NUMERIC;
        last_runinstance        NUMERIC;
        countRecords            NUMERIC;
        v_sqlcode               TEXT;
        tmp_results             NUMERIC;
        error_status            NUMERIC;
BEGIN

        -- Initialize...
        error_status := -1;
        error_message := 'spMD_ReleaseObject ERROR: ';

        -- Check for parameter errors
        IF ( userobject IS NULL OR etl_process IS NULL OR run_instance IS NULL OR has_error IS NULL OR (has_error = 0 AND last_syncdate IS NULL) OR has_error NOT IN (1, 0)) THEN
                -- Invalid arguments!
                error_message := error_message || 'Null or invalid arguments';
                -- raise_application_error (-20001, error_message);
                SELECT spMD_WriteEventLog ('spMD_ReleaseObject', '-1','', 'Null or invalid arguments!') into tmp_results;
                --COMMIT;
                RETURN 1;
        END IF;

        SELECT  COUNT(*)
        INTO    countRecords
        FROM    wfsdw.MD_ETLProcessRunInstance
        WHERE   ETLProcess      = etl_process
        AND     RunInstance     = run_instance;

        -- Check that the process/runinstance exist in wfsdw.MD_ETLProcessRunInstance
        IF countRecords = 0  THEN
                -- Invalid Process/RunInstance
                error_message := error_message || 'Invalid Process/RunInstance' ;
                --raise_application_error (-20001, error_message);
                SELECT spMD_WriteEventLog ('spMD_ReleaseObject', '-1','', 'Invalid Process/RunInstance!') into tmp_results;
                --COMMIT;
                --RETURN 1;  -- Comeback here
        END IF;

        -- Look up object name and old sync date
        SELECT  LockableObjectKey,  Name, LastSyncDate,   LastETLProcess, LastRunInstance
        INTO    objectkey_control, object_name ,old_sync_date,  last_etlprocess,last_runinstance
        FROM    wfsdw.MD_LockableObject
        WHERE   LockableObjectKey = userobject;

        -- Check that the Object exists!
        IF objectkey_control IS NULL THEN
                error_message := error_message || 'Object ' || TO_CHAR (userobject,'9999999999') || ' does not exist';
                SELECT spMD_LogErrorMessage (etl_process, run_instance, error_message) into tmp_results;
                --raise_application_error (-20001, error_message);
                SELECT spMD_WriteEventLog ('spMD_ReleaseObject', '-1', '', 'Object ' || TO_CHAR (userobject,'9999999999') || ' does not exist') into tmp_results;
                --COMMIT;
                RETURN 1;
        END IF;

        -- Check if it's the same Object specified in wfsdw.MD_ETLProcess. The process won't fail if they're different, but it
        -- will issue a warning.
        SELECT  LockableObjectLoaded INTO etlprocess_object
        FROM    wfsdw.MD_ETLProcess
        WHERE   ETLProcessKey = etl_process;

        IF COALESCE(nonverbose, 0) <> 1 THEN
                IF etlProcess_object IS NULL THEN
                        SELECT spMD_LogInfoMessage(etl_process,run_instance,'Warning: wfsdw.MD_ETLProcess.LockableObjectLoaded is NULL. Please update this field for reporting purposes.') into tmp_results;
                ELSIF etlProcess_object <> userobject THEN
                        SELECT spMD_LogInfoMessage(etl_process,run_instance,'Warning: wfsdw.MD_ETLProcess.LockableObjectLoaded is different from object parameter. Please update this field for reporting purposes.') 
                        INTO tmp_results;
                END IF;
        END IF;

        -- Check if this process is the same one that acquired the lock...
        IF last_etlprocess <> etl_process OR last_runinstance <> run_instance THEN
                raise notice '.....at exception of spMD_ReleaseObject.....etl_process, run_instance are null : % %', etl_process, run_instance ;
                error_message := error_message || 'Object [' || object_name || ']' || ' was locked by Proc/Inst (' || TO_CHAR(last_etlprocess,'9999999999' ) ||'/' || TO_CHAR(last_runinstance,'9999999999') ||'). Release not authorized.';
                SELECT spMD_LogErrorMessage (etl_process, run_instance, error_message) into tmp_results;
                --raise_application_error(-20001, error_message);
                SELECT spMD_WriteEventLog ('spMD_ReleaseObject', '-1','',  error_message) into tmp_results;
                --COMMIT;
                RETURN 1;
        END IF;

        IF has_error = 0 THEN
                new_load_status := 0;   -- OK!
                new_sync_date := last_syncdate;
        ELSE
                new_load_status := 2;   -- FAILED!
                new_sync_date := old_sync_date;--Preserve it!
        END IF;

        -- OK, release object!
        -----------------------------------------------------------------------------
        BEGIN

                UPDATE  wfsdw.MD_LockableObject
                SET     LastObjectStatus    = new_load_status,
                        LastSyncDate        = new_sync_date,
                        LastETLProcess      = etl_process,
                        LastRunInstance     = run_instance,
                        UpdateDate          = now()::timestamp
                WHERE   LockableObjectKey   = userobject;

        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        raise notice '.....at exception of 1) spMD_ReleaseObject.....Error : % %', sqlstate, sqlerrm ;
                        --ROLLBACK;
                        error_message := error_message || ' Error updating wfsdw.MD_LockableObject while releasing Object [' || object_name || ']. Object could not be released.';
                        SELECT spMD_LogErrorMessage (etl_process,  run_instance, error_message) into tmp_results;
                        SELECT spMD_WriteEventLog ('spMD_ReleaseObject', v_sqlcode, '', 'Releasing object') into tmp_results;
                        --COMMIT;
                        RETURN 1;
        END;

        -- Done!
        --COMMIT;
        -- Update the process/runinstance to indicate the new status.
        UPDATE  wfsdw.MD_ETLProcessRunInstance
        SET     LockableObjectLoaded        = userobject,
                LockableObjectStatus        = new_load_status,
                NewSyncDate                 = new_sync_date
        WHERE   ETLProcess                  = etl_process
        AND     RunInstance                 = run_instance;

        -- Everything OK...
        IF COALESCE (nonverbose, 0) <> 1 THEN
                message := 'spMD_ReleaseObject: Object [' || object_name || '] has been released.';
                SELECT spMD_LogInfoMessage(etl_process,run_instance,message) into tmp_results;
        END IF;
        -- Done!
        --COMMIT;
        -- Return last synchronization date...
        error_status := 0;
        RETURN error_status;

EXCEPTION
        WHEN OTHERS THEN
                --ROLLBACK;
                raise notice '.....at exception of spMD_ReleaseObject.....Error : % %', sqlstate, sqlerrm ;
                --error_message := 'sqlcode : '|| sqlcode || '--' || error_message || 'Error updating wfsdw.MD_ETLProcessRunInstance AFTER releasing Object [' || object_name || ']. Please check this table.';
                error_message :=  error_message || 'Error updating wfsdw.MD_ETLProcessRunInstance AFTER releasing Object [' || object_name || ']. Please check this table.';
                SELECT spMD_LogErrorMessage(etl_process, run_instance, error_message) into tmp_results;
                --COMMIT;
                RETURN 1;
END;
$func_spMD_ReleaseObject$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_LockObject
        [ETL METADATA]
        *****************************************************************************
        Attempts to "lock" a LockableObject (hmmm, DUH)
        This procedure should be called prior to loading any LockableObject (standard
        fact tables and dimension tables), so that any other procedures that attempt
        to do the same thing can know there's somebody else updating that table.

        [RETURNS]       ERROR: Returns -1.

        [NOTE]          At this point I'm not raising application errors as this code
                                will only be called from within Oracle. If needed, just uncomment
                                the raise_application_error calls.

        *****************************************************************************
        [USED BY]       Should be used by every ETL process that performs an
                                incremental load of a standard fact table or a dimension
                                table.
        *****************************************************************************
        [SQL Server version by Laura Francheri 01/24/2002]
        [Oracle version by Venkat - revised by Laura on 09/10/2006]
        *****************************************************************************
        Change Log:

        *****************************************************************************
*/
--7
CREATE OR REPLACE FUNCTION spMD_LockObject (
        userobject              NUMERIC,
        etl_process             NUMERIC,
        run_instance            NUMERIC,
        force_lock              NUMERIC  DEFAULT 0, -- By default, do not force lock.
        --last_sync_date_output   OUT TIMESTAMP,
        nonverbose              NUMERIC DEFAULT 0)
RETURNS table ( error_status numeric, last_sync_date_output TIMESTAMP) AS $func_spMD_LockObject$
DECLARE
        --PRAGMA AUTONOMOUS_TRANSACTION;
        error_message           VARCHAR (500);
        objectkey_control       NUMERIC;
        object_name             VARCHAR (50);
        last_load_status        NUMERIC;
        etlprocess_object       NUMERIC;
        message                 VARCHAR (500);
        last_etl_process        NUMERIC;
        last_run_instance       NUMERIC;
        countRecords            NUMERIC;
        last_sync_date          TIMESTAMP;
        v_sqlcode               TEXT;
        tmp_results             NUMERIC;
        error_status            NUMERIC;
        last_sync_date_output   TIMESTAMP;
        ret                     record;
BEGIN
        -- Initialize...
        error_message := 'spMD_LockObject ERROR: ';
        last_sync_date_output := NULL;
        error_status := -1;
        -- Check for parameter errors
        IF userobject IS NULL OR etl_process IS NULL OR run_instance IS NULL THEN
               -- Process not found or NULL
                error_message := error_message || 'Null arguments';
                -- raise_application_error (-20001, error_message);
                SELECT spMD_WriteEventLog ('spMD_LockObject', '-1','', 'Null arguments!') into tmp_results;
                --COMMIT;
                RETURN query SELECT error_status, last_sync_date_output;
        END IF;

        SELECT  COUNT(*)
        INTO    countRecords
        FROM    wfsdw.MD_ETLProcessRunInstance
        WHERE   ETLProcess  = etl_process
        AND     RunInstance  = run_instance;

        -- Check that the process/runinstance exist in wfsdw.MD_ETLProcessRunInstance
        IF countRecords = 0  THEN
                -- Invalid Process/RunInstance
                error_message := error_message || 'Invalid Process/RunInstance' ;
                raise notice 'Invalid Process/RunInstance at spMD_LockObject';
                --raise_application_error (-20001, error_message);
                SELECT spMD_WriteEventLog ('spMD_LockObject', '-1','', 'Invalid Process/RunInstance!') into tmp_results;
                --COMMIT;
                RETURN query SELECT error_status, last_sync_date_output;
        END IF;

        -- Lock wfsdw.MD_LockableObject for update!
        BEGIN
                -- Look up last sync date and status
                SELECT  LockableObjectKey,
                        Name,
                        LastObjectStatus,
                        LASTSYNCDATE,
                        LastETLPRocess,
                        LastRunInstance
                INTO    objectkey_control,
                        object_name,
                        last_load_status,
                        last_sync_date,
                        last_etl_process,
                        last_run_instance
                FROM wfsdw.MD_LockableObject
                WHERE LockableObjectKey = userobject
                FOR UPDATE;

        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        --ROLLBACK;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogProcessStart ERROR: could not acquire lock of wfsdw.MD_ETLPROCESSRUNINSTANCE table.');
                        SELECT spMD_WriteEventLog ('spMD_LockObject', v_sqlcode, '','SELECT from wfsdw.MD_LockableObject FOR UPDATE') into tmp_results;
                        error_message := error_message || 'DB error when doing SELECT from wfsdw.MD_LockableObject FOR UPDATE';
                        SELECT spMD_LogErrorMessage (etl_process,  run_instance, error_message) into tmp_results;
                        --COMMIT;
                        RETURN query SELECT error_status, last_sync_date_output;
        END;

        -- Check if it's the same Object specified in wfsdw.MD_ETLProcess. The process won't fail if they're different, but it
        -- will issue a warning.
        SELECT  LockableObjectLoaded INTO etlprocess_object
        FROM    wfsdw.MD_ETLProcess
        WHERE   ETLProcessKey = etl_process;

        IF COALESCE (nonverbose, 0) <> 1 THEN
                IF etlProcess_object IS NULL THEN
                        SELECT spMD_LogInfoMessage(etl_process, run_instance, 'Warning: wfsdw.MD_ETLProcess.LockableObjectLoaded is NULL. Please update this field for reporting purposes.') into tmp_results;
                ELSIF etlProcess_object <> userobject THEN
                        SELECT spMD_LogInfoMessage(etl_process, run_instance, 'Warning: wfsdw.MD_ETLProcess.LockableObjectLoaded is different from object parameter. Please update this field for reporting purposes.') 
                        INTO tmp_results;
                END IF;
        END IF;

        -- Check the status of the Object to determine if it can be locked.
        IF last_load_status = 1 THEN -- Loading
                --ROLLBACK;
                error_message := error_message || 'Attempt to lock Object [' || object_name || '] FAILED: object is currently locked by process ('
                || TO_CHAR(last_etl_process, '9999999999') || '/'
                || TO_CHAR (last_run_instance,'9999999999') || ')';
                SELECT spMD_LogErrorMessage(etl_process, run_instance, error_message) into tmp_results;
                --COMMIT;
                RETURN query SELECT error_status, last_sync_date_output;
        END IF;

        IF last_load_status = 2 AND force_lock = 0  THEN  -- Failed and not set to force the lock
                --ROLLBACK;
                error_message := error_message || 'Attempt to lock Object [' || object_name ||
                         '] FAILED: object''s last status is FAILED. Use spMD_ReleaseObject or call procedure with @force_lock = 1.';
                SELECT spMD_LogErrorMessage(etl_process,  run_instance, error_message) into tmp_results;
                --COMMIT;
                RETURN query SELECT error_status, last_sync_date_output;
        END IF;

        -- Force lock?
        IF COALESCE (nonverbose, 0) <> 1 THEN
                IF last_load_status = 2 AND force_lock = 1  THEN -- Failed. Lock will be forced.
                        message := 'spMD_LockObject: last load for Object [' || object_name || '] failed. Attempting to force lock...';
                        SELECT spMD_LogInfoMessage(etl_process,  run_instance, message) into tmp_results;
                END IF;
        END IF;

        -- Everything OK! Lock object!
        -----------------------------------------------------------------------------
        BEGIN
                UPDATE  wfsdw.MD_LockableObject
                SET     LastObjectStatus        =  1,   -- Loading
                        LastETLProcess          =  etl_process,
                        LastRunInstance         =  run_instance,
                        UpdateDate              =  now()::TIMESTAMP
                WHERE   LockableObjectKey       =  userobject;

        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        --ROLLBACK;
                        error_message := error_message || 'updating wfsdw.MD_LockableObject while locking Object ['|| object_name ||']. Object could not be locked.';
                        SELECT spMD_LogErrorMessage (etl_process,  run_instance, error_message) into tmp_results;
                        SELECT spMD_WriteEventLog ('spMD_LockObject', v_sqlcode, '', 'Locking object') into tmp_results;
                        --COMMIT;
                        RETURN query SELECT error_status, last_sync_date_output;
        END;

        -- Done!
        --COMMIT;

        --raise notice 'At spMD_LockObject.....10';
        -- Update the process/runinstance to indicate the new status.
        UPDATE  wfsdw.MD_ETLProcessRunInstance
        SET     LockableObjectLoaded =  userobject,
                LockableObjectStatus =  1,      -- Running
                PreviousSyncDate =  last_sync_date
        WHERE   ETLProcess  = etl_process
        AND     RunInstance = run_instance;

        -- Everything OK...
        IF COALESCE (nonverbose, 0) <> 1 THEN
                --raise notice 'At spMD_LockObject.....11a';
                message := 'spMD_LockObject: Object [' || object_name || '] has been locked.';
                SELECT spMD_LogInfoMessage(etl_process, run_instance, message) into tmp_results;
                --raise notice 'At spMD_LockObject.....11b';
        END IF;

        -- Done!
        --COMMIT;

        -- Return last synchronization date...
        last_sync_date_output := last_sync_date;
        error_status := 0;
        --ret := ( error_status, last_sync_date_output );
        --raise notice 'At spMD_LockObject.....13';
        RETURN query SELECT error_status, last_sync_date_output;

EXCEPTION
        WHEN OTHERS THEN
                raise notice 'At spMD_LockObject.....exception : %',sqlerrm;
                --ROLLBACK;
                error_message := error_message||'Error updating wfsdw.MD_ETLProcessRunInstance AFTER locking Object [' || object_name || ']. Please check this table.';
                SELECT spMD_LogErrorMessage(etl_process, run_instance, error_message) into tmp_results;
                --COMMIT;
                RETURN query SELECT error_status, last_sync_date_output;

END;
$func_spMD_LockObject$ LANGUAGE plpgsql;

/*
        *****************************************************************************
        CREATE OR REPLACE FUNCTION: spMD_LogProcessStart
        [ETL METADATA]
        *****************************************************************************
        Inserts an entry in the Metadata table wfsdw.MD_ETLProcessRunInstance, indicating
        that a process has started.

        [RETURNS]       A RunInstance (sequential numeric), which will be used to
                                log messages and any other references to this RunInstance.
                                ERROR: error_status will be -1 and new_run_instance will be -1
                                as well.

        [NOTE]          At this point I'm not raising application errors as this code
                                will only be called from within Oracle. If needed, just uncomment
                                the raise_application_error calls.

        *****************************************************************************
        [USED BY]       Should be used by every ETL process right at the beginning.
        *****************************************************************************
        [SQL Server version by Laura Francheri 01/24/2002]
        [Oracle version by Venkat - revised by Laura on 09/10/2006]
        *****************************************************************************
        Change Log: 1) Added the Proceducre spMD_ProcessReport on 10/02/2006 by
                   Suresh Neravati to print the Process Run Instance information.
        *****************************************************************************
*/
-- 8
CREATE OR REPLACE FUNCTION spMD_LogProcessStart (
	etl_process                         NUMERIC,
	parent_etl_process                  NUMERIC  DEFAULT NULL,
	parent_etl_process_runinstance      NUMERIC  DEFAULT NULL,
	comments                            VARCHAR  DEFAULT NULL)
RETURNS table (error_status numeric, new_run_instance numeric) AS $func_spMD_LogProcessStart$
DECLARE
	--PRAGMA AUTONOMOUS_TRANSACTION;
	process_name               VARCHAR (50);
	lockableobject             NUMERIC;
	lockableobject_name        VARCHAR (50);
	urror_message              VARCHAR (500);
	count_records              NUMERIC := 0;
	SPID                       NUMERIC;
	comments_var               VARCHAR (500) := comments;
	v_sqlcode                  TEXT;
	tmp_results                NUMERIC;
	error_status               NUMERIC;
	new_run_instance           NUMERIC;
	ret                        record;
BEGIN
        -- Initialize...
        error_status := -1;
        new_run_instance := -1;
        ret := (error_status, new_run_instance);
        BEGIN
                -- Process name and object being loaded
                SELECT  Name, LockableObjectLoaded
                INTO    process_name, lockableobject
                FROM    wfsdw.MD_ETLProcess
                WHERE   ETLProcessKey = etl_process;

        EXCEPTION
                WHEN OTHERS THEN
                        v_sqlcode := SQLSTATE;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogProcessStart ERROR: error inserting wfsdw.MD_ETLProcessMessage');
                        raise notice 'spMD_LogProcessStart : Null arguments or ETL process does not exist!';
                        SELECT spMD_WriteEventLog ('spMD_LogProcessStart', v_sqlcode,'', 'Null arguments or ETL process does not exist!') into tmp_results;
                        --COMMIT;
                        RETURN query SELECT error_status, new_run_instance;
        END;

        -- Check for errors
        IF etl_process IS NULL OR process_name IS NULL THEN
                -- Process not found or NULL
                -- Not raising application errors at this point!
                -- raise_application_error(-20001, 'spMD_LogProcessStart ERROR: Null arguments or ETL process does not exist');
                SELECT spMD_WriteEventLog ('spMD_LogProcessStart', '-1','', 'Null arguments or ETL process does not exist!') into tmp_results;
                --COMMIT;
        RETURN query SELECT error_status, new_run_instance;
        END IF;

        SELECT  count(*)
        INTO    count_records
        FROM    wfsdw.MD_ETLProcessRunInstance
        WHERE   ETLProcess = parent_etl_process
        AND     RunInstance = parent_etl_process_runinstance;

        -- If a parent process is supplied, then a parentruninstance must be supplied
        -- and they both must exist already in the wfsdw.MD_ETLProcessRunInstance table.
        IF parent_etl_process IS NOT NULL THEN
                IF parent_etl_process_runinstance IS NULL OR count_records = 0 THEN
                        -- Parent process not found
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogProcessStart ERROR: Null parentprocess runinstance or parentprocess/runinstance does not exist!');
                        SELECT spMD_WriteEventLog ('spMD_LogProcessStart','-1','', 'Null parentprocess runinstance or parentprocess/runinstance does not exist!') into tmp_results;
                        --COMMIT;
                        RETURN query SELECT error_status, new_run_instance;
                END IF;
        END IF;

        -- Set up message if not provided...
        IF comments_var IS NULL OR trim(comments_var)= '' THEN
                IF lockableobject IS NULL THEN
                        comments_var := process_name || ' started...';
                ELSE
                        SELECT  process_name || ' (loads: ' || Name || ') started...' INTO comments_var
                        FROM    wfsdw.MD_LockableObject
                        WHERE   LockableObjectKey = lockableobject;
                END IF;
        END IF;

        -- Acquire a lock of this table so I can make sure nobody else does an insert until I do!
        BEGIN
                LOCK TABLE wfsdw.MD_ETLPROCESSRUNINSTANCE IN EXCLUSIVE MODE;
     
        EXCEPTION
                WHEN OTHERS THEN

                        --ROLLBACK;
                        v_sqlcode := SQLSTATE;
                        -- Not raising application errors at this point!
                        -- raise_application_error(-20001, 'spMD_LogProcessStart ERROR: could not acquire lock of wfsdw.MD_ETLPROCESSRUNINSTANCE table.');
                        SELECT spMD_WriteEventLog ('spMD_LogProcessStart', v_sqlcode, '', 'Locking wfsdw.MD_ETLProcessRunInstance') into tmp_results;
                        --COMMIT;
                        RETURN query SELECT error_status, new_run_instance;
        END;

        -- Look for the next run instance and lock tableMD_ETLProcessRunInstance
        SELECT  MAX(RunInstance) + 1  INTO new_run_instance
        FROM    wfsdw.MD_ETLProcessRunInstance
        WHERE   ETLProcess = etl_process;

        IF coalesce (new_run_instance, -1) = -1 THEN
                new_run_instance := 1; --First time
        END IF;

        -- Let's do it!
        SELECT pg_backend_pid() INTO SPID;

        INSERT INTO wfsdw.MD_ETLProcessRunInstance VALUES (
                etl_process,
                new_run_instance,
                parent_etl_process,
                parent_etl_process_runinstance,
                1,
                comments_var,
                NULL,
                NULL,
                NULL,
                NULL,
                NULL,
                NULL,
                SPID,
                now()::timestamp,
                NULL);

        --COMMIT ;

        new_run_instance := new_run_instance;
        error_status := 0;
        ret := (error_status, new_run_instance);

        RETURN query
            SELECT error_status, new_run_instance;
EXCEPTION
        WHEN OTHERS THEN
                raise notice 'spMD_LogProcessStart Inserting into wfsdw.MD_ETLProcessRunInstance % - %', SQLSTATE, SQLERRM;
                v_sqlcode := SQLSTATE;
                --ROLLBACK;
                -- Not raising application errors at this point!
                -- raise_application_error(-20001, 'spMD_LogProcessStart ERROR: error inserting wfsdw.MD_ETLProcessMessage');
                
                SELECT spMD_WriteEventLog ('spMD_LogProcessStart', v_sqlcode, '', 'Inserting into wfsdw.MD_ETLProcessRunInstance') into tmp_results;
                --COMMIT;
                new_run_instance := -1;
                error_status := -1;
                ret := (error_status, new_run_instance);
                RETURN query SELECT error_status, new_run_instance;
END;
$func_spMD_LogProcessStart$ LANGUAGE plpgsql;
