﻿-- DROP FUNCTION WFSDW.FN_STR_WITH_SEPS(iStr text, iSep text);

CREATE OR REPLACE FUNCTION WFSDW.FN_STR_WITH_SEPS(iStr text, iSep text) 
RETURNS text AS $func_str_with_seps$
DECLARE
    strOut  text;
BEGIN
    if ( length(iStr) > 0  AND length(replace(iStr,iSep,'')) <> length(iStr) )then
        SELECT replace (iStr,', ',',') INTO iStr;
        SELECT ''''|| replace (iStr,iSep,''''||iSep||'''') || '''' INTO strOut;
    else
        SELECT ''''|| iStr || '''' INTO strOut;
    end if;
    return strOut;
END;
$func_str_with_seps$ LANGUAGE plpgsql;

/*
*****************************************************************************
	PACKAGE: REPORT FUNCTIONS

      -- @version $Id: createwfsreport_dw.sql,v 1.4 2014/07/23 16:30:56 sneravati Exp $

*****************************************************************************
This package contains stored functions that provide data required for the Reports.
*****************************************************************************
[USED BY]	JReports
*****************************************************************************
[By Laura Francheri	- 10/16/2006]
    [Major rewrite for new structures by Chris Scherlacher 4/25/2007]
*****************************************************************************
Change Log:
            1)  Added the Procedure spRPTWorkOrders on 10/16/2006
                by Suresh Neravati.
            2)  Added the Procedure spStr_To_Strs on 10/16/2006
                by Suresh Neravati.
            3)  Added the Procedure spRPTWOsForAllProv on 11/09/2006
                by Imran Naqvi.
            4)  Added the Procedure spRPTAvgQTimeByRG on 11/10/2006
                by Imran Naqvi.
            5)  Added the Procedure spRPTAvgTaskDurByRG on 11/10/2006
                by Imran Naqvi.
            6)  Added the Procedure spRPTAvgTaskDurByRG on 11/10/2006
                by Kinjal Meheta.
            7)  Added the Procedure spContntListByProvdrAssetTyp on 11/10/2006
                by Sahil Verma.
            8)  Added the Procedure spWOsLicenseWindowMiss on 11/10/2006
                by Sahil Verma.
            9)  Added the Procedure spFailureTrendByResGroup on 11/14/2006
                by Suresh Neravati.
            10) Added the Procedure spFailureTrendByResource on 11/15/2006
                by Suresh Neravati.
            11) Added the Procedure spRPTDriveUsage on 11/15/2006
                by Silva Vijay.
            12) Added the Procedure spWOsPerPackageAssetID on 11/15/2006
                by Jakobac, Vladimir.
            13) Added the Procedure spRPTWOsPurge on 11/15/2006
                by Imran Naqvi.
            14) Rewrite of spRPTWorkOrders on 4/25/2007
            15) Modified seven procedures on 08/7/2010 by Suresh Neravati
                  to change parameters from varchar2 to date and modified the 
                  code to use the FN_STR_WITH_SEPS procedure for multi list.
            16) Added a new proc for Purged Titles report by Suresh Neravati 
                on 09/06/2010.                  
            17) Added a new proc for Expiring Content report for CMS2.0 
                by Suresh Neravati on 03/14/2011.
            18) Added a new proc for Missing Content report for CMS2.0 
                by Suresh Neravati on 03/18/2011.

            19) Converted the Stored procedures to Functions for Postgresql DB
                in CMS6.0 by Suresh Neravati - 07/12/2017 through 07/19/2017
*****************************************************************************
*/
/*
***************************************************************************************
	PROCEDURE: spRPTWOsForAllProv (ContentVolumesNoProvider)
	[REPORTS]
***************************************************************************************
	This provides the Data required for the Workorders by Template and by Time
        for all Providers
	[RETURNS] WorkOrders RefCursor which contain the work orders generated
                with in that period and for the templates list.
***************************************************************************************
	[USED BY]	Should be used to generate the WFS Report count of Work Orders
                by Template and Time.
***************************************************************************************
        [Developed by Imran Naqvi 11/09/2006]
        [ReWritten by Chris Scherlacher 4/26/2007]
***************************************************************************************
	Change Log: 
***************************************************************************************
*/

CREATE OR REPLACE FUNCTION funcRPTWOsForAllProv (  from_       IN  varchar,
                                                   to_         IN  varchar,
                                                   template_   IN  text,
                                                   wosData     OUT refcursor) 
RETURNS refcursor AS $funcRPTWOsForAllProv$
DECLARE
        vSQLCode      text;
        vErrMessage   varchar(500);
        procName      varchar(50);
        diff_days     int;
        paramsvalid   boolean:=true;
        stmtstr       text;
        vtemplates    text:='';
        tmp_results   smallint;
BEGIN
        procName := 'spRPTWOsForAllProv';
        template_ := replace(replace(template_,'[',''),']','');
        -- Validation of Parameters
        IF ((from_ IS NULL OR from_='') OR (to_ IS NULL OR to_='') OR (template_ IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(to_,'YYYY-MM-DD') - to_date(from_,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(to_) - date(from_) INTO diff_days;
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

        If Paramsvalid Then
            If template_ != 'ALL' Then
                SELECT wfsdw.Fn_Str_With_Seps(template_,',') INTO template_;
                vtemplates:= vtemplates||'  AND t.Name in ('||template_||')';
            End If;

            stmtstr:= 'select x.name, x.monthwk, coalesce(y.wocnt,0) wo_count from '||
                              '(select a.templatewk, a.name, d.monthwk::text '||
                              '   from  (select distinct t.templatewk, t.name '||
                              '            from (select distinct templatewk  '||
                              '                   from wfsdw.fact_workorderday  '||
                              '                  where datewk::text between '''||to_char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' and '''||to_char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''') wod '||
                              '            join (select templatewk, name from wfsdw.vdim_template  '||
                              '                   where processdefinitiontypeid = 2) t  '||
                              '            on wod.templatewk=t.templatewk  '||vtemplates||
                              ' ) a '||
                              '  cross join wfsdw.dim_month d '||
                              '  where d.monthwk::text between '''||to_char(to_date(from_,'YYYY-MM-DD'),'YYYYMM')||''' and '''||to_char(to_date(to_,'YYYY-MM-DD'),'YYYYMM')||''''||
                              ') x left join  '||
                              '(select distinct t.templatewk, t.name, substring(datewk::text from 1 for 6) monthwk_, sum(wod.workordercount) wocnt '||
                              '            from (select templatewk, sum(workordercount) as workordercount, datewk from wfsdw.fact_workorderday  '||
                              '               where datewk::text between '''||to_char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' and '''||to_char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' group by templatewk,datewk) wod '||
                              '            join (select templatewk, name from wfsdw.vdim_template where processdefinitiontypeid = 2) t  '||
                              '            on wod.templatewk=t.templatewk '||vtemplates||
                              '            GROUP BY t.templatewk, t.name, substring(datewk::text from 1 for 6) '||
                              ') y '||
                              ' on x.templatewk = y.templatewk and x.monthwk = y.monthwk_  and x.name = y.name '||
                              ' order by 1,2';

        -- Selecting the data into RefCursor
            OPEN wosData FOR EXECUTE stmtStr;
        ELSE
            -- Creating an empty cursor to return
            OPEN wosData FOR EXECUTE 'SELECT t.Name, m.YEARNUM || ''-'' || LPAD(m.MONTHNUM,2,''0''), wom.WorkOrderCount'||
                                     '  FROM wfsdw.FACT_WORKORDERMONTH wom '||
                                     '  JOIN wfsdw.DIM_PROVIDER p ON wom.ProviderWK = p.VariableInstanceWK'||
                                     '  JOIN wfsdw.DIM_TEMPLATE t ON wom.TemplateWK = t.TemplateWK'||
                                     '  JOIN wfsdw.DIM_MONTH m ON wom.MonthWK = m.MonthWK'||
                                     '  WHERE m.MonthWK BETWEEN to_char(to_date('''||to_date(from_,'YYYY-MM-DD')||''',''YYYY-MM-DD''),''YYYYMM'') AND to_char(to_date('''||to_||''',''YYYY-MM-DD''),''YYYYMM'')'||
                                     '  AND 1 = 2';
        END IF;
        
EXCEPTION
	WHEN NO_DATA_FOUND THEN
	    vSQLCode := SQLSTATE;
	    select wfsdw.spMD_WriteEventLog ('spRPTWOsForAllProv', vSQLCode, 'Query - No data found for the period from '||from_||' to '||to_||' for the templateWK '||template_) into tmp_results;
	WHEN OTHERS THEN
	    vSQLCode := SQLSTATE;
	    vErrMessage := procName || ': DB error.  Can''t provide data from the spRPTWosByTemplateTimeProvider..!';
	    select wfsdw.spMD_WriteEventLog ('spRPTWosByTemplateTimeProvider', vSQLCode, '',  vErrMessage) into tmp_results;
END;
$funcRPTWOsForAllProv$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spRPTAvgQTimeByRG (Average Queue Time By Resource Group)
	[REPORTS]
	*****************************************************************************
	This provides the Data required for the average time a resource was used by
    Resource Groups.
	[RETURNS] 	RefCursor which contain the resource groups, their
    average time used for a particular month.
	*****************************************************************************
	[USED BY]	Should be used to generate the Average Queue Time by Resource
    Group.
    *****************************************************************************
    	[SQL Server version by Imran Naqvi 11/10/2006]
	[ReWritten by Chris Scherlacher 4/26/2007]
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/

CREATE OR REPLACE FUNCTION funcRPTAvgQTimeByRG( from_          IN  varchar,
                                                to_            IN  varchar,
                                                resourceGroup  IN  text,
                                                wosData        OUT refcursor) 
RETURNS refcursor AS $funcRPTAvgQTimeByRG$
DECLARE
        vSQLCode     text;
        vErrMessage  varchar(500);
        procName     varchar(50);
        Diff_Days    Int;
        Paramsvalid  Boolean:=True;
        Stmtstr      text;
        vRGs         text:='';
        tmp_results  smallint;
      BEGIN
        procName := 'spRPTAvgQTimeByRG';
        resourceGroup := replace(replace(resourceGroup,'[',''),']','');
        -- Validation of Parameters
        IF ((from_ IS NULL OR from_='') OR (to_ IS NULL OR to_='') OR (resourceGroup IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(to_,'YYYY-MM-DD') - to_date(from_,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(to_) - date(from_) INTO diff_days;
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;        

        If Paramsvalid Then
            If Resourcegroup != 'ALL' Then
                SELECT wfsdw.Fn_Str_With_Seps(Resourcegroup,',') INTO Resourcegroup;
                vRGs:= vRGs||'  AND rg.ResourceGroupName in ('||Resourcegroup||')';
            End If;
            
            Stmtstr:='Select A.Resourcegroupname, A.Monthwk, coalesce(b.avg_qtime,0) From '|| 
                           '(Select Distinct ResourceGroupName, d.monthwk::text From '|| 
                           '       (select distinct rg.ResourceGroupName '|| 
                           '          From wfsdw.Fact_Resourcequeueday f '|| 
                           '          Join wfsdw.Dim_Resourcegroup Rg On F.Resourcegroupwk = Rg.Resourcegroupwk '|| 
                           '         where f.datewk::text between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vRGs||' '|| 
                           '         Group By Rg.Resourcegroupname) x '|| 
                           ' Cross Join wfsdw.Dim_Month D  '|| 
                           '             Where D.Monthwk::text Between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMM')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMM')||''') a '|| 
                           'Left Join '|| 
                           '( select rg.ResourceGroupName, substring(datewk::text from 1 for 6) monthwk_, '|| 
                           '         coalesce((SUM(totalDuration)/60)/(CASE WHEN SUM(totalQueued)=0 THEN NULL ELSE SUM(totalQueued) END),0) avg_qtime '|| 
                           '    from wfsdw.fact_resourcequeueday f '|| 
                           '    join wfsdw.dim_resourcegroup rg on f.resourcegroupwk = rg.resourcegroupwk '|| 
                           '   Where F.Datewk::text Between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vRGs||' '|| 
                           '   Group By Rg.Resourcegroupname, Substring(Datewk::text from 1 for 6)) b '|| 
                     '   On A.Resourcegroupname = B.Resourcegroupname '|| 
                     '  And A.Monthwk = B.Monthwk_ '|| 
                     '   order by 1,2';

            -- Selecting the data into RefCursor
            OPEN wosData FOR EXECUTE stmtStr;
        ELSE
            -- Creating an empty cursor to return
            OPEN wosData FOR EXECUTE 'SELECT rg.RESOURCEGROUPNAME, m.YEARNUM || ''-'' || LPAD(m.MONTHNUM,2,''0''), f.TotalDURATION '||
                                     '  FROM wfsdw.FACT_RESOURCEQUEUEMONTH f '||
                                     '  JOIN wfsdw.DIM_RESOURCEGROUP rg '||
                                     '    ON f.RESOURCEGROUPWK = rg.RESOURCEGROUPWK '||
                                     '  JOIN wfsdw.DIM_MONTH m '||
                                     '    ON f.MONTHWK = m.MONTHWK '||
                                     '  WHERE 1 = 2';
        END IF;

EXCEPTION
        WHEN NO_DATA_FOUND THEN
	    vSQLCode := SQLSTATE;
            select wfsdw.spMD_WriteEventLog ('spRPTAvgQTimeByRG', vSQLCode, 'Query - No data found for the period from '||from_||' to '||to_||' for the resourceGroup '||resourceGroup) into tmp_results;
        WHEN OTHERS THEN
            vSQLCode := SQLSTATE;
            vErrMessage := procName || ': DB error.  Can''t provide data from the spRPTAvgQTimeByRG..!';
            select wfsdw.spMD_WriteEventLog ('spRPTAvgQTimeByRG', vSQLCode, '',  vErrMessage) into tmp_results;

END;
$funcRPTAvgQTimeByRG$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spRPTAvgTaskDurByRG (Average Task Duration By Resource Group)
	[REPORTS]
	*****************************************************************************
	This provides the Data required for the average task duration for a resource
    by Resource Groups.
	[RETURNS] 	RefCursor which contain the resource groups, their
    average time used for a particular month.
	*****************************************************************************
	[USED BY]	Should be used to generate the Average Queue Time by Resource
    Group.
    *****************************************************************************
    [SQL Server version by Imran Naqvi 11/10/2006]
    [ReWritten by Chris Scherlacher 6/4/2007]
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcRPTAvgTaskDurByRG( from_            IN  varchar,
                                                  to_              IN  varchar,
                                                  resourceGroup    IN  text,
                                                  wosData          OUT refcursor) 
RETURNS refcursor AS $funcRPTAvgTaskDurByRG$
DECLARE
        vSQLCode            text;
        vErrMessage         varchar(500);
        procName            varchar(50);
        diff_days           int;
        Paramsvalid         Boolean:=True;
        stmtStr             text;
        vRGs                text:='';
        tmp_results         smallint;
      BEGIN
        procName := 'spRPTAvgTaskDurByRG';
        resourceGroup := replace(replace(resourceGroup,'[',''),']','');
        -- Validation of Parameters
        IF ((from_ IS NULL OR from_='') OR (to_ IS NULL OR to_='') OR (resourceGroup IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(to_,'YYYY-MM-DD') - to_date(from_,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(to_) - date(from_) INTO diff_days;
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF; 

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

        IF paramsValid THEN
            -- Report is desplaying data in average minutes
            If resourceGroup != 'ALL' Then
                SELECT wfsdw.Fn_Str_With_Seps(resourceGroup,',') INTO resourceGroup;
                vRGs:= vRGs||'  AND rg.ResourceGroupName in ('||resourceGroup||')';
            End If;

            Stmtstr:='Select A.Resourcegroupname, A.Monthwk, coalesce(b.avg_qtime,0) From '|| 
                           '(Select Distinct ResourceGroupName, d.monthwk::text From '|| 
                           '       (select distinct rg.ResourceGroupName '|| 
                           '          From wfsdw.Fact_ResourceTaskday f '|| 
                           '          Join wfsdw.Dim_Resourcegroup Rg On F.Resourcegroupwk = Rg.Resourcegroupwk '|| 
                           '         where f.datewk::text between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vRGs||' '|| 
                           '         Group By Rg.Resourcegroupname) x'|| 
                           ' Cross Join wfsdw.Dim_Month D  '|| 
                           '             Where D.Monthwk::text Between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMM')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMM')||''') a '|| 
                           'Left Join '|| 
                           '( select rg.ResourceGroupName, substring(datewk::text from 1 for 6) monthwk_, '|| 
                           ' ROUND(coalesce((SUM(f.totalDuration)/60)/(CASE WHEN SUM(f.totalTasks)=0 THEN NULL ELSE SUM(f.totalTasks) END),0),2) Avg_Qtime'||
                           '    from wfsdw.fact_resourceTaskday f '|| 
                           '    join wfsdw.dim_resourcegroup rg on f.resourcegroupwk = rg.resourcegroupwk '|| 
                           '   Where F.Datewk::text Between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vRGs||' '|| 
                           '   Group By Rg.Resourcegroupname, Substring(Datewk::text from 1 for 6)) b '|| 
                     '   On A.Resourcegroupname = B.Resourcegroupname '|| 
                     '  And A.Monthwk = B.Monthwk_ '|| 
                     '   order by 1,2';

            -- Selecting the data into RefCursor
            OPEN wosData FOR EXECUTE stmtStr;
        ELSE
            -- Creating an empty cursor to return
            OPEN wosData FOR EXECUTE 'SELECT rg.RESOURCEGROUPNAME, m.YEARNUM || ''-'' || LPAD(m.MONTHNUM,2,''0''), ROUND(f.TotalDURATION,2)'||
                          ' FROM wfsdw.FACT_RESOURCETASKMONTH f '||
                          ' JOIN wfsdw.DIM_RESOURCEGROUP rg ON f.RESOURCEGROUPWK = rg.RESOURCEGROUPWK '||
                                      '	JOIN wfsdw.DIM_MONTH m ON f.MONTHWK = m.MONTHWK '||
                                      ' WHERE 1 = 2';
        END IF;

EXCEPTION
        WHEN NO_DATA_FOUND THEN
	    vSQLCode := SQLSTATE;
            select wfsdw.spMD_WriteEventLog ('spRPTAvgTaskDurByRG', vSQLCode, '',  'Query - No data found for the period from '||from_||' to '||to_||' for the resourceGroup '||resourceGroup) into tmp_results;
        WHEN OTHERS THEN
            vSQLCode := SQLSTATE;
            vErrMessage := procName || ': DB error.  Can''t provide data from the spRPTAvgTaskDurByRG..!';
            select wfsdw.spMD_WriteEventLog ('spRPTAvgQTimeByRG', vSQLCode, '',  vErrMessage) into tmp_results;
END;
$funcRPTAvgTaskDurByRG$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spTaskDetailsByResource (Task Details By Resource)
	[REPORTS]
	*****************************************************************************
	Retrieves task details for selected resources
        *****************************************************************************
        [SQL Server version by Kinjal Meheta 11/10/2006]
	[Procedure Rewritten by Chris Scherlacher 6/8/2007]
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcTaskDetailsByResource( startdate IN varchar,
                                                      enddate   IN varchar,
                                                      resources IN text,
                                                      wosData   OUT refcursor)
RETURNS refcursor AS $funcTaskDetailsByResource$
DECLARE
        stmtStr       text;
        paramsValid   boolean:=true;
        diff_days     int;
        vSQLCode      text;
        vErrMessage   varchar(500);
        procName      varchar(50);
        tmp_results   smallint;
BEGIN
        procName := 'spTaskDetailsByResource';
        resources := replace(replace(resources,'[',''),']','');
        -- Validation of Parameters
        IF ((startdate IS NULL OR startdate='') OR (enddate IS NULL OR enddate='') OR (resources IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(enddate,'YYYY-MM-DD') - to_date(startdate,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(enddate) - date(startdate) INTO diff_days; 
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

        IF paramsValid THEN

            stmtStr :='SELECT dr.ResourceName, dwo.ProcessInstanceID, dt.Name as TemplateName ,'''' as taskName, dwo.TITLE,'||  -- dn.Name
                            ' to_char(dsd.CALENDERDATE,''YYYY-MM-DD'')||'' ''||to_char(dst.TIMEOFDAY,''HH:MI:SS AM'') as starttime,'||
                            ' to_char(dld.CALENDERDATE,''YYYY-MM-DD'')||'' ''||to_char(dlt.TIMEOFDAY,''HH:MI:SS AM'') as endtime,'||
                            ' Duration/60 as Duration, dwos.Name as TaskStatus'||
                      ' FROM wfsdw.FACT_RESOURCETASKDETAIL f'||
                      ' JOIN wfsdw.DIM_RESOURCE dr ON f.RESOURCEWK = dr.RESOURCEWK'||
                      ' JOIN wfsdw.DIM_WORKORDER dwo ON f.WORKORDERWK = dwo.WORKORDERWK'||
                      ' JOIN wfsdw.DIM_TEMPLATE dt ON f.TEMPLATEWK = dt.TEMPLATEWK'||
                      --' JOIN wfsdw.DIM_NODE dn ON f.NODEWK = dn.NODEWK'||
                      ' JOIN wfsdw.DIM_DATE dsd ON f.LOCALENTERDATEWK = dsd.DATEWK'||
                      ' JOIN wfsdw.DIM_TIME dst ON f.LOCALENTERTIMEWK = dst.TIMEWK'||
                      ' JOIN wfsdw.DIM_DATE dld ON f.LOCALLEAVEDATEWK = dld.DATEWK'||
                      ' JOIN wfsdw.DIM_TIME dlt ON f.LOCALLEAVETIMEWK = dlt.TIMEWK'||
                      ' JOIN wfsdw.DIM_WORKORDERSTATUS dwos ON f.OperStatusWK = dwos.WORKORDERSTATUSWK'||
                     ' WHERE dsd.CALENDERDATE >= '''||to_date(startdate,'YYYY-MM-DD')||''''||
                     '   AND dld.CALENDERDATE <= '''||to_date(enddate,'YYYY-MM-DD')||'''';
            
            IF resources!= 'ALL' THEN
                SELECT wfsdw.Fn_Str_With_Seps(resources,',') INTO resources;
                stmtStr:= stmtStr||' AND dr.ResourceName IN ('||resources||')'; 
                --stmtStr:= stmtStr||' AND dr.ResourceName IN ('''||wfsdw.FN_STR_WITH_SEPS(resources,',')||''')'; 
            END IF;   
            stmtStr := stmtStr||' ORDER BY dr.ResourceName, to_char(dsd.CALENDERDATE,''YYYY-MM-DD'')||'' ''||to_char(dst.TIMEOFDAY,''HH:MI:SS AM'')';
            OPEN wosData FOR EXECUTE stmtStr;
        ELSE
            OPEN wosData FOR EXECUTE 'SELECT dr.ResourceName, dwo.ProcessInstanceID, dt.Name as TemplateName, '''' as TaskName, dwo.Title,'||
                            ' to_char(dsd.CALENDERDATE,''YYYY-MM-DD'')||'' ''||to_char(dst.TIMEOFDAY,''HH:MI:SS AM'') as starttime,'||
                            ' to_char(dld.CALENDERDATE,''YYYY-MM-DD'')||'' ''||to_char(dlt.TIMEOFDAY,''HH:MI:SS AM'') as endtime,'||
                            ' Duration/60 as Duration, dwos.Name as TaskStatus '||
            ' FROM wfsdw.FACT_RESOURCETASKDETAIL f '||
            ' JOIN wfsdw.DIM_RESOURCE dr ON f.RESOURCEWK = dr.RESOURCEWK '||
            ' JOIN wfsdw.DIM_WORKORDER dwo ON f.WORKORDERWK = dwo.WORKORDERWK '||
            ' JOIN wfsdw.DIM_TEMPLATE dt ON f.TEMPLATEWK = dt.TEMPLATEWK '||
            --' JOIN wfsdw.DIM_NODE dn ON f.NODEWK = dn.NODEWK '||
            ' JOIN wfsdw.DIM_DATE dsd ON f.LOCALENTERDATEWK = dsd.DATEWK '||
            ' JOIN wfsdw.DIM_TIME dst ON f.LOCALENTERTIMEWK = dst.TIMEWK '||
            ' JOIN wfsdw.DIM_DATE dld ON f.LOCALLEAVEDATEWK = dld.DATEWK '||
            ' JOIN wfsdw.DIM_TIME dlt ON f.LOCALLEAVETIMEWK = dlt.TIMEWK '||
            ' JOIN wfsdw.DIM_WORKORDERSTATUS dwos ON f.OperStatusWK = dwos.WORKORDERSTATUSWK '||
            ' WHERE 1 = 2';
         END IF;
EXCEPTION
        WHEN NO_DATA_FOUND THEN
	    vSQLCode := SQLSTATE;
            select wfsdw.spMD_WriteEventLog ('spTaskDetailsByResource', vSQLCode, '',  'Query - No data found for the period from '||startdate||' to '||enddate||' for the resources '||resources) into tmp_results;
        WHEN OTHERS THEN
            vSQLCode := SQLSTATE;
            vErrMessage := procName || ': DB error.  Can''t provide data from the spTaskDetailByResource..!';
            select wfsdw.spMD_WriteEventLog ('spTaskDetailsByResource', vSQLCode, '',  vErrMessage) into tmp_results;
END;
$funcTaskDetailsByResource$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spFailureTrendByResGroup (Failure Rate By Resource Group)
	[REPORTS]
	*****************************************************************************
	This provides the Failure percentage Data required for the Resource Groups.
	[RETURNS] 	RGData RefCursor contains the Month wise Resource Group Failure
                percentage.

	[NOTE]		As per the Development team request at this point I'm not raising
                application errors . Just uncommenting the raise_application_error
                calls.
	*****************************************************************************
	[USED BY]	Should be used to generate the WFS Report Month wise Resource
                Group Failure percentage
	*****************************************************************************
	[SQL Server version by Suresh Neravati 11/14/2006]
	[Oracle version by Suresh - revised by Suresh on 11/14/2006]
	[Re-Written by Chris Scherlacher 6/14/07]
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcFailureTrendByResGroup ( FromDate       IN  varchar,
                                                        ToDate         IN  varchar,
                                                        ResourceGroup  IN  text,
                                                        RGData         OUT refcursor) 
RETURNS refcursor AS $funcFailureTrendByResGroup$
DECLARE
        vSQLCode      		text;
        vErrMessage   		varchar(500);
        procName      		varchar(50);
        Diff_Days     		Int;
        Paramsvalid   		Boolean:=True;
        Stmtstr       		text;
        vRGs                    text:='';
        tmp_results             smallint;
      Begin
        Procname := 'spFailureTrendByResGroup';
        ResourceGroup := replace(replace(ResourceGroup,'[',''),']','');
        -- Validation of Parameters
        IF ((FromDate IS NULL OR FromDate='') OR (ToDate IS NULL OR ToDate='') OR (ResourceGroup IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(ToDate,'YYYY-MM-DD') - to_date(FromDate,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(ToDate) - date(FromDate) INTO diff_days;
                     EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

        If Paramsvalid Then
            If resourceGroup != 'ALL' Then
                SELECT wfsdw.Fn_Str_With_Seps(resourceGroup,',') INTO resourceGroup;
                vRGs:= vRGs||'  AND rg.ResourceGroupName in ('||resourceGroup||')';
            End If;
            
            Stmtstr:='Select A.Resourcegroupname, A.Monthwk, coalesce(b.Avg_Errors,0) From '|| 
                           '(Select Distinct ResourceGroupName, d.monthwk::text From '|| 
                           '       (select distinct rg.ResourceGroupName '|| 
                           '          From wfsdw.Fact_ResourceTaskday f '|| 
                           '          Join wfsdw.Dim_Resourcegroup Rg On F.Resourcegroupwk = Rg.Resourcegroupwk '|| 
                           '         where f.datewk::text between '''||To_Char(to_date(FromDate,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(ToDate,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vRGs||' '|| 
                           '         Group By Rg.Resourcegroupname) x'|| 
                           ' Cross Join wfsdw.Dim_Month D  '|| 
                           '             Where D.Monthwk::text Between '''||To_Char(to_date(FromDate,'YYYY-MM-DD'),'YYYYMM')||''' AND '''||To_Char(to_date(ToDate,'YYYY-MM-DD'),'YYYYMM')||''') a '|| 
                           'Left Join '|| 
                           '( select rg.ResourceGroupName, substring(datewk::text from 1 for 6) monthwk_, '|| 
                           ' ROUND((SUM(f.totalErrors)/case when SUM(f.totalTasks)=0 then 0 else SUM(f.totalTasks) end)*100,2) Avg_Errors'||
                           '    from wfsdw.fact_resourceTaskday f '|| 
                           '    join wfsdw.dim_resourcegroup rg on f.resourcegroupwk = rg.resourcegroupwk '|| 
                           '   Where F.Datewk::text Between '''||To_Char(to_date(FromDate,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(ToDate,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vRGs||' '|| 
                           '   Group By Rg.Resourcegroupname, Substring(Datewk::text from 1 for 6)) b '|| 
                     '   On A.Resourcegroupname = B.Resourcegroupname '|| 
                     '  And A.Monthwk = B.Monthwk_ '|| 
                     '   order by 1,2';
                     
            -- Selecting the data into RefCursor
            OPEN RGData FOR EXECUTE stmtStr;
        ELSE
            -- Creating an empty cursor to return
            OPEN RGData FOR EXECUTE 'SELECT rg.RESOURCEGROUPNAME, m.YEARNUM || ''-'' || LPAD(m.MONTHNUM,2,''0''), f.TotalDURATION'||
                                    '  FROM wfsdw.FACT_RESOURCETASKMONTH f'||
                                    '  JOIN wfsdw.DIM_RESOURCEGROUP rg ON f.RESOURCEGROUPWK = rg.RESOURCEGROUPWK'||
                                    '  JOIN wfsdw.DIM_MONTH m ON f.MONTHWK = m.MONTHWK'||
                                    ' WHERE 1 = 2';
        END IF;
EXCEPTION
        WHEN NO_DATA_FOUND THEN
	    vSQLCode := SQLSTATE;
            select wfsdw.spMD_WriteEventLog ('spFailureTrendByResGroup', vSQLCode, '',  'Query - No data found for the period from '||FromDate||' to '||ToDate||' for the Resource Group '||ResourceGroup) 
            into tmp_results;
        WHEN OTHERS THEN
            vSQLCode := SQLSTATE;
            vErrMessage := procName || ': DB error.  Can''t provide data from the spFailureTrendByResGroup..!';
            select wfsdw.spMD_WriteEventLog ('spFailureTrendByResGroup', vSQLCode, '',  vErrMessage) into tmp_results;
END;
$funcFailureTrendByResGroup$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spFailureTrendByResource (Failure Rate By Resource)
	[REPORTS]
	*****************************************************************************
	This provides the Failure percentage Data required for the Resources.
	[RETURNS] 	RGData RefCursor contains the Month wise Resources Failure
                percentage.

	[NOTE]		As per the Development team request at this point I'm not raising
                application errors . Just uncommenting the raise_application_error
                calls.
	*****************************************************************************
	[USED BY]	Should be used to generate the WFS Report Month wise Resources
                Failure percentage.
	*****************************************************************************
	[SQL Server version by Suresh Neravati 11/15/2006]
	[Oracle version by Suresh - revised by Suresh on 11/15/2006]
	[ReWritten by Chris Scherlacher 6/18/2007]
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcFailureTrendByResource ( FromDate     IN  varchar,
                                                        ToDate       IN  varchar,
                                                        Resource_    IN  text,
                                                        ResourceData OUT refcursor) 
RETURNS refcursor AS $funcFailureTrendByResource$
DECLARE
        vSQLCode      text;
        vErrMessage   varchar(500);
        procName      varchar(50);
        diff_days     int;
        paramsValid   boolean:=true;
        stmtStr       text;
        vResources    text:='';
        tmp_results   smallint;
      BEGIN
        procName := 'spFailureTrendByResource';
        Resource_ := replace(replace(Resource_,'[',''),']','');
        -- Validation of Parameters
        IF ((FromDate IS NULL OR FromDate='') OR (ToDate IS NULL OR ToDate='') OR (Resource_ IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(ToDate,'YYYY-MM-DD') - to_date(FromDate,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(ToDate) - date(FromDate) INTO diff_days; 
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

        If Paramsvalid Then
            If Resource_ != 'ALL' Then
                SELECT wfsdw.Fn_Str_With_Seps(Resource_,',') INTO Resource_;
                vResources:= vResources||'  AND dr.ResourceName in ('||Resource_||')';
            End If;
            
            Stmtstr:='Select A.ResourceName, A.Monthwk, coalesce(b.Avg_Errors,0) From '|| 
                           '(Select Distinct ResourceName, d.monthwk::text From '|| 
                           '       (select distinct rg.ResourceName '|| 
                           '          From wfsdw.Fact_ResourceTaskday f '|| 
                           '          Join wfsdw.Dim_Resource Rg On F.Resourcewk = Rg.Resourcewk '|| 
                           '         where f.datewk::text between '''||To_Char(to_date(FromDate,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(ToDate,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vResources||' '|| 
                           '         Group By Rg.Resourcename) x'|| 
                           ' Cross Join wfsdw.Dim_Month D  '|| 
                           '             Where D.Monthwk::text Between '''||To_Char(to_date(FromDate,'YYYY-MM-DD'),'YYYYMM')||''' AND '''||To_Char(to_date(ToDate,'YYYY-MM-DD'),'YYYYMM')||''') a '|| 
                           'Left Join '|| 
                           '( select rg.ResourceName, substring(datewk::text from 1 for 6) monthwk_, '|| 
                           ' ROUND((SUM(f.totalErrors)/case when SUM(f.totalTasks)=0 then 0 else SUM(f.totalTasks) end)*100,2) Avg_Errors'||
                           '    from wfsdw.fact_resourceTaskday f '|| 
                           '    join wfsdw.dim_resource rg on f.resourcewk = rg.resourcewk '|| 
                           '   Where F.Datewk::text Between '''||To_Char(to_date(FromDate,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(ToDate,'YYYY-MM-DD'),'YYYYMMDD')||''' '||vResources||' '|| 
                           '   Group By Rg.Resourcename, Substring(Datewk::text from 1 for 6)) b '|| 
                     '   On A.Resourcename = B.Resourcename '|| 
                     '  And A.Monthwk = B.Monthwk_ '|| 
                     '   order by 1,2';
                     
        -- Selecting the data into RefCursor
           OPEN ResourceData FOR EXECUTE stmtStr;
        ELSE
            -- Creating an empty cursor to return
            --OPEN ResourceData FOR 'SELECT dr.RESOURCENAME, m.YEARNUM || ''-'' || LPAD(m.MONTHNUM,2,''0''), f.TotalDURATION'||
            OPEN ResourceData FOR EXECUTE 'SELECT NULL, NULL, NULL'||
                                          ' FROM wfsdw.FACT_RESOURCETASKMONTH f'||
                                          '  JOIN wfsdw.DIM_RESOURCE dr ON f.RESOURCEWK = dr.RESOURCEWK'||
                                          '  JOIN wfsdw.DIM_MONTH m ON f.MONTHWK = m.MONTHWK '||
                                          ' WHERE 1 = 2';
        END IF;
EXCEPTION
        WHEN NO_DATA_FOUND THEN
	    vSQLCode := SQLSTATE;
            select wfsdw.spMD_WriteEventLog ('spFailureTrendByResource', vSQLCode, '',  'Query - No data found for the period from '||FromDate||' to '||ToDate||' for the Resources '||Resource_) 
			  into tmp_results;
        WHEN OTHERS THEN
            vSQLCode := SQLSTATE;
            vErrMessage := procName || ': DB error.  Can''t provide data from the spFailureTrendByResource..!';
            select wfsdw.spMD_WriteEventLog ('spFailureTrendByResource', vSQLCode, '',  vErrMessage) into tmp_results;
END;
$funcFailureTrendByResource$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spRPTDriveUsage (Average SAN Usage By Drive)
	[REPORTS]
	*****************************************************************************
	This Stored Procedure provides the Data that shows the change of the SAN
    Drive usage over time.
	[NOTE]		As per the Development team request at this point I'm not raising
                application errors . Just uncommenting the raise_application_error
                calls.
	*****************************************************************************
	[USED BY]	Should be used to generate the WFS Report.
	*****************************************************************************
	Developed by Silva Vijay 10/16/2006
	[ReWritten by Chris Scherlacher 7/11/2007]
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcRPTDriveUsage ( from_       IN  varchar,
                                               to_         IN  varchar,
                                               driveNames  IN  text,
                                               wosData     OUT refcursor) 
RETURNS refcursor AS $funcRPTDriveUsage$
DECLARE
        vSQLCode      text;
        vErrMessage   varchar(500);
        procName      varchar(50);
        diff_days     int;
        paramsValid   boolean:=true;
        Stmtstr       text := '';
        vDrives       text := '';
        tmp_results   smallint;
    BEGIN
        procName := 'spRPTDriveUsage';
        driveNames := replace(replace(driveNames,'[',''),']','');
        -- Validation of Parameters
        IF ((from_ IS NULL OR from_='') OR (to_ IS NULL OR to_='') OR (driveNames IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(to_,'YYYY-MM-DD') - to_date(from_,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(to_) - date(from_) INTO diff_days;
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

      If Paramsvalid Then
            If driveNames != 'ALL' Then
                SELECT wfsdw.Fn_Str_With_Seps(driveNames,',') INTO driveNames;
                vDrives:= vDrives||'  AND Md.Name_ in ('||driveNames||')';
            End If;
            
            Stmtstr :='Select x.drivename, x.monthwk, coalesce(y.Maxusedpercentage,0) from '||
                        '  (Select Distinct A.Drivename, M.Monthwk::text  From '||
                        '          (Select distinct Md.Name_ As Drivename, substring(d.datewk::text from 1 for 6) As Monthname '||
                        '             From wfsdw.Fact_Mdhistoryday F '||
                        '             Join wfsdw.Dim_Monitoreddrive Md On F.Monitoreddrivewk = Md.Monitoreddrivewk '||
                        '             Join wfsdw.Dim_Date D On F.Localdatewk = D.Datewk '||
                        '            Where F.Localdatewk::text Between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' '||Vdrives||
                        ' ) a  Cross Join wfsdw.Dim_Month M Where M.Monthwk::text between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMM')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMM')||''') x '||
                        ' left join '||
                        ' (Select Md.Name_ As Drivename, substring(d.datewk::text from 1 for 6) As Monthname, max(F.percentused) As Maxusedpercentage '||
                        '    FROM wfsdw.FACT_MDHISTORYDAY f '||
                        '    Join wfsdw.Dim_Monitoreddrive Md On F.Monitoreddrivewk = Md.Monitoreddrivewk '||
                        '    Join wfsdw.Dim_Date D On F.Localdatewk = D.Datewk '||
                        '   Where D.Datewk::text Between '''||To_Char(to_date(from_,'YYYY-MM-DD'),'YYYYMMDD')||''' AND '''||To_Char(to_date(to_,'YYYY-MM-DD'),'YYYYMMDD')||''' '||Vdrives||
                        '   Group By Md.Name_, Substring(D.Datewk::text from 1 for 6)) y'||
                        '      On x.Drivename = y.Drivename And '||
                        '         x.Monthwk = y.Monthname '||
                    '  Order By 1,2';

          -- Selecting the data into RefCursor
          OPEN wosData FOR EXECUTE stmtStr;
      ELSE
          -- Creating an empty cursor to return
          OPEN wosData FOR EXECUTE 'SELECT ''Drive Name'' AS "DRIVE_NAME", ''Month'' AS "MONTH", 0 AS "USED_PERCENTAGE" FROM DUAL WHERE 1 = 2';
      END IF;
EXCEPTION
      WHEN NO_DATA_FOUND THEN
	  vSQLCode := SQLSTATE;
          select wfsdw.spMD_WriteEventLog (procName, vSQLCode, 'Query - No data found for the period from '||from_||' to '||to_||' for the SAN Drive: '|| driveNames) into tmp_results;
      WHEN OTHERS THEN
          vSQLCode := SQLSTATE;
          vErrMessage := procName || ': DB error.  Can''t provide data.';
          select wfsdw.spMD_WriteEventLog (procName, vSQLCode, '',  vErrMessage) into tmp_results;
END;
$funcRPTDriveUsage$ LANGUAGE plpgsql;
/*
	*****************************************************************************
	PROCEDURE: spRPTTitlePurge (TitlePurge)
	[REPORTS]
	*****************************************************************************
	This Stored Procedure provides the Data that shows the purged Titles.
	[NOTE]		As per the Development team request at this point I'm not raising
            application errors . Just uncommenting the raise_application_error
            calls.
	*****************************************************************************
	[USED BY]	Should be used to generate the WFS Report.
	*****************************************************************************
	Developed by Suresh Neravati 09/06/2010
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcRPTTitlePurge ( From_            In  varchar,
                                               To_              In  varchar,
                                               Providers        In  text,
                                               TitlePurgeData   OUT refcursor) 
RETURNS refcursor AS $funcRPTTitlePurge$
DECLARE
        vSQLCode      text;
        vErrMessage   varchar(500);
        procName      varchar(50);
        diff_days     int;
        paramsvalid   boolean:=True;
        stmtStr       text := '';
        tmp_results   smallint;
    BEGIN
        procName := 'spRPTTitlePurge';
        Providers := replace(replace(Providers,'[',''),']','');
        -- Validation of Parameters
        IF ((from_ IS NULL OR from_='') OR (to_ IS NULL OR to_='') OR (Providers IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(to_,'YYYY-MM-DD') - to_date(from_,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(to_) - date(from_) INTO diff_days;
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

      If Paramsvalid Then
          Stmtstr :='Select T.Title_Id, T.Provider_Id, T.Provider_Name, T.Purge_Date, T.Title_Brief, '||
                                  'T.Package_Asset_Id, Asset_Name, License_Start, License_End, Createdate, Updatedate '||
                    '  FROM Wfsdw.V_Titlepurge T'||
                    --' WHERE to_date(to_char(T.Purge_Date,''DD-MON-YYYY''),''DD-MON-YYYY'') BETWEEN '''||
                    ' WHERE T.Purge_Date BETWEEN '''||
                        --to_char(to_date(from_,'YYYY-MM-DD'),'YYYY-MM-DD')||''' AND '''||to_char(to_date(to_,'YYYY-MM-DD'),'YYYY-MM-DD')||'''';
                        to_date(from_,'YYYY-MM-DD')||''' AND '''||to_date(to_,'YYYY-MM-DD')||'''';

          If Providers!= 'ALL' Then
              SELECT wfsdw.Fn_Str_With_Seps(Providers,',') INTO Providers;
              stmtStr:= stmtStr||' AND T.Provider_Id IN ('||Providers||')';
              --stmtStr:= stmtStr||' AND T.Provider_Id IN ('''||wfsdw.FN_STR_WITH_SEPS(Providers,',')||''')';
          END IF;
          STMTSTR := STMTSTR ||' ORDER BY 1';
          --dbms_output.put_line(Stmtstr);
          -- Selecting the data into RefCursor
          Open Titlepurgedata FOR EXECUTE stmtStr;
      ELSE
          -- Creating an empty cursor to return
          OPEN TitlePurgeData FOR EXECUTE 'SELECT T.Title_Id, T.Provider_Id, T.Provider_Name, T.Purge_Date, T.Title_Brief, '||
                                                 'T.Package_Asset_Id, Asset_Name, License_Start, License_End, Createdate, Updatedate '||
                                          '  FROM wfsdw.V_Titlepurge T'||
                                          ' WHERE 1 = 2';
      END IF;
EXCEPTION
      When No_Data_Found Then
	  vSQLCode := SQLSTATE;
          select wfsdw.spMD_WriteEventLog (procName, vSQLCode, '',  'Query - No data found for the period from '||from_||' to '||to_||' ') into tmp_results;
      WHEN OTHERS THEN
          vSQLCode := SQLSTATE;
          vErrMessage := procName || ': DB error.  Can''t provide data.';
          select wfsdw.spMD_WriteEventLog (PROCNAME, VSQLCODE, '',  VERRMESSAGE) into tmp_results;
END;     
$funcRPTTitlePurge$ LANGUAGE plpgsql;

/*
	*****************************************************************************
	PROCEDURE: spRPTExpiringContent (ExpiringContent)
	[REPORTS]
	*****************************************************************************
	This Stored Procedure provides the Data that shows the Expiring Contents.
	[NOTE]		As per the Development team request at this point I'm not raising
            application errors . Just uncommenting the raise_application_error
            calls.
	*****************************************************************************
	[USED BY]	Should be used to generate the WFS Report.
	*****************************************************************************
	Developed by Suresh Neravati 03/14/2011
	*****************************************************************************
	Change Log:
	*****************************************************************************
*/
CREATE OR REPLACE FUNCTION funcRPTExpiringContent ( from_            in  varchar,
                                                    to_              in  varchar,
                                                    providers        in  text,
                                                    ExpiringContentData  OUT refcursor) 
RETURNS refcursor AS $funcRPTExpiringContent$
DECLARE
        vSQLCode      text;
        vErrMessage   varchar(500);
        procName      varchar(50);
        diff_days     int;
        paramsvalid   boolean:=True;
        stmtStr       text := '';
        tmp_results   smallint;
BEGIN
        procname := 'spRPTExpiringContent';
        providers := replace(replace(providers,'[',''),']','');
        -- Validation of Parameters
        IF ((from_ IS NULL OR from_='') OR (to_ IS NULL OR to_='') OR (providers IS NULL)) THEN 
            --raise_application_error(-20101,'Dates should not be null');
            paramsValid := false;
        Else
            BEGIN
                SELECT replace(to_date(to_,'YYYY-MM-DD') - to_date(from_,'YYYY-MM-DD'),' days','') INTO diff_days;
            EXCEPTION
                WHEN OTHERS THEN
                    BEGIN
                        SELECT date(to_) - date(from_) INTO diff_days;
                    EXCEPTION
                        WHEN OTHERS THEN paramsValid := false;
                     END;      
            END;
        END IF;

        IF (diff_days < 0) THEN 
            paramsValid := false;
        END IF;

      IF paramsvalid THEN
          Stmtstr := 'SELECT p.PROVIDERID AS "PROVIDER_ID", ' ||
                            'p.NAME AS "PROVIDER_NAME", ' ||
                            'ec.ASSET_NAME AS "ASSET_NAME", ' ||
                            'ec.PACKAGE_ASSETID AS "PACKAGE_ASSETID", ' || 
                            'to_char(ec.LICENSING_STARTDT,''DD-MON-YY'') as "LICENSING_STARTDT", ' ||
                            'to_char(ec.LICENSING_ENDDT,''DD-MON-YY'') AS "LICENSING_ENDDT",  ' ||
                            'ec.TITLE_BRIEF AS TITLE_BRIEF ' ||
                       'FROM wfs.VDW_MDM_Metadata_Fields ec ' ||
                       'JOIN wfs.pmm_title t ON (t.mdmdocid = ec.document_id) ' ||
                       'JOIN wfs.pmm_partner p ON (p.id_ = t.partner_id) ' ||
                     ' where ec.LICENSING_ENDDT between '|| 
                     'to_date('''||from_||''',''YYYY-MM-DD'') AND 
                      to_timestamp('''||to_||' 23:59:59'||''',''YYYY-MM-DD hh24:mi:ss'')::TIMESTAMP WITHOUT TIME ZONE';
                     
          IF Providers!= 'ALL' THEN
              SELECT wfsdw.Fn_Str_With_Seps(Providers,',') INTO Providers;
              stmtStr:= stmtStr||' AND p.providerid IN ('||Providers||')';
              --stmtStr:= stmtStr||' AND p.providerid IN ('''||wfsdw.FN_STR_WITH_SEPS(Providers,',')||''')';
          END IF;
          --stmtstr := stmtstr ||' ORDER BY 1';
          --dbms_output.put_line(Stmtstr);
          -- Selecting the data into RefCursor
          OPEN ExpiringContentData FOR EXECUTE stmtStr;
      ELSE
          -- Creating an empty cursor to return
          OPEN ExpiringContentData FOR EXECUTE 'SELECT p.PROVIDERID AS "PROVIDER_ID", ' ||
                                                        'p.NAME AS "PROVIDER_NAME", ' ||
                                                        'ec.ASSET_NAME AS "ASSET_NAME", ' ||
                                                        'ec.PACKAGE_ASSETID AS "PACKAGE_ASSETID", ' || 
                                                        'to_char(ec.LICENSING_STARTDT,''DD-MON-YY'') as "LICENSING_STARTDT", ' ||
                                                        'to_char(ec.LICENSING_ENDDT,''DD-MON-YY'') AS "LICENSING_ENDDT",  ' ||
                                                        'ec.TITLE_BRIEF AS TITLE_BRIEF ' ||
                                                ' FROM wfs.VDW_MDM_Metadata_Fields ec ' ||
                                                ' JOIN wfs.pmm_title t ON (t.mdmdocid = ec.document_id) ' ||
                                                ' JOIN wfs.pmm_partner p ON (p.id_ = t.partner_id) where 1 = 2';
      END IF;
EXCEPTION
      when no_data_found then
          vSQLCode := SQLSTATE;
          select wfsdw.spMD_WriteEventLog (procName, vSQLCode, '',  'Query - No data found for the period from '||from_||' to '||to_||' ') into tmp_results;
      WHEN OTHERS THEN
          vSQLCode := SQLSTATE;
          vErrMessage := procName || ': DB error.  Can''t provide data.';
          select wfsdw.spMD_WriteEventLog (procname, vsqlcode, '',  verrmessage) into tmp_results;
END;
$funcRPTExpiringContent$ LANGUAGE plpgsql;
