#!/usr/bin/python

import sys
sys.path.insert(1, './py_lib')
#sys.path.insert(1, './py3_lib')

import os
import re
import jinja2
import yaml
import argparse
import logging
import subprocess
import time
from datetime import datetime
import signal
import base64

#Interval seconds between stages
DEFAULT_STAGE_INTERVAL = 30

#Maximum amount for kept backup snapshots
MAX_BACKUP_SNAPSHOT_NUM = 5

#Timeout seconds for waiting pods to be terminated for kubectl scale command
SCALETIMEOUT = 50

#Default timeout setting for helm upgrade command
HELM_UPGRADE_TIMEOUT = "300s"

#timeout setting for helm rollback command
HELM_ROLLBACK_TIMEOUT = "1800s"

#components in stages less or equal than this value will enforce re-deploy each time running deploy
ENFORCE_REDEPLOY_STAGES = 0

#components in stages less or equal than this value will enforce adding the --wait and --timeout as helm option
ENFORCE_WAIT_OPTION_STAGES = 3

#components in stages less than this value will enforce the deployment to stopped when error happened.
ENFORCE_FAIL_STAGES = 5

PROFILE_FILE_PREFIX = "prof"
DPYPKG_FILE_PREFIX = "dpypkg"
PROFILE_FOLDER_NAME = "profiles"
DEPLOYPACKAGE_FOLDER_NAME = "deploypackages"
#PROFILE_FILE_PREFIX = "mdt-prof"
#DPYPKG_FILE_PREFIX = "mdt-dpypkg"
#PROFILE_FOLDER_NAME = "mdt-profiles"
#DEPLOYPACKAGE_FOLDER_NAME = "mdt-deploypackages"
CONFIGBUNDLE = "ConfigBundle"
STRING_JOINER = "_"
YAMM_FILE_SUFFIX = ".yaml"
CHART_FILE_SUFFIX = ".tgz"

DEFAULT_VAR_FILE_NAME = "products-var.yaml"
BACKUP_SNAPSHOT_SECRET_NAME = "cms-deployment-backup-snapshot"
BACKUP_SNAPSHOT_CONFIGMAP_NAME = "cms-deployment-backup-snapshot"
BACKUP_RESOURCE_FILE = "./py_lib/cms_backup_resources.yaml"
REPLICAS_FILE = "./py_lib/cms_replicas.yaml"

#init the helm error count 
helm_error = 0

#init the rollback_target_version, which will be set from method parse_k8s_replicas
rollback_target_version = ""

def printdot(seconds):
  for x in range(seconds):
    sys.stdout.write('.')
    sys.stdout.flush()
    time.sleep(1)
  

def text_indent(text, indent):
  panding = ' ' * indent
  #return '\n'.join(panding + re.sub(r'\t', ' ', line.strip()) for line in text.splitlines(True))
  return ''.join(panding + re.sub(r'\t', ' ', line) for line in text.splitlines(True))
  #return ''.join(panding + line for line in text.splitlines(True))


def exec_cmd(cmd):
  p = subprocess.Popen([cmd], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
  out_raw, err_raw = p.communicate()
  out = out_raw.decode('UTF-8')
  err = err_raw.decode('UTF-8')
  logger.debug("Running cmd:\n  " + str(cmd))
  logger.debug("Command output:\n  "+ str(out))
  return out, err


def exec_cmd_parallel(cmd):
  p = subprocess.Popen([cmd], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
  return p


def k8s_scale_0(kind, namespace, name):
  global helm_error
  global SCALETIMEOUT
  #result, err = exec_cmd("kubectl scale deploy --replicas=0 -n " + namespace +" "+ deploy.split(': ')[1])
  cmd = "kubectl scale " + kind + " --replicas=0 -n " + namespace +" "+ name
  p = subprocess.Popen([cmd], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
  #result, err = exec_cmd("kubectl scale " + kind + " --replicas=0 -n " + namespace +" "+ name)
  #if err:
  #  logger.error("File to Scale " + kind + " [" + name + "]")
  
  _continue = True

  check = 'start check pods'
  timeout = SCALETIMEOUT
  while check != '':
    result, err  = exec_cmd("kubectl get pod -n " + namespace + "|grep ^"+ name + " ")
    check = result.rstrip()
    printdot(1)
    timeout -= 1
    if timeout == 0:
      print('')
      logger.error("Timeout waiting for PODs of " + kind + " [" + name + "] to be terminated.")
      helm_error += 1
      _continue = False
      break
  print('')
  return _continue


def yamltodic(root, filepath):
  _dict = {}
  _file = open(filepath, 'r')
  _dict[root] = yaml.load(_file, Loader=yaml.FullLoader)
  _file.close()
  #logger.debug("yamltodic result:\n" + str(_dict) + "\n")
  return _dict


def yamltexttodic(root, text):
  _dict = {}
  _dict[root] = yaml.load(text, Loader=yaml.FullLoader)
  #logger.debug("yamltexttodic result:\n" + str(_dict) + "\n")
  return _dict


def dictoyamltext(dict):
  return yaml.dump(dict)
##### merge helm get values and values.yaml
def common_yamltexttodic(**kw):
  _dict = {}
  try:
    if kw.get('root') and kw.get('text'):
      _dict = yamltexttodic(kw.get('root'), kw.get('text'))
    elif kw.get('root') and kw.get('path'):
      _dict = yamltodic(kw.get('root'), kw.get('path'))
    else:
      _dict = yaml.load(kw['text'], Loader=yaml.FullLoader)
  except Exception as e:
    _dict = {}
  return _dict

def dictremoveolditems(mydict,keep_num):
  itemToDelete = list(reversed(sorted(mydict.keys())))[keep_num:]
  #print(str(itemToDelete))
  for item in itemToDelete:
    mydict.pop(item, None)
  
  logger.debug("Shorted dict is: " + str(mydict))
  return mydict

def update_helm_release_values(release, namespace, chart, chart_version, chart_file):
  # helm get values
  get_helm_values_cmd="helm get values -n " + namespace + " " + release + " -a"
  get_values, err=exec_cmd(get_helm_values_cmd)
  if err:
    logger.error("Error command: " + get_helm_values_cmd + ": " + err)
    exit(1)
    #dict_helm_get_value={}

  dict_helm_get_value=common_yamltexttodic(text=get_values)

  # helm show values
  #chart_file = os.path.join(CHARTS_FOLDER, chart + "-" + chart_version + CHART_FILE_SUFFIX)
  show_helm_values_cmd="helm show values " + chart_file
  show_values, err=exec_cmd(show_helm_values_cmd)
  if err:
    logger.error("Error command: " + show_helm_values_cmd + ": " + err)
    exit(1)
    #dict_helm_show_value={}

  dict_helm_show_value=common_yamltexttodic(text=show_values)

  # update dict
  if dict_helm_get_value and dict_helm_show_value != None:
    dict_helm_show_value.update(dict_helm_get_value)
  return dict_helm_show_value

def final_values(release, namespace, chart_name, chart_version, dpypkg_file, values_file, chart_file):

  # Does release exist?
  get_release_cmd="helm status " + "-n " + namespace + " " + release
  _, release_err=exec_cmd(get_release_cmd)
  if release_err:
    logger.debug("No such release: " + release + " ,set dict empty")
    dict_helm={}
  else:
    dict_helm=update_helm_release_values(release, namespace, chart_name, chart_version, chart_file)

  dpypkg_path = os.path.join(DEPLOYPACKAGE_FOLDER, dpypkg_file)
  helm_options, dict_dpypkg=render_values(dpypkg_path, VAR_FILE)

  if dict_dpypkg and dict_helm != None:
    dict_helm.update(dict_dpypkg)

  # Get the values portion and create output as values.yaml file
  #values_file = os.path.join(VALUES_FOLDER, STRING_JOINER.join(("values", release, chart_version)) + YAMM_FILE_SUFFIX)
  logger.debug("Start rendering: " + dpypkg_path + " to output: " + values_file + " with var file: " + VAR_FILE)
  with open(values_file, 'w') as f:
      yaml.dump(dict_helm, f, default_flow_style=False)
      f.truncate() 
  return helm_options


# Value None should be translated to '' so that Helm chart template can treat it as null. 
def out_finalize(var_value):
  if var_value is None:
    return ""   
  else:
    return var_value


def remove_quotes(file):
  with open(file, "r") as sources:
    lines = sources.readlines()
  with open(file+".noquotes", "w") as sources:
    for line in lines:
        sources.write(re.sub(r'"', '', line))


def system_check():
  print("Checking system ...")
  #Check kubectl and helm is installed
  out, err = exec_cmd("helm version")
  logger.debug(out)
  if err:
    logger.error("Helm is not installed in this node, make sure to run this scripts in the bootstrap node.")
    exit(1)

  out, err = exec_cmd("kubectl version")
  logger.debug(out)

  if err:
    if 'command not found' in err:
      logger.error("kubectl is not installed in this node, make sure to run this scripts in the bootstrap node.")
      exit(1)
    else:
      logger.warning("kubectl runs with warning: " + str(err))

def get_namespace():
  if not os.path.exists(DEFAULT_VAR_FILE):
    logger.error("Missing products-var file: " + DEFAULT_VAR_FILE)
    exit(1)

  var_dic = yamltodic('vars', DEFAULT_VAR_FILE)
  #if not var_dic['vars'].has_key('cms_namespace'):
  if 'cms_namespace' not in var_dic['vars']:
    logger.error("cms_namespace != set in products-var, fill in with proper namespace value and rerun.")
    exit(1)
  
  return var_dic['vars']['cms_namespace']


def label_nodes(label_map):

  logger.info("Start creating labels for nodes... ") 
  #TODO
  #Ensure all nodes mentioned in the yaml file exist first.
  
  nodes, err = exec_cmd("kubectl get node | grep -v NAME | awk '{print $1}'")
  nodes_list = nodes.splitlines()
  logger.debug("Cluster Nodes as below:\n" + str(nodes_list))
  if err:
    logger.error(err)
    exit(1)
  
  for node,labels in label_map.items():
    if node not in nodes_list:
      logger.error("Node [" + node + "] from matrix file != found in current cluster. Make sure the products-matrix file is configured properly.")
      exit(1)
    
    process_dict = {}
    for label in labels:
      cmd="kubectl label node "+ node + " " + label + "=enabled --overwrite=true"
      logger.info("Add label to node [" + node + "] with label [" + label + "=enabled]")
      cmd_proc = exec_cmd_parallel(cmd)
      if cmd_proc != None:
        process_dict[node + "%" + label] = cmd_proc

    for label,p in process_dict.items():
      out_raw, err_raw = p.communicate()
      out = out_raw.decode('UTF-8')
      err = err_raw.decode('UTF-8')
      if err:
        logger.error("Add label [" + label + "] failed with err: " + err)
        exit(1)
      else:
        logger.debug(out)
  

def parse_matrix(matrix):
  logger.info("Start parsing node matrix based on products-matrix file: " + matrix) 
  fullpath = os.path.join(SCRIPT_PATH, CONFIGBUNDLE, matrix)
  if not os.path.exists(matrix) and not os.path.exists(fullpath):
    logger.error("Matrix file " + matrix + " not exist, make sure the file is exist in bundle folder " + os.path.join(SCRIPT_PATH, CONFIGBUNDLE))
    exit(1)

  if not os.path.exists(matrix):
    matrix = fullpath

  matrix_dict = yamltodic('nodes', matrix)

  #Generate list for profiles and versions to pass to gen_deploy_list(profile_list) function
  profile_list = []
  for node in matrix_dict['nodes']:
    for profile in node['profiles']:
      if profile not in profile_list:
        profile_list.append(profile)

  '''
  profile_list is: [{'version': '9.0.0', 'name': 'cms'}, {'version': '9.0.0', 'name': 'cms-es-application'}, {'version': '9.0.0', 'name': 'cms-database'}]
  '''
  logger.debug("profile_list is: " + str(profile_list))

  #Generate map for node and labes mapping to pass to label_nodes(label_map) function
  label_map = {}
  for node in matrix_dict['nodes']:
    label_map[node['node']] = []
    for profile in node['profiles']:
      label_map[node['node']].append(profile['name'])

  '''
  label_map is: {'jacel-mdt-50-181': ['cms', 'cms-es-application'], 'jacel-mdt-50-183': ['cms', 'cms-es-application'], 'jacel-mdt-50-182': ['cms', 'cms-database'], 'jacel-mdt-50-184': ['cms', 'cms-es-application', 'cms-database']}
  '''
  logger.debug("label_map is: " + str(label_map))
  
  return profile_list, label_map


def parse_profile(profile):
  logger.info("Start parsing profile: " + profile)
  fullpath = os.path.join(SCRIPT_PATH, PROFILE_FOLDER, profile)
  if not os.path.exists(profile) and not os.path.exists(fullpath):
    logger.error("Profile " + profile + " not exist, make sure the file is exist in bundle profiles folder " + os.path.join(SCRIPT_PATH, PROFILE_FOLDER))
    exit(1)

  if not os.path.exists(profile):
    profile = fullpath
  
  return yamltodic('profile', profile)


def parse_dpypkg(dpypkg):
  logger.info("Start parsing deploypackage: " + dpypkg)
  fullpath = os.path.join(SCRIPT_PATH, DEPLOYPACKAGE_FOLDER, dpypkg)
  if not os.path.exists(fullpath):
    logger.error("Deploypackage " + dpypkg + " not exist, make sure the file [" + fullpath + "] is exist.")
    exit(1)

  return yamltodic('package', fullpath)


def gen_deploy_list(profile_list):
  logger.info("Start generating deploy-info-list from profiles ...")

  deploy_info_dict = {}
  sorted_deploy_info_list = []
  
  for profile in profile_list:
    if isinstance(profile, dict):
      profile_path = STRING_JOINER.join((PROFILE_FILE_PREFIX, profile['name'], profile['version'])) + YAMM_FILE_SUFFIX
    else:
      profile_path = profile
    logger.debug("profile_path is: " + profile_path)
    profile_dict = parse_profile(profile_path)

    for package in profile_dict['profile']['packages']:
      #example:
      #item: {'deploy-package': 'cms-adi-server', 'version': '9.0.1'}
      #deploypackage file path: cms-adi-server/dpypkg_cms-adi-server_9.0.1.yaml
      deploypackage = os.path.join(package['deploy-package'], STRING_JOINER.join((DPYPKG_FILE_PREFIX, package['deploy-package'], package['version'])) + YAMM_FILE_SUFFIX)
      dpypkg_dict = parse_dpypkg(deploypackage)
      
      #if dpypkg_dict['package']['deploy-package'].has_key('deployment_stage'):
      if 'deployment_stage' in dpypkg_dict['package']['deploy-package']:
        stage_id = dpypkg_dict['package']['deploy-package']['deployment_stage']
      else:
        stage_id = None
       
      if stage_id is None:
        stage_id = "last"
      
      _simplify_dict = {}
      _simplify_dict['name'] = deploypackage
      _simplify_dict['releases'] = dpypkg_dict['package']['deploy_list']
      _simplify_dict['chart'] = {}
      _simplify_dict['chart']['name'] = dpypkg_dict['package']['chart']['name']
      _simplify_dict['chart']['version'] = dpypkg_dict['package']['chart']['version']
      _simplify_dict['stage'] = stage_id

      #if not deploy_info_dict.has_key(stage_id):
      if stage_id not in deploy_info_dict:
        deploy_info_dict[stage_id] = []
      if _simplify_dict not in deploy_info_dict[stage_id]:
        deploy_info_dict[stage_id].append(_simplify_dict)

  logger.debug("deploy_info_dict as below: \n" + str(deploy_info_dict))
  
  keys = deploy_info_dict.keys() 
  logger.debug("keys: \n" + str(keys))
  #sorted(keys) 
  sorted_deploy_info_list = list(map(deploy_info_dict.get, sorted(keys)))
  logger.debug("sorted_deploy_info_list as below: \n" + str(sorted_deploy_info_list))

  return sorted_deploy_info_list

def backup_resource(namespace):
  backup_resource_dict = {}
  resources_dict = yamltodic("resources", BACKUP_RESOURCE_FILE)
  for key,resource_list in resources_dict['resources'].items():
    backup_resource_dict[key] = {}
    for resource in resource_list:
      #cmd = "kubectl get -o yaml -n " + namespace + " " + key + " " + resource + " |yq d - 'metadata.annotations.\"kubectl.kubernetes.io/last-applied-configuration\"'|yq d - 'metadata.creationTimestamp'|yq d - 'metadata.resourceVersion' 2>/dev/null | base64 -w 0"
      cmd = "kubectl get -o yaml -n " + namespace + " " + key + " " + resource + " |yq 'del(.metadata.annotations.\"kubectl.kubernetes.io/last-applied-configuration\")' - |yq 'del(.metadata.creationTimestamp)' - |yq 'del(.metadata.resourceVersion)' - 2>/dev/null | base64 -w 0"
      resource_content,err = exec_cmd(cmd)
      if err:
        logger.warning("Backup resource ["+key+":"+resource+"] failed: " + err)
      else:
        backup_resource_dict[key][resource] = resource_content
  
  return dictoyamltext(backup_resource_dict)

def backup_snapshot(namespace, secret_name, message):
  logger.info("Start backup helm release revision information before running helm install ...")

  # Delete the deprecated backup snapshot configmap
  exec_cmd("kubectl delete cm " + BACKUP_SNAPSHOT_CONFIGMAP_NAME + " -n " + namespace)

  snapshot, err = exec_cmd("helm list -a -n " + namespace)
  if err:
    logger.error("Listing helm release revision failed with err: " + err)
    exit(1)

  sts_replicas, err = exec_cmd("kubectl get sts -n " + namespace + " | grep -v cms-pg-")
  if err:
    logger.warning("Get statefulset replicas failed with err: " + err)
    #exit(1)

  deploy_replicas, err = exec_cmd("kubectl get deploy -n " + namespace + " | grep -v cms-pg-")
  if err:
    logger.warning("Get deployment replicas failed with err: " + err)
    #exit(1)

  resources = backup_resource(namespace)

  history_data = ""
  history_data_dict ={}

  secret_data, err = exec_cmd("kubectl get secret " + secret_name + " -o yaml -n " + namespace)
  if secret_data != '':
    data_dict = yamltexttodic('cm', secret_data)
    history_data_dict = dictremoveolditems(data_dict['cm']['data'], MAX_BACKUP_SNAPSHOT_NUM)
    logger.debug("Snapshots in history_data_dict as below:\n" + str(history_data_dict.keys()))

    for key,value in history_data_dict.items():
      history_data = history_data + '\n' + text_indent(key,2) + ': ' + value
      logger.debug("History data inside Backup Snapshot Secret:\n" + history_data)

  if err:
    logger.warning("Secret " + secret_name + " not exists, there is no backup histroy, it could be the first time deployment.")
  
  message = text_indent(message, 2)
  snapshot = text_indent(snapshot, 2)
  sts_replicas = text_indent(sts_replicas, 2)
  deploy_replicas = text_indent(deploy_replicas, 2)
  resources = text_indent(resources, 2)

  backupdata = """Backup Description: |
%s
Snapshot: |
%s
STS_Replicas: |
%s
DEPLOY_Replicas: |
%s
RESOURCES: |
%s
""" %(message,snapshot,sts_replicas,deploy_replicas,resources)

  base64_backupdata = base64.b64encode(backupdata.encode('ascii')).decode('UTF-8')

  #logger.debug(backupdata)
  
  cmd = """cat <<EOF | kubectl apply -f -
apiVersion: v1
data:
  snapshot-%s: %s
%s
kind: Secret
type: Opaque
metadata:
  name: %s
  namespace: %s
EOF
  """ %(CURRENTTIME, base64_backupdata, history_data, secret_name, namespace)
  logger.debug("Command for updating secret " + secret_name + ":\n" + cmd)

  out, err = exec_cmd(cmd)
  if err and "Warning" not in err:
    logger.error("Creating Secret " + secret_name + " failed with err: " + err)
    exit(1)
  logger.info("Snapshot [snapshot-" + CURRENTTIME + "] created: " + out)
  

def print_stages(deploy_list):
  stagesReview = ""
  for stage in deploy_list:
    for chart in stage:
      stagesReview = stagesReview + "  " + str(stage[0]['stage']) + "\t" + chart['chart']['name'] + "\n"
    stagesReview = stagesReview + "  ------------------------------------\n"

  logger.info("Deployment Stages Review:\n ========================================================\n" +
               stagesReview )


def install_charts(deploy_list, namespace, interval, force, deletepod):
  
  global helm_error

  # If not set with --force, need to fetch the current helm releases deployment status
  # And store the status into status_dict and passed to helm_install_chart() function
  # Example status in status_dict with key formatted <release_name>-<chart_name>-<chart_version>: <status>
  # {'cms-monitoring-cms-monitoring-9.0.0-29': 'deployed',	'cms-adi-server-cms-adi-server-9.0.0-36': 'deployed'}
  if not force:
    status, err = exec_cmd("helm list -a -o yaml -n " + namespace)
    deploy_status_dict = yamltexttodic('status', status)
    if err:
      logger.error("Failed to get helm chart releases status with err: " + err)
    for state in deploy_status_dict['status']:
      status_dict[state['name']+"-"+state['chart']] = state['status']
    logger.debug("status_dict is:\n" + str(status_dict))

  # Print the stage list for review
  print_stages(deploy_list)
  
  # Delete all completed jobs before running upgrade to avoid job conflict.
  #_, err = exec_cmd("kubectl get cronjob cms-delete-completed-jobs -n " + namespace)
  #if err:
  #  logger.debug("Failed to get helm chart releases status with err: " + err)
  #  logger.debug("It could be the first time for CMS deployment, not for upgrade")
  #else:
  #  _, err = exec_cmd("kubectl create job delete-completed-jobs-$(date +%Y%m%d%H%M%S) --from=cronjob/cms-delete-completed-jobs -n " + namespace)
  #  if err:
  #    logger.debug("Failed to trigger job with err: " + err)
  _, err = exec_cmd("kubectl delete jobs --field-selector status.successful=1 -n " + namespace)
  if err:
    logger.debug("Failed to delete completed jobs with err: " + err)
  
  # Trigger helm install for each release in the same stage at the same time.
  # And run stages in sequence. 
  for stage in deploy_list:
    logger.info("Start deploying Charts in stage [" + str(stage[0]['stage']) +"]")
    
    process_dict = {}
    
    #Init the stage_wait as 0
    #Each chart will return 1 by helm_install_chart() function if the deploy processed
    stage_wait= 0
    
    for chart in stage:
      dpypkg_file = chart['name']
      chart_name = chart['chart']['name']
      chart_version = chart['chart']['version']
      #helm_options is now get after values template renderred. Not in here any more.
      #helm_options = chart['helm_options']

      for release in chart['releases']:
        cmd_proc = helm_install_chart(release, namespace, chart_name, chart_version, dpypkg_file, force, deletepod, stage[0]['stage'])
        if cmd_proc != None:
          process_dict[release] = cmd_proc

    # Wait for all command to be executed and get the results
    for release,p in process_dict.items():
      out_raw, err_raw = p.communicate()
      # Ignore warning messages such as
      # W0617 09:03:21.005581    3519 warnings.go:67] admissionregistration.k8s.io/v1beta1 MutatingWebhookConfiguration is deprecated in v1.16+, 
      # unavailable in v1.22+; use admissionregistration.k8s.io/v1 MutatingWebhookConfiguration
      
      #if err and not re.match('W[0-9]{4} ', err):
      #  helm_error += 1
      #  logger.error("Helm Chart Release ["+release+"] deploy failed with error message:\n  " + err)
      #else:
      #  logger.debug("Helm Chart deploy successful with message:\n  " + out)
      #  stage_wait += 1
      out = out_raw.decode('UTF-8')
      err = err_raw.decode('UTF-8')

      error_info = ""
      if err:
        for line in err.split('\n'):
          if not re.match('W[0-9]{4} ', line) and "[debug]" not in line:
            error_info = "  " + line + "\n" + error_info
          else:
            logger.debug(line)

      if error_info.replace('\n', '').replace('\r', '').strip() != "":
        helm_error += 1
        logger.error("Helm Chart Release ["+release+"] deploy failed with error message:\n" + error_info)
      else:  
        logger.debug("Helm Chart Release ["+release+"] deploy successful with message:\n  " + out)
        stage_wait += 1

    if stage[0]['stage'] < ENFORCE_FAIL_STAGES and helm_error > 0 :
      logger.error("""
    ====================================
    CMS Deployment Failed with Error found!
    Check ERROR information from the log.
    ====================================
      """)
      exit(1)

    # Wait INTERVAL seconds before going to deploy charts in next stage
    if stage_wait > 0:
      logger.info("Pause "+str(interval)+" seconds waiting for chart releases in current stage [" + str(stage[0]['stage']) +"] to be deployed.")
      printdot(interval)
      print('')


def install_single_chart(release, namespace, chart, dpypkg_version, deletepod):
  #Always act with --force as true
  force = True
  
  # Get the latest updated .yaml file from the deploypackage folder if the dpypkg_version is 'latest'
  if dpypkg_version == 'latest':
    filename, err = exec_cmd("ls -rt " + os.path.join(DEPLOYPACKAGE_FOLDER, chart) + "| grep .yaml$|tail -1")
    if err or filename == '':
      logger.error("No deploypackage file for " + chart + " in folder " + os.path.join(DEPLOYPACKAGE_FOLDER, chart) + "\n  " + err)
      exit(1)
    dpypkg_file = os.path.join(chart, filename.rstrip())
  else:
    dpypkg_file = os.path.join(chart, STRING_JOINER.join((DPYPKG_FILE_PREFIX, chart, dpypkg_version)) + YAMM_FILE_SUFFIX)
  
  # Parse the deploypackage file to get chart_version
  chart_version = parse_dpypkg(dpypkg_file)['package']['chart']['version']

  p = helm_install_chart(release, namespace, chart, chart_version, dpypkg_file, force, deletepod, 100)
  if p != None:
    out_raw, err_raw = p.communicate()
    out = out_raw.decode('UTF-8')
    err = err_raw.decode('UTF-8')

    error_info = ""
    if err:
      for line in err.split('\n'):
        if not re.match('W[0-9]{4} ', line) and "[debug]" not in line:
          error_info = "  " + line + "\n" + error_info
        else:
          logger.debug(line)
    
    if error_info.replace('\n', '').replace('\r', '').strip() != "":
      logger.error("Helm Chart Release ["+release+"] deploy failed with error message:\n" + error_info)
    else:  
      logger.debug("Helm Chart Release ["+release+"] deploy successful with message:\n  " + out)

#Return None if no helm install ture happen
#Return process object if the helm install command executed
def helm_install_chart(release, namespace, chart, chart_version, dpypkg_file, force, deletepod, stage_num):
  
  logger.debug("helm install Release:" + release + " NS:" + namespace + " Chart:" + chart + " Version:" + chart_version + " Force:" + str(force) + " Delete-POD:" + str(deletepod))
  
  #global helm_error
  
  #Helm upgrade will take any change to the pods if there is no changes towards pod definition
  #Helm upgrade will update the pods based on the update strategy which might cause the upgrade blocked if there is no sufficient resources
  #With --force set to false, will bypass the chart upgrade if the same version is already in deployed status
  #With --delete-pod set to true, will scale the deployment/statefulset to 0 before running the helm upgrade to ensure the old pod is deleted inadvance so that we get enough resources
  
  #Init the boolen installchart as True to run helm install by default 
  installchart = True

  #With --force set to false, will bypass the chart upgrade if the same version is already in deployed status
  #if not force and status_dict.has_key('-'.join((release, chart, chart_version))) and stage_num > ENFORCE_REDEPLOY_STAGES:
  if not force and '-'.join((release, chart, chart_version)) in status_dict and stage_num > ENFORCE_REDEPLOY_STAGES:
    if status_dict['-'.join((release, chart, chart_version))] == 'deployed':
      installchart = False
      logger.info("Chart release ["+ '_'.join((release, chart, chart_version)) +"] is already deployed, bypass current upgrade step.")
      #return 0
      return None
  
  if installchart:
    logger.info("Deploying Chart release ["+ ' : '.join((release, chart, chart_version)) +"] ...")
    
    # To determine whether to continue the helm install or not based on kubectl scale command result
    _continue = True

    #Render the values file based on the deploypackage file
    #dpypkg_path = os.path.join(DEPLOYPACKAGE_FOLDER, dpypkg_file)
    values_file = os.path.join(VALUES_FOLDER, STRING_JOINER.join(("values", release, chart_version)) + YAMM_FILE_SUFFIX)
    chart_file = os.path.join(CHARTS_FOLDER, chart + "-" + chart_version + CHART_FILE_SUFFIX)
    helm_options=final_values(release, namespace, chart, chart_version, dpypkg_file, values_file, chart_file)
    if "--wait" not in helm_options and stage_num <= ENFORCE_WAIT_OPTION_STAGES:
      helm_options = helm_options + " --wait"
    if "--timeout " not in helm_options and stage_num <= ENFORCE_WAIT_OPTION_STAGES:
      helm_options = helm_options + " --timeout " + HELM_UPGRADE_TIMEOUT
    #print("helm_option is: " + helm_options)

    helm_value_set_options = ""
    helm_value_set_options = helm_set_replica_value(CMS_REPLICAS_DICT, release, namespace, helm_value_set_options)

    #With --delete-pod set to true, will scale the deployment/statefulset to 0 before running the helm upgrade to ensure the old pod is deleted inadvance so that we get enough resources
    if deletepod:
      deploy_resources, err = exec_cmd("helm get manifest -n " + namespace +" "+ release + " |grep -A 5 'kind: Deployment'|grep 'name: ' ")
      if err:
        logger.warning("helm chart release [" + release + "] not found, bypassing the --delete-pod option.")
      else:
        if deploy_resources != '':
          for deploy in deploy_resources.splitlines():
            logger.info("Deleting PODs by cmd: kubectl scale deploy --replicas=0 -n " + namespace +" "+ deploy.split(': ')[1])
            #result, err = exec_cmd("kubectl scale deploy --replicas=0 -n " + namespace +" "+ deploy.split(': ')[1])
            _continue = k8s_scale_0("deployment", namespace, deploy.split(': ')[1])

      sts_resources, err = exec_cmd("helm get manifest -n " + namespace +" "+ release + " |grep -A 5 'kind: StatefulSet'|grep 'name: ' ")
      if err:
        logger.warning("helm chart release [" + release + "] not found, bypassing the --delete-pod option.")
      else:
        if sts_resources != '':
          for sts in sts_resources.splitlines():
            logger.info("Deleting PODs by cmd: kubectl scale sts --replicas=0 -n " + namespace +" "+ sts.split(': ')[1])
            #result, err = exec_cmd("kubectl scale sts --replicas=0 -n " + namespace +" "+ sts.split(': ')[1])
            _continue = k8s_scale_0("statefulset", namespace, sts.split(': ')[1])
    
    if _continue:
      #Run the helm upgrade --install command
      #chart_file = os.path.join(CHARTS_FOLDER, chart + "-" + chart_version + CHART_FILE_SUFFIX)
      helm_cmd = "helm upgrade --install --debug -n " + namespace + " " + release + " --values=" + values_file + " " + helm_value_set_options + " " + chart_file + " " + helm_options
      #helm_cmd = ""
      logger.info("Running helm cmd to deploy [" + release + "]\n > " + helm_cmd)
      p = exec_cmd_parallel(helm_cmd)
      #if err:
      #  logger.error("Helm Chart deploy failed for [" + release + "] with error message:\n  " + err)
      #  helm_error += 1
      #  return 0
      #else:
      #  return 1
      return p
    else:
      return None

def helm_set_replica_value(replicas_field_dict, release, namespace, helm_set_value_option):
  #if replicas_field_dict['replicas'].has_key(release):
  if release in replicas_field_dict['replicas']:
    resources = replicas_field_dict['replicas'].get(release)
    for resource in resources:
      cmd = "kubectl get -o=jsonpath='{.status.replicas}' -n " + namespace + " " + resource['type'] + " " + resource['resource']
      logger.debug("Run kubectl to get replicas [" + resource['resource'] + "]")
      replicas, err = exec_cmd(cmd)
      if err:
        if "(NotFound):" in err:
          logger.debug("Warn: Fail to get replicas for [" + resource['resource'] + "] due to resource not exist\n  " + err)
        else:
          logger.error("Fail to get replicas for [" + resource['resource'] + "]\n  " + err)
          exit(1)
      elif str(replicas).replace('\n', '').replace('\r', '').strip() == "":
        helm_set_value_option = "--set " + resource['replica_account_field'] + "=0" + helm_set_value_option
      else:
        helm_set_value_option = "--set " + resource['replica_account_field'] + "=" + replicas + " " + helm_set_value_option
  
  logger.debug("helm_set_value_option for [" + release + "] is: " + helm_set_value_option.strip())
  return helm_set_value_option.strip()


def render_values(dpypkg, var):
  
  #LoggingUndefined = jinja2.make_logging_undefined(logger=logger,base=jinja2.Undefined)

  templates_path = os.path.dirname(dpypkg)
  template_name = os.path.basename(dpypkg) + ".noquotes"

  DEFAULT_VAR_PARENT_KEY = re.sub(r'-', '_', os.path.splitext(DEFAULT_VAR_FILE_NAME)[0])
  default_var_dict = yamltodic(DEFAULT_VAR_PARENT_KEY, DEFAULT_VAR_FILE)
  
  if DEFAULT_VAR_FILE != var:
    VAR_PARENT_KEY = re.sub(r'-', '_', os.path.splitext(os.path.basename(var))[0])
    var_dict = yamltodic(VAR_PARENT_KEY, var)
    default_var_dict.update(var_dict)

  #VAR_PARENT_KEY = re.sub(r'-', '_', os.path.splitext(os.path.basename(var))[0])
  #var_dict = yamltodic(VAR_PARENT_KEY, var)

  # Remove all double quotes from the deploypackage yaml file before rendering
  remove_quotes(dpypkg)

  # Start rendering
  j2_env = jinja2.Environment(
      loader=jinja2.FileSystemLoader(templates_path, encoding='utf-8'),
      trim_blocks=True, 
      lstrip_blocks=True,
      finalize=out_finalize,
      #undefined=jinja2.DebugUndefined,
      undefined=jinja2.StrictUndefined,
      autoescape=False
      )
  template = j2_env.get_template(template_name)

  # The render output is in YAML format
  rendered_template = template.render(default_var_dict)
  
  logger.debug("Rendered_template as: \n" + rendered_template + "\n") 
  dic_dpypkg_product=yaml.load(rendered_template, Loader=yaml.FullLoader)['values']

  #if yaml.load(rendered_template, Loader=yaml.FullLoader).has_key('helm_options'):
  if 'helm_options' in yaml.load(rendered_template, Loader=yaml.FullLoader):
    helm_options = yaml.load(rendered_template, Loader=yaml.FullLoader)['helm_options']
  else:
    helm_options = ''
  return helm_options, dic_dpypkg_product

def delete_charts(deploy_list, namespace):
  # If not set with --force, need to fetch the current helm releases deployment status
  # And store the status into status_dict and passed to helm_install_chart() function
  # Example status in status_dict with key formatted <release_name>-<chart_name>-<chart_version>: <status>
  # {'cms-monitoring-cms-monitoring-9.0.0-29': 'deployed',	'cms-adi-server-cms-adi-server-9.0.0-36': 'deployed'}

  global helm_error

  # Print the stage list for review
  print_stages(deploy_list)

  # Reverse the stages
  deploy_list.reverse()
  #logger.debug("Reversed deploy_list as below: \n" + str(deploy_list))
  
  # Trigger helm install for each release in the same stage at the same time.
  # And run stages in sequence. 
  for stage in deploy_list:
    logger.info("Start deleting Chart Releases in stage [" + str(stage[0]['stage']) +"]")
    
    process_list = []

    for chart in stage:
      for release in chart['releases']:
        logger.info("Running helm cmd to delete [" + release + "]\n > helm delete -n " + namespace + " " + release)
        cmd_proc = exec_cmd_parallel("helm delete -n " + namespace + " " + release)
        process_list.append(cmd_proc)
    
    # Wait for all command to be executed and get the results
    for p in process_list:
      out_raw, err_raw = p.communicate()
      out = out_raw.decode('UTF-8')
      err = err_raw.decode('UTF-8')
      if err and not re.match('W[0-9]{4} ', err):
        logger.error("Helm Chart delete failed for with error message:\n  " + err)
        helm_error += 1
      else:
        logger.info(out)
        #time.sleep(2)

def parse_chart_revision(revisions):
  global rollback_target_version
  logger.debug("Start parsing helm chart revisions")
  revision_dict = {}
  for line in revisions.split('\n'):
    if line.startswith("NAME "):
      continue
    elif line != '':
      if line.split()[0] == "cms-product-version":
        rollback_target_version = line.split()[9]
      # key:value = release_name:release_revision
      #logger.debug("handling: " + line)
      revision_dict[line.split()[0]]=line.split()[2]
  #logger.debug("Get helm chart revisions dict:\n" + str(revision_dict))    
  return revision_dict

def parse_k8s_replicas(k8s_replicas):
  logger.debug("Start parsing resource replicas")
  replicas_dict = {}
  for line in k8s_replicas.split('\n'):
    if line.startswith("NAME "):
      continue
    elif line != '':
      # key:value = resource_name:replicas_amount
      #logger.debug("handling: " + line)
      replicas_dict[line.split()[0]]=line.split()[1].split('/')[1]
  logger.debug("Get K8S resource replicas dict:\n" + str(replicas_dict))    
  return replicas_dict

def parse_current_helm_list():
  logger.info("Get current helm chart releases revisions")
  logger.debug("Running helm cmd to get current helm chart releases revisions\n > helm list -a -n " + namespace)
  current_revisions,err = exec_cmd("helm list -a -n " + namespace)
  if err:
    logger.error("Failed to get current helm chart releases revisions!")
    exit(1)
  
  logger.debug("Current helm chart releases revisions:\n" + current_revisions )
  return parse_chart_revision(current_revisions)

def parse_backup_snapshot(snapshot_name):
  # Fetch latest chart releases revision by helm list command, save the result to dict_latest_revision with release_name and release_revision
  # If snapshot_name is empty, get the latest backup snapshot from cm BACKUP_SNAPSHOT_SECRET_NAME (cms-deployment-backup-snapshot)
  # Else get the specified backup snapshot from cm
  # Save the result to dict_target_revision with release_name and release_revision
  logger.debug("Start parsing snapshot: " + snapshot_name )

  backup_snapshots,err = exec_cmd("kubectl get secret -o yaml -n " + namespace + " " + BACKUP_SNAPSHOT_SECRET_NAME)
  if err:
    logger.error("Failed to get helm chart releases revisions from backup snapshot secret!")
    exit(1)

  snapshots_dict = yamltexttodic("snapshots", backup_snapshots)

  if snapshot_name != '':
    logger.info("Get helm chart releases revision from " + snapshot_name + " in Secret [" + BACKUP_SNAPSHOT_SECRET_NAME +"]")
    wanted_snapshot_dict = yamltexttodic("wanted", base64.b64decode(snapshots_dict['snapshots']['data'].get(snapshot_name)))
    revision_dict = parse_chart_revision(wanted_snapshot_dict['wanted']['Snapshot'])
    sts_replicas_dict = parse_k8s_replicas(wanted_snapshot_dict['wanted']['STS_Replicas'])
    deploy_replicas_dict = parse_k8s_replicas(wanted_snapshot_dict['wanted']['DEPLOY_Replicas'])
    resources_dict = yamltexttodic("RESOURCES",wanted_snapshot_dict['wanted']['RESOURCES'])['RESOURCES']
    return revision_dict,sts_replicas_dict,deploy_replicas_dict,resources_dict
  
  else:
    logger.info("Get last helm chart releases revision from Secret [" + BACKUP_SNAPSHOT_SECRET_NAME +"]")
    logger.info("snapshots are: "+str(list(reversed(sorted(snapshots_dict['snapshots']['data'].keys())))))
    latest_snapshot = list(reversed(sorted(snapshots_dict['snapshots']['data'].keys())))[0]
    logger.info("latest_snapshot name is: " + str(latest_snapshot))
    latest_snapshot_dict = yamltexttodic("latest", base64.b64decode(snapshots_dict['snapshots']['data'].get(latest_snapshot))) 
    revision_dict = parse_chart_revision(latest_snapshot_dict['latest']['Snapshot'])
    sts_replicas_dict = parse_k8s_replicas(latest_snapshot_dict['latest']['STS_Replicas'])
    deploy_replicas_dict = parse_k8s_replicas(latest_snapshot_dict['latest']['DEPLOY_Replicas'])    
    resources_dict = yamltexttodic("RESOURCES",latest_snapshot_dict['latest']['RESOURCES'])['RESOURCES']
    return revision_dict,sts_replicas_dict,deploy_replicas_dict,resources_dict

def restore_replicas(dict_sts_replicas, dict_deploy_replicas):
  global helm_error
  process_dict = {}
  for sts,replica in dict_sts_replicas.items():
    cmd = "kubectl scale sts -n " + namespace + " " + sts + " --replicas=" + replica
    logger.debug("Running kubectl cmd to scale [" + sts + "] with replicas (" + replica + ")\n > "+cmd)
    cmd_proc = exec_cmd_parallel(cmd)
    if cmd_proc != None:
      process_dict[sts] = cmd_proc    
  for deploy,replica in dict_deploy_replicas.items():
    cmd = "kubectl scale deploy -n " + namespace + " " + deploy + " --replicas=" + replica
    logger.debug("Running kubectl cmd to scale [" + deploy + "] with replicas (" + replica + ")\n > "+cmd)
    cmd_proc = exec_cmd_parallel(cmd)
    if cmd_proc != None:
      process_dict[deploy] = cmd_proc
  
  # Wait for all command to be executed and get the results
  for resource,p in process_dict.items():
    out_raw, err_raw = p.communicate()
    out = out_raw.decode('UTF-8')
    err = err_raw.decode('UTF-8')
    if err:
      helm_error += 1
      logger.error("Kubectl scale failed for ["+resource+"] with error message:\n  " + err)

def restore_resources(dict_resources):
  global helm_error
  process_dict = {}

  for kind,resources in dict_resources.items():
    # When resources dict != empty
    if resources:
      logger.info("Restoring CMS "+ kind + " resources")
      for resource,content in resources.items():
        cmd = "echo " + content + " | base64 -d | kubectl apply -f -"
        logger.debug("Running kubectl cmd to restore [" + kind + ":" + resource + "]\n > "+cmd)
        cmd_proc = exec_cmd_parallel(cmd)
        if cmd_proc != None:
          process_dict[resource] = cmd_proc          
  
  # Wait for all command to be executed and get the results
  for resource,p in process_dict.items():
    out_raw, err_raw = p.communicate()
    out = out_raw.decode('UTF-8')
    err = err_raw.decode('UTF-8')
    if err and not re.match('Warning: ', err):
      helm_error += 1
      logger.error("Restore resource failed for ["+resource+"] with error message:\n  " + err)

def rollback_charts(deploy_list, namespace, interval, snapshot_name):
  # With dict_current_revision and dict_target_revision, compare each release current revision and target revision, 
  # rollback to target revision once the current revision is different.
  
  global helm_error
  helm_option = " --wait --timeout " + HELM_ROLLBACK_TIMEOUT
  # Print the stage list for review
  print_stages(deploy_list)

  dict_current_revisions = parse_current_helm_list()
  dict_target_revisions,dict_sts_replicas,dict_deploy_replicas,dict_resources = parse_backup_snapshot(snapshot_name)

  logger.info("Rollback to target CMS version: " + rollback_target_version)

  # Pre-Rollback special steps
  
  # Delete all completed jobs before running upgrade to avoid job conflict.
  _, err = exec_cmd("kubectl delete jobs --field-selector status.successful=1 -n " + namespace)
  if err:
    logger.debug("Failed to delete completed jobs with err: " + err)


  if rollback_target_version == "9.0.0":
    logger.debug("=====================================================================")
    logger.debug("Pre-Rollback special steps for target version 9.0.0")
    
    logger.debug("Running kubectl cmd to delete cms-haproxy clusterRoleBinding resource")
    _, err = exec_cmd("kubectl delete clusterrolebinding cms-haproxy")
    if err:
      logger.debug("Faile to delete clusterrolebinding cms-haproxy with error message:\n  " + err)

    #if dict_current_revisions.has_key('cms-kibana'):
    if 'cms-kibana' in dict_current_revisions:
      logger.debug("Bypassing rollback for cms-kibana by setting target_revision as current_revision.")
      dict_target_revisions['cms-kibana'] = dict_current_revisions['cms-kibana']
  
    logger.debug("Stop cms-etcd service")
    _, err = exec_cmd("kubectl scale sts cms-etcd --replicas=0 -n " + namespace)
    if err:
      logger.debug("Faile to scale statefulset cms-etcd:\n  " + err)
    logger.debug("=====================================================================")
    
  logger.debug("Get current helm chart revisions dict:\n" + str(dict_current_revisions))  
  logger.debug("Get target helm chart revisions dict:\n" + str(dict_target_revisions))  
  logger.debug("Get statefulset replicas dict:\n" + str(dict_sts_replicas))  
  logger.debug("Get deployment replicas dict:\n" + str(dict_deploy_replicas))  
  logger.debug("Get resources dict:\n" + str(dict_resources))

  
  # Trigger helm install for each release in the same stage at the same time.
  # And run stages in sequence. 
  for stage in deploy_list:
    logger.info("Start rolling back Chart Releases in stage [" + str(stage[0]['stage']) +"]")
    
    stage_wait= 0

    #process_list = []
    process_dict = {}

    for chart in stage:
      for release in chart['releases']:
        current_revision = str(dict_current_revisions.get(release))
        target_revision = str(dict_target_revisions.get(release))
        logging.debug("Release [" + release + "] revision: current (" + current_revision + "), target (" + target_revision + ")")
        #if dict_current_revisions.has_key(release) and current_revision != target_revision:
        if release in dict_current_revisions and current_revision != target_revision:
          # when target_revision exist, rollback to that revision
          if target_revision != 'None':
            cmd = "helm rollback --debug -n " + namespace + " " + release + " " + target_revision + helm_option
            logger.info("Running helm cmd to rollback [" + release + "] to revision (" + target_revision + ")\n > " + cmd)
            cmd_proc = exec_cmd_parallel(cmd)
            #process_list.append(cmd_proc)
            if cmd_proc != None:
              process_dict[release] = cmd_proc
          # when target_revision not exist, means this release did not exist in last version, we will delete this release as rollback
          else:
            logger.info("Running helm cmd to delete [" + release + "] as rollback\n > helm delete -n " + namespace + " " + release)
            cmd_proc = exec_cmd_parallel("helm delete -n " + namespace + " " + release + " --timeout " + HELM_ROLLBACK_TIMEOUT)
            if cmd_proc != None:
              process_dict[release] = cmd_proc
        else:
          logger.info("Bypass rollback for [" + release + "] as current revision is same as target revision")
        
    # Wait for all command to be executed and get the results
    for release,p in process_dict.items():
      out_raw, err_raw = p.communicate()
      out = out_raw.decode('UTF-8')
      err = err_raw.decode('UTF-8')
      # Ignore warning messages such as
      # W0617 09:03:21.005581    3519 warnings.go:67] admissionregistration.k8s.io/v1beta1 MutatingWebhookConfiguration is deprecated in v1.16+, 
      # unavailable in v1.22+; use admissionregistration.k8s.io/v1 MutatingWebhookConfiguration
      error_info = ""
      if err:
        for line in err.split('\n'):
          if not re.match('W[0-9]{4} ', line) and "[debug]" not in line:
            error_info = "  " + line + "\n" + error_info
          else:
            logger.debug(line)

      if error_info.replace('\n', '').replace('\r', '').strip() != "":
        helm_error += 1
        logger.error("Helm Chart Release ["+release+"] rollback failed with error message:\n" + error_info)
      else:  
        logger.debug("Helm Chart Release ["+release+"] rollback successful with message:\n  " + out)
        stage_wait += 1

    if helm_error > 0:
      break
    elif stage_wait > 0:
      logger.info("Pause "+str(interval)+" seconds waiting for chart releases in current stage [" + str(stage[0]['stage']) +"] to be rollback.")
      printdot(interval)
      print('')      
  
  # Post-Rollback special steps
  # After rollback to CMS 9.0.0
  if helm_error == 0 and rollback_target_version == "9.0.0":
    logger.debug("=====================================================================")
    logger.debug("Post-Rollback special steps for target version 9.0.0")
    
    logger.debug("Delete cms-cli pod to enforce cms-cli restart to finish the rollback")
    _, err = exec_cmd("kubectl delete pod cms-cli-0 -n " + namespace)
    if err:
      logger.debug("Faile to delete cms-cli-0 pod:\n  " + err)
    
    logger.debug("Clean cms-cli dummy daemonset")
    _, err = exec_cmd("kubectl delete ds cms-cli -n " + namespace)
    if err:
      logger.debug("Faile to delete daemonset cms-cli:\n  " + err)
    
    logger.debug("=====================================================================")
     
  # Set replicas to align with the amount from the backup snapshot data.
  if helm_error == 0:
    logger.info("Set replicas to align with the amount from the backup snapshot data")
    restore_replicas(dict_sts_replicas, dict_deploy_replicas)
  
  # Restore backup resources such as configmaps and secrets
  if helm_error == 0:
    restore_resources(dict_resources)
  
if __name__ == "__main__":

  #################################
  # Parse the scripts arguments
  #################################
  usage = """
  %(prog)s [[-mx PRODUCT_MATRIX | -p PROFILE] [-i INTERVAL] [-f] [-m BACKUP_MESSAGE] | -n CHART_RELEASE_NAME -c CHART [-v DEPLOY_PACKAGE_VERSION]] [--delete-pod] [--debug] [--var VAR_FILE_NAME] | [--reset PRODUCT_MATRIX] | [--rollback PRODUCT_MATRIX [SNAPSHOT_NAME]]
    
    Examples:
      python install.py -mx products-matrix-9.0.0.yaml -i 180 -f --debug -m "Backup for 1st update."
      python install.py -p prof_cms-common_9.0.0.yaml -i 180 -f --debug -m "Backup for 1st update."
      python install.py -n cms-workflow -c cms-workflow -v 9.0.0
      python install.py --reset products-matrix-9.0.0.yaml
      python install.py --rollback products-matrix-9.0.0.yaml
  """

  parser = argparse.ArgumentParser(
    description='Install CMS into Kubernetes Cluster with CMS Helm Charts.', 
    usage=usage)
  exgroup = parser.add_mutually_exclusive_group(required=True)
  exgroup.add_argument("--matrix", "-mx", dest='matrix', metavar='PRODUCT_MATRIX', help="Specify the matrix file that includes the nodes and profiles mappings.")
  exgroup.add_argument("--profile", "-p", metavar='PROFILE', help="Specify the profile that includes the deploypackage list.")
  exgroup.add_argument("--name", "-n", metavar='CHART_RELEASE_NAME', help="Specify the helm chart release name to be installed.")
  exgroup.add_argument("--reset", metavar='PRODUCT_MATRIX', help="Delete all deployed CMS charts based on the specific products-matrix file.")
  exgroup.add_argument("--rollback", metavar='PRODUCT_MATRIX', help="rollback all deployed CMS charts based on the specific products-matrix file.")
  parser.add_argument("--chart", "-c", help="Specify the helm chart name to be installed.")
  parser.add_argument("--version", "-v", metavar='DEPLOY_PACKAGE_VERSION', help="Specify the version of deploypackage to be used, e.g. 9.0.0. Latest version within the deploypackages folder will be choosn if not specified.")
  parser.add_argument("--interval", "-i", type=int, default=DEFAULT_STAGE_INTERVAL, help="Time inteval between stages, default is " + str(DEFAULT_STAGE_INTERVAL) + ", unit is second.")
  parser.add_argument("--force", "-f", action="store_true", default=False, help="if set, will re-deploy the chart even when the release with same chart version is already in deployed status.")
  parser.add_argument("--delete-pod", dest='deletepod', action="store_true", default=False, help="Delete the exist pod by scaling the deployed deployment or statefulset with replicas 0 before updating each component. This will result in bypassing the rollingupdate upgrade strategy.")
  parser.add_argument("--message", "-m", metavar='BACKUP_MESSAGE', default="Revision Information Backup",help="Optional, message as description in the Chart release revision backup snapshot.")
  parser.add_argument("--debug", action="store_true", default=False, help="Enable debug logs.")
  parser.add_argument("--var", metavar='VAR_FILE_NAME', default="products-var.yaml",help="Optional, to pass the variable file name, which should be located in ConfigBundle folder. Default is products-var.yaml.")
  parser.add_argument("--snapshot-name", "-sn", dest='snapshotname', metavar='SNAPSHOT_NAME', default="",help="Optional, the specific snapshot name in Secret cms-deployment-backup-snapshot to rollback. Leave this empty to use the latest backup snapshot.")
  parser.add_argument("--without-node-label", dest='notlabelnodes', action="store_true", default=False, help="Not to label the nodes.")
  parser.add_argument("--no-db",dest='nodb',action="store_true",default=False,help="Optional, specify to bypass database rollback.")
  parser.add_argument("--db-types", dest='dbtypes', default="all", help="Optional, specify the database types to bypass during rollback, default:all, available:all,epg,mm(metadata-manager),workflow, can specify multiple types splited by comma. Use with --no-db.")
  parser.add_argument("--no-es",dest='noes',action="store_true",default=False,help="Optional, specify to bypass elasticsearch rollback.(kibana willbe bypassed automatically if --no-es specified.)")
    
  
  args = parser.parse_args()
  MATRIX_NAME = args.matrix
  PROFILE_NAME = args.profile
  VAR_FILE_NAME = args.var
  RELEASE_NAME = args.name
  CHART_NAME = args.chart
  DPYPKG_VERSION = args.version
  STAGE_INTERVAL = args.interval
  FORCE_DEPLOY = args.force
  DELETE_POD_ENABLE = args.deletepod
  BACKUP_MESSAGE = args.message
  DEBUG_ENABLE = args.debug
  RESET_MATRIX = args.reset
  ROLLBACK_MATRIX = args.rollback
  SNAPSHOT_NAME = args.snapshotname
  NOT_LABEL_NODES = args.notlabelnodes
  
  NO_DB_ROLLBACK = args.nodb
  NO_ES_ROLLBACK = args.noes
  NO_DB_ROLLBACK_TYPES = args.dbtypes
   
  if RELEASE_NAME != None and CHART_NAME is None:
    parser.error("--chart should not be empty when using --name.")
  if CHART_NAME != None and RELEASE_NAME is None:
    parser.error("--name should not be empty when using --chart.")
  if DPYPKG_VERSION != None and CHART_NAME is None and RELEASE_NAME is None:
    parser.error("--version only can be used with --name.")
  

  #################################
  # Variables
  #################################
  CURRENTTIME = datetime.now().strftime("%Y%m%d%H%M%S")
  SCRIPT_PATH = os.path.dirname(os.path.abspath(__file__))
  PROFILE_FOLDER = os.path.join(SCRIPT_PATH, CONFIGBUNDLE, PROFILE_FOLDER_NAME)
  DEPLOYPACKAGE_FOLDER = os.path.join(SCRIPT_PATH, CONFIGBUNDLE, DEPLOYPACKAGE_FOLDER_NAME)
  VAR_FILE = os.path.join(SCRIPT_PATH, CONFIGBUNDLE, VAR_FILE_NAME)
  #NODE_MATRIX_FILE = os.path.join(SCRIPT_PATH, CONFIGBUNDLE, "products-matrix.yaml")


  CHARTS_FOLDER = os.path.join(SCRIPT_PATH,"charts") 
  VALUES_FOLDER = os.path.join(SCRIPT_PATH,"charts","values") 
  DEFAULT_VAR_FILE = os.path.join(SCRIPT_PATH, CONFIGBUNDLE, DEFAULT_VAR_FILE_NAME)

  CMS_REPLICAS_DICT = yamltodic("replicas", REPLICAS_FILE)
  
  #################################
  # Logs settings
  #################################
  LOG_FOLDER = os.path.join(SCRIPT_PATH, "logs")
  #LOG_FILE = os.path.join(SCRIPT_PATH, LOG_FOLDER, "cms-install."+CURRENTTIME+".log")
  LOG_FILE = os.path.join(SCRIPT_PATH, LOG_FOLDER, "cms-install.log")

  if not os.path.exists(LOG_FOLDER):
    os.makedirs(LOG_FOLDER)
  
  logFormatter = '%(asctime)s - [%(levelname)s] - %(message)s'
  if DEBUG_ENABLE :
    logLevel = logging.DEBUG
  else:
    logLevel = logging.INFO

  logging.basicConfig(
    filename=LOG_FILE,
    #level=logLevel,
    level=logging.DEBUG,
    format=logFormatter
    #format='%(asctime)s | %(name)s | %(levelname)s | %(message)s'
    )
  
  logger = logging.getLogger()

  handler = logging.StreamHandler(sys.stdout)
  handler.setLevel(logLevel)
  formatter = logging.Formatter(logFormatter)
  handler.setFormatter(formatter)
  logger.addHandler(handler)

  # Define signal to handle Ctrl+C
  main_sigint_handler = signal.signal(signal.SIGINT, signal.SIG_IGN)
  signal.signal(signal.SIGINT, main_sigint_handler)
  
  #################################
  # Start installation
  #################################
  if RESET_MATRIX is None and ROLLBACK_MATRIX is None: 
    logger.info("""
    ====================================
           CMS Deployment Started
    ====================================
    """)
  elif RESET_MATRIX != None:
    logger.info("""
    ====================================
              CMS Reset Started
    ====================================
    """)
  else:
    logger.info("""
    ====================================
             CMS Rollback Started
    ====================================
    """)
  
  #status_dict to store helm chart releases status
  status_dict = {}

  logger.debug("Log file: " + LOG_FILE)
  logger.debug("Charts in folder: " + CHARTS_FOLDER)
  logger.debug("Values in folder: " + VALUES_FOLDER)
  logger.debug("Proflies in folder: " + PROFILE_FOLDER)
  logger.debug("Deploypackages in folder: " + DEPLOYPACKAGE_FOLDER)
  logger.debug("Variable file: " + VAR_FILE)
  if MATRIX_NAME != None:
    logger.debug(" Deploying with Matrix file: " + MATRIX_NAME)
  if PROFILE_NAME != None:
    logger.debug(" Deploying with Profile file: " + PROFILE_NAME)
  if RELEASE_NAME != None:
    logger.debug("Deploying release name: " + RELEASE_NAME)  
    logger.debug("Deploying with Chart name: " + CHART_NAME)  
    if DPYPKG_VERSION != None:
      logger.debug("Deploying with deploypackage version: " + DPYPKG_VERSION)  


  if not os.path.exists(VALUES_FOLDER):
    os.makedirs(VALUES_FOLDER)

  system_check()

  namespace = get_namespace()
  logger.info("Namespace is: " + namespace)

  try:
    if MATRIX_NAME != None:
      profile_list, label_map = parse_matrix(MATRIX_NAME)
      if NOT_LABEL_NODES:
        logger.info("Bypass Nodes labeling.")
      else:
        label_nodes(label_map)
      deploy_list = gen_deploy_list(profile_list)
      backup_snapshot(namespace, BACKUP_SNAPSHOT_SECRET_NAME, BACKUP_MESSAGE)
      install_charts(deploy_list, namespace, STAGE_INTERVAL, FORCE_DEPLOY, DELETE_POD_ENABLE)
  
    if PROFILE_NAME != None:
      profile_list = [PROFILE_NAME]
      deploy_list = gen_deploy_list(profile_list)
      backup_snapshot(namespace, BACKUP_SNAPSHOT_SECRET_NAME, BACKUP_MESSAGE)
      install_charts(deploy_list, namespace, STAGE_INTERVAL, FORCE_DEPLOY, DELETE_POD_ENABLE)
    
    if RELEASE_NAME != None:
      if DPYPKG_VERSION is None:
        DPYPKG_VERSION = "latest"
      install_single_chart(RELEASE_NAME, namespace, CHART_NAME, DPYPKG_VERSION, DELETE_POD_ENABLE)
  
    if RESET_MATRIX != None:
      profile_list, label_map = parse_matrix(RESET_MATRIX)
      deploy_list = gen_deploy_list(profile_list)
      delete_charts(deploy_list, namespace)
  
    if ROLLBACK_MATRIX != None:
        bypass_db_list = []
        bypass_es_list = []
	
        if NO_DB_ROLLBACK:
          if NO_DB_ROLLBACK_TYPES == 'all':
            bypass_db_list.append("cms-pg-epg")
            bypass_db_list.append("cms-pg-metadata-manager")
            bypass_db_list.append("cms-pg-workflow")
          else:
            bypass_dbtypes = NO_DB_ROLLBACK_TYPES.split(",")
            for item in bypass_dbtypes:
              if item == "epg":
                bypass_db_list.append("cms-pg-epg")
              elif item == "mm":
                bypass_db_list.append("cms-pg-metadata-manager")
              elif item == "workflow":
                bypass_db_list.append("cms-pg-workflow")
              elif item == "all":
                bypass_db_list.append("cms-pg-epg")
                bypass_db_list.append("cms-pg-metadata-manager")
                bypass_db_list.append("cms-pg-workflow")
              else:
                pass  #not match any available type, ignore
        if NO_ES_ROLLBACK:
          bypass_es_list.append("cms-es-application")
          bypass_es_list.append("cms-kibana")

        logger.info("----------------------------------------------------------")
        logger.info("rollback matrix: "+ROLLBACK_MATRIX )
        logger.info("rollback bypass db: "+(NO_DB_ROLLBACK_TYPES+", bypass_db_list: "+str(bypass_db_list) if NO_DB_ROLLBACK else "no"))
        logger.info("rollback bypass es: "+("all, bypass_es_list: "+str(bypass_es_list) if NO_ES_ROLLBACK else "no"))
        logger.info("----------------------------------------------------------")

        profile_list = []
        _profile_list, label_map = parse_matrix(ROLLBACK_MATRIX)
        for _index, _item in enumerate(_profile_list):
          if _item["name"] in bypass_db_list or _item["name"] in bypass_es_list:
            logger.info( "**** Rollback of profile:"+ _item["name"]+" will be bypassed ****")
          else:
            profile_list.append(_item)

        deploy_list = []
        _deploy_list = gen_deploy_list(profile_list)
        for _stage_index, _stage in enumerate(_deploy_list):
          _stage_deployes = []
          for _deploy_index, _deploy in enumerate(_stage):
            _releases = []
            for _release_index, _release in enumerate(_deploy['releases']):
              if _release in bypass_db_list or _release in bypass_es_list:
                logger.info( "**** Rollback of deploypackage:"+ _release+" will be bypassed ****")
              else:
                _releases.append(_release)
            if len(_releases) > 0:
              _deploy['releases'] = _releases
              _stage_deployes.append(_deploy)
				
          deploy_list.append(_stage_deployes)

        rollback_charts(deploy_list, namespace, STAGE_INTERVAL, SNAPSHOT_NAME)

    if RESET_MATRIX is None and ROLLBACK_MATRIX is None:
      if helm_error == 0:
        logger.info("""
    ====================================
          CMS Deployment Completed
    ====================================
        """)
      else:
        logger.error("""
    ====================================
    CMS Deployment Failed with Error found!
    Check ERROR information from the log.
    ====================================
        """)
        exit(1)
    elif RESET_MATRIX != None:
      if helm_error == 0:
        logger.info("""
    ====================================
       ALL CMS Chart Releases Deleted
    ====================================
        """)
      else:
        logger.error("""
    ====================================
    CMS Reset Failed with Error found!
    Check ERROR information from the log.
    ====================================
        """)
        exit(1)
    else:
      if helm_error == 0:
        logger.info("""
    ====================================
       CMS Rollback Completed
    ====================================
        """)
      else:
        logger.error("""
    ====================================
    CMS Rollback Failed with Error found!
    Check ERROR information from the log.
    ====================================
        """)
        exit(1)
  except KeyboardInterrupt:
    logger.warning("Process stopped by user!")
    sys.exit(127)

