#!/bin/bash
set +H
function_help(){
	printf "Help:"
	printf "    -? Help            Prints usage.\n"
	printf "\nMigrate postgres.\n"
	printf "    -h Source Database IP                           [Required] IP of CMS database which migrate from. It should be db1 or db2 node ip in cms7.2 or cms8.0 \n"
    printf "    -U Database super user name                     Super user name of source CMS database. If not specified, the defalut value(postgres) will be used. \n"
    printf "    -P Password of database super user              Super user password of source CMS database. If not specified, the defalut value(n2bbsys)will be used.\n"
    printf "    -u Database CMS user name                       CMS user name of source CMS database. If not specified, the defalut value(wfs) will be used. \n"
    printf "    -p Password of database CMS user                CMS user password of source CMS database. If not specified, the defalut value (wfs$1234) will be used.\n"
    printf "    -n Azure db name                                Resource name of azure postgres database. It's required when do migration from azure db\n"
    printf "    -F                                              Force to execute db migration again when first attempt failed(support azure db only) . Defalut value is false\n"
    printf "    -s Skip the DB replication check                If you don't need or have done the DB replication manually, use -s to skip the check. Defalut value is false.\n"
	printf "\nExample:\n"
	printf " sh migrate_db.sh -h 10.116.54.132\n"
	printf "\n"
	exit 1
}
function info_log(){
   echo -e "$(date) [INFO] "$1 
}
function error_log(){
   echo -e "$(date) [ERROR] [Migrate-DB-Failed] "$1
}  
function warn_log(){
   echo -e "$(date) [WARN] "$1
}

function check_stolon_keeper(){
    STOLON_CLUSTER=$1
    MAX_SLEEP_SECOND=300
    while (( MAX_SLEEP_SECOND >= 0 )); do
      RESULT=$(kubectl get pods -l=stolon-cluster="${STOLON_CLUSTER}",component="stolon-keeper" -n ${CMS_NAMESPACE})
      if [ -z "${RESULT}" ]; then
        info_log "Replication [$STOLON_CLUSTER] is stopped"
        return 
      fi
      sleep 20
      MAX_SLEEP_SECOND=$((MAX_SLEEP_SECOND - 20))
    done;
    error_log "Timeout of stoping replication[$STOLON_CLUSTER] in $MAX_SLEEP_SECOND seconds."
    exit 1
}

function wait_for_upgrade_job_completed(){
    JOB_NAME=$1
    kubectl wait --for=condition=complete --timeout=${TIMEOUT_SECONDS_FOR_UPGRADE_JOB}s job/${JOB_NAME} -n ${CMS_NAMESPACE}
    if [ $? -ne 0 ]; then 
      error_log "Timed out (${TIMEOUT_SECONDS_FOR_UPGRADE_JOB} seconds) while waiting for the Job ${JOB_NAME} to complete."
      exit 1
    fi
}

function uninstall_container_before_upgrade(){
  if helm list --namespace ${CMS_NAMESPACE} |awk '{print $1}' |grep -q -x cms-container-pg-upgrade ; then
    warn_log "The release cms-container-pg-upgrade exist. Uninstall it..."
    helm uninstall cms-container-pg-upgrade -n ${CMS_NAMESPACE} 

    MAX_SLEEP_SECOND=300
    info_log "Deleting release cms-container-pg-upgrade.... "
    while (( MAX_SLEEP_SECOND > 0 )); do
      if helm list --namespace ${CMS_NAMESPACE} |awk '{print $1}' |grep -q -x cms-container-pg-upgrade ; then
        sleep 5
        MAX_SLEEP_SECOND=$((MAX_SLEEP_SECOND - 5))
      else
        info_log "Deleted ..... ok "
        break
      fi
    done
    if [ $MAX_SLEEP_SECOND -eq 0 ]; then
      error_log "Time out of uninstall release cms-container-pg-upgrade, please check it and try again."
      exit 1
    fi
  fi
}

while getopts "h:U:P:u:p:s:n:F:" opt; do
    case $opt in
        [h])  HOST=$OPTARG;;
        [U])  SUPER_USER_NAME=$OPTARG;;
        [P])  SUPER_USER_PASSWORD=$OPTARG;;
        [u])  CMS_USER_NAME=$OPTARG;;
        [p])  CMS_USER_PASSWORD=$OPTARG;;
        [s])  SKIP_DB_REPLICATION_CHECK=true;;
        [n])  AZURE_DB_NAME=$OPTARG;;
        [F])  FORCE_UPGRADE=$OPTARG;;
        *)  function_help
        esac
done

# Check required param.
if [ -z "$HOST" ]; then
    echo "Source CMS database hostname or IP is required.";
    exit 1;
fi

if [ -z "$AZURE_DB_NAME" ]; then
  info_log "Check the host $HOST of source database ..."
  ping -c 1 $HOST >/dev/null 2>&1
  if [ $? -eq 0 ]; then
     echo 
  else
    error_log "Host '$HOST' is not accessible, please input a valid host IP and try again."
    exit 1
  fi
else
  echo
fi

if [ -z "$FORCE_UPGRADE" ]; then 
  FORCE_UPGRADE="false"
fi

if [ "$CMS_USER_PASSWORD" = "Wf\$1234"  ]; then
  CMS_USER_PASSWORD="Wf\\\$1234"
fi

SUPER_USER_NAME="${SUPER_USER_NAME:-postgres}"
SUPER_USER_PASSWORD="${SUPER_USER_PASSWORD:-n2bbsys}"
CMS_USER_NAME="${CMS_USER_NAME:-wfs}"
CMS_USER_PASSWORD="${CMS_USER_PASSWORD:-Wf\\\$1234}"
SKIP_DB_REPLICATION_CHECK="${SKIP_DB_REPLICATION_CHECK:false}"
MAX_STREAMING_CHECK_RETRY_TIMES=30
CURRENT_RETRY_TIMES=0
STREAMING_CHECK_RETRY_INTERVAL=20
TIMEOUT_SECONDS_FOR_UPGRADE_JOB=1800
TIMEOUT_SECONDS_FOR_POD=300
HELM_REPO_PATH="/opt/cms/charts"
CMS_VAR_PATH="/opt/cms/ConfigBundle/products-var.yaml"
CMS_NAMESPACE=$(cat ${CMS_VAR_PATH}|grep cms_namespace|tr '"' ' '|awk -F ":" '{print $2}'|sed s/[[:space:]]//g)
if [ $? -ne 0 ]; then 
    error_log "The namespace cannot be obtained from $CMS_VAR_PATH with the following command: cat ${CMS_VAR_PATH}|grep cms_namespace|tr '\"' ' '|awk -F \"=\" '{print \$2}'|sed s/[[:space:]]//g"
    exit 1
fi
if [ -z "$CMS_NAMESPACE" ]; then 
  warn_log "Cannot find the namespace from $CMS_VAR_PATH, use \"default\" as the defalut."
  CMS_NAMESPACE="default"
fi
IMAGE_REGISTRY=$(cat ${CMS_VAR_PATH} |grep -E ^docker_registry |tr '"' ' '|sed s/[[:space:]]//g | sed s/docker_registry://g)
if [ -z "$IMAGE_REGISTRY" ]; then 
  warn_log "Can not find the image registry from $CMS_VAR_PATH, use \"docker-registry.kube-system:5001\" as the defalut."
  IMAGE_REGISTRY="docker-registry.kube-system:5001"
fi
EXEC_CMS_PG_CONTAINER="kubectl exec -i -n ${CMS_NAMESPACE} cms-pg-metadata-manager-stolon-keeper-0 -c cms-postgresql -- bash -c "
EXEC_DB_REPLICATION_PG_CONTAINER="kubectl exec -i -n ${CMS_NAMESPACE} db-replication-metadata-manager-stolon-keeper-0 -c db-replication-metadata-manager-stolon-keeper -- bash -c "

EPG_ENABLE=false
if kubectl get sts -n $CMS_NAMESPACE cms-pg-epg-stolon-keeper &>/dev/null ; then
  DB_REPLICATION_CLUSTERS_LIST=(metadata-manager-database-replication epg-database-replication database-replication-backup workflow-database-replication)
  DB_REPLICATION_KEEPER_LIST=(db-replication-metadata-manager-stolon-keeper db-replication-epg-stolon-keeper db-replication-backup-stolon-keeper db-replication-workflow-stolon-keeper)
  CMS_PG_KEEPER_LIST=(cms-pg-metadata-manager-stolon-keeper cms-pg-workflow-stolon-keeper cms-pg-epg-stolon-keeper)
  CMS_PG_UPGRADE_JOB_LIST=(upgrade-metadata-manager-pg-to-12 upgrade-workflow-pg-to-12 upgrade-epg-pg-to-12)
  CMS_PG_UPGRADE_AKS_JOB_LIST=(epg-db-migration-aks mm-db-migration-aks)
  CMS_PG_UPGRADE_CONTAINER_LIST=(upgarde-metadata-manager upgarde-workflow upgarde-epg)
  CMS_PG_POST_UPGRADE_JOB_LIST=(post-upgrade-metadata-manager-db post-upgrade-workflow-db post-upgrade-epg-db)
  CMS_PG_SERVICE_LIST=(dbserver dbserver-workflow dbserver-epg)
  OTHER_STREAMING_LIST="epg-database-replication metadata-manager-database-replication database-replication-backup"
  EPG_ENABLE=true
else
  DB_REPLICATION_CLUSTERS_LIST=(metadata-manager-database-replication database-replication-backup workflow-database-replication)
  DB_REPLICATION_KEEPER_LIST=(db-replication-metadata-manager-stolon-keeper db-replication-backup-stolon-keeper db-replication-workflow-stolon-keeper)
  CMS_PG_KEEPER_LIST=(cms-pg-metadata-manager-stolon-keeper cms-pg-workflow-stolon-keeper)
  CMS_PG_UPGRADE_JOB_LIST=(upgrade-metadata-manager-pg-to-12 upgrade-workflow-pg-to-12)
  CMS_PG_UPGRADE_AKS_JOB_LIST=(mm-db-migration-aks)
  CMS_PG_UPGRADE_CONTAINER_LIST=(upgarde-metadata-manager upgarde-workflow)
  CMS_PG_POST_UPGRADE_JOB_LIST=(post-upgrade-metadata-manager-db post-upgrade-workflow-db)
  CMS_PG_SERVICE_LIST=(dbserver dbserver-workflow)
  OTHER_STREAMING_LIST="metadata-manager-database-replication database-replication-backup"
  info_log "No EPG database deployed, will skip EPG database migration. "
fi


if [ -z "$AZURE_DB_NAME" ]; then
  echo 
else
  info_log "Start to migrate Azure DB ...."
  info_log "Disable db related cronjobs....."
  if [ "$EPG_ENABLE" = true ]; then
    kubectl patch cronjobs cms-common-backup cms-pg-epg-fullbackup cms-pg-epg-vacuum-analyzeonly cms-pg-epg-vacuum-vacuum cms-pg-metadata-manager-fullbackup cms-pg-metadata-manager-vacuum-analyzeonly cms-pg-metadata-manager-vacuum-vacuum cms-pg-purge cms-pg-workflow-fullbackup cms-pg-workflow-vacuum-analyzeonly cms-pg-workflow-vacuum-vacuum  -p '{"spec" : {"suspend" : true }}' -n ${CMS_NAMESPACE}
  else
    kubectl patch cronjobs cms-common-backup cms-pg-metadata-manager-fullbackup cms-pg-metadata-manager-vacuum-analyzeonly cms-pg-metadata-manager-vacuum-vacuum cms-pg-purge cms-pg-workflow-fullbackup cms-pg-workflow-vacuum-analyzeonly cms-pg-workflow-vacuum-vacuum  -p '{"spec" : {"suspend" : true }}' -n ${CMS_NAMESPACE}
  fi
  
  if [ $? -ne 0 ]; then 
    error_log "Disable db related cronjobs fail. Please fix the error and execute this script again."
    exit 1
  fi
  uninstall_container_before_upgrade
  
  info_log "Deploy the Helm Chart to upgrade postgreSQL and migrate database data."
  TGZ_FILE=$(ls ${HELM_REPO_PATH}|grep -E 'cms-container-pg-upgrade.*tgz')
  helm install cms-container-pg-upgrade --set image.repository=${IMAGE_REGISTRY},db_host=$HOST,db_name=$AZURE_DB_NAME,azuredb=true,epgEnable=$EPG_ENABLE,force_upgrade=$FORCE_UPGRADE,pg_password=$SUPER_USER_PASSWORD -n ${CMS_NAMESPACE} ${HELM_REPO_PATH}/${TGZ_FILE}

  if [ $? -ne 0 ]; then 
    error_log "Deploy the Helm Chart files fail. Please fix the error and execute this script again."
    exit 1
  fi
  
  if [ "$EPG_ENABLE" = true ]; then
    count_to_retry=0
    while :;do
      if kubectl get pod -n ${CMS_NAMESPACE} | grep -P 'epg-db-migration-aks-.+1/1\s+Running' &>/dev/null; then
        break
      fi
      ((count_to_retry++))
      if [[ $count_to_retry -gt 40 ]]; then
        error_log "Time out of starting job epg-db-migration-aks, please check the pod status."
        exit 1
      fi
      if kubectl get pod -n ${CMS_NAMESPACE} | grep -P 'epg-db-migration-aks-.+0/1\s+Completed' &>/dev/null; then
        break
      fi
      sleep 3
      info_log "Wait for epg-db-migration-aks to start..."
    done
  fi
  
  sleep 30
  for i in $(seq 0 $((${#CMS_PG_UPGRADE_AKS_JOB_LIST[@]}-1))); do
    kubectl logs -l job-name="${CMS_PG_UPGRADE_AKS_JOB_LIST[$i]}" -n ${CMS_NAMESPACE} --tail=400 -f
    if [ $? -ne 0 ]; then 
      error_log "Deploy Upgrade job ${CMS_PG_UPGRADE_AKS_JOB_LIST[$i]} failed."
      exit 1
    fi
    sleep 1
    if [ $(kubectl get pods -l=app=${CMS_PG_UPGRADE_AKS_JOB_LIST[$i]} -n ${CMS_NAMESPACE} | grep Completed | awk '{print $1}') ]; then
       info_log "Upgrade job ${CMS_PG_UPGRADE_AKS_JOB_LIST[$i]} is completed."
    else
       error_log "Execute Job ${CMS_PG_UPGRADE_AKS_JOB_LIST[$i]} failed."
       exit 1
    fi
  done
  info_log "Azure db migrate successfully."
  info_log "\nNOTE: Metadata Revisions data migration is running as an another job [revision-migration-aks]. To check the status, please excute [kubectl logs -l job-name=revision-migration-aks -n ${CMS_NAMESPACE} --tail=200 -f ]. "
  exit 0
fi

# Check the CMS9.0 postgres service before upgrading
for i in $(seq 0 $((${#CMS_PG_KEEPER_LIST[@]}-1)));do
  EXISTS_KEEPER_NUM=$(kubectl get pods -n ${CMS_NAMESPACE} | grep ${CMS_PG_KEEPER_LIST[$i]} | wc | awk '{print $1}')
  if [ ${EXISTS_KEEPER_NUM} -gt 0 ]; then
    error_log "The Postgres service must be stoped before upgrading."
    info_log "Please execute command: 'cmscli service stop -s ${CMS_PG_KEEPER_LIST[$i]}' to stop Postgres service first."
    exit 1 
  fi
done

echo -e "*************************************************************\n* Start to migrate database at $(date) *\n*************************************************************"
#If re-run after db replication has been promoted, skip this step.
if [ ! -e /tmp/db-replication-done ] && [ ! ${SKIP_DB_REPLICATION_CHECK} ]; then 
  #1. Check streaming status
  for cluster in ${DB_REPLICATION_CLUSTERS_LIST[@]}; do
    STREAMING_POD=$(kubectl get pods -l=stolon-cluster=${cluster},component=stolon-keeper -n ${CMS_NAMESPACE})
    if [ -z "${STREAMING_POD}" ]; then
      error_log "No streaming pod of cluster ${cluster} is detected. Please check the database replication based on the instructions in the documents. "
      exit 1 
    fi
  done
  
  info_log "Check the Workflow database replication streaming..."
  CURRENT_RETRY_TIMES=0
  while true; do
      CURRENT_RETRY_TIMES=$((CURRENT_RETRY_TIMES + 1))
      STREAMIN_STATUS_TEXT=$(${EXEC_DB_REPLICATION_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h ${HOST} -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"select case when count(1) = 1 then 't' else 'f' end from pg_stat_replication where state='streaming' and sent_location = write_location and application_name='workflow-database-replication'\"")
      STREAMIN_STATUS=$(echo $STREAMIN_STATUS_TEXT | tr '\n' ' ' | tr '\r' ' '| awk '{print $3}')
      if [ "${STREAMIN_STATUS}x" != "tx" ]; then 
          error_log "The workflow-database-replication streaming is not completed.";
      else
          info_log "The workflow-database-replication streaming is completed.";
          break;
      fi
      if [ ${CURRENT_RETRY_TIMES} -ge ${MAX_STREAMING_CHECK_RETRY_TIMES} ]; then 
        error_log "The maximum retry number is reached. Please check the streaming status in the old CMS DB."
        exit 1
      fi
      info_log "Sleep ${STREAMING_CHECK_RETRY_INTERVAL} seconds for next check..."
      sleep ${STREAMING_CHECK_RETRY_INTERVAL};
  done
  
  #check other streaming.
  for application_name in ${OTHER_STREAMING_LIST[@]}; do
    info_log "Check ${application_name} streaming..."
    CURRENT_RETRY_TIMES=0
    while true; do
      CURRENT_RETRY_TIMES=$((CURRENT_RETRY_TIMES + 1))
      STREAMIN_STATUS_TEXT=$(${EXEC_DB_REPLICATION_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h database-replication-service -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"select case when count(1) = 1 then 't' else 'f' end from pg_stat_replication where state='streaming' and sent_location = write_location and application_name='${application_name}'\"")
      STREAMIN_STATUS=$(echo $STREAMIN_STATUS_TEXT | tr '\n' ' ' | tr '\r' ' '| awk '{print $3}')
      if [ "${STREAMIN_STATUS}x" != "tx" ]; then 
          warn_log "The ${application_name} streaming is not completed.";
      else
          info_log "The ${application_name} streaming is completed.";
          break;
      fi
      if [ ${CURRENT_RETRY_TIMES} -ge ${MAX_STREAMING_CHECK_RETRY_TIMES} ]; then 
        error_log "The maximum retry number is reached. Check the streaming status through the database hostname database-replication-service."
        exit 1
      fi
      info_log "Sleep ${STREAMING_CHECK_RETRY_INTERVAL} seconds for next check..."
      sleep ${STREAMING_CHECK_RETRY_INTERVAL};
    done
  done
else
  info_log "The DB replication has been promoted or there is no DB replication. Skip the streaming status check ."
fi

#Promoting db replication clusters to breakout streaming
if [ ! -e /tmp/db-replication-done ]; then
  for cluster in ${DB_REPLICATION_CLUSTERS_LIST[@]}; do
    info_log "Promoting the ${cluster} to discontinue the data streaming..."
    ${EXEC_DB_REPLICATION_PG_CONTAINER} "stolonctl --cluster-name=$cluster --store-backend=kubernetes --kube-resource-kind=configmap promote --yes"
  done
  #sleep for promote
  sleep 20
  #Promote flag 
  touch /tmp/db-replication-done
fi

#Stop all database replication services
info_log "Stoping all database replication services..."
for keeper in ${DB_REPLICATION_KEEPER_LIST[@]}; do
    kubectl scale --replicas=0 statefulset/${keeper} -n ${CMS_NAMESPACE}
done

#Make sure the postgres is stopped.
for cluster in ${DB_REPLICATION_CLUSTERS_LIST[@]}; do
  info_log "Wait for the Postgres of ${cluster} to stop..."
  check_stolon_keeper $cluster
done


#2. Deploy the Chart files
uninstall_container_before_upgrade

info_log "Deploy the Helm Chart to upgrade postgreSQL and migrate database data."
TGZ_FILE=$(ls ${HELM_REPO_PATH}|grep -E 'cms-container-pg-upgrade.*tgz')
helm install cms-container-pg-upgrade --set image.repository=${IMAGE_REGISTRY},epgEnable=$EPG_ENABLE -n ${CMS_NAMESPACE} ${HELM_REPO_PATH}/${TGZ_FILE}
if [ $? -ne 0 ]; then 
    error_log "Deploy the Helm Chart files fail. Please fix the error and execute this script again."
    exit 1
fi

#3. Wait for upgrade job done and start the stolon keeper as master role
for i in $(seq 0 $((${#CMS_PG_UPGRADE_JOB_LIST[@]}-1)));do
  info_log "Wait for the upgrade job ${CMS_PG_UPGRADE_JOB_LIST[$i]} to complete, which will take some time..."
  sleep 5
  POD_NAME=$(kubectl get pods -o wide -l=app="${CMS_PG_UPGRADE_JOB_LIST[$i]}"  -n ${CMS_NAMESPACE} | awk 'NR>=2{ print $1 }' | tr '\n' ' ')
  kubectl logs -f ${POD_NAME} -c  ${CMS_PG_UPGRADE_CONTAINER_LIST[$i]} -n ${CMS_NAMESPACE} &
  kubectl wait --for=condition=complete --timeout=${TIMEOUT_SECONDS_FOR_UPGRADE_JOB}s job/${CMS_PG_UPGRADE_JOB_LIST[$i]} -n ${CMS_NAMESPACE}
  if [ $? -ne 0 ]; then 
    error_log "Timed out (${TIMEOUT_SECONDS_FOR_UPGRADE_JOB} seconds) while waiting for Job ${CMS_PG_UPGRADE_JOB_LIST[$i]} to complete."
    exit 1
  fi
  info_log "Upgrade job is completed. Starting the postgres service..."
  
done

info_log "Starting the cms-pg-workflow service..."
PGTGZ_FILE=$(ls -lt ${HELM_REPO_PATH} |grep -E 'cms-postgresql-12.3.0-.+tgz' |head -n 1 | awk '{print $9}')
WF_PGVALUES_FILE=$(ls -lt /opt/cms/charts/values | grep -P 'values_cms-pg-workflow_12.3.0-.+yaml' | head -n 1 | awk '{print $9}')
helm upgrade --install -n ${CMS_NAMESPACE} cms-pg-workflow --set wfskeeper.replicaCount=1 --values=/opt/cms/charts/values/$WF_PGVALUES_FILE  ${HELM_REPO_PATH}/$PGTGZ_FILE --no-hooks
if [ $? -ne 0 ]; then 
    error_log "Starting cms-pg-workflow service failed."
    exit 1
fi

if [ "$EPG_ENABLE" = true ]; then
  info_log "Starting the cms-pg-epg service..."
  EPG_PGVALUES_FILE=$(ls -lt /opt/cms/charts/values | grep -P 'values_cms-pg-epg_12.3.0-.+yaml' | head -n 1 | awk '{print $9}')
  helm upgrade --install -n ${CMS_NAMESPACE} cms-pg-epg --set epgkeeper.replicaCount=1 --values=/opt/cms/charts/values/$EPG_PGVALUES_FILE  ${HELM_REPO_PATH}/$PGTGZ_FILE --no-hooks
  if [ $? -ne 0 ]; then 
    error_log "Starting the cms-pg-epg service failed."
    exit 1
  fi
fi

info_log "Starting the cms-pg-metadata-manager service..."  
MM_PGVALUES_FILE=$(ls -lt /opt/cms/charts/values | grep -P 'values_cms-pg-metadata-manager_12.3.0-.+yaml' | head -n 1 | awk '{print $9}')
helm upgrade --install -n ${CMS_NAMESPACE} cms-pg-metadata-manager --set cmskeeper.replicaCount=1 --values=/opt/cms/charts/values/$MM_PGVALUES_FILE  ${HELM_REPO_PATH}/$PGTGZ_FILE --no-hooks
if [ $? -ne 0 ]; then 
    error_log "Starting the cms-pg-metadata-manager service failed."
    exit 1
fi
#4. Wait for post-upgrade-db job done
info_log "Wait for the post-upgrade to complete, which will take some time..."
for job in ${CMS_PG_POST_UPGRADE_JOB_LIST[@]}; do
    kubectl wait --for=condition=complete --timeout=${TIMEOUT_SECONDS_FOR_UPGRADE_JOB}s job/${job} -n ${CMS_NAMESPACE}
    if [ $? -ne 0 ]; then 
      error_log "Timed out (${TIMEOUT_SECONDS_FOR_UPGRADE_JOB} seconds) while waiting for Job ${job} to complete."
      exit 1
    fi
done
info_log "All post upgrade jobs are completed."

#5. Uninstal the cms-container-pg-upgrade release.
info_log "Uninstall the cms-container-pg-upgrade release..."
helm uninstall cms-container-pg-upgrade -n ${CMS_NAMESPACE}

#6. Reindex in metadata-manager DB after upgrading.
info_log "Fix some broken indexs in metadata-manager database after upgrading..."
${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h dbserver -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"reindex index wfs.idx_tre_actionparmval_value;\""
${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h dbserver -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"reindex index wfs.idx_cms_setting_sname_grpid;\""

info_log "Remove deprecated resource type/group from workflow database after upgrading..."
${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h dbserver-workflow -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"delete from wfs.ttv_resourcegroupmembership where resourcegroupid in (select resourcegroupid from wfs.ttv_resourcegroup a where exists (select 1 from wfs.ttv_resourcetype b where a.resourcetypeid=b.resourcetypeid and b.systemid in ('09','NG')));\""
${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h dbserver-workflow -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"delete from wfs.ttv_resourcegroup a where exists (select 1 from wfs.ttv_resourcetype b where a.resourcetypeid=b.resourcetypeid and b.systemid in ('09','NG'));\""
${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h dbserver-workflow -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"delete from wfs.ttv_resource a where exists (select 1 from wfs.ttv_resourcetype b where a.resourcetypeid=b.resourcetypeid and b.systemid in ('09','NG'));\""
${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=\"${SUPER_USER_PASSWORD}\" psql -h dbserver-workflow -d ttv -U ${SUPER_USER_NAME} -p 5432 -c \"delete from wfs.ttv_resourcetype where systemid in ('09','NG');\""


#7. Generate optimizer statistics
#for service in ${CMS_PG_SERVICE_LIST[@]}; do
#    info_log "Generate optimizer statistics in ${service}."
#    ${EXEC_CMS_PG_CONTAINER} "PGPASSWORD=${SUPER_USER_PASSWORD} /usr/lib/postgresql/12/bin/vacuumdb -U ${SUPER_USER_NAME} --all --analyze-in-stages -h ${service}"
#done

#8. enable db cronjobs after upgrade
info_log "Enable db related cronjobs after postgresql is upgraded....."
if [ "$EPG_ENABLE" = true ]; then
  kubectl patch cronjobs cms-common-backup cms-pg-epg-fullbackup cms-pg-epg-vacuum-analyzeonly cms-pg-epg-vacuum-vacuum cms-pg-metadata-manager-fullbackup cms-pg-metadata-manager-vacuum-analyzeonly cms-pg-metadata-manager-vacuum-vacuum cms-pg-purge cms-pg-workflow-fullbackup cms-pg-workflow-vacuum-analyzeonly cms-pg-workflow-vacuum-vacuum  -p '{"spec" : {"suspend" : false }}' -n ${CMS_NAMESPACE}
else
  kubectl patch cronjobs cms-common-backup cms-pg-metadata-manager-fullbackup cms-pg-metadata-manager-vacuum-analyzeonly cms-pg-metadata-manager-vacuum-vacuum cms-pg-purge cms-pg-workflow-fullbackup cms-pg-workflow-vacuum-analyzeonly cms-pg-workflow-vacuum-vacuum  -p '{"spec" : {"suspend" : false }}' -n ${CMS_NAMESPACE}
fi
#9. Use cms-cli command to restore the replicas of keeper
echo "********************************************"
info_log "Master node of postgres migration is completed.."
info_log "Starting standby keeper of postgresql to sync up with master node....."
for keeper in ${CMS_PG_KEEPER_LIST[@]}; do
  kubectl scale --current-replicas=1 --replicas=2 statefulset/${keeper} -n ${CMS_NAMESPACE}
done
info_log "The standby of postgres need some time to sync up with master, you can check it later."

info_log "Uninstall cms-container-db-replication release .."
helm uninstall cms-container-db-replication -n ${CMS_NAMESPACE}
info_log "****************The database migration is completed successfully.*************************"

