#!/bin/bash


function gen_kube_config()
{
    serviceaccount="$1"
    #Replace $api_server with kube-config server parameter
    api_server="$2"

    if ! secret="$(kubectl -n $NAME_SPACE get serviceaccount "$serviceaccount" -o 'jsonpath={.secrets[0].name}' 2>/dev/null)"; then
        echo "serviceaccounts \"$serviceaccount\" not found." >&2
        exit 2
    fi

    if [[ -z "$secret" ]]; then
        echo "serviceaccounts \"$serviceaccount\" doesn't have a serviceaccount token." >&2
        exit 2
    fi

    # context
    context="$(kubectl -n $NAME_SPACE config current-context)"
    # cluster
    cluster="$(kubectl -n $NAME_SPACE config view -o "jsonpath={.contexts[?(@.name==\"$context\")].context.cluster}")"
    if [ -z $api_server ];then
        server="$(kubectl -n $NAME_SPACE config view -o "jsonpath={.clusters[?(@.name==\"$cluster\")].cluster.server}")"
    else
        server=$api_server
    fi

    # token
    ca_crt_data="$(kubectl -n $NAME_SPACE get secret "$secret" -o "jsonpath={.data.ca\.crt}" | openssl enc -d -base64 -A)"
    namespace="$(kubectl -n $NAME_SPACE get secret "$secret" -o "jsonpath={.data.namespace}" | openssl enc -d -base64 -A)"
    token="$(kubectl -n $NAME_SPACE get secret "$secret" -o "jsonpath={.data.token}" | openssl enc -d -base64 -A)"

    export KUBECONFIG="$(mktemp)"
    kubectl config set-credentials "$serviceaccount" --token="$token" >/dev/null
    ca_crt="$(mktemp)"; echo "$ca_crt_data" > $ca_crt
    kubectl config set-cluster "$cluster" --server="$server" --certificate-authority="$ca_crt" --embed-certs >/dev/null
    kubectl config set-context "$context" --cluster="$cluster" --namespace="$namespace" --user="$serviceaccount" >/dev/null
    kubectl config use-context "$context" >/dev/null

    cat "$KUBECONFIG" | base64 -w 0
}

function ct_registration()
{
    trap 'echo "[Error] Scripts failed when running function [lib/${FUNCNAME[0]}] - error line: $LINENO, error cmd: $BASH_COMMAND"' ERR

    reset_function_timer

    #Replace $API_SERVER_URL with kube-config server parameter
    API_SERVER_URL=$1

    if [[ -z $API_SERVER_URL ]]; then
      API_SERVER_URL=$CLUSTER_API_SERVER_URL
    fi

    NAME_SPACE=$(get_product_parameter cms_namespace)
    CORE_VIP=$(kubectl -n $NAME_SPACE get endpoints app -o=jsonpath='{.subsets[0].addresses[0].ip}')

    #base_info
    CLUSTER_NAME=$(get_product_parameter cms_cluster_name)
    HOST=$(get_product_parameter cms_cluster_name)
    CLUSTER_VIP=$(get_product_parameter cms_vip)
    CLUSTER_PORT=$(get_product_parameter cms_haproxy.https8443ExtPort)
    DISPLAY_NAME=$(get_product_parameter cms_cluster_name)

    #cert_info
    SERVER_KEYSTORE_B64=$(get_product_parameter cms_ssl.serverkeystoreb64)
    CLIENT_AUTH=$(get_product_parameter cms_ssl.client_auth_keystorejksb64)
    TRUST_STORE_JKS=$(get_product_parameter cms_ssl.cms_truststorejksb64)
    KEYSTORE_PASS=$(get_product_parameter cms_ssl.keystorepassb64)
    CMS_FINGERPRINT=$(get_product_parameter cms_ssl.cms_fingerprintb64)
    APP_PEM=$(get_product_parameter cms_ssl.apppemb64)
    CLIENT_PEM=$(get_product_parameter cms_ssl.clientpemb64)

    #redis_info
    REDIS_PORT=6379
    #REDIS_PASSWORD=$(get_product_parameter cms_redis.password)
    REDIS_PASSWORD=$(kubectl get secrets -n $NAME_SPACE secret-cms-redis -o yaml|awk '/redis-password:/{print $2}'|base64 -d)

    #THe $APIAPI_SERVER_URL variables will replace the server parameters in kube-config
    #kube_info
    KUBE_INFO=$(gen_kube_config cms-common $API_SERVER_URL)
    
    echo "{
    \"base_info\": {
        \"cluster_name\": \"$CLUSTER_NAME\",
        \"host\": \"$HOST\",
        \"cluster_vip\": \"$CLUSTER_VIP\",
        \"cluster_port\": \"$CLUSTER_PORT\",
        \"display_name\": \"$DISPLAY_NAME\"
    },
    \"cert_info\": {
        \"client.pem\": \"$CLIENT_PEM\",
        \"app.pem\": \"$APP_PEM\",
        \"client_auth_keystorejks\": \"$CLIENT_AUTH\",
        \"cms_fingerprint\": \"$CMS_FINGERPRINT\",
        \"cms_truststorejks\": \"$TRUST_STORE_JKS\",
        \"keystorepass\": \"$KEYSTORE_PASS\"
    },
    \"redis_info\": {
        \"redis_port\": \"$REDIS_PORT\",
        \"redis_password\": \"$REDIS_PASSWORD\"
    },
    \"kube_info\": {
        \"cms_common_kubeconfig\": \"$KUBE_INFO\"
      }
    }" > /tmp/cms_ct_info.json

    ####CT information######
    echo "===================External CT information=================="
    cat /tmp/cms_ct_info.json

    #Generate client.pem file for CT registeration.
    PEER_CLIENT_PEM=$(get_product_parameter cms_peer_ssl.clientpemb64)
    #echo "$PEER_CLIENT_PEM" | base64 --decode > /tmp/client.pem
    echo "$PEER_CLIENT_PEM" | base64 -d > /tmp/client.pem
    
    #Send the CT registration request
    REGSITER=$(curl -s -m 10 -k -H 'Content-type':'application/json' --cert /tmp/client.pem -X PUT -d @/tmp/cms_ct_info.json https://$CORE_VIP:8443/ctmanager/registration)
    echo $REGSITER
    if [[ $REGSITER =~ 200 ]];then
        echo "[INFO]Register external CT to Core [$CORE_VIP] successfully."
    else
        exit 1
    fi
    
    #Delete tmp file
    rm -rf /tmp/client.pem
    rm -rf /tmp/cms_ct_info.json

    running_time
}

function unregister_remote_ct_in_geo()
{
  trap 'echo "[Error] Scripts failed when running function [lib/${FUNCNAME[0]}] - error line: $LINENO, error cmd: $BASH_COMMAND"' ERR

  reset_function_timer

  NAME_SPACE=$(get_product_parameter cms_namespace)
  CORE_VIP=$(kubectl -n $NAME_SPACE get endpoints app -o=jsonpath='{.subsets[0].addresses[0].ip}')
  geo_enabled=$(kubectl -n $NAME_SPACE get configmap cms-geo-setup-status-configmap  -o "jsonpath={.data.geoEnabled}")
  if [ 'XY' != 'X'$geo_enabled ]; then
    return 0
  fi
  echo "Geo enabled. "
  last_action=$(kubectl -n $NAME_SPACE get configmap cms-geo-configmap  -o "jsonpath={.data.lastAction}")
  echo "Local Geo last action is $last_action"
  if [ "X"$last_action != "Xfailover" ] &&  [ "X"$last_action != "Xswitchover" ]; then
    return 0
  fi

  update_redis_output_config
  if [ $? -ne 0 ]; then
    exit 1
  fi

  echo "Unregister remote external CT before register local external CT to Core."
  #Generate client.pem file for CT unregisteration.
  PEER_CLIENT_PEM=$(get_product_parameter cms_peer_ssl.clientpemb64)
  #echo "$PEER_CLIENT_PEM" | base64 --decode > /tmp/client.pem
  echo "$PEER_CLIENT_PEM" | base64 -d > /tmp/client.pem
  GEO_REMOTE_CLUSTER_NAME=$(get_product_parameter cms_geo.remote_cluster_name)
  UNREGSITER=$(curl -s -m 10 -k --cert /tmp/client.pem -X GET https://$CORE_VIP:8443/ctmanager/unregistration?clustername=$GEO_REMOTE_CLUSTER_NAME)
  echo $UNREGSITER
  if [[ $UNREGSITER =~ 200 ]];then
      echo "[INFO]Unregister remote external CT [$GEO_REMOTE_CLUSTER_NAME] from Core [$CORE_VIP] successfully."
  else
      exit 1
  fi

  #Delete tmp file
  rm -rf /tmp/client.pem
  running_time

}

function update_redis_output_config(){
  echo "Update redis output config."
  kubectl -n $NAME_SPACE exec -it cms-geo-cli-0 -- bash /opt/cms/geo-cli/util/geo_change_redis_config.sh ${last_action}
  if [ $? -eq 0 ]; then
    echo "Update redis output successfully."
  else
    echo "Update redis output failed."
    exit 1
  fi
}
