#!/usr/bin/env bash
set -e

# Get full path of this script file
SCRIPT=$(readlink -f "$0")
# Get folder path of this script
SCRIPTPATH=$(dirname "$SCRIPT")

# Define the log file for this scripts
LOG="${SCRIPTPATH}/logs/$(basename "$0" .sh).log"

# Make sure the log file is ready
mkdir -p $(dirname $LOG)
touch $LOG

trap ctrl_c INT
trap 'echo "Error in scripts '$0': error line: $LINENO,error cmd: $BASH_COMMAND" |& tee -i -a $LOG' ERR

function ctrl_c() {
   echo "** Trapped CTRL-C ** Stopped by User"
}

#########################
# Function: check_packages
# Make sure the needed packages are ready in ${SCRIPTPATH}/packages-prepack folder
#########################

function check_packages(){
  local packagesToCheck=$1
  cd $SCRIPTPATH
  cd ../packages-prepack

  for packageToCheck in ${packagesToCheck}
  do
    if [[  $(ls ${packageToCheck} > /dev/null 2>&1; echo $?) -ne 0 ]]; then
      echo "[Error] Missing package $packageToCheck in folder ${SCRIPTPATH}/packages-prepack. Make sure all needed packages are put into the packages folder before running this scripts again."
      echo "$packagesToCheck" |& tee -a $LOG
      exit 1
    fi
  done

  # Change back to previous folder
  cd - > /dev/null
}

function process_check(){
  if [ ${PIPESTATUS[0]} -ne 0 ];then
     echo "[Error] $1" |& tee -a $LOG
     exit 1
  fi
}

function prepare_images(){
  cd $SCRIPTPATH

  local _packageName=$1

  case $_packageName in
      *.tar|*.tgz )  _type=PACKAGE;;
      * ) echo -e $help_message; exit 1;;
  esac

  # Operation for single package
  if [[ $_type == "PACKAGE" ]]; then

    check_packages $_packageName

    # Clean up the information for loaded images
    cat /dev/null > ${SCRIPTPATH}/loaded_images.txt

    # Load images from the container packages into bastion machine
    cd ../packages-prepack
    echo "[Info] Load images from $_packageName."
    if [[  $(sudo docker load -i $_packageName >> ${SCRIPTPATH}/loaded_images.txt 2>&1; echo $?) -ne 0 ]]; then
      echo "[Error] Failed to load images from $packagename by docker load command."
      exit 1
    fi

    echo -e "\n[Info] Images To Be Pushed:\n=================================================================="
    cat ${SCRIPTPATH}/loaded_images.txt | sort | awk -F ': ' '{print "  "NR"\t", $2}'
    echo -e "==================================================================\n\n"

    for _image in $(cat ${SCRIPTPATH}/loaded_images.txt | sort | awk -F ': ' '{print $2}')
    do
      image=${_image##*/}
      echo "[Info] Pushing image $image to Container Image Registry [$PREPACK_IMAGE_REGISTRY]."
      sudo docker tag $image $PREPACK_IMAGE_REGISTRY/$image
      sudo docker push $PREPACK_IMAGE_REGISTRY/$image
      wait
    done
    echo "[Info] Complete pushing images."
  fi

  #wait

  # Change back to previous folder
  cd - > /dev/null
}

############################################################
# MAIN
############################################################

mkdir -p ${SCRIPTPATH}/logs

PREPACK_IMAGE_REGISTRY=$1
if [ -z "$PREPACK_IMAGE_REGISTRY" ]; then
   echo -e "[ERROR] Prepack Image Registry does not specify a value.\n"
   exit 1
fi
# Prepare Docker Images
prepare_images ${@:2} |& tee -i -a $LOG
process_check "Prepare images failed!"


