#!/usr/bin/env bash

#set -ex
#set -x
set -e

RUN_AS_CONTAINER=${RUN_AS_CONTAINER:-false}

# Get full path of this script file
SCRIPT=$(readlink -f "$0")
# Get folder path of this script
SCRIPTPATH=$(dirname "$SCRIPT")
PARENTPATH=$(dirname "$SCRIPTPATH")

BUNDLEFOLDER="${PARENTPATH}/ConfigBundle"
PRODUCTSVAR_FILE="${BUNDLEFOLDER}/products-var.yaml"
CMSCONF="$PARENTPATH/cms.var"
CLOUD_ENV="$PARENTPATH/aks.env"

# Define the log file for this scripts
LOG="${PARENTPATH}/logs/$(basename "$0" .sh).log"

##############################
# Include all functions from shell_lib
##############################
for f in ${PARENTPATH}/shell_lib/*.sh; do source $f; done

# Parameters from cms.var
CLOUDPLATFORM=$(get_parameter_value cloud_platform)
if [[ $CLOUDPLATFORM != "AZURE" ]];then
  echo "[ERROR] This scripts is only for Azure AKS PVC migration." |& tee -i -a $LOG
  exit 1
fi

#Common parameters

CLUSTER_NAME=$(get_parameter_value cluster_name)
CLOUD_CLUSTER_INSTANCE_NAME="$CLUSTER_NAME-cms-cluster"
CMS_NAMESPACE=$(get_parameter_value cms_namespace)
CLUSTER_ROLE=$(get_parameter_value cluster_role)
INTERNAL_CT_ENABLED=$(get_parameter_value internal_ct_enabled)
CONTENT_NAS_SIZE=$(get_parameter_value content_nas_size)

AZURE_REGION=$(get_parameter_value azure_region)
ACR_RESOURCEGROUP="cmsacr-rg-$AZURE_REGION"

#IMAGE_TAG=$(yq r $PRODUCTSVAR_FILE 'cms_images_tag.cms_common_backup_restore')
IMAGE_TAG=$(yq '.cms_images_tag.cms_common_backup_restore' $PRODUCTSVAR_FILE)
ACR_NAME=$(get_parameter_value acr_name)
IMAGE_REGISTRY=$(get_parameter_value image_registry)

if [[ -z "$IMAGE_REGISTRY" ]] && [[ -z "$ACR_NAME" ]]; then
  echo "[ERROR] Both image_registry and acr_name are empty, make sure one of them has proper value provided in cms.var file."
  exit 1
fi


RESOURCEGROUP_NAME="$CLUSTER_NAME-cms-resourcegroup"

#AZUREFILE_SHARES="cms-content cms-content-heavy cms-metadata-manager-content cms-conf cms-plugin backup-restore cmscli-plugin cms-sync-utils cms-service-client cms-workflow-runtime"
AZUREFILE_SHARES="cms-content cms-metadata-manager-content cms-conf cms-plugin backup-restore cmscli-plugin cms-sync-utils cms-service-client cms-workflow-runtime cms-watchfolder-runtime"
AZURE_STORAGE_ACCOUNT_NAME=$(grep OUTPUT_STORAGEACCOUNT_NAME ${CLOUD_ENV} | awk -F '=' '{print $2}')

trap ctrl_c INT
trap 'echo "Error in scripts '$0': error line: $LINENO,error cmd: $BASH_COMMAND" |& tee -i -a $LOG' ERR

start_information_print(){
  echo "[INFO] *********************************************" |& tee -i -a $LOG
  echo "[INFO] Start time : $(date +"%Hh%Mm%Ss%3Nms")" |& tee -i -a $LOG
  echo "[INFO] Run As Container : $RUN_AS_CONTAINER" |& tee -i -a $LOG
  START_TIME=$(date "+%s")
  
  if [[ -z "$IMAGE_REGISTRY" ]];then
    echo "[INFO] Getting Azure ACR information."
    IMAGE_REGISTRY=$(sudo az acr list -g $ACR_RESOURCEGROUP -o table --query [].loginServer|grep -i $ACR_NAME|tail -1)
  fi

  echo "" |& tee -i -a $LOG
  echo "=========== All Scripts Parameters ======================" |& tee -i -a $LOG
  echo "  SCRIPTPATH=$SCRIPTPATH"|& tee -i -a $LOG
  echo "  LOG=$LOG"|& tee -i -a $LOG
  echo "  BUNDLEFOLDER=$BUNDLEFOLDER"|& tee -i -a $LOG
  echo "  CMSCONF=$CMSCONF"|& tee -i -a $LOG
  echo "  CLUSTER_NAME=$CLUSTER_NAME"|& tee -i -a $LOG
  echo "  CLUSTER_ROLE=$CLUSTER_ROLE"|& tee -i -a $LOG
  echo "  INTERNAL_CT_ENABLED=$INTERNAL_CT_ENABLED"|& tee -i -a $LOG
  echo "  IMAGE_REGISTRY=$IMAGE_REGISTRY"|& tee -i -a $LOG
  echo "  ACR_NAME=$ACR_NAME"|& tee -a $LOG
  echo "  RESOURCEGROUP_NAME=$RESOURCEGROUP_NAME"|& tee -a $LOG
  echo "  AZURE_STORAGE_ACCOUNT_NAME=$AZURE_STORAGE_ACCOUNT_NAME"|& tee -a $LOG
  echo "=========================================================" |& tee -i -a $LOG
  echo "" |& tee -i -a $LOG
}

##################
#  Script usage  #
################## 
usage_function() {
    echo "This script is for migrating azure-file dynamic share to static shares."
    echo "----------------------------------------------------------------"
    echo "Usage: $0 <function>"
    echo
    echo "Available Functions:"
    echo "  all         : Migrage all existing azure-file PVCs from dynamic share to static shares"
    echo "  clean       : Clean all previous azure-file PVCs after migration completed"
    echo
    echo "Examples:"
    echo "  $0 all"
    echo "  $0 clean"
    echo "  $0 -h"
    echo
}


############################################################
# MAIN
############################################################


if [ "$1" = "all" ] ; then
  start_information_print
  # Prepare Azure StorageClass for static auzre-file shares
  for sharename in $AZUREFILE_SHARES;
  do
    echo "[Info] Creating StorageClass azure-file-$sharename."
    cat <<EOF | kubectl apply -f -
kind: StorageClass
apiVersion: storage.k8s.io/v1
metadata:
  name: azure-file-${sharename}
provisioner: kubernetes.io/azure-file
allowVolumeExpansion: true
reclaimPolicy: Retain
mountOptions:
  - dir_mode=0755
  - file_mode=0755
  - uid=99
  - gid=99
  - mfsymlinks
  - cache=strict
parameters:
  storageAccount: $AZURE_STORAGE_ACCOUNT_NAME
  resourcegroup: $RESOURCEGROUP_NAME
  shareName: ${sharename}
EOF
    
    # Create Template PVC and migrate the data only if PVC is existing
    if [[ $(kubectl get -n $CMS_NAMESPACE pvc pvc-$sharename > /dev/null 2>&1; echo $?) -eq 0 ]];then
    echo "[Info] Creating tempory PVC pvc-temp-$sharename."
    storage_size=$(kubectl get -n $CMS_NAMESPACE pvc pvc-$sharename -o=jsonpath='{.status.capacity.storage}' --request-timeout 60s)
    cat <<EOF | kubectl apply -f -
apiVersion: v1
kind: PersistentVolumeClaim
metadata:
  name: pvc-temp-${sharename}
  namespace: ${CMS_NAMESPACE}
spec:
  accessModes:
    - ReadWriteMany
  storageClassName: azure-file-${sharename}
  resources:
    requests:
      storage: ${storage_size}
EOF

    echo "[Info] Runnig Job to migrate data from [pvc-$sharename] to [pvc-temp-$sharename]."
    if [[ $(kubectl get -n $CMS_NAMESPACE job migrate-azure-file-pvc-$sharename --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      kubectl delete -n $CMS_NAMESPACE job migrate-azure-file-pvc-$sharename
    fi
    cat <<EOF | kubectl apply -f -
apiVersion: batch/v1
kind: Job
metadata:
  name: migrate-azure-file-pvc-${sharename}
  namespace: ${CMS_NAMESPACE}
  labels:
    job: migrate-azure-file
    app: cms-job
spec:
  backoffLimit: 1
  template:
    metadata:
      labels:
        job: migrate-azure-file
        app: cms-job
    spec:
      restartPolicy: Never
      imagePullSecrets: [{name: cms-image-pull-secret}]
      containers:
      - name: migrate-azure-file
        image: ${IMAGE_REGISTRY}/cms-common-backup-restore:${IMAGE_TAG}
        command: 
          - sh
          - '-c'
          - |
            touch /mnt/old-dynamic/startSyncing.tmp
            rsync -av --no-perms --no-owner --no-group --delete /mnt/old-dynamic/* /mnt/new-static
            if [[ $? -ne 0 ]];then
              echo "[Error] Sync Failed."
              exit 1
            fi
            sleep 5
            rm -f /mnt/new-static/startSyncing.tmp
            echo "[Info] Sync Completed."
            exit 0
        resources:
          requests:
            cpu: 200m
            memory: 128Mi
          limits:
            cpu: 500m
            memory: 512Mi
        securityContext:
          allowPrivilegeEscalation: false
          readOnlyRootFilesystem: true
          runAsUser: 99
          runAsGroup: 99
          runAsNonRoot: true
        volumeMounts:
        - mountPath: '/mnt/old-dynamic'
          name: volume-old-dynamic
        - mountPath: '/mnt/new-static'
          name: volume-new-static
      volumes:
        - name: volume-old-dynamic
          persistentVolumeClaim:
            claimName: pvc-${sharename}
        - name: volume-new-static
          persistentVolumeClaim:
            claimName: pvc-temp-${sharename}    
EOF

    fi
  echo ""
  done
  
  running_amount=10
  for (( i=1; i<=30; i++ ))
  do
    sleep 5
    running_amount=$(kubectl get pod -n $CMS_NAMESPACE --request-timeout 60s | grep migrate-azure-file | grep Running | wc -l)
    error_amount=$(kubectl get pod -n $CMS_NAMESPACE --request-timeout 60s | grep migrate-azure-file | grep Error | wc -l)
    if [[ $error_amount -ne 0 ]];then
      echo "[Error] Job failed, check the migrate-azure-file jobs status for details."
      exit 1
    fi
    if [[ $running_amount -eq 0 ]];then
      echo "[Info] All migrate-azure-file jobs completed."
      break
    fi
  done
  if [[ $running_amount -eq 0 ]];then
    echo "[Info] All CMS data have been migrated to new AzureFile static shares."
  else
    echo "[Warning] Checking timeout, migrate-auzre-file jobs still running."
    echo "Use 'kubectl get pod -n $CMS_NAMESPACE | grep migrate-azure-file' command to check status manually."
    echo "Ensure all pods are in Completed status before continue."
    exit 1
  fi

  # TODO: Clean all temp PVCs 

elif [ "$1" == "clean" ] ; then
  user_confirm "Deleting All Old Azure-File PVC! 
Please confirm all data have been migrated to static shares by command:
- sh migrate_aks_pvc.sh all
"
  user_confirm "Please confirm cms services have been stopped by below commands:
- cmscli service stop
- cmscli service stop -s cms-pg-epg-stolon-keeper
- cmscli service stop -s cms-pg-metadata-manager-stolon-keeper
- cmscli service stop -s cms-pg-workflow-stolon-keeper
- kubectl scale -n $CMS_NAMESPACE sts cms-cli --replicas=0
- kubectl scale -n $CMS_NAMESPACE sts cms-config --replicas=0
- kubectl patch cronjobs cms-common-backup -n $CMS_NAMESPACE -p '{\"spec\" : {\"suspend\" : true }}'
"
  # Delete all PVC migrate jobs
  kubectl delete job -l job=migrate-azure-file -n $CMS_NAMESPACE --request-timeout 60s

  # Delete all old dynamic PVCs
  for sharename in $AZUREFILE_SHARES;
  do
    if [[ $(kubectl get -n $CMS_NAMESPACE pvc pvc-$sharename --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      echo "[Info] Deleting PVC pvc-${sharename}..."
      kubectl delete pvc -n $CMS_NAMESPACE pvc-$sharename --request-timeout 60s
    fi
    if [[ $(kubectl get -n $CMS_NAMESPACE pvc pvc-temp-$sharename --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      echo "[Info] Deleting PVC pvc-temp-${sharename}..."
      kubectl delete pvc -n $CMS_NAMESPACE pvc-temp-$sharename --request-timeout 60s
    fi
  done 
  echo "[Info] All Old Azure-file PVCs deleted."
elif [ "$1" == "-h" ] || [ "$1" == "--help" ] ; then
  usage_function

else
  usage_function
fi
