#!/usr/bin/python
import argparse
import os
import re
import subprocess

import sys

INSTALLER_SCRIPT_PATH = "/opt/cms"
sys.path.insert(1, "{}/py_lib".format(INSTALLER_SCRIPT_PATH))
sys.path.insert(1, "{}/py3_lib".format(INSTALLER_SCRIPT_PATH))

import yaml

CMS_VAR_FILE = os.path.join(INSTALLER_SCRIPT_PATH, "cms.var")
PP_VAR_FILE = os.path.join(INSTALLER_SCRIPT_PATH, "prepack.var")
PRODUCTS_VAR_PREPACK_FILE = os.path.join(INSTALLER_SCRIPT_PATH, "ConfigBundle", "products-var-prepack.yaml")

ROOT_AVAILABLE_COMPONENTS = 'available_components'
KEY_CMS_PREPACK = 'cms_prepack'
KEY_COMPONENTS_TO_INSTALL = 'components_to_install'
KEY_INIT_CONTAINERS = 'init_containers'
LIST_FORMAT = '  - {}'

PREPACK_INSTALLATION_PROMPT = "Would you like to continue the PrePack installation? Type \'yes\' to proceed OR \'no\' to terminate the installation: "

# Name of ConfigMap to save PrePack information
PP_CM_NAME = 'cms-prepack'
# Key of installed components in ConfigMap "cms-prepack"
CM_KEY_INSTALLED_COMPONENTS = 'installedComponents'
# Key of installed customization in ConfigMap "cms-prepack"
CM_KEY_CUSTOMIZATION = 'customization'

# Blacklist for pulling images. Component types in this list will not be setup for init containers.
NO_IMAGES = ['cdn', 'packager', 'transport', 'dai', 'qc']

ADDITIONAL_COMPONENT_TO_INSTALL = {
    ('mkp-geo'): 'mkp',
    ('mkeo-geo'): 'mkeo',
    ('mediafirst', 'mkp'): 'mediafirst-with-mkp',
    ('mediafirst', 'mkp-geo'): 'mediafirst-with-mkp'
}


def list_available_components():
    all_components = get_all_components(PRODUCTS_VAR_PREPACK_FILE)
    installed = get_installed_components_list()
    available = [a for a in all_components.keys() if a not in installed]
    if not available:
        print('All the available components have been installed.')
    else:
        print_components_list('Available Components for PrePack Installation:', sorted(available), all_components)


def list_installed_components():
    installed = get_installed_components_list()

    customization = get_installed_customization()
    if not installed and not customization:
        print("No PrePack component installed yet.")
    else:
        if installed:
            print_components_list('Installed PrePack components:', sorted(installed),
                                  get_all_components(PRODUCTS_VAR_PREPACK_FILE))
        if customization:
            print('Customization:')
            for c in customization:
                print(LIST_FORMAT.format(c))


def merge_var_files():
    """ Merge values in prepack.var into products-var-prepack.yaml. """
    with open(PP_VAR_FILE) as source, open(PRODUCTS_VAR_PREPACK_FILE) as p:
        # Load variables from products-var-prepack.yaml file
        vars_in_products_var_pp = yaml.safe_load(p)
        pp_vars_in_products_var_pp = vars_in_products_var_pp.get(KEY_CMS_PREPACK)

        # Load values from prepack.var
        pp_var_values = yaml.safe_load(source)

        # if installing PrePack without any "components_to_install" in prepack.var:
        if not pp_var_values.get(KEY_COMPONENTS_TO_INSTALL):
            # program will exit
            print('[WARN] PrePack installation stopped since no component is selected.')
            exit(1)

        # else: setup components to install
        setup_init_containers(pp_var_values)

        # Prompt for user confirmation on installing PrePack without --always-continue arg
        if not CONTINUE:
            proceed = prompt_yes_or_no(PREPACK_INSTALLATION_PROMPT)
            if not proceed:
                print('[INFO] PrePack installation cancelled.')
                exit(1)

        for key, value in pp_var_values.items():
            if key in pp_vars_in_products_var_pp:
                pp_vars_in_products_var_pp[key] = value
            elif key in vars_in_products_var_pp:
                vars_in_products_var_pp[key] = value
            else:
                vars_in_products_var_pp[key] = value

        with open(PRODUCTS_VAR_PREPACK_FILE, 'w') as destination:
            yaml.safe_dump(vars_in_products_var_pp, destination)


def get_namespace_from_shell():
    print(get_namespace())
    exit(0)


def set_entry(key, value):
    with open(PRODUCTS_VAR_PREPACK_FILE) as f:
        variables = yaml.safe_load(f)
        variables[key] = '"{}"'.format(value)
        with open(PRODUCTS_VAR_PREPACK_FILE, 'w') as s:
            yaml.safe_dump(variables, s)


def get_all_components(var_yaml_path):
    """ Get available components from preinstall products-var-prepack file """
    with open(var_yaml_path) as f:
        return yaml.safe_load(f).get(ROOT_AVAILABLE_COMPONENTS)


def get_namespace():
    try:
        with open(CMS_VAR_FILE, 'r') as f:
            for line in f:
                key, value = line.strip().partition('=')[::2]
                if 'cms_namespace' == key.strip():
                    return value.strip().replace('"', '')
                else:
                    continue
        return ''
    except IOError:
        print("[ERROR] cms.var cannot be found. Please make sure that this var file is under proper directory.")
        exit(1)


def get_installed_components_list():
    return get_content_from_configmap(CM_KEY_INSTALLED_COMPONENTS)


def get_installed_customization():
    return get_content_from_configmap(CM_KEY_CUSTOMIZATION)


def get_content_from_configmap(key):
    namespace = get_namespace()
    if not namespace:
        print("[ERROR] cms_namespace is not set. Please fill in with proper namespace value and rerun the installer.")
        exit(1)
    try:
        p = subprocess.Popen('kubectl get cm {} -n {} -o yaml'.format(PP_CM_NAME, namespace), stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE, shell=True)
        cm, err = p.communicate()
        if err:
            return []
        cm_data = yaml.safe_load(cm)['data']
        if key not in cm_data:
            return []
        return cm_data[key].split()
    except Exception:
        print("Exception while getting installed components.")
        exit(1)


def setup_init_containers(pp_var_values):
    # "init_containers" is not set in prepack.var, so create a list to store common components first.
    pp_var_values[KEY_INIT_CONTAINERS] = ['base']

    # get components from prepack.var
    components_to_install = pp_var_values.get(KEY_COMPONENTS_TO_INSTALL)

    # remove the duplicates
    components_to_install = list(set(components_to_install))

    # get available components from preinstall prepack-var file
    available_components_dict = get_all_components(PRODUCTS_VAR_PREPACK_FILE)
    for component_to_install in components_to_install[:]:
        if component_to_install not in available_components_dict:
            components_to_install.remove(component_to_install)
            continue
        dirs = available_components_dict[component_to_install].strip().split('/')
        # replace '_' with '-' to avoid setup error in Kubernetes
        component_type = re.sub(r'_', '-', dirs[-2].lower())
        # setup downstream / device base into init containers
        if component_type not in pp_var_values[KEY_INIT_CONTAINERS] and component_type not in NO_IMAGES:
            pp_var_values[KEY_INIT_CONTAINERS].append(component_type)

    print_components_list('Components listed below will be installed:', sorted(components_to_install),
                          available_components_dict)

    # install some extra components
    components_to_install = install_additional_components(components_to_install)
    pp_var_values[KEY_COMPONENTS_TO_INSTALL] = components_to_install

    # set components_to_install into init_containers also
    pp_var_values[KEY_INIT_CONTAINERS].extend(sorted(components_to_install))


def prompt_yes_or_no(message):
    user_input = ''
    while user_input.lower() not in ['yes', 'y', 'no', 'n']:
        user_input = get_raw_input(message)
    return user_input.lower() in ['yes', 'y']


def get_raw_input(message):
    try:
        if is_python3():
            return input(message)
        return raw_input(message)
    except (IOError, EOFError):
        return get_raw_input("")

def is_python3():
    return sys.version_info >= (3,0)

def print_components_list(header, components, mappings):
    print(header)
    devices = [k for k in components if k in mappings and 'device' in mappings[k].lower()]
    if devices:
        print('Devices:')
        for component in devices:
            print(LIST_FORMAT.format(component))
    downstream = [k for k in components if k in mappings and 'downstream' in mappings[k].lower()]
    if downstream:
        print('PrePack Solutions:')
        for component in downstream:
            print(LIST_FORMAT.format(component))


def install_additional_components(components_to_install):
    additional_component_key = ADDITIONAL_COMPONENT_TO_INSTALL.keys()
    components_to_install = list(components_to_install)
    for key in list(additional_component_key):
        if (isinstance(key, str) and key in components_to_install) or set(key) < set(components_to_install):
            additional_component = ADDITIONAL_COMPONENT_TO_INSTALL.get(key)
            components_to_install.append(additional_component)
    # remove the duplicates
    return list(set(components_to_install))


if __name__ == '__main__':
    """ Parse input arguments """
    usage = """
%(prog)s [-h] {available,installed,merge,namespace,save} [--always-continue] [--key KEY] [--value VALUE]

Examples:
  python DeployHelper.py available
  python DeployHelper.py installed
  python DeployHelper.py merge
  python DeployHelper.py merge --always-continue
  python DeployHelper.py namespace
  python DeployHelper.py save --key key --value value
    """
    parser = argparse.ArgumentParser(description='Deploy PrePack Helper', usage=usage)
    parser.add_argument(dest='option', choices=['available', 'installed', 'merge', 'namespace', 'save'],
                        help="Option to execute: available, installed, merge, namespace or save")
    parser.add_argument('--always-continue', dest='always_continue', action='store_true', help="Key to skip prompts.")
    parser.add_argument('--key', dest='key', help="Key to be added into products-var-prepack.yaml.")
    parser.add_argument('--value', dest='value', help="Value to be added into products-var-prepack.yaml.")
    args = parser.parse_args()

    OPTION = args.option
    KEY = args.key
    VALUE = args.value
    CONTINUE = args.always_continue

    if OPTION == 'save' and (KEY is None or VALUE is None):
        parser.error("Key/value should be provided while calling \'save\'.")
    elif OPTION == 'available':
        list_available_components()
    elif OPTION == 'installed':
        list_installed_components()
    elif OPTION == 'merge':
        merge_var_files()
    elif OPTION == 'namespace':
        get_namespace_from_shell()
    elif OPTION == 'save':
        set_entry(KEY, VALUE)
