#!/usr/bin/env bash

######################################
# ======      Instructions      ======
# * Before running this script,ensure to have below steps performed:
# 1. cms.var has been updated with below variables and values,
# - use_azure_file_storage = true
# - use_azure_blob_storage = true
# 2. Blow commands have been rerun to update AKS to support Azure Blob usage:
# - bash deploy-cms.sh prepare
# - bash deploy-cms.sh setup-cluster
# 3. Stop CMS services and DB/ES services by below command,
# cmscli service stop;\
# cmscli service stop -s cms-pg-epg-stolon-keeper;\
# cmscli service stop -s cms-pg-metadata-manager-stolon-keeper;\
# cmscli service stop -s cms-pg-workflow-stolon-keeper;\
# cmscli service stop -s cms-es-application-data;\
# cmscli service stop -s cms-es-application-master;\
# cmscli service stop -s cms-es-logging-data;\
# cmscli service stop -s cms-es-logging-master;\
# kubectl scale -n <cms_namespace> sts cms-cli --replicas=0;\
# kubectl scale -n <cms_namespace> sts cms-config --replicas=0;\
# kubectl patch cronjobs cms-common-backup -n <cms_namespace> -p '{"spec" : {"suspend" : true }}'
# 
# * Run this Scripts to migrate files from azure-file to azure-blob:
# bash scripts/migrate_aks_pvc.sh all
# 
# * Run this Scripts to clean temporary PVs during migration:
# bash scripts/migrate_aks_pvc.sh clean
# 
# * Update CMS config and update CMS deployment with azure-blob PVs
# bash deploy-cms.sh init-config
# bash deploy-cms.sh deploy -n cms-pvc -c cms-pvc
# 
# * Start cmscli
# kubectl scale -n <cms_namespace> sts cms-cli --replicas=1;\
# kubectl scale -n <cms_namespace> sts cms-config --replicas=1
# 
# * Start databases
# cmscli service start -s cms-pg-epg-stolon-keeper;\
# cmscli service start -s cms-pg-metadata-manager-stolon-keeper;\
# cmscli service start -s cms-pg-workflow-stolon-keeper
# 
# * Start elasticsearch
# cmscli service start -s cms-es-application-master;\
# cmscli service start -s cms-es-application-data;\
# cmscli service start -s cms-es-logging-master;\
# cmscli service start -s cms-es-logging-data
# 
# * Start CMS services
# cmscli service start
######################################

#set -ex
#set -x
set -e

RUN_AS_CONTAINER=${RUN_AS_CONTAINER:-false}

# Get full path of this script file
SCRIPT=$(readlink -f "$0")
# Get folder path of this script
SCRIPTPATH=$(dirname "$SCRIPT")
PARENTPATH=$(dirname "$SCRIPTPATH")

BUNDLEFOLDER="${PARENTPATH}/ConfigBundle"
PRODUCTSVAR_FILE="${BUNDLEFOLDER}/products-var.yaml"
CMSCONF="$PARENTPATH/cms.var"
CLOUD_ENV="$PARENTPATH/aks.env"

# Define the log file for this scripts
LOG="${PARENTPATH}/logs/$(basename "$0" .sh).log"

##############################
# Include all functions from shell_lib
##############################
for f in ${PARENTPATH}/shell_lib/*.sh; do source $f; done

# Parameters from cms.var
CLOUDPLATFORM=$(get_parameter_value cloud_platform)
if [[ $CLOUDPLATFORM != "AZURE" ]];then
  echo "[ERROR] This scripts is only for Azure AKS PVC migration." |& tee -i -a $LOG
  exit 1
fi

if [[ $CLOUDPLATFORM == "AZURE" ]];then
  TERRAFORM_FOLDER="tf-aks"
  TF_DEFAULT_VARIABLES="$PARENTPATH/$TERRAFORM_FOLDER/variables.tf"
  if $RUN_AS_CONTAINER; then
    CLOUD_ENV="$PARENTPATH/data/aks.env"
  else
    CLOUD_ENV="$PARENTPATH/aks.env"
  fi
fi

#Common parameters

CLUSTER_NAME=$(get_parameter_value cluster_name)
CLOUD_CLUSTER_INSTANCE_NAME="$CLUSTER_NAME-cms-cluster"
CMS_NAMESPACE=$(get_parameter_value cms_namespace)
CLUSTER_ROLE=$(get_parameter_value cluster_role)
INTERNAL_CT_ENABLED=$(get_parameter_value internal_ct_enabled)
CONTENT_NAS_SIZE=$(get_parameter_value content_nas_size)

AZURE_REGION=$(get_parameter_value azure_region)
ACR_RESOURCEGROUP="cmsacr-rg-$AZURE_REGION"

#IMAGE_TAG=$(yq r $PRODUCTSVAR_FILE 'cms_images_tag.cms_common_backup_restore')
IMAGE_TAG=$(yq '.cms_images_tag.cms_common_backup_restore' $PRODUCTSVAR_FILE)
ACR_NAME=$(get_parameter_value acr_name)
IMAGE_REGISTRY=$(get_parameter_value image_registry)

if [[ -z "$IMAGE_REGISTRY" ]] && [[ -z "$ACR_NAME" ]]; then
  echo "[ERROR] Both image_registry and acr_name are empty, make sure one of them has proper value provided in cms.var file."
  exit 1
fi


# Azure specific parameters
AZURE_SUBID=$(get_parameter_value subscription_id)
AZURE_TENANTID=$(get_parameter_value tenant_id)
AZURE_CLIENTID=$(get_parameter_value client_id)
AZURE_CLIENTSECRET=$(get_parameter_value client_secret)

#AZUREFILE_SHARES="cms-content cms-content-heavy cms-metadata-manager-content cms-conf cms-plugin backup-restore cmscli-plugin cms-sync-utils cms-service-client cms-workflow-runtime"
AZUREFILE_SHARES="cms-content cms-metadata-manager-content cms-conf cms-plugin backup-restore cmscli-plugin cms-sync-utils cms-service-client cms-workflow-runtime cms-watchfolder-runtime cms-es-application-snapshot cms-es-logging-snapshot"
AZURE_STORAGE_ACCOUNT_NAME=$(grep OUTPUT_STORAGEACCOUNT_NAME ${CLOUD_ENV} | awk -F '=' '{print $2}')
AZURE_BLOB_STORAGE_ACCOUNT_NAME=$(grep OUTPUT_BLOB_STORAGEACCOUNT_NAME ${CLOUD_ENV} | awk -F '=' '{print $2}')

trap ctrl_c INT
trap 'echo "Error in scripts '$0': error line: $LINENO,error cmd: $BASH_COMMAND" |& tee -i -a $LOG' ERR

start_information_print(){
  echo "[INFO] *********************************************" |& tee -i -a $LOG
  echo "[INFO] Start time : $(date +"%Hh%Mm%Ss%3Nms")" |& tee -i -a $LOG
  echo "[INFO] Run As Container : $RUN_AS_CONTAINER" |& tee -i -a $LOG
  START_TIME=$(date "+%s")
  
  if [[ -z "$IMAGE_REGISTRY" ]];then
    echo "[INFO] Getting Azure ACR information."
    IMAGE_REGISTRY=$(sudo az acr list -g $ACR_RESOURCEGROUP -o table --query [].loginServer|grep -i $ACR_NAME|tail -1)
  fi

  echo "" |& tee -i -a $LOG
  echo "=========== All Scripts Parameters ======================" |& tee -i -a $LOG
  echo "  SCRIPTPATH=$SCRIPTPATH"|& tee -i -a $LOG
  echo "  LOG=$LOG"|& tee -i -a $LOG
  echo "  BUNDLEFOLDER=$BUNDLEFOLDER"|& tee -i -a $LOG
  echo "  CMSCONF=$CMSCONF"|& tee -i -a $LOG
  echo "  CLUSTER_NAME=$CLUSTER_NAME"|& tee -i -a $LOG
  echo "  CLUSTER_ROLE=$CLUSTER_ROLE"|& tee -i -a $LOG
  echo "  INTERNAL_CT_ENABLED=$INTERNAL_CT_ENABLED"|& tee -i -a $LOG
  echo "  IMAGE_REGISTRY=$IMAGE_REGISTRY"|& tee -i -a $LOG
  echo "  ACR_NAME=$ACR_NAME"|& tee -a $LOG
    if $RUN_AS_CONTAINER; then
      if [[ -z "$EXISTING_AKS_RESOURCE_GROUP_NAME" ]]; then
        RESOURCEGROUP_NAME="rg-$CLUSTER_NAME"
      else
        RESOURCEGROUP_NAME="$EXISTING_AKS_RESOURCE_GROUP_NAME"
      fi
    else
    # Ensure to use the correct resource group name
    #Use az to check whether the resource group with legacy resource group name "$CLUSTER_NAME-cms-resourcegroup" exist
    #If exist, use this legacy resource group name, otherwise, use the new formated resource group name as "rg-$CLUSTER_NAME"
      ORG_SCRIPTPATH=$SCRIPTPATH
      SCRIPTPATH=$PARENTPATH
      set_azure_resource_group
      SCRIPTPATH=$ORG_SCRIPTPATH
    fi
  echo "  RESOURCEGROUP_NAME=$RESOURCEGROUP_NAME"|& tee -a $LOG
  echo "  AZURE_STORAGE_ACCOUNT_NAME=$AZURE_STORAGE_ACCOUNT_NAME"|& tee -a $LOG
  echo "  AZURE_BLOB_STORAGE_ACCOUNT_NAME=$AZURE_BLOB_STORAGE_ACCOUNT_NAME"|& tee -a $LOG
  echo "=========================================================" |& tee -i -a $LOG
  echo "" |& tee -i -a $LOG
}

##################
#  Script usage  #
################## 
usage_function() {
    echo "This script is for migrating azure-file dynamic share to static shares."
    echo "----------------------------------------------------------------"
    echo "Usage: $0 <function>"
    echo
    echo "Available Functions:"
    echo "  all         : Migrage all existing azure-file PVCs to static azure-blob shares"
    echo "  clean       : Clean all previous azure-file PVCs after migration completed"
    echo
    echo "Examples:"
    echo "  $0 all"
    echo "  $0 clean"
    echo "  $0 -h"
    echo
}


############################################################
# MAIN
############################################################


if [ "$1" = "all" ] ; then
  user_confirm "Migrating files from Azure-File shares to Azure-Blob shares.
Please confirm before continue:
1. cms.var has been updated with below variables and values,
- use_azure_file_storage = true
- use_azure_blob_storage = true
2. Blow commands have been rerun to update AKS to support Azure Blob usage:
- bash deploy-cms.sh prepare
- bash deploy-cms.sh setup-cluster
"
  start_information_print
  # Prepare Azure Blob PVC and PV for cms shares, not including es-snapshot PVCs and PVs.
  for sharename in $AZUREFILE_SHARES;
  do
    if [[ $sharename =~ "cms-es-" ]];then
      # For cms-es snapshot
      azurefile_claimname=$sharename
      uid=1000
    else
      azurefile_claimname="pvc-$sharename"
      uid=99
    fi
    
    # Create Template PVC and migrate the data only if PVC is existing
    if [[ $(kubectl get -n $CMS_NAMESPACE pvc $azurefile_claimname > /dev/null 2>&1; echo $?) -eq 0 ]];then
    echo "[Info] Creating temporary azure-blob PVC and PV for $sharename."
    storage_size=$(kubectl get -n $CMS_NAMESPACE pvc $azurefile_claimname -o=jsonpath='{.status.capacity.storage}' --request-timeout 60s)

    cat <<EOF | kubectl apply -f -
apiVersion: v1
kind: PersistentVolume
metadata:
  name: pv-pvc-${sharename}-blob
spec:
  capacity:
    storage: ${storage_size}
  accessModes:
    - ReadWriteMany
  persistentVolumeReclaimPolicy: Retain  # If set as "Delete" container would be removed after pvc deletion
  storageClassName: azureblob-nfs-premium
  csi:
    driver: blob.csi.azure.com
    readOnly: false
    volumeHandle: pvc-${sharename}
    volumeAttributes:
      resourceGroup: ${RESOURCEGROUP_NAME}
      storageAccount: ${AZURE_BLOB_STORAGE_ACCOUNT_NAME}
      containerName: pvc-${sharename}
      protocol: nfs
      mountPermissions: '0777'
EOF

    cat <<EOF | kubectl apply -f -
kind: PersistentVolumeClaim
apiVersion: v1
metadata:
  name: pvc-${sharename}-blob
  namespace: ${CMS_NAMESPACE}
spec:
  accessModes:
    - ReadWriteMany
  resources:
    requests:
      storage: ${storage_size}
  volumeName: pv-pvc-${sharename}-blob
  storageClassName: azureblob-nfs-premium
EOF

    echo "[Info] Runnig Job to migrate data from [pvc-$sharename] to [pv-pvc-$sharename-blob]."
    if [[ $(kubectl get -n $CMS_NAMESPACE job migrate-azure-file-pvc-$sharename --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      kubectl delete -n $CMS_NAMESPACE job migrate-azure-file-pvc-$sharename
    fi

    cat <<EOF | kubectl apply -f -
apiVersion: batch/v1
kind: Job
metadata:
  name: migrate-azure-file-pvc-${sharename}
  namespace: ${CMS_NAMESPACE}
  labels:
    job: migrate-azure-file
    app: cms-job
spec:
  backoffLimit: 1
  template:
    metadata:
      labels:
        job: migrate-azure-file
        app: cms-job
    spec:
      restartPolicy: Never
      imagePullSecrets: [{name: cms-image-pull-secret}]
      containers:
      - name: migrate-azure-file
        image: ${IMAGE_REGISTRY}/cms-common-backup-restore:${IMAGE_TAG}
        command: 
          - sh
          - '-c'
          - |
            touch /mnt/old-azure-file/startSyncing.tmp
            rsync -av --no-perms --no-owner --no-group --delete /mnt/old-azure-file/* /mnt/new-azure-blob
            if [[ $? -ne 0 ]];then
              echo "[Error] Sync Failed."
              exit 1
            fi
            sleep 5
            rm -f /mnt/old-azure-file/startSyncing.tmp
            rm -f /mnt/new-azure-blob/startSyncing.tmp
            echo "[Info] Sync Completed."
            exit 0
        resources:
          requests:
            cpu: 200m
            memory: 128Mi
          limits:
            cpu: 500m
            memory: 512Mi
        securityContext:
          allowPrivilegeEscalation: false
          readOnlyRootFilesystem: true
          runAsUser: ${uid}
          runAsGroup: ${uid}
          runAsNonRoot: true
        volumeMounts:
        - mountPath: '/mnt/old-azure-file'
          name: volume-old-azure-file
        - mountPath: '/mnt/new-azure-blob'
          name: volume-new-azure-blob
      volumes:
        - name: volume-old-azure-file
          persistentVolumeClaim:
            claimName: ${azurefile_claimname}
        - name: volume-new-azure-blob
          persistentVolumeClaim:
            claimName: pvc-${sharename}-blob
EOF

    fi
  echo ""
  done
  
  running_amount=10
  for (( i=1; i<=30; i++ ))
  do
    #sleep 5
    sleep_pot 10
    running_amount=$(kubectl get pod -n $CMS_NAMESPACE --request-timeout 60s | grep migrate-azure-file | grep Running | wc -l)
    error_amount=$(kubectl get pod -n $CMS_NAMESPACE --request-timeout 60s | grep migrate-azure-file | grep Error | wc -l)
    if [[ $error_amount -ne 0 ]];then
      echo "[Error] Job failed, check the migrate-azure-file jobs status for details."
      exit 1
    fi
    if [[ $running_amount -eq 0 ]];then
      echo "[Info] All migrate-azure-file jobs completed."
      break
    fi
  done
  if [[ $running_amount -eq 0 ]];then
    echo "[Info] All CMS data have been migrated to new AzureFile static shares."
  else
    echo "[Warning] Checking timeout, migrate-auzre-file jobs still running."
    echo "Use 'kubectl get pod -n $CMS_NAMESPACE | grep migrate-azure-file' command to check status manually."
    echo "Ensure all pods are in Completed status before continue."
    exit 1
  fi

  # TODO: Clean all temp PVCs 

elif [ "$1" == "clean" ] ; then
  user_confirm "Deleting All Old Azure-File PVC! 
Please confirm all data have been migrated to azure-blob shares by command:
- bash scripts/migrate_aks_pvc.sh all
"
  user_confirm "Please confirm cms services have been stopped by below commands:
- cmscli service stop
- cmscli service stop -s cms-pg-epg-stolon-keeper
- cmscli service stop -s cms-pg-metadata-manager-stolon-keeper
- cmscli service stop -s cms-pg-workflow-stolon-keeper
- cmscli service stop -s cms-es-application-data
- cmscli service stop -s cms-es-application-master
- cmscli service stop -s cms-es-logging-data
- cmscli service stop -s cms-es-logging-master
- kubectl scale -n $CMS_NAMESPACE sts cms-cli --replicas=0
- kubectl scale -n $CMS_NAMESPACE sts cms-config --replicas=0
- kubectl patch cronjobs cms-common-backup -n $CMS_NAMESPACE -p '{\"spec\" : {\"suspend\" : true }}'
"
  # Delete all PVC migrate jobs
  kubectl delete job -l job=migrate-azure-file -n $CMS_NAMESPACE --request-timeout 60s

  # Delete all old dynamic PVCs
  for sharename in $AZUREFILE_SHARES;
  do
    if [[ $sharename =~ "cms-es-" ]];then
      azurefile_claimname=$sharename
    else
      azurefile_claimname="pvc-$sharename"
    fi

    if [[ $(kubectl get -n $CMS_NAMESPACE pvc $azurefile_claimname --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      echo "[Info] Deleting old azure-file PVC ${azurefile_claimname}..."
      kubectl delete pvc -n $CMS_NAMESPACE $azurefile_claimname --request-timeout 60s
    fi
    if [[ $(kubectl get -n $CMS_NAMESPACE pvc pvc-$sharename-blob --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      echo "[Info] Deleting temporary azure-blob PVC pvc-${sharename}-blob..."
      kubectl delete pvc -n $CMS_NAMESPACE pvc-$sharename-blob --request-timeout 60s
    fi
    if [[ $(kubectl get pv pv-pvc-$sharename-blob --request-timeout 60s > /dev/null 2>&1; echo $?) -eq 0 ]];then
      echo "[Info] Deleting temporary azure-blob PV pv-pvc-${sharename}-blob..."
      kubectl delete pv pv-pvc-$sharename-blob --request-timeout 60s
    fi  done 
  echo "[Info] All old Azure-file PVCs and temporary Azure-Blob PVCs deleted."
  echo "[Info] Instructions to clean up Azure-file StorageAccount."
  echo " 1. Run below command to delete all Released azure-file PVs:"
  echo "    # kubectl get pv | grep Released | grep $CMS_NAMESPACE | grep azure-file | awk '{print \$1}' | xargs kubectl delete pv"
  echo " 2. Edit cms.var and set use_azure_file_storage = false"
  echo " 3. Run setup-cluster to destory all azure-file storageaccount related Azure components."
  echo "    # bash deploy-cms.sh setup-cluster"

elif [ "$1" == "-h" ] || [ "$1" == "--help" ] ; then
  usage_function

else
  usage_function
fi
