#!/bin/bash


trap 'echo "[Error] Scripts failed when running $0 - error line: $LINENO, error cmd: $BASH_COMMAND"' ERR

source shell_lib/common.sh

TIMESTAMP=$(date --utc +%Y%m%d_%H%M)
SCRIPT=$(readlink -f "$0")
SCRIPTPATH=$(dirname "$SCRIPT")
CMSCONF="$SCRIPTPATH/cms.var"
BUNDLEFOLDER="${SCRIPTPATH}/ConfigBundle"
PRODUCTSVAR_FILE="${BUNDLEFOLDER}/products-var.yaml"

HELM_REPO_PATH="/opt/cms/charts/"
TGZ_FILE=$(ls -lt ${HELM_REPO_PATH}cms-migrate*tgz|awk 'NR==1{print $NF}')
IMAGE_REGISTRY=$(get_product_parameter docker_registry)
CMS_NAMESPACE=$(get_product_parameter cms_namespace)
CMS_ROLE=$(get_product_parameter cms_role)
INTERNAL_CT_ENABLED=$(get_product_parameter internal_ct_enabled)
CLOUD_PLATFORM=$(get_parameter_value cloud_platform)

[[ ! -d /opt/cms/logs/migrate/ ]] && mkdir -p /opt/cms/logs/migrate/

usage(){
  cat<<'END'
This script is used for migrating ES data, DB data and configuration files.
General Arguments:
  -h | --help                       Displays usage
  -t | --type                       es or db or config or post_upgrade. If the value is post_upgrade, the upgraded operation will be performed.
  -a | --all                        Migrate data of config, es, db
  --storageclass                    Specify the storageclass to save the migrate log

elasticsearch argument:
  --esip (required)                 ES IPs of remote cluster for re-indexing ES indices
  -w | --workorder (optional)       Number of days for which workorder history logs to re-index
  -d (optional)                     To compare number of documents on CMS 7.2/8.0 and CMS 9.0
  -o (optional)                     Only reindex history_log indices till previous day

db argument:
  --dbip (required)                 IP of CMS database which migrate from. It should be db1 or db2 node ip in cms7.2 or cms8.0
  -U | --superuser (optional)       Super user name of source CMS database. If not specified, the defalut value(postgres) will be used.
  -P | --superpassword (optional)   Super user password of source CMS database. If not specified, the defalut value(n2bbsys)will be used.
                                    The password string should be double-quoted, for example "password".
                                    If the password string contains dollar sign ($), the dollar sign should be prepended with a backslash (\), for example "pas\$word".
                                    If the password string contains double quotation marks (""), both quotation marks should be prepended with a backslah, for example "pa\"s\$wo\"rd".
  -u | --user (optional)            CMS user name of source CMS database. If not specified, the defalut value(wfs) will be used.
  -p | --password (optional)        CMS user password of source CMS database. If not specified, the defalut value (Wf\$1234) will be used.
                                    The password string should be double-quoted, for example "password".
                                    If the password string contains dollar sign ($), the dollar sign should be prepended with a backslash (\), for example "pas\$word".
                                    If the password string contains double quotation marks (""), both quotation marks should be prepended with a backslah, for example "pa\"s\$wo\"rd".
  -n | --azure_dbname (optional)    Resource name of azure postgres database. It's required when do migration from azure db.
  -F | --force (optional)           Force to execute db migration again when first attempt failed. Defalut value is false.
  -s (optional)                     If you don't need or have done the DB replication manually, use -s to skip the check. Defalut value is false.

configuration argument:
  -f | --file (required)            Backup file
  -c (Optional)                     File used as the restore template


Usage:
  bash migrate-cms.sh -t <type> [<command options>]

END
exit
}

org_arg="$@"
PARSED_ARGUMENTS=$(getopt -a -o ht:U:P:u:p:n:F:f:,w:dsc:ao --long help,type:,superuser:,superpassword:,user:,password:,azure_dbname:,force:,file:,workorder:,esip:,storageclass:,dbip:,all -- "$@")
if [ $? != "0" ]; then
  usage
fi
eval set -- "$PARSED_ARGUMENTS"

while [ ! -z "$1" ];do
  case $1 in
    -a|--all)
      type=all
      shift
      ;;
    -s)
      skip=true
      shift
      ;;
    -o)
      only=true
      shift
      ;;
    -d)
      compare=true
      shift
      ;;
    -h | --help)
      usage
      ;;
    -c)
      restore_template_file=$2
      customized=true
      shift 2
      ;;
    -t | --type)
      type=$2
      shift 2
      ;;
    --dbip)
      dbip=$2
      shift 2
      ;;
    --esip)
      esip=$2
      shift 2
      ;;
    --storageclass)
      storageclass=$2
      shift 2
      ;;
    -U | --superuser)
      superuser=$2
      shift 2
      ;;
    -P | --superpassword)
      superpassword=$2
      shift 2
      ;;
    -u | --user)
      user=$2
      shift 2
      ;;
    -p | --password)
      password=$2
      shift 2
      ;;
    -n | --azure_dbname)
      azure_dbname=$2
      shift 2
      ;;
    -F | --force)
      force_upgrade=$2
      shift 2
      ;;
    -f | --file)
      file=$2
      shift 2
      ;;
    -w | --workorder)
      day=$2
      shift 2
      ;;
    --)
      shift
      break
      ;;
    *)
      echo "[ERROR] Invalid argument detected: $1."
      usage
  esac
done


info_log(){
   echo -e "$(date) [INFO] "$1
}
error_log(){
   echo -e "$(date) [ERROR] "$1
   exit 1
}
warn_log(){
   echo -e "$(date) [WARN] "$1
}

exception(){
  if [[ -z $org_arg ]];then
    usage
  fi

  # all
  if [[ $type == "all" ]];then
    if [[ -z $dbip || -z $esip || -z $file ]];then
      error_log "Parameters error. If you specify the -a|--all parameter, you must pass in the --dbip, --esip, -f parameters"
    fi
    if [[ x"$skip" == "xtrue" || x"$only" == "xtrue" || x"$compare" == "xtrue" || $type != "all" || ! -z $day ]];then
      error_log "Parameters error. If you specify the -a|--all parameter, it cannot take -s, -o, -d, -t, -w parameters."
    fi
  fi

  # type
  if [[ $type != "db" && $type != "es" && $type != "config" && $type != "post_upgrade" && $type != "all" ]];then
    error_log "Type error. Must be db or es or config or post_upgrade or use -a|--all parameter."
  fi

  # post upgrade
  if [[ $type == "post_upgrade" ]];then
    if [[ $(echo "$org_arg"|sed -e 's/-t *post_upgrade//' -e 's/ *//g') ]];then
      error_log "Type error. When type is post_upgrade, it cannot take other parameters."
    fi
  fi

  # ip
  if [[ $type == "es" || $type == "db" ]];then
    if [[ -z $esip && -z $dbip ]];then
        error_log "IP error. If the type is $type, IP must be specified."
    fi
    if [[ -n $dbip && $type == "es" ]];then
        error_log "If the type is $type, it cannot take the --dbip parameter."
    fi
    if [[ -n $esip && $type == "db" ]];then
        error_log "If the type is $type, it cannot take the --esip parameter."
    fi
    if [[ -n $file ]];then
        error_log "If the type is $type, it cannot take the -f parameter."
    fi
    if [[ -n $restore_template_file ]];then
        error_log "If the type is $type, it cannot take the -c parameter."
    fi
  fi


  if [[ $type == "db" || $type == "config" || $type == "post_upgrade" ]];then
    if [[ -n $day ]];then
        error_log "If the type is $type, it cannot take the -w parameter."
    fi
  fi

  if [[ $type == "es" || $type == "config" || $type == "post_upgrade" ]];then
    if echo "$org_arg" | grep -qiP '\-u|\-p';then
      error_log "Redundant parameter. If the type is $type, not necessary to specify the user name or password."
    fi
  fi

  if [[ $type == "config" ]];then
    if [[ -n $esip || -n $dbip ]];then
      error_log "If the type is $type, IP cannot be specified."
    elif [[ -z $file ]];then
      error_log "If the type is $type, backup file must be specified."
    elif [[ ! -f $file ]];then
      error_log "The specified file does not exist."
    fi
  fi
}


helm_install_migrate() {
  if ! helm list --namespace ${CMS_NAMESPACE} |awk '{print $1}' |grep -q -x cms-migrate ; then
    info_log "Deploy the cms-migrate Helm Chart."
    if [[ -z "$storageclass" ]];then
      if [[ "x$CLOUD_PLATFORM" == "xAZURE" ]];then
          helm install cms-migrate --set image.repository=${IMAGE_REGISTRY} --set cms_role=${CMS_ROLE} --set internal_ct_enabled=${INTERNAL_CT_ENABLED} --set migratelog.storageClassName=cms-azure-file -n ${CMS_NAMESPACE} ${TGZ_FILE}
      else
          helm install cms-migrate --set image.repository=${IMAGE_REGISTRY} --set cms_role=${CMS_ROLE} --set internal_ct_enabled=${INTERNAL_CT_ENABLED} -n ${CMS_NAMESPACE} ${TGZ_FILE}
      fi
    else
          helm install cms-migrate --set image.repository=${IMAGE_REGISTRY} --set cms_role=${CMS_ROLE} --set internal_ct_enabled=${INTERNAL_CT_ENABLED} --set migratelog.storageClassName=$storageclass -n ${CMS_NAMESPACE} ${TGZ_FILE}
    fi
    if [[ $? -ne 0 ]];then
      error_log "cms-migrate failed to deploy."
    fi
  fi
  while :;do
    if kubectl get pod -n ${CMS_NAMESPACE}|grep -P 'cms-migrate\s+1/1\s+Running' &>/dev/null; then
      break
    fi
    sleep 5
    ((count++))
    if [[ $count -gt 20 ]];then
      error_log "cms-migrate pod failed to start, please check."
    fi
    info_log "Wait for cms-migrate to start..."
  done
}

db(){
  new_db_arg="$1"
  {
    info_log "Start migrating DB data..."
    reset_global_timer
    reset_function_timer
    bash ./scripts/migrate_db.sh $new_db_arg;
    echo "$?">/tmp/.db.txt
    running_time
    if [[ $(cat /tmp/.db.txt) -ne 0 ]];then
      error_log "DB migrate error, please check error details from log /opt/cms/logs/migrate/cms-db-migrate-$TIMESTAMP.log"
    fi
  } |& tee -i /opt/cms/logs/migrate/cms-db-migrate-$TIMESTAMP.log
  #echo "bash ./scripts/migrate_db.sh $new_db_arg"
}

es(){
  ip="$1"
  other_arg="$2"
  {
    info_log "Start migrating es data..."
    reset_global_timer
    reset_function_timer
    kubectl exec -n $CMS_NAMESPACE cms-migrate -c cms-migrate -- bash -c "cd /opt/tandbergtv/scripts && bash ./migrate-es.sh -e $ip $other_arg";
    echo "$?">/tmp/.es.txt
    running_time
    if [[ $(cat /tmp/.es.txt) -ne 0 ]];then
      error_log "Kubectl command error, please check error details from log /opt/cms/logs/migrate/cms-es-migrate-$TIMESTAMP.log"
    fi
  } |& tee -i /opt/cms/logs/migrate/cms-es-migrate-$TIMESTAMP.log
  #echo "kubectl exec -n $CMS_NAMESPACE cms-migrate -c cms-migrate -- bash -c \"cd /opt/tandbergtv/scripts && bash ./migrate-es.sh -e $ip $other_arg\""
}

config(){
  file=$1
  restore_template_file=$2
  if [[ $customized == "true" ]];then
    if [[ $restore_template_file != "app_only.csv" ]] && [[ $restore_template_file != "ct.csv" ]];then
      kubectl -n $CMS_NAMESPACE cp $restore_template_file cms-migrate:/opt/tandbergtv/scripts/restore_conf/$(basename $restore_template_file)
      [[ $? -eq 0 ]] || error_log "Failed to copy file $restore_template_file to cms-migrate pod."
    fi
  fi
  restore_template_file="restore_conf/$(basename $restore_template_file)"

  {
    info_log "Start migrating config data..."
    reset_global_timer
    reset_function_timer
    #echo "kubectl -n $CMS_NAMESPACE cp $file cms-migrate:/cms_migrate/backup_$genre.tgz && \
    #kubectl exec -n $CMS_NAMESPACE cms-migrate -c cms-migrate -- bash -c \"cd /opt/tandbergtv/scripts && bash ./restore-config.sh -c $restore_template_file -t /cms_migrate/backup_$genre.tgz\" && \
    #kubectl exec -n $CMS_NAMESPACE cms-migrate -- bash -c \"cd /opt/tandbergtv/scripts && ./migrate-config.sh\""
    kubectl -n $CMS_NAMESPACE cp $file cms-migrate:/cms_migrate/backup_$genre.tgz && \
    kubectl exec -n $CMS_NAMESPACE cms-migrate -c cms-migrate -- bash -c "cd /opt/tandbergtv/scripts && bash ./restore-config.sh -c $restore_template_file -t /cms_migrate/backup_$genre.tgz" && \
    kubectl exec -n $CMS_NAMESPACE cms-migrate -- bash -c "cd /opt/tandbergtv/scripts && ./migrate-config.sh";
    echo "$?">/tmp/.config.txt
    running_time
    if [[ $(cat /tmp/.config.txt) -ne 0 ]];then
      error_log "Kubectl command error, please check error details from log /opt/cms/logs/migrate/cms-config-migrate-$TIMESTAMP.log"
    fi
  } |& tee -i /opt/cms/logs/migrate/cms-config-migrate-$TIMESTAMP.log
}

post_upgrade_step(){
  echo
  info_log "Start CMS related services..."
  reset_global_timer
  reset_function_timer
  kubectl exec -i -n $CMS_NAMESPACE cms-cli-0 -c cms-cli -- cmscli service start -a
  #echo
  #info_log "Delete cms-migrate release..."
  #helm delete -n $CMS_NAMESPACE cms-migrate 2>/dev/null
  running_time
  exit
# other post upgrade command
#
}

post_upgrade(){
  post_upgrade_step
}


all_type(){
  # ./migrate-cms.sh -a --dbip 10 -U u -P p -u u -p p -f /etc/passwd --esip 10
  new_db_arg=$(echo "$org_arg"|sed -e 's/-a[[:space:]]\|-a$\|--all[[:space:]]\|--all$//' -e 's/--dbip\|-dbip/-h/' -e 's#-f *[^ ]*\|--file *[^ ]*##' -e 's#--esip *[^ ]*##' -e 's/-c \S*//' -e 's#--storageclass *[^ ]*##')

  beginLogging
  { echo "[INFO] Start Elasticsearch indices migrate, logging file /opt/cms/logs/migrate/cms-es-migrate-$TIMESTAMP.log"; es "$esip" "-w 0" >/dev/null; } &
  { echo "[INFO] Start DB migrate, logging file /opt/cms/logs/migrate/cms-db-migrate-$TIMESTAMP.log"; db "$new_db_arg" >/dev/null; } &
  { echo "[INFO] Start Configuration migrate, logging file /opt/cms/logs/migrate/cms-config-migrate-$TIMESTAMP.log"; config $file $restore_template_file >/dev/null; } &
  sleep 1
  echo "These steps take time to completed, please be patient."

  arr_type=(es config db)
  while :;do
    if [ ${#arr_type[@]} -eq 0 ]; then
      break
    fi
    for type in ${arr_type[*]}; do

      if [[ $type == "config" ]];then
        mytype="Configuration"
        myerror='\[ERROR\]'
      elif [[ $type == "db" ]];then
        mytype="DB"
        myerror='Migrate-DB-Failed'
      elif [[ $type == "es" ]];then
        mytype="Elasticsearch Indices"
        myerror='\[ERROR\]'
      fi

        if grep -q 'Elapsed Time' /opt/cms/logs/migrate/cms-$type-migrate-$TIMESTAMP.log;then

          #if grep -q '\[ERROR\]' /opt/cms/logs/migrate/cms-$type-migrate-$TIMESTAMP.log; then
          if grep -q "$myerror" /opt/cms/logs/migrate/cms-$type-migrate-$TIMESTAMP.log; then
            info="[ERROR] $mytype Migration Failed with error, check error details from log /opt/cms/logs/migrate/cms-$type-migrate-$TIMESTAMP.log"
            eval $type="[Fail]"
          else
            info="[INFO] $mytype Migration Completed."
            eval $type="[Success]"
          fi

          echo
          echo "$info"

          arr_type=($(echo ${arr_type[*]}|sed "s/$type//"))
          break
        else
          sleep 5
          echo -n .
        fi
    done
  done

  echo
  echo "[INFO] All Migration process stopped with below result:"
  cat <<END
DB Migration........................................$db
Elasticsearch Indices Migration.....................$es
Configuration Migration.............................$config

END

  if [[ $db == "[Success]" && $es == "[Success]" && $config == "[Success]" && $customized != "true" ]];then
    post_upgrade
  fi
        
exit
}
  
#if [[ $type == "config" ]];then
if [[ $CMS_ROLE  == "core" ]];then
  if [[ $INTERNAL_CT_ENABLED == "true" ]];then
    genre=app
  else
    genre=app_only
  fi
elif [[ $CMS_ROLE == "externalct" ]];then
  genre=ct
fi
#fi


exception
[[ $type != "post_upgrade" ]] && helm_install_migrate

restore_template_file=${restore_template_file:="restore_conf/$genre.csv"}

if [[ $type == "all" ]];then
  all_type |& tee -a -i /opt/cms/logs/migrate/cms-$type-migrate.log
  echo "Please check the log /opt/cms/logs/migrate/cms-$type-migrate.log for more details."
  exit
fi



case $type in
  db)
    new_db_arg=$(echo "$org_arg"|sed -e 's/--type *db\|-t *db//' -e 's/--dbip\|-dbip/-h/' -e 's#--storageclass *[^ ]*##')
    { beginLogging; db "$new_db_arg"; } |& tee -a -i /opt/cms/logs/migrate/cms-db-migrate.log
    ;;
  es)
    other_arg=$(echo "$org_arg"|sed -e 's/-t *es\|--type *es//' -e 's/--esip\|-esip//'|sed -e "s#$esip##g" -e 's#--storageclass *[^ ]*##')
    { beginLogging; es $esip "$other_arg"; } |& tee -a -i /opt/cms/logs/migrate/cms-es-migrate.log
    ;;
  config)
    { beginLogging; config $file $restore_template_file; } |& tee -a -i /opt/cms/logs/migrate/cms-config-migrate.log
    ;;
  post_upgrade)
    { beginLogging; post_upgrade; } |& tee -a -i /opt/cms/logs/migrate/cms-post_upgrade.log
    exit
    ;;
esac


if [[ $(cat /tmp/.$type.txt) -eq 0 ]];then
  info_log "$type data migration successful.\nPlease check the log /opt/cms/logs/migrate/cms-$type-migrate.log for more details."
else
  error_log "$type data migration failed.\nPlease check the log /opt/cms/logs/migrate/cms-$type-migrate.log for more details."
fi
