resource "azurerm_storage_account" "storageaccount" {
  count =  var.use_azure_file_storage ? 1 : 0
  # The name must conform to the following pattern: '^[a-zA-Z0-9]*$ and less than 24 charaters.
  # This name must be unique across the entire Azure service, not just within the resource group.
  name                     = substr(replace("${var.cluster_name}-storage","-",""),0,23)
  resource_group_name      = azurerm_resource_group.cms_rg.name
  location                 = var.azure_region
  # Add dev_mode for account_tier
  account_tier             = var.account_tier
  account_replication_type = var.account_replication_type
  account_kind             = var.account_kind
  access_tier              = var.access_tier
  #enable_file_encryption   = "false"
  min_tls_version          = var.sa_min_tls_version

  # Need to enable the hns and nfsv3 for using SC azureblob-nfs-premium
  #is_hns_enabled = true
  #nfsv3_enabled  = true

  ##Deny all public connections
  #network_rules {
  #  default_action             = "Deny"
  #  #Do not add allowed IP ranges/ or addresses in the firewall. This is to prevent public IPs from accessing your storage account.
  #  #For details, see: https://aka.ms/storagenetworksecurity.
  #  #ip_rules                   = split(",", (replace(join(",", var.external_whitelist), "/32", "")))
  #  virtual_network_subnet_ids = [azurerm_subnet.cms_subnet.id]
  #}

  dynamic "network_rules" {
    for_each = var.enable_sa_network_rules ? ["yes"] : []
    content {
      default_action             = "Deny"
      virtual_network_subnet_ids = [azurerm_subnet.cms_subnet.id]
    }
  }

  tags = var.custom_tags
}

#No need to define share, AKS PVC will automatically create share whin Azure File Share for each PV.
#resource "azurerm_storage_share" "cmsshare" {
#  name = "cmsshare-${var.cluster_name}"
#  storage_account_name = azurerm_storage_account.storageaccount.name
#  quota = var.storage_share_quota
#}

resource "azurerm_subnet" "cms_storage_pe_subnet" {
  count =  var.use_azure_file_storage ? 1 : 0
  name                 = "${var.cluster_name}-sa-pe-subnet"
  resource_group_name  = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
  virtual_network_name = var.existing_virtual_network_name == "" ? join("",azurerm_virtual_network.cms_network.*.name) : var.existing_virtual_network_name
  address_prefixes     = [var.cms_storage_private_cidr]

  #enforce_private_link_endpoint_network_policies = true
  private_endpoint_network_policies_enabled = true
}

# Private DNS Zone for storage account azure file
resource "azurerm_private_dns_zone" "cms_private_azure_file_zone" {
  count =  var.use_azure_file_storage ? (var.existing_virtual_network_name == "" ? 1 : (var.with_existing_azure_file_private_dns_zone ? 0 : 1)) : 0
  name                = "privatelink.file.core.windows.net"
  resource_group_name  = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
}

resource "azurerm_private_dns_zone_virtual_network_link" "cms_file_private_dns_vnet_link" {
  count =  var.use_azure_file_storage ? (var.existing_virtual_network_name == "" ? 1 : (var.with_existing_azure_file_private_dns_zone ? 0 : 1)) : 0
  name                  = "${var.cluster_name}-file-p-dns-vnet-link"
  depends_on            = [azurerm_private_dns_zone.cms_private_azure_file_zone]
  resource_group_name  = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
  private_dns_zone_name = "privatelink.file.core.windows.net"
  virtual_network_id    = "/subscriptions/${var.subscription_id}/resourceGroups/${var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name}/providers/Microsoft.Network/virtualNetworks/${var.existing_virtual_network_name == "" ? join("",azurerm_virtual_network.cms_network.*.name) : var.existing_virtual_network_name}"
  #registration_enabled  = true
  tags = var.custom_tags
}

# Add the private endpoint as A record into the private dns zone for Azure file in storage account
resource "azurerm_private_dns_a_record" "sa_file_endpoint_a_record" {
  count =  var.use_azure_file_storage ? 1 : 0
  name                = azurerm_storage_account.storageaccount[0].name
  depends_on          = [azurerm_private_dns_zone.cms_private_azure_file_zone]
  resource_group_name = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
  zone_name           = "privatelink.file.core.windows.net"
  ttl                 = 3600
  records             = [azurerm_private_endpoint.cms_2_storageacount[0].private_service_connection[0].private_ip_address]
}

# Private Endpoint for storageaccount connection
resource "azurerm_private_endpoint" "cms_2_storageacount" {
  count =  var.use_azure_file_storage ? 1 : 0
  name                 = "${var.cluster_name}-sa-pe"
  depends_on = [azurerm_storage_account.storageaccount]
  location             = var.azure_region
  resource_group_name  = azurerm_resource_group.cms_rg.name
  subnet_id            = azurerm_subnet.cms_storage_pe_subnet[0].id

  private_service_connection {
    name                           = "${var.cluster_name}-sa-private-connection"
    is_manual_connection           = false
    private_connection_resource_id = azurerm_storage_account.storageaccount[0].id
    subresource_names              = ["file"]
  }

  #private_dns_zone_group {
  #  name                  = "${var.cluster_name}-sa-private-dns-zone"
  #  private_dns_zone_ids  = [azurerm_private_dns_zone.cms_private_azure_file_zone.id]
  #}

  tags = var.custom_tags
}

output "OUTPUT_STORAGEACCOUNT_NAME" {
  depends_on = [azurerm_storage_account.storageaccount]
  value = var.use_azure_file_storage ? (azurerm_storage_account.storageaccount[0].name) : ""
}
output "OUTPUT_STORAGEACCOUNT_ACCESSKEY" {
  depends_on = [azurerm_storage_account.storageaccount]
  value = var.use_azure_file_storage ? (azurerm_storage_account.storageaccount[0].primary_access_key) : ""
  sensitive = true
}
