# this file manages the networks created as part of the deployment
# VM Specific networking is managed in vm.tf and tied to the VMs being created


# Create virtual network
resource "azurerm_virtual_network" "cms_network" {
  count =  var.existing_virtual_network_name == "" ? 1 : 0
  name                = "${var.cluster_name}-network"
  address_space       = [var.vnet_cidr]
  location            = var.azure_region
  resource_group_name = azurerm_resource_group.cms_rg.name

  ## Since one subscription can only have one DDOS plan, we will leave this for user to create and bind to VNet by himself if necessary.
  ## Message="It is not allowed to create more than 1 DDoS protection plan for a subscription in the region."
  #ddos_protection_plan {
  #  id     = azurerm_network_ddos_protection_plan.cms_ddospplan[0].id
  #  enable = true
  #}
  dynamic "ddos_protection_plan" {
    for_each = var.create_ddos_protect_plan ? ["yes"] : []
    content {
      id     = azurerm_network_ddos_protection_plan.cms_ddospplan[0].id
      enable = true
    }
  }

  tags = var.custom_tags
}

# Create DDOS protect plan for Vnet
# Since one subscription can only have one DDOS plan, user need to make the decision on whether to create it or not by scripts.
# Message="It is not allowed to create more than 1 DDoS protection plan for a subscription in the region."
resource "azurerm_network_ddos_protection_plan" "cms_ddospplan" {
  count =  var.existing_virtual_network_name == "" ? (var.create_ddos_protect_plan ? 1 : 0) : 0
  name                = "${var.cluster_name}-ddospplan"
  location            = var.azure_region
  resource_group_name = azurerm_resource_group.cms_rg.name
}

# Create subnet
resource "azurerm_subnet" "cms_subnet" {
  name                 = "${var.cluster_name}-subnet"
  resource_group_name  = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
  virtual_network_name = var.existing_virtual_network_name == "" ? join("",azurerm_virtual_network.cms_network.*.name) : var.existing_virtual_network_name
  address_prefixes     = [var.cms_private_cidr]
  #service_endpoints    = ["Microsoft.Storage","Microsoft.Sql"]
  service_endpoints    = ["Microsoft.Storage"]
  #enforce_private_link_endpoint_network_policies = var.setup_private_aks
  private_endpoint_network_policies_enabled = var.setup_private_aks
}

# NOTE: A Route Table must be configured on this Subnet.
# ERROR: Clusters using managed identity do not support bringing your own route table. Please see https://aka.ms/aks/customrt for more information
resource "azurerm_route_table" "cms_subnet_rt" {
  name                          = "${var.cluster_name}-subnet-route-table"
  location                      = var.azure_region
  resource_group_name           = azurerm_resource_group.cms_rg.name
  disable_bgp_route_propagation = true

  route {
    name           = "${var.cluster_name}-route"
    address_prefix = var.cms_private_cidr
    #next_hop_type  = "vnetlocal"
    next_hop_type  = "VnetLocal"
  }

  dynamic "route" {
    for_each = var.custom_route_table_routes
    content {
      name     = route.value.name
      address_prefix = route.value.address_prefix
      next_hop_type = route.value.next_hop_type
      next_hop_in_ip_address = route.value.next_hop_in_ip_address
    }
  }

  tags = var.custom_tags
}

# Should not set this route table for private AKS
resource "azurerm_subnet_route_table_association" "cms_subnet_rta" {
  count          = var.setup_private_aks ? 0 : 1
  subnet_id      = azurerm_subnet.cms_subnet.id
  route_table_id = azurerm_route_table.cms_subnet_rt.id
}

# Internal VIP?

# External VIP
resource "azurerm_public_ip" "ext_ip" {
  name                = "${var.cluster_name}-ext-ip"
  location            = var.azure_region
  #availability_zone   = join(",", var.availability_zones) == "" ? "No-Zone" : "Zone-Redundant"
  zones = var.availability_zones
  resource_group_name = azurerm_resource_group.cms_rg.name
  allocation_method   = "Static"
  sku                 = var.azure_lb_sku
  domain_name_label   = "ui-ip-${var.cluster_name}"
  tags = var.custom_tags
}

# Public IP for SLB managed outbound IP
resource "azurerm_public_ip" "slb_outbound_ip" {
  name                = "${var.cluster_name}-slb-outbound-ip"
  location            = var.azure_region
  #availability_zone   = join(",", var.availability_zones) == "" ? "No-Zone" : "Zone-Redundant"
  zones = var.availability_zones
  resource_group_name = azurerm_resource_group.cms_rg.name
  allocation_method   = "Static"
  sku                 = var.azure_lb_sku
  domain_name_label   = "slb-outbound-ip-${var.cluster_name}"
  tags = var.custom_tags
}

#data "azurerm_public_ip" "ext_ip" {
#  name                = azurerm_public_ip.ext_ip.name
#  resource_group_name = azurerm_resource_group.cms_rg.name
#}
#
#output "OUTPUT_EXTERNAL_VIP" {
#  value = data.azurerm_public_ip.ext_ip.ip_address
#}

output "OUTPUT_EXTERNAL_VIP" {
  value = azurerm_public_ip.ext_ip.ip_address
}

output "OUTPUT_INTERNAL_VIP" {
  #value = cidrhost(var.cms_private_cidr, var.default_node_count + var.cms_node_count + 10)
  #Use the next to last IP from the private CIDR as the internal VIP
  value = cidrhost(var.cms_private_cidr, pow(2, 32-split("/", var.cms_private_cidr)[1])-2)
}

output "OUTPUT_OUTBOUND_IP" {
  value = azurerm_public_ip.slb_outbound_ip.ip_address
}

# Peer from Bstion host (default) to CMS instances (cms-network)
resource "azurerm_virtual_network_peering" "bastion2cms" {
  count = var.setup_bastion_peering ? 1 : 0
  name                  = "bastion-2-${var.cluster_name}"
  resource_group_name   = var.bastion_resource_group_name
  virtual_network_name  = var.bastion_vnet_name
  #remote_virtual_network_id = var.existing_virtual_network_id == "" ? join("",azurerm_virtual_network.cms_network.*.id) : var.existing_virtual_network_id
  remote_virtual_network_id = var.existing_virtual_network_name == "" ? join("",azurerm_virtual_network.cms_network.*.id) : "/subscriptions/${var.subscription_id}/resourceGroups/${var.existing_vnet_resource_group_name}/providers/Microsoft.Network/virtualNetworks/${var.existing_virtual_network_name}"
}

resource "azurerm_virtual_network_peering" "cms2bastion" {
  count = var.setup_bastion_peering ? 1 : 0
  name                  = "${var.cluster_name}-2-bastion"
  resource_group_name  = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
  virtual_network_name = var.existing_virtual_network_name == "" ? join("",azurerm_virtual_network.cms_network.*.name) : var.existing_virtual_network_name
  #remote_virtual_network_id = var.bastion_vnet_id
  remote_virtual_network_id = "/subscriptions/${var.subscription_id}/resourceGroups/${var.bastion_resource_group_name}/providers/Microsoft.Network/virtualNetworks/${var.bastion_vnet_name}"
}
