#----------------------------------------------------------------------------
# AKS cluster creation
#----------------------------------------------------------------------------

## Private dns zone for private AKS will be created automatically.
## No need to create by terraform scripts.
#resource "azurerm_private_dns_zone" "cms_aks_private_dns" {
#  count = var.setup_private_aks ? 1 : 0
#  #The private dns zone name should be in this format 'privatelink.<region>.azmk8s.io'.
#  #Please refer to https://aka.ms/aks/private-cluster for detail
#  name                = "privatelink.${var.azure_region}.azmk8s.io"
#  resource_group_name = azurerm_resource_group.cms_rg.name
#}
#
#resource "azurerm_private_dns_zone_virtual_network_link" "private_dns_to_bastion" {
#  count = var.setup_private_aks ? (var.add_private_dns_zone_to_bastion ? 1 : 0) : 0
#  name                  = "${var.cluster_name}_private_dns_2_bastion"
#  resource_group_name   = azurerm_resource_group.cms_rg.name
#  private_dns_zone_name = join("",azurerm_private_dns_zone.cms_aks_private_dns.*.name)
#  virtual_network_id    = var.bastion_vnet_id
#}

resource "azurerm_subnet" "cms_aks_api_subnet" {
  count =  var.setup_private_aks ? 1 : 0
  depends_on           = [azurerm_route_table.cms_subnet_rt]
  name                 = "${var.cluster_name}-aks-api-subnet"
  resource_group_name  = var.existing_vnet_resource_group_name == "" ? azurerm_resource_group.cms_rg.name : var.existing_vnet_resource_group_name
  virtual_network_name = var.existing_virtual_network_name == "" ? join("",azurerm_virtual_network.cms_network.*.name) : var.existing_virtual_network_name
  address_prefixes     = [var.cms_aks_api_subnet_cidr == "" ? cidrsubnet(var.vnet_cidr, 4, 15) : var.cms_aks_api_subnet_cidr]

  delegation {
    name = "delegation"
    service_delegation {
      name    = "Microsoft.ContainerService/managedClusters"
      #actions = ["Microsoft.Network/virtualNetworks/subnets/join/action", "Microsoft.Network/virtualNetworks/subnets/prepareNetworkPolicies/action"]
    }
  }
}

resource "azurerm_kubernetes_cluster" "cms_aks" {
  name                = var.cluster_name
  location            = var.azure_region
  resource_group_name = azurerm_resource_group.cms_rg.name
  depends_on          = [azurerm_route_table.cms_subnet_rt]

  dns_prefix          = var.cluster_name
  #node_resource_group = "MC_rg-${var.cluster_name}-cms-cluster_${var.azure_region}"

  default_node_pool {
    name       = "defaultpool"
    node_count = var.default_node_count
    #vm_size    = "Standard_D4s_v3"
    vm_size    = var.default_vm_size
    #availability_zones = var.availability_zones
    zones = var.availability_zones
    #node_labels = {
    #  cms    = "enabled"
    #}
    orchestrator_version = var.k8s_version
    os_disk_size_gb = var.default_disk_size_gb
    vnet_subnet_id = azurerm_subnet.cms_subnet.id

    # Possible values are AvailabilitySet and VirtualMachineScaleSets. Defaults to VirtualMachineScaleSets
    type = "VirtualMachineScaleSets"

    tags = var.custom_tags
  }

  ### To aviod Error: parsing "": cannot parse an empty string for subnet_id in public AKS scenario,
  ### have to use dynamic block to define the api_server_access_profile
  ##private_cluster_enabled = var.setup_private_aks
  ##api_server_access_profile {
  ## authorized_ip_ranges = var.setup_private_aks ? [] : (var.bastion_vm_public_ip == "" ? concat(["${azurerm_public_ip.slb_outbound_ip.ip_address}/32"], var.external_whitelist) : concat(["${var.bastion_vm_public_ip}/32"], ["${azurerm_public_ip.slb_outbound_ip.ip_address}/32"], var.external_whitelist))
  ## vnet_integration_enabled = var.setup_private_aks
  ## subnet_id = var.setup_private_aks ? azurerm_subnet.cms_aks_api_subnet[0].id : ""
  ##}

  # For Private AKS scenario
  dynamic "api_server_access_profile" {
    for_each = var.setup_private_aks ? ["yes"] : []
    content {
      authorized_ip_ranges = []
      vnet_integration_enabled = true
      subnet_id = azurerm_subnet.cms_aks_api_subnet[0].id
    }
  }

  # For Public AKS scenario
  dynamic "api_server_access_profile" {
    for_each = var.setup_private_aks ? [] : ["yes"]
    content {
      authorized_ip_ranges = var.bastion_vm_public_ip == "" ? concat(["${azurerm_public_ip.slb_outbound_ip.ip_address}/32"], var.external_whitelist) : concat(["${var.bastion_vm_public_ip}/32"], ["${azurerm_public_ip.slb_outbound_ip.ip_address}/32"], var.external_whitelist)
      vnet_integration_enabled = false
    }
  }

  #private_dns_zone_id = var.setup_private_aks ? azurerm_private_dns_zone.cms_aks_private_dns[0].id : "None"
  kubernetes_version = var.k8s_version

  #linux_profile{}

  auto_scaler_profile {
    balance_similar_node_groups = true
  }

  network_profile {
    # Currently supported values are azure and kubenet
    network_plugin = "kubenet"
    network_policy = "calico"
    dns_service_ip = cidrhost(var.kubernetes_service_cidr, 10)
    service_cidr = var.kubernetes_service_cidr
    pod_cidr = var.kubernetes_pod_cidr
    #docker_bridge_cidr = var.kubernetes_docker_bridge_cidr
    load_balancer_sku = lower(var.azure_lb_sku)
    load_balancer_profile {
      idle_timeout_in_minutes = 30
      outbound_ip_address_ids = [azurerm_public_ip.slb_outbound_ip.id]
    }
  }

  storage_profile {
    blob_driver_enabled = true
  }

  ### Error: Unsupported block type
  ###
  ###   on cluster.tf line 32, in resource "azurerm_kubernetes_cluster" "cms_aks":
  ###   32:   storage_profile {
  ###
  ### Blocks of type "storage_profile" are not expected here.


  #identity {
  #  type = "SystemAssigned"
  #}

  service_principal {
    client_id = var.client_id
    client_secret = var.client_secret
  }

  #role_based_access_control {
  #  enabled = true
  #  #azure_active_directory {
  #  #  managed = true
  #  #  tenant_id = var.tenant_id
  #  #  azure_rbac_enabled = true
  #  #}
  #}

  azure_policy_enabled = true
  #addon_profile {
  #  azure_policy {
  #    enabled = true
  #  }
  #}

  # sku_tier with value Free or Paid, default is Free.
  sku_tier = var.aks_sku_tier

  tags = var.custom_tags
}

#output "client_certificate" {
#  value = azurerm_kubernetes_cluster.cms_aks.kube_config.0.client_certificate
#}
#
#output "kube_config" {
#  value = azurerm_kubernetes_cluster.cms_aks.kube_config_raw
#}


resource "azurerm_kubernetes_cluster_node_pool" "cms_node_pool" {
  # name must start with a lowercase letter, have max length of 12, and only have characters a-z0-9
  name                  = "cmspool"
  kubernetes_cluster_id = azurerm_kubernetes_cluster.cms_aks.id
  node_count = var.use_spot_vm ? 0 : var.cms_node_count
  enable_auto_scaling = var.use_spot_vm ? false : true
  max_count = var.use_spot_vm ? 0 : (var.cms_node_count + 2)
  min_count = var.use_spot_vm ? 0 : (var.cms_type == "compact" ? 1 : 3)

  #vm_size    = "Standard_D2_v2"
  vm_size    = var.cms_vm_size
  #availability_zones = var.availability_zones
  zones = var.availability_zones

  orchestrator_version = var.cms_nodes_k8s_version == "" ? var.k8s_version : var.cms_nodes_k8s_version
  os_disk_size_gb = var.cms_disk_size_gb

  vnet_subnet_id = azurerm_subnet.cms_subnet.id

  tags = var.custom_tags

  # Update label requires node-pool to be replaced!
  node_labels = {
    cms-common              = "enabled"
    cms                     = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-es-application      = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-db-metadata-manager = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-db-workflow         = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-db-epg              = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-etcd                = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-ct                  = var.cluster_role == "core" ? (var.internal_ct_enabled ? "enabled" : "disabled") : "enabled"
  }
}

resource "azurerm_kubernetes_cluster_node_pool" "cms_spot_node_pool" {
  count = var.use_spot_vm ? 1 : 0
  # name must start with a lowercase letter, have max length of 12, and only have characters a-z0-9
  name                  = "cmsspotpool"
  kubernetes_cluster_id = azurerm_kubernetes_cluster.cms_aks.id

  priority = "Spot"
  # An Eviction Policy can only be configured when priority is set to Spot and will default to Delete unless otherwise specified.
  eviction_policy = "Delete"
  spot_max_price = var.cms_spot_max_price
  #spot_max_price = 0.03611

  node_count = var.use_spot_vm ? (var.cms_type == "compact" ? 1 : 3) : 0
  enable_auto_scaling = true
  max_count = var.use_spot_vm ? ( var.cms_node_count + 2 ) : 0
  min_count = var.use_spot_vm ? (var.cms_type == "compact" ? 1 : 3) : 0

  #vm_size    = "Standard_D2_v2"
  vm_size    = var.cms_spot_vm_size == "" ? var.cms_vm_size : var.cms_spot_vm_size
  #availability_zones = var.availability_zones
  zones = var.availability_zones

  orchestrator_version = var.cms_nodes_k8s_version == "" ? var.k8s_version : var.cms_nodes_k8s_version
  os_disk_size_gb = var.cms_spot_disk_size_gb

  vnet_subnet_id = azurerm_subnet.cms_subnet.id

  tags = var.custom_tags

  # Update label requires node-pool to be replaced!
  node_labels = {
    cms-common              = "enabled"
    cms                     = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-es-application      = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-db-metadata-manager = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-db-workflow         = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-db-epg              = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-etcd                = var.cluster_role == "core" ? "enabled" : "disabled"
    cms-ct                  = var.cluster_role == "core" ? (var.internal_ct_enabled ? "enabled" : "disabled") : "enabled"
    "kubernetes.azure.com/scalesetpriority" = "spot"
  }

  node_taints = [
    "kubernetes.azure.com/scalesetpriority=spot:NoSchedule"
  ]
}

#resource "azurerm_kubernetes_cluster_node_pool" "db_node_pool" {
#  # name must start with a lowercase letter, have max length of 12, and only have characters a-z0-9
#  name                  = "dbpool"
#  kubernetes_cluster_id = azurerm_kubernetes_cluster.cms_aks.id
#  node_count = var.db_node_count
#  #vm_size    = "Standard_D2_v2"
#  vm_size    = var.db_vm_size
#  availability_zones = var.availability_zones
#
#  orchestrator_version = var.k8s_version
#  os_disk_size_gb = var.db_disk_size_gb
#
#  vnet_subnet_id = azurerm_subnet.cms_subnet.id
#
#  tags = {
#    environment = var.tag_environment
#  }
#
#  node_labels = {
#    cms-db = "enabled"
#  }
#}

# Setup virtual_network_link between private dns zone and bastion
# So that bastion vm can resolve the aks's private api server
resource "azurerm_private_dns_zone_virtual_network_link" "dns_to_bastion_link" {
  count = var.setup_private_aks ? (var.setup_bastion_peering ? 1 : 0) : 0
  name                  = "${var.cluster_name}-2-bastion"
  resource_group_name   = azurerm_kubernetes_cluster.cms_aks.node_resource_group
  private_dns_zone_name = join(".", slice(split(".", azurerm_kubernetes_cluster.cms_aks.private_fqdn), 1, length(split(".", azurerm_kubernetes_cluster.cms_aks.private_fqdn))))
  #virtual_network_id    = var.bastion_vnet_id
  virtual_network_id    = "/subscriptions/${var.subscription_id}/resourceGroups/${var.bastion_resource_group_name}/providers/Microsoft.Network/virtualNetworks/${var.bastion_vnet_name}"
}

# Fetch the Network Security Group Resource
##There is terraform bug for using azurerm_resources
##data "azurerm_resources" "aks_nsg" {
##  resource_group_name = azurerm_kubernetes_cluster.cms_aks.node_resource_group
##  type                = "Microsoft.Network/networkSecurityGroups"
##  depends_on          = [azurerm_kubernetes_cluster_node_pool.cms_node_pool]
##}

# Fetch the Network Security Group Name
data "external" "aks_nsg_name" {
  program = [
    "bash",
    "${path.cwd}/scripts/az_get.sh",
    "networkSecurityGroup",
    azurerm_kubernetes_cluster.cms_aks.node_resource_group
  ]
  depends_on = [azurerm_kubernetes_cluster_node_pool.cms_node_pool]
}

resource "azurerm_network_security_rule" "additional_allow_to_8443" {
  count = var.additional_external_cidr_to_8443 == [] ? 0 : 1
  name                        = "additional_allow_to_8443"
  priority                    = 100
  direction                   = "Inbound"
  access                      = "Allow"
  protocol                    = "Tcp"
  source_port_range           = "*"
  destination_port_range      = "8443"
  source_address_prefixes     = var.additional_external_cidr_to_8443
  destination_address_prefix  = azurerm_public_ip.ext_ip.ip_address
  resource_group_name         = azurerm_kubernetes_cluster.cms_aks.node_resource_group
  network_security_group_name = data.external.aks_nsg_name.result.nsg_name
  depends_on          = [azurerm_kubernetes_cluster_node_pool.cms_node_pool]
}

output "OUTPUT_AKS_NODE_GROUP_NAME" {
  value = azurerm_kubernetes_cluster.cms_aks.node_resource_group
}

output "OUTPUT_AKS_PRIVATE_DNS_ZONE_NAME" {
  value = join(".", slice(split(".", azurerm_kubernetes_cluster.cms_aks.private_fqdn), 1, length(split(".", azurerm_kubernetes_cluster.cms_aks.private_fqdn))))
}

output "OUTPUT_AKS_NSG_NAME" {
    #value = data.azurerm_resources.aks_nsg.resources.0.name
    value = data.external.aks_nsg_name.result.nsg_name
}
