#!/usr/bin/env python

__all__ = ['CMSService', 'FilemanagerService', 'WorkflowService', 'GenericService']

import time
import socket
import os

from HttpRequestDecorator import HttpRequestDecorator
from KubernetesConstants import *


class APPService(object):
	CMD_SERVICE_OPERATOR = "kubectl exec cms-cli-0 -n " + CMS_NAMESPACE + " -- cmscli service %(cmd)s -s %(name)s"
	CMD_MONITOR_UP = "kubectl get deploy -n " + CMS_NAMESPACE + " %s --no-headers=true | awk '{print $2}'"
	CMD_MONITOR_DOWN = "kubectl get pods -n " + CMS_NAMESPACE + " -l=app=%s"
	
	def __init__(self, name):
		self.__service_name = name
	
	def start(self):
		command = APPService.CMD_SERVICE_OPERATOR % {'name': self.__service_name, 'cmd': 'start'}
		print os.popen(command, "r").read()
	
	def stop(self):
		command = APPService.CMD_SERVICE_OPERATOR % {'name': self.__service_name, 'cmd': 'stop'}
		print os.popen(command, "r").read()
	
	def restart(self):
		command = APPService.CMD_SERVICE_OPERATOR % {'name': self.__service_name, 'cmd': 'restart'}
		print os.popen(command, "r").read()
	
	def status(self):
		command = APPService.CMD_SERVICE_OPERATOR % {'name': self.__service_name, 'cmd': 'status'}
		print os.popen(command, "r").read()
	
	def waitForServiceUp(self, timeout=300):
		return self._waitForServiceUp()
	
	def _waitForServiceUp(self, url=None, timeout=300, successStatus=['200']):
		if not self.__waitForPodsReady():
			return False
		
		if not url:
			return True
		
		waitTime = 15
		while timeout >= 0:
			try:
				print "Waiting %(serviceName)s [%(url)s] to startup..." \
					% {'serviceName': self.__service_name, 'url': url}
				resp = HttpRequestDecorator.request(url)
				if resp[0]["status"] in successStatus:
					print "%s service is up" % self.__service_name
					return True
			except socket.error:
				pass
			time.sleep(waitTime)
			timeout = timeout - waitTime
		
		print "%s service is still down" % self.__service_name
		return False
	
	def __waitForPodsReady(self, timeout=3600):
		waitTime = 10
		while timeout >= 0:
			readyData = os.popen(APPService.CMD_MONITOR_UP % self.__service_name, "r").read().strip()
			print "Waiting %(serviceName)s pods ready %(readyData)s" \
				% {'serviceName': self.__service_name, 'readyData': readyData}
			if readyData and "/" in readyData:
				readyDatas = readyData.split("/")
				if readyDatas[0] == readyDatas[1]:
					return True
			time.sleep(waitTime)
			timeout = timeout - waitTime
		
		print "Pods of %s is still not ready" % self.__service_name
		return False

	def waitForServiceDown(self, timeout=3600):
		waitTime = 10
		while timeout >= 0:
			query_result = os.popen(APPService.CMD_MONITOR_DOWN % self.__service_name, "r").read().strip()
			# if pods down, query result is ''
			if query_result:
				print "[INFO] Waiting %(serviceName)s pods down:\n %(query_result)s" \
					% {'serviceName': self.__service_name, 'query_result': query_result}
			else:
				return True
			time.sleep(waitTime)
			timeout = timeout - waitTime
		
		print "Pods of %s is still not ready" % self.__service_name
		return False


class CMSService(APPService):
	def __init__(self):
		APPService.__init__(self, "cms-metadata-manager")
	
	def waitForServiceUp(self, timeout=3600):
		url = "https://app:8443/portal"
		return self._waitForServiceUp(url, timeout, ['200', '302'])


class FilemanagerService(APPService):
	def __init__(self):
		APPService.__init__(self, "cms-filemanager")
	
	def waitForServiceUp(self, timeout=270):
		return self._waitForServiceUp(timeout=timeout)


class WorkflowService(APPService):
	def __init__(self):
		APPService.__init__(self, "cms-workflow")
	
	def waitForServiceUp(self, timeout=3600):
		url = "https://app:8443/workflow/services/AdaptorService"
		return self._waitForServiceUp(url, timeout)


class GenericService(APPService):
	def __init__(self, svcname):
		APPService.__init__(self, svcname)


if __name__ == "__main__":
	workflowService = WorkflowService()
	workflowService.restart()
	workflowService.waitForServiceUp()
	
	filemanagerService = FilemanagerService()
	filemanagerService.restart()
	filemanagerService.waitForServiceUp()
