#!/usr/bin/env python

import json
import sys

import time

import BaseClass
from HttpRequestDecorator import HttpRequestDecorator
from SiteConstants import *


class SitesRestMgr(BaseClass.BaseHandler):
	SITE_SERVICE_LOCATION = "https://app:8443/v2/content/sites"
	
	def __init__(self, is_override=False):
		super(SitesRestMgr, self).__init__(isOverride=is_override)
		self.header = {'Accept': 'application/json', 'Content-Type': 'application/json'}
		print "SitesRestMgr is initializing..."
	
	def listSites(self):
		requestUrl = SitesRestMgr.SITE_SERVICE_LOCATION
		return HttpRequestDecorator.request(requestUrl, "GET", None, headers=self.header)
	
	def get_all_sites(self):
		resp = self.listSites()
		if resp[0]["status"] != '200':
			print "[ERROR]", resp
			return False
		
		resp_payload = json.loads(resp[1])
		if resp_payload["resourceList"] == "":
			return False
		else:
			site_info = resp_payload["resourceList"]["site"]
			if type(site_info) is list:
				return site_info
			if type(site_info) is dict:
				return [site_info]
		
		return False
	
	# Return site id
	def checkExistence(self, data):
		siteName = data[0]
		resp = self.listSites()
		if resp[0]["status"] != '200':
			print "[ERROR]", resp
			return False
		
		respPayload = json.loads(resp[1])
		
		if respPayload["resourceList"] == "":
			return False
		else:
			siteInfo = respPayload["resourceList"]["site"]
			
			if type(siteInfo) is list:
				for site in siteInfo:
					if site["name"] == siteName:
						siteHref = site["href"]
						siteId = siteHref[(siteHref.rfind("/") + 1):]
						return siteId
			if type(siteInfo) is dict:
				if siteInfo["name"] == siteName:
					siteHref = siteInfo["href"]
					siteId = siteHref[(siteHref.rfind("/") + 1):]
					return siteId
		
		return False
	
	def getSiteById(self, siteId):
		getSiteByIdUrl = SitesRestMgr.SITE_SERVICE_LOCATION + "/" + siteId
		return HttpRequestDecorator.request(getSiteByIdUrl, "GET", None, headers=self.header)
	
	# data[0]     NAME
	# data[1]     EXTERNALID
	# data[2]     ACTIVE
	# data[3]     DISTRIBUTION_OPTION
	# data[4]     METADATA_FORMAT
	# data[5]     FILE_LOOKUP_KEYS
	# data[6]     DIST_TEMPLATE_NAME
	# data[7]     ALERT_DELAY_PERIOD
	# data[8]     TYPE
	# data[9]     ASSOCIATE_RESOURCE
	# data[10]    PARAMETERS
	# data[11]    PARTIALOFFERDISTRIBUTION
	# data[12]    PARTIAL_DEPENDS_ON_DISTRIBUTION
	# data[13]    DEPENDS_ON_SITE
	# data[14]    MONITORED_RESOURCES
	# data[15]    DISTRIBUTION_STATUS_PLUGIN
	def buildSiteBody(self, siteRecord):
		siteMainBody = {}
		siteMainBody["name"] = siteRecord[0]
		siteMainBody["externalID"] = siteRecord[1]
		siteMainBody["activated"] = siteRecord[2]
		siteMainBody["distributionOption"] = siteRecord[3]
		siteMainBody["metadataFormat"] = siteRecord[4]
		siteMainBody["distributionConfiguration"] = siteRecord[6]
		siteMainBody["alertDelayPeriod"] = siteRecord[7]
		lookupKeysBody = {}
		lookupKeysBody["lookupKey"] = siteRecord[5].split(",")
		siteMainBody["lookupKeys"] = lookupKeysBody
		
		parameters = siteRecord[10]
		entryBody = {}
		entryBodyDict = {}
		for parameter in parameters:
			entryBodyDict["key"] = parameter.get("NAME")
			entryBodyDict["value"] = parameter.get("VALUE")
		entryBody["entry"] = entryBodyDict
		siteMainBody["distConfParams"] = entryBody
		
		dependsOnSites = siteRecord[13]
		if dependsOnSites and len(dependsOnSites) > 0:
			dependsOnSiteList = []
			for dependsOnSite in dependsOnSites:
				dependsOnSiteList.append({"id": dependsOnSite[SITE_ID]})
			siteMainBody["dependsOnSites"] = {"dependsOnSite": dependsOnSiteList}
		
		monitorResourceConfig = siteRecord[14]
		if monitorResourceConfig and len(monitorResourceConfig) > 0 and SITE_RESOURCE in monitorResourceConfig:
			monitorResources = []
			for resourceEntryConfig in monitorResourceConfig[SITE_RESOURCE]:
				resourceBody = {"groupName": resourceEntryConfig[SITE_GROUP_NAME]}
				if SITE_RESOURCE_NAME in resourceEntryConfig:
					resourceBody["resourceName"] = resourceEntryConfig[SITE_RESOURCE_NAME]
				monitorResources.append(resourceBody)
			siteMainBody["monitoredResources"] = {"resource": monitorResources}
		
		siteMainBody["partialOfferDistribution"] = siteRecord[11]
		siteMainBody["partialDependsOnDistribution"] = siteRecord[12]
		siteMainBody["distributionStatusPlugin"] = siteRecord[15]
		
		siteBody = {}
		siteType = siteRecord[8]
		siteBody[siteType] = siteMainBody
		body = json.dumps(siteBody)
		return body
	
	def insert(self, id_, siteRecord):
		body = self.buildSiteBody(siteRecord)
		print "Insert Site str(body): " + str(body)
		
		resp = HttpRequestDecorator.request(SitesRestMgr.SITE_SERVICE_LOCATION, "POST", body, headers=self.header)
		if resp[0]["status"] != '200':
			print "[ERROR]", resp
			sys.exit(1)
		return True
	
	# siteRecord[0]  name
	def update(self, siteRecord):
		siteId = self.checkExistence(siteRecord)
		if not siteId:
			print "Update: the site '%s' doesn't exist!" % siteRecord[0]
			return False
		
		requestUrl = SitesRestMgr.SITE_SERVICE_LOCATION + "/" + str(siteId)
		print "Update Site by Url: " + requestUrl
		
		body = self.buildSiteBody(siteRecord)
		print "Update Site: " + str(body)
		
		resp = HttpRequestDecorator.request(requestUrl, "PUT", body, headers=self.header)
		if resp[0]["status"] != '200':
			print resp
			sys.exit(1)
		return True
	
	# data[0] : site name
	def delete(self, data):
		siteId = self.checkExistence(data)
		if siteId == False:
			print "delete: the site '%s' doesn't exist!" % data[0]
			return True
		
		requestUrl = SitesRestMgr.SITE_SERVICE_LOCATION + "/" + str(siteId)
		print "deleting site by Url: " + requestUrl
		resp = HttpRequestDecorator.request(requestUrl, "DELETE", None, headers=self.header)
		
		if resp[0]["status"] != '204':
			print resp
	
	def set_fields_in_update(self, update_site):
		cms_sites = self.get_all_sites()
		if not cms_sites:
			return None
		for cms_site in cms_sites:
			if update_site[SITE_NAME] == cms_site["name"]:
				cms_site[SITE_NAME] = update_site[SITE_NAME]
				self.__set_fields_for_matched_site(cms_site, update_site, cms_sites)
				return cms_site
	
	@staticmethod
	def __set_fields_for_matched_site(site, update, cms_sites):
		site[SITE_EXTERNALID] = update[SITE_FIELDS_TO_UPDATE][SITE_EXTERNALID] \
			if SITE_EXTERNALID in update[SITE_FIELDS_TO_UPDATE] else site["externalID"]
		
		site[SITE_ACTIVE] = update[SITE_FIELDS_TO_UPDATE][SITE_ACTIVE] \
			if SITE_ACTIVE in update[SITE_FIELDS_TO_UPDATE] else site["activated"]
		
		site[SITE_DISTRIBUTION_OPTION] = update[SITE_FIELDS_TO_UPDATE][SITE_DISTRIBUTION_OPTION] \
			if SITE_DISTRIBUTION_OPTION in update[SITE_FIELDS_TO_UPDATE] else site["distributionOption"]
		
		site[SITE_METADATA_FORMAT] = update[SITE_FIELDS_TO_UPDATE][SITE_METADATA_FORMAT] \
			if SITE_METADATA_FORMAT in update[SITE_FIELDS_TO_UPDATE] else site["metadataFormat"]
		
		site[SITE_FILE_LOOKUP_KEYS] = update[SITE_FIELDS_TO_UPDATE][SITE_FILE_LOOKUP_KEYS] \
			if SITE_FILE_LOOKUP_KEYS in update[SITE_FIELDS_TO_UPDATE] else ",".join(site["lookupKeys"]["lookupKey"])
		
		site[SITE_DIST_TEMPLATE_NAME] = update[SITE_FIELDS_TO_UPDATE][SITE_DIST_TEMPLATE_NAME] \
			if SITE_DIST_TEMPLATE_NAME in update[SITE_FIELDS_TO_UPDATE] else site["distributionConfiguration"]
		
		site[SITE_ALERT_DELAY_PERIOD] = update[SITE_FIELDS_TO_UPDATE][SITE_ALERT_DELAY_PERIOD] \
			if SITE_ALERT_DELAY_PERIOD in update[SITE_FIELDS_TO_UPDATE] else site["alertDelayPeriod"]
		
		site[SITE_TYPE] = update[SITE_FIELDS_TO_UPDATE][SITE_TYPE] \
			if SITE_TYPE in update[SITE_FIELDS_TO_UPDATE] else site["type"]
		
		site[SITE_ASSOCIATE_RESOURCE] = update[SITE_FIELDS_TO_UPDATE][SITE_ASSOCIATE_RESOURCE] \
			if SITE_ASSOCIATE_RESOURCE in update[SITE_FIELDS_TO_UPDATE] else ""
		
		site[SITE_PARTIAL_OFFER_DISTRIBUTION] = update[SITE_FIELDS_TO_UPDATE][SITE_PARTIAL_OFFER_DISTRIBUTION] \
			if SITE_PARTIAL_OFFER_DISTRIBUTION in update[SITE_FIELDS_TO_UPDATE] else site["partialOfferDistribution"]
		
		site[SITE_PARTIAL_DEPENDS_ON_DISTRIBUTION] = update[SITE_FIELDS_TO_UPDATE][SITE_PARTIAL_DEPENDS_ON_DISTRIBUTION] \
			if SITE_PARTIAL_DEPENDS_ON_DISTRIBUTION in update[SITE_FIELDS_TO_UPDATE] else site[
			"partialDependsOnDistribution"]

		if SITE_DISTRIBUTION_STATUS_PLUGIN in update[SITE_FIELDS_TO_UPDATE]:
			site[SITE_DISTRIBUTION_STATUS_PLUGIN] = update[SITE_FIELDS_TO_UPDATE][SITE_DISTRIBUTION_STATUS_PLUGIN]
		else:
			if "distributionStatusPlugin" in site:
				site[SITE_DISTRIBUTION_STATUS_PLUGIN] = site["distributionStatusPlugin"]
			else:
				site[SITE_DISTRIBUTION_STATUS_PLUGIN] = []
		
		# appends site to depends on instead of override
		finalDependsOnSiteIds = []
		if "dependsOnSites" in site and "dependsOnSite" in site["dependsOnSites"]:
			dependsOnSites = site["dependsOnSites"]["dependsOnSite"]
			# the dependsOnSites maybe a dictionary when there is only one depends on site
			if not isinstance(dependsOnSites, list):
				dependsOnSites = [dependsOnSites]
			for dependsOnSite in dependsOnSites:
				finalDependsOnSiteIds.append(dependsOnSite["id"])
		# merge update site configure into site get from CMS
		if SITE_DEPENDS_ON_SITE in update[SITE_FIELDS_TO_UPDATE]:
			for dependEntry in update[SITE_FIELDS_TO_UPDATE][SITE_DEPENDS_ON_SITE]:
				for cms_site in cms_sites:
					if not dependEntry[SITE_NAME] == cms_site["name"]:
						continue
					siteHref = cms_site["href"]
					siteId = siteHref[(siteHref.rfind("/") + 1):]
					finalDependsOnSiteIds.append(int(siteId))
					break
			# remove duplicate id
			finalDependsOnSiteIds = list(set(finalDependsOnSiteIds))
		site[SITE_DEPENDS_ON_SITE] = []
		for dependsOnSiteId in finalDependsOnSiteIds:
			site[SITE_DEPENDS_ON_SITE].append({SITE_ID: dependsOnSiteId})
		
		# appends monitor resources instead of override
		finalMonitorResources = []
		monitoredGroup = set()
		monitoredResource = set()
		if "monitoredResources" in site and "resource" in site["monitoredResources"]:
			site_monitorResources = site["monitoredResources"]["resource"]
			# the site_monitorResources maybe a dictionary when there is only one monitored resource
			if not isinstance(site_monitorResources, list):
				site_monitorResources = [site_monitorResources]
			for monitoredResourceEntry in site_monitorResources:
				monitoredGroup.add(monitoredResourceEntry["groupName"])
				monitorResourceBody = {SITE_GROUP_NAME: monitoredResourceEntry["groupName"]}
				if "resourceName" in monitoredResourceEntry:
					monitoredResource.add(monitoredResourceEntry["resourceName"])
					monitorResourceBody[SITE_RESOURCE_NAME] = monitoredResourceEntry["resourceName"]
				finalMonitorResources.append(monitorResourceBody)
		# merge update site configure into site get from CMS
		if SITE_MONITORED_RESOURCES in update[SITE_FIELDS_TO_UPDATE]:
			for resourceEntry in update[SITE_FIELDS_TO_UPDATE][SITE_MONITORED_RESOURCES][SITE_RESOURCE]:
				if resourceEntry[SITE_GROUP_NAME] in monitoredGroup or \
						(SITE_RESOURCE_NAME in resourceEntry and resourceEntry[SITE_RESOURCE_NAME] in monitoredResource):
					# resource group or resource already in monitor
					continue
				resourceBody = {SITE_GROUP_NAME: resourceEntry[SITE_GROUP_NAME]}
				if SITE_RESOURCE_NAME in resourceEntry:
					resourceEntry[SITE_RESOURCE_NAME] = resourceEntry[SITE_RESOURCE_NAME]
				finalMonitorResources.append(resourceBody)
		site[SITE_MONITORED_RESOURCES] = {SITE_RESOURCE: finalMonitorResources}
		
		if SITE_PARAMETERS in update[SITE_FIELDS_TO_UPDATE]:
			site[SITE_PARAMETERS] = update[SITE_FIELDS_TO_UPDATE][SITE_PARAMETERS]
		else:
			entry = site["distConfParams"].get("entry")
			param_dict = {"NAME": entry.get("key"), "VALUE": entry.get("value")}
			site[SITE_PARAMETERS] = [param_dict]
	
	def backUpExisting(self, args):
		siteId = self.checkExistence(args)
		if siteId == False:
			return False
		
		resp = self.getSiteById(siteId)
		if resp[0]["status"] != '200':
			print resp
			return False
		
		respPayload = json.loads(resp[1])
		for key, value in respPayload.iteritems():
			siteType = key
			sitePayload = value
		lookupKeyList = sitePayload.get("lookupKeys").get("lookupKey")
		
		parametersList = None
		if sitePayload.get("distConfParams"):
			entryDict = sitePayload.get("distConfParams").get("entry")
			parametersList = []
			parametersList.append({"NAME": entryDict["key"], "VALUE": entryDict["value"]})
		
		result = list([sitePayload.get("name"), sitePayload.get("externalID"), sitePayload.get("activated"),
		               sitePayload.get("distributionOption"),
		               sitePayload.get("metadataFormat"), ','.join(lookupKeyList),
		               sitePayload.get("distributionConfiguration"),
		               sitePayload.get("alertDelayPeriod"), siteType, sitePayload.get("associateResource"),
		               parametersList])
		print "backUpExisting: " + str(result)
		return result
	
	def getNextId(self):
		return None
	
	def checkAndWaitSiteServiceAvailable(self, waitForUp=True, interval=5, timeout=60):
		timeToWait = timeout
		
		while waitForUp and timeToWait > 0:
			try:
				resp = HttpRequestDecorator.request(SitesRestMgr.SITE_SERVICE_LOCATION, "GET", None,
				                                    headers=self.header)
				
				if resp[0]["status"] == '200':
					print "Site Service is up."
					return True
				else:
					print "Site Service is not up yet, wait for " + str(interval) + " seconds to check again."
			except Exception:
				pass
			time.sleep(interval)
			timeToWait = timeToWait - interval
		print "Site Service is still not up after " + str(timeout) + " seconds. Please check the network."
		return False


def test(is_override, s):
	site = SitesRestMgr(is_override)
	
	if is_override and SITE_FIELDS_TO_UPDATE in s:  # upgrade
		s = site.set_fields_in_update(s)
		print s
	if s:
		deleteSite = False
		site.handle(s[SITE_NAME], s[SITE_EXTERNALID], s[SITE_ACTIVE], s[SITE_DISTRIBUTION_OPTION],
		            s[SITE_METADATA_FORMAT], s[SITE_FILE_LOOKUP_KEYS], s[SITE_DIST_TEMPLATE_NAME],
		            s[SITE_ALERT_DELAY_PERIOD], s[SITE_TYPE], s[SITE_ASSOCIATE_RESOURCE], s[SITE_PARAMETERS],
		            s[SITE_PARTIAL_OFFER_DISTRIBUTION], s[SITE_PARTIAL_DEPENDS_ON_DISTRIBUTION], s[SITE_DEPENDS_ON_SITE],
		            s[SITE_DISTRIBUTION_STATUS_PLUGIN], deleteSite)


def main():
	site_to_update1 = {
		"NAME": "test bill Site-new site-1",
		"EXTERNALID": "ExternalIngest",
		"ACTIVE": "true",
		"DISTRIBUTION_OPTION": "IMMEDIATE",
		"METADATA_FORMAT": "MSTV2",
		"FILE_LOOKUP_KEYS": "PACKAGE-ORIGINAL,BOXCOVER-ORIGINAL,MOVIE-ORIGINAL,POSTER-ORIGINAL,PREVIEW-ORIGINAL,ENCRYPTED-ORIGINAL",
		"DIST_TEMPLATE_NAME": "Distribution Template",
		"ALERT_DELAY_PERIOD": 60,
		"TYPE": "distributionSite",
		"PARAMETERS": [
			{
				"NAME": "Distribution Profile",
				"VALUE": "dist-mediafirst-onPremise-sample"
			}
		],
		"PARTIALOFFERDISTRIBUTION": True,
		"ASSOCIATE_RESOURCE": "",
		"PARTIALDEPENDSONDISTRIBUTION": "",
		"DEPENDSONSITE": "",
		"DISTRIBUTION_STATUS_PLUGIN": "Media First Distribution Plugin"
	}
	
	site_to_update2 = {
		"NAME": "test bill Site ",
		"FIELDS_TO_UPDATE": {
			"PARTIALOFFERDISTRIBUTION": "true",
			"PARTIALDEPENDSONDISTRIBUTION": "true",
			"DEPENDSONSITE": [
				{
					"NAME": "MKP Secondary"
				},
				{
					"NAME": "MKP Site"
				}
			]
		}
	}

	test(True, site_to_update1)

if __name__ == "__main__":
	main()
