#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jerish Lin
# Created: Dec 11, 2012
# Description:
# Generic PrePackage Deployment Script
#

from InstallComponentCollector import InstallComponentCollector
from InstallConfigObjectConstructor import InstallConfigObjectConstructor
from InstallHelper import InstallHelper
from InstallGeoHelper import InstallGeoHelper
from deploy import Deploy
from Common import Common
import Tee
import getopt
import os
import sys
import traceback
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy
import time
from DeployFileHelper import DeployFileHelper

class Install(Common):
    CUSTOMER_MAPPER_CONFIG = {'43686172746572': 'CUS0001',
                              '43656E747572794C696E6B': 'CUS0002',
                              '43686172746573': 'CUS0003'}

    def __init__(self, mode, customerId=None, geoDomain=None):
        Common.__init__(self)
        self.config["mode"] = mode
        InstallGeoHelper.autoDetectGeoDomain(self.config,geoDomain)
        self.initPolicy()
        self.initParameters()
        self.installComponentCollector = InstallComponentCollector(customerId)
        self.combinedPath = None

    def install(self, preConfiguredFile):
        
        result = self.installComponentCollector.processToCollectAllSelections(preConfiguredFile)
        if not result:
            return False
        
        fullInstallComponents, componentsToInstall, componentsToDelete = result

        configObjectConstructor = InstallConfigObjectConstructor(fullInstallComponents, componentsToInstall, componentsToDelete)
        if configObjectConstructor.isInvolveNothing():
            self.output("There is no operations selected to handle. System would skip the installation.")
            return False
        
        # Start to construct the final config object
        configObject = configObjectConstructor.constructFinalJsonObj(self.COMPONENTS_PATH_PREFIX)
        configObject = self.mergeConfig(configObject, self.config)
        self.combinedPath = self.helper.saveCombinedConfigObject(configObject)

        if not self.deploy(configObject):
            return False

        # Save the installed Components at the end.
        DeployFileHelper.deleteFile(InstallHelper.INSTALLED_COMPONENTS)
        self.installComponentCollector.saveCurrentInstalledComponents(componentsToInstall)
        return True
    
    def deploy(self, configObject):
        self.helper.logActions(InstallHelper.DEPLOY, self.backupVersion, InstallHelper.START)
        result = self.doDeploy(configObject)
        if result:
            self.helper.logActions(InstallHelper.DEPLOY, self.backupVersion, InstallHelper.FINISH)
            DeployFileHelper.deleteFile(InstallHelper.PREPACK_VERSION_FILE)
            self.helper.saveNewPrepackVersion(self.installerVersion)
        return result

    def doDeploy(self, configObject):
        try:
            dep = Deploy(configObject)
            dep.setCombinedConfigJsonPath(self.combinedPath)

            # call deploySecondary when geoDomain is SECONDARY
            if InstallGeoHelper.isGeoSecondary(self.config):
                return dep.deploySecondary()
            elif InstallGeoHelper.isDefaultOrGeoPrimary(self.config):
                if InstallGeoHelper.isGeoEnabled():
                    InstallGeoHelper.excludeFoldersFromGeoSync()
                return dep.deploy()
            else:
                return False
        except Exception:
            print traceback.format_exc()
            return False


###########################################################################
#
# Main
#
###########################################################################
def output(outputstr):
    print outputstr


def getCustomerFeatureId(customerId):
    if customerId == None:
        return None
    
    customerFeatureId = None
    if not os.path.exists(InstallComponentCollector.CUSTOMIZE_COMPONENTS_CONFIG):
        output("[Error] configuration missing: " + InstallComponentCollector.CUSTOMIZE_COMPONENTS_CONFIG + " is not exists!!")
        sys.exit(1)
        
    if Install.CUSTOMER_MAPPER_CONFIG.has_key(customerId):
        customerFeatureId = Install.CUSTOMER_MAPPER_CONFIG[customerId]
    if customerFeatureId == None:
        output("[Error] invalid customer feature Id " + customerId + ", please contact PDU to get the correct customer feature Id!")
        sys.exit(1)
    
    return customerFeatureId


def main():
    print "%s Prepack Installation Start." % time.asctime(time.localtime(time.time()))
    preConfigFile = None
    mode = 'Fresh'
    customerId = None
    geoDomain = None
    try:
        opts, args = getopt.getopt(sys.argv[1:], "h:p:m:c:v:s:", ["help", "preconfigFile", "mode", "customerId", "site"])
    except getopt.GetoptError as err:
        output("install.py [-p preconfigFile]" + str(err))
        sys.exit(2)
    for o, a in opts:
        if o in ("-h", "--help"):
            output('install.py [-p preconfigFile] -m <mode>')
            sys.exit(0)
        elif o in ("-p", "--preconfigFile"):
            preConfigFile = a
        elif o in ("-m", "--mode"):
            mode = a
        elif o in ("-c", "--customer"):
            customerId = a
        elif o in ("-s", "--site"):
            geoDomain = a

    customerFeatureId = getCustomerFeatureId(customerId)

    result = False
    try:
        ins = Install(mode, customerFeatureId, geoDomain)
        result = ins.install(preConfigFile)
    finally:
        if result:
            output('%s Prepack Installation Completed.' % time.asctime(time.localtime(time.time())))
        else:
            output('%s Installation Failed Or Skipped.' % time.asctime(time.localtime(time.time())))
        CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().cleanDatabaseConnection()
        
    if not result:
        sys.exit(2)
    sys.exit(0)


if __name__ == "__main__":
    logFile = Tee.Tee("/opt/tandbergtv/cms/conf/prepack/install.log", "a", True)
    main()
