#!/usr/bin/env bash
#
# Functions of certificates generating/updating
#

#check wether a cert secret already exist and intact
function check_secret_state {
    if [[ -n "$1" ]]; then
	kubectl -n ${CMS_NAMESPACE} get secret --request-timeout 10s $1 1>/dev/null 2>&1
    else
	return 1
    fi
    return $?
}

OPENSSL_VERSION=$(openssl version | cut -d' ' -f2)

#check wether a set of cert secrets already exist and intact
function check_cert_state {
    local _name=$1
    local _result="false"

    if [[ "$_name" == "haproxy-core" ]]; then
	check_secret_state cms-haproxy && \
	check_secret_state cms-trusted-ssl-cert && \
	check_secret_state cms-app-server-cert && \
	check_secret_state cms-ct-ssl && \
	check_secret_state peer-cms-trusted-ssl-cert && _result="true"
    elif [[ "$_name" == "haproxy-externalct" ]]; then
	check_secret_state cms-haproxy && \
	check_secret_state cms-trusted-ssl-cert && \
	check_secret_state cms-app-server-cert && \
	check_secret_state cms-rabbitmq-ssl-certfile && \
	check_secret_state peer-cms-trusted-ssl-cert && _result="true"
    elif [[ "$_name" == "rabbitmq" ]]; then
	check_secret_state cms-rabbitmq-ssl-certfile && _result="true"
    elif [[ "$_name" == "oauth-core" ]]; then
	check_secret_state cms-oauth-token-secret && \
	check_secret_state cms-oauth-token-cert && _result="true"
    elif [[ "$_name" == "oauth-externalct" ]]; then
	check_secret_state cms-oauth-token-cert && _result="true"
    fi

    echo $_result
}

#update cert secrets with annotation: keep
function update_annotation {
    local _name=$1

    if [[ "$_name" == "haproxy-core" ]]; then
	patch_secret_annotation_keep cms-haproxy
	patch_cms_haproxy_extra_data
	patch_secret_annotation_keep cms-trusted-ssl-cert
	patch_secret_annotation_keep peer-cms-trusted-ssl-cert
	patch_secret_annotation_keep cms-app-server-cert
	patch_secret_annotation_keep cms-ct-ssl
    elif [[ "$_name" == "haproxy-externalct" ]]; then
	patch_secret_annotation_keep cms-haproxy
	patch_cms_haproxy_extra_data
	patch_secret_annotation_keep cms-trusted-ssl-cert
	patch_secret_annotation_keep peer-cms-trusted-ssl-cert
	patch_secret_annotation_keep cms-app-server-cert
	patch_secret_annotation_keep cms-rabbitmq-ssl-certfile
    elif [[ "$_name" == "rabbitmq" ]]; then
	patch_secret_annotation_keep cms-rabbitmq-ssl-certfile
    elif [[ "$_name" == "oauth-core" ]]; then
	patch_secret_annotation_keep cms-oauth-token-secret
	patch_secret_annotation_keep cms-oauth-token-cert
    elif [[ "$_name" == "oauth-externalct" ]]; then
	patch_secret_annotation_keep cms-oauth-token-cert
    fi
}

#create empty secret with keep annotation and app:cms-installer
function create_a_secret {
 if [[ -n "$1" ]]; then
  cat <<- EOF | kubectl apply --request-timeout 10s -f -
apiVersion: v1
kind: Secret
metadata:
  annotations:
    helm.sh/resource-policy: keep
  labels:
    app: cms-installer
  name: $1
  namespace: ${CMS_NAMESPACE}
type: Opaque
data:
EOF
 fi
}

#delete a secret, be careful to use this function
function delete_a_secret {
  if [[ -n "$1" ]]; then
  	kubectl -n ${CMS_NAMESPACE} delete secret --request-timeout 10s $1 2>/dev/null
  fi
}

#create empty secrets for generating
function create_empty_secrets {
    local _name=$1

    info "Preparing $_name secrets.."
    if [[ "$_name" == "haproxy-core" ]]; then
	create_a_secret cms-haproxy
	create_a_secret cms-trusted-ssl-cert
	create_a_secret peer-cms-trusted-ssl-cert
	create_a_secret cms-app-server-cert
	create_a_secret cms-ct-ssl
    elif [[ "$_name" == "haproxy-externalct" ]]; then
	create_a_secret cms-haproxy
	create_a_secret cms-trusted-ssl-cert
	create_a_secret peer-cms-trusted-ssl-cert
	create_a_secret cms-app-server-cert
	create_a_secret cms-rabbitmq-ssl-certfile
    elif [[ "$_name" == "rabbitmq" ]]; then
	create_a_secret cms-rabbitmq-ssl-certfile
    elif [[ "$_name" == "oauth-core" ]]; then
	create_a_secret cms-oauth-token-secret
	create_a_secret cms-oauth-token-cert
    elif [[ "$_name" == "oauth-externalct" ]]; then
	create_a_secret cms-oauth-token-cert
    fi
}

#clear all cert secrets generated by cms-installer
function clear_all_cert_secrets {
    info "Clearing cert secrets from namespace: $CMS_NAMESPACE.."

    delete_a_secret cms-haproxy
    delete_a_secret cms-trusted-ssl-cert
    delete_a_secret peer-cms-trusted-ssl-cert
    delete_a_secret cms-app-server-cert
    delete_a_secret cms-ct-ssl
    delete_a_secret cms-rabbitmq-ssl-certfile
    delete_a_secret cms-oauth-token-secret
    delete_a_secret cms-oauth-token-cert
}

#clear and re-create cert dir, return dir name
function prepare_cert_dir {

    local _name=$1
    local _cert_dir=""

    if [[ "$_name" == "haproxy" ]]; then
	_cert_dir="haproxy-certs"
    elif [[ "$_name" == "rabbitmq" ]]; then
	_cert_dir="rabbitmq-ssl-certs"
    elif [[ "$_name" == "oauth" ]]; then
	_cert_dir="oauth-certs"
    elif [[ "$_name" == "cert-gen" ]]; then
	_cert_dir="cms-cert-gen"
    fi

    if [[ -n "$_cert_dir" ]]; then
	rm -rf $_cert_dir
    	mkdir -p $_cert_dir
    fi

    echo $_cert_dir
}

#--------------main function of certificates generating/updating(invoked by init-config)------------#

#generate/update certificates and java keystores of: haproxy, rabbitmq, oauth. This function is not for webhook and elastic certs, they have their logics already.
function cert_gen_update {

  SCRIPTPATH=${SCRIPTPATH:-"/opt/cms"}
  CLUSTER_ROLE=${CLUSTER_ROLE:-"core"}
  CMS_NAMESPACE=${CMS_NAMESPACE:-"mkcms"}

  local _script_path=${SCRIPTPATH:-"/opt/cms"}
  local _temp_dir="/tmp"

  local _cert_days=${CERT_DAYS:-"3650"}
  local _cluster_role=${CLUSTER_ROLE:-"core"}
  local _name_space=${CMS_NAMESPACE:-"mkcms"}
  local _user_provided_pem="false"

  local _bypass_haproxy="false"
  local _bypass_rabbitmq="false"
  local _bypass_oauth="false"

  absent_cert_files="false"
  _absent_haproxy_certs="false"
  _absent_rabbitmq_certs="false"
  _absent_oauth_certs="false"

  cd $_temp_dir

  _cert_gen_dir="$(prepare_cert_dir cert-gen)"
  cd $_cert_gen_dir

  if [[ -f $_script_path/ssl/app.pem ]]; then
	_app_pem_checksum_old="$(get_secret_data cms-haproxy app.pem.checksum)"
	_app_pem_checksum_new="$(cat $_script_path/ssl/app.pem|md5sum|cut -c 1-32|tr -d '\n')"

	if [[ "$_app_pem_checksum_old" == "$_app_pem_checksum_new" ]]; then
		info "User provided app.pem is already updated, rename it to app.pem.updated"
		mv -f $_script_path/ssl/app.pem  $_script_path/ssl/app.pem.updated
	else
		info "User provided new version app.pem"
		_user_provided_pem="true"
	fi
  fi

  if [[ -f $_script_path/ssl/clients.pem ]]; then
	_clients_pem_checksum_old="$(get_secret_data cms-haproxy clients.pem.checksum)"
	_clients_pem_checksum_new="$(cat $_script_path/ssl/clients.pem|md5sum|cut -c 1-32|tr -d '\n')"

	if [[ "$_clients_pem_checksum_old" == "$_clients_pem_checksum_new" ]]; then
		info "User provided clients.pem is already updated, rename it to clients.pem.updated"
		mv -f $_script_path/ssl/clients.pem  $_script_path/ssl/clients.pem.updated
	else
		info "User provided new version clients.pem"
		_user_provided_pem="true"
	fi
  fi

  if [[ "$_cluster_role" == "core" ]]; then
  	_haproxy_mode="haproxy-core"
	_oauth_mode="oauth-core"
  else
  	_haproxy_mode="haproxy-externalct"
	_oauth_mode="oauth-externalct"
  fi

  if [[ "$(check_cert_state $_haproxy_mode)" == "false" ]];then
    #fresh install

    if [[ "$CLUSTER_ROLE" == "externalct" ]]; then
      if [[ ! -f $SCRIPTPATH/ssl/peer/app.pem ]]; then
	warn "File app.pem does not exist in $SCRIPTPATH/ssl/peer during externalct fresh install"
	_absent_haproxy_certs="true"
      fi
      if [[ ! -f $SCRIPTPATH/ssl/peer/clients.pem ]]; then
	warn "File clients.pem does not exist in $SCRIPTPATH/ssl/peer during externalct fresh install"
	_absent_haproxy_certs="true"
      fi
      if [[ ! -f $SCRIPTPATH/ssl/peer/rabbitmq.cacert ]]; then
	warn "File rabbitmq.cacert does not exist in $SCRIPTPATH/ssl/peer during externalct fresh install"
    	_absent_rabbitmq_certs="true"
      fi
      if [[ ! -f $SCRIPTPATH/ssl/peer/rabbitmq.cert ]]; then
	warn "File rabbitmq.cert does not exist in $SCRIPTPATH/ssl/peer during externalct fresh install"
    	_absent_rabbitmq_certs="true"
      fi
      if [[ ! -f $SCRIPTPATH/ssl/peer/rabbitmq.key ]]; then
	warn "File rabbitmq.key does not exist in $SCRIPTPATH/ssl/peer during externalct fresh install"
    	_absent_rabbitmq_certs="true"
      fi
      if [[ ! -f $SCRIPTPATH/ssl/peer/oauth_jwt_publickey.cer ]]; then
	warn "File oauth_jwt_publickey.cer does not exist in $SCRIPTPATH/ssl/peer during externalct fresh install"
	_absent_oauth_certs="true"
      fi
      if [ "$_absent_haproxy_certs" == "true" -o "$_absent_rabbitmq_certs" == "true" -o "$_absent_oauth_certs" == "true" ]; then
	error "Neccessary cert files absented, please copy wanning files above from core cluster to the correct location, aborting.."
	absent_cert_files="true"
	return 1
      fi
    fi

    #prepare empty secrets and cert dir, start generate/update certs
    create_empty_secrets $_haproxy_mode
    _dir_haproxy="$(prepare_cert_dir haproxy)"

    #once generated, should be updated by 'cmscli cert-rotate'
    gen_haproxy_cert $_cert_days $_name_space $_user_provided_pem
  else
    #upgrade install

    if [[ "$_user_provided_pem" == "true" ]]; then
    	_dir_haproxy="$(prepare_cert_dir haproxy)"

	gen_haproxy_cert $_cert_days $_name_space $_user_provided_pem $(get_secret_data cms-trusted-ssl-cert keystore.pass)
	info "For new certificates of haproxy to take effect, please run following commands:\n\t  cmscli service restart -s cms-haproxy\n\t  cmscli service restart"

  	if [[ "$_cluster_role" == "core" ]]; then
	  info "If you have externalct connected, please run:\n\t  cmscli cert-rotate -haproxy\n\tto apply the updated certs to it."
	else
	  info "If you connected to a core cluster, please run:\n\t  cmscli cert-rotate\n\tto apply the updated certs to it."
	fi
    else
    	info "Haproxy certs already exist, bypass cert generation"
    	update_annotation $_haproxy_mode

    	_bypass_haproxy="true"
    fi

  fi

  if [[ "$_cluster_role" == "core" ]]; then
    #core, generate rabbitmq certs
    if [[ "$(check_cert_state rabbitmq)" == "false" ]];then
    	create_empty_secrets rabbitmq
    	_dir_rabbitmq="$(prepare_cert_dir rabbitmq)"

    	#once generated, should be updated by 'cmscli cert-rotate'
    	gen_rabbitmq_cert $_cert_days $_name_space

    	rm -f $_dir_rabbitmq/cert.crt
    	rm -f $_dir_rabbitmq/cacert.crt
    else
    	info "Rabbitmq certs already exist, bypass cert generation"
    	update_annotation rabbitmq

    	_bypass_rabbitmq="true"
    fi

  else
    #externalct, no need to generate rabbitmq certs
    _bypass_rabbitmq="true"
  fi

  if [[ "$(check_cert_state $_oauth_mode)" == "false" ]];then
	#core&ct all need oauth
	create_empty_secrets $_oauth_mode
    	_dir_oauth="$(prepare_cert_dir oauth)"

    	#once generated, should be updated by 'cmscli cert-rotate'
    	gen_oauth_cert $_cert_days $_name_space
  else
    	info "Oauth certs already exist, bypass cert generation"
    	update_annotation $_oauth_mode

    	_bypass_oauth="true"
  fi

  if [[ "$absent_cert_files" == "false" ]]; then
   if [ "$_bypass_haproxy" == "false" -o "$_bypass_rabbitmq" == "false" -o "$_bypass_oauth" == "false" ]; then
    info "Please wait 15 secs for all secrets to be synchronized.."
    sleep 15

    info "Listing generated/updated certs and keystores status:"
    walk_certs
    walk_keystores
   fi
  fi

  #clean tmp dir
  cd $_script_path
  rm -rf $_temp_dir/$_cert_gen_dir
}

##-----------------------Base functions of cert generation-----------------------##
function info {
  echo -e "[INFO] ""$1"
}

function warn {
  echo "[WARN] ""$1"
}

function error {
  echo "[ERROR] ""$1"
}

function show_cert_info {
    cert_info=`openssl x509 -text -in $1 | grep -E "Issuer|DNS" -A3 -B1`
    echo "$cert_info"
}

function show_keystore_info {
    if [[ -n "$2" ]]; then
      keystore_info=`keytool -list -keystore $1 -storepass $2 2>/dev/null`
    else
      keystore_info=`echo -e "\n" | keytool -list -keystore $1 2>/dev/null`
    fi
    echo "$keystore_info"
}

function show_keystore_info_v {
    if [[ -n "$2" ]]; then
      keystore_info=`keytool -v -list -keystore $1 -storepass $2 2>/dev/null`
    else
      keystore_info=`echo -e "\n" | keytool -v -list -keystore $1 2>/dev/null`
    fi
    echo "$keystore_info"
}

function table_head_cert {
  table_split_line
  echo "|Secret:cert                        |Issuer/CN                |Create Time              |Expire Time              |"
  table_split_line
}

function table_head_keystore {
  table_split_line
  echo "|Secret/keystore:entry              |Issuer/CN                |Create Time              |Expire Time              |"
  table_split_line
}

function table_split_line {
  echo "-------------------------------------------------------------------------------------------------------------------"
}

function table_tr_c {
  _name="$1"
  (( ${#_name} > 31 )) && _name="${_name:0:29}.."
  printf "|%-35s|%-25s|%-25s|%-25s|\n" "cms-$_name" "$2" "$3" "$4"
}

function table_tr_k {
  _fname="$2"
  (( ${#_fname} > 33 )) && _fname="${_fname:0:33}.."
  printf "|%-35s|%-25s|%-25s|%-25s|\n" "cms-$1" "" "" ""
  printf "|%-35s|%-25s|%-25s|%-25s|\n" "$_fname" "" "" ""
  printf "|%-35s|%-25s|%-25s|%-25s|\n" "$3" "$4" "$5" "$6"
}

#list certificates in given list
function walk_certs {
  table_head_cert

  _cert_files=`find . -type f | grep -E "(\.pem|\.crt|\.cer)$" | grep -Ev "key\.pem" | grep -Ev "_.+\.pem"`
  for file in $_cert_files
  do
    _dir=`echo $file|awk -F "/" '{print $2}'`
    _fname=`echo $file|awk -F "/" '{print $3}'`

    _info="$(show_cert_info $file)"

    _issuer=`echo "$_info" | grep Issuer|awk -F "=" '{print $2}'| awk -F "," '{print $1}' | sed 's/^\s//g'`
    _create_time=`echo "$_info" | grep "Not Before" |awk -F "Not Before: " '{print $2}'`
    _expire_time=`echo "$_info" | grep "Not After" |awk -F "Not After : " '{print $2}'`

    table_tr_c "$_dir:$_fname" "$_issuer" "$_create_time" "$_expire_time"
  done

  table_split_line
}

#list keystores in given list
function walk_keystores {
  table_head_keystore

  _keystore_files=`find . -type f | grep -E "(\.keystore|\.jks|keystore\.p12|keystorejks|truststorejks)$"`
  for file in $_keystore_files
  do
    _dir=`echo $file|awk -F "/" '{print $2}'`
    _fname=`echo $file|awk -F "/" '{print $3}'`

    _store_pass=`cat $_dir/keystore.pass 2>/dev/null`
    if [[ -z "$_store_pass" ]]; then
	_store_pass=`cat $_dir/default.keystorepass 2>/dev/null`
    fi
    _storepass=`echo $_store_pass | tr -d '\n'`
    if [[ "$_fname" =~ server\.keystore ]]; then
	_storepass="watchpoint-cms"
    fi

    _info="$(show_keystore_info $file $_storepass)"
    _info_v="$(show_keystore_info_v $file $_storepass)"

    _cert=`echo "$_info" | grep -E "CertEntry|PrivateKeyEntry" | awk -F "," '{print $1}'`
    _issuer=`echo "$_info_v" | grep Issuer|awk -F "=" '{print $2}'`
    _create_time=`echo "$_info_v" | grep "Valid from: " | awk -F "from: " '{print $2}' |awk -F " until: " '{print $1}' | cut -c 5-`
    _expire_time=`echo "$_info_v" | grep "Valid from: " | awk -F " until: " '{print $2}' |cut -c 5-`

    if [[ ! -z "$_cert" ]]; then
      table_tr_k "$_dir/" "$_fname" "$_cert" "$_issuer" "$_create_time" "$_expire_time"
      table_split_line
    fi
  done
}

#get value from a secret's data with specific key name
function get_secret_data {
  if [[ -n "$1" ]]; then
    echo -n $(kubectl -n ${CMS_NAMESPACE} get secret --request-timeout 10s $1 -o yaml 2>/dev/null|yq ".data.\"$2\""|base64 -d)
  fi
}

#get value in from a secret's data with specific key name(base64 format)
function get_secret_data_b64 {
  if [[ -n "$1" ]]; then
    echo -n $(kubectl -n ${CMS_NAMESPACE} get secret --request-timeout 10s $1 -o yaml 2>/dev/null|yq ".data.\"$2\"")
  fi
}

#get value from a secret's metadata with specific key name
function get_secret_meta {
  if [[ -n "$1" ]]; then
    echo -n $(kubectl -n ${CMS_NAMESPACE} get secret --request-timeout 10s $1 -o yaml 2>/dev/null|yq ".metadata.$2")
  fi
}

#use kubectl to patch a given secret name, with annotation:keep and app:cms-installer
function patch_secret_annotation_keep {
  if [[ -n "$1" ]]; then
    if [[ "$(get_secret_meta $1 annotations.\"helm.sh/resource-policy\")" =~ keep ]]; then
    	if [[ "$(get_secret_meta $1 labels.app)" =~ cms-installer ]]; then
		true
	else
		kubectl -n ${CMS_NAMESPACE} patch secret --request-timeout 10s $1 \
          	  -p "{\"metadata\": {\"annotations\":{\"helm.sh/resource-policy\":\"keep\"},
				      \"labels\": {\"app\":\"cms-installer\"}
	      	      }}"
	fi
    else 
	kubectl -n ${CMS_NAMESPACE} patch secret --request-timeout 10s $1 \
          -p "{\"metadata\": {\"annotations\":{\"helm.sh/resource-policy\":\"keep\"},
			      \"labels\": {\"app\":\"cms-installer\"}
	      }}"
    fi
  fi
}

#patch secret cms_haproxy to add app.pem.checksum,app.pem.updated,clients.pem.checksum,clients.pem.upfated fields for CMS upgrade install
function patch_cms_haproxy_extra_data {
  if [[ "$(get_secret_data_b64 cms-haproxy app.pem.checksum)" == "null" ]]; then
    kubectl -n ${CMS_NAMESPACE} patch secret --request-timeout 10s cms-haproxy \
	-p "{\"data\":{\"app.pem.checksum\":\"\"}}"
  fi
  if [[ "$(get_secret_data_b64 cms-haproxy app.pem.updated)" == "null" ]]; then
    kubectl -n ${CMS_NAMESPACE} patch secret --request-timeout 10s cms-haproxy \
	-p "{\"data\":{\"app.pem.updated\":\"\"}}"
  fi
  if [[ "$(get_secret_data_b64 cms-haproxy clients.pem.checksum)" == "null" ]]; then
    kubectl -n ${CMS_NAMESPACE} patch secret --request-timeout 10s cms-haproxy \
	-p "{\"data\":{\"clients.pem.checksum\":\"\"}}"
  fi
  if [[ "$(get_secret_data_b64 cms-haproxy clients.pem.updated)" == "null" ]]; then
    kubectl -n ${CMS_NAMESPACE} patch secret --request-timeout 10s cms-haproxy \
	-p "{\"data\":{\"clients.pem.updated\":\"\"}}"
  fi
}

#generate/update haproxy&common certificates and keystores
function gen_haproxy_cert {

  _tmpdir="/tmp/cms-cert-gen/haproxy-certs"
  _secret="cms-haproxy"

  _user_pems_updated="true"
  _self_app_pem="true"
  _self_clients_pem="true"
  _user_provided_app_pem="false"
  _user_provided_clients_pem="false"
  _externalct_connected="false"

  _days=$1
  _namespace=$2
  _user_provided=$3
  _storepass=$4

  if [[ ! -n "$_storepass" ]]; then
  	_storepass=`echo $RANDOM | md5sum | cut -c1-10`
  	_user_pems_updated="false"
  fi

  if [[ "$_user_pems_updated" == "false" ]]; then
    info "Generating haproxy certs"
  else
    info "User provided haproxy certs found, updating"
  fi

  rm -rf $_tmpdir
  mkdir -p $_tmpdir

  #generate private key
  openssl genrsa -out ${_tmpdir}/app.key 2048 &>/dev/null
  openssl genrsa -out ${_tmpdir}/clients.key 2048 &>/dev/null

  #generate certificate
  openssl req -new -x509 -days ${_days} -key ${_tmpdir}/app.key -out ${_tmpdir}/_app.pem -subj "/CN=app"
  openssl req -new -x509 -days ${_days} -key ${_tmpdir}/clients.key -out ${_tmpdir}/_clients.pem -subj "/CN=clientservice"

  #add private key to certificate
  cat ${_tmpdir}/app.key ${_tmpdir}/_app.pem > ${_tmpdir}/app.pem
  cat ${_tmpdir}/clients.key ${_tmpdir}/_clients.pem > ${_tmpdir}/clients.pem

  if [[ ! $? -eq 0 ]]; then
    error "Some error during rotating haproxy certs.."
  fi
  
  #output envs
  APP_PEM_CHECKSUM=""
  CLIENTS_PEM_CHECKSUM=""
  APP_PEM_UPDATED=$(echo -n "true"|base64 -w 0)
  CLIENTS_PEM_UPDATED=$(echo -n "true"|base64 -w 0)

  if [[ "$_user_pems_updated" == "true" ]]; then
  	APP_PEM_UPDATED=$(echo -n "false"|base64 -w 0)
  	CLIENTS_PEM_UPDATED=$(echo -n "false"|base64 -w 0)
  fi

  if [[ "$_user_provided" == "true" ]]; then
  	if [[ -f $SCRIPTPATH/ssl/app.pem ]]; then
		_user_provided_app_pem="true"
		_self_app_pem="false"

		info "Using ssl/app.pem"
		cp -f $SCRIPTPATH/ssl/app.pem ${_tmpdir}/app.pem

		app_key_begin=`cat ${_tmpdir}/app.pem | grep -n "BEGIN .*PRIVATE KEY"| awk -F ":" '{print $1}'`
		app_key_end=`cat ${_tmpdir}/app.pem | grep -n "END .*PRIVATE KEY"| awk -F ":" '{print $1}'`
		sed -n "${app_key_begin},${app_key_end}p" ${_tmpdir}/app.pem > ${_tmpdir}/app.key
		app_cert_begin=`cat ${_tmpdir}/app.pem | grep -n "BEGIN CERTIFICATE"| awk -F ":" '{print $1}'`
		app_cert_end=`cat ${_tmpdir}/app.pem | grep -n "END CERTIFICATE"| awk -F ":" '{print $1}'`
		sed -n "${app_cert_begin},${app_cert_end}p" ${_tmpdir}/app.pem > ${_tmpdir}/_app.pem

  		APP_PEM_CHECKSUM=$(cat ${_tmpdir}/app.pem|md5sum|cut -c 1-32|tr -d '\n'|base64 -w 0)
  		APP_PEM_UPDATED=$(echo -n "true"|base64 -w 0)
  	fi
  	if [[ -f $SCRIPTPATH/ssl/clients.pem ]]; then
		_user_provided_clients_pem="true"
		_self_clients_pem="false"

		info "Using ssl/clients.pem"
		cp -f $SCRIPTPATH/ssl/clients.pem ${_tmpdir}/clients.pem

		#clients_key_begin=`cat ${_tmpdir}/clients.pem | grep -n "BEGIN .*PRIVATE KEY"| awk -F ":" '{print $1}'`
		#clients_key_end=`cat ${_tmpdir}/clients.pem | grep -n "END .*PRIVATE KEY"| awk -F ":" '{print $1}'`
		#sed -n "${clients_key_begin},${clients_key_end}p" ${_tmpdir}/clients.pem > ${_tmpdir}/clients.key
		#clients_cert_begin=`cat ${_tmpdir}/clients.pem | grep -n "BEGIN CERTIFICATE"| awk -F ":" '{print $1}'`
		#clients_cert_end=`cat ${_tmpdir}/clients.pem | grep -n "END CERTIFICATE"| awk -F ":" '{print $1}'`
		#sed -n "${clients_cert_begin},${clients_cert_end}p" ${_tmpdir}/clients.pem > ${_tmpdir}/_clients.pem

		CLIENTS_PEM_CHECKSUM=$(cat ${_tmpdir}/clients.pem|md5sum|cut -c 1-32|tr -d '\n'|base64 -w 0)
  		CLIENTS_PEM_UPDATED=$(echo -n "true"|base64 -w 0)
  	fi
  fi

  APP_PEM=$(cat $_tmpdir/app.pem|base64 -w 0)
  CLIENTS_PEM=$(cat $_tmpdir/clients.pem|base64 -w 0)
  STORE_PASS=$(echo -n "$_storepass"|base64 -w 0)

  echo "$_storepass" > $_tmpdir/keystore.pass
  #update related java keystore: trusted-ssl-cert:cms_truststore.jks
  keytool -importcert -alias app_cert -keystore ${_tmpdir}/trusted-ssl-certs-cms_truststore.jks \
	-storetype jks -file ${_tmpdir}/_app.pem -storepass $_storepass -noprompt

  TRUSTED_SSL_TRUSTSTORE_JKS=$(cat $_tmpdir/trusted-ssl-certs-cms_truststore.jks|base64 -w 0)

  #update related java keystore: trusted-ssl-cert:client_auth_keystore.jks

  if [[ "$OPENSSL_VERSION" =~ ^3\. ]]; then
        openssl pkcs12 -export -out ${_tmpdir}/clients.pfx -in ${_tmpdir}/clients.pem -passout pass:$_storepass -legacy
  else
        openssl pkcs12 -export -out ${_tmpdir}/clients.pfx -in ${_tmpdir}/clients.pem -passout pass:$_storepass
  fi

  keytool -importkeystore \
	-srckeystore ${_tmpdir}/clients.pfx -srcstoretype PKCS12 \
	-destkeystore ${_tmpdir}/trusted-ssl-certs-client_auth_keystore.jks -deststoretype JKS \
	-srcstorepass $_storepass -deststorepass $_storepass -noprompt 2>/dev/null

  TRUSTED_SSL_CLIENTAUTHSTORE_JKS=$(cat $_tmpdir/trusted-ssl-certs-client_auth_keystore.jks|base64 -w 0)

  #update trusted-ssl-cert:cms_fingerprint
  keytool -list -storepass $_storepass \
	-keystore ${_tmpdir}/trusted-ssl-certs-client_auth_keystore.jks -v 2>/dev/null \
       	|grep "Certificate fingerprints" -A3| grep SHA1 | awk -F "SHA1: " '{print $2}'|sed 's/\://g' \
	|xargs -i printf "NeptuneComponent:{}:TrustedClient" \
	|sha1sum | awk '{printf $1}' > ${_tmpdir}/trusted-ssl-certs-cms_fingerprint

  TRUSTED_SSL_FINGERPRINT=$(cat $_tmpdir/trusted-ssl-certs-cms_fingerprint|base64 -w 0)

  #update related java keystore: peer-cms-trusted-ssl-cert:cms_truststore.jks
  #_storepass=`cat peer-cms-trusted-ssl-cert/keystore.pass | tr -d '\n'`
  keytool -importcert -alias app_cert -keystore ${_tmpdir}/peer-cms-trusted-ssl-cert-cms_truststore.jks \
	-storetype jks -file ${_tmpdir}/_app.pem -storepass $_storepass -noprompt

  #update related java keystore: peer-cms-trusted-ssl-cert:client_auth_keystore.jks
  keytool -importkeystore \
	-srckeystore ${_tmpdir}/clients.pfx -srcstoretype PKCS12 \
	-destkeystore ${_tmpdir}/peer-cms-trusted-ssl-cert-client_auth_keystore.jks -deststoretype JKS \
	-srcstorepass $_storepass -deststorepass $_storepass -noprompt 2>/dev/null

  #update peer-cms-trusted-ssl-cert:cms_fingerprint
  keytool -list -storepass $_storepass \
       -keystore ${_tmpdir}/peer-cms-trusted-ssl-cert-client_auth_keystore.jks -v 2>/dev/null \
       |grep "Certificate fingerprints" -A3| grep SHA1 | awk -F "SHA1: " '{print $2}'|sed 's/\://g' \
       |xargs -i printf "NeptuneComponent:{}:TrustedClient" \
       |sha1sum | awk '{printf $1}' > ${_tmpdir}/peer-cms-trusted-ssl-cert-cms_fingerprint

  if [ "$CLUSTER_ROLE" == "externalct" -a "$_user_pems_updated" == "false" ]; then
    if [[ -f $SCRIPTPATH/ssl/peer/app.pem ]]; then
	info "Using ssl/peer/app.pem to gen peer-cms-trusted-ssl-cert:cms_truststore.jks"

	cp -f $SCRIPTPATH/ssl/peer/app.pem ${_tmpdir}/peer_app_pem
	peer_app_key_begin=`cat ${_tmpdir}/peer_app_pem|grep -n "BEGIN .*PRIVATE KEY"| awk -F ":" '{print $1}'`
	peer_app_key_end=`cat ${_tmpdir}/peer_app_pem|grep -n "END .*PRIVATE KEY"| awk -F ":" '{print $1}'`
	sed -n "${peer_app_key_begin},${peer_app_key_end}p" ${_tmpdir}/peer_app_pem > ${_tmpdir}/peer_app_key
	peer_app_cert_begin=`cat ${_tmpdir}/peer_app_pem | grep -n "BEGIN CERTIFICATE"| awk -F ":" '{print $1}'`
	peer_app_cert_end=`cat ${_tmpdir}/peer_app_pem | grep -n "END CERTIFICATE"| awk -F ":" '{print $1}'`
	sed -n "${peer_app_cert_begin},${peer_app_cert_end}p" ${_tmpdir}/peer_app_pem > ${_tmpdir}/peer_app_cert

	#re-gen peer cms_truststore.jks using user provided peer/app.pem instead
	rm -f ${_tmpdir}/peer-cms-trusted-ssl-cert-cms_truststore.jks
  	keytool -importcert -alias app_cert -keystore ${_tmpdir}/peer-cms-trusted-ssl-cert-cms_truststore.jks \
	  -storetype jks -file ${_tmpdir}/peer_app_cert -storepass $_storepass -noprompt
    fi
    if [[ -f $SCRIPTPATH/ssl/peer/clients.pem ]]; then
	info "Using ssl/peer/clients.pem to gen peer-cms-trusted-ssl-cert:client_auth_keystore.jks"

	cp -f $SCRIPTPATH/ssl/peer/clients.pem ${_tmpdir}/peer_clients_pem
	PEER_CLIENTS_PEM=$(cat $_tmpdir/peer_clients_pem|base64 -w 0)
	#peer_clients_key_begin=`cat ${_tmpdir}/peer_clients_pem|grep -n "BEGIN .*PRIVATE KEY"| awk -F ":" '{print $1}'`
	#peer_clients_key_end=`cat ${_tmpdir}/peer_clients_pem|grep -n "END .*PRIVATE KEY"| awk -F ":" '{print $1}'`
	#sed -n "${peer_clients_key_begin},${peer_clients_key_end}p" ${_tmpdir}/peer_clients_pem > ${_tmpdir}/peer_clients_key
	#peer_clients_cert_begin=`cat ${_tmpdir}/peer_clients_pem | grep -n "BEGIN CERTIFICATE"| awk -F ":" '{print $1}'`
	#peer_clients_cert_end=`cat ${_tmpdir}/peer_clients_pem | grep -n "END CERTIFICATE"| awk -F ":" '{print $1}'`
	#sed -n "${peer_clients_cert_begin},${peer_clients_cert_end}p" ${_tmpdir}/peer_clients_pem > ${_tmpdir}/peer_clients_cert

	#re-gen peer client_auth_keystore.jks&fingerprint using user provided peer/app.pem instead
	rm -f ${_tmpdir}/peer-cms-trusted-ssl-cert-client_auth_keystore.jks
	rm -f ${_tmpdir}/peer-cms-trusted-ssl-cert-cms_fingerprint
    if [[ "$OPENSSL_VERSION" =~ ^3\. ]]; then
          openssl pkcs12 -export -out ${_tmpdir}/peer_clients_pfx -in ${_tmpdir}/peer_clients_pem -passout pass:$_storepass -legacy
    else
          openssl pkcs12 -export -out ${_tmpdir}/peer_clients_pfx -in ${_tmpdir}/peer_clients_pem -passout pass:$_storepass
    fi
  	keytool -importkeystore \
	  -srckeystore ${_tmpdir}/peer_clients_pfx -srcstoretype PKCS12 \
	  -destkeystore ${_tmpdir}/peer-cms-trusted-ssl-cert-client_auth_keystore.jks -deststoretype JKS \
	  -srcstorepass $_storepass -deststorepass $_storepass -noprompt 2>/dev/null
  	keytool -list -storepass $_storepass \
          -keystore ${_tmpdir}/peer-cms-trusted-ssl-cert-client_auth_keystore.jks -v 2>/dev/null \
          |grep "Certificate fingerprints" -A3| grep SHA1 | awk -F "SHA1: " '{print $2}'|sed 's/\://g' \
          |xargs -i printf "NeptuneComponent:{}:TrustedClient" \
          |sha1sum | awk '{printf $1}' > ${_tmpdir}/peer-cms-trusted-ssl-cert-cms_fingerprint
    fi
    if [[ "$_absent_rabbitmq_certs" == "false" ]]; then
	update_rabbitmq_ssl_cert_ct "$(cat $SCRIPTPATH/ssl/peer/rabbitmq.key)" "$(cat $SCRIPTPATH/ssl/peer/rabbitmq.cert)" "$(cat $SCRIPTPATH/ssl/peer/rabbitmq.cacert)" $_namespace
    fi
  fi

  PEER_CMS_TRUSTED_SSL_TRUSTSTORE_JKS=$(cat $_tmpdir/peer-cms-trusted-ssl-cert-cms_truststore.jks|base64 -w 0)
  PEER_CMS_TRUSTED_SSL_CLIENTAUTHSTORE_JKS=$(cat $_tmpdir/peer-cms-trusted-ssl-cert-client_auth_keystore.jks|base64 -w 0)
  PEER_CMS_TRUSTED_SSL_FINGERPRINT=$(cat $_tmpdir/peer-cms-trusted-ssl-cert-cms_fingerprint|base64 -w 0)

  if [[ "$CLUSTER_ROLE" == "core" ]]; then
    #update related java keystore: ct-ssl:default.cms_truststorejks
    #_storepass=`cat ct-ssl/default.keystorepass |tr -d '\n'`
    keytool -importcert -alias app_cert -keystore ${_tmpdir}/ct-ssl-default.cms_truststorejks \
        -storetype jks -file ${_tmpdir}/_app.pem -storepass $_storepass -noprompt

    CT_SSL_TRUSTSTORE_JKS=$(cat $_tmpdir/ct-ssl-default.cms_truststorejks|base64 -w 0)

    #update related java keystore: ct-ssl:default.client_auth_keystorejks
    keytool -importkeystore \
        -srckeystore ${_tmpdir}/clients.pfx -srcstoretype PKCS12 \
        -destkeystore ${_tmpdir}/ct-ssl-default.client_auth_keystorejks -deststoretype jks \
        -srcstorepass $_storepass -deststorepass $_storepass -noprompt 2>/dev/null

    CT_SSL_CLIENTAUTHSTORE_JKS=$(cat $_tmpdir/ct-ssl-default.client_auth_keystorejks|base64 -w 0)
  fi

  #update related java keystore: cns-app-server-cert:server.keystore
  _app_pass="watchpoint-cms"
  keytool -genkeypair -keystore ${_tmpdir}/app-server-cert-server.keystore \
	-keyalg RSA -keysize 2048 -storepass $_app_pass -validity ${_days} \
	-keypass $_app_pass -dname "CN=watchpoint-app" -alias "watchpoint-app" 2>/dev/null

  APP_SERVER_CERT_JKS=$(cat $_tmpdir/app-server-cert-server.keystore|base64 -w 0)

  #patch secrets
  if [ "$_user_provided" == "true" -a "$_user_pems_updated" == "true" ]; then
	#update user provided certs only

  	if [[ "$_user_provided_app_pem" == "true" ]]; then
		#update app.pem in secret cms-haproxy
  		kubectl -n ${_namespace} patch secret ${_secret} \
   	  	  -p "{\"data\":{\"app.pem\":\"${APP_PEM}\", \"app.pem.checksum\":\"${APP_PEM_CHECKSUM}\", \"app.pem.updated\":\"${APP_PEM_UPDATED}\"}}"

  		#patch secret cms-trusted-ssl-cert
  		kubectl -n ${_namespace} patch secret cms-trusted-ssl-cert \
  	  	  -p "{\"data\":{\"cms_truststore.jks\":\"${TRUSTED_SSL_TRUSTSTORE_JKS}\", \"keystore.pass\":\"${STORE_PASS}\"}}"

  		if [[ "$CLUSTER_ROLE" == "core" ]]; then
  		  #patch secret cms-ct-ssl
		  kubectl -n ${_namespace} patch secret cms-ct-ssl \
	  	    -p "{\"data\":{\"default.cms_truststorejks\":\"${CT_SSL_TRUSTSTORE_JKS}\", \"default.keystorepass\":\"${STORE_PASS}\"}}"

		  if [[ "$(kubectl -n ${CMS_NAMESPACE} get configmap cms-ct-info -o yaml| yq ".data.\"cms-ct-info.json\""| grep type| grep externalct 1>/dev/null && echo -n "true")" != "true" ]]; then
  		  	#if no externalct connected, patch secret peer-cms-trusted-ssl-cert
  		  	kubectl -n ${_namespace} patch secret peer-cms-trusted-ssl-cert \
  	  	    	  -p "{\"data\":{\"cms_truststore.jks\":\"${PEER_CMS_TRUSTED_SSL_TRUSTSTORE_JKS}\", \"keystore.pass\":\"${STORE_PASS}\"}}"
		  else
			_externalct_connected="true"
		  fi
		fi

  		#patch secret cms-app-server-cert
  		kubectl -n ${_namespace} patch secret cms-app-server-cert \
  	  	  -p "{\"data\":{\"server.keystore\":\"${APP_SERVER_CERT_JKS}\"}}"
	fi
  	if [[ "$_user_provided_clients_pem" == "true" ]]; then
		#update clients.pem in secret cms-haproxy
  		kubectl -n ${_namespace} patch secret ${_secret} \
   	  	  -p "{\"data\":{\"clients.pem\":\"${CLIENTS_PEM}\", \"clients.pem.checksum\":\"${CLIENTS_PEM_CHECKSUM}\", \"clients.pem.updated\":\"${CLIENTS_PEM_UPDATED}\"}}"

  		#patch secret cms-trusted-ssl-cert
  		kubectl -n ${_namespace} patch secret cms-trusted-ssl-cert \
  	  	  -p "{\"data\":{\"client_auth_keystore.jks\":\"${TRUSTED_SSL_CLIENTAUTHSTORE_JKS}\", \"cms_fingerprint\":\"${TRUSTED_SSL_FINGERPRINT}\", \"keystore.pass\":\"${STORE_PASS}\"}}"

  		if [[ "$CLUSTER_ROLE" == "core" ]]; then
  		  #patch secret cms-ct-ssl
		  kubectl -n ${_namespace} patch secret cms-ct-ssl \
	  	    -p "{\"data\":{\"default.client_auth_keystorejks\":\"${CT_SSL_CLIENTAUTHSTORE_JKS}\", \"default.keystorepass\":\"${STORE_PASS}\"}}"

		  if [[ "$(kubectl -n ${CMS_NAMESPACE} get configmap cms-ct-info -o yaml| yq ".data.\"cms-ct-info.json\""| grep type| grep externalct 1>/dev/null && echo -n "true")" != "true" ]]; then
  		  	#if no externalct connected, patch secret peer-cms-trusted-ssl-cert
  		  	kubectl -n ${_namespace} patch secret peer-cms-trusted-ssl-cert \
  	  	    	  -p "{\"data\":{\"client_auth_keystore.jks\":\"${PEER_CMS_TRUSTED_SSL_CLIENTAUTHSTORE_JKS}\", \"cms_fingerprint\":\"${PEER_CMS_TRUSTED_SSL_FINGERPRINT}\", \"keystore.pass\":\"${STORE_PASS}\"}}"
		  else
			_externalct_connected="true"
		  fi
		fi
	fi
  else
	#generate all secrets for fresh install

  	#patch secret cms-haproxy
  	kubectl -n ${_namespace} patch secret ${_secret} \
   	  -p "{\"data\":{\"app.pem\":\"${APP_PEM}\", \"app.pem.checksum\":\"${APP_PEM_CHECKSUM}\", \"app.pem.updated\":\"${APP_PEM_UPDATED}\", \"clients.pem\":\"${CLIENTS_PEM}\", \"clients.pem.checksum\":\"${CLIENTS_PEM_CHECKSUM}\", \"clients.pem.updated\":\"${CLIENTS_PEM_UPDATED}\"}}"

  	#patch secret cms-trusted-ssl-cert
  	kubectl -n ${_namespace} patch secret cms-trusted-ssl-cert \
  	  -p "{\"data\":{\"cms_truststore.jks\":\"${TRUSTED_SSL_TRUSTSTORE_JKS}\", \"client_auth_keystore.jks\":\"${TRUSTED_SSL_CLIENTAUTHSTORE_JKS}\", \"cms_fingerprint\":\"${TRUSTED_SSL_FINGERPRINT}\", \"keystore.pass\":\"${STORE_PASS}\"}}"

  	if [[ "$CLUSTER_ROLE" == "core" ]]; then
  	  #patch secret cms-ct-ssl
	  kubectl -n ${_namespace} patch secret cms-ct-ssl \
	    -p "{\"data\":{\"default.cms_truststorejks\":\"${CT_SSL_TRUSTSTORE_JKS}\", \"default.client_auth_keystorejks\":\"${CT_SSL_CLIENTAUTHSTORE_JKS}\", \"default.keystorepass\":\"${STORE_PASS}\", \".role\":\"$(echo -n "core"| base64)\"}}"
  	  #patch secret peer-cms-trusted-ssl-cert
  	  kubectl -n ${_namespace} patch secret peer-cms-trusted-ssl-cert \
  	    -p "{\"data\":{\"cms_truststore.jks\":\"${PEER_CMS_TRUSTED_SSL_TRUSTSTORE_JKS}\", \"client_auth_keystore.jks\":\"${PEER_CMS_TRUSTED_SSL_CLIENTAUTHSTORE_JKS}\", \"cms_fingerprint\":\"${PEER_CMS_TRUSTED_SSL_FINGERPRINT}\", \"keystore.pass\":\"${STORE_PASS}\", \"clients.pem\":\"${CLIENTS_PEM}\"}}"
	else
  	  #patch secret peer-cms-trusted-ssl-cert
  	  kubectl -n ${_namespace} patch secret peer-cms-trusted-ssl-cert \
  	    -p "{\"data\":{\"cms_truststore.jks\":\"${PEER_CMS_TRUSTED_SSL_TRUSTSTORE_JKS}\", \"client_auth_keystore.jks\":\"${PEER_CMS_TRUSTED_SSL_CLIENTAUTHSTORE_JKS}\", \"cms_fingerprint\":\"${PEER_CMS_TRUSTED_SSL_FINGERPRINT}\", \"keystore.pass\":\"${STORE_PASS}\", \"clients.pem\":\"${PEER_CLIENTS_PEM}\"}}"
	fi

  	#patch secret cms-app-server-cert
  	kubectl -n ${_namespace} patch secret cms-app-server-cert \
  	  -p "{\"data\":{\"server.keystore\":\"${APP_SERVER_CERT_JKS}\"}}"
  fi

  #clean unused certs/keystores during update phase, for listing status
  if [[ "$_user_pems_updated" == "true" ]]; then
	if [ "$_user_provided_app_pem" == "true" -a "$_user_provided_clients_pem" == "false" ]; then
		rm -f ${_tmpdir}/*client*jks ${_tmpdir}/*clients.*
	fi
	if [ "$_user_provided_app_pem" == "false" -a "$_user_provided_clients_pem" == "true" ]; then
		rm -f ${_tmpdir}/*truststore*jks ${_tmpdir}/*app.* ${_tmpdir}/*server.keystore
	fi
  	if [ "$CLUSTER_ROLE" == "core" -a "$_externalct_connected" == "true" ]; then
		#core should not update peer-cms-trusted-ssl-cert, when externalct connected
		rm -f ${_tmpdir}/peer-cms*
	fi
  	if [[ "$CLUSTER_ROLE" == "externalct" ]]; then
		#ct should not update peer-cms-trusted-ssl-cert, core should do it
		rm -f ${_tmpdir}/peer-cms*
	fi
  fi
}

#generate rabbitmq server certificates and keystores
function gen_rabbitmq_cert {

  _tmpdir="/tmp/cms-cert-gen/rabbitmq-ssl-certs"

  _days=$1
  _namespace=$2
  _secret="cms-rabbitmq-ssl-certfile"
  _storepass=`echo $RANDOM | md5sum | cut -c1-10`

  info "Generating rabbitmq server certs.."

  rm -rf $_tmpdir
  mkdir -p $_tmpdir
  cat > $_tmpdir/openssl.cnf <<- EOF
		[req]
		distinguished_name = req_distinguished_name
		req_extensions = v3_req
		[req_distinguished_name]
		[v3_req]
		basicConstraints = CA:FALSE
		keyUsage = nonRepudiation, digitalSignature, keyEncipherment
		EOF

  #generate private key
  openssl genrsa -out ${_tmpdir}/key.pem 2048 &>/dev/null

  #generate certificate
  openssl req -new -x509 -days ${_days} -key ${_tmpdir}/key.pem -out ${_tmpdir}/cert.crt \
	-subj "/CN=cms_ca" -config ${_tmpdir}/openssl.cnf 2>/dev/null
  openssl req -new -x509 -days ${_days} -key ${_tmpdir}/key.pem -out ${_tmpdir}/cacert.crt \
	-subj "/CN=cms_ca" -config ${_tmpdir}/openssl.cnf 2>/dev/null

  #generate certificate sign request
  openssl req -new -key ${_tmpdir}/key.pem > ${_tmpdir}/cert.csr \
	-subj "/CN=cms_ca" -config ${_tmpdir}/openssl.cnf 2>/dev/null
  openssl req -new -key ${_tmpdir}/key.pem > ${_tmpdir}/cacert.csr \
	-subj "/CN=cms_ca" -config ${_tmpdir}/openssl.cnf 2>/dev/null

  #sign certificate with self key, output certificate
  openssl x509 -in ${_tmpdir}/cert.csr -out ${_tmpdir}/cert.pem -req \
	-signkey ${_tmpdir}/key.pem -days ${_days} -extensions v3_req -extfile $_tmpdir/openssl.cnf 2>/dev/null
  openssl x509 -in ${_tmpdir}/cacert.csr -out ${_tmpdir}/cacert.pem -req \
	-signkey ${_tmpdir}/key.pem -days ${_days} -extensions v3_req -extfile $_tmpdir/openssl.cnf 2>/dev/null

  if [[ ! $? -eq 0 ]]; then
    error "Some error during rotating rabbitmq certs.."
  fi
  
  #output env
  KEY_PEM=$(cat $_tmpdir/key.pem|base64 -w 0)
  CERT_PEM=$(cat $_tmpdir/cert.pem|base64 -w 0)
  CACERT_PEM=$(cat $_tmpdir/cacert.pem|base64 -w 0)

  echo "$_storepass" > $_tmpdir/keystore.pass
  #update related java keystore: rabbitmq-ssl-certfile:cms_truststore.jks
  keytool -importcert -alias rabbitmq -keystore ${_tmpdir}/truststore.jks \
	-file ${_tmpdir}/cacert.pem -storepass $_storepass -noprompt

  RABBITMQ_TRUSTSTORE_JKS=$(cat $_tmpdir/truststore.jks|base64 -w 0)

  #update related java keystore: rbbitmq-ssl-certfile:auth_keystore.jks
  keytool -importcert -alias rabbitmq -keystore ${_tmpdir}/auth_keystore.jks \
	-file ${_tmpdir}/cert.pem -storepass $_storepass -noprompt

  RABBITMQ_AUTHSTORE_JKS=$(cat $_tmpdir/auth_keystore.jks|base64 -w 0)

  #update rabbitmq-ssl-certfile:cms_fingerprint
  #keytool -list -storepass $_storepass \
  #	-keystore ${_tmpdir}/auth_keystore.jks -v 2>/dev/null \
  #      |grep "Certificate fingerprints" -A3| grep SHA1 | awk -F "SHA1: " '{print $2}'|sed 's/\://g' \
  #	|xargs -i printf "NeptuneComponent:{}:TrustedClient" \
  #	|sha1sum | awk '{printf $1}' > ${_tmpdir}/cms_fingerprint

  #RABBITMQ_CMS_FINGERPRINT=$(cat $_tmpdir/cms_fingerprint|base64 -w 0)

  STORE_PASS=$(echo -n "$_storepass"|base64 -w 0)

  #patch secret
  kubectl -n ${_namespace} patch secret $_secret \
   -p "{\"data\":{\"key.pem\":\"${KEY_PEM}\", \"cacert.pem\":\"${CACERT_PEM}\", \"cert.pem\":\"${CERT_PEM}\", \"truststore.jks\":\"${RABBITMQ_TRUSTSTORE_JKS}\", \"auth_keystore.jks\":\"${RABBITMQ_AUTHSTORE_JKS}\", \"keystore.pass\": \"${STORE_PASS}\"}}"
   #-p "{\"data\":{\"key.pem\":\"${KEY_PEM}\", \"cacert.pem\":\"${CACERT_PEM}\", \"cert.pem\":\"${CERT_PEM}\", \"truststore.jks\":\"${RABBITMQ_TRUSTSTORE_JKS}\", \"auth_keystore.jks\":\"${RABBITMQ_AUTHSTORE_JKS}\", \".cms_fingerprint\":\"${RABBITMQ_CMS_FINGERPRINT}\", \"keystore.pass\": \"${STORE_PASS}\"}}"

  #kubectl -n ${_namespace} patch secret cms-rabbitmq-cert \
  # -p "{\"data\":{\"key.pem\":\"${KEY_PEM}\", \"cacert.pem\":\"${CACERT_PEM}\", \"cert.pem\":\"${CERT_PEM}\"}}"
}

#generat rabbitmq ssl certificates and keystores for externalct only
function update_rabbitmq_ssl_cert_ct {

  _rbtmpdir="/tmp/cms-cert-gen/rabbitmq-ssl-certs"

  _rbnamespace=$4
  _rbsecret="cms-rabbitmq-ssl-certfile"
  _rbstorepass=`echo $RANDOM | md5sum | cut -c1-10`
   
  if [[ "$CLUSTER_ROLE" == "externalct" ]]; then
    info "Updating rabbitmq client ssl certs for externalct.."
  else
    return
  fi

  rm -rf $_rbtmpdir
  mkdir -p $_rbtmpdir

  #use certs passed by calling side
  echo "$1" > ${_rbtmpdir}/key.pem
  echo "$2" > ${_rbtmpdir}/cert.pem
  echo "$3" > ${_rbtmpdir}/cacert.pem

  RABBITMQ_KEY_PEM=$(echo "$1"|base64 -w 0)
  RABBITMQ_CERT_PEM=$(echo "$2"|base64 -w 0)
  RABBITMQ_CACERT_PEM=$(echo "$3"|base64 -w 0)

  echo "$_rbstorepass" > $_rbtmpdir/keystore.pass
  #update related java keystore: rabbitmq-ssl-certfile:cms_truststore.jks
  keytool -importcert -alias rabbitmq -keystore ${_rbtmpdir}/truststore.jks \
	-file ${_rbtmpdir}/cacert.pem -storepass $_rbstorepass -noprompt

  RABBITMQ_TRUSTSTORE_JKS=$(cat $_rbtmpdir/truststore.jks|base64 -w 0)

  #update related java keystore: rbbitmq-ssl-certfile:auth_keystore.jks
  keytool -importcert -alias rabbitmq -keystore ${_rbtmpdir}/auth_keystore.jks \
	-file ${_rbtmpdir}/cert.pem -storepass $_rbstorepass -noprompt

  RABBITMQ_AUTHSTORE_JKS=$(cat $_rbtmpdir/auth_keystore.jks|base64 -w 0)

  #update rabbitmq-ssl-certfile:cms_fingerprint
  #keytool -list -storepass $_rbstorepass \
  #	-keystore ${_rbtmpdir}/auth_keystore.jks -v 2>/dev/null \
  #      |grep "Certificate fingerprints" -A3| grep SHA1 | awk -F "SHA1: " '{print $2}'|sed 's/\://g' \
  #	|xargs -i printf "NeptuneComponent:{}:TrustedClient" \
  #	|sha1sum | awk '{printf $1}' > ${_rbtmpdir}/cms_fingerprint

  #RABBITMQ_CMS_FINGERPRINT=$(cat $_rbtmpdir/cms_fingerprint|base64 -w 0)

  RABBITMQ_STORE_PASS=$(echo -n "$_rbstorepass"|base64 -w 0)

  kubectl -n ${_rbnamespace} patch secret $_rbsecret \
   -p "{\"data\":{\"key.pem\":\"${RABBITMQ_KEY_PEM}\", \"cacert.pem\":\"${RABBITMQ_CACERT_PEM}\", \"cert.pem\":\"${RABBITMQ_CERT_PEM}\", \"truststore.jks\":\"${RABBITMQ_TRUSTSTORE_JKS}\", \"auth_keystore.jks\":\"${RABBITMQ_AUTHSTORE_JKS}\", \"keystore.pass\": \"${RABBITMQ_STORE_PASS}\"}}"
   #-p "{\"data\":{\"key.pem\":\"${RABBITMQ_KEY_PEM}\", \"cacert.pem\":\"${RABBITMQ_CACERT_PEM}\", \"cert.pem\":\"${RABBITMQ_CERT_PEM}\", \"truststore.jks\":\"${RABBITMQ_TRUSTSTORE_JKS}\", \"auth_keystore.jks\":\"${RABBITMQ_AUTHSTORE_JKS}\", \".cms_fingerprint\":\"${RABBITMQ_CMS_FINGERPRINT}\", \"keystore.pass\": \"${RABBITMQ_STORE_PASS}\"}}"
}

#generat oauth certificates and keystores
function gen_oauth_cert {

  _tmpdir="/tmp/cms-cert-gen/oauth-certs"

  _days=$1
  _namespace=$2

  info "Generating oauth certs.."

  rm -rf $_tmpdir
  mkdir -p $_tmpdir

  if [[ "$CLUSTER_ROLE" == "core" ]]; then
    _storepass=`echo $RANDOM | md5sum | cut -c1-10`
    echo "$_storepass" > $_tmpdir/keystore.pass

    keytool -genkey -v -alias jwt -keyalg RSA -storetype PKCS12 -keystore $_tmpdir/oauth_jwt_keystore.p12 \
	-dname "CN=www.mediakind.com,OU=mk,O=mk,L=gz,ST=gd,C=cn" -storepass $_storepass -keypass $_storepass \
	-validity $_days -noprompt 2>/dev/null
    keytool -export -alias jwt -keystore $_tmpdir/oauth_jwt_keystore.p12 \
	-storetype PKCS12 -storepass $_storepass -rfc -file $_tmpdir/oauth_jwt_publickey.cer -noprompt 2>/dev/null

    OAUTH_JWT_KS_P12=$(cat $_tmpdir/oauth_jwt_keystore.p12|base64 -w 0)
    STORE_PASS=$(echo -n "$_storepass"|base64 -w 0)

    kubectl -n ${_namespace} patch secret cms-oauth-token-secret \
     -p "{\"data\":{\"oauth_jwt_keystore.p12\":\"${OAUTH_JWT_KS_P12}\", \"keystore.pass\": \"${STORE_PASS}\"}}"
  else
    if [[ -f $SCRIPTPATH/ssl/peer/oauth_jwt_publickey.cer ]]; then
	info "Using ssl/peer/oauth_jwt_publickey.cer to gen cms-oauth-token-cert"
	cp -f $SCRIPTPATH/ssl/peer/oauth_jwt_publickey.cer $_tmpdir/oauth_jwt_publickey.cer
    fi
  fi

  OAUTH_JWT_PB_CER=$(cat $_tmpdir/oauth_jwt_publickey.cer|base64 -w 0)
  kubectl -n ${_namespace} patch secret cms-oauth-token-cert \
   -p "{\"data\":{\"oauth_jwt_publickey.cer\":\"${OAUTH_JWT_PB_CER}\"}}"

  if [[ ! $? -eq 0 ]]; then
    error "Some error during rotating oauth certs.."
  fi
  
}
