#!/usr/bin/env bash

#set -ex
#set -x
set -e

#Set script to use a fixed yq of path /usr/bin, depiting other path like /usr/local/bin
shopt -s expand_aliases
alias yq='/usr/bin/yq'

RUN_AS_CONTAINER=${RUN_AS_CONTAINER:-false}

# Profile version used by init-cfg-bundle function
CMS_PROFILE_VERSION="11.0.6"
COMMON_PROFILE_VERSION="11.0.6"
CT_PROFILE_VERSION="11.0.6"
ES_PROFILE_VERSION="11.0.6"
DB_PROFILE_VERSION="11.0.6"
ETCD_PROFILE_VERSION="11.0.6"

# Profiles mapping
declare -A PROFILES
PROFILES[cms]=$CMS_PROFILE_VERSION
PROFILES[cms-common]=$COMMON_PROFILE_VERSION
PROFILES[cms-ct]=$CT_PROFILE_VERSION
PROFILES[cms-es-application]=$ES_PROFILE_VERSION
PROFILES[cms-db-epg]=$DB_PROFILE_VERSION
PROFILES[cms-db-metadata-manager]=$DB_PROFILE_VERSION
PROFILES[cms-db-workflow]=$DB_PROFILE_VERSION
PROFILES[cms-etcd]=$ETCD_PROFILE_VERSION

# Get full path of this script file
SCRIPT=$(readlink -f "$0")
# Get folder path of this script
SCRIPTPATH=$(dirname "$SCRIPT")

LICENSE_FILE="${SCRIPTPATH}/license/license.xml"
SSL_FOLDER="${SCRIPTPATH}/ssl"
PEER_SSL_FOLDER="${SCRIPTPATH}/ssl/peer"
CHART_FOLDER="${SCRIPTPATH}/charts"
BUNDLEFOLDER="${SCRIPTPATH}/ConfigBundle"
PROFILEFOLDER="${BUNDLEFOLDER}/profiles"
RPMFOLDER="${SCRIPTPATH}/tool-rpms"
MATRIXFILE="${BUNDLEFOLDER}/products-matrix.yaml"
PRODUCTSVAR_FILE="${BUNDLEFOLDER}/products-var.yaml"
CMSCONF="$SCRIPTPATH/cms.var"
CMSGEOCONF="$SCRIPTPATH/cmsgeo.var"
CMSCONF_COMBINE="/tmp/cms.var.combine"

#TERRAFORM_VERSION="0.12.29"
#TERRAFORM_VERSION="0.14.5"
TERRAFORM_VERSION="1.5.2"
TF_DEFAULT_VARIABLES="$SCRIPTPATH/_variables.tf"
# update APIVersion in the manifest files to align with latest kubernetes kubeapi version
updateAPIVersion="true"

DEFAULT_CMS_ADMIN_SERVICE_ACCOUNT_NAME="cms-admin"

# Define the log file for this scripts
LOG="${SCRIPTPATH}/logs/$(basename "$0" .sh).log"

##############################
# Include all functions from shell_lib
##############################
for f in ${SCRIPTPATH}/shell_lib/*.sh; do source $f; done

# Parameters from cms.var
CLOUDPLATFORM=$(get_parameter_value cloud_platform)
if [[ $CLOUDPLATFORM == "AZURE" ]];then
  TERRAFORM_FOLDER="tf-aks"
  TF_DEFAULT_VARIABLES="$SCRIPTPATH/$TERRAFORM_FOLDER/variables.tf"
  if $RUN_AS_CONTAINER; then
    CLOUD_ENV="$SCRIPTPATH/data/aks.env"
  else
    CLOUD_ENV="$SCRIPTPATH/aks.env"
  fi
fi
if [[ $CLOUDPLATFORM == "GCP" ]];then
  TERRAFORM_FOLDER="tf-gke"
  TF_DEFAULT_VARIABLES="$SCRIPTPATH/$TERRAFORM_FOLDER/variables.tf"
  if $RUN_AS_CONTAINER; then
    CLOUD_ENV="$SCRIPTPATH/data/gke.env"
  else
    CLOUD_ENV="$SCRIPTPATH/gke.env"
  fi
  GCP_REGION=$(get_parameter_value gcp_region)
  GCP_GCR=$(get_parameter_value gcp_gcr)
  GCP_PROJECT=$(get_parameter_value gcp_project_id)
fi

if [[ $CLOUDPLATFORM == "AWS" ]];then
  TERRAFORM_FOLDER="tf-aks"
  TF_DEFAULT_VARIABLES="$SCRIPTPATH/$TERRAFORM_FOLDER/locals.tf"
  if $RUN_AS_CONTAINER; then
    CLOUD_ENV="$SCRIPTPATH/data/aws.env"
  else
    CLOUD_ENV="$SCRIPTPATH/aws.env"
  fi

  AWS_EKS_ARN=$(grep OUTPUT_EKS_ARN "${CLOUD_ENV}" | awk -F '=' '{gsub(/"/, "", $2); print $2}')
  AWS_REGION=$(grep OUTPUT_AWS_REGION "${CLOUD_ENV}" | awk -F '=' '{gsub(/"/, "", $2); print $2}')

` `


fi

# 
ARR_USER_PASSWD=(image_pull_user image_pull_password elasticstack_monitor_user elasticstack_monitor_password elasticstack_manage_user elasticstack_manage_password)
ARR_CLOUD_VAR=(client_id client_secret)

#Common parameters
K8S_PLATFORM=$(get_parameter_value k8s_platform "")
ALLOW_WEBHOOK=$(get_parameter_value allow_webhook true)
ALLOW_CREATE_NODE_LABEL=$(get_parameter_value allow_create_node_label true)

K8S_CLUSTER_DOMAIN=$(get_parameter_value k8s_cluster_domain "cluster.local")
DEV_MODE=$(get_parameter_value dev_mode)
MEDIAFIRST_INTEGRATION=$(get_parameter_value mediafirst_integration)
# Since USE_SPOT_VM will be used in common function from load_cms_charts,
# need to init USE_SPOT_VM with false.
# The values will be later set for Azure deployment based on value passed from cms.var.
USE_SPOT_VM=false

CMS_TYPE=$(get_parameter_value cms_type)
CMS_TYPE_LOWER=$(echo "$CMS_TYPE" | awk '{print tolower($0)}')

CLUSTER_ROLE=$(get_parameter_value cluster_role)
CLUSTER_API_SERVER_URL=$(get_parameter_value cluster_api_server_url "")

if $RUN_AS_CONTAINER && [[ $CLUSTER_ROLE == "externalct" ]]; then
  if [[ -z $CLUSTER_API_SERVER_URL ]]; then
    echo "[ERROR] Value of variable cluster_api_server_url in cms.var should not be empty." |& tee -i -a $LOG
    echo "  -- Provide proper value for cluster_api_server_url into configmap cmsbootstrap-cms-var and rerun this deploy-cms.sh command." |& tee -i -a $LOG
    exit 1
  fi
fi

CLUSTER_NAME=$(get_parameter_value cluster_name)
#CLOUD_CLUSTER_INSTANCE_NAME="$CLUSTER_NAME-cms-cluster"
CLOUD_CLUSTER_INSTANCE_NAME="$CLUSTER_NAME"
RUN_FILEBEAT_USERID=$(get_parameter_value run_filebeat_as_user_id)
RUN_FILEBEAT_GROUPID=$(get_parameter_value run_filebeat_as_group_id)
CMS_NAMESPACE=$(get_parameter_value cms_namespace)
CMS_TIMEZONE=$(get_parameter_value cms_timezone)
CMS_TOMCAT_UMASK=$(get_parameter_value cms_tomcat_umask)
INTERNAL_CT_ENABLED=$(get_parameter_value internal_ct_enabled)
CMS_ADMIN_SERVICE_ACCOUNT_NAME=$(get_parameter_value cms_admin_service_account_name)
APPARMOR_PROFILE_NAME=$(get_parameter_value apparmor_profile_name)

LIMITED_DEPLOYMENT=$(get_parameter_value limited_deployment false)
POD_SECURITY_POLICY_NAME=$(get_parameter_value pod_security_policy_name)
STORAGECLASS_NAME=$(get_parameter_value storageclass_name)

ALLOW_ROOT_USER_FOR_INIT_CONTAINER=$(get_parameter_value allow_root_user_for_init_container true)
ALLOW_HOST_PATH=$(get_parameter_value allow_host_path true)


EXT_WHITE_LIST=$(get_parameter_value external_whitelist)

CONTENT_NAS_SIZE=$(get_parameter_value content_nas_size)

#Elastic stack users
ELASTICSTACK_MANAGE_USER=$(get_parameter_value elasticstack_manage_user "elastic")
ELASTICSTACK_MANAGE_PASSWORD=$(get_parameter_value elasticstack_manage_password 'mkcm$elastic!234')
ELASTICSTACK_MONITOR_USER=$(get_parameter_value elasticstack_monitor_user "cmsmonitor")
ELASTICSTACK_MONITOR_PASSWORD=$(get_parameter_value elasticstack_monitor_password 'mkcm$monitor!234')

#FTP parameters
FTP_ENABLED=$(get_parameter_value ftp_enable)
FTP_PASSIVEPORTMIN=$(get_parameter_value ftp_passiveportMin)
FTP_PASSIVEPORTMAX=$(get_parameter_value ftp_passiveportMax)
FTP_USER=$(get_parameter_value ftp_user)
FTP_PASSWORD_B64=$(echo $(get_parameter_value ftp_password)|base64)

#GEO parameters
GEO_KUBERNETES_SERVER_FQDN=$(get_geo_parameter_value kubernetes_server_fqdn)
GEO_REMOTE_CLUSTER_IP=$(get_geo_parameter_value remote_cluster_ip)
GEO_REMOTE_CLUSTER_NAME=$(get_geo_parameter_value remote_cluster_name)
GEO_REMOTE_KUBE_CONFIG_BASE64=$(get_geo_parameter_value remote_kube_config_base64)
GEO_REMOTE_NAMESPACE=$(get_geo_parameter_value remote_namespace)
GEO_SSH_AUTH_KEY=$(get_geo_parameter_value ssh_auth_key)
GEO_SSH_PRIVATE_KEY=$(get_geo_parameter_value ssh_private_key)
GEO_FILERSYNC_SSH_PORT=$(get_geo_parameter_value filersync_ssh_port)
GEO_RABBITMQ_RSYNC_SSH_PORT=$(get_geo_parameter_value rabbitmq_rsync_ssh_port)
GEO_METADATA_MANAGER_STOLON_PROXY_NODE_PORT=$(get_geo_parameter_value metadata_manager_stolon_proxy_node_port)
GEO_WORKFLOW_STOLON_PROXY_NODE_PORT=$(get_geo_parameter_value workflow_stolon_proxy_node_port)
GEO_EPG_STOLON_PROXY_NODE_PORT=$(get_geo_parameter_value epg_stolon_proxy_node_port)
GEO_ES_APPLICATION_CLIENT_PORT=$(get_geo_parameter_value es_application_client_port)

IMAGE_REGISTRY=$(get_parameter_value image_registry)
IMAGE_REGISTRY_INSTANCES=$(get_parameter_value image_registry_instances)
IMAGE_PULL_USER=$(get_parameter_value image_pull_user)
IMAGE_PULL_PASSWORD=$(get_parameter_value image_pull_password)
IMAGE_PULL_SECRET_NAME="cms-image-pull-secret"

#Parameters for Bare Metal deployment specific
if [[ $CLOUDPLATFORM == "" ]];then
  #SETUP_VIP=$(get_parameter_value setup_vip)
  EXTERNAL_VIP=$(get_parameter_value external_vip)
  INTERNAL_VIP=$(get_parameter_value internal_vip)
  CORE_VIP=$(get_parameter_value core_vip)
  
  NAS_SERVER_IP=$(get_parameter_value nas_server_ip)
  NAS_SHARE=$(get_parameter_value nas_share)
  CONTENT_NAS_SERVER_IP=$(get_parameter_value content_nas_server_ip)
  CONTENT_NAS_SHARE=$(get_parameter_value content_nas_share)
  BACKUP_NAS_SERVER_IP=$(get_parameter_value backup_nas_server_ip)
  BACKUP_NAS_SHARE=$(get_parameter_value backup_nas_share)
  USER_MODIFIED_STORAGECLASS=$(get_parameter_value user_modified_storageclass false)
  DB_LOCAL_VOLUME_PREFIX=$(get_parameter_value db_local_volume_prefix |sed -e 's/\/$//') 
fi

if [[ -n "$CLOUDPLATFORM" ]];then
  K8S_SERVER_VERSION=$(get_parameter_value k8s_version)
fi

if [[ $CLOUDPLATFORM == "AZURE" ]];then
  # No need to create node labels for CMS deployment on AKS,
  # as the labels are already being set in the node pool level.
  ALLOW_CREATE_NODE_LABEL=false

  #Parameters for Cloud deployment spcific
  #PG_DISK_SIZE=$(get_parameter_value pg_disk_size)
  SC_SKU=$(get_parameter_value storageclass_sku)

  CMS_NODES_K8S_VERSION=$(get_parameter_value cms_nodes_k8s_version)
  
  ## Set value based on cms.var only for Azure deployment secnario.
  USE_SPOT_VM=$(get_parameter_value use_spot_vm false)
  
  # Azure specific parameters
  AZURE_SUBID=$(get_parameter_value subscription_id)
  AZURE_TENANTID=$(get_parameter_value tenant_id)
  AZURE_CLIENTID=$(get_parameter_value client_id)
  AZURE_CLIENTSECRET=$(get_parameter_value client_secret)
  AZURE_REGION=$(get_parameter_value azure_region)

  SETUP_PRIVATE_AKS=$(get_parameter_value setup_private_aks "false")

  # Azure Container Registry 'ACR_NAME' must conform to the following pattern: '^[a-zA-Z0-9]*$'
  ACR_RESOURCEGROUP="cmsacr-rg-$AZURE_REGION"
  #ACR_NAME="MKCMSACR$AZURE_REGION"
  ACR_NAME=$(get_parameter_value acr_name)
  ACR_SKU="Basic"
  mapfile -t ACR_TAGS <<<$(get_parameter_mapvalue custom_tags)
  
  # AKS Resource Group
  RESOURCEGROUP_NAME=""
  EXISTING_AKS_RESOURCE_GROUP_NAME=$(get_parameter_value existing_aks_resource_group_name)

  ## Azure Storage Account
  #AZURE_STORAGE_ACCOUNT_NAME=$(echo "$CLUSTER_NAME-storage" | sed -e 's/-//g' | cut -c1-23)

  USE_AZURE_FILE_STORAGE=$(get_parameter_value use_azure_file_storage true)
  USE_AZURE_BLOB_STORAGE=$(get_parameter_value use_azure_blob_storage false)

  # Azure-file Static share folder names 
  #AZUREFILE_SHARES="cms-content cms-content-heavy cms-metadata-manager-content cms-conf cms-plugin backup-restore cmscli-plugin cms-sync-utils cms-service-client cms-workflow-runtime"
  AZUREFILE_SHARES="cms-content cms-metadata-manager-content cms-conf cms-plugin backup-restore cmscli-plugin cms-sync-utils cms-service-client cms-workflow-runtime cms-watchfolder-runtime"

  # Database disk size, common for both aks and gke
  PG_DISK_SIZE=$(get_parameter_value pg_disk_size "30")

  #Pod CIDR
  POD_CIDR=$(get_parameter_value kubernetes_pod_cidr)

fi

if [[ $CLOUDPLATFORM == "AWS" ]];then
  # Database disk size, since Azure size set above
  PG_DISK_SIZE=$(get_parameter_value pg_disk_size "30")
fi

if [[ $CLOUDPLATFORM == "GCP" ]];then
  # Database disk size, since Azure size set above
  PG_DISK_SIZE=$(get_parameter_value pg_disk_size "30")
fi

# Chart Packages
CMS_CHARTS_PACKAGE="cms-charts-*.tgz"

# Container Image Packages
#  mdt-containers_cms_9.0.000.818.tgz
#  mdt-containers_cms3pp_9.0.000.823.tgz
CMS_IMAGE_PACKAGES="
  cms-containers-*.tgz
  cms-3pp-containers-*.tgz
  "

# Configure Bundle Package
#  config-bundle-cms-9.0.000.826.tgz
CONFIG_PACKAGE="cms-config-bundle-*.tgz"

# Utility rpms Package
#  rpms-cms-9.0.000.826.tgz
CMS_RPM_PACKAGE="cms-tool-rpms-*.tgz"


## DEPLOY_LSERV=false
## 
## # TODO use 30001 as in BareMetal/OS/GCE
## UI_PORT=8443

# Make sure the log file is ready
mkdir -p $(dirname $LOG)
touch $LOG

trap ctrl_c INT
trap 'echo "Error in scripts '$0': error line: $LINENO,error cmd: $BASH_COMMAND" |& tee -i -a $LOG' ERR

start_information_print(){
  echo "[INFO] *********************************************" |& tee -i -a $LOG
  echo "[INFO] Start time : $(date +"%Hh%Mm%Ss%3Nms")" |& tee -i -a $LOG
  echo "[INFO] Run As Container : $RUN_AS_CONTAINER" |& tee -i -a $LOG
  START_TIME=$(date "+%s")
  export START_TIME
  
  echo "" |& tee -i -a $LOG
  echo "=========== All Scripts Parameters ======================" |& tee -i -a $LOG
  echo "  SCRIPTPATH=$SCRIPTPATH"|& tee -i -a $LOG
  echo "  LOG=$LOG"|& tee -i -a $LOG
  echo "  BUNDLEFOLDER=$BUNDLEFOLDER"|& tee -i -a $LOG
  echo "  CMSCONF=$CMSCONF"|& tee -i -a $LOG
  echo "  CMS_TYPE=$CMS_TYPE"|& tee -i -a $LOG
  echo "  LIMITED_DEPLOYMENT=$LIMITED_DEPLOYMENT"|& tee -i -a $LOG
  echo "  CLUSTER_NAME=$CLUSTER_NAME"|& tee -i -a $LOG
  echo "  CLUSTER_ROLE=$CLUSTER_ROLE"|& tee -i -a $LOG
  echo "  INTERNAL_CT_ENABLED=$INTERNAL_CT_ENABLED"|& tee -i -a $LOG
  echo "  DEV_MODE=$DEV_MODE"|& tee -i -a $LOG
  echo "  MEDIAFIRST_INTEGRATION=$MEDIAFIRST_INTEGRATION"|& tee -i -a $LOG
  echo "  EXT_WHITE_LIST=$EXT_WHITE_LIST"|& tee -i -a $LOG
  if [[ $CLOUDPLATFORM == "" ]];then
  echo "  IMAGE_REGISTRY=$IMAGE_REGISTRY"|& tee -i -a $LOG
  echo "  IMAGE_REGISTRY_INSTANCES=$IMAGE_REGISTRY_INSTANCES"|& tee -i -a $LOG
  echo "  EXTERNAL_VIP=$EXTERNAL_VIP"|& tee -i -a $LOG
  echo "  INTERNAL_VIP=$INTERNAL_VIP"|& tee -i -a $LOG
  echo "  CORE_VIP=$CORE_VIP"|& tee -i -a $LOG
  echo "  NAS_SERVER_IP=$NAS_SERVER_IP"|& tee -i -a $LOG
  echo "  NAS_SHARE=$NAS_SHARE"|& tee -i -a $LOG
  echo "  CONTENT_NAS_SERVER_IP=$CONTENT_NAS_SERVER_IP"|& tee -i -a $LOG
  echo "  CONTENT_NAS_SHARE=$CONTENT_NAS_SHARE"|& tee -i -a $LOG
  echo "  DB_LOCAL_VOLUME_PREFIX=$DB_LOCAL_VOLUME_PREFIX"|& tee -i -a $LOG
  elif [[ $CLOUDPLATFORM == "AZURE" ]];then
  echo ""|& tee -a $LOG
  echo "Azure Specific Parameters:"|& tee -a $LOG
  echo "----------------------------------------"|& tee -a $LOG
  echo "  USE_SPOT_VM=$USE_SPOT_VM"|& tee -a $LOG
  echo "  K8S_SERVER_VERSION=$K8S_SERVER_VERSION"|& tee -a $LOG
  echo "  CMS_NODES_K8S_VERSION=$CMS_NODES_K8S_VERSION"|& tee -a $LOG
    if [[ -z "$IMAGE_REGISTRY" ]];then
    echo "  ACR_RESOURCEGROUP=$ACR_RESOURCEGROUP"|& tee -a $LOG
    echo "  ACR_NAME=$ACR_NAME"|& tee -a $LOG
    echo "  ACR_SKU=$ACR_SKU"|& tee -a $LOG
    else
    echo "  IMAGE_REGISTRY=$IMAGE_REGISTRY"|& tee -i -a $LOG
    fi
    if $RUN_AS_CONTAINER; then
      if [[ -z "$EXISTING_AKS_RESOURCE_GROUP_NAME" ]]; then
        RESOURCEGROUP_NAME="rg-$CLUSTER_NAME"
      else
        RESOURCEGROUP_NAME="$EXISTING_AKS_RESOURCE_GROUP_NAME"
      fi
    else
    # Ensure to use the correct resource group name
    #Use az to check whether the resource group with legacy resource group name "$CLUSTER_NAME-cms-resourcegroup" exist
    #If exist, use this legacy resource group name, otherwise, use the new formated resource group name as "rg-$CLUSTER_NAME"
      set_azure_resource_group
    fi
  echo "  RESOURCEGROUP_NAME=$RESOURCEGROUP_NAME"|& tee -a $LOG
  echo "  CLOUD_CLUSTER_INSTANCE_NAME=$CLOUD_CLUSTER_INSTANCE_NAME"|& tee -a $LOG
  #echo "  AZURE_STORAGE_ACCOUNT_NAME=$AZURE_STORAGE_ACCOUNT_NAME"|& tee -a $LOG
  echo "  Azure-Disk-StorageClass_SKU=$SC_SKU"|& tee -a $LOG
  echo "  USE_AZURE_FILE_STORAGE=$USE_AZURE_FILE_STORAGE"|& tee -a $LOG
  echo "  USE_AZURE_BLOB_STORAGE=$USE_AZURE_BLOB_STORAGE"|& tee -a $LOG
  elif [[ $CLOUDPLATFORM == "AWS" ]];then

  AWS_REGION=$(get_parameter_value aws_region)


  echo ""|& tee -a $LOG
  echo "AWS Specific Parameters:"|& tee -a $LOG
  echo "----------------------------------------"|& tee -a $LOG
  echo "  K8S_SERVER_VERSION=$K8S_SERVER_VERSION"|& tee -a $LOG
  echo "  CMS_NODES_K8S_VERSION=$CMS_NODES_K8S_VERSION"|& tee -a $LOG
  echo "  CLOUD_CLUSTER_INSTANCE_NAME=$CLOUD_CLUSTER_INSTANCE_NAME"|& tee -a $LOG
  echo "  USE_AWS_FILE_STORAGE=$USE_AWS_FILE_STORAGE"|& tee -a $LOG
  echo "  USE_AWS_BLOB_STORAGE=$USE_AWS_BLOB_STORAGE"|& tee -a $LOG
  echo "  AWS_EKS_ARN=$AWS_EKS_ARN"|& tee -a $LOG
  echo "  AWS_REGION=$AWS_REGION"|& tee -a $LOG

  fi
  echo "=========================================================" |& tee -i -a $LOG
  echo "" |& tee -i -a $LOG
}

##################
#  Script usage  #
################## 
usage_function() {
    echo "This script is for deploying CMS components."
    echo "----------------------------------------------------------------"
    echo "Usage: $0 <function>"
    echo
    echo "Available Functions:"
    echo "  prepare        : Prepare the Bootstrap node with necessary tools (Only available in non-containerized environments)"
    echo "  image          : Prepare the docker images and push to docker image registry (Only available in non-containerized environments)"
    echo "  setup-cluster  : Create a cloud cluster from scratch (Only available in non-containerized environments)"
    echo "                   Use --always-continue to automatically confirm the manual confirmation request."
    echo "  clean          : Destroy the whole cloud cluster (Only available in non-containerized environments)"
    echo "  init-config    : Initialize the config bundle files"
    echo "  deploy         : Deploy CMS components into the Kubernetes cluster based on default products-matrix.yaml"
    echo "  reset          : Remove CMS components from the Kubernetes cluster based on default products-matrix.yaml"
    echo "  reset all      : Delete the whole CMS namespace"
    echo "  ct-register    : Register external CT information to CMS core side"
    echo "  rollback       : Rollback CMS to previous backup revision"
    echo "  es-snapshot    : Take elasticsearch snapshot backup, query backup or restore. Sub-command: backup, query, restore"
    echo "  post-check     : Check CMS status after installation, should use CMS namespace as input parameter"
    echo "  -h | --help    : Show help section"
    echo
    echo "Examples:"
    echo "  $0 prepare"
    echo "  $0 image cms"
    echo "  $0 setup-cluster"
    echo "  $0 setup-cluster --always-continue"
    echo "  $0 init-config"
    echo "  $0 deploy"
    echo "  $0 reset"
    echo "  $0 reset all"
    echo "  $0 clean"
    echo "  $0 ct-register"
    echo "  $0 rollback"
    echo "  $0 es-snapshot backup logging [prefix_name]"
    echo "  $0 post-check <cms_namespace>"
    echo "  $0 -h"
    echo
}

##################
#  Script Helper #
################## 
help_function() {
    usage_function
    echo "More Advanced Usage for Function --deploy-- :"
    echo "----------------------------------------------------------------"
    python install.py -h | sed -e 's#[python ]*install.py#  '$0' deploy#'
    echo ""
}


############################################################
# MAIN
############################################################
mkdir -p ${SCRIPTPATH}/logs
reset_global_timer

if [[ "$1" == "prepare" || "$1" == "setup-cluster" || "$1" == "clean" || "$1" == "image" ]];then
  if $RUN_AS_CONTAINER; then
    echo "[ERROR] $1 subcommand cannot be run in a container environmen." |& tee -i -a $LOG
    exit
  fi
fi

if [ "$1" = "prepare" ] ; then
  # Prepare Bastion Machine / Bootstrap node
  prepare_installer |& tee -i -a $LOG
  process_check "Prepare installer failed!"
   
elif [ "$1" = "image" ] ; then
  start_information_print 
  # Prepare Docker Images
  prepare_images ${@:2} |& tee -i -a $LOG
  process_check "Prepare images failed!"
   
elif [ "$1" = "setup-cluster" ] ; then 
  start_information_print 
  # Deploy Cloud Kubernetes cluster using Terraform
  instanciate $2|& tee -i -a $LOG
  process_check "Deploy Kubernetes cluster on Cloud failed!"
  
elif [ "$1" = "clean" ] ; then
  start_information_print 
  user_confirm "Destroy the Cloud Kubernetes cluster including the CMS deployment!"
  cleanup |& tee -i -a $LOG
  process_check "Clean Cloud Kubernetes cluster failed!"

elif [ "$1" = "deploy" ] ; then
  # Only asking for confirmation when deploy with on-prem default matrix file.
  if [[ $CLOUDPLATFORM == "" ]] && [[ -z "${@:2}" ]] && $ALLOW_CREATE_NODE_LABEL;then
    user_confirm "!! *NOTICE* !! Do you confirm you have the products-matrix file configured as expected?
Enter 'No' if not. Refer to Installation Guide and ensure you have the products-matrix configured as expected first;
Enter 'Yes' to continue this deployment scripts."
  fi
  [[ $2 != "-h" && $2 != "--help" ]] && start_information_print 
  # Check packages
  check_packages $CMS_CHARTS_PACKAGE |& tee -i -a $LOG
  process_check "Check packages failed!"
  
  [[ $2 != "-h" && $2 != "--help" ]] && ! $RUN_AS_CONTAINER && prepare_helm |& tee -i -a $LOG

  if [[ "${@:2}" =~ "-n " ]];then
    echo "[WARN] Bypassing load charts for single release installation." |& tee -i -a $LOG
  elif [[ ! ("${@:2}" =~ "-h") ]] && [[ ! ("${@:2}" =~ "--help") ]];then
    load_cms_charts |& tee -i -a $LOG
    process_check "Prepare CMS charts failed!"
  fi

  if [ -z $2 ]; then
    deploy_products -mx products-matrix.yaml |& tee -i -a $LOG
    process_check "Deploy CMS failed!"

    if [ $CLUSTER_ROLE = "externalct" ];then
      ct_registration |& tee -i -a $LOG
      process_check "Failed to register external CT! Make sure CMS Core cluster is accessible and rerun the register step by command 'sh deploy-cms.sh ct-register'"
    fi

    post_check $CMS_NAMESPACE
  else
    deploy_products ${@:2} |& tee -i -a $LOG
    [[ $2 != "-h" && $2 != "--help" ]] && process_check "Deploy CMS failed!"
  fi

elif [ "$1" = "reset" ] ; then
  start_information_print
  if [ "$2" = "all" ] ; then
    user_confirm "!! *CAUTION* !! Delete whole CMS namespace from Kubernetes cluster!"
    reset_products "all" |& tee -i -a $LOG
    process_check "Reset Products failed!"
  elif [ -z "$2" ] ; then
    user_confirm "Remove the CMS from Kubernetes cluster!"
    reset_products |& tee -i -a $LOG
    process_check "Reset Products failed!"
  else
    usage_function
  fi

elif [ "$1" = "rollback" ] ; then
  rollback_products ${@:2} |& tee -i -a $LOG
  process_check "Rollback Products failed!"

elif [ "$1" = "init-config" ] ; then
  start_information_print 
  init_cfgBundle |& tee -i -a $LOG
  process_check "Initialize configuration bundle failed!"

elif [ "$1" = "ct-register" ] ; then
  start_information_print
  #The value of $2 can replace address of the kubernetes api server
  #for example:
  #bash deploy-cms.sh ct-register https://10.116.50.2:6443
  ct_registration $2 |& tee -i -a $LOG
  process_check "Failed to register external CT! Make sure CMS Core cluster is accessible and rerun the register step by command 'sh deploy-cms.sh ct-register'"

elif [ "$1" = "es-snapshot" ] ; then
  if [[ "$2" == '-h' ]] || [[ "$2" == '--help' ]]; then
    echo "es-snapshot Usage:"
    echo "  Take elasticsearch snapshots:                             backup <logging|application> [prefix_name]"
    echo "  Query the snapshots and status:                           query <logging|application>"
    echo "  Delete elasticsearch snapshots:                           delete <logging|application> [prefix_name]"
    echo "  Restore elasticsearch indices from the snapshot backups:  restore <logging|application> <prefix_name_timestamp> [--no-metrics]"
    echo "                                                            restore <logging|application> <timestamp> [--no-metrics]"
    echo "                                                            restore <logging|application> <snapshot_full_name> [--no-metrics]"
    exit
  fi
  es_tls_result=$(tls_or_not)
  if [[ $es_tls_result -eq 0 ]]; then
    es_tls_setting="tls_disable"
  elif [[ $es_tls_result -eq 1 ]]; then
    es_tls_setting="tls_enable"
  elif [[ $es_tls_result -eq 2 ]]; then
    echo "[ERROR] Elasticsearch not accessable. Make sure the logging Elasticsearch is normal"
    exit 1
  elif [[ $es_tls_result -eq 3 ]]; then
    echo "[ERROR] No Elasticsearch pod found. Make sure the K8S cluster is ready and CMS elasticsearch pods are running"
    exit 1
  else
    echo "[ERROR] Failed to check Elasticsearch tls state"
    exit 1
  fi
  if [[ "$2" == "backup" ]]; then
    es_backup "$es_tls_setting" "$3" "$4"
  elif [[ "$2" == "query" ]]; then
    es_query "$es_tls_setting" "$3"
  elif [[ "$2" == "restore" ]]; then
    es_restore "$es_tls_setting" "$3" "$4" "$5"
  elif [[ "$2" == "delete" ]]; then
    es_remove "$es_tls_setting" "$3" "$4"
  elif [[ -z "$2" ]]; then
    echo "Please specific the operation for target elasticsearch: backup, query, restore or delete, -h or --help for more details"
    exit
  fi

elif [ "$1" == "post-check" ]; then 
  if [[ -z "$2" ]]; then
    echo "[ERROR] Need to pass the CMS namespace as input parameter for post_check function"
    exit 1
  fi
  post_check $2

elif [ "$1" == "-h" ] || [ "$1" == "--help" ] ; then
  help_function

else
  usage_function
fi
