#!/usr/bin/env bash

get_product_parameter()
{
    PRODUCT_KEY=$1
    #yq r $PRODUCTSVAR_FILE "$PRODUCT_KEY" 	#change to yq4	
    yq ".$PRODUCT_KEY" $PRODUCTSVAR_FILE
}

OPENSSL_VERSION=$(openssl version | cut -d' ' -f2)

function ctrl_c() {
        echo "** Trapped CTRL-C ** Stopped by User"
}

function reset_global_timer {
    export SEC0=$(date --utc +%s)
}

function reset_function_timer {
    export SEC1=$(date --utc +%s)
}

function running_time {
    SEC2=$(date --utc +%s); DIFFSEC=$((${SEC2} - ${SEC1})); printf "\nSection Time: $(date +%H:%M:%S -ud @${DIFFSEC})\n"
    SEC2=$(date --utc +%s); DIFFSEC=$((${SEC2} - ${SEC0})); printf "Elapsed Time: $(date +%H:%M:%S -ud @${DIFFSEC})\n\n"
}

function process_check(){
  if [ ${PIPESTATUS[0]} -ne 0 ];then
     echo "[Error] $1" |& tee -a $LOG
     exit 1
  fi
}

function beginLogging(){
  echo "[INFO] ==================================================="
  echo "[INFO] $1"
  echo "[INFO] Start time : $(date +"%Y-%m-%d %H:%M:%S_%3N")"
}

function sleep_pot(){
  local sleepTime=$1
  for (( i=1; i<=$1; i++ ))
  do
   printf "."
   sleep 1
  done
  echo ""
}

#########################
# Function: get_parameter_value
# Fetch the value for parameter from CMSCONF first
# If no value found, then fetch the default value from TF_DEFAULT_VARIABLES
# If still no value found, use the default value passed 
# usage: get_parameter_value <parameter_name> <default_value>
#########################
function get_parameter_value(){
  local _parmName=$1
  local _defaultValue=$2
  local _parmValue=$(grep "^$_parmName[' ']*=" $CMSCONF|tail -1|awk -F '=' '{print $2}'|sed -e 's/ //g' -e 's/"//g')

  if [[ ${ARR_USER_PASSWD[@]/$_parmName/} != ${ARR_USER_PASSWD[@]} ]]; then
    if [[ $(env|grep $_parmName) =~ $_parmName\=[^\s] ]]; then
      env|grep $_parmName|grep -Po '(?<=\=).*'
      return 0
    fi
  fi
  # cloud var
  if [[ ${ARR_CLOUD_VAR[@]/$_parmName/} != ${ARR_CLOUD_VAR[@]} ]]; then
    if [[ $(env|grep TF_VAR_$_parmName) =~ TF_VAR_$_parmName\=[^\s] ]]; then
      #sed -i "/^$_parmName/s/^/#/" $CMSCONF
      #sed -i "/^$_parmName\|^\s*$_parmName/s/^/#/" $CMSCONF
      eval echo \$$"TF_VAR_$_parmName"
      # terraform plan -var-file=$CMSCONF -out=cms.tfplan
      return 0
    fi
  fi

  if [[ -z $_parmValue ]] && [[ -f $TF_DEFAULT_VARIABLES ]];then
    _parmValue=$(sed -n '/variable "'$_parmName'"/{:a;N;/\}/!{$!ba};/default/p}' $TF_DEFAULT_VARIABLES | grep -v ^[' ']*# | grep "default[' ']*=" |awk -F '=' '{print $2}'|sed -e 's/ //g' -e 's/"//g')
  fi

  if [[ -z $_parmValue ]] && [[ -n $_defaultValue ]];then
    _parmValue=$_defaultValue
  fi

  echo $_parmValue
}

#########################
# Function: get_parameter_mapvalue
# Fetch the map value ({}) for parameter from CMSCONF first
# If no value found, then fetch the default value from TF_DEFAULT_VARIABLES
#########################
function get_parameter_mapvalue(){
  local _parmName=$1
  local _parmValue=$(sed -n '/'$_parmName'.*=.*{/, /}/ {s/'$_parmName'.*=.*{//g; s/}//g; s/\t\+//g; s/^ \+//g; /^$/d; s/ \+= \+/=/g; s/= \+/=/g; s/ \+=/=/g; p}' $CMSCONF)

  if [[ -z $_parmValue ]] && [[ -f $TF_DEFAULT_VARIABLES ]];then
    _parmValue=$(sed -n '/variable.*"'$_parmName'".*{/, /^}/ {/default.*{/, /^}/ {s/default.*{//g; s/}//g; s/\t\+//g; s/^ \+//g; /^$/d; s/ \+= \+/=/g; s/ \+=/=/g; s/= \+/=/g; p }}' $TF_DEFAULT_VARIABLES )
  fi

  echo "$_parmValue"
}

#########################
# Function: get_geo_parameter_value
# Fetch the value for parameter from CMSGEOCONF
#########################
function get_geo_parameter_value(){
  local _parmName=$1
  local _parmValue=$(grep "^$_parmName[' ']*=" $CMSGEOCONF|sed "s/^$_parmName[' ']*=//"|sed -e 's/ //g' -e 's/"//g')
  echo $_parmValue
}

#########################
# Function: check_packages
# Make sure the needed packages are ready in ${SCRIPTPATH}/packages folder
#########################

function check_packages(){
  local packagesToCheck=$1
  cd $SCRIPTPATH/packages

  for packageToCheck in ${packagesToCheck}
  do
    if [[  $(ls ${packageToCheck} > /dev/null 2>&1; echo $?) -ne 0 ]]; then
      echo "[Error] Missing package $packageToCheck in folder ${SCRIPTPATH}/packages. Make sure all needed packages are put into the packages folder before running this scripts again."
      echo "$packagesToCheck" |& tee -a $LOG
      exit 1
    fi
  done

  # Change back to previous folder
  cd - > /dev/null
}


#########################
# Function: gen_base64()
# create encrypted string by base64 for a file and put the result to specific key in products-var.yaml file.
#########################

function gen_base64(){
  # full path of the file to be transfered to base64
  local filepath=$1
  # key format as foo.bar
  local varkey=$2
  # level is ERROR or WARN
  local level=$3
  
  #KEYS=($(echo $varkey | tr "." "\n"))
  #KEY_LENTH=${#KEYS[@]}

  if [[ ! -f $filepath ]]; then
    if [[ $level == "ERROR" ]]; then
      echo "[Error] Missing file [$filepath]. Make sure to have this file prepared before running install scripts again." |& tee -a $LOG
      exit 1
    fi
    if [[ $level == "WARN" ]]; then
      echo "[Warn] Missing file [$filepath]. Use default value in products-var.yaml file." |& tee -a $LOG
      return
    fi
  fi  

  base64_value=$(cat $filepath | base64 | tr -d '\n')
  #yq w -i ${PRODUCTSVAR_FILE} "${varkey}" "${base64_value}"
  yq -i ".${varkey}=\"${base64_value}\"" ${PRODUCTSVAR_FILE}

  #if [[ $KEY_LENTH -eq 1 ]]; then
  #  sed -i 's/^'$KEYS':.*/'$KEYS': '$base64_value'/' $PRODUCTSVAR_FILE
  #fi
  #if [[ $KEY_LENTH -eq 2 ]]; then
  #  key_1=${KEYS[0]}
  #  key_2=${KEYS[1]}
  #  sed -i '/^'$key_1':/{:a;N;/  '$key_2':.*/!{$!ba};s/  '$key_2':.*/  '$key_2': '$base64_value'/}' $PRODUCTSVAR_FILE
  #fi
}


############################
#### Function: get_mdt_cli_pod
#### Fetch the pod name of mdt-cli pod
############################
###
###function get_mdt_cli_pod(){
###    # Get mdt-cli POD name
###    MDT_CLI=$(kubectl get pod -n ${MDT_NAMESPACE} | grep 'mdt-cli-' | awk '{print $1}')
###}

#########################
# Function: wait_container_running
# Wait till the pod is in running status
#########################

function wait_container_running() {
    #set +ex
    local _name=$1
    echo "wait_container_running $_name"
    kubectl -n ${MDT_NAMESPACE} get pod -o wide | grep ${_name} | grep "Running"
    while [ $? -ne 0 ]; do
        sleep 10
        kubectl -n ${MDT_NAMESPACE} get pod -o wide | grep ${_name} | grep "Running"
    done
    #set -ex
}

#########################
# Function: ensure_kubectl_connection
# To ensure the kubectl have kube-config provided properly to connecto to the K8S cluster
#########################

function ensure_kubectl_connection() {
  echo '[INFO] Checking kubectl connection.'
  for (( i=1; i<=3; i++ ))
  do
    if [[  $(kubectl version --output=yaml > /dev/null 2>&1; echo $?) -eq 0 ]]; then
      break
    else
      if [[ $i -eq 3 ]]; then
        kubectl version --output=yaml
        echo '[ERROR] Make sure the kube-config file is configured properly and the K8S cluser is reachable before rerunning the cms deployment scripts.'
        exit 1
      fi
      sleep 5
    fi
  done
}


#########################
# Function: user_confirm
# Asking for user confirmation
#########################

function user_confirm() {
    local _message=$1
    echo "######################################"
    echo "##          !! WARNING !!           ##"
    echo "######################################"
    echo -e "$_message"

    while true; do
        read -p "Are you sure to continue? (yes/no)" answer
        case $answer in
            Yes|yes|YES|y|Y ) echo 'now continue ...'; break;;
            No|no|NO|n|N ) exit;;
            * ) echo 'please type "yes" or "no".';;
        esac
    done
}

#########################
# Function: split_list
# Split a string list with slice number
# e.g.
# a1 a2 a3 a4 a5 --> a1+++a2 a3+++a4 a5
# We can later on use the awk to split to sub list and handle the sub list
#########################
function split_list(){
  _array=($1)
  _slicenum=$2

  array_length=${#_array[@]}
  #echo lengh is $array_length
  
  new_string=""

  for (( i=0;i<$array_length;i=$i+$_slicenum ));do 
    e=$(($i + $_slicenum - 1))
    
    slice=""

    if [[ $e -ge $array_length ]];then
      #echo "[DEBUG] Slicing index from $i to $array_length"
      for (( j=$i;j<=$array_length;j++ ));do
        slice="$slice${_array[j]}+++"
      done
    else
      #echo "[DEBUG] Slicing index from $i to $e"
      for (( j=$i;j<=$e;j++ ));do
        slice="$slice${_array[j]}+++"
      done
      slice=$(echo $slice|sed -e 's/+++$//')" "
    fi
    
    #echo "[DEBUG] slice is: $slice"
    new_string="$new_string$slice"
  done
  echo $new_string|sed -e 's/+*$//'
}

# detect api version, use stable version first, then beta, alpha
function get_api_apiversion() {
  _api=$1
  _api_version=$(kubectl api-versions)
  if echo "$_api_version" | grep -qw "^${_api}/v1" ; then
    printf "v1"
  elif echo "$_api_version" | grep -qw "^${_api}/v1beta1" ; then
    printf "v1beta1"
  elif echo "$_api_version" | grep -qw "^${_api}/v1alpha1" ; then
    printf "v1alpha1"
  fi
}

#########################
# Function: gen_self_cert
# Generating self-assigned certifications with openssl
# This function makes use of kubectl csr, so it can be used only after cluster is ready
#########################

function gen_self_cert() {
    ca_folder="/tmp/cms_ca"
    _commonname=$1
    _namespace=$2
    _clustername=$3

    stable_csr_apiversion=$(get_api_apiversion certificates.k8s.io)

    rm -rf $ca_folder
    mkdir -p $ca_folder
    if [[ -z "$_clustername" ]]; then
      #kubectl config view --raw -o yaml|yq r - 'clusters[0].cluster.certificate-authority-data'  > $ca_folder/ca.crt
      kubectl config view --raw -o yaml|yq '.clusters[0].cluster.certificate-authority-data' - > $ca_folder/ca.crt
    else
      if [[ ${CLOUDPLATFORM} == "GCP" ]]; then
	#gke_tvaas-poc_us-west2_cms-cluster-casey
        kubectl config view --raw -o yaml|yq e ".clusters[]|select(.name==\"gke_${GCP_PROJECT}_${GCP_REGION}_${_clustername}\")" -| yq '.cluster.certificate-authority-data' - > $ca_folder/ca.crt
      elif [[ ${CLOUDPLATFORM} == "AWS" ]]; then  
        kubectl config view --raw -o yaml|yq e ".clusters[]|select(.name==\"$AWS_EKS_ARN\")" -| yq '.cluster.certificate-authority-data' - > $ca_folder/ca.crt
      else
        kubectl config view --raw -o yaml|yq e ".clusters[]|select(.name==\"$_clustername\")" -| yq '.cluster.certificate-authority-data' - > $ca_folder/ca.crt
      fi
      #kubectl config view --raw -o yaml|yq r - 'clusters.(name=='$_clustername')'|yq r - 'cluster.certificate-authority-data'  > $ca_folder/ca.crt
    fi

    #==================================================
    #for cms-webhook
    #==================================================
    if [[ "${_commonname}" != "elastic" ]]; then
      cat > $ca_folder/openssl.cnf <<- EOF
		[ req ]
		distinguished_name = req_distinguished_name
		# The extensions to add to a certificate request
		req_extensions = v3_req
		[req_distinguished_name]
		[ v3_req ]
		# Extensions to add to a certificate request
		basicConstraints = CA:FALSE
		keyUsage = nonRepudiation, digitalSignature, keyEncipherment
		subjectAltName = @alt_names
		[alt_names]
		DNS.1 = ${_commonname}.${_namespace}.svc
		DNS.2 = ${_commonname}.${_namespace}.svc.$K8S_CLUSTER_DOMAIN
		EOF
      openssl genrsa -out $ca_folder/tls.key 2048 2>/dev/null
      if [[ "$stable_csr_apiversion" == "v1beta1" ]]; then
        openssl req -new -key $ca_folder/tls.key -out $ca_folder/req.csr -subj "/CN=${_commonname}.${_namespace}" -config $ca_folder/openssl.cnf
        csr_base64=$(cat $ca_folder/req.csr | base64 -w 0)
		cat > $ca_folder/csr.yaml <<- EOF
			apiVersion: certificates.k8s.io/v1beta1
			kind: CertificateSigningRequest
			metadata:
			  name: ${_commonname}-${_namespace}-csr
			spec:
			  request: ${csr_base64}
			  usages:
			  - digital signature
			  - key encipherment
			  - server auth
			  - client auth
			EOF
      elif [[ "$stable_csr_apiversion" == "v1" ]]; then
        openssl req -new -key $ca_folder/tls.key -out $ca_folder/req.csr -subj "/CN=system:node:${_commonname}.${_namespace};/O=system:nodes" -config $ca_folder/openssl.cnf
		csr_base64=$(cat $ca_folder/req.csr | base64 -w 0)
		#Use K8S CSR to genrate the certificate
		cat > $ca_folder/csr.yaml <<- EOF
			apiVersion: certificates.k8s.io/v1
			kind: CertificateSigningRequest
			metadata:
			  name: ${_commonname}-${_namespace}-csr
			spec:
			  signerName: kubernetes.io/kubelet-serving
			  request: ${csr_base64}
			  usages:
			  - digital signature
			  - key encipherment
			  - server auth
			EOF
      fi

      timeout 10 kubectl get csr ${_commonname}-${_namespace}-csr > /dev/null 2>&1 && kubectl delete csr ${_commonname}-${_namespace}-csr
      timeout 10 kubectl apply -f $ca_folder/csr.yaml
      sleep 3
      timeout 10 kubectl certificate approve ${_commonname}-${_namespace}-csr
      sleep 3
      #while [[ -z "$(cat $ca_folder/tls.crt 2>/dev/null)" ]] || grep -q "null" $ca_folder/tls.crt;do
      #  #timeout 10 kubectl get csr ${_commonname}-${_namespace}-csr -o yaml|yq r - 'status.certificate' > $ca_folder/tls.crt
      #  timeout 10 kubectl get csr ${_commonname}-${_namespace}-csr -o yaml|yq '.status.certificate' - > $ca_folder/tls.crt
      #
      #  sleep 1
      #done

      echo "[INFO] Get tls.crt for ${_commonname}"
      openssl x509 -req -in /tmp/cms_ca/req.csr -signkey /tmp/cms_ca/tls.key -out /tmp/cms_ca/tls.crt -days 3650
      #cat $ca_folder/tls.crt
      #echo "[INFO] ------------------------------------------------"

      export SELF_CA_CRT=$(cat $ca_folder/ca.crt|base64 -w 0)
      export SELF_TLS_CRT=$(cat $ca_folder/tls.crt|base64 -w 0)
      export SELF_TLS_KEY=$(cat $ca_folder/tls.key|base64 -w 0)

    #==================================================
    # for elastic
    #==================================================
    else

      cat > $ca_folder/openssl.cnf <<- EOF
		[ req ]
		distinguished_name = req_distinguished_name
		# The extensions to add to a certificate request
		req_extensions = v3_req

		[req_distinguished_name]

		[ v3_req ]
		# Extensions to add to a certificate request
		basicConstraints = CA:FALSE
		keyUsage = nonRepudiation, digitalSignature, keyEncipherment
		extendedKeyUsage = serverAuth, clientAuth
		subjectAltName = @alt_names

		[alt_names]
		DNS.1 = cms-es-logging-discovery
		DNS.2 = elasticsearch
		DNS.3 = elasticsearch-data
		DNS.4 = cms-kibana
		DNS.5 = cms-logstash
		DNS.6 = cms-es-application-discovery
		DNS.7 = cms-es-application-master
		DNS.8 = cms-es
		DNS.9 = cms-logstash-0.cms-logstash
		DNS.10 = cms-logstash-1.cms-logstash
		DNS.11 = cms-logstash-2.cms-logstash
		DNS.12 = cms-logstash-3.cms-logstash
		DNS.13 = cms-logstash-4.cms-logstash
		EOF

      #Create Self Signed CA (ca.crt and ca.key)
      openssl req -x509 -nodes -days 3650 -newkey rsa:2048 -keyout $ca_folder/ca.key -out $ca_folder/ca.crt -subj "/CN=${_commonname}.${_namespace}"  2>/dev/null
      
      openssl genrsa -out $ca_folder/tls.key 2048 2>/dev/null
      openssl req -new -key $ca_folder/tls.key -out $ca_folder/req.csr -subj "/CN=${_commonname}.${_namespace}" -config $ca_folder/openssl.cnf

      #Create certificate with self signed CA
      openssl x509 -req -in $ca_folder/req.csr -CA $ca_folder/ca.crt -CAkey $ca_folder/ca.key -CAcreateserial -out $ca_folder/tls.crt -days 3650 -extensions v3_req -extfile $ca_folder/openssl.cnf
      
      export SELF_CA_CRT=$(cat $ca_folder/ca.crt|base64 -w 0)
      export SELF_TLS_CRT=$(cat $ca_folder/tls.crt|base64 -w 0)
      export SELF_TLS_KEY=$(cat $ca_folder/tls.key|base64 -w 0)

      openssl pkcs8 -in $ca_folder/tls.key -topk8 -nocrypt -out $ca_folder/tls.pkcs8.key
      cat $ca_folder/tls.crt $ca_folder/tls.key > $ca_folder/tls.pem
      export SELF_TLS_PKCS8_KEY=$(cat $ca_folder/tls.pkcs8.key|base64 -w 0)
      export SELF_TLS_PEM=$(cat $ca_folder/tls.pem|base64 -w 0)
      
    fi
    #==================================================  

    #rm -rf $ca_folder
}

function gen_java_cert() {
    ca_folder="/tmp/cms_java_cert"
    _namespace=$1

    rm -rf $ca_folder
    mkdir -p $ca_folder

    pass="watchpoint-cms"

    keystore=$(kubectl get secret --request-timeout 30s -n $_namespace elastic-tls -o jsonpath='{.data.tls\.keystore}')
    if [ $keystore ] ; then
      echo "[INFO] ELK java certs exists, skip create"
    else
      kubectl get secret --request-timeout 30s -n $_namespace elastic-tls -o jsonpath='{.data.tls\.crt}' | base64 -d - > $ca_folder/tls.crt
      kubectl get secret --request-timeout 30s -n $_namespace elastic-tls -o jsonpath='{.data.ca\.crt}' | base64 -d - > $ca_folder/ca.crt
      kubectl get secret --request-timeout 30s -n $_namespace elastic-tls -o jsonpath='{.data.tls\.key}' | base64 -d - > $ca_folder/tls.key
    if [[ "$OPENSSL_VERSION" =~ ^3\. ]]; then
      openssl pkcs12 -export -in $ca_folder/tls.crt -inkey $ca_folder/tls.key -out $ca_folder/tls.p12 -passout pass:$pass -legacy
    else
      openssl pkcs12 -export -in $ca_folder/tls.crt -inkey $ca_folder/tls.key -out $ca_folder/tls.p12 -passout pass:$pass
    fi
      keytool -importkeystore -v -srckeystore $ca_folder/tls.p12 -srcstoretype pkcs12 -destkeystore $ca_folder/tls.keystore -deststoretype jks -storepass $pass -srcstorepass $pass -noprompt 2>/dev/null
      keytool -import -trustcacerts -alias ca -file $ca_folder/ca.crt -keystore $ca_folder/ca.truststore -storetype jks -storepass $pass -noprompt

      JAVA_CA_TRUSTSTORE=$(cat $ca_folder/ca.truststore|base64 -w 0)
      JAVA_TLS_KEYSTORE=$(cat $ca_folder/tls.keystore|base64 -w 0)
      STORE_PASSWORD=$(echo -n $pass|base64 -w 0)

      kubectl patch secrets elastic-tls -n $_namespace -p "{\"data\":{\"ca.truststore\": \"$JAVA_CA_TRUSTSTORE\", \"tls.keystore\": \"$JAVA_TLS_KEYSTORE\", \"keystore.pass\": \"$STORE_PASSWORD\"}}"
    fi
}

function prepare_helm() {
  
  ensure_kubectl_connection

  K8S_MINOR_VERSION=$(kubectl version --output=yaml|yq '.serverVersion.minor')
 
  # Prepare helm
  if [[ $K8S_MINOR_VERSION -ge 18 ]];then
    helm_version="3.6.2-mk"
  elif [[ $K8S_MINOR_VERSION -ge 16 ]];then
    helm_version="3.4.2-mk"
  else
    helm_version="3.2.4-mk"
  fi
  
  # In containerized installer, we will pre-install the 3.6.2-mk, so actuall will not run into this install helm commands.
  if [[  $(helm version > /dev/null 2>&1; echo $?) -ne 0 || $(helm version | grep Version:\"$helm_version\" | wc -l) -eq 0 ]]; then
    
    if [[ ! -f $RPMFOLDER/helm/helm-v$helm_version-linux-amd64.tar.gz ]]; then
      check_packages "$CMS_RPM_PACKAGE"
      mkdir -p $RPMFOLDER
      for rpmpkg in $CMS_RPM_PACKAGE
      do
        sudo tar -xzf $(ls -rt packages/$rpmpkg | tail -1) -C $RPMFOLDER
      done
    fi

    if [[ -f $RPMFOLDER/helm/helm-v$helm_version-linux-amd64.tar.gz ]]; then
      sudo mkdir -p /tmp/helm_temp
      echo "[INFO] Preparing helm ..."
      sudo tar -zxf $RPMFOLDER/helm/helm-v$helm_version-linux-amd64.tar.gz -C /tmp/helm_temp
      sudo mv -f /tmp/helm_temp/linux-amd64/helm /usr/local/bin/helm
      sudo rm -rf /tmp/helm_temp
    else
      echo "[ERROR] No helm/helm-v$helm_version-linux-amd64.tar.gz found in $RPMFOLDER/helm!"
      exit 1
    fi  
  fi

  echo "[INFO] Installed Helm Version:"
  helm version

  ## Prepare public helm repository
  #if [[  $(helm repo list | grep stable > /dev/null 2>&1; echo $?) -ne 0 ]]; then
  #  echo "[INFO] Preparing public stable helm repository."
  #  #helm repo add stable https://kubernetes-charts.storage.googleapis.com/
  #  helm repo add stable https://charts.helm.sh/stable
  #  helm repo update
  #  helm repo list
  #fi
}


function azure_login() {
  if $RUN_AS_CONTAINER; then
    echo "[INFO] Not running any azure-cli command as this installation is running inside container."
  else
    # Login to azure to ensure using the expected account and subsciption from the bootstrap node.
    echo "[INFO] Initilizing login to Azure ..."
    mkdir -p ~/.kube
    az login --service-principal -u $AZURE_CLIENTID -p $AZURE_CLIENTSECRET --tenant $AZURE_TENANTID
    az account set --subscription $AZURE_SUBID
  fi
}

function azure_get_aks_credentials() {
  if $RUN_AS_CONTAINER; then
    echo "[INFO] Not running any azure-cli command as this installation is running inside container."
  else
    ## Login to azure to ensure using the expected account and subsciption from the bootstrap node.
    #azure_login
    # Get Azure kubernetes cluster credential
    if [ "$mediafirst_integration" = false ]; then
      az aks get-credentials --overwrite-existing --resource-group "${RESOURCEGROUP_NAME}" --name "${CLOUD_CLUSTER_INSTANCE_NAME}" --admin
    else
      az aks get-credentials --overwrite-existing --resource-group "${RESOURCEGROUP_NAME}" --name "${CLOUD_CLUSTER_INSTANCE_NAME}"
    fi
    #if [[ "$(whoami)" != "root" ]] && [[ ! $RUN_AS_CONTAINER ]]; then
    #  mkdir -p ~/.kube
    #  sudo \cp /root/.kube/config ~/.kube/config
    #  sudo chown -R $(whoami):cms ~/.kube > /dev/null 2>&1
    #fi
  fi
}

function set_aks_cluster_name() {
  _resourceGroupName=$1
  
  # Once the cms aks has already been created with old style name end with '-cms-cluster', we need to keep it to avoid cluster recreation.
  if [[ -f $CLOUD_ENV ]] && [[ $(grep OUTPUT_AKS_NODE_GROUP_NAME $CLOUD_ENV|grep -q _$CLUSTER_NAME-cms-cluster_; echo $?) -eq 0 ]]; then
    # old style name end with '-cms-cluster' recorded and found in env file, update the cluster name in cluster.tf file
    echo "  [Warning] Setting aks cluster name with suffix '-cms-cluster'"
    sed -i '/resource "azurerm_kubernetes_cluster" "cms_aks"/{:a;N;/name[[:space:]]*= var.cluster_name$/!{$!ba};s/name[[:space:]]*= var.cluster_name$/name = "${var.cluster_name}-cms-cluster"/}' $SCRIPTPATH/tf-aks/cluster.tf
    CLOUD_CLUSTER_INSTANCE_NAME="$CLUSTER_NAME-cms-cluster"
  elif [[ -f $CLOUD_ENV ]] && [[ $(grep OUTPUT_AKS_NODE_GROUP_NAME $CLOUD_ENV|grep -q _${CLUSTER_NAME}_; echo $?) -eq 0 ]]; then
    # new style name end without '-cms-cluster' recorded and found in env file, ensure the cluster name without suffix in cluster.tf file
    sed -i '/resource "azurerm_kubernetes_cluster" "cms_aks"/{:a;N;/name[[:space:]]*= "${var.cluster_name}-cms-cluster"/!{$!ba};s/name[[:space:]]*= "${var.cluster_name}-cms-cluster"/name = var.cluster_name/}' $SCRIPTPATH/tf-aks/cluster.tf
    #break
  else
    azure_login > /dev/null
    if [[ $(az aks show --name $CLUSTER_NAME-cms-cluster --resource-group $_resourceGroupName > /dev/null 2>&1; echo $?) -eq 0 ]]; then
      # old style name end with '-cms-cluster' found in azure, update the cluster name in cluster.tf file
      echo "  [Warning] Setting aks cluster name with suffix '-cms-cluster'"
      sed -i '/resource "azurerm_kubernetes_cluster" "cms_aks"/{:a;N;/name[[:space:]]*= var.cluster_name$/!{$!ba};s/name[[:space:]]*= var.cluster_name$/name = "${var.cluster_name}-cms-cluster"/}' $SCRIPTPATH/tf-aks/cluster.tf
      CLOUD_CLUSTER_INSTANCE_NAME="$CLUSTER_NAME-cms-cluster"
    else
      # Ensure the cluster name without suffix in cluster.tf file
      sed -i '/resource "azurerm_kubernetes_cluster" "cms_aks"/{:a;N;/name[[:space:]]*= "${var.cluster_name}-cms-cluster"/!{$!ba};s/name[[:space:]]*= "${var.cluster_name}-cms-cluster"/name = var.cluster_name/}' $SCRIPTPATH/tf-aks/cluster.tf
    fi
  fi
}

function set_azure_resource_group() {
  legacy_rg_name="$CLUSTER_NAME-cms-resourcegroup"
  if [ "$MEDIAFIRST_INTEGRATION" = false ]; then
    new_rg_name="rg-${CLUSTER_NAME}"
  else
    new_rg_name="${CLUSTER_NAME}_K8S"
  fi

  if grep -q "default[[:space:]]*= \"$legacy_rg_name\"" $TF_DEFAULT_VARIABLES; then
    # When using legacy resource groupname, it means the AKS cluster has been deployed
    # In this case, not to set additional node resource group name to avoid cluster recreation
    sed -i 's/node_resource_group[[:space:]]*=/#node_resource_group =/' $SCRIPTPATH/tf-aks/cluster.tf
    RESOURCEGROUP_NAME=$legacy_rg_name
  elif grep -q "default[[:space:]]*= \"$new_rg_name\"" $TF_DEFAULT_VARIABLES; then
    RESOURCEGROUP_NAME=$new_rg_name
  else
    # Login to azure to ensure using the expected account and subsciption from the bootstrap node.
    azure_login > /dev/null
  
    # Ensure to use the correct resource group name
    #Use az to check whether the resource group with legacy resource group name exist
    #If exist, use this legacy resource group name, otherwise, use the new formated resource group name
    legacyresourcegroup=$(az group exists -n $legacy_rg_name)
  
    if [[ $legacyresourcegroup == "true" ]]; then
      #sed -i 's/"rg-cmsaks-default"/"'$legacy_rg_name'"/' $SCRIPTPATH/tf-aks/variables.tf
      sed -i '/variable "resource_group_name"/{:a;N;/default[[:space:]]*=.*/!{$!ba};s/default[[:space:]]*=.*/default = "'$legacy_rg_name'"/}' $TF_DEFAULT_VARIABLES
      #echo "[INFO] Reource Group with legacy name format exists, setting resource group name with legacy name [$legacy_rg_name] in variables.tf"
      
      # When using legacy resource groupname, it means the AKS cluster has been deployed
      # In this case, not to set additional node resource group name to avoid cluster recreation
      sed -i 's/node_resource_group[[:space:]]*=/#node_resource_group =/' $SCRIPTPATH/tf-aks/cluster.tf
      
      RESOURCEGROUP_NAME=$legacy_rg_name
    else
      #sed -i 's/"rg-cmsaks-default"/"'$new_rg_name'"/' $SCRIPTPATH/tf-aks/variables.tf
      sed -i '/variable "resource_group_name"/{:a;N;/default[[:space:]]*=.*/!{$!ba};s/default[[:space:]]*=.*/default = "'$new_rg_name'"/}' $TF_DEFAULT_VARIABLES
      #echo "[INFO] Setting resource group name as [$new_rg_name] in variables.tf"
      RESOURCEGROUP_NAME=$new_rg_name
    fi
  fi

  set_aks_cluster_name $RESOURCEGROUP_NAME
}

function create_namespace() {
  _namespace=$1
  
  if [[  $(kubectl get ns $_namespace > /dev/null 2>&1; echo $?) -eq 0 ]]; then
    echo "[INFO] Bypass CMS namespace [$_namespace] creation as it already exists." 
  else
    echo "[INFO] Create CMS namespace [$_namespace]."
    cat <<EOF | kubectl apply -f -
apiVersion: v1
kind: Namespace
metadata:
  name: ${_namespace}
EOF

  fi
}

#Function: check_extra_labels
#Check cms.var for cms_*_extra_labels vars, which in map format
#If a var define is found, replace the corresponding extraLabels part of products-var.yaml
#Usage: check_extra_labels
function check_extra_labels() {
  local param
  local lines
  local apps=`grep -oE "^[ ]*cms_.+_extra_labels" $CMSCONF`

  for param in $apps
  do
    app=`echo $param | awk -F "_extra_labels" '{print $1}' | sed 's/^[ ]*//g'`
    lines="$(get_parameter_mapvalue $param)"

    yq -i "del(.$app.extraLabels)" $PRODUCTSVAR_FILE
    yq -i ".$app.extraLabels=\"\" | .$app.extraLabels tag=\"!!null\"" $PRODUCTSVAR_FILE

    echo "$lines" > /tmp/exxx
    while read line; do
      _key=`echo $line | awk -F "=" '{print $1}' | sed 's/\"//g' | sed 's/^[ ]*//g' | sed 's/[ ]$//g'`
      _value=`echo $line | awk -F "=" '{print $2}' | sed 's/\"//g' | sed 's/^[ ]*//g' | sed 's/[ ]$//g'`

      if [[ ${_key} =~ "#" ]]; then
        continue
      fi

      yq -i ".$app.extraLabels.$_key=\"$_value\"" $PRODUCTSVAR_FILE
    done < /tmp/exxx
    rm -f /tmp/exxx
  done
}


#Function: insert_tolerations
#When use_spot_vm set to true, 
#a tolerations should be set in helm chart template file for deploy/sts/ds/job/cronjob/pod.
function insert_tolerations() {
  local chart_folder=$1
  #echo "[DEBUG] Updating helm chart [$chart_folder] templates to insert tolerations."
  yamls=$(find -L $chart_folder/*/templates -name '*.yaml')
  kind_array=("Deployment StatefulSet DaemonSet Job CronJob Pod")
  
  cd $CHART_FOLDER/temp
  for template in ${yamls}
  do
    # Check the kind of each yaml file, only handle deploy/sts/ds/job/cronjob/pod kinds
    kind=$(grep "^kind: " $template|awk -F ': ' {'print $2'})
    if [[ "${kind_array[*]}" =~ "${kind}" ]]; then
      #echo "[Debug] Setting tolerations for $kind : $template"
      
      # Ensure to insert "tolerations:" keyword into the template
      value_tolerations_count=$(grep ".Values.[^\.]*[.]*tolerations" $template|wc -l)
      if [[ $value_tolerations_count -eq 0 ]]; then
        # No tolerations field defined in Values.yaml. 
        # In this case, we assume there could be some existing tolerations defined directly within the template.
        if [[ $kind == "Pod" ]]; then
          # Specifically handling for cms-migrate Pod
          sed -i '/^spec:/a\  tolerations:' $template
        elif [[ $kind == "CronJob" ]]; then
          grep -q "^          tolerations:$" $template || sed -i '/^        spec:/a\          tolerations:' $template
        else
          grep -q "^      tolerations:$" $template || sed -i '/^    spec:/a\      tolerations:' $template
        fi
      else
        # There is tolerations field defined in Values.yaml.
        # However, based on our CMS current chart design, tolerations field defined in Values.yaml never being truly used.
        # So in this case, we can assume there is no any existing tolerations defined directly within the template and simply inset the keywork.
        if [[ $kind == "CronJob" ]]; then
          sed -i '/^        spec:/a\          tolerations:' $template
        else
          sed -i '/^    spec:/a\      tolerations:' $template
        fi
      fi
      
      # The INSERT_SPOT_TOLERATIONS ENV was deined from scripts load_cms_charts.sh
      if $INSERT_SPOT_TOLERATIONS ; then
        # Insert spot toleration into the template under "tolerations:" keyword.
        # The file '$CHART_FOLDER/temp/spot_toleration.txt' was created in scripts load_cms_charts.sh.
        #echo "[Debug] Adding spot_toleration for [$template]"
        if [[ $kind == "Pod" ]]; then
          # Specifically handling for cms-migrate Pod
          sed -i '/^  tolerations:$/r spot_toleration.txt' $template
        elif [[ $kind == "CronJob" ]]; then
          sed -i '/^          tolerations:$/r spot_toleration_cronjob.txt' $template
        else
          sed -i '/^      tolerations:$/r spot_toleration.txt' $template
        fi
      fi
    fi
  done
  cd - > /dev/null

}
