#!/usr/bin/env bash

#set -ex
set -e

###################
#  Script Helper  #
###################
help_function() {
  echo "This script deploys CMS PrePack."
  echo "------------------------------------------"
  echo "Usage: $0 <function>"
  echo
  echo "Available Functions:"
  echo "  [no input]                           : Deploy CMS PrePack."
  echo "  --upgrade                            : Upgrade CMS PrePack."
  echo "  -c <customization_id>                : Deploy specific customized CMS PrePack."
  echo "  -s | --site <geo_domain_name>        : Deploy CMS PrePack as Primary / Secondary geo domain."
  echo "  --always-continue                    : Automatically confirm the manual confirmation request."
  echo "  -h | --help                          : Show the help text."
  echo
}

# Get full path of this script file
SCRIPT=$(readlink -f "$0")
# Get folder path of this script
SCRIPT_PATH=$(dirname "$SCRIPT")
PREPACK_BIN_PATH="${SCRIPT_PATH}/prepack-bin"
PREPACKCONF="$SCRIPT_PATH/prepack.var"
RELEASE_NAME="cms-prepack-installer"
CHART_NAME=$RELEASE_NAME

PRODUCTS_VAR_PREPACK_FILE="${SCRIPT_PATH}/ConfigBundle/products-var-prepack.yaml"

PACKAGES_PATH="${SCRIPT_PATH}/packages-prepack/"
CHARTS_PACKAGE_PATTERN="cms-prepack-charts-*.tgz"
CONTAINERS_FILE_PATTERN="cms-prepack-images-*.tgz"

######################
#  Script Functions  #
######################
function get_prepack_var_value() {
  local _parmName=$1
  local _parmValue=$(yq e ".$_parmName" "$PREPACKCONF")
  echo "$_parmValue"
}

function resolve_script_arguments() {
  while [ $# -ge 1 ]; do
    case "$1" in
    -c)
      shift
      if [ -z "$1" ] || [[ "$1" =~ ^- ]]; then
        echo '[ERROR] Customization ID should be specified with -c option.'
        exit 1
      fi
      customization="$1"
      ;;
    -s | --site)
      shift
      if [ -z "$1" ] || [[ "$1" =~ ^- ]]; then
        echo '[ERROR] Geo domain should be specified with -s option.'
        exit 1
      fi
      geo_domain="$1"
      ;;
    -h | --help)
      help_func=true
      break
      ;;
    -x)
      push_pp_images=false
      ;;
    --upgrade)
      to_upgrade=true
      ;;
    --always-continue)
      always_continue=true
      ;;
    *) ;;
    esac

    shift
  done
}

function validate_customization_id() {
  if [ $# == 0 ]; then
    echo 'No customization input.'
    exit 1
  fi
  cus_id=$1
  case "${cus_id}" in
  43686172746572 | 43686172746573) # Charter | Title migration customization ID
    echo "$cus_id will be installed."
    ;;
  *)
    echo "Customization $cus_id is not supported."
    exit 1
    ;;
  esac
}

function validate_geo_domain() {
  if [ $# == 0 ]; then
    echo 'No geo domain input.'
    exit 1
  fi
  geo_domain=$1
  case "${geo_domain}" in
  Primary | Secondary)
    echo "Domain $geo_domain is set."
    ;;
  *)
    echo "Geo domain $geo_domain is not supported. Please input the correct geo domain: Primary or Secondary"
    exit 1
    ;;
  esac
}

function prepare_chart() {
  # get the latest "cms-prepack-charts" file
  charts_package=$(find "$PACKAGES_PATH" -name "$CHARTS_PACKAGE_PATTERN" | xargs -r ls -1 -t | head -1)
  if [ ! -f "$charts_package" ]; then
    echo -e "[ERROR] Installation is terminated since no charts package is found for installing PrePack.\n"
    exit 1
  fi
  tar -xvf "$charts_package" -C "$SCRIPT_PATH/charts"
  echo '[INFO] Charts are loaded into the /charts folder.'
}

function push_images() {
  # get the latest "cms-prepack-images" file
  image=$(find "$PACKAGES_PATH" -name "$CONTAINERS_FILE_PATTERN" | xargs -r ls -1 -t | head -1)
  if [ -z "$image" ]; then
    echo -e "[ERROR] Installation is terminated since no images are found for installing PrePack.\n"
    exit 1
  fi
  echo "[INFO] Pushing $image into the docker registry..."

  if [ -z "$PREPACK_IMAGE_REGISTRY" ]; then
    bash "$SCRIPT_PATH"/deploy-cms.sh image "$image"
  else
    bash "$PREPACK_BIN_PATH"/push-prepack-image.sh "$PREPACK_IMAGE_REGISTRY" "$image"
  fi
}

function post_check() {
  count=0
  completed=false
  NAMESPACE=$1
  echo -e "\n[INFO] Checking PrePack PODs in namespace: ${NAMESPACE}..."
  while true; do
    pod_stats=$(echo "$(kubectl get pod -n $NAMESPACE --no-headers=true | grep ^$RELEASE_NAME)")
    pod_stats_completed=$(echo "$pod_stats" | awk '($3 == "Completed")')
    pod_stats_error=$(echo "$pod_stats" | awk '($3 == "Error")')
    if [[ -n $pod_stats_completed ]]; then
      completed=true
      break
    fi
    if [[ -n $pod_stats_error ]]; then
      completed=false
      echo "$pod_stats_error" >/tmp/pp_pod_stats_err.output
      break
    fi
    echo -e "PODs not completed:\n$pod_stats"
    echo
    sleep 30
    ((++count))
    if [[ $count -gt 180 ]]; then # wait over one and a half hour
      completed=false
      echo "$pod_stats" >/tmp/pp_pod_stats_err.output
      break
    fi
    continue
  done

  echo
  if [[ $completed == true ]]; then
    echo -e '\n[INFO] PrePack deployment is completed.'
  else
    echo -e "\n[ERROR] PrePack deployment failed. Please find the following Kubernetes PODs that are not completed:"
    cat /tmp/pp_pod_stats_err.output
    exit 1
  fi
}

############################################################
# MAIN
############################################################
help_func=false
push_pp_images=true
to_upgrade=false
customization=

# Process command line arguments
resolve_script_arguments "$@"

PREPACK_IMAGE_REGISTRY=$(get_prepack_var_value docker_registry)

if [[ $help_func == true ]]; then
  help_function

else
  echo -e "\nKick off PrePack deployment.\n"

  # Dealing with PrePack charts
  prepare_chart

  # Dealing with PrePack containers(images)
  if [[ $push_pp_images == true ]]; then
    push_images
  fi

  if [ -n "$customization" ]; then
    echo -e "--- Validating customized values in prepack.var... ---\n"
    validate_customization_id "$customization"
    python "$PREPACK_BIN_PATH"/DeployHelper.py save --key "customization_id" --value "$customization"
  fi

  if [ -n "$geo_domain" ]; then
    echo -e "--- Setting geo domain in prepack.var... ---\n"
    validate_geo_domain "$geo_domain"
    python "$PREPACK_BIN_PATH"/DeployHelper.py save --key "geo_domain" --value "$geo_domain"
  fi

  if [[ $to_upgrade == true ]]; then
    echo -e "--- Setting upgrade flag in prepack.var... ---\n"
  fi
  python "$PREPACK_BIN_PATH"/DeployHelper.py save --key "upgrade" --value $to_upgrade

  [[ $always_continue == true ]] && option='--always-continue'
  eval "python $PREPACK_BIN_PATH/DeployHelper.py merge $option"
  echo -e "\n--- Customized values in prepack.var are merged. ---\n"

  echo -e "--- Begin to deploy PrePack... ---\n"
  NAME_SPACE=$(python "$PREPACK_BIN_PATH"/DeployHelper.py namespace)
  existing_job=$(echo "$(kubectl get job -n "$NAME_SPACE" | grep ^$RELEASE_NAME)")
  if [[ $existing_job == *$RELEASE_NAME* ]]; then
    echo '[INFO] Installation job already exists. The installer will remove the original job and then create a new one.'
    kubectl delete job $RELEASE_NAME -n "$NAME_SPACE"
    echo
  fi
  bash "$SCRIPT_PATH"/deploy-cms.sh deploy -n $RELEASE_NAME -c $CHART_NAME --var "$PRODUCTS_VAR_PREPACK_FILE"
  post_check "$NAME_SPACE"

fi
