# this file manages the VMs created as part of the deployment
# VM Specific networking is also in this file
# General netwoking is managed in network.tf


# Create virtual machine es1
resource "azurerm_virtual_machine" "es" {
    name                  = "es${count.index + 1}-${var.install_name}"
    count = "${var.es_count}"
    location              = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name   = "${azurerm_resource_group.cms-rg.name}"
    network_interface_ids = ["${azurerm_network_interface.es_nic[count.index].id}"]
    vm_size               = "${var.dev_mode ? var.aws_instance_type_es_dev : var.aws_instance_type_es}"

    depends_on = ["azurerm_virtual_machine.cs[0]"]

    storage_os_disk {
        name              = "es${count.index + 1}-${var.install_name}-os-disk"
        caching           = "ReadWrite"
        create_option     = "FromImage"
        managed_disk_type = "${var.root_volume_type}"
#        disk_size_gb      = "${var.instance_block_size_es}"
    }

    delete_os_disk_on_termination = "${var.delete_disks_on_termination}"

    storage_data_disk {
        name              = "es${count.index + 1}-${var.install_name}-data-disk"
        caching           = "ReadWrite"
        create_option     = "Empty"
        lun               = "3"
        managed_disk_type = "${var.root_volume_type}"
        disk_size_gb      = "${var.instance_block_size_es}"
    }

    delete_data_disks_on_termination = "${var.delete_disks_on_termination}"

    storage_image_reference {
        publisher = "${var.os_publisher}"
        offer     = "${var.os_offer}"
        sku       = "${var.os_sku}"
        version   = "${var.os_version}"
    }

    os_profile {
        computer_name  = "es${count.index + 1}-${var.install_name}"
        admin_username = "${var.ssh_user}"
        admin_password = "${var.ssh_user_password}"
    }

    os_profile_linux_config {
        disable_password_authentication = false
        ssh_keys {
            path     = "/home/${var.ssh_user}/.ssh/authorized_keys"
            key_data = "${file(var.public_key_path)}"
        }
    }

	tags = {
        environment = "CMS-${var.install_name}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo -S <<< ${var.ssh_user_password} sh -c \"echo -e '${var.ssh_user} ALL=(ALL) NOPASSWD: ALL\n' >> /etc/sudoers\"",
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            host = "${azurerm_public_ip.es_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
        connection {
            host = "${azurerm_public_ip.es_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
#            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_es}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.es_count} es${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            host = "${azurerm_public_ip.es_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
}


# App Virtual NIC
resource "azurerm_network_interface" "es_nic" {
    name                = "nic-es${count.index + 1}-${var.install_name}"
    location            = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_id = "${azurerm_network_security_group.admin.id}"
    count = "${var.es_count}"
    ip_configuration {
        name                          = "nic-es${count.index + 1}-${var.install_name}"
        subnet_id                     = "${azurerm_subnet.cms-subnet.id}"
        private_ip_address_allocation = "Dynamic"
        public_ip_address_id          = "${azurerm_public_ip.es_public_ip[count.index].id}"
    }

    tags = {
        environment = "CMS-${var.install_name}"
    }
}


# App public IPs
resource "azurerm_public_ip" "es_public_ip" {
    name                         = "public-ip-es${count.index + 1}-${var.install_name}"
    count = "${var.es_count}"
    location                     = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
    allocation_method            = "Dynamic"
    domain_name_label             = "es${count.index + 1}-pub-${var.install_name}"


    tags = {
        environment = "CMS-${var.install_name}"
    }
}

output "es_nodes" {
    value = "${azurerm_public_ip.es_public_ip.*.ip_address}"
}
