# Build number: 7.2.000.27

# Installs CMS on AWS

provider "aws" {
    region = "${var.aws_region}"
    access_key = "${var.aws_access_key}"
    secret_key = "${var.aws_secret_key}"
    version = "~> 1.13"
}

provider "null" {
    version = "~> 1.0"
}

provider "template" {
    version = "~> 1.0"
}

data "aws_ami" "image_id" {
    most_recent = true
    filter {
        name = "product-code"
        values = [ "${var.aws_ami_product_code}" ]
    }
}

data "aws_availability_zones" "available" {}

resource "aws_key_pair" "keypair" {
    key_name = "cms-ssh-${var.install_name}"
    public_key = "${file(var.public_key_path)}"
}

resource "aws_vpc" "default" {
    cidr_block = "${var.aws_vpc_cidr}"
    enable_dns_hostnames = true
    tags {
        Name = "CMS-${var.install_name}"
    }
}

resource "aws_internet_gateway" "default" {
    vpc_id = "${aws_vpc.default.id}"
    tags {
        Name = "CMS-${var.install_name}"
    }
}

resource "aws_route" "internet_access" {
    route_table_id = "${aws_vpc.default.main_route_table_id}"
    destination_cidr_block = "0.0.0.0/0"
    gateway_id = "${aws_internet_gateway.default.id}"
}

resource "aws_subnet" "default" {
    count = "${min(length(data.aws_availability_zones.available.names),var.az_count)}"
    vpc_id = "${aws_vpc.default.id}"
    cidr_block = "${cidrsubnet(var.aws_subnet_cidr, ceil(log(min(length(data.aws_availability_zones.available.names),var.az_count), 2)), count.index)}"
    map_public_ip_on_launch = true
    availability_zone = "${data.aws_availability_zones.available.names[count.index]}"
    tags {
        Name = "CMS-${var.install_name}-${count.index}"
    }
}

resource "aws_subnet" "rds_subnet" {
    vpc_id = "${aws_vpc.default.id}"
    cidr_block = "${var.rds_subnet_cidr}"
    map_public_ip_on_launch = true
    availability_zone = "${data.aws_availability_zones.available.names[length(data.aws_availability_zones.available.names) - 1]}"
    tags {
        Name = "cms-db-subnet-${var.install_name}"
    }
}

resource "aws_db_subnet_group" "db_subnet_group" {
    name = "cms-db-subnet-group-${var.install_name}"
    subnet_ids = ["${aws_subnet.default.*.id}", "${aws_subnet.rds_subnet.id}"]
    tags {
        Name = "cms-db-subnet-group-${var.install_name}"
    }
}

resource "aws_security_group" "ui" {
    name = "CMS UI Security Group for ${var.install_name}"
    vpc_id = "${aws_vpc.default.id}"

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-UI-${var.install_name}"
    }
}

resource "aws_security_group_rule" "ui" {
    count = "${length(var.ui_ports)}"
    type = "ingress"
    from_port = "${element(var.ui_ports, count.index)}"
    to_port = "${element(var.ui_ports, count.index)}"
    protocol = "tcp"
    cidr_blocks = "${var.external_whitelist}"
    security_group_id = "${aws_security_group.ui.id}"
}

resource "aws_security_group_rule" "ui_eip" {
    count = "${var.app_count}"
    type = "ingress"
    from_port = 0
    to_port = 65535
    protocol = "all"
    cidr_blocks = ["${element(aws_eip.app_eip.*.public_ip, count.index)}/32"]
    security_group_id = "${aws_security_group.ui.id}"
}

resource "aws_security_group" "cs" {
    name = "CMS CS Security Group for ${var.install_name}"
    vpc_id = "${aws_vpc.default.id}"

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-CS-${var.install_name}"
    }
}

resource "aws_security_group_rule" "cs" {
    count = "${length(var.cs_ports)}"
    type = "ingress"
    from_port = "${element(var.cs_ports, count.index)}"
    to_port = "${element(var.cs_ports, count.index)}"
    protocol = "tcp"
    cidr_blocks = "${var.external_whitelist}"
    security_group_id = "${aws_security_group.cs.id}"
}

resource "aws_security_group_rule" "ftp" {
    type = "ingress"
    from_port = 10000
    to_port = "${9999 + 100 * var.app_count}"
    protocol = "tcp"
    cidr_blocks = "${var.external_whitelist}"
    security_group_id = "${aws_security_group.ui.id}"
}

resource "aws_security_group" "admin" {
    name = "Admin Access Security Group for ${var.install_name}"
    vpc_id = "${aws_vpc.default.id}"

    # White list of SSH access from bastion network
    ingress {
        from_port = 22
        to_port = 22
        protocol = "tcp"
        cidr_blocks = "${var.bastion_public_cidr}"
    }

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-Admin-${var.install_name}"
    }
}

resource "aws_security_group_rule" "admin" {
    count = "${length(var.admin_whitelist) > 0  ? 1 : 0}"
    type = "ingress"
    from_port = 0
    to_port = 65535
    protocol = "tcp"
    cidr_blocks = "${var.admin_whitelist}"
    security_group_id = "${aws_security_group.admin.id}"
}

resource "aws_security_group" "private" {
    name = "Private Subnet Security Group for ${var.install_name}"
    vpc_id = "${aws_vpc.default.id}"

    # Full TCP access to subnet
    ingress {
        from_port = 0
        to_port = 65535
        protocol = "tcp"
        cidr_blocks = [ "${var.aws_subnet_cidr}" ]
    }

    # Full UDP access to subnet
    ingress {
        from_port = 0
        to_port = 65535
        protocol = "udp"
        cidr_blocks = [ "${var.aws_subnet_cidr}"]
    }

    # Support Ping
    ingress {
        from_port = 8
        to_port = 0
        protocol = "icmp"
        cidr_blocks = [ "${var.aws_subnet_cidr}"]
    }

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-Private-${var.install_name}"
    }
}

resource "aws_security_group" "rds_security_group" {
    name = "RDS Subnet Security Group for ${var.install_name}"
    description = "Allow all inbound traffic"
    vpc_id = "${aws_vpc.default.id}"

    ingress {
        from_port = 0
        to_port = 65535
        protocol = "TCP"
        cidr_blocks = [ "${var.aws_subnet_cidr}"]
    }

    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = ["0.0.0.0/0"]
    }

    tags {
        Name = "CMS-RDS-${var.install_name}"
    }
}

# Network load balancer for UI; internet-facing
resource "aws_lb" "ui_vip" {
    name = "${var.install_name}"
    load_balancer_type = "network"
    internal = false
    subnets = ["${aws_subnet.default.*.id}"]
    depends_on = ["aws_network_interface.cs_int"]
}

# Network load balancer for internal
resource "aws_lb" "internal_vip" {
    name = "${var.install_name}-int"
    load_balancer_type = "network"
    internal = true
    subnets = ["${aws_subnet.default.*.id}"]
    depends_on = ["aws_network_interface.cs_int"]
}

# Load balancer target group for UI ports
resource "aws_lb_target_group" "back_end_ui" {
    count = "${length(var.ui_ports)}"
    port = "${element(var.ui_ports, count.index)}"
    protocol = "TCP"
    vpc_id = "${aws_vpc.default.id}"
    target_type = "instance"
}

# Load balancer target group for FTP ports (10 per app)
resource "aws_lb_target_group" "back_end_ftp" {
    count = "${var.app_count * 10}"
    port = "${10000 + count.index % 10 + 100 * (count.index / 10)}"
    protocol = "TCP"
    vpc_id = "${aws_vpc.default.id}"
    target_type = "instance"
}

# Load balancer target group for internal load-balanced ports
resource "aws_lb_target_group" "back_end_internal" {
    count = "${length(var.internal_lb_ports)}"
    port = "${element(var.internal_lb_ports, count.index)}"
    protocol = "TCP"
    vpc_id = "${aws_vpc.default.id}"
    target_type = "ip"
}

# Load balancer target group for UI ports for internal load balancer
resource "aws_lb_target_group" "back_end_internal_ui" {
    count = "${length(var.ui_ports)}"
    port = "${element(var.ui_ports, count.index)}"
    protocol = "TCP"
    vpc_id = "${aws_vpc.default.id}"
    target_type = "ip"
}

# Load balancer target group for FTP ports (10 per app) for internal load balancer
resource "aws_lb_target_group" "back_end_internal_ftp" {
    count = "${var.app_count * 10}"
    port = "${10000 + count.index % 10 + 100 * (count.index / 10)}"
    protocol = "TCP"
    vpc_id = "${aws_vpc.default.id}"
    target_type = "ip"
}

# Load balancer target group attachments for UI load balancer; each port to each CS node
resource "aws_lb_target_group_attachment" "ui" {
    count = "${length(var.ui_ports) * var.cs_count}"
    target_group_arn = "${element(aws_lb_target_group.back_end_ui.*.arn, count.index / var.cs_count)}"
    target_id = "${element(aws_instance.cs.*.id, count.index)}"
    port = "${element(var.ui_ports, count.index / var.cs_count)}"
}

resource "aws_lb_target_group_attachment" "ftp" {
    count = "${var.app_count * 10 * var.cs_count}"
    target_group_arn = "${element(aws_lb_target_group.back_end_ftp.*.arn, count.index / var.cs_count)}"
    target_id = "${element(aws_instance.cs.*.id, count.index)}"
    port = "${10000 + count.index / var.cs_count  % 10 + 100 * (count.index / var.cs_count / 10)}"
}

# Load balancer target group attachments for internal load balancer; each port to each CS node
resource "aws_lb_target_group_attachment" "internal" {
    count = "${length(var.internal_lb_ports) * var.cs_count}"
    target_group_arn = "${element(aws_lb_target_group.back_end_internal.*.arn, count.index / var.cs_count)}"
    target_id = "${element(flatten(aws_network_interface.cs_int.*.private_ips), count.index)}"
    port = "${element(var.internal_lb_ports, count.index / var.cs_count)}"
}

resource "aws_lb_target_group_attachment" "internal_ui" {
    count = "${length(var.ui_ports) * var.cs_count}"
    target_group_arn = "${element(aws_lb_target_group.back_end_internal_ui.*.arn, count.index / var.cs_count)}"
    target_id = "${element(flatten(aws_network_interface.cs_int.*.private_ips), count.index)}"
    port = "${element(var.ui_ports, count.index / var.cs_count)}"
}

resource "aws_lb_target_group_attachment" "internal_ftp" {
    count = "${var.app_count * 10 * var.cs_count}"
    target_group_arn = "${element(aws_lb_target_group.back_end_internal_ftp.*.arn, count.index / var.cs_count)}"
    target_id = "${element(flatten(aws_network_interface.cs_int.*.private_ips), count.index)}"
    port = "${10000 + count.index / var.cs_count  % 10 + 100 * (count.index / var.cs_count / 10)}"
}

# Load balancer listener for UI ports for UI load balancer
resource "aws_lb_listener" "front_end_ui" {
    count = "${length(var.ui_ports)}"
    load_balancer_arn = "${aws_lb.ui_vip.arn}"
    port = "${element(var.ui_ports, count.index)}"
    protocol = "TCP"

    default_action {
        target_group_arn = "${element(aws_lb_target_group.back_end_ui.*.arn, count.index)}"
        type = "forward"
    }
}

# Load balancer listener for FTP ports for UI load balancer
resource "aws_lb_listener" "front_end_ftp" {
    count = "${var.app_count * 10}"
    load_balancer_arn = "${aws_lb.ui_vip.arn}"
    port = "${10000 + count.index % 10 + 100 * (count.index / 10)}"
    protocol = "TCP"

    default_action {
        target_group_arn = "${element(aws_lb_target_group.back_end_ftp.*.arn, count.index)}"
        type = "forward"
    }
}

# Load balancer listener for internal ports for internal load balancer
resource "aws_lb_listener" "front_end_internal" {
    count = "${length(var.internal_lb_ports)}"
    load_balancer_arn = "${aws_lb.internal_vip.arn}"
    port = "${element(var.internal_lb_ports, count.index)}"
    protocol = "TCP"

    default_action {
        target_group_arn = "${element(aws_lb_target_group.back_end_internal.*.arn, count.index)}"
        type = "forward"
    }
}

# Load balancer listener for UI ports for internal load balancer
resource "aws_lb_listener" "front_end_internal_ui" {
    count = "${length(var.ui_ports)}"
    load_balancer_arn = "${aws_lb.internal_vip.arn}"
    port = "${element(var.ui_ports, count.index)}"
    protocol = "TCP"

    default_action {
        target_group_arn = "${element(aws_lb_target_group.back_end_internal_ui.*.arn, count.index)}"
        type = "forward"
    }
}

# Load balancer listener for FTP ports for internal load balancer
resource "aws_lb_listener" "front_end_internal_ftp" {
    count = "${var.app_count * 10}"
    load_balancer_arn = "${aws_lb.internal_vip.arn}"
    port = "${10000 + count.index % 10 + 100 * (count.index / 10)}"
    protocol = "TCP"

    default_action {
        target_group_arn = "${element(aws_lb_target_group.back_end_internal_ftp.*.arn, count.index)}"
        type = "forward"
    }
}

# CMS Configuration config.json files
resource "template_dir" "config" {
    source_dir = "${path.module}/config_templates",
    destination_dir = "${path.cwd}/config"

    vars {
        customerLicenseId = "${var.cms_license_fingerprint}"
        privateNetmask = "${cidrnetmask(var.aws_subnet_cidr)}"
        # DNS services are on .2 of the vpc subnet by AWS
        vpc_dns = "${cidrhost(var.aws_vpc_cidr, 2)}"
        consul_servers = "${var.cs_count >= 3 ? "\"cs1\",\"cs2\",\"cs3\"" : "\"cs1\"" }"
        named_master = "${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}"
        named_slaves = "${jsonencode(slice(flatten(aws_network_interface.cs_int.*.private_ips), 1, length(flatten(aws_network_interface.cs_int.*.private_ips))))}"
        cms_nas_server = "${aws_efs_file_system.cms.dns_name}"
        content_nas_server = "${aws_efs_file_system.content.dns_name}"
        mountOptions = "${join(",", var.mount_options)}"
        remount = "${var.remount}"
        privateDomain = "${var.cms_dns_domain}"
        ui_eip = "${aws_lb.ui_vip.dns_name}"
        app_eip = "${aws_lb.internal_vip.dns_name}"
        timezone = "${var.timezone}"
        dbserver = "${aws_db_instance.rds.address}"
        snmpServer = "${var.snmp_server}"
        privateNetwork = "${cidrhost(var.aws_subnet_cidr,0)}"
        customerInstanceId = "${var.customerInstanceId}"
        installName = "${var.install_name}"
    }
}

# Creating EFS file system for cms mount
resource "aws_efs_file_system" "cms" {
    creation_token = "cms_mount-${var.install_name}"
    tags {
        Name = "cms_mount-${var.install_name}"
    }
}

# Creating EFS mount target for cms mount
resource "aws_efs_mount_target" "cms" {
    count = "${min(length(data.aws_availability_zones.available.names),var.az_count)}"
    file_system_id = "${aws_efs_file_system.cms.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index)}"
    security_groups = ["${aws_security_group.private.id}"]
    depends_on = ["aws_network_interface.cs_int"]
}

# Creating EFS file system for content mount
resource "aws_efs_file_system" "content" {
    creation_token = "content_mount-${var.install_name}"
    tags {
        Name = "content_mount-${var.install_name}"
    }
}

# Creating EFS mount target for content mount
resource "aws_efs_mount_target" "content" {
    count = "${min(length(data.aws_availability_zones.available.names),var.az_count)}"
    file_system_id = "${aws_efs_file_system.content.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index)}"
    security_groups = ["${aws_security_group.private.id}"]
    depends_on = ["aws_network_interface.cs_int"]
}

resource "aws_network_interface" "cs_int" {
    count = "${var.cs_count}"
    description = "CS${count.index + 1} reserved private IP"
    subnet_id = "${element(aws_subnet.default.*.id, count.index)}"
    security_groups = ["${aws_security_group.ui.id}","${aws_security_group.cs.id}","${aws_security_group.private.id}","${aws_security_group.admin.id}"]
}

resource "aws_instance" "cs" {
    tags {
        Name = "cs${count.index + 1}-${var.install_name}"
    }
    count = "${var.cs_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }
    
    depends_on = [ "aws_lb_target_group_attachment.internal", "aws_lb_listener.front_end_internal", "aws_lb_listener.front_end_internal_ui", "aws_efs_mount_target.cms" ]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.dev_mode ? var.aws_instance_type_cs_dev : var.aws_instance_type_cs}"
    key_name = "${aws_key_pair.keypair.id}"

    network_interface {
        network_interface_id = "${element(aws_network_interface.cs_int.*.id, count.index)}"
        device_index = 0
    }

    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    # additional disk space
    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_cs}"
        volume_type = "${var.block_volume_type}"
    }

    # block space for nfs server
    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sde"
        volume_size = "${var.content_nas_volume_size}"
        volume_type = "${var.content_nas_volume_type}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/cms/staging/config",
            "sudo chown -R ${var.ssh_user}:${var.ssh_user} /opt/cms/staging"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "${template_dir.config.destination_dir}"
        destination = "/opt/cms/staging/"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "${var.package_location}"
        destination = "~${var.ssh_user}/packages"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "30m"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup space and NFS share for use by other vms
            "sudo bash /opt/vmTools/mountVMdisk.sh /vol1 \"0% 100%\"",
            "sudo bash /opt/vmTools/setupNFSServer.sh /vol1",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_cs}${count.index + 1}",

            # have to tell the network NOT to create an /etc/resolv.conf based on what comes back from AWS DHCP - we need cms/ours not theirs
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #turn off SELinux
            "sudo setenforce 0",
            "sudo sed -i --follow-symlinks 's/^SELINUX=.*/SELINUX=disabled/g' /etc/sysconfig/selinux",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            "if [ ${count.index} == '0' ]; then",
                #install rpms in the packages folder, remove rpms when done to conserve space
                "cd ~${var.ssh_user}/packages/",
                "sudo yum install -y `ls *.rpm`",

                #create cms repo
                "sudo bash /opt/repos/scripts/create_httpd_repo.sh ${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}",
                "sudo systemctl restart httpd",

                ## not needed on CS unless we tar files from here
                # add the cms license file
                "sudo mv /opt/cms/staging/config /opt/cms/installer/staging/.",
                "sudo cat << EOF | sudo tee /opt/cms/installer/staging/files/license.xml",
                "${file(var.license_file_path)}",
                "EOF",

                #Update configuration jsons
                #"sudo bash /opt/vmTools/UpdateConfigJSONs.sh ${var.cs_count} ${var.app_count} ${var.es_count} ${cidrhost(var.aws_subnet_cidr, 0)}",

                #make a tarball for other nodes
                "sudo bash /opt/cms/installer/create_config_package.sh",
            "fi",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.cs_count} cs${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
}

resource "aws_db_instance" "rds" {
    depends_on = ["aws_db_subnet_group.db_subnet_group","aws_network_interface.cs_int"]
    tags {
        Name = "rds${count.index + 1}-${var.install_name}"
    }
    multi_az = "${var.rds_multi_az}"
    allocated_storage = 100
    storage_type = "gp2"
    skip_final_snapshot = true
    engine = "postgres"
    engine_version = "${var.rds_db_version}.${var.rds_db_minor_version}"
    instance_class = "${var.dev_mode ? var.aws_instance_type_rds_dev : var.aws_instance_type_rds}"
    name = "${var.rds_db_name}"
    port = "${var.rds_db_port}"
    username = "${var.rds_db_user}"
    password = "${var.rds_db_pass}"
    db_subnet_group_name = "${aws_db_subnet_group.db_subnet_group.id}"
    parameter_group_name = "default.postgres${var.rds_db_version}"
    vpc_security_group_ids = ["${aws_security_group.rds_security_group.id}"]
}


# Prepack Configuration file - install_sysInfo.tpl -> install_sysInfo.json
data "template_file" "init2" {
    template = "${file(var.config_install_sysInfo_template)}"

    vars {
        installerIP = "${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}"
    }
}

resource "aws_eip" "app_eip" {
    count = "${var.app_count}"
    vpc = true
    depends_on = ["aws_internet_gateway.default"]
}

resource "aws_eip_association" "app_eip_assoc" {
    count = "${var.app_count}"
    instance_id   = "${element(aws_instance.app.*.id, count.index)}"
    allocation_id = "${element(aws_eip.app_eip.*.id, count.index)}"
}

resource "aws_instance" "app" {
    tags {
        Name = "app${count.index + 1}-${var.install_name}"
    }

    count = "${var.app_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }

    depends_on = ["aws_db_instance.rds","aws_instance.cs","aws_efs_mount_target.cms","aws_efs_mount_target.content"]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.dev_mode ? var.aws_instance_type_app_dev : var.aws_instance_type_app}"
    key_name = "${aws_key_pair.keypair.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index + 1)}"
    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_app}"
        volume_type = "${var.block_volume_type}"
    }

    vpc_security_group_ids = ["${aws_security_group.private.id}","${aws_security_group.admin.id}"]

    # Need to enable ssh
    provisioner "remote-exec" {
        inline = [
            #send keep alives to client to keep AWS from dropping ssh connection during long processes with no output
            "echo -e '\nClientAliveInterval 60' | sudo tee -a /etc/ssh/sshd_config",
            "sudo systemctl restart sshd"
        ]
    connection {
        type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup for using nfs client
            "sudo systemctl enable rpcbind.service",
            "sudo systemctl start rpcbind",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_app}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",

            #make create a settings file to override dbsettings
            "sudo mkdir -p /opt/db/scripts/install",
            "echo '#!/bin/bash' | sudo tee /opt/db/scripts/install/settings",
            "echo 'DB_ADMIN_PW=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'PGPD=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_PORT=${var.rds_db_port}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_TYPE=postgres' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_USER=${var.rds_db_user}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_VERSION=${var.rds_db_version}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_NAME=${var.rds_db_name}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'HOST_NAME=dbserver' | sudo tee -a /opt/db/scripts/install/settings",
            "sudo chown nobody:nobody /opt/db/scripts/install/settings",
            "sudo chmod 400 /opt/db/scripts/install/settings",
            
            #sleep to stagger app instances
            "sleep ${270*count.index}",

            "sudo bash installcmsnode.sh -n ${var.app_count} app${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
    
    provisioner "remote-exec" {
        inline = [
            # setup vsftpd
            "sudo sed -i 's/^pasv_max_port=.*/pasv_max_port=$((pasv_min_port + 9))/g' /etc/puppetlabs/code/modules/vsftpd/files/vsftpdsetup.sh",
            "echo 'pasv_promiscuous=YES' | sudo tee -a /etc/vsftpd/vsftpd.conf",
            "sudo /opt/puppetlabs/bin/puppet apply -e 'include vsftpd'",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }
}

resource "aws_instance" "es" {
    tags {
        Name = "es${count.index + 1}-${var.install_name}"
    }
    count = "${var.es_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }

    depends_on = ["aws_db_instance.rds","aws_instance.cs"]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.dev_mode ? var.aws_instance_type_es_dev : var.aws_instance_type_es}"
    key_name = "${aws_key_pair.keypair.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index + 2)}"

    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_es}"
        volume_type = "${var.block_volume_type}"
    }

    vpc_security_group_ids = ["${aws_security_group.private.id}","${aws_security_group.admin.id}"]

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }
    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_es}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.es_count} es${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
}

resource "aws_instance" "pt" {
    tags {
        Name = "pt${count.index + 1}-${var.install_name}"
    }
    count = "${var.pt_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }

    depends_on = ["aws_db_instance.rds","aws_instance.cs"]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.aws_instance_type_pt}"
    key_name = "${aws_key_pair.keypair.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index + 3)}"

    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_pt}"
        volume_type = "${var.block_volume_type}"
    }

    vpc_security_group_ids = ["${aws_security_group.private.id}","${aws_security_group.admin.id}"]

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_pt}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.pt_count} pt${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
}

#create a file with variables pointing to deployed nodes for use by jenkins jobs
data "template_file" "nodes" {
    template = "${file("nodes.sh.tpl")}"

    vars {
        customerInstanceId = "${var.customerInstanceId}"
        uiVip = "${aws_lb.ui_vip.dns_name}"
        csIps = "${join(" ",aws_instance.cs.*.public_ip)}"
        esIps = "${join(" ",aws_instance.es.*.public_ip)}"
        appIps = "${join(" ",aws_instance.app.*.public_ip)}"
        ptIps = "${join(" ",aws_instance.pt.*.public_ip)}"
        dbHost = "${replace(aws_db_instance.rds.endpoint,"/(.*):.*/","$1")}"
    }
}

resource "null_resource" "nodes" {
    provisioner "local-exec" {
        command = "echo \"${data.template_file.nodes.rendered}\" | sudo tee nodes.sh"
    }
}

output "cms_operator_ui" {
    value = "https://${aws_lb.ui_vip.dns_name}:8443"
}

output "cms_central_logger_ui" {
    value = "http://${aws_lb.ui_vip.dns_name}:9292"
}

output "cms_consul_ui" {
    value = "http://${aws_lb.ui_vip.dns_name}:8501"
}

output "cs_nodes" {
    value = "${aws_instance.cs.*.public_ip}"
}

output "es_nodes" {
    value = "${aws_instance.es.*.public_ip}"
}

output "rds endpoint" {
    value = "${aws_db_instance.rds.endpoint}"
}

output "app_nodes" {
    value = "${aws_eip.app_eip.*.public_ip}"
}

output "pt_nodes" {
    value = "${aws_instance.pt.*.public_ip}"
}


