# Create virtual machine ct1
resource "azurerm_virtual_machine" "ct" {
    count = "${ (!var.cloud_ext_ct ? 1 : 0) * (var.ct_count) }"
    name                  = "ct${count.index + 1}-${var.install_name}"
    location              = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name   = "${azurerm_resource_group.cms-rg.name}"
    network_interface_ids = ["${azurerm_network_interface.ct_nic[count.index].id}"]
    vm_size               = "${var.dev_mode ? var.aws_instance_type_ct_dev : var.aws_instance_type_ct}"
    availability_set_id   = "${azurerm_availability_set.cms-avset-ct[0].id}"


    depends_on = ["azurerm_virtual_machine.cs[0]"]

    storage_os_disk {
        name              = "ct${count.index + 1}-${var.install_name}-os-disk"
        caching           = "ReadWrite"
        create_option     = "FromImage"
        managed_disk_type = "${var.root_volume_type}"
#        disk_size_gb      = "${var.instance_block_size_ct}"
    }

    delete_os_disk_on_termination = "${var.delete_disks_on_termination}"

    #storage_data_disk {
    #    name              = "ct${count.index + 1}-${var.install_name}-data-disk"
    #    caching           = "ReadWrite"
    #    create_option     = "Empty"
    #    lun               = "3"
    #    managed_disk_type = "${var.root_volume_type}"
    #    disk_size_gb      = "${var.instance_block_size_ct}"
    #}
#
    #delete_data_disks_on_termination = "${var.delete_disks_on_termination}"

    storage_image_reference {
        publisher = "${var.os_publisher}"
        offer     = "${var.os_offer}"
        sku       = "${var.os_sku}"
        version   = "${var.os_version}"
    }

    os_profile {
        computer_name  = "ct${count.index + 1}-${var.install_name}"
        admin_username = "${var.ssh_user}"
        admin_password = "${var.ssh_user_password}"
    }

    os_profile_linux_config {
        disable_password_authentication = false
        ssh_keys {
            path     = "/home/${var.ssh_user}/.ssh/authorized_keys"
            key_data = "${file(var.public_key_path)}"
        }
    }

	tags = {
        environment = "${var.tag_environment}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo -S <<< ${var.ssh_user_password} sh -c \"echo -e '${var.ssh_user} ALL=(ALL) NOPASSWD: ALL\n' >> /etc/sudoers\"",
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            host = "${azurerm_public_ip.ct_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
        connection {
            host = "${azurerm_public_ip.ct_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

#            # create new disk space
#           "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup for using nfs client
            "sudo systemctl enable rpcbind.service",
            "sudo systemctl start rpcbind",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_ct}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            ## #Create the SMB mount point with Azure File Share
            ## "sudo mkdir -p /data/{cms,backup,content}",
            ## "echo \"${replace(azurerm_storage_share.appshare.url, "https:", "")} /data/cms cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            ## "echo \"${replace(azurerm_storage_share.backupshare.url, "https:", "")} /data/backup cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            ## "echo \"${replace(azurerm_storage_share.ctcontentshare[0].url, "https:", "")} /data/content cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            ## "sudo mount -a",
            ## "sudo mkdir -p /opt/tandbergtv;sudo ln -fs /data/cms /opt/tandbergtv/cms",
            ## "sudo mkdir -p /mnt;sudo ln -fs /data/backup /mnt/backup",
            ## "sudo ln -fs /data/content /content",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}:8008/cms-node-config.tgz",

            #sleep to stagger ct instances
            "sleep ${270*count.index}",

            "sudo bash installcmsnode.sh -t 900 -n ${var.ct_count} ct${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            host = "${azurerm_public_ip.ct_public_ip[count.index].fqdn}"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }

}

# Availability Set
resource "azurerm_availability_set" "cms-avset-ct" {
  name                         = "${var.install_name}-cms-availabilityset-ct"
  count                        = "${ (var.ct_count == 0 ? 0 : 1) }"
  location                     = "${azurerm_resource_group.cms-rg.location}"
  resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
  platform_fault_domain_count  = "${var.platform_fault_domain_count}"
  platform_update_domain_count = "${var.platform_update_domain_count}"
  managed                      = true
}

# CT Virtual NIC
resource "azurerm_network_interface" "ct_nic" {
    count              = "${ (!var.cloud_ext_ct ? 1 : 0) * (var.ct_count) }"
    name                = "nic-ct${count.index + 1}-${var.install_name}"
    location            = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_id = "${azurerm_network_security_group.admin.id}"
    ip_configuration {
        name                          = "nic-ct${count.index + 1}-${var.install_name}"
        subnet_id                     = "${azurerm_subnet.cms-subnet.id}"
        private_ip_address_allocation = "Dynamic"
        public_ip_address_id          = "${azurerm_public_ip.ct_public_ip[count.index].id}"
    }

    tags = {
        environment = "${var.tag_environment}"
    }
}


# CT public IPs
resource "azurerm_public_ip" "ct_public_ip" {
    count                       = "${ (!var.cloud_ext_ct ? 1 : 0) * (var.ct_count) }"
    name                         = "public-ip-ct${count.index + 1}-${var.install_name}"
    location                     = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
    allocation_method            = "Dynamic"
    domain_name_label             = "ct${count.index + 1}-pub-${var.install_name}"


    tags = {
        environment = "${var.tag_environment}"
    }
}

output "ct_nodes" {
    value = "${azurerm_public_ip.ct_public_ip.*.ip_address}"
}
