# this file manages the VMs created as part of the deployment
# VM Specific networking is also in this file
# General netwoking is managed in network.tf


# Create virtual machine app1
resource "azurerm_virtual_machine" "app" {
    name                  = "app${count.index + 1}-${var.install_name}"
    count = "${var.app_count}"
    location              = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name   = "${azurerm_resource_group.cms-rg.name}"
    network_interface_ids = ["${azurerm_network_interface.app_nic[count.index].id}"]
    vm_size               = "${var.dev_mode ? var.instance_type_app_dev : var.instance_type_app}"
    availability_set_id   = "${azurerm_availability_set.cms-avset-app.id}"

    depends_on = ["azurerm_virtual_machine.cs[0]"]
    #depends_on = ["azurerm_virtual_machine.nfs"]

    storage_os_disk {
        name              = "app${count.index + 1}-${var.install_name}-os-disk"
        caching           = "ReadWrite"
        create_option     = "FromImage"
        managed_disk_type = "${var.root_volume_type}"
#        disk_size_gb      = "${var.instance_block_size_app}"
    }

    delete_os_disk_on_termination = "${var.delete_disks_on_termination}"

    storage_data_disk {
        name              = "app${count.index + 1}-${var.install_name}-data-disk"
        caching           = "ReadWrite"
        create_option     = "Empty"
        lun               = "3"
        managed_disk_type = "${var.root_volume_type}"
        disk_size_gb      = "${var.dev_mode ? var.instance_block_size_app_dev : var.instance_block_size_app}"
    }

    delete_data_disks_on_termination = "${var.delete_disks_on_termination}"

    storage_image_reference {
        publisher = "${var.os_publisher}"
        offer     = "${var.os_offer}"
        sku       = "${var.os_sku}"
        version   = "${var.os_version}"
    }

    os_profile {
        computer_name  = "app${count.index + 1}-${var.install_name}"
        admin_username = "${var.ssh_user}"
        admin_password = "${var.ssh_user_password}"
    }

    os_profile_linux_config {
        disable_password_authentication = false
        ssh_keys {
            path     = "/home/${var.ssh_user}/.ssh/authorized_keys"
            key_data = "${file(var.public_key_path)}"
        }
    }

	tags = {
        environment = "${var.tag_environment}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo -S <<< ${var.ssh_user_password} sh -c \"echo -e '${var.ssh_user} ALL=(ALL) NOPASSWD: ALL\n' >> /etc/sudoers\"",
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            host = "${azurerm_public_ip.app_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
        connection {
            host = "${azurerm_public_ip.app_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }  
}

# Install 1st APP
resource "null_resource" "app1_installcms" {
    depends_on = ["azurerm_network_security_rule.app_sr","null_resource.cs_installcms"]
    count = "${var.app_count > 0 ? 1 : 0}"

  # run this in all app nodes
    connection {
        host = "${data.azurerm_public_ip.app_pub_ip[count.index].ip_address}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup for using nfs client
            "sudo systemctl enable rpcbind.service",
            "sudo systemctl start rpcbind",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_app}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            ##Create the SMB mount point with Azure File Share
            #"sudo mkdir -p /data/{cms,backup,content}",
            #"echo \"${replace(azurerm_storage_share_directory.cmsfolder.id, "https:", "")} /data/cms cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"echo \"${replace(azurerm_storage_share_directory.contentfolder.id, "https:", "")} /data/content cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"echo \"${replace(azurerm_storage_share_directory.backupfolder.id, "https:", "")} /data/backup cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"sudo mount -a",
            #"sudo mkdir -p /opt/tandbergtv;sudo ln -fs /data/cms /opt/tandbergtv/cms",
            #"sudo mkdir -p /mnt;sudo ln -fs /data/backup /mnt/backup",
            #"sudo ln -fs /data/content /content",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",

            #make create a settings file to override dbsettings
            "sudo mkdir -p /opt/db/scripts/install",
            "echo '#!/bin/bash' | sudo tee /opt/db/scripts/install/settings",
            "echo 'DB_ADMIN_PW=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'PGPD=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_PORT=${var.rds_db_port}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_TYPE=azure' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_USER=${var.rds_db_user}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_VERSION=${var.rds_db_version}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_NAME=${var.rds_db_name}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'HOST_NAME=dbserver' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'AZURE_DB=${var.install_name}edb1' | sudo tee -a /opt/db/scripts/install/settings",
            "sudo chown nobody:nobody /opt/db/scripts/install/settings",
            "sudo chmod 400 /opt/db/scripts/install/settings",

            #sleep to stagger app instances
            #"sleep ${270*count.index}",

            "sudo bash installcmsnode.sh -t 900 -n ${var.app_count} app${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",

            #Additional modification on APP node to fit the Azure SQL service DB username format
            "sudo sed -i '/<user>/d' /opt/tandbergtv/cms/conf/jreport/bin/datasource.xml",
            "sudo sed -i '/>ConnectionWFSDW</{:stop;n;{s/<\\/url>/<\\/url>\\n\\        <user>wfsdw@${var.install_name}edb1<\\/user>/};/<\\/datasource>/b;bstop}' /opt/tandbergtv/cms/conf/jreport/bin/datasource.xml",
            "sudo sed -i '/>ConnectionWFS</{:stop;n;{s/<\\/url>/<\\/url>\\n\\        <user>wfs@${var.install_name}edb1<\\/user>/};/<\\/datasource>/b;bstop}' /opt/tandbergtv/cms/conf/jreport/bin/datasource.xml",
            #"sudo sed -i '/<password>/d' /opt/tandbergtv/cms/conf/jreport/bin/datasource.xml",
            #"sudo sed -i '/>ConnectionWFSDW</{:stop;n;{s/<\\/url>/<\\/url>\\n\\        <password>wfsdw<\\/password>/};/<\\/datasource>/b;bstop}' /opt/tandbergtv/cms/conf/jreport/bin/datasource.xml",
            #"sudo sed -i '/>ConnectionWFS</{:stop;n;{s/<\\/url>/<\\/url>\\n\\        <password>Wf$1234<\\/password>/};/<\\/datasource>/b;bstop}' /opt/tandbergtv/cms/conf/jreport/bin/datasource.xml",
            "sudo sed -i '/jdbc.username=wfs/c jdbc.username=wfs@${var.install_name}edb1' /opt/tandbergtv/cms/scripts/sync_utils/conf/specids.properties",
            "sudo sed -i '/jdbc.username=wfs/c jdbc.username=wfs@${var.install_name}edb1' /opt/tandbergtv/cms/scripts/sync_utils/conf/sync.properties",
            #Need to restart CMS for jreport change to take effect
            "sudo systemctl restart cms",
            
            # setup vsftpd
            "if [ -d /etc/puppetlabs/code/modules/vsftpd ];then ",
            "sudo sed -i 's/^pasv_max_port=.*/pasv_max_port=$((pasv_min_port + 9))/g' /etc/puppetlabs/code/modules/vsftpd/files/vsftpdsetup.sh",
            "echo 'pasv_promiscuous=YES' | sudo tee -a /etc/vsftpd/vsftpd.conf",
            "sudo /opt/puppetlabs/bin/puppet apply -e 'include vsftpd'",
            "fi",
        ]
    }
}

# Install other APPs after APP1 installed
resource "null_resource" "appx_installcms" {
    depends_on = ["azurerm_network_security_rule.app_sr","null_resource.prepare_rpms","null_resource.app1_installcms"]
    count = "${var.app_count > 1 ? var.app_count - 1 : 0}"

  # run this in all app nodes
    connection {
        host = "${data.azurerm_public_ip.app_pub_ip[count.index + 1].ip_address}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

#            # create new disk space
#           "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",


            # setup for using nfs client
            "sudo systemctl enable rpcbind.service",
            "sudo systemctl start rpcbind",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_app}${count.index + 2}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            ##Create the SMB mount point with Azure File Share
            #"sudo mkdir -p /data/{cms,backup,content}",
            #"echo \"${replace(azurerm_storage_share_directory.cmsfolder.id, "https:", "")} /data/cms cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"echo \"${replace(azurerm_storage_share_directory.contentfolder.id, "https:", "")} /data/content cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"echo \"${replace(azurerm_storage_share_directory.backupfolder.id, "https:", "")} /data/backup cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"sudo mount -a",
            #"sudo mkdir -p /opt/tandbergtv;sudo ln -fs /data/cms /opt/tandbergtv/cms",
            #"sudo mkdir -p /mnt;sudo ln -fs /data/backup /mnt/backup",
            #"sudo ln -fs /data/content /content",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",

            #make create a settings file to override dbsettings
            "sudo mkdir -p /opt/db/scripts/install",
            "echo '#!/bin/bash' | sudo tee /opt/db/scripts/install/settings",
            "echo 'DB_ADMIN_PW=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'PGPD=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_PORT=${var.rds_db_port}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_TYPE=azure' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_USER=${var.rds_db_user}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_VERSION=${var.rds_db_version}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_NAME=${var.rds_db_name}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'HOST_NAME=dbserver' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'AZURE_DB=${var.install_name}edb1' | sudo tee -a /opt/db/scripts/install/settings",
            "sudo chown nobody:nobody /opt/db/scripts/install/settings",
            "sudo chmod 400 /opt/db/scripts/install/settings",

            #sleep to stagger app instances
            #"sleep ${270*count.index}",

            "sudo bash installcmsnode.sh -t 900 -n ${var.app_count} app${count.index + 2} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",

            ##Additional modification on APP node to fit the Azure SQL service DB username format
            ##No need to do this again for APPX since it has been modified in APP1

            # setup vsftpd
            "if [ -d /etc/puppetlabs/code/modules/vsftpd ];then ",
            "sudo sed -i 's/^pasv_max_port=.*/pasv_max_port=$((pasv_min_port + 9))/g' /etc/puppetlabs/code/modules/vsftpd/files/vsftpdsetup.sh",
            "echo 'pasv_promiscuous=YES' | sudo tee -a /etc/vsftpd/vsftpd.conf",
            "sudo /opt/puppetlabs/bin/puppet apply -e 'include vsftpd'",
            "fi",
         ]
    }
}

# Availability Set
resource "azurerm_availability_set" "cms-avset-app" {
  name                         = "${var.install_name}-cms-availabilityset-app"
  location                     = "${azurerm_resource_group.cms-rg.location}"
  resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
  platform_fault_domain_count  = "${var.platform_fault_domain_count}"
  platform_update_domain_count = "${var.platform_update_domain_count}"
  managed                      = true
}
# App Virtual NIC
resource "azurerm_network_interface" "app_nic" {
    name                = "nic-app${count.index + 1}-${var.install_name}"
    location            = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_id = "${azurerm_network_security_group.admin.id}"
    count = "${var.app_count}"
    ip_configuration {
        name                          = "nic-app${count.index + 1}-${var.install_name}"
        subnet_id                     = "${azurerm_subnet.cms-subnet.id}"
        private_ip_address_allocation = "Dynamic"
        public_ip_address_id          = "${azurerm_public_ip.app_public_ip[count.index].id}"
    }

	tags = {
        environment = "${var.tag_environment}"
    }
}


# App public IPs
resource "azurerm_public_ip" "app_public_ip" {
    name                         = "public-ip-app${count.index + 1}-${var.install_name}"
    count = "${var.app_count}"
    location                     = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
    allocation_method            = "Static"
    domain_name_label             = "app${count.index + 1}-pub-${var.install_name}"


	tags = {
        environment = "${var.tag_environment}"
    }
}

data "azurerm_public_ip" "app_pub_ip" {
  count               = "${var.app_count}"
  name                = "${azurerm_public_ip.app_public_ip[count.index].name}"
  resource_group_name = "${azurerm_virtual_machine.app[count.index].resource_group_name}"
}

resource "azurerm_network_security_rule" "app_sr" {
  count                       = "${var.app_count}"
  depends_on = ["azurerm_virtual_machine.app","data.azurerm_public_ip.app_pub_ip"]
  name                        = "sr-app${count.index + 1}-${var.install_name}"
  priority                    = "${201 + count.index}"
  direction                   = "Inbound"
  access                      = "Allow"
  protocol                    = "*"
  source_port_range           = "*"
  destination_port_range      = "*"
  #source_address_prefix       = "${azurerm_public_ip.app_public_ip[count.index].ip_address}"
  source_address_prefix       = "${data.azurerm_public_ip.app_pub_ip[count.index].ip_address}"
  destination_address_prefix  = "*"
  resource_group_name         = "${azurerm_resource_group.cms-rg.name}"
  network_security_group_name = "admin_sg"
}

output "app_public_ip_address" {
  value = "${data.azurerm_public_ip.app_pub_ip.*.ip_address}"
}

#output "app_nodes" {
#    value = "${azurerm_public_ip.app_public_ip.*.ip_address}"
#}
