# this file manages the VMs created as part of the deployment
# VM Specific networking is also in this file
# General netwoking is managed in network.tf


# Create virtual machine cs1
resource "azurerm_virtual_machine" "cs" {
    name                  = "cs${count.index + 1}-${var.install_name}"
    count = "${var.cs_count}"
    location              = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name   = "${azurerm_resource_group.cms-rg.name}"
    network_interface_ids = ["${azurerm_network_interface.cs_nic[count.index].id}"]
    vm_size               = "${var.dev_mode ? var.instance_type_cs_dev : var.instance_type_cs}"
    availability_set_id   = "${azurerm_availability_set.cms-avset-cs.id}"

    storage_os_disk {
        name              = "cs${count.index + 1}-${var.install_name}-os-disk"
        caching           = "ReadWrite"
        create_option     = "FromImage"
        managed_disk_type = "${var.root_volume_type}"
#        disk_size_gb      = "${var.instance_block_size_cs}"
    }

    delete_os_disk_on_termination = "${var.delete_disks_on_termination}"

    storage_data_disk {
        name              = "cs${count.index + 1}-${var.install_name}-data-disk"
        caching           = "ReadWrite"
        create_option     = "Empty"
        lun               = "3"
        managed_disk_type = "${var.root_volume_type}"
        disk_size_gb      = "${var.dev_mode ? var.instance_block_size_cs_dev : var.instance_block_size_cs}"
    }

    delete_data_disks_on_termination = "${var.delete_disks_on_termination}"

    storage_image_reference {
        publisher = "${var.os_publisher}"
        offer     = "${var.os_offer}"
        sku       = "${var.os_sku}"
        version   = "${var.os_version}"
    }

    os_profile {
        computer_name  = "cs${count.index + 1}-${var.install_name}"
        admin_username = "${var.ssh_user}"
        admin_password = "${var.ssh_user_password}"
    }

    os_profile_linux_config {
        disable_password_authentication = false
        ssh_keys {
            path     = "/home/${var.ssh_user}/.ssh/authorized_keys"
            key_data = "${file(var.public_key_path)}"
        }
    }

	tags = {
        environment = "${var.tag_environment}"
    }

    #depends_on = ["azurerm_storage_share.appshare","azurerm_storage_share.backupshare"]
    #depends_on = ["azurerm_virtual_machine.nfs"]
    depends_on = ["azurerm_storage_share_directory.cmsfolder","azurerm_storage_share_directory.backupfolder","azurerm_storage_share_directory.contentfolder"]

    provisioner "remote-exec" {
        inline = [
            "sudo -S <<< ${var.ssh_user_password} sh -c \"echo -e '${var.ssh_user} ALL=(ALL) NOPASSWD: ALL\n' >> /etc/sudoers\"",
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            host = "${azurerm_public_ip.cs_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
        connection {
            host = "${azurerm_public_ip.cs_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    # Put the private keys to the VM
    provisioner "file" {
        source = "${var.private_key_path}"
        destination = "~${var.ssh_user}/.ssh/id_rsa"
        connection {
            host = "${azurerm_public_ip.cs_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/cms/staging/config",
            "sudo chown -R ${var.ssh_user}:${var.ssh_user} /opt/cms/staging"
        ]
        connection {
            host = "${azurerm_public_ip.cs_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "${template_dir.config.destination_dir}"
        destination = "/opt/cms/staging/"
        connection {
            host = "${azurerm_public_ip.cs_public_ip[count.index].fqdn}"
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }



}


#Put CMS SWO and installer rpms to cs1 node as installer node
resource "null_resource" "prepare_rpms" {
    depends_on = ["azurerm_virtual_machine.cs[0]"]
    # Changes to any instance of the cluster requires re-provisioning
    #triggers = {
    #    cs1_id = "${azurerm_virtual_machine.cs[0].id}"
    #}

    # only need to put rpms to cs1 node as installer node
    connection {
        host = "${azurerm_public_ip.cs_public_ip[0].fqdn}"
        #host = "${data.azurerm_public_ip.cs_pub_ip[0].ip_address}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    # Put the packages folder to CS1
    provisioner "file" {
        source = "${var.package_location}"
        destination = "~${var.ssh_user}"
    }

}

# Operations to be done on all cs nodes
resource "null_resource" "cs_installcms" {
    depends_on = ["azurerm_network_security_rule.cs_sr","null_resource.prepare_rpms"]
    count = "${var.cs_count}"
    ## Changes to any instance of the cluster requires re-provisioning
    #triggers = {
    #    cs_ids = "${azurerm_virtual_machine.cs[0].id)}"
    #}

  # run this in all cs nodes
    connection {
        #host = "${azurerm_public_ip.cs_public_ip[count.index].fqdn}"
        host = "${data.azurerm_public_ip.cs_pub_ip[count.index].ip_address}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # make sure the private key can be used for remote login
            "chmod 600 ~${var.ssh_user}/.ssh/id_rsa",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /var/lib/elasticsearch \"26% 100%\"",

            ## Mount the data disk for storing elasticsearch indices
            #"sudo bash /opt/vmTools/mountVMdisk_azure.sh /var/lib/elasticsearch \"0% 100%\"",

            #bypass the nas server setup since we are using Azure File Share
            #"sudo mkdir /vol1",
            #"sudo bash /opt/vmTools/setupNFSServer.sh /vol1",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_cs}${count.index + 1}",

            # have to tell the network NOT to create an /etc/resolv.conf based on what comes back from AWS DHCP - we need cms/ours not theirs
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #turn off SELinux
            "sudo setenforce 0",
            "sudo sed -i --follow-symlinks 's/^SELINUX=.*/SELINUX=disabled/g' /etc/sysconfig/selinux",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            "if [ ${count.index} == '0' ]; then",
                #install rpms in the packages folder, remove rpms when done to conserve space
                "cd ~${var.ssh_user}/packages/",
                "sudo yum install -y `ls *.rpm`",
                #This is for temp usage for DB_user issues new rpms
                "sudo mv ~${var.ssh_user}/packages/updates/*.rpm /opt/repos/ericsson-cms",
                "sudo chown root:root /opt/repos/ericsson-cms/*.rpm",
                "sudo yum install -y createrepo",
                "sudo yum clean all; sudo createrepo -c cache -s sha --database /opt/repos/ericsson-cms",

                #create cms repo
                "sudo bash /opt/repos/scripts/create_httpd_repo.sh ${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}",
                "sudo systemctl restart httpd",

                ## not needed on CS unless we tar files from here
                # add the cms license file
                "sudo mv /opt/cms/staging/config /opt/cms/installer/staging/.",
                "sudo cat << EOF | sudo tee /opt/cms/installer/staging/files/license.xml",
                "${file(var.license_file_path)}",
                "EOF",

                #Update configuration jsons
                #"sudo bash /opt/vmTools/UpdateConfigJSONs.sh ${var.cs_count} ${var.app_count} ${var.es_count} ${cidrhost(var.aws_subnet_cidr, 0)}",

                #make a tarball for other nodes
                "sudo bash /opt/cms/installer/create_config_package.sh",
            "fi",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",

            ##Create the SMB mount point with Azure File Share
            #"sudo mkdir -p /data/{cms,backup}",
            ##"sudo mount -t cifs //${azurerm_storage_account.storageaccount.name}.file.core.windows.net/appshare-${var.install_name} /data/cms -o vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99",
            ##"sudo mount -t cifs //${azurerm_storage_account.storageaccount.name}.file.core.windows.net/backupshare-${var.install_name} /data/backup -o vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99",
            ##"echo \"${replace(azurerm_storage_share.appshare.url, "https:", "")} /data/cms cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            ##"echo \"${replace(azurerm_storage_share.backupshare.url, "https:", "")} /data/backup cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"echo \"${replace(azurerm_storage_share_directory.cmsfolder.id, "https:", "")} /data/cms cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"echo \"${replace(azurerm_storage_share_directory.backupfolder.id, "https:", "")} /data/backup cifs vers=3.0,username=${azurerm_storage_account.storageaccount.name},password=${azurerm_storage_account.storageaccount.primary_access_key},dir_mode=0777,file_mode=0777,uid=99,gid=99,mfsymlinks\" | sudo tee -a /etc/fstab",
            #"sudo mount -a",
            #"sudo mkdir -p /opt/tandbergtv;sudo ln -fs /data/cms /opt/tandbergtv/cms",
            #"sudo mkdir -p /mnt;sudo ln -fs /data/backup /mnt/backup",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(azurerm_network_interface.cs_nic.*.private_ip_addresses), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -t 900 -n ${var.cs_count} cs${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
    }
}

# Availability Set
resource "azurerm_availability_set" "cms-avset-cs" {
  name                         = "${var.install_name}-cms-availabilityset-cs"
  location                     = "${azurerm_resource_group.cms-rg.location}"
  resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
  platform_fault_domain_count  = "${var.platform_fault_domain_count}"
  platform_update_domain_count = "${var.platform_update_domain_count}"
  managed                      = true
}

# App Virtual NIC
resource "azurerm_network_interface" "cs_nic" {
    count               = "${var.cs_count}"
    name                = "nic-cs${count.index + 1}-${var.install_name}"
    location            = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_id = "${azurerm_network_security_group.admin.id}"
    ip_configuration {
        name                          = "nic-cs${count.index + 1}-${var.install_name}"
        subnet_id                     = "${azurerm_subnet.cms-subnet.id}"
        private_ip_address_allocation = "Dynamic"
        public_ip_address_id          = "${azurerm_public_ip.cs_public_ip[count.index].id}"
    }

    tags = {
        environment = "${var.tag_environment}"
    }
}

# App public IPs
resource "azurerm_public_ip" "cs_public_ip" {
    name                         = "public-ip-cs${count.index + 1}-${var.install_name}"
    count = "${var.cs_count}"
    location                     = "${azurerm_resource_group.cms-rg.location}"
    resource_group_name          = "${azurerm_resource_group.cms-rg.name}"
    allocation_method            = "Static"
    sku                          = "${var.azure_lb_sku}"
    domain_name_label             = "cs${count.index + 1}-pub-${var.install_name}"


    tags = {
        environment = "${var.tag_environment}"
    }
}

data "azurerm_public_ip" "cs_pub_ip" {
  count               = "${var.cs_count}"
  name                = "${azurerm_public_ip.cs_public_ip[count.index].name}"
  resource_group_name = "${azurerm_virtual_machine.cs[count.index].resource_group_name}"
}

resource "azurerm_network_security_rule" "cs_sr" {
  count                       = "${var.cs_count}"
  depends_on = ["azurerm_virtual_machine.cs","data.azurerm_public_ip.cs_pub_ip"]
  name                        = "sr-cs${count.index + 1}-${var.install_name}"
  priority                    = "${101 + count.index}"
  direction                   = "Inbound"
  access                      = "Allow"
  protocol                    = "*"
  source_port_range           = "*"
  destination_port_range      = "*"
  #source_address_prefix       = "${azurerm_public_ip.cs_public_ip[count.index].ip_address}"
  source_address_prefix       = "${data.azurerm_public_ip.cs_pub_ip[count.index].ip_address}"
  destination_address_prefix  = "*"
  resource_group_name         = "${azurerm_resource_group.cms-rg.name}"
  network_security_group_name = "admin_sg"
}

resource "azurerm_network_security_rule" "ui_sr" {
    name                          = "sr-ui"
    depends_on                    = ["azurerm_virtual_machine.cs"]
    priority                      = 200
    direction                     = "Inbound"
    access                        = "Allow"
    protocol                      = "Tcp"
    source_port_range             = "*"
    destination_port_ranges       = "${var.ui_ports}"
    source_address_prefixes       = "${var.external_whitelist}"
#   destination_address_prefix    = data.azurerm_public_ip.cs_pub_ip.*.ip_address
    destination_address_prefixes  = azurerm_network_interface.cs_nic.*.private_ip_address
    resource_group_name           = "${azurerm_resource_group.cms-rg.name}"
    network_security_group_name   = "admin_sg"
}

output "cs_public_ip_address" {
  value = "${data.azurerm_public_ip.cs_pub_ip.*.ip_address}"
}

#output "cs_nodes" {
#    value = "${azurerm_public_ip.cs_public_ip.*.ip_address}"
#}
