#Put CMS SWO and installer rpms to cs1 node as installer node
resource "null_resource" "prepare_rpms" {
    depends_on = ["aws_instance.cs"]
    # only need to put rpms to cs1 node as installer node
    connection {
        host = "${element(aws_instance.cs.*.public_ip, 0)}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    # Put the packages folder to CS1
    provisioner "file" {
        source = "${var.package_location}"
        destination = "~${var.ssh_user}/packages"
    }

}

resource "aws_eip" "cs_eip" {
    count = "${(var.cloud_ext_ct ? 1 : 0) * (var.cs_count)}"
    vpc = true
    depends_on = ["aws_internet_gateway.default"]
}

resource "aws_eip_association" "cs_eip_assoc" {
    count = "${(var.cloud_ext_ct ? 1 : 0) * (var.cs_count)}"
    instance_id   = "${element(aws_instance.cs.*.id, count.index)}"
    allocation_id = "${element(aws_eip.cs_eip.*.id, count.index)}"
}


resource "aws_instance" "cs" {
    tags {
        Name = "cs${count.index + 1}-${var.install_name}"
    }
    count = "${var.cs_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }
    
    depends_on = [ "aws_lb_target_group_attachment.internal", "aws_lb_listener.front_end_internal", "aws_lb_listener.front_end_internal_ui", "aws_efs_mount_target.cms" ]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.dev_mode ? var.aws_instance_type_cs_dev : var.aws_instance_type_cs}"
    key_name = "${aws_key_pair.keypair.id}"

    network_interface {
        network_interface_id = "${element(aws_network_interface.cs_int.*.id, count.index)}"
        device_index = 0
    }

    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    # additional disk space
    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_cs}"
        volume_type = "${var.block_volume_type}"
    }

    # block space for nfs server
    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sde"
        volume_size = "${var.content_nas_volume_size}"
        volume_type = "${var.content_nas_volume_type}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/cms/staging/config",
            "sudo chown -R ${var.ssh_user}:${var.ssh_user} /opt/cms",
            "sudo chmod 755 /opt/cms"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "${template_dir.config.destination_dir}"
        destination = "/opt/cms/staging/"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "${var.package_location}"
        destination = "~${var.ssh_user}/packages"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "30m"
        }
    }
}


resource "null_resource" "cs_installcms" {
    depends_on = ["aws_instance.cs","null_resource.prepare_rpms"]
    count = "${var.cs_count}"

  # run this in all cs nodes
    connection {
        host = "${aws_instance.cs.private_ip[count.index]}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup space and NFS share for use by other vms
            "sudo bash /opt/vmTools/mountVMdisk.sh /vol1 \"0% 100%\"",
            "sudo bash /opt/vmTools/setupNFSServer.sh /vol1",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_cs}${count.index + 1}",

            # have to tell the network NOT to create an /etc/resolv.conf based on what comes back from AWS DHCP - we need cms/ours not theirs
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #turn off SELinux
            "sudo setenforce 0",
            "sudo sed -i --follow-symlinks 's/^SELINUX=.*/SELINUX=disabled/g' /etc/sysconfig/selinux",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            "if [ ${count.index} == '0' ]; then",
                #install rpms in the packages folder, remove rpms when done to conserve space
                "cd ~${var.ssh_user}/packages/",
                "sudo yum install -y `ls *.rpm`",
                #remove the packages after installed to save the disk space
                "sudo rm -f *.rpm",

                #create cms repo
                "sudo bash /opt/repos/scripts/create_httpd_repo.sh ${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}",
                "sudo systemctl restart httpd",

                ## not needed on CS unless we tar files from here
                # add the cms license file
                "sudo mv /opt/cms/staging/config /opt/cms/installer/staging/.",
                "sudo cat << EOF | sudo tee /opt/cms/installer/staging/files/license.xml",
                "${file(var.license_file_path)}",
                "EOF",

                #Update configuration jsons
                #"sudo bash /opt/vmTools/UpdateConfigJSONs.sh ${var.cs_count} ${var.app_count} ${var.es_count} ${cidrhost(var.aws_subnet_cidr, 0)}",

                #make a tarball for other nodes
                "sudo bash /opt/cms/installer/create_config_package.sh",
                #change the mode to 755 for the created package file
                "sudo chmod 755 /opt/cms/installer/cms-node-config.tgz",
            "fi",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",
            "sudo systemctl stop iptables",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.cs_count} cs${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
}

output "cs_nodes" {
    value = "${aws_instance.cs.*.public_ip}"
}
