resource "aws_instance" "ct" {
    tags {
        Name = "ct${count.index + 1}-${var.install_name}"
    }
    count = "${ (!var.cloud_ext_ct ? 1 : 0) * (var.ct_count) }"

    lifecycle {
        ignore_changes = ["ami"]
    }

    depends_on = ["aws_efs_mount_target.ct","aws_efs_mount_target.ct_content","aws_instance.cs"]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.aws_instance_type_ct}"
    key_name = "${aws_key_pair.keypair.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index + 3)}"

    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_ct}"
        volume_type = "${var.block_volume_type}"
    }

    vpc_security_group_ids = ["${aws_security_group.private.id}","${aws_security_group.admin.id}"]
}

# Operations to be done on all ct nodes
resource "null_resource" "ct_installcms" {
    depends_on = ["aws_instance.ct","null_resource.prepare_rpms"]
    count = "${var.app_count}"

  # run this in all app nodes
    connection {
        host = "${aws_instance.ct[count.index].private_ip}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_ct}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",
            "sudo systemctl stop iptables",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/cms",
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.ct_count} ct${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]

        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }

        provisioner "remote-exec" {
        inline = [
            # setup vsftpd
            "if [ -d /etc/puppetlabs/code/modules/vsftpd ];then ",
            "sudo sed -i 's/^pasv_max_port=.*/pasv_max_port=$((pasv_min_port + 9))/g' /etc/puppetlabs/code/modules/vsftpd/files/vsftpdsetup.sh",
            "echo 'pasv_promiscuous=YES' | sudo tee -a /etc/vsftpd/vsftpd.conf",
            "sudo /opt/puppetlabs/bin/puppet apply -e 'include vsftpd'",
            "fi"
        ]
    }
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
}

output "ct_nodes" {
    value = "${aws_instance.ct.*.public_ip}"
}
