provider "google" {
  project = "tvaas-poc"
  region  = var.region
}

provider "google-beta" {
  project = "tvaas-poc"
  region  = var.region
}

provider "null" {
  version = "~> 2.1"
}

provider "template" {
  version = "~> 2.1"
}

data "google_compute_image" "my_image" {
  family  = "centos-7"
  project = "centos-cloud"
}

resource "google_compute_network" "cms-network" {
  name                    = "cms-network"
  description             = "Network that is use for host CMS installation"
  auto_create_subnetworks = "false"
}

### Add Cloud NAT 

resource "google_compute_router_nat" "cloud-nat" {
  provider                           = google
  name                               = "cloud-nat"
  router                             = google_compute_router.cloud-nat.name
  nat_ip_allocate_option             = "AUTO_ONLY"
  source_subnetwork_ip_ranges_to_nat = "ALL_SUBNETWORKS_ALL_IP_RANGES"
  region                             = var.region
}

resource "google_compute_router" "cloud-nat" {
  name    = "cloud-nat"
  network = google_compute_network.cms-network.self_link
}

data "google_compute_network" "default" {
  name = "bastion"
}

# Peer from Bstion host (default) to CMS instances (cms-network)
resource "google_compute_network_peering" "bastion2cms" {
  name         = "bastion2cms"
  network      = data.google_compute_network.default.self_link
  peer_network = google_compute_network.cms-network.self_link
  #auto_create_routes = true
}

resource "google_compute_network_peering" "cms2bastion" {
  name         = "cms2bastion"
  network      = google_compute_network.cms-network.self_link
  peer_network = data.google_compute_network.default.self_link
  #auto_create_routes = true
}

## Peer VPCs with MFVP project
#resource "google_compute_network_peering" "tvaas-peering-cms-mfvp" {
#  name               = "tvaas-peering-cms-mfvp"
#  network            = "${google_compute_network.cms-network.self_link}"
#  peer_network       = "https://www.googleapis.com/compute/v1/projects/tvaas-poc-mfvp/global/networks/tvaas-mfvp"
#  auto_create_routes = true
#}
#
## Peer VPCs with VSPP project
#resource "google_compute_network_peering" "tvaas-peering-mfvp-vspp" {
#  name               = "tvaas-peering-mfvp-vspp"
#  network            = "${google_compute_network.cms-network.self_link}"
#  peer_network       = "https://www.googleapis.com/compute/v1/projects/tvaas-poc-vspp/global/networks/default"
#  auto_create_routes = true
#}

resource "google_compute_subnetwork" "subnet-us-west1-tvaas-poc" {
  name          = "subnet-us-west1-tvaas-poc"
  ip_cidr_range = var.gcp_subnet
  region        = var.region
  network       = google_compute_network.cms-network.name
  #private_ip_google_access = "true"
}

resource "google_compute_firewall" "defaut-ingress" {
  name    = "default-${var.install_name}-ingress"
  network = google_compute_network.cms-network.name

  allow {
    protocol = "TCP"
    ports    = concat(var.ui_ports, var.internal_lb_ports)
  }

  direction = "INGRESS"
}

resource "google_compute_firewall" "defaut-internal-ingress" {
  name    = "default-${var.install_name}-internal-ingress"
  network = google_compute_network.cms-network.name

  source_ranges = ["192.168.0.0/24", "10.128.0.0/9", "10.0.0.0/24"]

  allow {
    protocol = "all"
  }

  direction = "INGRESS"
}

resource "google_compute_firewall" "defaut-egress" {
  name    = "default-${var.install_name}-egress"
  network = google_compute_network.cms-network.name

  allow {
    protocol = "all"
  }

  direction = "EGRESS"
}

resource "google_compute_address" "cs_vip_internal" {
  name         = "cs-${var.name}-${var.region}-ip-internal"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"
}

resource "google_compute_address" "cs_vip_external" {
  name         = "cs-${var.name}-${var.region}-ip-external"
  region       = var.region
  address_type = "EXTERNAL"
}

# External Forwarding rules
resource "google_compute_forwarding_rule" "cs_forwarding_rule_external" {
  count                 = length(concat(var.ui_ports, var.internal_lb_ports))
  project               = var.project
  name                  = "${var.name}-cs-internal-${element(concat(var.ui_ports, var.internal_lb_ports), count.index)}"
  region                = var.region
  load_balancing_scheme = "EXTERNAL"
  target                = google_compute_target_pool.cs_pool.self_link
  ip_address            = google_compute_address.cs_vip_external.self_link
  ip_protocol           = var.ip_protocol

  port_range = element(concat(var.ui_ports, var.internal_lb_ports), count.index)
}

# Internal Forwardign rules
/*
resource "google_compute_forwarding_rule" "cs_forwarding_rule_internal" {
  #count                 = "${length(concat(var.ui_ports,var.internal_lb_ports))}"
  project               = "${var.project}"
  #name                  = "${var.name}-cs-internal-${element(concat(var.ui_ports,var.internal_lb_ports),count.index)}"
  name                  = "${var.name}-cs-internal"
  region                = "${var.region}"
  network               = "${google_compute_network.cms-network.self_link}"
  subnetwork            = "${google_compute_subnetwork.subnet-us-west1-tvaas-poc.self_link}"
  load_balancing_scheme = "INTERNAL"
  backend_service       = "${google_compute_region_backend_service.cs_backend.self_link}"
  ip_address            = "${google_compute_address.cs_vip_internal.self_link}"
  ip_protocol           = "${var.ip_protocol}"

  #ports                 = ["${element(concat(var.ui_ports,var.internal_lb_ports),count.index)}"]
  #allports = "true"
}
*/

resource "google_compute_region_backend_service" "cs_backend" {
  project          = var.project
  name             = "${var.name}-cs"
  region           = var.region
  protocol         = var.ip_protocol
  timeout_sec      = 10
  session_affinity = var.session_affinity

  backend {
    group = google_compute_instance_group.cs_group.self_link
  }

  health_checks = [google_compute_health_check.cs-consule.self_link]
}

resource "google_compute_health_check" "cs-consule" {
  name = "consule-8501-ui"

  http_health_check {
    port         = "8501"
    request_path = "/ui/"
  }
}

# Target pool for CS servers
resource "google_compute_target_pool" "cs_pool" {
  name      = "${var.name}-cs-pool"
  instances = google_compute_instance.cs.*.self_link
  //health_checks = ["${element(compact(concat(google_compute_health_check.tcp.*.self_link)), 0)}"]
}

# Unmanaged Instance group for CS
resource "google_compute_instance_group" "cs_group" {
  name        = "cs-test"
  description = "CS unmanaged group"
  zone        = "${var.region}-${var.zone}"
  network     = google_compute_network.cms-network.self_link

  instances  = google_compute_instance.cs.*.self_link
  depends_on = [google_compute_instance.cs]
}

# CMS Configuration config.json files
resource "template_dir" "config" {
  source_dir      = "${path.module}/config_templates"
  destination_dir = "${path.cwd}/config"

  vars = {
    customerLicenseId = var.cms_license_fingerprint
    privateNetmask    = cidrnetmask(var.gcp_subnet)
    # DNS services are on .2 of the vpc subnet by AWS
    # due to google set it to 8.8.8.8
    vpc_dns        = "8.8.8.8"
    consul_servers = var.cs_count >= 3 ? "\"cs1\",\"cs2\",\"cs3\"" : "\"cs1\""
    named_master = element(
      flatten(google_compute_address.cs_with_subnet_and_address.*.address),
      0,
    )
    named_slaves = jsonencode(
      slice(
        flatten(google_compute_address.cs_with_subnet_and_address.*.address),
        1,
        length(
          flatten(google_compute_address.cs_with_subnet_and_address.*.address),
        ),
      ),
    )
    cms_nas_server     = google_compute_instance.nfs.network_interface[0].network_ip
    content_nas_server = google_compute_instance.nfs.network_interface[0].network_ip
    mountOptions       = join(",", var.mount_options)
    remount            = var.remount
    privateDomain      = var.cms_dns_domain
    ui_eip             = google_compute_address.cs_vip_external.address
    app_eip            = google_compute_address.cs_vip_external.address
    timezone           = var.timezone
    #dbserver           = "${google_compute_address.pg_with_subnet_and_address.address}"
    dbserver           = google_sql_database_instance.cloudsql_master.private_ip_address
    snmpServer         = var.snmp_server
    privateNetwork     = cidrhost(var.gcp_subnet, 0)
    customerInstanceId = var.customerInstanceId
    installName        = var.install_name
  }
}

data "template_file" "init2" {
  template = file(var.config_install_sysInfo_template)

  vars = {
    installerIP = google_compute_address.cs_with_subnet_and_address[0].address
  }
}

# Create NFS server
resource "google_compute_instance" "nfs" {
  name         = "nfs"
  machine_type = "n1-highmem-2"

  //
  zone = "${var.region}-${var.zone}"

  boot_disk {
    initialize_params {
      image = "centos-cloud/centos-7"
    }
  }

  attached_disk {
    source      = google_compute_disk.nfs_disk.id
    device_name = google_compute_disk.nfs_disk.name
  }

  network_interface {
    subnetwork = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
    network_ip = cidrhost(var.gcp_subnet, 5)
    //access_config = {}
  }

  metadata_startup_script = file("scripts/nfs.sh")

  metadata = {
    device_name = "google-${google_compute_disk.nfs_disk.name}"
  }

  depends_on = [google_sql_database_instance.cloudsql_master]
}

# Create persistent disk
resource "google_compute_disk" "nfs_disk" {
  name = "nfs-disk"
  type = "pd-standard"
  zone = "${var.region}-${var.zone}"
  size = "3300"

  labels = {
    service = "nfs"
  }
}

resource "google_compute_address" "cs_with_subnet_and_address" {
  # Static internal IPs for CS compute instances
  count        = var.cs_count
  name         = "cs-adress-${count.index}"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"
  address      = "10.0.0.${count.index + 16}"
}

resource "google_compute_instance" "cs" {
  # Scpecification of CS instances.
  count        = var.cs_count
  name         = "cs-${count.index}"
  machine_type = var.dev_mode ? var.gcp_instance_type_cs_dev : var.gcp_instance_type_cs

  //
  zone = "${var.region}-${var.zone}"

  boot_disk {
    initialize_params {
      image = "centos-cloud/centos-7"
    }
  }

  network_interface {
    subnetwork = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
    network_ip = element(
      google_compute_address.cs_with_subnet_and_address.*.address,
      count.index,
    )
    // //access_config = {}
  }

  depends_on = [google_compute_instance.nfs]

  metadata_startup_script = "sed -i 's/enabled=1/enabled=0/g' /etc/yum.repos.d/epel.repo;echo -n 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf;sysctl -p;ip addr add ${google_compute_address.cs_vip_external.address}/32 dev lo;ip addr add ${google_compute_address.cs_vip_internal.address}/32 dev lo;"

  metadata = {
    ssh-keys = "test:${file(var.public_key_path)}"
  }

  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/vmTools",
      "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools",
    ]

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = "vmTools/"
    destination = "/opt/vmTools"

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/cms/staging/config",
      "sudo chown -R ${var.ssh_user}:${var.ssh_user} /opt/cms/staging",
    ]

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = template_dir.config.destination_dir
    destination = "/opt/cms/staging/"

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = var.package_location
    destination = "~${var.ssh_user}/packages"

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "30m"
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo chmod +x /opt/vmTools/*.sh",
      "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",
      "sudo bash /opt/vmTools/mountVMdisk.sh /vol1 \"0% 100%\"",
      "sudo bash /opt/vmTools/setupNFSServer.sh /vol1",
      "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_cs}${count.index + 1}",
      "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",
      "sudo setenforce 0",
      "sudo sed -i --follow-symlinks 's/^SELINUX=.*/SELINUX=disabled/g' /etc/sysconfig/selinux",
      "sudo yum erase -y python-setuptools",
      "if [ ${count.index} == '0' ]; then",
      "cd ~${var.ssh_user}/packages/",
      "sudo yum install -y `ls *.rpm`",
      "sudo bash /opt/repos/scripts/create_httpd_repo.sh ${self.network_interface[0].network_ip}",
      "sudo systemctl restart httpd",
      "sudo mv /opt/cms/staging/config /opt/cms/installer/staging/.",
      "sudo cat << EOF | sudo tee /opt/cms/installer/staging/files/license.xml",
      file(var.license_file_path),
      "EOF",
      "sudo bash /opt/cms/installer/create_config_package.sh",
      "fi",
      "sudo systemctl stop firewalld",
      "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${google_compute_address.cs_with_subnet_and_address[0].address}:8008/cms-node-config.tgz",
    ]

    #"sudo echo -n 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf",
    #"sudo sysctl -p",

    #"sudo echo -n 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf",
    #"sudo sysctl -p",

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "60m"
    }
  }
}

resource "null_resource" "cs_cluster_vip_exist" {
  count = var.cs_count

  triggers = {
    cs_vip_loadbalancer = join(
      ",",
      google_compute_forwarding_rule.cs_forwarding_rule_external.*.self_link,
    )
  }

  connection {
    type = "ssh"
    host = element(
      google_compute_instance.cs.*.network_interface.0.network_ip,
      count.index,
    )
    user        = var.ssh_user
    private_key = file(var.private_key_path)
    timeout     = "60m"
  }

  provisioner "remote-exec" {
    inline = [
      "cd /opt/cms/staging",
      "sudo bash installcmsnode.sh -n ${var.cs_count} cs${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
    ]
  }

  depends_on = [google_compute_forwarding_rule.cs_forwarding_rule_external]
}

# Other VMs

resource "google_compute_address" "es_with_subnet_and_address" {
  count        = var.es_count
  name         = "es-adress-${count.index}"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"

  #region       = "${var.region}"
  address = "10.0.0.${count.index + 32}"
}

resource "google_compute_instance" "es" {
  count        = var.es_count
  name         = "es-${count.index}"
  machine_type = var.dev_mode ? var.gcp_instance_type_es_dev : var.gcp_instance_type_es

  //
  zone = "${var.region}-${var.zone}"

  boot_disk {
    initialize_params {
      image = "centos-cloud/centos-7"
    }
  }

  network_interface {
    subnetwork = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
    network_ip = element(
      google_compute_address.es_with_subnet_and_address.*.address,
      count.index,
    )
    //access_config = {}
  }

  metadata = {
    ssh-keys = "test:${file(var.public_key_path)}"
  }

  metadata_startup_script = "sed -i 's/enabled=1/enabled=0/g' /etc/yum.repos.d/epel.repo"
  depends_on              = [google_compute_instance.cs]

  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/vmTools",
      "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools",
    ]

    connection {
      host = element(
        google_compute_instance.es.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = "vmTools/"
    destination = "/opt/vmTools"

    connection {
      host = element(
        google_compute_instance.es.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo chmod +x /opt/vmTools/*.sh",
      "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",
      "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_es}${count.index + 1}",
      "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",
      "sudo yum erase -y python-setuptools",
      "sudo systemctl stop firewalld",
      "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${google_compute_address.cs_with_subnet_and_address[0].address}:8008/cms-node-config.tgz",
      "cd /opt/cms/staging",
      "sudo bash installcmsnode.sh -n ${var.es_count} es${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
    ]

    connection {
      host = element(
        google_compute_instance.es.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "60m"
    }
  }
}

resource "google_compute_address" "app_with_subnet_and_address" {
  count        = var.app_count
  name         = "app-adress-${count.index}"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"
  address      = "10.0.0.${count.index + 48}"
}

resource "google_compute_instance" "app" {
  count        = var.app_count
  name         = "app-${count.index}"
  machine_type = var.dev_mode ? var.gcp_instance_type_app_dev : var.gcp_instance_type_app

  //
  zone = "${var.region}-${var.zone}"

  boot_disk {
    initialize_params {
      image = "centos-cloud/centos-7"
    }
  }

  metadata = {
    ssh-keys = "test:${file(var.public_key_path)}"
  }

  metadata_startup_script = "sed -i 's/enabled=1/enabled=0/g' /etc/yum.repos.d/epel.repo"

  network_interface {
    subnetwork = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
    network_ip = element(
      google_compute_address.app_with_subnet_and_address.*.address,
      count.index,
    )
    //access_config = {}
  }

  depends_on = [google_compute_instance.cs]

  # Need to enable ssh
  # Need to enable ssh
  provisioner "remote-exec" {
    inline = [
      "echo -e '\nClientAliveInterval 60' | sudo tee -a /etc/ssh/sshd_config",
      "sudo systemctl restart sshd",
    ]

    connection {
      host = element(
        google_compute_instance.app.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/vmTools",
      "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools",
    ]

    connection {
      host = element(
        google_compute_instance.app.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = "vmTools/"
    destination = "/opt/vmTools"

    connection {
      host = element(
        google_compute_instance.app.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo chmod +x /opt/vmTools/*.sh",
      "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",
      "sudo systemctl enable rpcbind.service",
      "sudo systemctl start rpcbind",
      "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_app}${count.index + 1}",
      "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",
      "sudo yum erase -y python-setuptools",
      "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
      "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",
      "sudo systemctl stop firewalld",
      "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${google_compute_address.cs_with_subnet_and_address[0].address}:8008/cms-node-config.tgz",
      "cd /opt/cms/staging",
      "sudo mkdir -p /opt/db/scripts/install",
      "echo '#!/bin/bash' | sudo tee /opt/db/scripts/install/settings",
      "echo 'DB_ADMIN_PW=${var.db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'PGPD=${var.db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'DB_PORT=${var.db_port}' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'DB_TYPE=postgres' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'DB_USER=${var.db_user}' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'DB_VERSION=9.5' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'DB_NAME=${var.db_name}' | sudo tee -a /opt/db/scripts/install/settings",
      "echo 'HOST_NAME=dbserver' | sudo tee -a /opt/db/scripts/install/settings",
      "sudo chown nobody:nobody /opt/db/scripts/install/settings",
      "sudo chmod 400 /opt/db/scripts/install/settings",
      "sleep ${270 * count.index}",
      "sudo bash installcmsnode.sh -n ${var.app_count} app${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
    ]

    connection {
      host = element(
        google_compute_instance.app.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "60m"
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo sed -i 's/^pasv_max_port=.*/pasv_max_port=$((pasv_min_port + 9))/g' /etc/puppetlabs/code/modules/vsftpd/files/vsftpdsetup.sh",
      "echo 'pasv_promiscuous=YES' | sudo tee -a /etc/vsftpd/vsftpd.conf",
      "sudo /opt/puppetlabs/bin/puppet apply -e 'include vsftpd'",
    ]

    connection {
      host = element(
        google_compute_instance.app.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }
}

resource "google_compute_address" "pt_with_subnet_and_address" {
  count        = var.pt_count
  name         = "pt-adress-${count.index}"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"
  address      = "10.0.0.${count.index + 64}"
}

resource "google_compute_instance" "pt" {
  count        = var.pt_count
  name         = "pt-${count.index}"
  machine_type = var.gcp_instance_type_pt

  //
  zone = "${var.region}-${var.zone}"

  boot_disk {
    initialize_params {
      image = "centos-cloud/centos-7"
    }
  }

  metadata = {
    ssh-keys = "test:${file(var.public_key_path)}"
  }

  metadata_startup_script = "sed -i 's/enabled=1/enabled=0/g' /etc/yum.repos.d/epel.repo"

  network_interface {
    subnetwork = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
    network_ip = element(
      google_compute_address.pt_with_subnet_and_address.*.address,
      count.index,
    )
    //access_config = {}
  }

  depends_on = [google_compute_instance.cs]

  # Need to enable ssh
  # Need to enable ssh
  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/vmTools",
      "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools",
    ]

    connection {
      host = element(
        google_compute_instance.pt.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = "vmTools/"
    destination = "/opt/vmTools"

    connection {
      host = element(
        google_compute_instance.pt.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo chmod +x /opt/vmTools/*.sh",
      "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",
      "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_pt}${count.index + 1}",
      "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",
      "sudo yum erase -y python-setuptools",
      "sudo systemctl stop firewalld",
      "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${google_compute_address.cs_with_subnet_and_address[0].address}:8008/cms-node-config.tgz",
      "cd /opt/cms/staging",
      "sudo bash installcmsnode.sh -n ${var.pt_count} pt${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
    ]

    connection {
      host = element(
        google_compute_instance.pt.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "60m"
    }
  }
}

#### Createtig custom machine with PostgressSQL instaled on it
#resource "google_compute_address" "pg_with_subnet_and_address" {
#  name         = "pgdb-adress-${count.index}"
#  region       = "${var.region}"
#  subnetwork   = "${google_compute_subnetwork.subnet-us-west1-tvaas-poc.name}"
#  address_type = "INTERNAL"
#}
#
#resource "google_compute_instance" "postgress" {
#  name         = "${var.name}-postgressdb"
#  machine_type = "${var.dev_mode ? var.gcp_instance_type_rds_dev : var.gcp_instance_type_rds}"
#
#  //
#  zone = "${var.region}-${var.zone}"
#
#  boot_disk {
#    initialize_params {
#      image = "centos-cloud/centos-7"
#    }
#  }
#
#  depends_on = ["google_compute_network_peering.bastion2cms", "google_compute_network_peering.cms2bastion"]
#
#  metadata {
#    ssh-keys = "test:${file("${var.public_key_path}")}"
#  }
#
#  network_interface {
#    subnetwork = "${google_compute_subnetwork.subnet-us-west1-tvaas-poc.name}"
#    network_ip = "${google_compute_address.pg_with_subnet_and_address.address}"
#
#    //access_config = {}
#  }
#
#  provisioner "remote-exec" {
#    inline = [
#      #update existing system packages to lattest versino
#      "sudo yum update -y",
#
#      # add Postgresql repo
#      "sudo rpm -Uvh http://yum.postgresql.org/9.5/redhat/rhel-7-x86_64/pgdg-centos95-9.5-2.noarch.rpm",
#
#      # installling PostgreSQL 9.5
#      "sudo yum -y install postgresql95-server postgresql95",
#
#      #Init DB
#      "sudo /usr/pgsql-9.5/bin/postgresql95-setup initdb",
#
#      # Enable login from 0.0.0.0/0 and pass auth
#      "sudo sed -i 's/#listen_addresses/listen_addresses/g' /var/lib/pgsql/9.5/data/postgresql.conf",
#
#      "sudo sed -i 's/localhost/*/g' /var/lib/pgsql/9.5/data/postgresql.conf",
#      "sudo sed -i 's/max_connections = 100/max_connections = 500/g' /var/lib/pgsql/9.5/data/postgresql.conf",
#      "sudo sed -i 's/ident/md5/g' /var/lib/pgsql/9.5/data/pg_hba.conf",
#      "sudo sed -i -e \"\\$ahost    all     all     0.0.0.0/0       md5\" /var/lib/pgsql/9.5/data/pg_hba.conf",
#
#      #Enable PostgreSQl server on startup
#      "sudo systemctl enable postgresql-9.5",
#
#      # PostgreSQl server
#      "sudo systemctl start postgresql-9.5",
#
#      #"sudo -u postgres bash -c psql -c 'CREATE USER ${var.db_user} WITH PASSWORD '${var.db_pass}';' ",
#      "sudo  -u postgres  psql -c \"ALTER USER ${var.db_user} WITH PASSWORD '${var.db_pass}';\"",
#    ]
#
#    connection {
#      type        = "ssh"
#      user        = "${var.ssh_user}"
#      private_key = "${file(var.private_key_path)}"
#      timeout     = "60m"
#    }
#  }
#}

# Disabling CLoud SQL as it doesn't support libxml

resource "google_compute_global_address" "private_ip_address" {
  provider      = google-beta
  name          = "private-ip-address"
  purpose       = "VPC_PEERING"
  address_type  = "INTERNAL"
  prefix_length = 16
  network       = google_compute_network.cms-network.self_link
}

resource "google_service_networking_connection" "private_vpc_connection" {
  provider                = google-beta
  network                 = google_compute_network.cms-network.self_link
  service                 = "servicenetworking.googleapis.com"
  reserved_peering_ranges = [google_compute_global_address.private_ip_address.name]
}

resource "google_sql_database_instance" "cloudsql_master" {
  #name             = "db-mast-eb5er"
  database_version = "POSTGRES_9_6"

  depends_on = [google_service_networking_connection.private_vpc_connection]

  #"google_compute_network.cms-network"

  settings {
    tier = "db-custom-8-15360"

    #ip_configuration  {
    #  authorized_networks = [
    #    #name = "all"
    #    #value = "0.0.0.0/0"
    #    "${data.null_data_source.auth_netw_postgres_allowed_1.*.outputs}",
    #    "${data.null_data_source.auth_netw_postgres_allowed_2.*.outputs}",
    #  ]
    #}
    ip_configuration {
      ipv4_enabled    = "false"
      private_network = google_compute_network.cms-network.self_link
    }
  }
}

resource "google_sql_database" "cloudsql_db" {
  name     = var.db_name
  instance = google_sql_database_instance.cloudsql_master.name
}

# create database Users
resource "google_sql_user" "db_sql_user" {
  name     = var.db_user
  instance = google_sql_database_instance.cloudsql_master.name
  host     = "%"
  password = var.db_pass
}

