
resource "google_compute_address" "cs_vip_internal" {
  name         = "cs-${var.name}-${var.region}-ip-internal"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"
}

resource "google_compute_address" "cs_vip_external" {
  name         = "cs-${var.name}-${var.region}-ip-external"
  region       = var.region
  address_type = "EXTERNAL"
}

# External Forwarding rules
resource "google_compute_forwarding_rule" "cs_forwarding_rule_external" {
  count                 = length(concat(var.ui_ports, var.internal_lb_ports))
  project               = var.project
  name                  = "${var.name}-cs-internal-${element(concat(var.ui_ports, var.internal_lb_ports), count.index)}"
  region                = var.region
  load_balancing_scheme = "EXTERNAL"
  target                = google_compute_target_pool.cs_pool.self_link
  ip_address            = google_compute_address.cs_vip_external.self_link
  ip_protocol           = var.ip_protocol

  port_range = element(concat(var.ui_ports, var.internal_lb_ports), count.index)
}

# Internal Forwardign rules
/*
resource "google_compute_forwarding_rule" "cs_forwarding_rule_internal" {
  #count                 = "${length(concat(var.ui_ports,var.internal_lb_ports))}"
  project               = "${var.project}"
  #name                  = "${var.name}-cs-internal-${element(concat(var.ui_ports,var.internal_lb_ports),count.index)}"
  name                  = "${var.name}-cs-internal"
  region                = "${var.region}"
  network               = "${google_compute_network.cms-network.self_link}"
  subnetwork            = "${google_compute_subnetwork.subnet-us-west1-tvaas-poc.self_link}"
  load_balancing_scheme = "INTERNAL"
  backend_service       = "${google_compute_region_backend_service.cs_backend.self_link}"
  ip_address            = "${google_compute_address.cs_vip_internal.self_link}"
  ip_protocol           = "${var.ip_protocol}"

  #ports                 = ["${element(concat(var.ui_ports,var.internal_lb_ports),count.index)}"]
  #allports = "true"
}
*/

resource "google_compute_region_backend_service" "cs_backend" {
  project          = var.project
  name             = "${var.name}-cs"
  region           = var.region
  protocol         = var.ip_protocol
  timeout_sec      = 10
  session_affinity = var.session_affinity

  backend {
    group = google_compute_instance_group.cs_group.self_link
  }

  health_checks = [google_compute_health_check.cs-consule.self_link]
}

resource "google_compute_health_check" "cs-consule" {
  name = "consule-8501-ui"

  http_health_check {
    port         = "8501"
    request_path = "/ui/"
  }
}

# Target pool for CS servers
resource "google_compute_target_pool" "cs_pool" {
  name      = "${var.name}-cs-pool"
  instances = google_compute_instance.cs.*.self_link
  //health_checks = ["${element(compact(concat(google_compute_health_check.tcp.*.self_link)), 0)}"]
}

# Unmanaged Instance group for CS
resource "google_compute_instance_group" "cs_group" {
  name        = "cs-test"
  description = "CS unmanaged group"
  zone        = "${var.region}-${var.zone}"
  network     = google_compute_network.cms-network.self_link

  instances  = google_compute_instance.cs.*.self_link
  depends_on = [google_compute_instance.cs]
}

resource "google_compute_address" "cs_with_subnet_and_address" {
  # Static internal IPs for CS compute instances
  count        = var.cs_count
  name         = "cs-adress-${count.index}"
  region       = var.region
  subnetwork   = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
  address_type = "INTERNAL"
  address      = "10.0.0.${count.index + 16}"
}

resource "google_compute_instance" "cs" {
  # Scpecification of CS instances.
  count        = var.cs_count
  name         = "cs-${count.index}"
  machine_type = var.dev_mode ? var.gcp_instance_type_cs_dev : var.gcp_instance_type_cs

  //
  zone = "${var.region}-${var.zone}"

  boot_disk {
    initialize_params {
      image = "centos-cloud/centos-7"
    }
  }

  network_interface {
    subnetwork = google_compute_subnetwork.subnet-us-west1-tvaas-poc.name
    network_ip = element(
      google_compute_address.cs_with_subnet_and_address.*.address,
      count.index,
    )
    // //access_config = {}
  }

  depends_on = [google_compute_instance.nfs]

  metadata_startup_script = "sed -i 's/enabled=1/enabled=0/g' /etc/yum.repos.d/epel.repo;echo -n 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf;sysctl -p;ip addr add ${google_compute_address.cs_vip_external.address}/32 dev lo;ip addr add ${google_compute_address.cs_vip_internal.address}/32 dev lo;"

  metadata = {
    ssh-keys = "test:${file(var.public_key_path)}"
  }

  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/vmTools",
      "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools",
    ]

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = "vmTools/"
    destination = "/opt/vmTools"

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo mkdir -p /opt/cms/staging/config",
      "sudo chown -R ${var.ssh_user}:${var.ssh_user} /opt/cms/staging",
    ]

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = template_dir.config.destination_dir
    destination = "/opt/cms/staging/"

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
    }
  }

  provisioner "file" {
    source      = var.package_location
    destination = "~${var.ssh_user}/packages"

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "30m"
    }
  }

  provisioner "remote-exec" {
    inline = [
      "sudo chmod +x /opt/vmTools/*.sh",
      "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",
      "sudo bash /opt/vmTools/mountVMdisk.sh /vol1 \"0% 100%\"",
      "sudo bash /opt/vmTools/setupNFSServer.sh /vol1",
      "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_cs}${count.index + 1}",
      "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",
      "sudo setenforce 0",
      "sudo sed -i --follow-symlinks 's/^SELINUX=.*/SELINUX=disabled/g' /etc/sysconfig/selinux",
      "sudo yum erase -y python-setuptools",
      "if [ ${count.index} == '0' ]; then",
      "cd ~${var.ssh_user}/packages/",
      "sudo yum install -y `ls *.rpm`",
      "sudo bash /opt/repos/scripts/create_httpd_repo.sh ${self.network_interface[0].network_ip}",
      "sudo systemctl restart httpd",
      "sudo mv /opt/cms/staging/config /opt/cms/installer/staging/.",
      "sudo cat << EOF | sudo tee /opt/cms/installer/staging/files/license.xml",
      file(var.license_file_path),
      "EOF",
      "sudo bash /opt/cms/installer/create_config_package.sh",
      "fi",
      "sudo systemctl stop firewalld",
      "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${google_compute_address.cs_with_subnet_and_address[0].address}:8008/cms-node-config.tgz",
    ]

    #"sudo echo -n 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf",
    #"sudo sysctl -p",

    #"sudo echo -n 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf",
    #"sudo sysctl -p",

    connection {
      host = element(
        google_compute_instance.cs.*.network_interface.0.network_ip,
        count.index,
      )
      type        = "ssh"
      user        = var.ssh_user
      private_key = file(var.private_key_path)
      timeout     = "60m"
    }
  }
}
