resource "aws_instance" "pt" {
    tags {
        Name = "pt${count.index + 1}-${var.install_name}"
    }
    count = "${var.pt_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }

    depends_on = ["aws_db_instance.rds","aws_instance.cs"]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.aws_instance_type_pt}"
    key_name = "${aws_key_pair.keypair.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index + 3)}"

    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_pt}"
        volume_type = "${var.block_volume_type}"
    }

    vpc_security_group_ids = ["${aws_security_group.private.id}","${aws_security_group.admin.id}"]

}

# Operations to be done on all pt nodes
resource "null_resource" "pt_installcms" {
    depends_on = ["aws_instance.pt","null_resource.prepare_rpms"]
    count = "${var.pt_count}"

  # run this in all pt nodes
    connection {
        #host = "${aws_instance.pt[count.index].private_ip}"
        host = "${element(aws_instance.pt.*.public_ip, count.index)}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "file" {
        source = "vmTools/"
        destination = "/opt/vmTools"
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
        }
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_pt}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",
            "sudo systemctl stop iptables",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/cms",
            "cd /opt/cms/staging",
            "sudo bash installcmsnode.sh -n ${var.pt_count} pt${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
        connection {
            type = "ssh"
            user = "${var.ssh_user}"
            private_key = "${file(var.private_key_path)}"
            timeout = "60m"
        }
    }
}

output "pt_nodes" {
    value = "${aws_instance.pt.*.public_ip}"
}
