resource "aws_key_pair" "keypair" {
    key_name = "cms-ssh-${var.install_name}"
    public_key = "${file(var.public_key_path)}"
}

resource "aws_vpc" "default" {
    count = "${ var.existing_vpc_id == 0 ? 1 : 0 }"
    cidr_block = "${var.aws_vpc_cidr}"
    enable_dns_hostnames = true
    tags {
        Name = "CMS-${var.install_name}"
    }
}

resource "aws_internet_gateway" "default" {
    count = "${ var.existing_vpc_id == 0 ? 1 : 0 }"
    vpc_id = "${ aws_vpc.default.id }"
    tags {
        Name = "CMS-${var.install_name}"
    }
}

resource "aws_route" "internet_access" {
    count = "${ var.existing_vpc_id == 0 ? 1 : 0 }"
    route_table_id = "${ aws_vpc.default.main_route_table_id }"
    destination_cidr_block = "0.0.0.0/0"
    gateway_id = "${ aws_internet_gateway.default.id }"
}

resource "aws_security_group" "rds_security_group" {
    name = "RDS Subnet Security Group for ${var.install_name}"
    description = "Allow all inbound traffic"
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"

    ingress {
        from_port = 0
        to_port = 65535
        protocol = "TCP"
        cidr_blocks = [ "${var.aws_subnet_cidr}"]
    }

    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = ["0.0.0.0/0"]
    }

    tags {
        Name = "CMS-RDS-${var.install_name}"
    }
}

resource "aws_db_subnet_group" "db_subnet_group" {
    name = "cms-db-subnet-group-${var.install_name}"
    subnet_ids = ["${aws_subnet.default.*.id}", "${aws_subnet.rds_subnet.id}"]
    tags {
        Name = "cms-db-subnet-group-${var.install_name}"
    }
}

resource "aws_subnet" "rds_subnet" {
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"
    cidr_block = "${var.rds_subnet_cidr}"
    map_public_ip_on_launch = true
    availability_zone = "${data.aws_availability_zones.available.names[length(data.aws_availability_zones.available.names) - 1]}"
    tags {
        Name = "cms-db-subnet-${var.install_name}"
    }
}

resource "aws_security_group" "cs" {
    name = "CMS CS Security Group for ${var.install_name}"
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-CS-${var.install_name}"
    }
}

resource "aws_security_group_rule" "cs" {
    count = "${length(var.cs_ports)}"
    type = "ingress"
    from_port = "${element(var.cs_ports, count.index)}"
    to_port = "${element(var.cs_ports, count.index)}"
    protocol = "tcp"
    cidr_blocks = "${var.external_whitelist}"
    security_group_id = "${aws_security_group.cs.id}"
}

resource "aws_network_interface" "cs_int" {
    count = "${var.cs_count}"
    description = "CS${count.index + 1} reserved private IP"
    subnet_id = "${element(aws_subnet.default.*.id, count.index)}"
    security_groups = ["${aws_security_group.ui.id}","${aws_security_group.cs.id}","${aws_security_group.private.id}","${aws_security_group.admin.id}"]
}

resource "aws_security_group" "ui" {
    name = "CMS UI Security Group for ${var.install_name}"
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-UI-${var.install_name}"
    }
}

resource "aws_security_group_rule" "ui" {
    count = "${length(var.ui_ports)}"
    type = "ingress"
    from_port = "${element(var.ui_ports, count.index)}"
    to_port = "${element(var.ui_ports, count.index)}"
    protocol = "tcp"
    cidr_blocks = "${var.external_whitelist}"
    security_group_id = "${aws_security_group.ui.id}"
}

resource "aws_security_group_rule" "ui_eip" {
    count = "${var.app_count}"
    type = "ingress"
    from_port = 0
    to_port = 65535
    protocol = "all"
    cidr_blocks = ["${element(aws_eip.app_eip.*.public_ip, count.index)}/32"]
    security_group_id = "${aws_security_group.ui.id}"
}

resource "aws_subnet" "default" {
    count = "${min(length(data.aws_availability_zones.available.names),var.az_count)}"
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"
    cidr_block = "${cidrsubnet(var.aws_subnet_cidr, ceil(log(min(length(data.aws_availability_zones.available.names),var.az_count), 2)), count.index)}"
    map_public_ip_on_launch = true
    availability_zone = "${data.aws_availability_zones.available.names[count.index]}"
    tags {
        Name = "CMS-${var.install_name}-${count.index}"
    }
}

resource "aws_security_group_rule" "ftp" {
    type = "ingress"
    from_port = 10000
    to_port = "${9999 + 100 * var.app_count}"
    protocol = "tcp"
    cidr_blocks = "${var.external_whitelist}"
    security_group_id = "${aws_security_group.ui.id}"
}

resource "aws_security_group" "admin" {
    name = "Admin Access Security Group for ${var.install_name}"
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"

    # White list of SSH access from bastion network
    ingress {
        from_port = 22
        to_port = 22
        protocol = "tcp"
        cidr_blocks = "${var.bastion_public_cidr}"
    }

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-Admin-${var.install_name}"
    }
}

resource "aws_security_group_rule" "admin" {
    count = "${length(var.admin_whitelist) > 0  ? 1 : 0}"
    type = "ingress"
    from_port = 0
    to_port = 65535
    protocol = "all"
    cidr_blocks = "${var.admin_whitelist}"
    security_group_id = "${aws_security_group.admin.id}"
}

resource "aws_security_group" "private" {
    name = "Private Subnet Security Group for ${var.install_name}"
    #vpc_id = "${aws_vpc.default.id}"
    vpc_id = "${ var.existing_vpc_id == 0 ? join("",aws_vpc.default.*.id) : var.existing_vpc_id }"

    # Full TCP access to subnet
    ingress {
        from_port = 0
        to_port = 65535
        protocol = "tcp"
        cidr_blocks = [ "${var.aws_subnet_cidr}" ]
    }

    # Full UDP access to subnet
    ingress {
        from_port = 0
        to_port = 65535
        protocol = "udp"
        cidr_blocks = [ "${var.aws_subnet_cidr}"]
    }

    # Support Ping
    ingress {
        from_port = 8
        to_port = 0
        protocol = "icmp"
        cidr_blocks = [ "${var.aws_subnet_cidr}"]
    }

    # outbound internet access
    egress {
        from_port = 0
        to_port = 0
        protocol = "-1"
        cidr_blocks = [ "0.0.0.0/0" ]
    }

    tags {
        Name = "CMS-Private-${var.install_name}"
    }
}
