resource "aws_eip" "app_eip" {
    count = "${var.app_count}"
    vpc = true
}

resource "aws_eip_association" "app_eip_assoc" {
    count = "${var.app_count}"
    instance_id   = "${element(aws_instance.app.*.id, count.index)}"
    allocation_id = "${element(aws_eip.app_eip.*.id, count.index)}"
}

resource "aws_instance" "app" {
    tags {
        Name = "app${count.index + 1}-${var.install_name}"
    }

    count = "${var.app_count}"

    lifecycle {
        ignore_changes = ["ami"]
    }

    depends_on = ["aws_efs_mount_target.cms","aws_efs_mount_target.content"]

    ami = "${data.aws_ami.image_id.id}"
    instance_type = "${var.dev_mode ? var.aws_instance_type_app_dev : var.aws_instance_type_app}"
    key_name = "${aws_key_pair.keypair.id}"
    subnet_id = "${element(aws_subnet.default.*.id, count.index + 1)}"
    root_block_device  {
        delete_on_termination = true
        volume_type = "${var.root_volume_type}"
    }

    ebs_block_device {
        delete_on_termination = true
        device_name = "/dev/sdc"
        volume_size = "${var.aws_instance_block_size_app}"
        volume_type = "${var.block_volume_type}"
    }

    vpc_security_group_ids = ["${aws_security_group.private.id}","${aws_security_group.admin.id}"]

}

resource "null_resource" "app_installcms" {
    #The terraform version 0.12 will support null_resource.cs_installcms[0] parameter,So it need change this parameter later.
    depends_on = ["aws_db_instance.rds","aws_instance.app","null_resource.prepare_rpms","null_resource.cs_installcms"]
    count = "${var.app_count}"

    connection {
        host = "${element(aws_eip_association.app_eip_assoc.*.public_ip, count.index)}"
        type = "ssh"
        user = "${var.ssh_user}"
        private_key = "${file(var.private_key_path)}"
        timeout = "60m"
    }

    provisioner "remote-exec" {
        inline = [
            #send keep alives to client to keep AWS from dropping ssh connection during long processes with no output
            "echo -e '\nClientAliveInterval 60' | sudo tee -a /etc/ssh/sshd_config",
            "sudo systemctl restart sshd"
        ]
    }

    provisioner "remote-exec" {
        inline = [
            "sudo mkdir -p /opt/vmTools",
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/vmTools"
        ]
    }

    provisioner "file" {
        source = "../vmTools/"
        destination = "/opt/vmTools"
    }

    provisioner "remote-exec" {
        inline = [
            # make scripts runnable
            "sudo chmod +x /opt/vmTools/*.sh",

            # create new disk space
            "sudo bash /opt/vmTools/mountVMdisk.sh /var/log \"0% 25%\" /opt \"26% 100%\"",

            # setup for using nfs client
            "sudo systemctl enable rpcbind.service",
            "sudo systemctl start rpcbind",

            # set the hostname
            "sudo bash /opt/vmTools/setHostname.sh ${var.cms_hostname_app}${count.index + 1}",

            # avoid getting AWS resolv.conf
            "echo 'PEERDNS=no' | sudo tee -a /etc/sysconfig/network-scripts/ifcfg-eth0",

            #remove packages which will conflict with CMS packages
            "sudo yum erase -y python-setuptools",

            #Have problems with RPMS that add users to ssh if sshc_config doesn't end with a newline
            "sudo sed -i -e 's/\\(UseDNS.*\\)/\\1\\n/' /etc/ssh/sshd_config",
            #CMS install blindly adds a line for sftp service, comment out existing
            "sudo sed -i -e 's/\\(Subsystem sftp.*\\)/#\\1/' /etc/ssh/sshd_config",

            #iptables blocks access to httpd
            "sudo systemctl stop firewalld",
            "sudo systemctl stop iptables",

            # get the node installer
            "sudo bash /opt/vmTools/waitAndPullNodeInstall.sh http://${element(flatten(aws_network_interface.cs_int.*.private_ips), 0)}:8008/cms-node-config.tgz",

            # do the install
            "sudo chown ${var.ssh_user}:${var.ssh_user} /opt/cms",
            "cd /opt/cms/staging",

            #make create a settings file to override dbsettings
            "sudo mkdir -p /opt/db/scripts/install",
            "echo '#!/bin/bash' | sudo tee /opt/db/scripts/install/settings",
            "echo 'DB_ADMIN_PW=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'PGPD=${var.rds_db_pass}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_PORT=${var.rds_db_port}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_TYPE=postgres' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_USER=${var.rds_db_user}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_VERSION=${var.rds_db_version}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'DB_NAME=${var.rds_db_name}' | sudo tee -a /opt/db/scripts/install/settings",
            "echo 'HOST_NAME=dbserver' | sudo tee -a /opt/db/scripts/install/settings",
            "sudo chown nobody:nobody /opt/db/scripts/install/settings",
            "sudo chmod 400 /opt/db/scripts/install/settings",

            #sleep to stagger app instances
            "sleep ${270*count.index}",

            "sudo bash installcmsnode.sh -n ${var.app_count} app${count.index + 1} |& sudo tee -a /var/log/cms/installer/installcmsnode.log; test $${PIPESTATUS[0]} -eq 0",
        ]
    }

    provisioner "remote-exec" {
        inline = [
            # setup vsftpd
            "if [ -d /etc/puppetlabs/code/modules/vsftpd ];then ",
            "sudo sed -i 's/^pasv_max_port=.*/pasv_max_port=$((pasv_min_port + 9))/g' /etc/puppetlabs/code/modules/vsftpd/files/vsftpdsetup.sh",
            "echo 'pasv_promiscuous=YES' | sudo tee -a /etc/vsftpd/vsftpd.conf",
            "sudo /opt/puppetlabs/bin/puppet apply -e 'include vsftpd'",
            "fi"
        ]
    }
}

output "app_nodes" {
    value = "${aws_eip.app_eip.*.public_ip}"
}
