"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import sys
import signal
import argparse
import logging
import logging.handlers
import getpass

from ericsson.deploypattern.lib import errors
from ericsson.deploypattern.lib import ansibleplay
from ericsson.deploypattern.lib.config import Config, get_config_vars_dict
from ericsson.deploypattern.lib.inventory import inventory_command
from ericsson.deploypattern.lib.utils import load_yaml
from ericsson.deploypattern.plugins.deployplugins import load_plugin
from ericsson.deploypattern.plugins.pattern import pattern
from ericsson.deploypattern.tools import labels
from ericsson.deploypattern.tools.dfninit import mgnt_server_init, check_init
from ericsson.deploypattern.tools.yum import load_yum_repo
from ericsson.deploypattern.tools.apt import load_deb_repo
from ericsson.deploypattern.tools.registry import load_docker_registry
from ericsson.deploypattern.tools import helm
from ericsson.deploypattern.tools import bundle
from ericsson.deploypattern.version import __version__

# create logger
logger = logging.getLogger()
logger.setLevel(logging.INFO)

# Console Handler
ch = logging.StreamHandler()
ch.setLevel(logging.INFO)
# formatter = logging.Formatter('%(message)s')
# ch.setFormatter(formatter)
logger.addHandler(ch)

# File Handler
formatter = logging.Formatter('%(asctime)s:'
                              '%(levelname)s:'
                              '%(filename)s:'
                              '%(funcName)s:'
                              ' %(message)s')
fh = logging.handlers.RotatingFileHandler(
    '/tmp/deploypattern.log', 'a', 1000000, 1)
# fh = logging.FileHandler('/var/log/deploypattern.log')
fh.setLevel(logging.INFO)
fh.setFormatter(formatter)
logger.addHandler(fh)

# For Ansible Log
# Ansible add FileHandler to root logger
# Search ansible logger with its key: p=<PID ansible> u=<user> |
# Don't propagate the log to parent logger (root), not to have twice log written
# Add your Handler ansible logger to have Ansible log
for key in logging.Logger.manager.loggerDict.keys():
    if ' | ' in key:
        logging.Logger.manager.loggerDict[key].propagate = False
        logging.Logger.manager.loggerDict[key].addHandler(fh)

DFN_CONF_FILE = '/etc/deploypattern.conf'


class ArgumentParser(argparse.ArgumentParser):
    """
    Override error method in argparse.ArgmentParser
    """

    def error(self, message):
        self.print_help(sys.stderr)
        self.exit(2, '%s: error: %s\n' % (self.prog, message))


def run_pattern(deploy_pattern):
    """ Parse deployment pattern

    Launch the plugin related to the task type
    And run it

    """
    tasks_list = deploy_pattern['deployment_tasks']

    for task in tasks_list:
        logger.info("\n:: %s ::", task['name'])
        logger.info("--------------------------")

        # get task type
        __deployment_type = task['type']

        # load the plugin
        plugin = load_plugin(__deployment_type)
        plugin(task)


def parse_cmdline():
    """ Parse command line arguments """
    parser = ArgumentParser(prog="mdt",
                            usage='mdt [mdt_options] [command] [resource] [command_options]',
                            description=('''
    MDT: MediaFirst Deployment Toolkit
    ==================================
    Tool to deploy Kubernetes cluster and products

                                     '''),
                            formatter_class=argparse.RawDescriptionHelpFormatter)

    # Optional arguments for the MDT
    parser.add_argument('--version', action='version',
                        version="- %(prog)s {version} -"
                        .format(version=__version__))

    parser.add_argument("--debug",
                        action="store_true",
                        help="Enable Debug output messages")

    parser.add_argument("-c", "--config",
                        dest="conf",
                        help="configuration file; "
                        "default is " + DFN_CONF_FILE)

    # Subparser commands
    subparsers = parser.add_subparsers(dest='cmd')
    subparsers.required = False
    del_parser = subparsers.add_parser('delete',
                                       usage='mdt [mdt_options] delete [resource] [command_options].',
                                       help='Delete resource')
    deploy_parser = subparsers.add_parser('deploy',
                                          usage='mdt [mdt_options] deploy [resource] [command_options]',
                                          help='Deploy resource')
    get_parser = subparsers.add_parser('get',
                                       usage='mdt [mdt_options] get [resource] [command_options]',
                                       help='Get resource')
    init_parser = subparsers.add_parser('init',
                                        description=('''
Initialize the MDT software to setup the environment appropriately.
For allinone installation the local disk will be used instead of NFS server
                                        '''),
                                        formatter_class=argparse.RawDescriptionHelpFormatter,
                                        usage='mdt [mdt_options] init  [-i IP_ADDRESS] [-k SSH_KEY] [-g REGISTRY] [-m HELM_REPO_URL] [-d REGISTRY_NFS_ADDRESS] [-p REGISTRY_NFS_PATH] [-u USER] [-r {yum,deb}] [-a ANSIBLE_PATH]',
                                        help='MDT initialisation')
    load_parser = subparsers.add_parser('load',
                                        usage='mdt [mdt_options] load [resource] [command_options]',
                                        help='Load resources')
    reset_parser = subparsers.add_parser('reset',
                                         usage='mdt [mdt_options] reset [resource] [command_options]',
                                         help='Reset ressources')
    set_parser = subparsers.add_parser('set',
                                       usage='mdt [mdt_options] set [resource] [command_options]',
                                       help='Set resource')

    # Command deploy
    # -----------------
    deploy_sub_parser = deploy_parser.add_subparsers(dest='cmd_deploy')
    deploy_helm_parser = deploy_sub_parser.add_parser('helm',
                                                      description='Install Helm repository on MDT node and Helm server tiller, on one master.',
                                                      usage='mdt [mdt_options] deploy helm')

    deploy_k8scluster_parser = deploy_sub_parser.add_parser('k8s-cluster',
                                                            description=('''
Deploy the k8s-cluster, Kubernetes on the nodes masters and workers. The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
The configuration of Kubernetes cluster as HA registry, HA proxy, load balancer ... are given in k8s-config file in the config bundle folder.
                                                '''),
                                                            formatter_class=argparse.RawDescriptionHelpFormatter,
                                                            usage='mdt [mdt_options] deploy k8s-cluster')

    deploy_labels_parser = deploy_sub_parser.add_parser('labels',
                                                        description=('''
Deploy labels on all nodes from files of the config bundle directory.
With the option -a|--audit only display the labels to set per node.
                                                '''),
                                                        formatter_class=argparse.RawDescriptionHelpFormatter,
                                                        usage='mdt [mdt_options] deploy labels [-a]')

    deploy_products_parser = deploy_sub_parser.add_parser('products',
                                                          description=('''
Deploy the products on Kubernetes cluster nodes according to the MDT deploy packages defined in the config bundle folder.
If a profile is set, deploy only the packages link to the profile.
                                                '''),
                                                          formatter_class=argparse.RawDescriptionHelpFormatter,
                                                          usage='mdt [mdt_options] deploy products [-p PROFILE_NAME]')

    deploy_volumes_parser = deploy_sub_parser.add_parser('volumes',
                                                         description=('''
Deploy storage classes from k8s-volumes file in config bundle folder.
A storage class is defined with 4 parameters:
* name
* type
* address
* path
                                                '''),
                                                         formatter_class=argparse.RawDescriptionHelpFormatter,
                                                         usage='mdt [mdt_options] deploy volumes')

    # --> mdt deploy helm
    # no parameter

    # --> mdt deploy k8s-cluster
    # no parameter

    # --> mdt deploy labels
    deploy_labels_parser.add_argument("-a", "--audit",
                                      dest="audit",
                                      action="store_true",
                                      help="Display the list of labels per node without deploy them")

    # --> mdt deploy products [-p profile_name]
    deploy_products_parser.add_argument("-p", "--profile",
                                        dest="profile_name",
                                        default=None,
                                        help="Specific profile"
                                        )

    # --> mdt deploy volumes
    # no parameter

    # Command reset
    # ----------------
    reset_sub_parser = reset_parser.add_subparsers(dest='cmd_reset')
    reset_product_parser = reset_sub_parser.add_parser('products',
                                                       description='Reset all products installed by the MDT on all nodes',
                                                       usage='mdt [mdt_options] reset products')

    reset_k8scluster_parser = reset_sub_parser.add_parser('k8s-cluster',
                                                          description=('''
Reset the Kubernetes cluster on all nodes and clean load balancers. The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
                                                '''),
                                                          formatter_class=argparse.RawDescriptionHelpFormatter,
                                                          usage='mdt [mdt_options] reset k8s-cluster')

    # Command set
    # ----------------
    set_sub_parser = set_parser.add_subparsers(dest='cmd_set')
    set_label_parser = set_sub_parser.add_parser('label',
                                                 description=('''
                                  Set a label (key,value) on a node.
2 formats to pass label parameter: <key>=<value> or <key>:<value>
If there is already a label with same key on the node, the set label is not available.
You have to use option --force to set label with other value.
                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] set label -n NODE -l LABEL [--force]')

    # --> mdt set label
    set_label_parser.add_argument("-n", "--node",
                                  dest="node",
                                  required=True,
                                  help="Required - Node to label")
    set_label_parser.add_argument("-l", "--label",
                                  dest="label",
                                  required=True,
                                  help="Required - Label to add on node: key=value or key:value")
    set_label_parser.add_argument("--force",
                                  action="store_true",
                                  dest="force",
                                  help="Force to set the label on the node, even there is a label with same key on the node")

    # Command get
    # ---------------
    get_sub_parser = get_parser.add_subparsers(dest='cmd_get')
    get_label_parser = get_sub_parser.add_parser('label',
                                                 description=('''
Get all labels (k8s and mdt labels) from a node, if there is option -n or otherwise from all nodes.
Labels are displayed with yaml format, with 2 lists per node: 1 for MDT_labels and 1 for Other_labels.
If option -o, the yaml is saved in file which path is given by the option.
                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] get label [-n NODE] [--all] [-o OUTPUT_FILE]')

    get_cfg_parser = get_sub_parser.add_parser('cfg',
                                               description=('''
Get the entire configuration bundle currently applied and export all files as a tgz archive.
The output archive is created in the path given in command line with option -o.
                                  '''),
                                               formatter_class=argparse.RawDescriptionHelpFormatter,
                                               usage='mdt [mdt_options] get cfg -o OUTPUT_FILE')

    # --> mdt get label
    get_label_parser.add_argument("-n", "--node",
                                  dest="node",
                                  default="--all",
                                  help="Node to get label")
    get_label_parser.add_argument("--all",
                                  dest="node",
                                  action="store_const",
                                  const="--all",
                                  help="All nodes, default value")
    get_label_parser.add_argument("-o", "--output",
                                  dest="output_file",
                                  help="Path to output file")

    # --> mdt get cfg
    get_cfg_parser.add_argument("-o", "--output",
                                required=True,
                                dest="output_file",
                                help="Required - Path to archive file")

    # Command delete
    # ---------------
    del_sub_parser = del_parser.add_subparsers(dest='cmd_del')
    del_label_parser = del_sub_parser.add_parser('label',
                                                 description=('''
Delete a label (key or key=value or key:value) on a node.
Display a warning if the node doesn't exist or hasn't the key.
                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] delete label -n NODE -l LABEL')

    del_cfg_parser = del_sub_parser.add_parser('cfg')

    # --> mdt delete label -n <node> -l <label>
    # with label key=value, key:value, key
    del_label_parser.add_argument("-n", "--node",
                                  dest="node",
                                  required=True,
                                  help="Required - Node to delete label")
    del_label_parser.add_argument("-l", "--label",
                                  dest="label",
                                  required=True,
                                  help="Required - Label to delete on node")

    # Command load
    # -------------
    load_sub_parser = load_parser.add_subparsers(dest='cmd_load')
    load_cfg_parser = load_sub_parser.add_parser('cfg',
                                                 description=('''
Load the content of the given config bundle from CONF_BUNDLE_TGZ archive file or CONF_BUNDLE_DIR directory path.
If there is already a config bundle loaded, warning message is displayed.
Use the option --force to force the loading.
The config bundle is saved in ''' + Config().get('bundle', 'path')),
        formatter_class=argparse.RawDescriptionHelpFormatter,
        usage='mdt [mdt_options] load cfg (-f CONF_BUNDLE_TGZ | -d CONF_BUNDLE_DIR) [--force]')

    load_mdtchart_parser = load_sub_parser.add_parser('chart',
                                                      description=('''
Load Helm charts from HELM_TGZ archive file or HELM_DIR directory path.
The Helm charts are pushed to the MDT Helm repository: helm_repo.
                                                '''),
                                                      formatter_class=argparse.RawDescriptionHelpFormatter,
                                                      usage='mdt [mdt_options] load chart (-f CHART_TGZ | -d CHART_DIR)')

    load_container_parser = load_sub_parser.add_parser('container',
                                                       description=('''
Load containers from CONTAINER_TGZ archive file or CONTAINER_DIR directory path.
The containers are pushed to the MDT Docker registry (default).
See NFS configuration to share registry with K8S HA Docker registry.
                                                  '''),
                                                       formatter_class=argparse.RawDescriptionHelpFormatter,
                                                       usage='mdt [mdt_options] load container (-f CONTAINER_TGZ | -d CONTAINER_DIR)')

    load_rpm_parser = load_sub_parser.add_parser('rpm',
                                                 description=('''
Load rpm from RPM_TGZ archive file.
The rpm are pushed to the local yum repository.
                                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] load rpm (-f RPM_TGZ )')

    load_deb_parser = load_sub_parser.add_parser('deb',
                                                 description=('''
Load deb from DEB_TGZ archive file.
The deb are pushed to the local deb repository.
                                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] load deb (-f DEB_TGZ )')

    # --> mdt load rpm -f <tar.gz>
    load_rpm_parser.add_argument("-f", "--file",
                                 dest="rpm_tgz",
                                 help="Path to rpm tgz archive to load ")

    # --> mdt load deb -f <tar.gz>
    load_deb_parser.add_argument("-f", "--file",
                                 dest="deb_tgz",
                                 help="Path to deb tgz archive to load ")

    # --> mdt load cfg [--force] -f <tar.gz> | -d <folder>
    load_cfg_exclusive_parser = load_cfg_parser.add_mutually_exclusive_group(
        required=True)
    load_cfg_exclusive_parser.add_argument("-f", "--file",
                                           dest="conf_bundle_tgz",
                                           default=None,
                                           help="Path of the config bundle tgz file to be loaded")
    load_cfg_exclusive_parser.add_argument("-d", "--directory",
                                           dest="conf_bundle_dir",
                                           default=None,
                                           help="Path of the config bundle directory to be loaded")
    load_cfg_parser.add_argument("--force",
                                 action="store_true",
                                 dest="force",
                                 help="Force load, if there is already a config bundle")

    # --> mdt load chart -f <tar.gz> | -d <folder>
    load_mdtchart_exclusive_parser = load_mdtchart_parser.add_mutually_exclusive_group(
        required=True)
    load_mdtchart_exclusive_parser.add_argument("-f", "--file",
                                                dest="chart_tgz",
                                                help="Path to MDT packages (Helm charts) tgz archive to load ")
    load_mdtchart_exclusive_parser.add_argument("-d", "--directory",
                                                dest="chart_dir",
                                                help="Path to MDT packages (Helm charts) directory to load")

    # --> mdt load container [--force] -f <tar.gz> | -d <folder> [-r registry]
    load_container_exclusive_parser = load_container_parser.add_mutually_exclusive_group(
        required=True)
    load_container_exclusive_parser.add_argument("-f", "--file",
                                                 dest="container_tgz",
                                                 default=None,
                                                 help="Path of the container archive file to be loaded")
    load_container_exclusive_parser.add_argument("-d", "--directory",
                                                 dest="container_dir",
                                                 default=None,
                                                 help="Path of container directory to be loaded")
    load_container_parser.add_argument("-r", "--registry",
                                       dest="registry",
                                       default=None,
                                       help="Load containers in specific registry")

    # Command init
    # -------------

    init_parser.add_argument("--debug",
                             action="store_true",
                             help="Enable Debug output messages")
    init_parser.add_argument("-i", "--ip-address",
                             dest="ip_address",
                             help="IP address of the MDT to use for cluster nodes ")
    init_parser.add_argument("-k", "--ssh-key",
                             dest="ssh_key",
                             help="Full path to the SSH key to use for cluster node SSH connections")
    init_parser.add_argument("-g", "--docker-registry",
                             dest="registry",
                             help="Only if external insecure Docker registry is used: <host>:<port> ")
    init_parser.add_argument("-m", "--helm-repository",
                             dest="helm_repo_url",
                             help="Only if external Helm repository is used: <host>:<port> ")
    init_parser.add_argument("-d", "--registry-nfs-address",
                             dest="registry_nfs_address",
                             help="NFS address for registry storage")
    init_parser.add_argument("-p", "--registry-nfs-path",
                             dest="registry_nfs_path",
                             help="Remote NFS path "
                             "(with docker/registry/ path)")
    init_parser.add_argument("-u", "--user",
                             dest="user",
                             default="centos",
                             help="Linux user to connect to the remote nodes, default current user")
    init_parser.add_argument("-r", "--repo",
                             dest="repo",
                             choices=["yum", "apt"],
                             help="Repository to use yum (rpm packages) or apt(deb packages), default yum")
    init_parser.add_argument("-a", "--ansible-path",
                             dest="ansible_path",
                             default=Config().get('ansible', 'path'),
                             help="Ansible yaml path, default " + Config().get('ansible', 'path'))

    # Commands  MDT < 1.5
    #---------------------

    inventory_parser = subparsers.add_parser('inventory',
                                             help='Inventory management (MDT <1.5)')
    inventory_parser.add_argument("--debug",
                                  action="store_true",
                                  help="Enable Debug output messages")
    inventory_parser.add_argument("-l", "--list",
                                  action="store_true",
                                  help="List current inventory")
    inventory_parser.add_argument("-f", "--format",
                                  dest="format",
                                  default="ansible",
                                  help="list in format ansible/yaml")
    inventory_parser.add_argument("-i", "--import",
                                  dest="_import",
                                  help="import inventory")
    inventory_parser.add_argument("-e", "--export",
                                  dest="export",
                                  help="export inventory to ansible format")

    pattern_parser = subparsers.add_parser(
        'pattern', help='Deploy pattern (MDT <1.5)')
    pattern_parser.add_argument("--debug",
                                action="store_true",
                                help="Enable Debug output messages")
    pattern_parser.add_argument("-i", "--install",
                                dest="install",
                                help="install pattern")
    pattern_parser.add_argument("-l", "--list-config",
                                action="store_true",
                                help="list configuration pattern")
    pattern_parser.add_argument("-c", "--config",
                                dest="config",
                                help="set pattern configuration")
    pattern_parser.add_argument("-p", "--path",
                                dest="pattern_path",
                                help="set pattern configuration")
    pattern_parser.add_argument("-r", "--run",
                                action="store_true",
                                help="run pattern - "
                                "can be combined with -c/--config")

    if len(sys.argv[1:]) == 0:
        parser.print_help()
        parser.exit(2)

    return parser.parse_args()


def signal_handler(signal, frame):
    """ Catch signal """
    logging.info(':: exiting ...')
    sys.exit(0)


def main():
    """ Main """

    # setup signal handler
    signal.signal(signal.SIGINT, signal_handler)
    # parse cmdline args
    args = parse_cmdline()

    if args.debug:
        formatter = logging.Formatter('%(levelname)s:'
                                      '%(filename)s:'
                                      '%(funcName)s:'
                                      ' %(message)s')
        ch.setFormatter(formatter)
        logger.setLevel(logging.DEBUG)
        ch.setLevel(logging.DEBUG)
        fh.setLevel(logging.DEBUG)

    # Command mdt init
    # ---------------------
    # init config
    if args.conf:
        config = Config(args.conf)
    else:
        config = Config(DFN_CONF_FILE)

    # Launch init process
    if args.cmd == 'init':
        try:
            mgnt_server_init(ipaddr=args.ip_address,
                             sshkey=args.ssh_key,
                             registry=args.registry,
                             registry_nfs_addr=args.registry_nfs_address,
                             registry_nfs_path=args.registry_nfs_path,
                             helm_repo=args.helm_repo_url,
                             user=args.user,
                             repo=args.repo,
                             ansible_path=args.ansible_path)
        except Exception as e:
            logger.error("Initialization failed: %s" %
                         e.message, exc_info=True)
            return 1

    # Command mdt inventory < 1.5
    # --------------------------
    if args.cmd == 'inventory':
        try:
            inventory_command(args)
        except Exception as e:
            logger.error("Inventory command failed: %s" %
                         e.message, exc_info=True)
            return 1
        finally:
            exit(0)

    # check if init as been done before going on
    try:
        check_init(config)
    except Exception as e:
        logger.error("Init must be done before: %s" % e.message)
        return 1

    # launch deployment pattern
    if args.cmd == 'pattern':
        try:
            pattern(args)
        except Exception as e:
            logger.error("Deploying pattern failed: %s" %
                         e.message, exc_info=True)
            return 1


# NEW MDT 1.5

    # Command mdt deploy
    # --------------------------
    if args.cmd == 'deploy':
        if args.cmd_deploy == 'labels':
            try:
                labels.set_labels(args.audit)
            except Exception as e:
                logger.error("Deploy labels on nodes failed: %s" % e.message)
                return 1

        if args.cmd_deploy == 'k8s-cluster':
            try:
                bundle.Bundle().deploy_k8scluster()
            except Exception as e:
                logger.error("Deploying pattern failed: %s" % e.message)
                return 1

        if args.cmd_deploy == "helm":
            try:
                inventory = bundle.Bundle().get_inventory()
                k8s_vars_dict = bundle.Bundle().get_vars_k8s_config()
                # Deploy Helm server (container tiller) only on one master with deployment (replicas=1)
                # Add service to expose Helm server
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="deploy_helm",
                    vars=get_config_vars_dict(),
                    extra_vars=k8s_vars_dict)
                logger.debug("result : %s" % result)
                if result != 0:
                    raise errors.AnsibleError(
                        "Ansible playbook failed : deploy_helm ")

            except Exception as e:
                logger.error("Install Helm failed: %s" % e.message)
                return 1

        if args.cmd_deploy == 'products':
            try:
                bundle.Bundle().deploy_products(args.profile_name)
            except Exception as e:
                logger.error("Deploy products failed: %s" % e.message)
                return 1

        if args.cmd_deploy == "volumes":
            bundle.Bundle().deploy_volumes()

    # Command mdt reset
    # -------------------------
    if args.cmd == 'reset':
        if args.cmd_reset == 'products':
            # Reset all products
            helm.reset_all()

        if args.cmd_reset == 'k8s-cluster':
            try:
                bundle.Bundle().reset_k8scluster()
            except Exception as e:
                logger.error("Reset K8S failed: %s" % e.message)
                return 1

    # Command mdt load
    # -------------------------
    if args.cmd == 'load':
        if args.cmd_load == 'cfg':
            # load config bundle files:
            bundle.Bundle().load_bundle(args.conf_bundle_tgz,
                                        args.conf_bundle_dir, args.force)

        if args.cmd_load == 'chart':
            try:
                # load helm chart tgz / load chart tgz:
                if args.chart_tgz:
                    # load_to_helm_repo(args.chart_tgz,args.chart_dir)
                    tgz_dir = "none"
                    helm.load_to_helm_repo(args.chart_tgz, tgz_dir)
                elif args.chart_dir:
                    tgz = "none"
                    helm.load_to_helm_repo(tgz, args.chart_dir)
                else:
                    raise errors.LoadError(
                        "Did you specify -p <mdt_package> option or -d <directory_containing_mdt_packages> option ?")

            except Exception as e:
                logger.error("Loading MDT package failed: %s" % e.message)
                return 1

        if args.cmd_load == 'container':
            try:
                load_docker_registry(
                    args.container_tgz, args.container_dir, args.registry)

            except Exception as e:
                logger.error("Loading Containers failed: %s" % e.message)
                return 1

        if args.cmd_load == 'rpm':
            try:
                load_yum_repo(args.rpm_tgz)

            except Exception as e:
                logger.error("Loading Rpm failed: %s" % e.message)
                return 1

        if args.cmd_load == 'deb':
            try:
                load_deb_repo(args.deb_tgz)

            except Exception as e:
                logger.error("Loading deb failed: %s" % e.message)
                return 1

    # Command mdt-cli set
    # ----------------------
    if args.cmd == 'set':
        if args.cmd_set == 'label':
            try:
                labels.set_label(args.node, args.label, args.force)
            except Exception as e:
                logger.error("Set label failed: %s" % e.message)
                return 1

    # Command mdt-cli delete
    # ----------------------
    if args.cmd == 'delete':
        if args.cmd_del == 'label':
            try:
                labels.delete_label(args.node, args.label)
            except Exception as e:
                logger.error("Delete label failed: %s" % e.message)
                return 1

    # Command mdt-cli get
    # ----------------------
    if args.cmd == 'get':
        if args.cmd_get == 'label':
            try:
                if args.node != "--all":
                    labels.get_label(args.node, args.output_file)
                else:
                    labels.get_label(None, args.output_file)
            except Exception as e:
                logger.error("Get label failed: %s" % e.message)
                return 1

        if args.cmd_get == 'cfg':
            # get config bundle files:
            bundle.Bundle().get_bundle(args.output_file)

    # logger.info("\n")
    return 0


if __name__ == '__main__':
    main()
