"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import os
import jinja2
import json
import sys
import logging
from kubernetes import client as client
from kubernetes import config as config

import ericsson.deploypattern.lib.errors as errors
import ericsson.deploypattern.tools.bundle as bundle
from ericsson.deploypattern.lib.config import Config

logger = logging.getLogger(__name__)

class K8sApi(object):
    """

    """
    def __init__(self,config_file_path=None):
        if config_file_path == None:
            config_file_path = os.path.join(bundle.Bundle().get_bundle_path(), "pki","kubeconfig.yaml")
        try:
            k8s_conf = config.load_kube_config(config_file_path)
        except Exception as e:
            raise errors.K8sApiError("--> Error: %s" % e)

        try:
            client.configuration.debug = True
            self.v1api = client.CoreV1Api()
            template_dir_path = os.path.join(os.path.dirname(os.path.realpath(__file__)).rsplit("/",1)[0],"templates")
            self.j2_env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_dir_path),
                                autoescape=jinja2.select_autoescape(['json', 'yml', 'yaml']))
        except Exception as e:
            raise errors.K8sApiError("--> Error: %s" % e)



    def set_label(self,node=None,key=None,value=None):
        """
        Set a label (key, value) to K8S node
        :param node:
        :param key:
        :param value:
        :return:
        """
        if node == None:
          logger.info("WARNING: No node to label")
          return
        if key == None:
          logger.info("WARNING: No label given")
          return
        if value == None:
            value = "yes"

        logger.info("-> Set label %s on node %s"% (key,node))
        try:
            body = self.j2_env.get_template('label.json.j2').render(label=key,value=value)
            body = json.loads(body)
            api_response = self.v1api.patch_node(node, body)
            logger.debug(":: %s" % api_response)
        except Exception as e:
            raise errors.K8sApiError("--> Error: %s" % e)

    def del_label(self,node=None,key=None):
        """
        Delete a label (key) of a node
        :param node:
        :param key:
        :return:
        """
        if node == None:
          logger.info("WARNING: No node to delete label")
          return
        if key == None:
          logger.info("WARNING: No label given")
          return

        logger.info("-> Delete label %s on node %s" % (key,node))
        try:
            body = self.j2_env.get_template('del_label.json.j2').render(label=key)
            body = json.loads(body)
            body['metadata']['labels'][key]=None
            api_response = self.v1api.patch_node(node, body)
            logger.debug(":: %s" % api_response)
        except Exception as e:
            raise errors.K8sApiError("--> Error in del_label: %s" % e)

    def get_label(self,node=None):
        """
        Get all labels of K8S node
        :param node:
        :return: dictionary (key, value)
        """
        if node == None:
          logger.info("WARNING: No node to get label")
          return

        logger.info("-> Get labels on node %s" % node)
        try:
            api_response = self.v1api.read_node(node)
            return api_response.metadata.labels
        except Exception as e:
            raise errors.K8sApiError("--> Error in get_labels: %s" % e)

    def create_namespace(self,namespace):
        """
        Create namespace
        :param namespace:
        :return:
        """
        try:
            body = client.V1Namespace()
            body.metadata = client.V1ObjectMeta(name=namespace)
            api_response = self.v1api.create_namespace(body)
            logger.debug(":: %s" % api_response)
        except Exception as e:
            raise errors.K8sApiError("--> Error in create namespace: %s" % e)
