"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import json
import logging
import re

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.plugins.deployplugins import register_plugin
from ericsson.deploypattern.lib.ansibleplay import AnsibleRunner
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_config_vars_dict
from ericsson.deploypattern.lib.utils import ssh_command

logger = logging.getLogger(__name__)

REMOTE_NODE_DEST_DIR='~/ericsson/'


class Node(object):
    """ Class that handle a node object
    """
    __processing_type = ['mux', 'encoding', 'muconv']

    def __init__(self, node, compose_path):

        self.processing_spec = node['processing']
        if 'serverid' in node.keys():
          self.serverid_spec   = node['serverid']
        else:
          self.serverid_spec   = ""

        self.ip              = str(node['address'])
        self.envfile         = os.path.join(compose_path, 'env-' + self.ip)

        # check is network mode is specified
        # otherwise use host by default
        if 'net' in node.keys():
            self.net = node['net']
        else:
            self.net = 'host'

    def get_composes_file(self):
        """ construct a list of """
        compose_files = []

        # add by default controller compose file for mux and encoding
        if 'mux' in self.processing_spec or 'encoding' in  self.processing_spec:
            controller_file = 'controller-standalone-' + self.net + '.yaml'
            compose_files.append(controller_file)

        # then add only processing we want
        for processing in self.processing_spec:

            if processing in self.__processing_type:
                file = str(processing + '-standalone-' + self.net + '.yaml')
                compose_files.append(file)
            else:
                raise TypeError("processing %s not implemented" %
                                processing)
        return compose_files

    def get_serverid(self):
        _serverids = {}

        for prod in self.serverid_spec:
            for type, id in prod.items():
                _serverids[type] = id

        return _serverids


class Container(object):

    def __init__(self, container):
        self.name    = container['name']
        self.version = container['version']

        if 'vars' in container:
            self.vars = container['vars']

    def get_vars(self):
        if hasattr(self, 'vars'):
            return self.vars
        else:
            return None

    def get_versions(self):
        _versions = {}
        _versions['version_' + self.name] = self.version
        return _versions


class DeployContainerStandalone(object):
    """ Class to deploy containers through compose
    """
    def __init__(self, spec):
        self.containers = spec['containers']
        self.nodes      = spec['nodes']

        if 'vars' in spec:
            self.vars       = spec['vars']

        self.ansible = AnsibleRunner()

        try:
            self.compose_path = Config().get('compose', 'path')
            self.ssh_key = Config().get('ansible', 'ssh_key')
            self.remote_user = Config().get('ansible', 'remote_user')
        except Exception as e:
            raise errors.InvalidConfig(e)

    def deploy(self):
        """ Handle deployment for each nodes"""
        for __node in self.nodes:
            # intitialize Node object
            node = Node(__node, self.compose_path)

            # build env dict
            env = self._get_static_env()
            env.update(node.get_serverid())

            if hasattr(self, 'vars'):
                [env.update(_v) for _v in self.vars]

            for c in self.containers:
                container = Container(c)
                _vars = container.get_vars()
                if _vars is not None:
                    [env.update(_v) for _v in _vars]
                env.update(container.get_versions())

            # create env file
            self._create_env_file(node.envfile, env)

            # deploy on remote node
            self._deploy_node(node.ip, node.envfile, node.get_composes_file())

    def _deploy_node(self, ip, envfile, compose_files):
        """Deploy containers on the node """
        try:
            logger.info(" -> node : %s " % ip)

            # remote destination directory
            dest_dir = REMOTE_NODE_DEST_DIR
            self.ansible.create_dir([ip], dest_dir)

            # copy env file on remote node
            env_dest = os.path.join(dest_dir, '.env')
            self.ansible.copy_file([ip], envfile, env_dest, task='template')

            # copy compose file on remote nodes
            subcmd = ''
            for file in compose_files:
                file_src = os.path.join(self.compose_path, file)
                compose_dest = os.path.join(dest_dir, os.path.basename(file))
                self.ansible.copy_file([ip], file_src, compose_dest)
                subcmd = subcmd + ' -f ' + compose_dest
            
            # create log dir on remote node
            self._create_log_dir(ip)

            # execute compose up on remote node
            compose_cmd =  'sudo sh -c "'
            compose_cmd += 'cd %s ;' % dest_dir
            compose_cmd += 'docker-compose '
            compose_cmd += subcmd
            compose_cmd += ' up -d "'

            # exec compose up command
            #self.ansible.remote_command([ip], compose_cmd)
            logger.info("cmd=%s" % compose_cmd)
            ssh_command(ip, self.ssh_key, self.remote_user, compose_cmd, stream=True)

        except Exception as e:
            logger.error("Compose deployment failed")
            raise

    @staticmethod
    def _get_static_env():
        """create static var for compose env file"""
        env_dict = {}

        vars = get_config_vars_dict()
        env_dict['DOCKER_BRIDGE']   = vars['docker_bridge']
        env_dict['DOCKER_BR_CIDR']  = vars['docker_bridge_cidr']
        env_dict['DOCKER_REGISTRY'] = vars['docker_registry'] + '/'
        env_dict['REPOSITORY']      = ''

        return env_dict

    @staticmethod
    def _create_env_file(envfile, envkeys):
        """write dict of k=v in compose env files"""
        try:
            logger.debug(":: create env file %s" % envfile)
            file = open(envfile, 'w')

            for k, v in envkeys.items():
                line = k + '=' + v + '\n'
                logger.debug(" -> write line %s" % line)
                file.write(line)

            file.close()
        except Exception as e:
            logger.error("creating .env file failed")
            raise

    def _create_log_dir(self, ip):
        """Create all containers log dir in nodes as we assume a binding
        is done between containers and node at startup for logs"""
        path = '/var/log/envivio/'
        containers_list = ['alarm', 'command', 'statistic', 'spark-config',
                           'service-api', 'server-api', 'template-api',
                           'unified-ui', 'statmux', 'spark-mux', 'server-daemon',
                           'encoding-live-worker', 'encoding-live-config']
        folders_list = ['private', 'public']

        for cont in containers_list:
            for folder in folders_list:
                _dir = os.path.join(path, cont, folder)
                self.ansible.create_dir([ip], _dir)

        for cont in ['alarm', 'command', 'statistic']:
            _dir = os.path.join(path, cont, 'internal')
            self.ansible.create_dir([ip], _dir)

        # create symlink for retro compatibility with envivio
        mk_link = dict(src='/var/log/envivio', dest='/var/log/mfvp', state='link')
        self.ansible.file_task([ip], mk_link)


def deploy_containers(command):
    """ Class to deploy containers
    """
    __deployment_mode = ["standalone"]

    logger.debug("command : %s" % json.dumps(command, indent=4))

    # TODO: validate input field
    mode = command['mode']
    spec = command['spec']

    if mode not in __deployment_mode:
        raise TypeError("mode %s not implemented" % mode)

    if mode == "standalone":
        deployment = DeployContainerStandalone(spec)
        deployment.deploy()


# register the plugin
register_plugin("deploycontainers", deploy_containers)

