"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import logging
import tarfile
import subprocess
import re
import shlex

try:
    from subprocess import DEVNULL # py3k
except ImportError:
    import os
    DEVNULL = open(os.devnull, 'wb')

import ericsson.deploypattern.lib.errors as errors
from ericsson.deploypattern.lib.config import Config

logger = logging.getLogger(__name__)

KEY_GEN_CONF = "key-gen.conf"
CREATEREPO = "debian_repo.sh"


def __get_repo_path():
    repo = Config().get('repo', 'path') + '/' + \
           Config().get('repo', 'name')
    return repo


def __get_ansible_path():
    return Config().get('ansible', 'path')


def __get_script():
    script_path = os.path.join(__get_ansible_path(), "files")
    return os.path.join(script_path, CREATEREPO)


def __get_key_conf():
    return os.path.join(__get_ansible_path(), "files", KEY_GEN_CONF)


def create_apt_repo():
    update_apt_repo(create=True)


def update_apt_repo(create=False):
    """ Create a local deb repository """
    try:
        repo_path = __get_repo_path()
        createrepo = __get_script()

        if not os.path.exists(repo_path):
            os.makedirs(repo_path)

        cmd = createrepo
        cmd += " -r " + repo_path + " -g " + __get_key_conf()
        if create:
            logger.info(":: creating APT repository in %s" % repo_path)
            cmd += " -c "

        logger.debug(cmd)
        subprocess.call(shlex.split(cmd),
                        stdout=DEVNULL, stderr=subprocess.STDOUT)
    except Exception as e:
        raise errors.InitError(e)


def load_deb_repo(tgz):
    """ Load an apt repository """
    try:
        os.path.exists(tgz)
        repo_path = __get_repo_path()
        if not os.path.exists(repo_path):
            raise errors.LoadError("Did you call init before ?")

        logger.info(":: updating APT repository %s" % repo_path)

        if tarfile.is_tarfile(tgz) is not True:
            raise errors.LoadError("%s is not a tar file" % tgz)

        tar = tarfile.open(tgz)
        for file in tar.getmembers():
            if file.isfile():
                logger.debug(" - > tar members %s" % file.name)
                if file.name.lower().endswith('.deb'):
                    # check if there is not ../ in file path
                    if re.match(r'\.\./', file.name):
                        raise errors.LoadError(
                            "suspicious filename with /../ in path")

                    # check if filename not begin with /
                    if re.match('^/', file.name):
                        raise errors.LoadError(
                            "suspicious filename start with /")

                    logger.info(" - > extracting DEB %s" % file.name)
                    tar.extract(file, path=repo_path)
                else:
                    raise errors.LoadError("is %s contains dep ?" % tgz)
        tar.close()
        update_apt_repo()

    except Exception as e:
        raise errors.LoadError(e)
        #raise



