"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import glob
import logging
import tarfile
import shutil
import time
from random import randint

from ericsson.deploypattern.lib import ansibleplay
from ericsson.deploypattern.lib import errors
from ericsson.deploypattern.lib import utils
from ericsson.deploypattern.lib.ansibleplay import AnsibleRunner
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_config_vars_dict
from ericsson.deploypattern.lib.utils import load_csv, load_yaml
from ericsson.deploypattern.tools import helm

logger = logging.getLogger(__name__)


BUNDLE_FILE_LIST = ['k8s-config', 'k8s-volumes',
                    'mdt-matrix', 'application-version-list']
BUNDLE_FOLDER_LIST = ['mdt-profiles', 'mdt-deploypackages']
BUNDLE_SUFFIX = ('.csv', '.yaml', '.yml')
NODE = 'node'
TYPE = 'type'
LOAD_BALANCER = "load-balancer"
PROFILE = 'profile'
K8S_MASTER = "k8s-master"
K8S_WORKER = "k8s-worker"
KEYS_MATRIX_LIST = [NODE, TYPE, PROFILE]
DPKG = 'deploypackage'
VERSION = 'version'
KEYS_PROFILE_LIST = [DPKG, VERSION]
KEYS_VOLUME_LIST = ['path', 'name', 'address', 'type']
THIRD_PART_LIST = ['mongo', 'rabbitmq', 'redis']

class Bundle(object):
    """

    """

    def __init__(self):
        self._bundle_path = os.path.join(
            Config().get('global', 'path'), 'bundle')
        if not os.path.exists(self._bundle_path):
            os.makedirs(self._bundle_path)


    def _merge_bundle(self, bundle_path):
        """
        Check the consistency of the Config Bundle
        :param bundle_path:
        :return:
        """
        try:
            # Copy files to existing directory
            for f in os.listdir(bundle_path):
                src = os.path.join(bundle_path, f)
                if os.path.isfile(src):
                    shutil.copy(src, os.path.join(self._bundle_path, f))
                else:
                    try:
                        os.makedirs(os.path.join(self._bundle_path, f))
                    except:
                        pass
                    for ff in os.listdir(src):
                        shutil.copy(os.path.join(bundle_path, f, ff),
                                    os.path.join(self._bundle_path, f, ff))
            logger.info(":: load complete")

        except Exception as e:
            raise errors.LabelError(" %s" % e)

    def _check_bundle(self, path_to_check):
        """
        Check the consistency of the Config Bundle
        :param bundle_path:
        :return:
        """
        logger.info(":: Check configuration bundle ")

        for filename in BUNDLE_FILE_LIST:
            # Search if file or folder exists in the config bundle folder
            file_path = os.path.join(path_to_check, filename)
            file_list = glob.glob(file_path + "*")
            # Get usually one file or folder, more than one if conversion csv
            # -> yaml
            if len(file_list) == 0:
                logger.warning(
                    "WARNING: %s.yaml is not found in the new config bundle", file_path)
            # Check the format: yaml or csv for file
            for existing_file in file_list:
                if os.path.isfile(existing_file):
                    # Case file:
                    if not existing_file.endswith(BUNDLE_SUFFIX):
                        logger.warning(
                            "WARNING: File %s has not a csv or yaml extension", existing_file)

        for folder in BUNDLE_FOLDER_LIST:
            folder_path = os.path.join(path_to_check, folder)
            if os.path.isdir(folder_path):
                file_list = glob.glob(folder_path + "/*")
                for existing_file in file_list:
                    if os.path.isfile(existing_file):
                        # Case file:
                        if not existing_file.endswith(BUNDLE_SUFFIX):
                            logger.warning(
                                "WARNING: File %s has not a csv or yaml extension", existing_file)
            else:
                logger.warning(
                    "WARNING: The %s folder is missing in the new config bundle", folder)

    def _clean_label_nodes(self):
        """
        Clean persistent label storage
        :return:
        """
        bundle_path = self.get_bundle_path()
        try:
            os.remove(os.path.join(bundle_path,"pki","labels_history"))
        except:
            pass

    def get_labels(self):
        """
        Get a list of dictionary (node, label, value) to set labels on nodes.
        Per default, use the config bundle under the bundle folder
        Otherwise, raise error.
        :return: list of dictionaries: {node, label}
        """
        logger.info(":: Get list (node, label)")

        # Check matrix_path exists
        f_list = glob.glob(self._bundle_path + "/mdt-matrix*")
        if len(f_list) != 1:
            raise errors.BundleError(
                "Matrix file not found or more than one matrix rin config bundle! ")
        else:
            matrix_path = f_list[0]

        # Convert matrix file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(matrix_path)[1] == '.csv':
            matrix_dict_list = load_csv(matrix_path)
        else:
            matrix_dict_list = load_yaml(matrix_path)

        # Check file: keys are node, type, profile (optional)
        for m in matrix_dict_list:
            if (set(m.keys()) - set(KEYS_MATRIX_LIST)) == ([]):
                raise errors.BundleError("Key error in %s" % matrix_path)

        # Read files and create dictionary
        label_dict = {}
        for m in matrix_dict_list:
            # Add a label to a node corresponding to its type in K8S: master or
            # worker
            label_dict[m[NODE]] = []
            if m[TYPE] in [K8S_MASTER, K8S_WORKER]:
                label_dict[m[NODE]].append({'label': m[TYPE], 'value': "yes"})

                # Check if profile file exists
                # !! we presume that there is one profile per file, and there is the profile name in profile file
                try:
                    if 'profile' in m.keys():
                        profile_path = glob.glob(os.path.join(
                            self._bundle_path, 'mdt-profiles', "*" + m['profile'] + "*"))
                        if len(profile_path) == 0:
                            raise errors.BundleError(
                                "Profile file doesn't exist for %s" % m['profile'])
                        # Read Profile file and create dictionary
                        profile_path = profile_path[0]
                        # Convert profile file in list of dictionaries one per
                        # deploy package
                        if os.path.splitext(profile_path)[1] == '.csv':
                            profile_dict_list = load_csv(profile_path)
                        else:
                            profile_dict_list = load_yaml(profile_path)

                        # Check file, keys are deploypackage, version
                        for p in profile_dict_list:
                            if not set(p.keys()) == set(KEYS_PROFILE_LIST):
                                raise errors.BundleError(
                                    "Key error in %s" % profile_path)

                        for p in profile_dict_list:
                            # Check if deploy package file exists
                            dpypkg_path = glob.glob(os.path.join(self._bundle_path, 'mdt-deploypackages', "*" +
                                                                 p['deploypackage'] + "_" + p['version'] + "*"))
                            if len(dpypkg_path) == 0:
                                raise errors.BundleError("Deploy package file doesn't exist for %s_%s" %
                                                         (p['deploypackage'], p['version']))
                            dpypkg_path = dpypkg_path[0]
                            # Convert mdt-deploypackage file in list of
                            # dictionaries one per deypkg
                            if os.path.splitext(dpypkg_path)[1] == '.csv':
                                dpypkg_dict_list = load_csv(dpypkg_path)
                            else:
                                dpypkg_dict_list = load_yaml(dpypkg_path)
                            # If there is a nodeSelector directive, we presume there is only one
                            # Get the key and the value
                            if 'values' in dpypkg_dict_list.keys():
                                if 'nodeSelector' in dpypkg_dict_list['values']:
                                    # Remove duplicate label
                                    label = dpypkg_dict_list['values']['nodeSelector'].keys()[
                                        0]
                                    node_label_list = [l['label']
                                                       for l in label_dict[m[NODE]]]
                                    if not label in node_label_list:
                                        label_dict[m[NODE]].append({
                                            'label': dpypkg_dict_list['values']['nodeSelector'].keys()[0],
                                            'value': dpypkg_dict_list['values']['nodeSelector'][dpypkg_dict_list['values']['nodeSelector'].keys()[0]]})
                                    else:
                                        # Test if different value for the same
                                        # label
                                        if dpypkg_dict_list['values']['nodeSelector'][dpypkg_dict_list['values']['nodeSelector'].keys()[0]] != label_dict[m[NODE]][node_label_list.index(label)]['value']:
                                            raise errors.BundleError(
                                                "WARNING: Different values for the same label")
                except Exception as e:
                    raise errors.BundleError(e)

        # Return
        return label_dict

    def get_inventory(self):
        """
        Get inventory from matrix file
        :return: dictionary of lists (group,list nodes)
        """
        logger.info(":: Get list (group, node)")

        # Check matrix_path exists
        f_list = glob.glob(self._bundle_path + "/mdt-matrix*")
        if len(f_list) != 1:
            raise errors.BundleError(
                "Matrix file not found in config bundle! ")
        else:
            matrix_path = f_list[0]

        # Convert matrix file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(matrix_path)[1] == '.csv':
            matrix_dict_list = load_csv(matrix_path)
        else:
            matrix_dict_list = load_yaml(matrix_path)

        # Check file: keys are node, type, profile (optional)
        for m in matrix_dict_list:
            if (set(m.keys()) - set(KEYS_MATRIX_LIST)) == ([]):
                raise errors.BundleError("Key error in %s" % matrix_path)

        # Read files and create dictionaries list
        # one dictionary per node type
        # key = node type, value = list of nodes
        inventory_dict_list = []
        inventory_dict_list.append({K8S_MASTER: []})
        inventory_dict_list.append({K8S_WORKER: []})
        inventory_dict_list.append({LOAD_BALANCER: []})

        for m in matrix_dict_list:
            # m (node, type, profile)
            # Search in dictionary list, the dictionary corresponding to the
            # node type
            index = [i for i, j in enumerate(
                inventory_dict_list) if j.keys()[0] == m[TYPE]]
            # Add the node to the list of nodes for this type
            # node = dictionary name, ipaddress (current inventory used by
            # Ansible)
            inventory_dict_list[index[0]][m[TYPE]].append(
                {'name': m[NODE], "ipaddress": m[NODE]})

        return inventory_dict_list

    def get_vars(self):
        """
        Get K8S variable from k8s-config file
        :return: dictionary var:value
        """
        logger.info(":: Get K8S variables ")

        # Check k8s-config  exists
        f_list = glob.glob(self._bundle_path + "/k8s-config*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "k8s-config file not found in config bundle! ")
        else:
            k8s_config_path = f_list[0]

        # Convert k8sconfig file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(k8s_config_path)[1] == '.csv':
            k8s_vars_dict = load_csv(k8s_config_path)
        else:
            k8s_vars_dict = load_yaml(k8s_config_path)

        return k8s_vars_dict

    def get_vars_k8s_config(self):
        """

        :return:
        """
        # Build dictionary to pass to ansible playbook to overwrite default values
        bundle_var_dict = self.get_vars()
        var_dict = {}
        if 'k8s' in bundle_var_dict.keys():
            for key, value in bundle_var_dict['k8s'].items():
                var_dict[key] = value

        # Check consistency
        if 'haproxy' in bundle_var_dict['k8s'].keys() and \
                        'load_balancer' in bundle_var_dict['k8s'].keys():
            if var_dict['haproxy'] and var_dict['load_balancer']:
                raise errors.BundleError("Incompatibility")

        if 'ha_registry' in bundle_var_dict.keys():
            for key, value in bundle_var_dict['ha_registry'].items():
                var_dict['ha_registry_' + key] = value

        if 'registry' in bundle_var_dict.keys():
            for key, value in bundle_var_dict['ha_registry'].items():
                var_dict['registry_' + key] = value

        # Case load_balancer
        if 'load_balancer' in bundle_var_dict.keys() and \
                                'load_balancer' in var_dict.keys():
            if var_dict['load_balancer']:
                for key, value in bundle_var_dict['load_balancer'].items():
                    var_dict['lb_' + key] = value
                var_dict['haproxy'] = False


        # Case haproxy
        if 'haproxy' in bundle_var_dict.keys() and \
                            'haproxy'in var_dict.keys():
            if var_dict['haproxy']:
                for key, value in bundle_var_dict['haproxy'].items():
                    var_dict['haproxy_' + key] = value
                var_dict['load_balancer'] = False


        if 'cloud_provider' in bundle_var_dict.keys():
            if bundle_var_dict['cloud_provider']['type'] == "openstack":
                var_dict['cloud_provider'] = 'openstack'
                for key, value in bundle_var_dict['cloud_provider'].items():
                    var_dict['os_' + key] = value
            elif bundle_var_dict['cloud_provider']['type'] == "aws":
                var_dict['cloud_provider'] = 'aws'
                for key, value in bundle_var_dict['cloud_provider'].items():
                    var_dict['aws_' + key] = value

        # Case vip parameter common to all configurations
        if 'haproxy_vip' in var_dict.keys():
            var_dict['vip'] = var_dict['haproxy_vip']
            del var_dict['haproxy_vip']
        elif 'lb_vip' in var_dict.keys():
            var_dict['vip'] = var_dict['lb_vip']
            del var_dict['lb_vip']
        else:
            var_dict['vip'] = False
        return var_dict

    def get_volumes_vars(self):
        """
        Get volumes variables from k8s-volumes file
        :return: list of dictionary var:value
        """
        logger.info(":: Get storage variables ")

        # Check k8s-volumes  exists
        f_list = glob.glob(self._bundle_path + "/k8s-volumes*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "k8s-volumes file not found in config bundle! ")
        else:
            k8s_volumes_path = f_list[0]

        # Convert k8s-volumes file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(k8s_volumes_path)[1] == '.csv':
            k8s_volume_vars = load_csv(k8s_volumes_path)
        else:
            k8s_volume_vars = load_yaml(k8s_volumes_path)

        return k8s_volume_vars

    def get_bundle_path(self):
        return self._bundle_path


    def _get_namespace(self):
        """
        Get namespace from k8s-config file
        :return: namespace
        """
        k8s_vars_dict = self.get_vars()
        return k8s_vars_dict['k8s']['namespace']

    def create_namespace(self):
        """

        :return:
        """
        # Get namespace from k8s_config
        namespace = self._get_namespace()
        from ericsson.deploypattern.lib.k8sApi import K8sApi
        try:
            K8sApi().create_namespace(namespace)
        except:
            pass

    def _check_dpypkg_used(self,dpypkg_name):
        """
        Check if dpypkg is used at least one profile by at least one node
        :param dpypkg_name: name of the dpypkg
        :return: boolean
        """
        deploy = False
        # Get chart name from dpypkg file
        chart = dpypkg_name.split('_')[1]
        # Search profile files with this chart:
        f2_list = os.listdir(self._bundle_path + "/mdt-profiles")
        p_list = []
        for f2 in f2_list:
            if chart in open(os.path.join(self._bundle_path + "/mdt-profiles",f2)).read():
                p_list.append(f2)
        # Search if at least one profile is used
        with open(os.path.join(self._bundle_path + "/mdt-matrix.yaml")) as matrix:
            matrix = matrix.read()
        for p in p_list:
            # Get profile name from profile file:
            profile = p.split('_')[1].split('.')[0]
            # Search profile name in matrix file
            if profile in matrix:
                deploy = True
                break
        return deploy

    def install_thirdparty(self):
        """
        Install 3rd party from Config Bundle

        :return:
        """
        # Get namespace from k8s_config
        namespace = self._get_namespace()

        # Install first All 3rd parties
        for third_part in THIRD_PART_LIST:
            f_list = glob.glob(self._bundle_path + "/mdt-deploypackages/*" + third_part + "*")
            for f in f_list:
                if self._check_dpypkg_used(f):
                    # Convert mdt-deploypackage file in list of dictionaries one per
                    # dpypkg
                    if os.path.splitext(f)[1] == '.csv':
                        dpypkg_dict_list = load_csv(os.path.join(
                            self._bundle_path, "mdt-deploypackages", f))
                    elif os.path.splitext(f)[1] == '.yaml':
                        dpypkg_dict_list = load_yaml(os.path.join(
                            self._bundle_path, "mdt-deploypackages", f))
                    else:
                        raise("Incorrect file format for %s" % third_part)
                    helm.install_chart(namespace=namespace,
                               chart_name=dpypkg_dict_list['package'],
                               chart_version=dpypkg_dict_list['version'],
                               release_name=dpypkg_dict_list['package'][0:3],
                               values_dict=dpypkg_dict_list['values'])

        # Wait until 3rd parties are running
        return

    def _install_dpypkg(self, dpypkg_dict_list):
        """
        Install Helm charts
        Overwrite values in deploy chart by global variables in k8s_config file
        :param dpypkg_dict_list:
        :return:
        """
        # Get namespace -> global variable
        namespace = self._get_namespace()

        # Overwrite values with global values in k8s_config file
        # section products
        global_dict = self.get_vars()['products']
        for key, value in global_dict.items():
            if key in dpypkg_dict_list['values']:
                dpypkg_dict_list['values'][key] = value
            for key2 in dpypkg_dict_list['values'].keys():
                if type(dpypkg_dict_list['values'][key2]) == dict:
                    if key in dpypkg_dict_list['values'][key2]:
                        dpypkg_dict_list['values'][key2][key] = value

        # Install chart one or more time
        if 'deploy_list' in dpypkg_dict_list:
            for i in range(len(dpypkg_dict_list['deploy_list'])):
                helm.install_chart(namespace=namespace,
                                   chart_name=dpypkg_dict_list['package'],
                                   release_name=dpypkg_dict_list['deploy_list'][i],
                                   chart_version=dpypkg_dict_list['version'],
                                   values_dict=dpypkg_dict_list['values'])
        else:
            helm.install_chart(namespace=namespace,
                               chart_name=dpypkg_dict_list['package'],
                               chart_version=dpypkg_dict_list['version'],
                               release_name=dpypkg_dict_list['package'][0:3] + "-" + str(
                                   randint(0, 99)),
                               values_dict=dpypkg_dict_list['values'])

    # Resource cfg
    #=============

    def load_bundle(self, conf_bundle_tgz=None, conf_bundle_dir=None, force=False):
        """
        mdt load cfg [-h] (-f CONF_BUNDLE_TGZ | -d CONF_BUNDLE_DIR) [--force]

        Load the content of the given config bundle from CONF_BUNDLE_TGZ archive file or CONF_BUNDLE_DIR directory path.
        If there is already a config bundle loaded, warning message is displayed.
        Use the option --force to force the loading.
        :param conf_bundle_tgz: config bundle to load
        :param conf_bundle_dir: directory config bundle to load
        :param force: to force load
        :return:
        """
        tmp = os.path.join('/tmp', 'bundle_tmp')
        if conf_bundle_tgz is not None:
            try:
                logger.info(":: load configuration bundle %s - force=%s" %
                            (conf_bundle_tgz, force))

                # Check if file exist
                if not os.path.exists(conf_bundle_tgz):
                    raise errors.BundleError(
                        "file %s not found ! " % conf_bundle_tgz)

                # is a readable tar file ?
                if tarfile.is_tarfile(conf_bundle_tgz) is not True:
                    raise errors.BundleError(
                        "%s is not a tar file" % conf_bundle_tgz)

                # Check if config bundle is already load (there is already pki
                # folder => >1)
                if len(os.listdir(self._bundle_path)) > 1 and not force:
                    raise errors.BundleError(
                        "A config bundle is already loaded, use option --force to load the new one")

                # open and extract file and tgz
                tar = tarfile.open(conf_bundle_tgz)
                tar.extractall(path=tmp)
                tar.close()
                # Check Config Bundle: just Warning
                # We can load partially config bundle
                self._check_bundle(tmp)
                self._merge_bundle(tmp)

            except Exception as e:
                raise errors.LabelError(" %s" % e)

            finally:
                if os.path.exists(tmp):
                    shutil.rmtree(tmp)

        elif conf_bundle_dir is not None:
            logger.info(":: load configuration bundle %s - %s",
                        conf_bundle_dir, force)
            # Check if config bundle is already load (there is already pki
            # folder => >1)
            if len(os.listdir(self._bundle_path)) > 1 and not force:
                raise errors.BundleError(
                    "A config bundle is already loaded, use option --force to load the new one")
            # is a readable tar file ?
            if not os.path.isdir(conf_bundle_dir):
                raise errors.BundleError(
                    "folder %s not found !" % conf_bundle_dir)
            # Check Config Bundle: just Warning
            # We can load partially config bundle
            self._check_bundle(conf_bundle_dir)
            self._merge_bundle(conf_bundle_dir)

    def get_bundle(self, output_path=None):
        """
        mdt get cfg [-h] -o OUTPUT_FILE

        Get Bundle in archive path
        :param output_path: path is required
        :return: the entire archive in output_path
        """
        logger.info(":: Get Current Config Bundle file ")
        if output_path is not None:
            with tarfile.open(output_path, "w:gz") as tar:
                for f in os.listdir(self._bundle_path):
                    f = os.path.basename(f)
                    if f not in ['pki', 'labels_history']:
                        tar.add(os.path.join(self._bundle_path, f), f)
            tar.close()
        else:
            raise errors.BundleError("The output path is required")

    # Resource volume
    #================
    def deploy_volumes(self):
        """
        mdt deploy volumes

        Deploy storage classes from k8s-volumes file in config bundle folder.
        A storage class is defined with 4 parameters:
        * name
        * type
        * address
        * path
        :param:
        :return:
        """
        inventory = self.get_inventory()
        k8s_volumes_vars = self.get_volumes_vars()
        config_extra_vars = get_config_vars_dict()
        for storage in k8s_volumes_vars:
            try:
                intersection = set(storage.keys()) & set(KEYS_VOLUME_LIST)
            except Exception as e:
                raise errors.YamlError(
                    "The k8s_volumes file is not correct: %s" % e)
            in_file_only = set(storage.keys()) - set(KEYS_VOLUME_LIST)
            missing_in_file = set(KEYS_VOLUME_LIST) - set(storage.keys())
            if intersection == set(KEYS_VOLUME_LIST):
                if in_file_only != ([]):
                    logger.warn(":: The following parameters will be ignored: %s",
                                in_file_only)
                new_vars_dic = {}
                new_vars_dic["storage_name"] = storage["name"]
                new_vars_dic["nfs_ip"] = storage["address"]
                new_vars_dic["nfs_path"] = storage["path"]
                new_vars_dic["storage_type"] = storage["type"]

                tmp_extra_vars = config_extra_vars
                tmp_extra_vars.update(new_vars_dic)
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="deploy_storage",
                    extra_vars=tmp_extra_vars)
                logger.debug("result : %s" % result)
                if result != 0:
                    raise errors.AnsibleError(
                        "Ansible playbook failed :deploy_storage ")
            else:
                logger.error(
                    ":: the following storage class is not correct: %s.\
                     Missing the following parameters %s", storage, missing_in_file)



    # Resource k8s-cluster
    #=======================
    def deploy_k8scluster(self):
        """
	    mdt deploy k8s-cluster

        Deploy the k8s-cluster, Kubernetes on the nodes masters and workers.
        The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
        The configuration of Kubernetes cluster as HA registry, HA proxy, load balancer ... are given in
        k8s-config file in the config bundle folder.
        :return:
        """
        inventory = self.get_inventory()

        logger.info(":: Deploy K8S")

        var_dict = self.get_vars_k8s_config()

        # First configure /etc/hosts
        result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="configure_etchosts",
                    vars=get_config_vars_dict(),
                    extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError( "Ansible playbook failed: configure_etchost ")

        result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="install_k8s_HA",
                    vars=get_config_vars_dict(),
                    extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError( "Ansible playbook failed: install_k8s_HA ")

    def reset_k8scluster(self):
        """
        Reset K8s cluster: reset all nodes masters and workers and clean the
        load-balancers.
        The file to save history labels for nodes is removed
        :return:
        """
        inventory = self.get_inventory()
        logger.info(":: Reset K8S")
        var_dict = self.get_vars_k8s_config()

        # Reset K8S cluster on all nodes a
        result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="reset_k8s",
                    vars=get_config_vars_dict(),
                    extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError(
                        "Ansible playbook failed : reset_k8s ")

        # Clean load-balancers
        result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="reset_loadbalancer",
                    vars=get_config_vars_dict(),
                    extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError(
                        "Ansible playbook failed : reset_loadbalancer ")

        # Clean Label nodes
        self._clean_label_nodes()

    # Resource product
    #=================
    def deploy_products(self, profile_name=None):
        """
        Deploy products on nodes from Config Bundle files, or only from one profile name
        :param profile_name: name of the profile, found in file name: mfvpel -> mdt-prof_mfvpel.yaml
        :return:
        """

        # !!!! Create namespace from k8s-config: there or another command ?
        self.create_namespace()

        # !!! Install 3rd party: there or another command or mdt-dpypkg ?
        logger.info(":: Waiting Mongo ...")
        self.install_thirdparty()

        # Wait 3rd parties installed
        time.sleep(120)

        logger.info(":: Deploy products")
        if profile_name is not None:
            # Search if profile name exists
            profile_path = glob.glob(os.path.join(
                self._bundle_path, 'mdt-profiles', "*" + profile_name + "*"))
            if len(profile_path) == 0:
                raise errors.BundleError(
                    "Profile file doesn't exist for %s" % profile_name)
            # Read Profile file and create dictionary
            profile_path = profile_path[0]
            # Convert profile file in list of dictionaries one per deploy
            # package
            if os.path.splitext(profile_path)[1] == '.csv':
                profile_dict_list = load_csv(profile_path)
            else:
                profile_dict_list = load_yaml(profile_path)

            # Check file, keys are deploypackage, version
            for p in profile_dict_list:
                if not set(p.keys()) == set(KEYS_PROFILE_LIST):
                    raise errors.BundleError("Key error in %s" % profile_path)

            for p in profile_dict_list:
                # Check if deploy package file exists
                dpypkg_path = glob.glob(os.path.join(self._bundle_path, 'mdt-deploypackages', "*" +
                                                     p['deploypackage'] + "_" + p['version'] + "*"))
                if len(dpypkg_path) == 0:
                    raise errors.BundleError("Deploy package file doesn't exist for %s_%s" %
                                             (p['deploypackage'], p['version']))

                dpypkg_path = dpypkg_path[0]
                # Convert mdt-deploypackage file in list of dictionaries one
                # per dpypkg
                if os.path.splitext(dpypkg_path)[1] == '.csv':
                    dpypkg_dict_list = load_csv(dpypkg_path)
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
                elif os.path.splitext(dpypkg_path)[1] == '.yaml':
                    dpypkg_dict_list = load_yaml(dpypkg_path)
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
            return

        # Deploy packages that are used in matrix and not third parties
        f_list = os.listdir(self._bundle_path + "/mdt-deploypackages")
        # Remove 3rd party, not to install them twice
        f_list = [ f for f in f_list if f.find('mongo')== -1 and f.find('redis') == -1 and f.find('rabbit') == -1]

        for f in f_list:
            if self._check_dpypkg_used(f):
                # Convert mdt-deploypackage file in list of dictionaries one per
                # dpypkg
                if os.path.splitext(f)[1] == '.csv':
                    dpypkg_dict_list = load_csv(os.path.join(
                        self._bundle_path, "mdt-deploypackages", f))
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
                elif os.path.splitext(f)[1] == '.yaml':
                    dpypkg_dict_list = load_yaml(os.path.join(
                        self._bundle_path, "mdt-deploypackages", f))
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)