"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import logging
import os
import socket
import glob
import sys
import fileinput
import re

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.tools.registry import start_docker_registry
from ericsson.deploypattern.tools.registry import mount_registry_nfs
from ericsson.deploypattern.tools.helm import create_helm_repo_dir
from ericsson.deploypattern.tools.yum import create_yum_repo, load_yum_repo
from ericsson.deploypattern.tools.apt import create_apt_repo, load_deb_repo
from ericsson.deploypattern.lib.ansibleplay import run_playbook

logger = logging.getLogger(__name__)


def __configure_services():
    """

    :return:
    """
    logger.info(":: configure node services")
    res = run_playbook('config_pkg')
    if res != 0:
        raise errors.InitError("MDT configuration services failed")


def __validate_sshkey_conf(config):
    key = config.get('ansible', 'ssh_key')
    logger.info('check ssh-key : {}'.format(key))
    if key.startswith('~'):
        key=os.path.expanduser(key)
    if not os.path.exists(key):
        raise errors.FileNotFound('%s: ' % key)
    return True



def __validate_ipaddr_conf(config):
    addr = config.get('global', 'mgnt_ip_addr')
    try:
        socket.inet_aton(addr)
    except socket.error:
        raise errors.InvalidConfig("mgnt_ip_addr in config not valid")
    return True


def __update_config(config):
    with open(config.get_file_conf(), 'wb+') as configfile:
        config.write(configfile)


def check_init(config):
    try:
        __validate_ipaddr_conf(config)
        __validate_sshkey_conf(config)
    except Exception as e:
        raise e


def mgnt_server_init(ipaddr=None,
                     sshkey=None,
                     registry=None,
                     registry_nfs_addr=None,
                     registry_nfs_path=None,
                     helm_repo=None,
                     user=None,
                     repo=None,
                     ansible_path=None):
    """
    usage: mdt init [-h] [--debug] [-i IP_ADDRESS] [-k SSH_KEY] [-g REGISTRY]
                [-d REGISTRY_NFS_ADDRESS] [-p REGISTRY_NFS_PATH] [-u USER]
                [-r REPO] [-a ANSIBLE_PATH]

    optional arguments:
        -h, --help            show this help message and exit
        --debug               Enable Debug output messages
        -i IP_ADDRESS, --ip-address IP_ADDRESS
                        IP address of the MDT to use for cluster nodes
        -k SSH_KEY, --ssh-key SSH_KEY
                        ssh key to use for cluster node connections
        -g REGISTRY:PORT, --docker-registry REGISTRY:PORT
                        Specify Docker registry to use
        -m REPOSITORY:PORT, --helm-repository EPOSITORY:PORT
                        Specify Helm repository to use
        -d REGISTRY_NFS_ADDRESS, --registry-nfs-address REGISTRY_NFS_ADDRESS
                        Specify a NFS address for registry storage
        -p REGISTRY_NFS_PATH, --registry-nfs-path REGISTRY_NFS_PATH
                        Specify remote NFS path (with docker/registry/ path)
        -u USER, --user USER  Specify user to use
        -r REPO, --repo REPO  Specify repository to use yum or deb
        -a ANSIBLE_PATH, --ansible-path ANSIBLE_PATH
                        Ansible yaml path

    :return:
    """
    logger.info("== initializing deployment node server ==")
    logger.info("\n")

    try:
        # get current config
        config = Config()
        if ipaddr is not None:
            config.set('global', 'mgnt_ip_addr', ipaddr)
        __validate_ipaddr_conf(config)

        if sshkey is not None:
            config.set('ansible', 'ssh_key', sshkey)
        __validate_sshkey_conf(config)

        if user is not None:
            config.set('ansible', 'remote_user', user)
        if ansible_path:
            config.set('ansible', 'path', ansible_path)

        if registry is not None:
            config.set_registry(registry)
        else:
            # assume local MDT registry by default
            if not config.get('docker', 'registry_address'):
                config.set('docker', 'registry_address', \
                           config.get('global', 'mgnt_ip_addr'))

        if bool(registry_nfs_addr) !=  bool(registry_nfs_path):
            raise errors.InvalidConfig("--registry-nfs-address and "
                                       "--registry-nfs-path must both be set ")

        if registry_nfs_addr and registry_nfs_path:
            config.set('docker', 'registry_nfs_ip', registry_nfs_addr)
            config.set('docker', 'registry_nfs_path', registry_nfs_path)

        if helm_repo is not None:
            config.set('helm_repo', 'path', 'http://' + helm_repo)

        repo_saved = config.get('repo','type')

        if repo is not None:
            config.set('repo', 'type', repo)
        elif repo_saved == 'first':
            repo = 'yum'
            config.set('repo', 'type', 'yum')

        # write new config in file
        __update_config(config)

    except Exception as e:
        logger.error("updating config failed")
        raise


    try:
        # Create the yum RPM repository or apt DEB repository (by default yum)
        if repo_saved == 'first':
            if repo == "yum":
                create_yum_repo()
            elif repo == "apt":
                create_apt_repo()
        elif repo_saved == "yum" and repo == "apt":
            create_apt_repo()
        elif repo_saved == "apt" and repo == "yum":
            create_yum_repo()

        # Simply create directory that will host Helm charts later
        if helm_repo is None:
            create_helm_repo_dir()

        # Configure all necessary services: docker, ansible, httpd
        __configure_services()

        # mount registry storage
        if registry_nfs_addr:
            mount_registry_nfs(registry_nfs_addr, registry_nfs_path)

        # start the docker registry if not external
        if registry is None and config.get('docker', 'registry_address') == config.get('global', 'mgnt_ip_addr'):
            start_docker_registry()

    except Exception as e:
        raise
