"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import logging
import tarfile
# import subprocess
import os


# try:
#     from subprocess import DEVNULL # py3k
# except ImportError:
#     import os
#     DEVNULL = open(os.devnull, 'wb')

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.lib.config import Config

logger = logging.getLogger(__name__)

HELM_BIN='/usr/local/bin/helm'



def __get_helm_repo_path():
    helm_repo = Config().get('helm_repo', 'path') + '/' + \
           Config().get('helm_repo', 'name')
    return helm_repo


def create_helm_repo_dir():
    """ Create a local helm repository """
    try:
        helm_repo_path = __get_helm_repo_path()
        logger.info(":: creating Helm repository directory in %s" % helm_repo_path)

        if not os.path.exists(helm_repo_path):
            os.makedirs(helm_repo_path)

    except Exception as e:
        raise errors.InitError(e)


def load_to_helm_repo(tgz,tgz_dir):
    """ Load Charts to Helm repository """
    try:
        os.path.exists(tgz)
        helm_repo_path = __get_helm_repo_path()
        if not os.path.exists(helm_repo_path):
            raise errors.LoadError("Did you call init before ?")

        logger.info(":: updating Helm repository %s" % helm_repo_path)

        if tgz != "none":
            # load a tgz containing a Helm chart
            if tarfile.is_tarfile(tgz) is not True:
                raise errors.LoadError("%s is not a tar file" % tgz)
            else:
                # extract Helm chart to helm_repo_path
                tar = tarfile.open(tgz)
                for file in tar.getmembers():
                    if file.isfile():
                        logger.info(" - > extracting Helm chart %s" % file.name)
                        tar.extract(file, path=helm_repo_path)
                tar.close()

            # update repo index with helm linux command
            # subprocess.call([HELM_BIN, 'repo', 'index', helm_repo_path], stdout=subprocess.STDOUT, stderr=subprocess.STDOUT)
            os.system(HELM_BIN + " repo " + " index " + helm_repo_path)

        elif tgz_dir != "none":
            # load content of a directory
            for root, dirs, files in os.walk(tgz_dir):
                for the_tgz_file in files:
                    tgz_file = os.path.join(root, the_tgz_file)
                    if tarfile.is_tarfile(tgz_file) is not True:
                        logger.error("%s is not a tar file" % tgz_file)
                    else:
                        # extract Helm chart to helm_repo_path
                        tar = tarfile.open(tgz_file)
                        for file in tar.getmembers():
                            if file.isfile():
                                logger.info(" - > extracting Helm chart %s" % file.name)
                                tar.extract(file, path=helm_repo_path)
                        tar.close()

            # update repo index with helm linux command
            # subprocess.call([HELM_BIN, "repo", "index", helm_repo_path], stdout=subprocess.STDOUT, stderr=subprocess.STDOUT)
            os.system(HELM_BIN + " repo " + " index " + helm_repo_path)

        # update repo
        os.system(HELM_BIN + " repo update ")
        os.system("sudo " + HELM_BIN + " repo update ")

    except Exception as e:
        raise errors.LoadError(e)
        #raise


attributes = ""

def __get_helm_values(key,value):
    global attributes
    if isinstance(value, dict):
        for the_key in value.keys():
            if key == "":
                __get_helm_values(str(the_key), value[the_key])
            else:
                __get_helm_values(key + "." + str(the_key), value[the_key])
    else:
        attributes += " --set {}={}".format(key,str(value))
    return attributes


def install_chart(namespace,chart_name,chart_version,release_name,values_dict):
    """
    Install a chart on remote k8s cluster
    Use --host and --kube-context for remote Helm server
    :param namespace: K8S namespace to install chart
    :param chart_name:
    :param chart_version:
    :param release_name:
    :param values_dict: dictionary of variables/values to overwrite chart values
    :return:
    """
    try:
        # set Helm values list
        global attributes
        attributes = ""
        helm_values = __get_helm_values(key="",value=values_dict)
        # surcharge helm install linux command
        helm_cmd = "sudo " + HELM_BIN + " install --tls --host "+ Config().get('tiller', 'server') \
                   + ":" + Config().get('tiller', 'port') \
                + " helm_repo/" + chart_name + " --name " + release_name +  " --version " \
                + chart_version + " --namespace " +  namespace + helm_values
        print(helm_cmd)
        os.system(helm_cmd)

    except Exception as e:
        raise errors.DeployError(e)

def install_thirdparty(namespace,storage):
    """
    How install 3rd parties ???
    Case allinone -> just one replica and one node
    No more use with MDT > 1.5.03
    :return:
    """
    try:
        registry_docker  = Config().get('docker', 'registry_address') + ":" +  Config().get('docker', 'registry_port')
    except  Exception as e:
        logger.error('Config value not found')
        raise

    # Where find container version ?? in k8s-config ?
    mongo_version = "3.4.9"
    mongodb_install_version = "0.5"
    rabbimqha_version = "3.7-alpine"
    redis_version = "4.0.9"

    # Install Mongo
    try:
        helm_cmd = "sudo " + HELM_BIN + " install " + Config().get('helm_repo', 'name') + "/mongodb-replicaset --name mon --namespace " \
                   +  namespace + " --set installImage.name=" + registry_docker +"/mongodb-install --set installImage.tag=" \
                    + mongodb_install_version + " --set image.name=" \
                   + registry_docker + "/mongo --set image.tag=" + mongo_version + \
                  " --set auth.enabled=false  --set persistentVolume.enabled=true --set tls.enabled=false " + \
            " --set configmap.replication.replSetName=repldb --set replicas=1 --set replicaSet=repldb --set persistentVolume.storageClass=" + storage
        print(helm_cmd)
        os.system(helm_cmd)
    except Exception as e:
        raise errors.DeployError(e)
    # Install RabbitMQ
    try:
        helm_cmd = "sudo " + HELM_BIN + " install " + Config().get('helm_repo', 'name') + "/rabbitmq-ha --name rab --namespace " +  namespace + \
                  " --set image.repository=" + registry_docker + "/rabbitmq --set image.tag=" + rabbimqha_version + \
                  " --set replicaCount=1 " + \
            " --set rabbitmqPassword=guest,rbac.create=false"
        print(helm_cmd)
        os.system(helm_cmd)
    except Exception as e:
        raise errors.DeployError(e)
    # Install Redis
    try:
        helm_cmd = "sudo " + HELM_BIN + " install " + Config().get('helm_repo', 'name') + "/redis --name red --namespace " +  namespace + \
                    "  --set image=" + registry_docker + "/redis:" + redis_version + " --set persistence.enabled=false --set usePassword=false --set metrics.enabled=false " + \
                    " --set serviceType=ClusterIP --set networkPolicy.enabled=false"
        print(helm_cmd)
        os.system(helm_cmd)
    except Exception as e:
        raise errors.DeployError(e)

def reset_all():
    """
    Reset all charms
    :return:
    """
    try:
        helm_host = Config().get('tiller', 'server') + ":" + Config().get('tiller', 'port')
        helm_cmd = "sudo " + HELM_BIN +  " list  --tls --host " + helm_host \
                   + " | awk '{ if (NR != 1) {print $1}}' |  xargs sudo " + HELM_BIN +  \
                    " delete --purge --tls --host " + helm_host
        print(helm_cmd)
        os.system(helm_cmd)
    except Exception as e:
        raise errors.DeployError(e)
