#!/bin/bash



##################################################
### Install MDT                                ###
##################################################

SCRIPTNAME="`basename "$0"`"
SCRIPTDIR=$(dirname "$(readlink -f "$0")")

LOGFILE="mdt_$SCRIPTNAME.log"

REPO_NAME=mdt_repo
REPO_FILE=/etc/yum.repos.d/mdt.repo
REPO_DIR=rpm_repo
RESOURCES_DIR=resources

DOCKER_OVERLAY_FS="true"

HELM=helm-v*-linux-amd64.tar.gz

__exit_val=0


#######################
# Functions
#######################


show_help() {
   cat << EOF
Usage: $SCRIPTNAME [-h help]
Install MDT. Options are
    -h|--help             : Display this help and exit
    -f|--no-overlay-fs    : Disable docker overlayFS backend
EOF
}


# print error and exit
die ()
{
    local __str="$1"
    echo $__str
    exit 1
}


__cleanup()
{
#   echo -e "\nCleaning up..."

   echo -e "\nExiting..."
   exit $__exit_val
}


trap __cleanup SIGHUP SIGINT SIGTERM


create_local_repo()
{
   cat <<EOM >$REPO_FILE
[$REPO_NAME]
name=Local MDT repository
baseurl=file://$SCRIPTDIR/$REPO_DIR
enabled=1
gpgcheck=0
EOM
}


install_docker ()
{
    yum --disablerepo=* --enablerepo=$REPO_NAME -y install --setopt=obsoletes=0 docker-ce docker-ce-selinux
    usermod -aG docker root

    # Folowing configuration work with docker 1.12, not 1.9
    mkdir -p /etc/systemd/system/docker.service.d
    cat << EOF > /etc/systemd/system/docker.service.d/docker.conf
[Service]
EnvironmentFile=-/etc/sysconfig/docker
ExecStart=
ExecStart=/usr/bin/dockerd \$OPTIONS
EOF

    ## WORKAROUND docker 1.9 : https://github.com/docker/docker/issues/17653 --exec-opt native.cgroupdriver=cgroupfs

    DOCKER_OPTS='OPTIONS=" -H unix:///var/run/docker.sock  --exec-opt native.cgroupdriver=cgroupfs'
    if [ $DOCKER_OVERLAY_FS == "true" ]; then
        DOCKER_OPTS+=' --storage-driver=overlay'
    fi
    DOCKER_OPTS+='"'
    echo $DOCKER_OPTS > /etc/sysconfig/docker

    systemctl daemon-reload
    systemctl enable docker
    systemctl start docker
    systemctl status docker | grep running > /dev/null 2>&1
    [ $? -eq 0 ] || die "Error: [DOCKER] Docker service is not started"
}



#######################
# Main
#######################


exec > >(tee -a $LOGFILE)
exec 2>&1

cd $SCRIPTDIR
echo "============>"
echo "============> $(date)"
echo "============>"
echo


INSTALL_CONF=$RESOURCES_DIR/install.conf
[ -f $INSTALL_CONF ] && source $INSTALL_CONF || exit_on_error "$INSTALL_CONF is not a file"

cat /etc/redhat-release | grep $SUPPORTED_REDHAT_RELEASE #2>&1 > /dev/null
if [ $? -ne 0 ]; then
    die "Error: Red Hat release not supported: $SUPPORTED_REDHAT_RELEASE "
fi


while [ $# -gt 0 ]; do
   case "$1" in
      -h|--help)
         show_help;
         exit 1
         ;;
      -f|--no-overlay-fs)
         DOCKER_OVERLAY_FS="false"
         ;;
      *)
         echo "Invalid option"
         show_help
         exit 1
         ;;
   esac
   shift
done

echo -e "\n***** Creating local repository $REPO_NAME ..."
create_local_repo


echo -e "\n***** Installing $RPM_NAME package ..."
find $REPO_DIR -name "$RPM_NAME*.rpm" -exec yum --enablerepo=$REPO_NAME -y install {} \;

echo -e "\n***** Installing helm package ..."
tar -zxf $RESOURCES_DIR/$HELM && mv ./linux-amd64/helm /usr/bin/helm
rm -rf RESOURCES_DIR/linux-amd64

echo -e "\n***** Installing docker ..."
install_docker

echo -e "\n***** create path for static files ..."
mkdir -p /var/www/ericsson/static/  && chown daemon:daemon /var/www/ericsson/static/

rm -rf $REPO_FILE
echo
echo

exec 1>&- 2>&-

kill -TERM $$
