"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import sys
import signal
import argparse
import logging
import logging.handlers
import subprocess
import os
from ericsson.deploypattern.tools.mdtcli import exec_cmd_shell

from ericsson.deploypattern.lib import errors
from ericsson.deploypattern.lib import ansibleplay
from ericsson.deploypattern.lib import config as config_lib
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.inventory import inventory_command
from ericsson.deploypattern.plugins.deployplugins import load_plugin
from ericsson.deploypattern.plugins.pattern import pattern
from ericsson.deploypattern.tools import labels
from ericsson.deploypattern.tools.dfninit import mgnt_server_init, check_init
from ericsson.deploypattern.tools.yum import load_yum_repo
from ericsson.deploypattern.tools.apt import load_deb_repo
from ericsson.deploypattern.tools.registry import load_docker_registry
from ericsson.deploypattern.tools import helm
from ericsson.deploypattern.tools import bundle
from ericsson.deploypattern.tools import mdtcli
from ericsson.deploypattern.version import __version__

# create logger
logger = logging.getLogger()
logger.setLevel(logging.INFO)

# Console Handler
ch = logging.StreamHandler()
ch.setLevel(logging.INFO)
# formatter = logging.Formatter('%(message)s')
# ch.setFormatter(formatter)
logger.addHandler(ch)

# File Handler
formatter = logging.Formatter('%(asctime)s:'
                              '%(levelname)s:'
                              '%(filename)s:'
                              '%(funcName)s:'
                              ' %(message)s')
fh = logging.handlers.RotatingFileHandler(
    '/tmp/deploypattern.log', 'a', 1000000, 1)
# fh = logging.FileHandler('/var/log/deploypattern.log')
fh.setLevel(logging.INFO)
fh.setFormatter(formatter)
logger.addHandler(fh)

# For Ansible Log
# Ansible add FileHandler to root logger
# Search ansible logger with its key: p=<PID ansible> u=<user> |
# Don't propagate the log to parent logger (root), not to have twice log written
# Add your Handler ansible logger to have Ansible log
for key in logging.Logger.manager.loggerDict.keys():
    if ' | ' in key:
        logging.Logger.manager.loggerDict[key].propagate = False
        logging.Logger.manager.loggerDict[key].addHandler(fh)

DFN_CONF_FILE = '/etc/deploypattern.conf'


class ArgumentParser(argparse.ArgumentParser):
    """
    Override error method in argparse.ArgmentParser
    """

    def error(self, message):
        self.print_help(sys.stderr)
        self.exit(2, '%s: error: %s\n' % (self.prog, message))


def run_pattern(deploy_pattern):
    """ Parse deployment pattern

    Launch the plugin related to the task type
    And run it

    """
    tasks_list = deploy_pattern['deployment_tasks']

    for task in tasks_list:
        logger.info("\n:: %s ::", task['name'])
        logger.info("--------------------------")

        # get task type
        __deployment_type = task['type']

        # load the plugin
        plugin = load_plugin(__deployment_type)
        plugin(task)


def redirect_to_cli(command):
    """
    To manage Help
    :param command:
    :return:
    """
    def handle(parsed_args, unknown_args):
        if unknown_args.count("-h") != 0 or unknown_args.count("--help") != 0:
            cmd = 'sudo docker exec mdt-cli mdt {} {}'.format(
                command, ' '.join(unknown_args))
            returncode, stdout, stderr = exec_cmd_shell(cmd)
            if returncode == 0:
                logger.info(stdout)
                exit(0)
            else:
                logger.error("stderr : " + stderr)
                exit(1)
            try:
                unknown_args.remove("-h")
                unknown_args.remove("-help")
            except ValueError:
                pass
    return handle


def parse_cmdline():
    """
    Parse command line arguments
    """
    parser = ArgumentParser(prog="mdt",
                            usage='mdt [mdt_options] [command] [resource] [command_options]',
                            description=('''
    MDT: MediaFirst Deployment Toolkit
    ==================================
    Tool to deploy Kubernetes cluster and products

                                     '''),
                            formatter_class=argparse.RawDescriptionHelpFormatter)

    # Optional arguments for the MDT
    parser.add_argument('--version', action='version',
                        version="- %(prog)s {version} -"
                        .format(version=__version__))

    parser.add_argument("--debug",
                        action="store_true",
                        help="Enable Debug output messages")

    parser.add_argument("-c", "--config",
                        dest="conf",
                        help="configuration file; "
                        "default is " + DFN_CONF_FILE)

    # Subparser commands
    subparsers = parser.add_subparsers(dest='cmd')
    subparsers.required = False
    # del_parser = subparsers.add_parser('delete',
    #                                   usage='mdt [mdt_options] delete [resource] [command_options].',
    #                                   help='Delete resource')
    deploy_parser = subparsers.add_parser('deploy',
                                          usage='mdt [mdt_options] deploy [resource] [command_options]',
                                          help='Deploy resource')
    get_parser = subparsers.add_parser('get',
                                       usage='mdt [mdt_options] get [resource] [command_options]',
                                       help='Get resource')
    init_parser = subparsers.add_parser('init',
                                        description=('''
!!!! Command available only for MDT bootstrap !!!!

Initialize the MDT software to setup the environment appropriately.
For allinone installation the local disk will be used instead of NFS server
                                        '''),
                                        formatter_class=argparse.RawDescriptionHelpFormatter,
                                        usage='mdt [mdt_options] init  [-i IP_ADDRESS] [-k SSH_KEY] [-g REGISTRY] [-m HELM_REPO_URL] [-d REGISTRY_NFS_ADDRESS] [-p REGISTRY_NFS_PATH] [-u USER] [-r {yum,deb}] [-a ANSIBLE_PATH] [-c 3RDPARTY_PATH] [--http-port HTTP_PORT] [--skip_load] [--stop]',
                                        help='MDT initialisation')
    load_parser = subparsers.add_parser('load',
                                        usage='mdt [mdt_options] load [resource] [command_options]',
                                        help='Load resources')
    reset_parser = subparsers.add_parser('reset',
                                         usage='mdt [mdt_options] reset [resource] [command_options]',
                                         help='Reset ressources')
    # set_parser = subparsers.add_parser('set',
    #                                   usage='mdt [mdt_options] set [resource] [command_options]',
    #                                   help='Set resource')

    # Command deploy
    # -----------------
    deploy_sub_parser = deploy_parser.add_subparsers(dest='cmd_deploy')
    deploy_helm_parser = deploy_sub_parser.add_parser('helm',
                                                      description='Install Helm repository on MDT node and Helm server tiller, on one master.',
                                                      usage='mdt [mdt_options] deploy helm')

    deploy_kubecluster_parser = deploy_sub_parser.add_parser('kube-cluster',
                                                             description=('''
Deploy the kube-cluster, Kubernetes on the nodes masters and workers. The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
The configuration of Kubernetes cluster as HA registry, HA proxy, load balancer ... are given in kube-config file in the config bundle folder.
                                                '''),
                                                             formatter_class=argparse.RawDescriptionHelpFormatter,
                                                             usage='mdt [mdt_options] deploy kube-cluster')

    deploy_products_parser = deploy_sub_parser.add_parser(
        'products', add_help=False)
    deploy_products_parser.set_defaults(
        func=redirect_to_cli('deploy products'))

    deploy_volumes_parser = deploy_sub_parser.add_parser('volumes',
                                                         description=('''
Deploy storage classes from kube-volumes file in config bundle folder.
A storage class is defined with 4 parameters:
* name
* type
* address
* path
                                                '''),
                                                         formatter_class=argparse.RawDescriptionHelpFormatter,
                                                         usage='mdt [mdt_options] deploy volumes')

    # --> mdt deploy helm
    # no parameter

    # --> mdt deploy kube-cluster
    # no parameter

    # --> mdt deploy products [-p profile_name]
#    deploy_products_parser.add_argument("-p", "--profile",
#                                        dest="profile_name",
#                                        default=None,
#                                        help="Specific profile"
#                                        )

    # --> mdt deploy volumes
    # no parameter

    # Command reset
    # ----------------
    reset_sub_parser = reset_parser.add_subparsers(dest='cmd_reset')
    reset_product_parser = reset_sub_parser.add_parser(
        'products', add_help=False)
    reset_product_parser.set_defaults(func=redirect_to_cli('reset products'))

    reset_kubecluster_parser = reset_sub_parser.add_parser('kube-cluster',
                                                           description=('''
Reset the Kubernetes cluster on all nodes and clean load balancers. The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
                                                '''),
                                                           formatter_class=argparse.RawDescriptionHelpFormatter,
                                                           usage='mdt [mdt_options] reset kube-cluster')

    # Command get
    # ---------------
    get_sub_parser = get_parser.add_subparsers(dest='cmd_get')
    get_label_parser = get_sub_parser.add_parser('label', add_help=False)
    get_label_parser.set_defaults(func=redirect_to_cli('get label'))

    get_cfg_parser = get_sub_parser.add_parser('cfg', add_help=False)
    get_cfg_parser.set_defaults(func=redirect_to_cli('get cfg'))

    # Command load
    # -------------
    load_sub_parser = load_parser.add_subparsers(dest='cmd_load')
    load_cfg_parser = load_sub_parser.add_parser('cfg', add_help=False)
    load_cfg_parser.set_defaults(func=redirect_to_cli('load cfg'))

    load_mdtchart_parser = load_sub_parser.add_parser('chart',
                                                      description=('''
Load Helm charts from HELM_TGZ archive file or HELM_DIR directory path.
The Helm charts are pushed to the MDT Helm repository: helm_repo.
                                                '''),
                                                      formatter_class=argparse.RawDescriptionHelpFormatter,
                                                      usage='mdt [mdt_options] load chart (-f CHART_TGZ | -d CHART_DIR)')

    load_container_parser = load_sub_parser.add_parser('container',
                                                       description=('''
Load containers from CONTAINER_TGZ archive file or CONTAINER_DIR directory path.
The containers are pushed to the MDT Docker registry (default).
See NFS configuration to share registry with kube HA Docker registry.
                                                  '''),
                                                       formatter_class=argparse.RawDescriptionHelpFormatter,
                                                       usage='mdt [mdt_options] load container (-f CONTAINER_TGZ | -d CONTAINER_DIR)')

    load_rpm_parser = load_sub_parser.add_parser('rpm',
                                                 description=('''
Load rpm from RPM_TGZ archive file.
The rpm are pushed to the local yum repository.
                                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] load rpm (-f RPM_TGZ )')

    load_deb_parser = load_sub_parser.add_parser('deb',
                                                 description=('''
Load deb from DEB_TGZ archive file.
The deb are pushed to the local deb repository.
                                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] load deb (-f DEB_TGZ )')

    # --> mdt load rpm -f <tar.gz>
    load_rpm_parser.add_argument("-f", "--file",
                                 dest="rpm_tgz",
                                 help="Path to rpm tgz archive to load ")

    # --> mdt load deb -f <tar.gz>
    load_deb_parser.add_argument("-f", "--file",
                                 dest="deb_tgz",
                                 help="Path to deb tgz archive to load ")

    # --> mdt load chart -f <tar.gz> | -d <folder>
    load_mdtchart_exclusive_parser = load_mdtchart_parser.add_mutually_exclusive_group(
        required=True)
    load_mdtchart_exclusive_parser.add_argument("-f", "--file",
                                                dest="chart_tgz",
                                                help="Path to MDT packages (Helm charts) tgz archive to load ")
    load_mdtchart_exclusive_parser.add_argument("-d", "--directory",
                                                dest="chart_dir",
                                                help="Path to MDT packages (Helm charts) directory to load")

    # --> mdt load container [--force] -f <tar.gz> | -d <folder> [-r registry]
    load_container_exclusive_parser = load_container_parser.add_mutually_exclusive_group(
        required=True)
    load_container_exclusive_parser.add_argument("-f", "--file",
                                                 dest="container_tgz",
                                                 default=None,
                                                 help="Path of the container archive file to be loaded")
    load_container_exclusive_parser.add_argument("-d", "--directory",
                                                 dest="container_dir",
                                                 default=None,
                                                 help="Path of container directory to be loaded")
    # TODO when uncommnet don't forget to use this parameter in load_docker_registry function call
    # load_container_parser.add_argument("-r", "--registry",
    #                                    dest="registry",
    #                                    default=None,
    # help="Load containers in specific registry")

    # Command init
    # -------------
    init_parser.add_argument("--debug",
                             action="store_true",
                             help="Enable Debug output messages")
    init_parser.add_argument("-i ", "--ip-address",
                             dest="ip_address",
                             help="IP address of the MDT to use for cluster nodes ")
    init_parser.add_argument("-k ", "--ssh-key",
                             dest="ssh_key",
                             help="Full path to the SSH key to use for cluster node SSH connections")
    init_parser.add_argument("-g ", "--docker-registry",
                             dest="registry",
                             help="Only if external insecure Docker registry is used: <host>:<port> ")
    init_parser.add_argument("-m ", "--helm-repository",
                             dest="helm_repo_url",
                             help="Only if external Helm repository is used: <host>:<port> ")
    init_parser.add_argument("-d ", "--registry-nfs-address",
                             dest="registry_nfs_address",
                             help="NFS address for registry storage")
    init_parser.add_argument("-p ", "--registry-nfs-path",
                             dest="registry_nfs_path",
                             help="Remote NFS path "
                             "(with docker/registry/ path)")
    init_parser.add_argument("-u ", "--user",
                             dest="user",
                             help="Linux user to connect to the remote nodes, default is " + Config().get('ansible', 'remote_user'))
    init_parser.add_argument("-r ", "--repo",
                             dest="repo",
                             choices=["yum", "apt"],
                             help="Repository to use yum (rpm packages) or apt(deb packages), default yum")
    init_parser.add_argument("-a ", "--ansible-path",
                             dest="ansible_path",
                             help="Ansible yaml path, default " + Config().get('ansible', 'path'))
    init_parser.add_argument("-s ", "--stop",
                             action="store_true",
                             dest="stop",
                             help="Stop all containers launched by MDT bootstrap")
    init_parser.add_argument("--skip-load",
                             action="store_true",
                             dest="skip_load",
                             help="Skip load MDT and 3rd party containers to init MDT bootstrap")
    init_parser.add_argument("--http-port",
                             dest="http_port",
                             help="HTTP port to access MDT containers, default " + Config().get('global', 'http_port'))
    init_parser.add_argument("-c", "--container-3rdparty-path",
                             default=None,
                             dest="thirdparty_path",
                             help="Path of the 3rd party containers to load, default /opt/mfvp/deploypattern/resources")

    # Commands  MDT < 1.5
    #---------------------

    # inventory_parser = subparsers.add_parser('inventory',
    #                                          help='Inventory management (MDT <1.5)')
    # inventory_parser.add_argument("--debug",
    #                               action="store_true",
    #                               help="Enable Debug output messages")
    # inventory_parser.add_argument("-l", "--list",
    #                               action="store_true",
    #                               help="List current inventory")
    # inventory_parser.add_argument("-f", "--format",
    #                               dest="format",
    #                               default="ansible",
    #                               help="list in format ansible/yaml")
    # inventory_parser.add_argument("-i", "--import",
    #                               dest="_import",
    #                               help="import inventory")
    # inventory_parser.add_argument("-e", "--export",
    #                               dest="export",
    #                               help="export inventory to ansible format")

    # pattern_parser = subparsers.add_parser(
    #     'pattern', help='Deploy pattern (MDT <1.5)')
    # pattern_parser.add_argument("--debug",
    #                             action="store_true",
    #                             help="Enable Debug output messages")
    # pattern_parser.add_argument("-i", "--install",
    #                             dest="install",
    #                             help="install pattern")
    # pattern_parser.add_argument("-l", "--list-config",
    #                             action="store_true",
    #                             help="list configuration pattern")
    # pattern_parser.add_argument("-c", "--config",
    #                             dest="config",
    #                             help="set pattern configuration")
    # pattern_parser.add_argument("-p", "--path",
    #                             dest="pattern_path",
    #                             help="set pattern configuration")
    # pattern_parser.add_argument("-r", "--run",
    #                             action="store_true",
    #                             help="run pattern - "
    #                             "can be combined with -c/--config")

    parsed_args, unknown_args = parser.parse_known_args()

    if hasattr(parsed_args, 'func'):
        parsed_args.func(parsed_args, unknown_args)

    if len(sys.argv[1:]) == 0:
        parser.print_help()
        parser.exit(2)

    return parsed_args, unknown_args


def signal_handler(signal, frame):
    """ Catch signal """
    logging.info(':: exiting ...')
    sys.exit(0)


def __exec_cmd_shell(cmd, error_msg=None):
    """

    :param cmd:
    :param error_msg:
    :return:
    """
    p = subprocess.Popen(
        cmd, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    (stdout, stderr) = p.communicate()
    logger.debug(":: rc=%s for cmd='%s'" % (p.returncode, cmd))
    if p.returncode != 0:
        logger.info(stdout)
        logger.info(stderr)
        if error_msg:
            raise errors.DeployError(error_msg)
    return(p.returncode, stdout, stderr)

def main():
    """ Main """

    # setup signal handler
    signal.signal(signal.SIGINT, signal_handler)
    # parse cmdline args
    args, unknown_args = parse_cmdline()

    if args.debug:
        formatter = logging.Formatter('%(levelname)s:'
                                      '%(filename)s:'
                                      '%(funcName)s:'
                                      ' %(message)s')
        ch.setFormatter(formatter)
        logger.setLevel(logging.DEBUG)
        ch.setLevel(logging.DEBUG)
        fh.setLevel(logging.DEBUG)

    # Command mdt init
    # ---------------------
    # init config
    if args.conf:
        config = Config(args.conf)
    else:
        config = Config(DFN_CONF_FILE)

    # Launch init process
    # Available only for MDT bootstrap
    if args.cmd == 'init':
        # TODO Check if MDT bootstrap
        try:
            mgnt_server_init(ipaddr=args.ip_address,
                             sshkey=args.ssh_key,
                             registry=args.registry,
                             registry_nfs_addr=args.registry_nfs_address,
                             registry_nfs_path=args.registry_nfs_path,
                             helm_repo=args.helm_repo_url,
                             user=args.user,
                             repo=args.repo,
                             ansible_path=args.ansible_path,
                             skip_load=args.skip_load,
                             stop=args.stop,
                             thirdparty_path=args.thirdparty_path,
                             http_port=args.http_port)
        except Exception as msg:
            logger.error("Initialization failed: %s" %
                         msg.message, exc_info=True)
            return 1

    # Command mdt inventory < 1.5
    # --------------------------
    if args.cmd == 'inventory':
        try:
            inventory_command(args)
        except Exception as msg:
            logger.error("Inventory command failed: %s" %
                         msg.message, exc_info=True)
            return 1
        finally:
            exit(0)

    # check if init as been done before going on
    try:
        check_init(config)
    except Exception as msg:
        logger.error("Init must be done before: %s" % msg.message)
        return 1

    # launch deployment pattern
    if args.cmd == 'pattern':
        try:
            pattern(args)
        except Exception as msg:
            logger.error("Deploying pattern failed: %s" %
                         msg.message, exc_info=True)
            return 1


# NEW MDT 1.5

    # Command mdt deploy
    # --------------------------
    if args.cmd == 'deploy':
        if args.cmd_deploy == 'labels':
            try:
                labels.set_labels(args.audit)
            except Exception as msg:
                logger.error("Deploy labels on nodes failed: %s" % msg.message)
                return 1

        if args.cmd_deploy == 'kube-cluster':
            try:
                bundle.Bundle().deploy_kubecluster()
            except Exception as msg:
                logger.error("Deploying pattern failed: %s" % msg.message)
                return 1
            # Update mdt_api_products with Docker and Helm parameters after
            # Kube cluster deployment

        if args.cmd_deploy == "helm":
            try:
                inventory = bundle.Bundle().get_inventory()
                kube_vars_dict = bundle.Bundle().get_vars_kube_config()
                # Deploy Helm server (container tiller) only on one master with deployment (replicas=1)
                # Add service to expose Helm server
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="deploy_helm",
                    vars=config_lib.get_config_vars_dict(),
                    extra_vars=kube_vars_dict)
                logger.debug("result : %s" % result)
                if result != 0:
                    raise errors.AnsibleError(
                        "Ansible playbook failed : deploy_helm ")

            except Exception as msg:
                logger.error("Install Helm failed: %s" % msg.message)
                return 1

        if args.cmd_deploy == 'products':
            mdtcli.call_mdt_cli('deploy', 'products', unknown_args)

        if args.cmd_deploy == "volumes":
            bundle.Bundle().deploy_volumes()

    # Command mdt reset
    # -------------------------
    if args.cmd == 'reset':
        if args.cmd_reset == 'products':
            # Reset all products
            mdtcli.call_mdt_cli('reset', 'products', unknown_args)

        if args.cmd_reset == 'kube-cluster':
            try:
                bundle.Bundle().reset_kubecluster()
            except Exception as msg:
                logger.error("Reset kube failed: %s" % msg.message)
                return 1
            # Update mdt_api_products with Docker and Helm parameters after
            # Kube cluster reset
            # TODO: is it useful ?

    # Command mdt load
    # -------------------------
    if args.cmd == 'load':
        if args.cmd_load == 'cfg':
            # load config bundle files:
            mdtcli.call_mdt_cli('load', 'cfg', unknown_args)

        if args.cmd_load == 'chart':
            try:
                # load helm chart tgz / load chart tgz:
                if args.chart_tgz:
                    # load_to_helm_repo(args.chart_tgz,args.chart_dir)
                    tgz_dir = "none"
                    helm.load_to_helm_repo(args.chart_tgz, tgz_dir)
                elif args.chart_dir:
                    tgz = "none"
                    helm.load_to_helm_repo(tgz, args.chart_dir)
                else:
                    raise errors.LoadError(
                        "Did you specify -p <mdt_package> option or -d <directory_containing_mdt_packages> option ?")

            except Exception as e:
                logger.error("Loading MDT package failed: %s" % e.message)
                return 1

        if args.cmd_load == 'container':
            try:
                load_docker_registry(
                    args.container_tgz, args.container_dir)
# args.container_tgz, args.container_dir, args.registry) TODO use when
# regitry will be used

            except Exception as msg:
                logger.error("Loading Containers failed: %s" % msg.message)
                return 1

        if args.cmd_load == 'rpm':
            try:
                load_yum_repo(args.rpm_tgz)

            except Exception as msg:
                logger.error("Loading Rpm failed: %s" % msg.message)
                return 1

        if args.cmd_load == 'deb':
            try:
                load_deb_repo(args.deb_tgz)

            except Exception as msg:
                logger.error("Loading deb failed: %s" % msg.message)
                return 1

    # Command mdt-cli get
    # ----------------------
    if args.cmd == 'get':
        if args.cmd_get == 'label':
            # get label
            mdtcli.call_mdt_cli('get', 'label', unknown_args)

        if args.cmd_get == 'cfg':
            # get config bundle files:
            mdtcli.call_mdt_cli('get', 'cfg', unknown_args)

    if len(unknown_args) > 0:
        logger.error("Unknown arguments : {}".format(unknown_args))
        exit(2)
    # logger.info("\n")
    return 0


if __name__ == '__main__':
    main()
