"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.
 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import glob
import logging
import tarfile
import shutil
import time
import sys
import jinja2
import yaml
import requests
import subprocess
from random import randint

from ericsson.deploypattern.lib import ansibleplay
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_config_vars_dict
from ericsson.deploypattern.lib.utils import load_csv, load_yaml
from ericsson.deploypattern.tools import helm
from ericsson.deploypattern.lib import errors

logger = logging.getLogger(__name__)


BUNDLE_FILE_LIST = ['kube-config', 'kube-volumes',
                    'kube-matrix', 'application-version-list']
BUNDLE_FOLDER_LIST = ['mdt-profiles', 'mdt-deploypackages']
BUNDLE_SUFFIX = ('.csv', '.yaml', '.yml')
NODE = 'node'
TYPE = 'type'
LOAD_BALANCER = "load-balancer"
KUBE_MASTER = "kube-master"
KUBE_NODE = "kube-node"
ETCD = 'etcd'
KUBE_ALL = 'k8s-cluster'
PROFILE = 'profile'
KEYS_MATRIX_LIST = [NODE, TYPE, PROFILE]
DPKG = 'deploypackage'
VERSION = 'version'
KEYS_PROFILE_LIST = [DPKG, VERSION]
KEYS_VOLUME_LIST = ['path', 'name', 'address', 'type']
THIRD_PART_LIST = ['mongo', 'rabbitmq', 'redis']


class Bundle(object):
    """

    """

    def __init__(self):
        self._bundle_path = os.path.join(
            Config().get('global', 'path'), 'bundle')
        if not os.path.exists(self._bundle_path):
            os.makedirs(self._bundle_path)

    def _merge_bundle(self, bundle_path):
        """
        Check the consistency of the Config Bundle
        :param bundle_path:
        :return:
        """
        try:
            # Copy files to existing directory
            for file_name in os.listdir(bundle_path):
                src = os.path.join(bundle_path, file_name)
                if os.path.isfile(src):
                    shutil.copy(src, os.path.join(
                        self._bundle_path, file_name))
                else:
                    try:
                        os.makedirs(os.path.join(self._bundle_path, file_name))
                    except:
                        pass
                    for sub_file_name in os.listdir(src):
                        shutil.copy(os.path.join(bundle_path, file_name, sub_file_name),
                                    os.path.join(self._bundle_path, file_name, sub_file_name))
            logger.info(":: load complete")

        except Exception as msg:
            raise errors.LabelError(" %s" % msg)

    def _check_bundle(self, path_to_check):
        """
        Check the consistency of the Config Bundle
        :param bundle_path:
        :return:
        """
        logger.info(":: Check configuration bundle ")

        for filename in BUNDLE_FILE_LIST:
            # Search if file or folder exists in the config bundle folder
            file_path = os.path.join(path_to_check, filename)
            file_list = glob.glob(file_path + "*")
            # Get usually one file or folder, more than one if conversion csv
            # -> yaml
            if len(file_list) == 0:
                logger.warning(
                    "WARNING: %s.yaml is not found in the new config bundle", file_path)
            # Check the format: yaml or csv for file
            for existing_file in file_list:
                if os.path.isfile(existing_file):
                    # Case file:
                    if not existing_file.endswith(BUNDLE_SUFFIX):
                        logger.warning(
                            "WARNING: File %s has not a csv or yaml extension", existing_file)

        for folder in BUNDLE_FOLDER_LIST:
            folder_path = os.path.join(path_to_check, folder)
            if os.path.isdir(folder_path):
                file_list = glob.glob(folder_path + "/*")
                for existing_file in file_list:
                    if os.path.isfile(existing_file):
                        # Case file:
                        if not existing_file.endswith(BUNDLE_SUFFIX):
                            logger.warning(
                                "WARNING: File %s has not a csv or yaml extension", existing_file)
            else:
                logger.warning(
                    "WARNING: The %s folder is missing in the new config bundle", folder)

    def _clean_label_nodes(self):
        """
        Clean persistent label storage
        :return:
        """
        bundle_path = self.get_bundle_path()
        try:
            os.remove(os.path.join(bundle_path, "pki", "labels_history"))
        except:
            pass

    def get_labels(self):
        """
        Get a list of dictionary (node, label, value) to set labels on nodes.
        Per default, use the config bundle under the bundle folder
        Otherwise, raise error.
        :return: list of dictionaries: {node, label}
        """
        logger.info(":: Get list (node, label)")

        # Check matrix_path exists
        f_list = glob.glob(self._bundle_path + "/mdt-matrix*")
        if len(f_list) != 1:
            raise errors.BundleError(
                "Matrix file not found or more than one matrix rin config bundle! ")
        else:
            matrix_path = f_list[0]

        # Convert matrix file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(matrix_path)[1] == '.csv':
            matrix_dict_list = load_csv(matrix_path)
        else:
            matrix_dict_list = load_yaml(matrix_path)

        # Check file: keys are node, type, profile (optional)
        for matrix in matrix_dict_list:
            if (set(matrix.keys()) - set(KEYS_MATRIX_LIST)) == ([]):
                raise errors.BundleError("Key error in %s" % matrix_path)

        # Read files and create dictionary
        label_dict = {}
        for matrix in matrix_dict_list:
            # Add a label to a node corresponding to its type in K8S: master or
            # worker
            label_dict[matrix[NODE]] = []
            if matrix[TYPE] in [KUBE_MASTER, KUBE_NODE]:
                label_dict[matrix[NODE]].append(
                    {'label': matrix[TYPE], 'value': "yes"})

                # Check if profile file exists
                # !! we presume that there is one profile per file, and there is the profile name in profile file
                try:
                    if 'profile' in matrix.keys():
                        profile_path = glob.glob(os.path.join(
                            self._bundle_path, 'mdt-profiles', "*" + matrix['profile'] + "*"))
                        if len(profile_path) == 0:
                            raise errors.BundleError(
                                "Profile file doesn't exist for %s" % matrix['profile'])
                        # Read Profile file and create dictionary
                        profile_path = profile_path[0]
                        # Convert profile file in list of dictionaries one per
                        # deploy package
                        if os.path.splitext(profile_path)[1] == '.csv':
                            profile_dict_list = load_csv(profile_path)
                        else:
                            profile_dict_list = load_yaml(profile_path)

                        # Check file, keys are deploypackage, version
                        for profile in profile_dict_list:
                            if not set(profile.keys()) == set(KEYS_PROFILE_LIST):
                                raise errors.BundleError(
                                    "Key error in %s" % profile_path)

                        for profile in profile_dict_list:
                            # Check if deploy package file exists
                            dpypkg_path = glob.glob(os.path.join(self._bundle_path, 'mdt-deploypackages', "*" +
                                                                 profile['deploypackage'] + "_" + profile['version'] + "*"))
                            if len(dpypkg_path) == 0:
                                raise errors.BundleError("Deploy package file doesn't exist for %s_%s" %
                                                         (profile['deploypackage'], profile['version']))
                            dpypkg_path = dpypkg_path[0]
                            # Convert mdt-deploypackage file in list of
                            # dictionaries one per deypkg
                            if os.path.splitext(dpypkg_path)[1] == '.csv':
                                dpypkg_dict_list = load_csv(dpypkg_path)
                            else:
                                dpypkg_dict_list = load_yaml(dpypkg_path)
                            # If there is a nodeSelector directive, we presume there is only one
                            # Get the key and the value
                            if 'values' in dpypkg_dict_list.keys():
                                if 'nodeSelector' in dpypkg_dict_list['values']:
                                    # Remove duplicate label
                                    label = dpypkg_dict_list['values']['nodeSelector'].keys()[
                                        0]
                                    node_label_list = [l['label']
                                                       for l in label_dict[matrix[NODE]]]
                                    if not label in node_label_list:
                                        label_dict[matrix[NODE]].append({
                                            'label': dpypkg_dict_list['values']['nodeSelector'].keys()[0],
                                            'value': dpypkg_dict_list['values']['nodeSelector'][dpypkg_dict_list['values']['nodeSelector'].keys()[0]]})
                                    else:
                                        # Test if different value for the same
                                        # label
                                        if dpypkg_dict_list['values']['nodeSelector'][dpypkg_dict_list['values']['nodeSelector'].keys()[0]] != label_dict[matrix[NODE]][node_label_list.index(label)]['value']:
                                            raise errors.BundleError(
                                                "WARNING: Different values for the same label")
                except Exception as msg:
                    raise errors.BundleError(msg)

        # Return
        return label_dict

    def get_inventory(self):
        """
        Get inventory from kube matrix file
        :return: dictionary of lists (group,list nodes)
        """
        logger.info(":: Get list (group, node)")

        # Check matrix_path exists
        f_list = glob.glob(self._bundle_path + "/kube/kube-matrix*")
        if len(f_list) != 1:
            raise errors.BundleError(
                "Matrix file not found in config bundle! ")
        else:
            matrix_path = f_list[0]

        # Convert matrix file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(matrix_path)[1] == '.csv':
            matrix_dict_list = load_csv(matrix_path)
        else:
            matrix_dict_list = load_yaml(matrix_path)

        # Check file: keys are node, type, profile (optional)
        for matrix in matrix_dict_list:
            if (set(matrix.keys()) - set(KEYS_MATRIX_LIST)) == ([]):
                raise errors.BundleError("Key error in %s" % matrix_path)

        # Read files and create dictionaries list
        # one dictionary per node type
        # key = node type, value = list of nodes
        inventory_dict_list = []
        inventory_dict_list.append({KUBE_MASTER: []})
        inventory_dict_list.append({KUBE_NODE: []})
        inventory_dict_list.append({LOAD_BALANCER: []})
        inventory_dict_list.append({ETCD: []})
        inventory_dict_list.append({KUBE_ALL: []})

        for matrix in matrix_dict_list:
            # matrix (node, type)
            # Search in dictionary list, the dictionary corresponding to the
            # node type
            index = [i for i, j in enumerate(
                inventory_dict_list) if j.keys()[0] == matrix[TYPE]]
            # Add the node to the list of nodes for this type
            # node = dictionary name, ipaddress (current inventory used by
            # Ansible)
            inventory_dict_list[index[0]][matrix[TYPE]].append(
                {'name': matrix[NODE], "ipaddress": matrix[NODE]})
        # copy master to node group to be able to deploy pod on them
        inventory_dict_list[1][KUBE_NODE].append({'children': KUBE_MASTER})

        inventory_dict_list[3][ETCD].append({'children': KUBE_MASTER})
        inventory_dict_list[4][KUBE_ALL].append({'children': KUBE_MASTER})
        inventory_dict_list[4][KUBE_ALL].append({'children': KUBE_NODE})

        return inventory_dict_list

    def get_vars(self):
        """
        Get K8S variable from kube-config file
        :return: dictionary var:value
        """
        #logger.info(":: Get K8S variables ")

        # Check kube-config  exists
        f_list = glob.glob(self._bundle_path + "/kube/kube-config*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "kube-config file not found in config bundle! ")
        else:
            kube_config_path = f_list[0]

        # Convert k8sconfig file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(kube_config_path)[1] == '.csv':
            kube_vars_dict = load_csv(kube_config_path)
        else:
            kube_vars_dict = load_yaml(kube_config_path)
        return kube_vars_dict

    def get_vars_product(self):
        """
        Get K8S variable from products-config file
        :return: dictionary var:value
        """
        #logger.info(":: Get K8S variables ")

        # Check product-config  exists
        f_list = glob.glob(self._bundle_path + "/products/product-config*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "product-config file not found in config bundle! ")
        else:
            pdt_config_path = f_list[0]

        # Convert k8sconfig file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(pdt_config_path)[1] == '.csv':
            pdt_vars_dict = load_csv(pdt_config_path)
        else:
            pdt_vars_dict = load_yaml(pdt_config_path)
        return pdt_vars_dict

    def get_vars_kube_config(self):
        """

        :return:
        """
        # Build dictionary to pass to ansible playbook to overwrite default
        # values
        bundle_var_dict = self.get_vars()
        var_dict = {}
        for key, value in bundle_var_dict.items():
            var_dict[key] = value
        return var_dict

    def get_vars_common(self):
        """
        Get K8S variables from common file
        :return: dictionary var:value
        """
        #logger.info(":: Get K8S common variables ")

        # Check common  exists
        f_list = glob.glob(self._bundle_path + "/common*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "common-config file not found in config bundle! ")
        else:
            co_config_path = f_list[0]

        # Convert k8sconfig file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(co_config_path)[1] == '.csv':
            co_vars_dict = load_csv(co_config_path)
        else:
            co_vars_dict = load_yaml(co_config_path)
        return co_vars_dict

    def get_vars_products_config(self):
        """

        :return:
        """
        # Build dictionary for jinja template
        bundle_var_dict = self.get_vars_product()
        var_dict = {}
        for key, value in bundle_var_dict.items():
            var_dict[key] = value
        return var_dict

    def get_vars_common_config(self):
        """

        :return:
        """
        # Build dictionary for jinja template
        bundle_var_dict = self.get_vars_common()
        var_dict = {}
        for key, value in bundle_var_dict.items():
            var_dict[key] = value
        return var_dict

    def get_volumes_vars(self):
        """
        Get volumes variables from kube-volumes file
        :return: list of dictionary var:value
        """
        logger.info(":: Get storage variables ")

        # Check k8s-volumes  exists
        f_list = glob.glob(self._bundle_path + "/kube/kube-volumes*")
        if len(f_list) != 1:
            raise errors.FileNotFound(
                "kube-volumes file not found in config bundle! ")
        else:
            k8s_volumes_path = f_list[0]

        # Convert kube-volumes file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(k8s_volumes_path)[1] == '.csv':
            k8s_volume_vars = load_csv(k8s_volumes_path)
        else:
            k8s_volume_vars = load_yaml(k8s_volumes_path)

        return k8s_volume_vars

    def get_bundle_path(self):
        return self._bundle_path

    def _get_namespace(self):
        """
        Get namespace from kube-config file
        :return: namespace
        """
        k8s_vars_dict = self.get_vars()
        return k8s_vars_dict['k8s']['namespace']

    def create_namespace(self):
        """
        In common file, we can define several namespace
        :return:
        """
        # Get namespace from k8s_config
        namespaces = self._get_namespace()
        from ericsson.deploypattern.lib.k8sApi import K8sApi
        for key, value in namespaces.items():
            try:
                K8sApi().create_namespace(value)
            except:
                pass

    def _check_dpypkg_used(self, dpypkg_name):
        """
        Check if dpypkg is used at least one profile by at least one node
        :param dpypkg_name: name of the dpypkg
        :return: boolean
        """
        deploy = False
        # Get chart name from dpypkg file
        chart = dpypkg_name.split('_')[1]
        # Search profile files with this chart:
        f2_list = os.listdir(self._bundle_path + "/products/mdt-profiles")
        p_list = []
        for f2 in f2_list:
            if chart in open(os.path.join(self._bundle_path + "/products/mdt-profiles", f2)).read():
                p_list.append(f2)
        # Search if at least one profile is used
        with open(os.path.join(self._bundle_path + "/products/products-matrix.yaml")) as matrix:
            matrix = matrix.read()
        for profile in p_list:
            # Get profile name from profile file:
            profile = profile.split('_')[1].split('.')[0]
            # Search profile name in matrix file
            if profile in matrix:
                deploy = True
                break
        return deploy

    def install_thirdparty(self, jinja_dict):
        """
        Install 3rd party from Config Bundle
        Don't update csv case
        :return:
        """
        namespace = 'mfvp'
        # Install first All 3rd parties
        for third_part in THIRD_PART_LIST:
            f_list = glob.glob(
                self._bundle_path + "/products/mdt-deploypackages/*" + third_part + "*")
            for file_name in f_list:
                if self._check_dpypkg_used(file_name):
                    # Convert mdt-deploypackage file in list of dictionaries one per
                    # dpypkg
                    if os.path.splitext(file_name)[1] == '.csv':
                        dpypkg_dict_list = load_csv(os.path.join(
                            self._bundle_path, "mdt-deploypackages", file_name))
                    elif os.path.splitext(file_name)[1] == '.yaml':
                        with open(os.path.join(self._bundle_path, "/products/mdt-deploypackages", file_name)) as f_:
                            tmpl = jinja2.Template(f_.read())
                        render = tmpl.render(jinja_dict)
                        dpypkg_dict_list = yaml.load(render)
                    else:
                        raise "Incorrect file format for %s" % third_part
                    helm.install_chart(namespace=namespace,
                                       chart_name=dpypkg_dict_list['package'],
                                       chart_version=dpypkg_dict_list['version'],
                                       release_name=dpypkg_dict_list['package'][0:3],
                                       values_dict=dpypkg_dict_list['values'])

        # Wait until 3rd parties are running
        return

    def _install_dpypkg(self, dpypkg_dict_list):
        """
        Install Helm charts
        Overwrite values in deploy chart by global variables in k8s_config file
        :param dpypkg_dict_list:
        :return:
        """
        # Install chart one or more time
        if 'deploy_list' in dpypkg_dict_list:
            for i in range(len(dpypkg_dict_list['deploy_list'])):
                helm.install_chart(namespace=dpypkg_dict_list['values']['namespace'],
                                   chart_name=dpypkg_dict_list['package'],
                                   release_name=dpypkg_dict_list['deploy_list'][i],
                                   chart_version=dpypkg_dict_list['version'],
                                   values_dict=dpypkg_dict_list['values'])
        else:
            helm.install_chart(namespace=dpypkg_dict_list['values']['namespace'],
                               chart_name=dpypkg_dict_list['package'],
                               chart_version=dpypkg_dict_list['version'],
                               release_name=dpypkg_dict_list['package'][0:3] + "-" + str(
                                   randint(0, 99)),
                               values_dict=dpypkg_dict_list['values'])

    # Resource cfg
    #=============

    def load_bundle(self, conf_bundle_tgz=None, conf_bundle_dir=None, force=False):
        """
        mdt load cfg [-h] (-f CONF_BUNDLE_TGZ | -d CONF_BUNDLE_DIR) [--force]

        Load the content of the given config bundle from CONF_BUNDLE_TGZ archive file or CONF_BUNDLE_DIR directory path.
        If there is already a config bundle loaded, warning message is displayed.
        Use the option --force to force the loading.
        :param conf_bundle_tgz: config bundle to load
        :param conf_bundle_dir: directory config bundle to load
        :param force: to force load
        :return:
        """
        tmp = os.path.join('/tmp', 'bundle_tmp')
        if conf_bundle_tgz is not None:
            try:
                logger.info(":: load configuration bundle %s - force=%s" %
                            (conf_bundle_tgz, force))

                # Check if file exist
                if not os.path.exists(conf_bundle_tgz):
                    raise errors.BundleError(
                        "file %s not found ! " % conf_bundle_tgz)

                # is a readable tar file ?
                if tarfile.is_tarfile(conf_bundle_tgz) is not True:
                    raise errors.BundleError(
                        "%s is not a tar file" % conf_bundle_tgz)

                # Check if config bundle is already load (there is already pki
                # folder => >1)
                if len(os.listdir(self._bundle_path)) > 1 and not force:
                    raise errors.BundleError(
                        "A config bundle is already loaded, use option --force to load the new one")

                # open and extract file and tgz
                tar = tarfile.open(conf_bundle_tgz)
                tar.extractall(path=tmp)
                tar.close()
                # Check Config Bundle: just Warning
                # We can load partially config bundle
                self._check_bundle(tmp)
                self._merge_bundle(tmp)

            except Exception as msg:
                raise errors.LabelError(" %s" % msg)

            finally:
                if os.path.exists(tmp):
                    shutil.rmtree(tmp)

        elif conf_bundle_dir is not None:
            logger.info(":: load configuration bundle %s - %s",
                        conf_bundle_dir, force)
            # Check if config bundle is already load (there is already pki
            # folder => >1)
            if len(os.listdir(self._bundle_path)) > 1 and not force:
                raise errors.BundleError(
                    "A config bundle is already loaded, use option --force to load the new one")
            # is a readable tar file ?
            if not os.path.isdir(conf_bundle_dir):
                raise errors.BundleError(
                    "folder %s not found !" % conf_bundle_dir)
            # Check Config Bundle: just Warning
            # We can load partially config bundle
            self._check_bundle(conf_bundle_dir)
            self._merge_bundle(conf_bundle_dir)

    def get_bundle(self, output_path=None):
        """
        mdt get cfg [-h] -o OUTPUT_FILE

        Get Bundle in archive path
        :param output_path: path is required
        :return: the entire archive in output_path
        """
        logger.info(":: Get Current Config Bundle file ")
        if output_path is not None:
            with tarfile.open(output_path, "w:gz") as tar:
                for file_name in os.listdir(self._bundle_path):
                    file_name = os.path.basename(file_name)
                    if file_name not in ['pki', 'labels_history']:
                        tar.add(os.path.join(
                            self._bundle_path, file_name), file_name)
            tar.close()
        else:
            raise errors.BundleError("The output path is required")

    # Resource volume
    #================
    def deploy_volumes(self):
        """
        mdt deploy volumes

        Deploy storage classes from k8s-volumes file in config bundle folder.
        A storage class is defined with 4 parameters:
        * name
        * type
        * address
        * path
        :param:
        :return:
        """
        logger.info(":: Deploy Volumes")
        inventory = self.get_inventory()
        k8s_volumes_vars = self.get_volumes_vars()
        config_extra_vars = get_config_vars_dict()
        for storage in k8s_volumes_vars:
            try:
                intersection = set(storage.keys()) & set(KEYS_VOLUME_LIST)
            except Exception as msg:
                raise errors.YamlError(
                    "The kube_volumes file is not correct: %s" % msg)
            in_file_only = set(storage.keys()) - set(KEYS_VOLUME_LIST)
            missing_in_file = set(KEYS_VOLUME_LIST) - set(storage.keys())
            if intersection == set(KEYS_VOLUME_LIST):
                if in_file_only != ([]):
                    logger.warn(":: The following parameters will be ignored: %s",
                                in_file_only)
                new_vars_dic = {}
                new_vars_dic["storage_name"] = storage["name"]
                new_vars_dic["nfs_ip"] = storage["address"]
                new_vars_dic["nfs_path"] = storage["path"]
                new_vars_dic["storage_type"] = storage["type"]

                tmp_extra_vars = config_extra_vars
                tmp_extra_vars.update(new_vars_dic)
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="deploy_storage",
                    extra_vars=tmp_extra_vars)
                logger.debug("result : %s" % result)
                if result != 0:
                    raise errors.AnsibleError(
                        "Ansible playbook failed :deploy_storage ")
            else:
                raise errors.YamlError(
                    "In the following storage class is not correct: %s. Missing the following parameters %s" % (storage, list(missing_in_file)[0]))

    # Resource kube-cluster
    #=======================
    def deploy_kubecluster(self):
        """
            mdt deploy kube-cluster

        Deploy the kube-cluster, Kubernetes on the nodes masters and workers.
        The inventory, list of nodes, is given by the kube-matrix file in the config bundle folder.
        The configuration of Kubernetes cluster is given in
        kube-config file in the config bundle folder.
        Before deploy cluster, you have to get the current config bundle and save it
        in /opt/mfvp/deploybundle/bundle
        :return:
        """
        inventory = self.get_inventory()

        logger.info(":: Deploy Kubernetes")

        # By pass Helm install by Kubepray
        # because helm server (tiller) can be called just inside the cluster
        # and mdt bootstrap needs to call it (outside the cluster)
        kube_vars = self.get_vars_kube_config()
        common_vars = self.get_vars_common()
        commont_dict = {}
        commont_dict['common_var'] = common_vars
        var_dict = {}
        var_dict.update(kube_vars)
        var_dict.update(commont_dict)
        var_dict['helm_enabled'] = False
        var_dict['helm_install'] = True

        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="pre_kubespray",
            vars=get_config_vars_dict(),
            extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError(
                "Ansible playbook failed: pre_kubespray ")

        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="/opt/mfvp/deploypattern/ansible/kubespray/cluster",
            vars=get_config_vars_dict(),
            extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError("Ansible playbook failed: cluster ")

        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="post_kubespray",
            vars=get_config_vars_dict(),
            extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError(
                "Ansible playbook failed: post_kubespray ")

    def reset_kubecluster(self):
        """
        Reset kube cluster: reset all nodes masters and workers
        Warning: not clean flannel and delete mdt containers
        :return:
        """
        # Delete namespaces
        result = requests.delete(
            "http://127.0.0.1/api/mdt/kube/cluster/namespaces")
        if result.status_code != 200:
            logger.warn("Failed to Delete Namespaces : status_code = %s  message = %s " %
                        (result.status_code, result.text))

        inventory = self.get_inventory()
        logger.info(":: Reset K8S")
        var_dict = self.get_vars_kube_config()

        # Reset K8S cluster on all nodes a
        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="/opt/mfvp/deploypattern/ansible/kubespray/reset",
            vars=get_config_vars_dict(),
            extra_vars=var_dict)
        logger.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError(
                "Ansible playbook failed : reset ")

        # Clean Label nodes
        self._clean_label_nodes()

        # restart docker after reset
        #os.system('sudo systemctl restart docker')
        subprocess.call(['sudo', 'systemctl', 'restart', 'docker'])

    # Resource product
    #=================
    def deploy_products(self, profile_name=None):
        """
        Deploy products on nodes from Config Bundle files, or only from one profile name
        :param profile_name: name of the profile, found in file name: mfvpel -> mdt-prof_mfvpel.yaml
        :return:
        """

        # !!!! Create namespace from kube-config: there or another command ?
        self.create_namespace()

        # Get common variables for jinja2
        products_dict = self.get_vars_product()
        jinja_dict = {}
        jinja_dict['product_config'] = self.get_vars_product()
        config = get_config_vars_dict()
        jinja_dict['mdt'] = {'docker': {'registry': 'localhost:5000'}}
        jinja_dict['common_config'] = self.get_vars_common()

        logger.info("Jinja dict: %s" % jinja_dict)

        # !!! Install 3rd party: there or another command or mdt-dpypkg ?
        #logger.info(":: Waiting Mongo ...")
        self.install_thirdparty(jinja_dict)

        # Wait 3rd parties installed
        # time.sleep(120)
        sys.stdout.write("Waiting for Mongo  ")
        sys.stdout.flush()
        crono = 0
        while crono < 120:
            sys.stdout.write(".")
            sys.stdout.flush()
            time.sleep(0.5)
            crono += 0.5
        sys.stdout.write("\n")
        sys.stdout.flush()

        logger.info(":: Deploy products")
        if profile_name is not None:
            # Search if profile name exists
            profile_path = glob.glob(os.path.join(
                self._bundle_path, 'products/mdt-profiles', "*" + profile_name + "*"))
            if len(profile_path) == 0:
                raise errors.BundleError(
                    "Profile file doesn't exist for %s" % profile_name)
            # Read Profile file and create dictionary
            profile_path = profile_path[0]
            # Convert profile file in list of dictionaries one per deploy
            # package
            if os.path.splitext(profile_path)[1] == '.csv':
                profile_dict_list = load_csv(profile_path)
            else:
                profile_dict_list = load_yaml(profile_path)

            # Check file, keys are deploypackage, version
            for profile in profile_dict_list:
                if not set(profile.keys()) == set(KEYS_PROFILE_LIST):
                    raise errors.BundleError("Key error in %s" % profile_path)

            for profile in profile_dict_list:
                # Check if deploy package file exists
                dpypkg_path = glob.glob(os.path.join(self._bundle_path, 'mdt-deploypackages', "*" +
                                                     profile['deploypackage'] + "_" + profile['version'] + "*"))
                if len(dpypkg_path) == 0:
                    raise errors.BundleError("Deploy package file doesn't exist for %s_%s" %
                                             (profile['deploypackage'], profile['version']))

                dpypkg_path = dpypkg_path[0]
                # Convert mdt-deploypackage file in list of dictionaries one
                # per dpypkg
                if os.path.splitext(dpypkg_path)[1] == '.csv':
                    dpypkg_dict_list = load_csv(dpypkg_path)
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
                elif os.path.splitext(dpypkg_path)[1] == '.yaml':
                    dpypkg_dict_list = load_yaml(dpypkg_path)
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
            return

        # Deploy packages that are used in matrix and not third parties
        f_list = os.listdir(self._bundle_path + "/products/mdt-deploypackages")
        # Remove 3rd party, not to install them twice
        f_list = [file_name for file_name in f_list if file_name.find(
            'mongo') == -1 and file_name.find('redis') == -1 and file_name.find('rabbit') == -1]

        for file_name in f_list:
            if self._check_dpypkg_used(file_name):
                # Convert mdt-deploypackage file in list of dictionaries one per
                # dpypkg
                if os.path.splitext(file_name)[1] == '.csv':
                    dpypkg_dict_list = load_csv(os.path.join(
                        self._bundle_path, "mdt-deploypackages", file_name))
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
                elif os.path.splitext(file_name)[1] == '.yaml':
                    with open(os.path.join(
                            self._bundle_path, "products/mdt-deploypackages", file_name)) as f_:
                        tmpl = jinja2.Template(f_.read())
                    render = tmpl.render(jinja_dict)
                    dpypkg_dict_list = yaml.load(render)
                    # Install
                    self._install_dpypkg(dpypkg_dict_list)
