"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import os
import logging
import shelve
import yaml

import ericsson.deploypattern.lib.errors as errors
import ericsson.deploypattern.tools.bundle as bundle
from ericsson.deploypattern.lib.k8sApi import K8sApi

logger = logging.getLogger(__name__)


def set_labels(audit=False):
    """
	mdt deploy labels [-a|--audit]

	Set the labels on the nodes based on the bundle folder.
    The nodes are labeled according to their profile and the packages selected in the profile.
    The previous label are removed.
    The option -a|--audit only display the labels to set per node
    :param audit: Just display: node -> labels from config bundle
    :return:
    """
    logger.info(":: Set Labels " )

    # Get labels list
    label_dict = bundle.Bundle().get_labels()

    # Case only audit -> just display
    if audit:
        for n in label_dict:
            for l in label_dict[n]:
                logger.info("Node: %s -> %s=%s" % (n, l['label'], l['value']))
        return

    # Set labels:
    # History of labels are save in a dictionary (persistent with shelve module) -> labels_history with one key
    # per node (file under bundle_path)
    try:
        bundle_path = bundle.Bundle().get_bundle_path()
        previous_labels_dict = shelve.open(os.path.join(bundle_path,"pki","labels_history"),writeback=True)
        current_labels_dict = {}
    except Exception as e:
            raise errors.FileNotFound("%s" % e)

    # For all dictionaries (node, label) in the list
    for node in label_dict:
        try:
            # Check if it is useful to set label (not already in the node)
            for l in label_dict[node]:
                if (node in previous_labels_dict) and not (l['label'] in previous_labels_dict[node]) or \
                    (not node in previous_labels_dict):
                    K8sApi().set_label(node,l['label'],l['value'])
                # Save current labels
                if node in current_labels_dict:
                    if not l['label'] in current_labels_dict[node]:
                        current_labels_dict[node].append(l['label'])
                else:
                    current_labels_dict[node] = [l['label']]
        except Exception as e:
            raise errors.LabelError("--> Error: %s" % e)

    # Check label to remove and save current labels
    for node in current_labels_dict.keys():
        if node in previous_labels_dict:
            temp = previous_labels_dict[node]
        else:
            temp = []
        # Check if label to remove
        to_del_labels_set = set(temp).difference(set(current_labels_dict[node]))
        for label in to_del_labels_set:
            K8sApi().del_label(node,label)
        # Save current labels
        previous_labels_dict[node] = current_labels_dict[node]

    # Save persistent dictionary
    previous_labels_dict.close()


def set_label(node,label,force=False):
    """
    mdt set [--force] label -n|--node <node> -l|--label <key>=<value>|<key>:<value>

    Set a label (key,value) on a node.
    2 formats to pass label parameter: <key>=<value> or <key>:<value>
    If there is already a label with same key on the node, the set label is not available.
    You have to use option --force to set la label with other value.
    :param node: node to set
    :param label: 2 formats: key:value or key=value
    :param force: to force set command
    :return:
    """
    # Search if key or key:value or key=value
    key = label.split(":")[0]
    if key == label:
        key = label.split("=")[0]
        if key == label:
            raise errors.LabelError("Option -l must be key=value or key:value")
        else:
            value = label.split("=")[1]
    else:
        value = label.split(":")[1]


    # Search if label set on node
    bundle_path = bundle.Bundle().get_bundle_path()

    try:
        previous_labels_dict = shelve.open(os.path.join(bundle_path,"pki","labels_history"),writeback=True)
    except:
        previous_labels_dict = {}

    if node in previous_labels_dict:
        if key in previous_labels_dict[node]:
            if not force:
                logger.info("WARNING: label %s already set on node %s" % (key,node))
            else:
                K8sApi().set_label(node,key,value)
        else:
            K8sApi().set_label(node,key,value)
            # Update persistent dictionary
            temp = previous_labels_dict[node]
            temp.append(key)
            previous_labels_dict[node] = temp
    else:
        K8sApi().set_label(node,key,value)
        # Update persistent dictionary
        previous_labels_dict[node] = [key]
    previous_labels_dict.close()



def delete_label(node,label):
    """
	mdt delete label -n|--node <node>] -l|--label <key>|<key>=<value>|<key>:value

    Delete a label on node.
    Display a warning if the node doesn't exist or hasn't the key
    :param node:
    :param label: 3 formats: <key> or <key>=<value> or <key>:value
    :return:
    """
    # Search if key or key:value or key=value
    key = label.split(":")[0]
    key = key.split("=")[0]

    # Search if label set on node
    bundle_path = bundle.Bundle().get_bundle_path()
    try:
        previous_labels_dict = shelve.open(os.path.join(bundle_path,"pki","labels_history"),writeback=True)
    except:
        previous_labels_dict = {}

    if node in previous_labels_dict:
        if key in previous_labels_dict[node]:
            K8sApi().del_label(node,key)
            # Update persistent dictionary
            temp = previous_labels_dict[node]
            temp.remove(key)
            previous_labels_dict[node] = temp
            previous_labels_dict.close()
        else:
            logger.info("WARNING: label %s not set on node %s" % (key,node))
    else:
        logger.info("WARNING: no MDT label set on node %s" % (node))

def get_label(node,output_file=None):
    """
    mdt get label [-n|--node <node>] [-o|--output <output>]

    Get Get all labels (k8s and mdt labels) on a node, if there is option -n or otherwise on all nodes.
    Labels are displayed with yaml format, with 2 lists per node: 1 for MDT_labels and 1 fr Other_labels.
    If option -o, the yaml is saved in file which path is given by the option
    :param node: node to get label
    :param output_file: path to save labels per node information
    :return:
    """
    bundle_path = bundle.Bundle().get_bundle_path()
    try:
        previous_labels_dict = shelve.open(os.path.join(bundle_path,"pki","labels_history"),writeback=True)
    except:
        previous_labels_dict = {}

    if node == None:
        node_list = previous_labels_dict.keys()
    else:
        node_list = [node]

    yaml_dict = {}
    for node in node_list:
        labels_dict = K8sApi().get_label(node)
        yaml_dict[node] = {'MDT_labels':[],'Other_labels':[]}
        if node in previous_labels_dict:
            for key,value in labels_dict.items():
                if key in previous_labels_dict[node]:
                    yaml_dict[node]['MDT_labels'].append({key:value})
            for key,value in labels_dict.items():
                if not key in previous_labels_dict[node]:
                    yaml_dict[node]['Other_labels'].append({key:value})
        else:
            for key,value in labels_dict.items():
                yaml_dict[node]['Other_labels'].append({key:value})

    print (yaml.safe_dump(yaml_dict))

    # Save in output_file
    if output_file != None:
        yaml.safe_dump(yaml_dict,file(output_file,'w'), encoding='utf-8', allow_unicode=True )

