"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)


import subprocess
import os
import logging
import sys
import time
import ericsson.deploypattern.lib.errors as errors


logger = logging.getLogger(__name__)

MDT_CLI = "mdt-cli"


def exec_cmd_shell(cmd):
    # This function should be replaced by the one made in US51109
    p = subprocess.Popen(
        cmd, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    sys.stdout.write("working ")
    sys.stdout.flush()
    while True:
        if p.poll() != None:
            sys.stdout.write("\n")
            sys.stdout.flush()
            break
        else:
            sys.stdout.write(".")
            sys.stdout.flush()
        time.sleep(0.5)
    (stdout, stderr) = p.communicate()
    logger.debug(":: rc=%s for cmd='%s'" % (p.returncode, cmd))
    if p.returncode != 0:
        logger.debug(":: stdout=%s" % stdout)
        logger.debug(":: stderr=%s" % stderr)
        if stderr:
            raise SystemExit(stderr)
    return (p.returncode, stdout, stderr)


def get_index_of_arg(param_list, short_arg=None, long_arg=None):
    """
    This function return index for a list of param like this ['-o', 'test_file.txt', '-d', 'aDirectory', ...]
    :param param_list: the list of param/value
    :param short_arg: argument to find in short version (ex : -o)
    :param long_arg: argument to find in long version (ex:--output)
    :return: Index of argument
    """
    index = None

    if short_arg is not None and param_list.count(short_arg) > 0:
        index = param_list.index(short_arg)
    if long_arg is not None and param_list.count(long_arg) > 0:
        index = param_list.index(long_arg)

    return index


class cfg(object):
    """ Config bundle class
    """

    def __init__(self):
        self.resource = type(self).__name__

    def get(self, optional_param=[]):
        """ Get the entire configuration bundle currently applied and export all 
            files as a tgz archive. The output archive is created in the path
            given in command line with option -o.
        """
        index = get_index_of_arg(optional_param, '-o', '--output')
        if index is not None:
            opt_file = optional_param[index + 1]
            try:
                # Run mdt cli command
                docker_cmd = "sudo docker exec mdt-cli mdt get %s -o /tmp/file.tgz" % (
                    self.resource)
                exec_cmd_shell(docker_cmd)

                # Get file from container
                docker_cmd = "sudo docker cp mdt-cli:/tmp/file.tgz %s" % (
                    opt_file)
                exec_cmd_shell(docker_cmd)

                # Clean folder
                docker_cmd = "sudo docker exec mdt-cli python3.6 -c \"import os; os.remove('/tmp/file.tgz')\""
                exec_cmd_shell(docker_cmd)

            except (ValueError, OSError, SystemExit) as msg:
                # Case no config bundle to return
                raise errors.CliError("mdt-cli command failed: %s" % msg)
        else:
            raise errors.BundleError("Output file name is mandatory")
        del optional_param[index + 1]
        del optional_param[index]

    def load(self, optional_param=[]):
        """Load the content of the given config bundle from CONF_BUNDLE_TGZ archive 
           file or CONF_BUNDLE_DIR directory path. If there is already a config bundle
           loaded, warning message is displayed. Use the option --force to force the loading.
        """

        # Case Load with file parameter
        index = get_index_of_arg(optional_param, '-f', '--file')
        if index is not None:
            # case option file exists
            opt_file = optional_param[index + 1]
            if os.path.isfile(opt_file):
                try:
                    # Load file to container
                    docker_cmd = "sudo docker cp %s mdt-cli:/tmp/file.tgz" % (
                        opt_file)
                    exec_cmd_shell(docker_cmd)
                    # Run mdt cli command
                    if optional_param.count('--force') > 0:
                        index_force = get_index_of_arg(
                            optional_param, None, '--force')
                        docker_cmd = "sudo docker exec mdt-cli mdt load %s -f /tmp/file.tgz --force" % (
                            self.resource)
                        del optional_param[index_force]
                    else:
                        docker_cmd = "sudo docker exec mdt-cli mdt load %s -f /tmp/file.tgz " % (
                            self.resource)
                    logger.debug("%s" % docker_cmd)
                    exec_cmd_shell(docker_cmd)
                except (ValueError, OSError, SystemExit) as msg:
                    # Case no config bundle to return
                    raise errors.CliError("mdt-cli command failed: %s" % msg)
                finally:
                    # Clean folder
                    docker_cmd = "sudo docker exec mdt-cli python3.6 -c \"import os; os.remove('/tmp/file.tgz')\""
                    logger.debug("%s" % docker_cmd)
                    exec_cmd_shell(docker_cmd)
            else:
                # case option file does not exist
                logger.error("load %s failed: file %s not found" %
                             (self.resource, opt_file))
                raise errors.FileNotFound("config bundle not found")
            index = get_index_of_arg(optional_param, '-f', '--file')
            del optional_param[index + 1]
            del optional_param[index]

        # Case Load with folder parameter
        index = get_index_of_arg(optional_param, '-d', '--directory')
        if index is not None:
            # case option folder exists
            opt_dir = optional_param[index + 1]
            if os.path.isdir(opt_dir):
                try:
                    # Load folder to container
                    docker_cmd = "sudo docker cp %s mdt-cli:/tmp/cfg_dir" % (
                        opt_dir)
                    logger.debug("%s" % docker_cmd)
                    exec_cmd_shell(docker_cmd)
                    # Run mdt cli command
                    if optional_param.count('--force') > 0:
                        index_force = get_index_of_arg(
                            optional_param, None, '--force')
                        docker_cmd = "sudo docker exec mdt-cli mdt load %s -d /tmp/cfg_dir --force" % (
                            self.resource)
                        del optional_param[index_force]
                    else:
                        docker_cmd = "sudo docker exec mdt-cli mdt load %s -d /tmp/cfg_dir " % (
                            self.resource)
                    logger.debug("%s" % docker_cmd)
                    exec_cmd_shell(docker_cmd)
                except (ValueError, OSError, SystemExit) as msg:
                    # Case no config bundle to return
                    raise errors.CliError("mdt-cli command failed: %s" % msg)
                finally:
                    # Clean folder
                    docker_cmd = "sudo docker exec mdt-cli python3.6 -c \"import shutil; shutil.rmtree('/tmp/cfg_dir')\""
                    logger.debug("%s" % docker_cmd)
                    exec_cmd_shell(docker_cmd)
            else:
                # case option folder doesn't exist
                logger.error("load %s failed: folder %s not found" %
                             (self.resource, opt_dir))
                raise errors.DirectoryNotFound("config bundle not found")
            index = get_index_of_arg(optional_param, '-d', '--directory')
            del optional_param[index + 1]
            del optional_param[index]


class label(object):
    """ Label class
    """

    def __init__(self):
        self.resource = type(self).__name__

    def get(self, optional_param=[]):
        """Get all the labels from all the nodes or from one node by defining 
        the node name with the option -n. Labels are displayed in yaml format.
         If option -o is set, then the result will also be saved into the file.
         """
        index_node = get_index_of_arg(optional_param, '-n', '--node')
        index_file = get_index_of_arg(optional_param, '-o', '--output')
        try:
            if index_node is None and index_file is None:
                # Run mdt cli command
                docker_cmd = "sudo docker exec mdt-cli mdt get %s" % (
                    self.resource)
                # ouput of docker cmd is redirected on stderr
                (_, _, err) = exec_cmd_shell(docker_cmd)
                logger.info("%s" % err)
            elif index_file is None:
                node = optional_param[index_node + 1]
                # Get file from container
                docker_cmd = "sudo docker exec mdt-cli mdt get %s -n %s" % (self.resource,
                                                                            node)
                # ouput of docker cmd is redirected on stderr
                (_, _, err) = exec_cmd_shell(docker_cmd)
                logger.info("%s" % err)
            elif index_node is None:
                opt_file = optional_param[index_file + 1]
                # Get file from container
                docker_cmd = "sudo docker exec mdt-cli mdt get %s -o /tmp/label.txt" % (
                    self.resource)
                exec_cmd_shell(docker_cmd)

                # copy file
                docker_cmd = "sudo docker cp mdt-cli:/tmp/label.txt %s" % (
                    opt_file)
                exec_cmd_shell(docker_cmd)

                # Clean file
                docker_cmd = "sudo docker exec mdt-cli python3.6 -c \"import os; os.remove('/tmp/label.txt')\""
                exec_cmd_shell(docker_cmd)
            else:
                node = optional_param[index_node + 1]
                opt_file = optional_param[index_file + 1]
                # Get file from container
                docker_cmd = "sudo docker exec mdt-cli mdt get %s -n %s -o /tmp/label.txt" % (
                    self.resource, node)
                exec_cmd_shell(docker_cmd)

                # copy file
                docker_cmd = "sudo docker cp mdt-cli:/tmp/label.txt %s" % (
                    opt_file)
                exec_cmd_shell(docker_cmd)

                # Clean file
                docker_cmd = "sudo docker exec mdt-cli python3.6 -c \"import os; os.remove('/tmp/label.txt')\""
                exec_cmd_shell(docker_cmd)
        except (ValueError, OSError, SystemExit) as msg:
            raise errors.CliError("mdt-cli command failed: %s" % msg)

        # Remove arguments who are processed
        if index_node is not None:
            index_node = get_index_of_arg(optional_param, '-n', '--node')
            del optional_param[index_node+1]
            del optional_param[index_node]
        if index_file is not None:
            index_file = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index_file+1]
            del optional_param[index_file]


class products(object):
    """ Products class
    """

    def __init__(self):
        self.resource = type(self).__name__

    def deploy(self, optional_param=[]):
        """ Deploy all products
        """
        try:
            # Run mdt cli command
            docker_cmd = "sudo docker exec mdt-cli mdt deploy %s " % (
                self.resource)
            # ouput of docker cmd is redirected on stderr
            (_, _, err) = exec_cmd_shell(docker_cmd)
            logger.info("%s" % err)
            
        except (ValueError, OSError, SystemExit) as msg:
            # Case no config bundle to return
            raise errors.CliError("mdt-cli command failed: %s" % msg)

    def reset(self, optional_param=[]):
        """ Reset all products
        """
        try:
            # Run mdt cli command
            docker_cmd = "sudo docker exec mdt-cli mdt reset %s " % (
                self.resource)
            # ouput of docker cmd is redirected on stderr
            (_, _, err) = exec_cmd_shell(docker_cmd)
            logger.info("%s" % err)
            
        except (ValueError, OSError, SystemExit) as msg:
            # Case no config bundle to return
            raise errors.CliError("mdt-cli command failed: %s" % msg)


def call_mdt_cli(operation, resource, args=[]):
    """
    Run docker or kubectl command to call mdt-cli
    :param operation: [ load, get ]
    :param resource: [ cfg, label ]
    :return:
    """
    try:
        logger.info("calling %s operation for %s resource" %
                    (operation, resource))
        class_object = globals()[resource]()
        getattr(class_object, operation)(args)
        logger.info("done")
    except NameError as msg:
        raise errors.CliError("mdt-cli resource does not exist: %s" % msg)
    except AttributeError as msg:
        raise errors.CliError("mdt-cli operation is not allowed: %s" % msg)
    except Exception as msg:
        raise errors.CliError("%s" % msg)
